/********************************************************************************
   watsup    Linux system resources monitor

   Copyright 2007-2023 Michael Cornelison
   source code URL: https://kornelix.net
   contact: mkornelix@gmail.com

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. See https://www.gnu.org/licenses

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.

*********************************************************************************/

#include "zfuncs.h"

#define appname  "watsup"
#define release  "watsup-7.1"

int tiny_report();
int detail_report();


//  main program which starts either the small report or the detailed report

int main(int argc, char **argv)
{
   zinitapp(release,argc,argv);                                                  //  intiz. zfuncs

   if (argv[1] && strstr(argv[1],"detail")) detail_report();                     //  detail report window
   else tiny_report();                                                           //  small report window
   return 1;
}


/********************************************************************************/

//  Small report of the basics: CPU, disk, and network loadings
//  This is intended to be on the screen indefinitely, taking almost no space.

int         Zskip = 2;                                                           //  initial samples to skip
zdialog     *zd;
GtkWidget   *window, *report;
char        setsfile[200];
int         winx, winy;

int tiny_report()
{
   void menufunc(GtkWidget *, int line, int posn, int KBkey);
   int cycle_tiny_report(void *);
   
   FILE     *fid;
   int      nn, n1, n2;
   int      ww, hh;

   winx = winy = 200;

   snprintf(setsfile,200,"%s/.watsup/settings",getenv("HOME"));                  //  settings file

   fid = fopen(setsfile,"r");                                                    //  get window position from prior session
   if (fid) {
      nn = fscanf(fid,"%d %d",&n1,&n2);
      fclose(fid);
      if (nn == 2) {
         winx = n1;
         winy = n2;
      }
   }
   
   zd = zdialog_new("watsup",0,0);                                               //  dialog window, no buttons
   window = zdialog_gtkwidget(zd,"dialog");                                      //  GtkWidget

   zdialog_set_decorated(zd,0);                                                  //  no title bar
   zdialog_add_widget(zd,"report","report","dialog","initializing",0);
   report = zdialog_gtkwidget(zd,"report");                                      //  report text
   textwidget_set_eventfunc(report,menufunc);                                    //  set mouse/KB event function
   zd->popup_report_CB = (void *) menufunc;

   zdialog_run(zd,0);

   gtk_window_move(GTK_WINDOW(window),winx,winy);                                //  restore last position
   
   textwidget_replace(report,0,0," \n");
   textwidget_replace(report,0,1,"initializing\n");
   textwidget_replace(report,0,2," ");

   zmainsleep(1);
   gtk_window_get_size(GTK_WINDOW(window),&ww,&hh);                              //  add margin for window drag
   gtk_window_resize(GTK_WINDOW(window),ww,hh+10);                               //  7.1

   cycle_tiny_report(0);

   gtk_main();
   return 0;
}


//  dialog callback function for clicked report text
   
void menufunc(GtkWidget *widget, int line, int posn, int KBkey)
{
   cchar    *menu[3] = { "quit", "details", "user guide" };
   cchar    *choice;
   char     progexe[300];
   int      cc;

   choice = popup_picklist(widget,menu,0,3);
   if (! choice) return;

   if (strmatch(choice,"quit")) {
      zdialog_free(zd);
      gtk_main_quit();                                                           //  quit app
      return;
   }

   else if (strmatch(choice,"details"))                                          //  start detail report in new process
   {
      cc = readlink("/proc/self/exe",progexe,300);                               //  get own program path
      if (cc <= 0) {
         zmessageACK(0,"cannot get /proc/self/exe");
         return;
      }
      progexe[cc] = 0;
      zshell("log ack","%s details &",progexe);
      return;
   }
   
   else if (strmatch(choice,"user guide")) {                                     //  show user guide
      showz_textfile("data","userguide",window);
      return;
   }
}


//  periodic report of overall system loads in a tiny window

int cycle_tiny_report(void *)
{
   void  getloads(float &cpu, float &disk, float &net);

   int         coreT;
   float       cpuload;                                                          //  total CPU load (100% = 1 core)
   float       diskrate;                                                         //  disk IO rate, MB/sec
   float       netrate;                                                          //  network IO rate, KB/sec
   int         px, py;
   FILE        *fid;

   coreT = coretemp();                                                           //  get CPU temperature
   getloads(cpuload,diskrate,netrate);                                           //  get resource loadings

   if (Zskip) {                                                                  //  discard initial samples
      Zskip--;
      g_timeout_add(500,cycle_tiny_report,0);                                    //  0.5 sec. interval for these
      return 0;
   }

   textwidget_replace(report,0,0," CPU %d%c %d°C \n",int(cpuload), '%', coreT);
   textwidget_replace(report,0,1," I/O %d MB/s \n",int(diskrate));
   textwidget_replace(report,0,2," Net %d KB/s ",int(netrate));

   gtk_window_get_position(GTK_WINDOW(window),&px,&py);                          //  check window position
   if (px != winx || py != winy)                                                 //  changed
   {
      winx = px;                                                                 //  new position
      winy = py;

      fid = fopen(setsfile,"w");                                                 //  update settings file
      if (fid) {                                                                 //    for subsequent session
         fprintf(fid,"%d %d \n",winx,winy);
         fclose(fid);
      }
      else {
         Plog(0,"cannot write settings file: %s \n",strerror(errno));
         gtk_main_quit();
         return 0;
      }
   }

   g_timeout_add(1000,cycle_tiny_report,0);                                      //  1.0 sec. interval
   return 0;
}


//  Return CPU load in percent, disk IO in MB/sec, network IO in KB/sec.
//  Statistics are based on the interval between prior and current calls.
//  CPU load: 100 corresponds to one processor core fully loaded.
//  Zeros are returned for the first call.

void getloads(float &cpuload, float &diskrate, float &netrate)
{
   static int     ftf = 1;                                                       //  1st time flag
   FILE           *fid;
   char           *pp, buff[1000];
   double         val1, val2;
   timeval        timev;
   static double  jifftime;                                                      //  CPU time slice
   static double  time1, time2, elapsed;
   static double  jiff1, jiff2, dio1, dio2, net1, net2;

   if (ftf) jifftime = 1.0 / sysconf(_SC_CLK_TCK);                               //  "jiffy" time slice = 1.0 / HZ
   
   jiff2 = dio2 = net2 = 0;

   fid = fopen("/proc/stat","r");                                                //  read /proc/stat
   if (fid) {
      while ((pp = fgets(buff,999,fid)))
      {
         if (strncmp(pp,"cpu ",4) == 0) {                                        //  look for "cpu " rec. (all cores)
            parseprocrec(pp,2,&val1,4,&val2,NULL);                               //  get user and system jiffies
            jiff2 += val1 + val2;
         }
      }
      fclose(fid);
   }
   
   fid = fopen("/proc/diskstats","r");                                           //  read disk statistics
   if (fid) {
      while ((pp = fgets(buff,999,fid)))
      {
         if (! strmatchN(pp+13,"sd",2) &&                                        //  look for 'sd' (disk)
             ! strmatchN(pp+13,"sr",2) &&                                        //        or 'sr' (CD/DVD/BR)
             ! strmatchN(pp+13,"nv",2)) continue;                                //        or 'sr' (CD/DVD/BR)
         parseprocrec(pp,6,&val1,10,&val2,NULL);                                 //  get sectors read and written
         dio2 += val1 + val2;
      }
      fclose(fid);
   }
   
   fid = fopen("/proc/net/dev","r");                                             //  read network statistics
   if (fid) {
      while ((pp = fgets(buff,999,fid)))
      {
         pp = strchr(buff,':');                                                  //  look for device "eth0:" etc.
         if (! pp) continue;
         parseprocrec(pp+1,1,&val1,9,&val2,NULL);                                //  bytes received and transmitted
         net2 += val1 + val2;
      }
      fclose(fid);
   }
   
   gettimeofday(&timev,0);
   time2 = timev.tv_sec + 0.000001 * timev.tv_usec - 946684800.0;                //  seconds since 2000.01.01

   if (ftf) {
      ftf = 0;
      time1 = time2;
      jiff1 = jiff2;
      dio1 = dio2;
      net1 = net2;
      cpuload = diskrate = netrate = 0;
      return;
   }
   
   elapsed = time2 - time1;
   cpuload = (jiff2 - jiff1) * jifftime / elapsed * 100.0;                       //  percent CPU time
   diskrate = (dio2 - dio1) * 512.0 / 1048576.0 / elapsed;                       //  disk megabytes / sec.
   netrate = (net2 - net1) / 1024.0 / elapsed;                                   //  network kilobytes /sec.
   
   time1 = time2;                                                                //  save counters for next call
   jiff1 = jiff2;
   dio1 = dio2;
   net1 = net2;

   return;   
}


/********************************************************************************/

//  Detailed report of overall loadings and per-process loadings

//  program parameters and limits
#define  maxpid 1000                                                             //  max. PIDs sampled
#define  maxtop 100                                                              //  max. top PIDs to report
#define  proc_ignore "getty init"                                                //  processes to ignore
#define  mega (1024 * 1024)                                                      //  computer megabyte
#define  autoscroll GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC                    //  scroll window policy
#define  textwin GTK_TEXT_WINDOW_TEXT                                            //  GDK window of GTK text view

//  Linux system parameters
int      pagesize;                                                               //  memory page size
double   jiffy;                                                                  //  CPU time slice
int      Ncpus;                                                                  //  CPU count

//  editable parameters and default values
double   Pinterval = 1.0;                                                        //  sample interval, 1 sec.
char     matchtext[100] = "";                                                    //  process match text (wild patterns)
zlist_t  *zmatchhist;                                                            //  match text history
char     zhistfile[200];                                                         //  history file ~/.watsup/matchhist

//  calculated parameters
int      topmost = 20;                                                           //  topmost procs within window
int      txwhh;                                                                  //  height of text window

//  system statistics record layout
struct   sysrec {
   double   cpubusy;
   double   memused, memfree, memcache, swapfree;
   double   pfsoft, pfhard;
   double   Dops, Dblocks;
   double   netrecv, netxmit;
};

sysrec      srec1, srec2;                                                        //  system stats, sample 1 and sample 2

//  process statistics record layout
struct   pidrec { 
   int      pid;
   int      row;                                                                 //  array row
   double   weight;
   double   cpu;                                                                 //  CPU jiffies
   double   mem;                                                                 //  memory MB
   double   pfs;                                                                 //  hard fault count
   double   dios;                                                                //  IO count
   int      repline;                                                             //  report line number
   char     username[20];
   char     exefname[40];
};

pidrec      prec1[maxpid];                                                       //  proc recs, sample 1
pidrec      prec2[maxpid];                                                       //  proc recs, sample 2
pidrec      prec3[maxpid];                                                       //  proc recs for report
int         nrec1, nrec2, nrec3;                                                 //  record counts for precN arrays
int         rec1, rec2, rec3;                                                    //  indexes for precN arrays
int         repline;                                                             //  tracks report output lines

double      elapsed;                                                             //  elapsed time between 1 and 2
int         Fresort = 0;                                                         //  flag, re-sort topmost by weight
int         Pstart = 0, Pruns = 0;                                               //  flags, control process report
int         Fomitidle = 0;                                                       //  omit idle processes

namespace zfuncs {                                                               //  externals from zfuncs module
   extern GdkDisplay        *display;                                            //  X11 workstation (KB, mouse, screen)
   extern GdkDeviceManager  *manager;                                            //  knows screen / mouse associations
   extern GdkScreen         *screen;                                             //  monitor (screen)
   extern GdkDevice         *mouse;                                              //  pointer device
   extern GtkSettings       *settings;                                           //  screen settings
   extern GtkWidget         *mainwin;                                            //  main window for zfuncs parent
   extern char              *progexe;                                            //  executable image file
   extern char              *appimagexe;                                         //  appimage executable image file
   extern cchar             *build_date_time;                                    //  build date and time
}

GtkWidget   *Pwin, *mVbox, *mScroll, *mLog;                                      //  main window widgets

int m_draw(GtkWidget *, cairo_t *cr);                                            //  main window draw event

void callbackfunc2(GtkWidget *, int line, int pos, int kbkey);                   //  main window mouse click function

void menufunc(GtkWidget *item, const char *menu);                                //  menu functions
void m_parms();
void m_start();
void m_stop();
void m_sort();
void m_quit();
void m_help();

void load_zhist();                                                               //  load and save match history data
void save_zhist();

void get_sys_stats(sysrec *srec);                                                //  functions for system statistics
void report_sys_stats();

void get_proc_stats(pidrec *prec, int &nnp);                                     //  functions for process statistics
void report_proc_stats();


//  main function

int detail_report()
{
   GtkWidget   *tbar;
   int         err, prio;

   load_zhist();                                                                 //  load match history data

   Pwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                                   //  create window
   zfuncs::mainwin = Pwin;

   gtk_window_set_title(GTK_WINDOW(Pwin),release);
   gtk_window_set_default_size(GTK_WINDOW(Pwin),800,500);                        //  set size and location

   mVbox = gtk_box_new(VERTICAL,0);                                              //  vertical packing box
   gtk_container_add(GTK_CONTAINER(Pwin),mVbox);                                 //  add to main window

   tbar = create_toolbar(mVbox,32);                                              //  create tool bar and buttons
   add_toolbar_button(tbar,"parms","edit parameters","preferences.png",menufunc);
   add_toolbar_button(tbar,"start","start reporting","run.png",menufunc);
   add_toolbar_button(tbar,"stop","stop reporting","stop.png",menufunc);
   add_toolbar_button(tbar,"sort","resort top processes","refresh.png",menufunc);
   add_toolbar_button(tbar,"quit","quit watsup","quit.png",menufunc);
   add_toolbar_button(tbar,"help","user guide","help.png",menufunc);

   mScroll = gtk_scrolled_window_new(0,0);                                       //  scrolled window
   gtk_box_pack_start(GTK_BOX(mVbox),mScroll,1,1,0);                             //  add to main window mVbox
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mScroll),autoscroll);

   mLog = gtk_text_view_new();                                                   //  text window
   gtk_text_view_set_left_margin(GTK_TEXT_VIEW(mLog),2);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(mLog),0);
   gtk_container_add(GTK_CONTAINER(mScroll),mLog);                               //  add to scrolled window
   
   textwidget_set_eventfunc(mLog,callbackfunc2);                                 //  connect mouse click function
   G_SIGNAL(Pwin,"delete-event",m_quit,0);                                       //  connect window delete event
   G_SIGNAL(Pwin,"destroy",m_quit,0);                                            //  connect window destroy event
   G_SIGNAL(Pwin,"draw",m_draw,0);

   window_to_mouse(Pwin);                                                        //  move to mouse position
   gtk_widget_show_all(Pwin);                                                    //  show window

   jiffy = 1.0 / sysconf(_SC_CLK_TCK);                                           //  "jiffy" time slice = 1.0 / HZ
   pagesize = sysconf(_SC_PAGESIZE);                                             //  system page size
   Ncpus = sysconf(_SC_NPROCESSORS_ONLN);                                        //  SMP CPU count

   prio = getpriority(PRIO_PROCESS,0);                                           //  current priority
   if (prio > -2 && getuid() == 0) {
      err = setpriority(PRIO_PROCESS,0,-2);                                      //  boost priority if allowed
      if (err) Plog(1,"watsup: unable to boost priority \n");
      else Plog(1,"watsup: priority set to -2 \n");
   }

   m_start();                                                                    //  start report running

   gtk_main();                                                                   //  start main window loop

   return 0;
}


//  window draw event - recalibrate report window dimensions

int m_draw(GtkWidget *, cairo_t *cr)
{
   GdkWindow   *textwin;
   textwin = gtk_text_view_get_window(GTK_TEXT_VIEW(mLog),GTK_TEXT_WINDOW_WIDGET);
   txwhh = gdk_window_get_height(textwin);                                       //  text window height
   return 0;
}


//  mouse click event - get row clicked, get process data from current report

void callbackfunc2(GtkWidget *WIN, int line, int pos, int kbkey) 
{
   int         ii;
   int         kill, killpid;                                                    //  process to kill, PID
   char        killfname[100];                                                   //  executable filename
   
   for (ii = 0; ii < nrec3; ii++)
      if (prec3[ii].repline == line) break;                                      //  find clicked process record
   if (ii == nrec3) return;

   killpid = prec3[ii].pid;
   if (killpid <= 0) return;

   strncpy0(killfname,prec3[ii].exefname,99);

   kill = zmessageYN(Pwin,"Kill this process? \n %d  %s",killpid,killfname);
   if (kill) zshell("log ack","kill -KILL %d",killpid);

   return;
}


//  menu selection events

void menufunc(GtkWidget *item, const char *menu)
{
   if (strmatch(menu,"parms")) m_parms();
   if (strmatch(menu,"start")) m_start();
   if (strmatch(menu,"stop")) m_stop();
   if (strmatch(menu,"sort")) m_sort();
   if (strmatch(menu,"quit")) m_quit();
   if (strmatch(menu,"help")) m_help();
   return;
}


//  edit report parameters

void m_parms() 
{ 
   int   zd_parms_event(zdialog *zd, cchar *event);                              //  dialog event function

   zdialog     *zd;
   int         nn, ii;

/***
       ______________________________________
      |     edit watsup parameters           |
      |                                      |
      | sample period [____|-+]              |
      | [x] omit idle processes              |
      | match text: [_____________________]  |
      | history: [______________________|v]  |
      |                                      |
      |                      [OK] [add hist] |
      |______________________________________|

***/

   zd = zdialog_new("edit watsup parameters",Pwin,"OK","add hist",null);

   zdialog_add_widget(zd,"hbox","hbsamp","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","lab1","hbsamp","sample period:","space=5");
   zdialog_add_widget(zd,"spin","period","hbsamp","0.1|99.0|0.1|1.0");

   zdialog_add_widget(zd,"hbox","hbomit","dialog",0,"space=3");
   zdialog_add_widget(zd,"check","omitidle","hbomit","omit idle processes");

   zdialog_add_widget(zd,"hbox","hbmatch","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labmatch","hbmatch","match text:","space=5");
   zdialog_add_widget(zd,"zentry","matchtext","hbmatch",0,"size=25|expand");

   zdialog_add_widget(zd,"hbox","hbhist","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labhist","hbhist","history:","space=5");
   zdialog_add_widget(zd,"combo","matchhist","hbhist",0,"size=25|expand");

   zdialog_stuff(zd,"period",Pinterval);                                         //  stuff widgets with current params
   zdialog_stuff(zd,"matchtext",matchtext);
   zdialog_stuff(zd,"omitidle",Fomitidle);
   
   nn = zlist_count(zmatchhist);                                                 //  match text history
   for (ii = 0; ii < nn; ii++)
      zdialog_stuff(zd,"matchhist",zlist_get(zmatchhist,ii));
   zdialog_stuff(zd,"matchhist","(all)");                                        //  top entry is "(all)"

   zdialog_resize(zd,400,0);
   zdialog_run(zd,zd_parms_event,"parent");                                      //  run dialog, parallel
   return;
}


int zd_parms_event(zdialog *zd, cchar *event)                                    //  dialog event function
{
   int      ii;
   
   if (strmatch(event,"period"))                                                 //  get sample period
      zdialog_fetch(zd,"period",Pinterval);
   
   if (strmatch(event,"omitidle"))                                               //  get omit idle option
      zdialog_fetch(zd,"omitidle",Fomitidle);

   if (strmatch(event,"matchtext"))                                              //  get match text input
      zdialog_fetch(zd,"matchtext",matchtext,100);

   if (strmatch(event,"matchhist")) {                                            //  match history text selected
      zdialog_fetch(zd,"matchhist",matchtext,100);
      zdialog_stuff(zd,"matchtext",matchtext);                                   //  stuff match text in dialog
   }
   
   if (zd->zstat == 2) {                                                         //  [add hist]
      zd->zstat = 0;
      if (*matchtext) {
         zlist_prepend(zmatchhist,matchtext,1);                                  //  add new entry if not already
         while (zlist_count(zmatchhist) > 20)                                    //  prune to last 20 entries
            zlist_remove(zmatchhist,20);
         for (ii = 0; ii < zlist_count(zmatchhist); ii++)                        //  rebuild combobox
            zdialog_stuff(zd,"matchhist",zlist_get(zmatchhist,ii));
      }
   }      

   if (! zd->zstat) return 1;                                                    //  wait for completion
   
   zdialog_free(zd);
   return 1;
}


//  load and save match history data

void load_zhist()
{
   snprintf(zhistfile,200,"%s/matchhist",get_zhomedir());                        //  get .../.watsup/matchhist

   zmatchhist = zlist_from_file(zhistfile);                                      //  load match history data

   if (! zmatchhist || ! zlist_count(zmatchhist)) {                              //  none, create minimum dummy
      zmatchhist = zlist_new(1);
      zlist_put(zmatchhist,"(all)",0);
      zlist_to_file(zmatchhist,zhistfile);
   }

   return;
}

void save_zhist()
{
   if (! zmatchhist || ! zlist_count(zmatchhist)) {                              //  if no data, create minimum dummy
      zmatchhist = zlist_new(1);                                                 
      zlist_put(zmatchhist,"(all)",0);
   }
   
   zlist_to_file(zmatchhist,zhistfile);                                          //  save match history data to file
   return;
}


//  start reporting

void m_start()
{
   int proc_report(void *);
   if (Pruns) return;
   Pruns = 1;
   proc_report(0);
   return;
}


//  stop reporting

void m_stop()
{
   Pruns = 0;
   return;
}


//  re-sort top processes to top of list

void m_sort()
{ 
   Fresort = 1;                                                                  //  set flag for process report
   return;
}


//  exit process report

void m_quit()
{
   Pruns = 0;
   save_zhist();                                                                 //  save match history data
   gtk_widget_destroy(Pwin);                                                     //  kill window
   exit(0);                                                                      //  exit process 
   return;
}


//  show help menu

void m_help()
{
   void show_userguide(GtkWidget *, cchar *menu);
   void show_about(GtkWidget *, cchar *menu);

   GtkWidget   *menu;

   menu = create_popmenu();
   add_popmenu_item(menu,"user guide",show_userguide,0,0);
   add_popmenu_item(menu,"about watsup",show_about,0,0);
   popup_menu(Pwin,menu);

   return;
}


//  show the user guide in a popup window

void show_userguide(GtkWidget *, cchar *menu)
{
   showz_textfile("data","userguide",Pwin);
   return;
}


//  show the about text in a popup window

void show_about(GtkWidget *, cchar *menu)
{
   zabout();
   return;
}


//  process report main function

int proc_report(void *)
{
   int      millisecs = Pinterval * 1000;                                        //  sample interval, milliseconds
   static double   time1 = 0, time2;                                             //  sample times 1 and 2

   if (! Pruns) return 0;                                                        //  report stopped

   time2 = get_seconds();

   if (! time1) {                                                                //  first call, no time #1
      time1 = time2;
      get_sys_stats(&srec1);                                                     //  get system stats #1
      get_proc_stats(prec1,nrec1);                                               //  get process stats #1
      g_timeout_add(millisecs,proc_report,0);                                    //  reschedule and exit
      return 0;
   }
   
   get_sys_stats(&srec2);                                                        //  get system stats #2
   get_proc_stats(prec2,nrec2);                                                  //  get process stats #2

   elapsed = time2 - time1;
   report_sys_stats();                                                           //  system report for interval #1 to #2
   report_proc_stats();                                                          //  process report for interval #1 to #2

   time1 = time2;                                                                //  copy time #2 to time #1

   srec1 = srec2;                                                                //  copy system stats #2 to #1

   for (rec2 = 0; rec2 < nrec2; rec2++)                                          //  copy process stats #2 to #1
         prec1[rec2] = prec2[rec2];
   nrec1 = nrec2;

   g_timeout_add(millisecs,proc_report,0);                                       //  reschedule and exit
   return 0;
}


//  get system level statistics from various /proc/xxxx files

void get_sys_stats(sysrec *srec)
{
   int         ii;
   FILE        *fid;
   char        *pp, buff[1000];
   const char  *ppc;
   double      val1, val2, val3, val4;
   
   memset(srec,0,sizeof(sysrec));                                                //  zero data
   
   fid = fopen("/proc/stat","r");                                                //  read /proc/stat
   if (fid) {
      while ((pp = fgets(buff,999,fid)))
      {
         if (strncmp(pp,"cpu",3) == 0) {                                         //  look for "cpuN" record
            ii = atoi(pp+3);
            if (ii < 0 || ii >= Ncpus) continue;
            parseprocrec(pp,2,&val1,4,&val2,null);                               //  get user and system mode jiffies
            srec->cpubusy += val1 + val2;
            if (ii == Ncpus-1) break;
         }
      }
      fclose(fid);
   }

   parseprocfile("/proc/meminfo","MemTotal:",&val1,"MemFree:",&val2,             //  memory statistics
                                 "Cached:",&val3,"SwapFree:",&val4,null);
   srec->memused = val1 - val2 - val3;                                           //  used = total-free-cached
   srec->memfree = val2;
   srec->memcache = val3;
   srec->swapfree = val4;

   parseprocfile("/proc/vmstat","pgfault",&val1,"pgmajfault",&val2,null);        //  page fault statistics
   srec->pfsoft = val1;
   srec->pfhard = val2;

   fid = fopen("/proc/diskstats","r");                                           //  read disk statistics
   if (fid) {
      while ((pp = fgets(buff,999,fid)))
      {
         ppc = substring(buff,' ',3);                                            //  disk, sda sda1 sr0 nvme0n1
         if (! ppc) continue;
         if (ppc[0] == 's' && ppc[3]) continue;                                  //  sdaN  ignore partitions
         if (ppc[0] == 'n' && ppc[7] == 'p') continue;                           //  nvmexxxpN  ignore partitions
         parseprocrec(pp,4,&val1,6,&val2,8,&val3,10,&val4,null);
         srec->Dops += val1 + val3;                                              //  read/write counts and blocks
         srec->Dblocks += val2 + val4;
      }
      fclose(fid);
   }
   
   fid = fopen("/proc/net/dev","r");                                             //  read network statistics
   if (fid) {
      while ((pp = fgets(buff,999,fid)))
      {
         pp = strchr(buff,':');                                                  //  look for device "eth0:" etc.
         if (! pp) continue;
         parseprocrec(pp+1,1,&val1,9,&val2,null);                                //  get bytes received and transmitted
         srec->netrecv += val1;
         srec->netxmit += val2;
      }
      fclose(fid);
   }
   
   return;   
}


//  get process level statistics from /proc/<pid>/stat files
      
void get_proc_stats(pidrec *prec, int &nnp)
{
   DIR            *procdir;
   FILE           *fid;
   struct dirent  *procent;
   char           *ppid, *pp, *pp2;
   char           username[20], exefname[40];
   char           buff1[300], buff2[3000];
   int            ii, cc, pid;
   double         uid, cpu1, cpu2, cpu3, cpu4;
   double         mem, pfs1, pfs2, rio, wio, rbs, wbs;

   procdir = opendir("/proc");
   if (! procdir) zappcrash("open /proc failed: %s",strerror(errno));
   
   nnp = 0;

   while (true)                                                                  //  loop per PID
   {
      procent = readdir(procdir);                                                //  read /proc directory
      if (! procent) break;
      ppid = procent->d_name;
      if (*ppid < '1' || *ppid > '9') continue;                                  //  skip non-pid entries
      
      pid = atoi(ppid);
      pfs1 = pfs2 = cpu1 = cpu2 = cpu3 = cpu4 = mem = 0;
      *exefname = *username = 0;
      
      snprintf(buff1,300,"/proc/%s/status",ppid);                                //  read file /proc/<pid>/status
      parseprocfile(buff1,"Uid:",&uid,null);                                     //  get real UID

      snprintf(buff1,300,"/proc/%s/environ",ppid);                               //  read file /proc/<pid>/environ
      fid = fopen(buff1,"r");
      if (! fid) continue;
      pp = fgets(buff2,3000,fid);
      fclose(fid);
      if (! pp) continue;
      
      if (uid == 0) strcpy(username,"root");                                     //  uid 0, user = root
      
      else {
         strcpy(username,"unknown");
         while (*pp) {                                                           //  loop null-terminated strings
            pp2 = strstr(pp,"USER");                                             //  look for xxxUSERxxx=name
            if (pp2) {
               pp2 = strchr(pp2,'=');
               if (pp2) {
                  strncpy0(username,pp2+1,19);                                   //  found, use it
                  break;
               }
            }
            pp += strlen(pp) + 1;                                                //  next string
         }
      }

      snprintf(buff1,300,"/proc/%s/exe",ppid);                                   //  proc/<pid>/exe
      cc = readlink(buff1,buff2,999);                                            //  executable pathname/filename
      if (cc <= 0) continue;
      buff2[cc] = 0;
      pp = (char *) strrchr(buff2,'/');                                          //  isolate filename
      if (pp) pp++;
      else pp = buff2;
      if (strstr(proc_ignore,pp)) continue;                                      //  ignore some processes
      strncpy0(exefname,pp,39);                                                  //  keep filename only

      if (*matchtext && ! strmatch(matchtext,"(all)")) { 
         for (ii = 1; ; ii++) {                                                  //  match text strings
            pp = (char *) substring(matchtext,' ',ii);                           //  (multiple blank-separated)
            if (! pp) break;
            if (MatchWild(pp, ppid) == 0) break;                                 //  match process ID
            else if (MatchWild(pp, username) == 0) break;                        //  match user name
            else if (MatchWild(pp, exefname) == 0) break;                        //  match exe file name
         }

         if (! pp) continue;                                                     //  no matches, skip this process
      }

      snprintf(buff1,300,"/proc/%s/stat",ppid);                                  //  read file /proc/<pid>/stat
      fid = fopen(buff1,"r");
      if (! fid) continue;
      pp = fgets(buff2,999,fid);
      fclose(fid);
      if (! pp) continue;
      
      pp = strchr(pp,')');                                                       //  closing ')' after (short) filename
      if (pp) parseprocrec(pp+1, 10,&pfs1, 11,&pfs2, 12,&cpu1,                   //  parse and get wanted fields
                       13,&cpu2, 14,&cpu3, 15,&cpu4, 22,&mem, null);
      
      snprintf(buff1,300,"/proc/%s/io",ppid);                                    //  read file /proc/<pid>/io
      parseprocfile(buff1,"syscr:",&rio, "syscw:",&wio,                          //  get read, write IO operations
                          "read_bytes:",&rbs, "write_bytes:",&wbs, null);        //    and read, write byte counts 

      prec[nnp].pid = pid;                                                       //  build proc record
      prec[nnp].cpu = cpu1 + cpu2 + cpu3 + cpu4;                                 //  user and system jiffies
      prec[nnp].mem = mem;                                                       //  real memory
      prec[nnp].pfs = pfs1 + pfs2;                                               //  major page faults
      prec[nnp].dios = rio + wio;                                                //  I/O counts
      strcpy(prec[nnp].username,username);                                       //  user name
      strcpy(prec[nnp].exefname,exefname);                                       //  executable file
      
      if (++nnp == maxpid) zappcrash("exceeded %d processes",maxpid);
   }

   closedir(procdir);
   return;
}


//  create report from two sets of system data and time interval

void report_sys_stats()
{
   int         ii, jj, lines = 4;
   double      cpu1, cpupct, MBused, MBfree, MBcache, MBswap;
   double      pfsoft, pfhard;
   double      Dops, diskMBs;
   double      netrecv, netxmit;
   char        *pp, sysline[4][100];
   const char  *header = " period    CPU    memory MB     PFs/sec      disk/sec    net KB/sec \n";

   repline = 0;                                                                  //  track report lines 
   textwidget_replace(mLog,1,repline++,header);

   cpu1 = srec2.cpubusy - srec1.cpubusy;                                         //  CPU busy time
   cpupct = cpu1 * 100.0 * jiffy / elapsed;

   MBused = srec2.memused / 1024.0;                                              //  get used, free, cached memory, MB
   MBfree = srec2.memfree / 1024.0;
   MBcache = srec2.memcache / 1024.0;
   MBswap = srec2.swapfree / 1024.0; 
   
   pfsoft = (srec2.pfsoft - srec1.pfsoft) / elapsed;                             //  hard and soft faults / sec
   pfhard = (srec2.pfhard - srec1.pfhard) / elapsed;
   
   Dops = (srec2.Dops - srec1.Dops) / elapsed;                                   //  disk IOs and KB / sec
   diskMBs = (srec2.Dblocks - srec1.Dblocks) / elapsed;
   diskMBs = diskMBs / 2048;                                                     //  blocks to megabytes

   netrecv = (srec2.netrecv - srec1.netrecv) / elapsed / 1024;                   //  network KB / sec
   netxmit = (srec2.netxmit - srec1.netxmit) / elapsed / 1024;

   memset(sysline,' ',sizeof(sysline));                                          //  clear report lines

   for (ii = 0; ii < lines; ii++)
   {
      pp = sysline[ii];                                                          //  pointer to line 0 ... 3
      
      if (ii == 0) sprintf(pp,"%7.2f",elapsed);                                  //  elapsed time on line 0
      
      if (ii == 0) sprintf(pp+10,"%4.0f",cpupct);                                //  CPU % on line 0

      if (ii == 0) sprintf(pp+17," used: %.0f",MBused);                          //  memory statistics on lines 0-2
      if (ii == 1) sprintf(pp+17,"cache: %.0f",MBcache);
      if (ii == 2) sprintf(pp+17,"avail: %.0f (swap: %.0f)",MBfree,MBswap);
      
      if (ii == 0) sprintf(pp+32,"soft: %.0f",pfsoft);                           //  page fault stats on lines 0-1
      if (ii == 1) sprintf(pp+32,"hard: %.0f",pfhard);
      
      if (ii == 0) sprintf(pp+45,"IOs: %.0f",Dops);                              //  disk I/O stats on lines 0-1
      if (ii == 1) sprintf(pp+45,"MBs: %.0f",diskMBs);
      
      if (ii == 0) sprintf(pp+57,"recv: %.0f",netrecv);                          //  network stats on lines 0-1
      if (ii == 1) sprintf(pp+57,"xmit: %.0f",netxmit);
   }
   
   for (ii = 0; ii < lines; ii++)                                                //  output the lines
   {
      pp = sysline[ii];
      for (jj = 0; jj < 99; jj++) if (pp[jj] < ' ') pp[jj] = ' ';                //  remove imbedded nulls
      for (jj = 98; jj; jj--) if (pp[jj] > ' ') break;                           //  and trailing blanks
      pp[jj+1] = 0;
      textwidget_replace(mLog,0,repline++,"%s \n",pp);
   }
}


//  create report from two sets of process data and time interval

void report_proc_stats()
{
   static int     ftf = 1, ycoord, linehh = 20;
   GtkTextBuffer  *textbuff;
   GtkTextIter    iter;

   const char  *header2 = "    PID    CPU     MB   PF/s   IO/s   User         Executable \n";

   double      cpupct, MB, dios, weight;
   char        scpupct[8];
   int         row, nrep;
   int         rowmap[maxpid];
                                                                                 //  sort keys:
   int      key1[2][3] = { { 8, 8, 8 }, { 0, 4, 3 } };                           //    weight descending, PID ascending
   int      key2[1][3] = { { 4, 4, 3 } };                                        //    row ascending
   int      key3[1][3] = { { 0, 4, 3 } };                                        //    PID ascending

   if (ftf) {                                                                    //  get text line height
      ftf = 0;
      textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mLog));
      gtk_text_buffer_get_iter_at_line(textbuff,&iter,0);
      gtk_text_view_get_line_yrange(GTK_TEXT_VIEW(mLog),&iter,&ycoord,&linehh);
   }

   textwidget_replace(mLog,1,repline++,header2);
   
   for (rec2 = 0; rec2 < nrec2; rec2++)                                          //  copy proc recs #2 to #3 for reporting
      prec3[rec2] = prec2[rec2];
   nrec3 = nrec2;

   for (rec1 = rec3 = 0; rec3 < nrec3; rec1++, rec3++)                           //  loop through proc recs #3
   {
      if (rec1 >= nrec1 || prec1[rec1].pid != prec3[rec3].pid) {                 //  find same PID in proc recs #1
         for (rec1 = 0; rec1 < nrec1; rec1++)                                    //  (utilize order, but don't count on it)
            if (prec1[rec1].pid == prec3[rec3].pid) break;
      }
      
      if (rec1 < nrec1) {                                                        //  if found, calculate deltas:
         prec3[rec3].cpu -= prec1[rec1].cpu;                                     //    cpu jiffies
         prec3[rec3].pfs -= prec1[rec1].pfs;                                     //    hard page faults
         prec3[rec3].dios -= prec1[rec1].dios;                                   //    I/O count
         prec3[rec3].weight = prec1[rec1].weight;                                //  copy weight and row from proc rec #1
         prec3[rec3].row = prec1[rec1].row;                                      //    to proc rec #3
      }

      else {
         prec3[rec3].weight = 0;                                                 //  not found (new process was started)
         prec3[rec3].row = 0;                                                    //  weight and row = 0
      }
   }

   for (rec3 = 0; rec3 < nrec3; rec3++)                                          //  calc. new sort weights
   {
      weight = prec3[rec3].cpu 
             + 0.001 * (prec3[rec3].pfs 
             + 0.010 * prec3[rec3].dios);                                        //  lumps PF + disk + widget I/O
      weight = 0.25 * weight + 0.75 * prec3[rec3].weight;                        //  use historical weighted average
      prec3[rec3].weight = weight;
   }
      
   MemSort((char *) prec3, sizeof(pidrec), nrec3, key1, 2);                      //  sort proc recs by weight descending

   topmost = txwhh / linehh - repline - 1;                                       //  proc recs fitting report window
   if (topmost < 3) topmost = 3;
   if (topmost < nrec3) nrep = topmost;                                          //  topmost proc recs to report
   else nrep = nrec3;

   for (row = 0; row < nrep; row++) rowmap[row] = 0;                             //  clear row map

   for (rec3 = 0; rec3 < nrep; rec3++)                                           //  assign report row to topmost proc recs
   {
      row = prec3[rec3].row;                                                     //  keep prior report row
      if (row > 0 && row < nrep) rowmap[row-1] = 1;                              //    if available and in range
      else prec3[rec3].row = 0;
   }
   
   for (rec3 = row = 0; rec3 < nrep; rec3++)                                     //  assign rest from top down
   {
      if (prec3[rec3].row > 0) continue;
      while (rowmap[row] > 0) row++;
      prec3[rec3].row = 1 + row++;
   }

   for (rec3 = nrep; rec3 < nrec3; rec3++)                                       //  clear all rows after topmost
      prec3[rec3].row = 0;
   
   if (Fresort) {                                                                //  if a re-sort was requested - 
      for (rec3 = 0; rec3 < nrep; rec3++)                                        //    keep current order by weight
            prec3[rec3].row = rec3 + 1;
      Fresort = 0;
   }

   MemSort((char *) prec3, sizeof(pidrec), nrep, key2, 1);                       //  sort topmost proc recs by row

   for (rec3 = 0; rec3 < nrep; rec3++)                                           //  output each proc record
   {
      cpupct = prec3[rec3].cpu * 100.0 * jiffy / elapsed;                        //  format 0.0 to 9.9, then 10 to 100
      if (cpupct > 101 * Ncpus) cpupct = 0;                                      //  terminated child added to parent
      if (cpupct < 9.95) sprintf(scpupct,"%4.1f",cpupct);                        //  (stop double-counts)
      else sprintf(scpupct,"%4.0f",cpupct);

      MB = prec3[rec3].mem * pagesize / mega;                                    //  memory MB
      dios = prec3[rec3].dios / elapsed;                                         //  IO/sec

      if (strmatchN(prec3[rec3].username,"systemd",7))                           //  truncate systemd-xxxxxx user name 
         prec3[rec3].username[7] = 0;
      
      if (Fomitidle)
         if (cpupct < 1 && dios < 1) continue;
      
      prec3[rec3].repline = repline;                                             //  remember process report line

      textwidget_replace(mLog,0,repline++,
               "%7d %6s %6.0f %6.0f %6.0f   %-12s %s \n",                        //  7-digit PID, 12 char. username 
               prec3[rec3].pid, scpupct, MB+0.5, prec3[rec3].pfs,                //  MB/sec 
               dios, prec3[rec3].username, prec3[rec3].exefname);                //  IO/sec
   }

   textwidget_clear(mLog,repline);                                               //  clear rest of window
   
   MemSort((char *) prec2, sizeof(pidrec), nrec2, key3, 1);                      //  sort proc recs #2 by PID ascending
   MemSort((char *) prec3, sizeof(pidrec), nrec3, key3, 1);                      //  sort proc recs #3 by PID ascending
   
   for (rec2 = 0; rec2 < nrec2; rec2++)                                          //  copy new weights and rows
   {                                                                             //    to proc recs #2
      if (prec2[rec2].pid != prec3[rec2].pid) zappcrash("bug 23");
      prec2[rec2].weight = prec3[rec2].weight;
      prec2[rec2].row = prec3[rec2].row;
   }
   
   return;
}


//  dialog KB event function (not used but mandatory)

void KBevent(GdkEventKey *event) 
{
   printf("KB event\n");
   return; 
}




