=head1 NAME

iPE::Annotation::Segment - Defines a contiguous segment of a layered annotation where all layers are consistent.

=head1 DESCRIPTION

This is the smallest unit of the final annotation.  Each annotation file is grouped into annotations of locuses which overlap.  These are divided into segments which have the consistent state annotations in all layers of the annotation.  

A segment contains information on each layer of the annotation including the state that the gHMM is in for the entirety of the segment, as well as an iPE::Annnotation::Transcript object describing the entire isoform.

=head1 FUNCTIONS

=over 8

=cut

package iPE::Annotation::Segment;
use strict;

=item new (memberHash)

Creates a new annotation Segment with the keys of the member hash.  Required keys are:

=over 8

=item start 

The beginning coordinate of the segment.

=item end 

The ending coordinate of the segment.

=item features 

Array reference to references to features in each layer.

=item transcripts 

Array reference to references to the transcripts in each layer.

=back

=cut
sub new {
    my ($class, $m) = @_;
    my $this = bless {}, $class;

    die "Incomplete instantiation of $class.  Required keys are \n".
         "start, end, features, and locuses.\n"
        if(!defined $m->{start}  || !defined $m->{end} || 
           !defined $m->{features} || !defined $m->{transcripts});

    $this->{$_."_"} = $m->{$_}  for (keys %$m); 
    for (my $i = 0; $i < scalar(@{$m->{features}}); $i++) {
        next if (!defined $m->{features}->[$i]);
        #$this->{features_}->[$i] = $m->{features}->[$i]->clone;
    }
    $this->{validLayers_} = 0;
    for my $layer (@{$this->{features_}}) {
        $this->{validLayers_}++ if defined $layer;
    }

    return $this;
}

=item start (), end ()

The start and end coordinates of the segment, respectively.

=item features ()

The feature layers of the segment.  For each layer there is either a defined feature or undef.

=item transcripts ()

The transcripts of the segment.  As with features, these are either defined as a locus or undef.  These layers give you an overall picture of the coordinates of the locus, which features are where and a transcript ID if provided.  See L<iPE::Annotation::Transcript> for more details.

=item layers ()

The total number of layers in this segment.

=item validLayers ()

The number of valid layers in this segment.  There may be more layers that are empty, but they are not counted in this function.

=cut
sub start       { shift->{start_}       }
sub end         { shift->{end_}         }
sub features    { shift->{features_}    }
sub transcripts { shift->{transcripts_} }
sub layers      { scalar(shift->{features_}) }
sub validLayers { shift->{validLayers_} }

=back

=head1 SEE ALSO

L<iPE::Annotation::SegmentedAnnotation>, L<iPE::Annotation>

=head1 AUTHOR

Bob Zimmermann (rpz@cse.wustl.edu).

=cut
1;
