/* Licensed to Stichting The Commons Conservancy (TCC) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * TCC licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * mod_ca.h
 *
 *  Author: Graham Leggett
 */

/**
 * @file mod_ca.h
 * @brief Certificate authority hooks.
 *
 * @defgroup MOD_CA mod_ca
 * @ingroup  APACHE_MODS
 * @{
 */

#ifndef MOD_CA_H_
#define MOD_CA_H_

#include "httpd.h"
#include "apr_hash.h"
#include "apr_hooks.h"
#include "apr_pools.h"
#include "apr_time.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Create a set of CA_DECLARE(type), CA_DECLARE_NONSTD(type) and
 * CA_DECLARE_DATA with appropriate export and import tags for the platform
 */
#if !defined(WIN32)
#define CA_DECLARE(type)        type
#define CA_DECLARE_NONSTD(type) type
#define CA_DECLARE_DATA
#elif defined(CA_DECLARE_STATIC)
#define CA_DECLARE(type)        type __stdcall
#define CA_DECLARE_NONSTD(type) type
#define CA_DECLARE_DATA
#elif defined(CA_DECLARE_EXPORT)
#define CA_DECLARE(type)        __declspec(dllexport) type __stdcall
#define CA_DECLARE_NONSTD(type) __declspec(dllexport) type
#define CA_DECLARE_DATA         __declspec(dllexport)
#else
#define CA_DECLARE(type)        __declspec(dllimport) type __stdcall
#define CA_DECLARE_NONSTD(type) __declspec(dllimport) type
#define CA_DECLARE_DATA         __declspec(dllimport)
#endif

typedef struct ca_asn1_t ca_asn1_t;
struct ca_asn1_t
{
    const unsigned char *val;
    apr_size_t len;
};

typedef enum ca_cert_status_e ca_cert_status_e;
enum ca_cert_status_e
{
    CA_CERT_STATUS_GOOD = 0,
    CA_CERT_STATUS_REVOKED = 1,
    CA_CERT_STATUS_UNKNOWN = 2
};

#define CA_SERIAL_NUMBER "serialNumber"
#define CA_THIS_UPDATE "thisUpdate"
#define CA_NEXT_UPDATE "nextUpdate"
#define CA_CERT_STATUS "certStatus"
#define CA_REVOCATION_REASON "revocationReason"
#define CA_REVOCATION_TIME "revocationTime"
#define CA_HOLD_INSTRUCTION_CODE "holdInstructionCode"
#define CA_INVALIDITY_DATE "invalidityDate"

/**
 * Sign the certificate sign request passed in.
 *
 * The buffer is expected to contain a DER encoded certificate sign
 * request, which on return is expected to contain a DER encoded
 * X509 certificate.
 *
 * @param r The request.
 * @param params Optional extra parameters. May include the certificate
 * subject in 'subject', proof of possession in the form of a certificate
 * sign request in 'popCertificateSignRequest' or 'popSpkac'.
 * @param cert Pointer to a DER encoded PKCS7 containing certificates. If
 * the certificate was issued immediately, a pointer to the signed
 * certificate will be returned at this location allocated from the
 * request pool, and OK will be returned. The full certificate chain is
 * returned.
 * @param len Pointer to the DER encoded PKCS7 length. The size of the
 * returned DER encoded PKCS7 will be written to this location on return
 * if necessary.
 * @return If the certificate was issued immediately, we return OK,
 * and the certificate is written to the buffer. If the certificate is
 * to be returned later, perhaps after an offline process is completed,
 * we return DONE. On failure, we return an HTTP return code.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_sign,
        (request_rec *r, apr_hash_t *params, const unsigned char **buffer,
                apr_size_t *len));

/**
 * Authorize the certificate sign request.
 *
 * The buffer is expected to contain a DER encoded certificate sign
 * request, and the hook is expected to verify the details within
 * the CSR and abort if necessary.
 *
 * @param r The request.
 * @param params Optional extra parameters, such the proof of possession
 * of the private key.
 * @param cert Pointer to a DER encoded X509 certificate sign request.
 * @param len The DER encoded X509 certificate sign request length.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_reqauthz,
        (request_rec *r, apr_hash_t *params, const unsigned char *buffer,
                apr_size_t len));

/**
 * Store the certificate.
 *
 * The buffer is expected to contain the signed certificate,
 * and the hook is expected to store the certificate in a suitable
 * place.
 *
 * @param r The request.
 * @param params A hash containing additional optional parameters,
 * such as the transactionID. May be NULL.
 * @param cert Pointer to a DER encoded X509 certificate.
 * @param len The DER encoded X509 certificate length.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_certstore,
        (request_rec *r, apr_hash_t *params, const unsigned char *buffer, apr_size_t len));

/**
 * Get the CA certificate.
 *
 * Return the DER encoded CA certificate for this CA. An unsigned char
 * buffer and its length will be written to the pointers provided,
 * allocated from the pool.
 *
 * @param r The request.
 * @param cacert Pointer to the CA certificate returned at this location,
 * allocated from the request pool.
 * @param len The size of the returned DER encoded X509 CA certificate
 * will be written to this location on return.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_getca,
        (request_rec *r, const unsigned char **cacert, apr_size_t *len, apr_time_t *validity));

/**
 * Get the next CA certificate.
 *
 * Return the DER encoded next CA certificate for this CA. An unsigned
 * char buffer and its length will be written to the pointers provided,
 * allocated from the pool. If a pointer is provided, the notAfter date
 * will be written as validity.
 *
 * If the next CA certificate is not available for any reason, DECLINED
 * should be returned.
 *
 * @param r The request.
 * @param cacert Pointer to the CA certificate returned at this location,
 * allocated from the request pool.
 * @param len The size of the returned DER encoded X509 CA certificate
 * will be written to this location on return.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_getnextca,
        (request_rec *r, const unsigned char **cacert, apr_size_t *len, apr_time_t *validity));

/**
 * Get the certificate chain.
 *
 * Return the DER encoded certificate chain for this CA. An unsigned char
 * buffer and its length will be written to the pointers provided,
 * allocated from the pool.
 *
 * @param r The request.
 * @param chain Pointer to the certificate chain returned at this location,
 * allocated from the request pool.
 * @param len The size of the returned DER encoded X509 certificate chain
 * will be written to this location on return.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_getchain,
        (request_rec *r, const unsigned char **chain, apr_size_t *len, apr_time_t *validity));

/**
 * Get the CRL.
 *
 * Return the DER encoded CRL for this CA. An unsigned char
 * buffer and its length will be written to the pointers provided,
 * allocated from the pool.
 *
 * @param r The request.
 * @param crl Pointer to the CA certificate returned at this location,
 * allocated from the request pool.
 * @param len The size of the returned DER encoded X509 CRL
 * will be written to this location on return.
 * @param validity If specified, pointer to an apr_time_t where the
 * next update time of the CRL will be written if present.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_getcrl,
        (request_rec *r, const unsigned char **crl, apr_size_t *len, apr_time_t *validity));

/**
 * Get the status of a certificate.
 *
 * A hash is passed in containing at least the serial number of the
 * certificate, and implementations are expected to update the hash
 * with the certificate status, and any further extensions supported.
 *
 * Used primary for responding to OCSP requests.
 *
 * @param r The request.
 * @param certstatus Hash containing the certificate serialNumber. On
 * return, the hash is expected to contain the certificateStatus, in
 * addition to optional extension data.
 * @param validity If specified, pointer to an apr_time_t where the
 * next update time of this status will be written if present.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_getcertstatus,
        (request_rec *r, apr_hash_t *certstatus, apr_time_t *validity));

/**
 * Get the certificate, if it exists.
 *
 * A hash is passed in containing at least the serial number of the
 * certificate, or the transaction ID of the original request, and if
 * the certificate exists, the certificate will be returned.
 *
 * @param r The request.
 * @param search Hash containing the certificate serialNumber, or
 * the transactionID.
 * @param cert Pointer to the DER encoded X509 certificate will be returned
 * at this location, allocated from the request pool.
 * @param len The size of the returned DER encoded X509 certificate
 * will be written to this location on return.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_getcert,
        (request_rec *r, apr_hash_t *search, const unsigned char **cert, apr_size_t *len));

/**
 * Get the next serial number.
 *
 * An optional hash is passed in containing the CSR, or the transactionID of the
 * original request. The next serial number is expected to be returned as a DER
 * encoded ASN1 integer.
 *
 * @param r The request.
 * @param params Hash containing parameters such as the CSR and optional
 * transactionID.
 * @param serial Pointer to the DER encoded ASN1 integer serial number will be returned
 * at this location, allocated from the request pool.
 * @param len The size of the returned DER encoded ASN1 integer serial number
 * will be written to this location on return.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_makeserial,
        (request_rec *r, apr_hash_t *params, const unsigned char **serial, apr_size_t *len));

/**
 * Create a private key.
 *
 * The private key is returned as a DER encoded PKCS8 structure.
 *
 * @param r The request.
 * @param params Hash containing parameters.
 * @param key Pointer to the DER encoded PKCS8 private key will be returned
 * at this location, allocated from the request pool.
 * @param len The size of the returned DER encoded PKCS8 private key
 * will be written to this location on return.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_makekey,
        (request_rec *r, apr_hash_t *params, const unsigned char **key, apr_size_t *len));

/**
 * Get the current time for timestamping or certificate signing.
 *
 * Return a DER encoded ASN1 generalizedTime representing the current time for the
 * purposes of signing certificates or timestamping.
 *
 * @param r The request.
 * @param time If specified, return the number of microseconds since 1 January 1970 UTC.
 * @param as If specified, return the accuracy of the seconds of this time source.
 * @param ams If specified, return the accuracy of the milliseconds of this time source.
 * @param amicro If specified, return the accuracy of the microseconds of this time source.
 * @return OK on success, HTTP return code on failure.
 */
APR_DECLARE_EXTERNAL_HOOK(ap, CA, int, ca_gettime,
        (request_rec *r, apr_time_t *time, apr_interval_time_t *as, apr_interval_time_t *ams, apr_interval_time_t *amicro));

#ifdef __cplusplus
}
#endif

#endif /* MOD_CA_H_ */
/** @} */
