#.rst:
# ECMAddCompilerFlag
# ------------------
#
# Add compiler one or more flag conditionally
#
# ::
#
#  ECM_ADD_CXX_COMPILER_FLAGS_IF_SUPPORTED(FLAGS <flag|flags>
#     [SUPPORTED_IF <support_condition>]
#     [QUERY_IF <query_condition>])
#
# C-language equivalent::
#
#  ECM_ADD_C_COMPILER_FLAGS_IF_SUPPORTED(FLAGS <flag|flags>
#     [SUPPORTED_IF <support_condition>]
#     [QUERY_IF <query_condition>])
#
# add ``<flag>`` or ``<flags>`` to CMAKE_CXX_FLAGS if the compiler supports them.
# Support is determined by the ``SUPPORTED_IF`` expression if provided or by
# querying the compiler directly if a ``QUERY_IF`` expression is true. The
# ``QUERY_IF`` expression takes precedence if it evaluates to true.
# The compiler is also queried if no conditions are given at all.
# The compiler is queried for each flag in turn and only the supported
# flag(s) are added. The ``SUPPORTED_IF`` condition applies to all flags.
#
# examples:
#
# add flags -a, -b and -c when using a GCC compiler or Clang:
#
# .. code-block:: cmake
#
#   ecm_add_cxx_compiler_flags_if_supported(FLAGS -a -b -c SUPPORTED_IF CMAKE_CXX_COMPILER_ID STREQUAL "GNU" OR CMAKE_CXX_COMPILER_ID MATCHES "Clang")
#
# add flags -d, -e and/or -f if the compiler supports them:
#
# .. code-block:: cmake
#
#   ecm_add_cxx_compiler_flags_if_supported(FLAGS -d -e -f)
#
# add flag -a when using a GCC > 5 compiler, or query the compiler when running on a Mac
# (a more specific test would be APPLE AND CMAKE_CXX_COMPILER_ID MATCHES "Clang" but there is
# little chance another compiler will be used on that platform):
#
# .. code-block:: cmake
#
#   ecm_add_c_compiler_flags_if_supported(FLAGS -g
#     SUPPORTED_IF CMAKE_CXX_COMPILER_ID STREQUAL "GNU" AND NOT CMAKE_CXX_COMPILER_VERSION VERSION_LESS "5.0.0"
#     QUERY_IF APPLE)
#
# Since 5.xx

#=============================================================================
# Copyright 2018,9      René J.V. Bertin <rjvbertin@gmail.com>
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

include(CMakeParseArguments)
include(CheckCXXCompilerFlag)
include(CheckCCompilerFlag)


function(ECM_ADD_CXX_COMPILER_FLAGS_IF_SUPPORTED)
    set(_OPTIONS_ARGS)
    set(_ONE_VALUE_ARGS)
    set(_MULTI_VALUE_ARGS FLAGS SUPPORTED_IF QUERY_IF)

    cmake_parse_arguments(EASCXXFLAGS "${_OPTIONS_ARGS}" "${_ONE_VALUE_ARGS}" "${_MULTI_VALUE_ARGS}" ${ARGN})
    if(NOT EASCXXFLAGS_FLAGS)
        message(FATAL_ERROR "ecm_add_cxx_compiler_flags_if_supported: 'FLAGS' is a required argument.")
    endif()
    # if the user provided conditions, evaluate them now to simplify things later
    if(EASCXXFLAGS_SUPPORTED_IF AND (${EASCXXFLAGS_SUPPORTED_IF}))
        set(EASCXXFLAGS_is_supported ON)
    endif()
    if((EASCXXFLAGS_QUERY_IF AND (${EASCXXFLAGS_QUERY_IF}))
        OR (NOT EASCXXFLAGS_SUPPORTED_IF AND NOT EASCXXFLAGS_QUERY_IF))
        set(EASCXXFLAGS_needs_query ON)
    endif()
    if(EASCXXFLAGS_needs_query)
        # without conditions, or when QUERY_IF is true we'll need to ask the compiler directly.
        # one flag at a time:
        set(flags)
        foreach(flag IN ITEMS ${EASCXXFLAGS_FLAGS})
            # use a standardised and informative cached test variable
            set(HASFLAG "${CMAKE_CXX_COMPILER_ID}++_ACCEPTS${flag}")
            check_cxx_compiler_flag(${flag} ${HASFLAG})
            if(${${HASFLAG}})
                set(flags "${flags} ${flag}")
            endif()
        endforeach()
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} ${flags}" PARENT_SCOPE)
    elseif(EASCXXFLAGS_is_supported)
        # all flags can be appended at once
        string(REPLACE ";" " " FLAGS "${EASCXXFLAGS_FLAGS}")
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} ${FLAGS}" PARENT_SCOPE)
    endif()
endfunction()

function(ECM_ADD_C_COMPILER_FLAGS_IF_SUPPORTED)
    set(_OPTIONS_ARGS)
    set(_ONE_VALUE_ARGS)
    set(_MULTI_VALUE_ARGS FLAGS SUPPORTED_IF QUERY_IF)

    cmake_parse_arguments(EASCFLAGS "${_OPTIONS_ARGS}" "${_ONE_VALUE_ARGS}" "${_MULTI_VALUE_ARGS}" ${ARGN})
    if(NOT EASCFLAGS_FLAGS)
        message(FATAL_ERROR "ecm_add_c_compiler_flags_if_supported: 'FLAGS' is a required argument.")
    endif()
    # if the user provided conditions, evaluate them now to simplify things later
    if(EASCFLAGS_SUPPORTED_IF AND (${EASCFLAGS_SUPPORTED_IF}))
        set(EASCFLAGS_is_supported ON)
    endif()
    if((EASCFLAGS_QUERY_IF AND (${EASCFLAGS_QUERY_IF}))
        OR (NOT EASCFLAGS_SUPPORTED_IF AND NOT EASCFLAGS_QUERY_IF))
        set(EASCFLAGS_needs_query ON)
    endif()
    if(EASCFLAGS_needs_query)
        set(flags)
        foreach(flag IN ITEMS ${EASCFLAGS_FLAGS})
            set(HASFLAG "${CMAKE_C_COMPILER_ID}_ACCEPTS${flag}")
            check_c_compiler_flag(${flag} ${HASFLAG})
            if(${${HASFLAG}})
                set(flags "${flags} ${flag}")
            endif()
        endforeach()
        set(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} ${flags}" PARENT_SCOPE)
    elseif(EASCFLAGS_is_supported)
        string(REPLACE ";" " " FLAGS "${EASCFLAGS_FLAGS}")
        set(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} ${FLAGS}" PARENT_SCOPE)
    endif()
endfunction()

