! WHIZARD 2.8.2 Oct 24 2019
!
! Copyright (C) 1999-2019 by
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!
!     with contributions from
!     cf. main AUTHORS file
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'

module dglap_remnant

  use kinds, only: default, double
  use iso_varying_string, string_t => varying_string
  use numeric_utils
  use diagnostics
  use constants
  use physics_defs
  use pdg_arrays
  use phs_fks, only: isr_kinematics_t
  use fks_regions, only: region_data_t

  use nlo_data

  implicit none
  private

  public :: dglap_remnant_t

  type :: dglap_remnant_t
     type(nlo_settings_t), pointer :: settings => null ()
     type(region_data_t), pointer :: reg_data => null ()
     type(isr_kinematics_t), pointer :: isr_kinematics => null ()
     real(default), dimension(:), allocatable :: sqme_born
     real(default), dimension(:,:), allocatable :: sf_factors
   contains
     procedure :: init => dglap_remnant_init
     procedure :: evaluate => dglap_remnant_evaluate
     procedure :: final => dglap_remnant_final
  end type dglap_remnant_t


contains

  subroutine dglap_remnant_init (dglap, settings, reg_data, isr_kinematics)
    class(dglap_remnant_t), intent(inout) :: dglap
    type(nlo_settings_t), intent(in), target :: settings
    type(region_data_t), intent(in), target :: reg_data
    integer :: n_flv_born
    type(isr_kinematics_t), intent(in), target :: isr_kinematics
    dglap%reg_data => reg_data
    n_flv_born = reg_data%get_n_flv_born ()
    allocate (dglap%sf_factors (reg_data%n_regions, 0:reg_data%n_in))
    dglap%sf_factors = zero
    dglap%settings => settings
    allocate (dglap%sqme_born(n_flv_born))
    dglap%sqme_born = zero
    dglap%isr_kinematics => isr_kinematics
  end subroutine dglap_remnant_init

  subroutine dglap_remnant_evaluate (dglap, alpha_s, separate_alrs, sqme_dglap)
    class(dglap_remnant_t), intent(inout) :: dglap
    real(default), intent(in) :: alpha_s
    logical, intent(in) :: separate_alrs
    real(default), intent(inout), dimension(:) :: sqme_dglap
    integer :: alr, emitter
    real(default) :: sqme_alr
    logical, dimension(:,:,:), allocatable :: evaluated
    real(default) :: sb, fac_scale2
    sb = dglap%isr_kinematics%sqrts_born**2
    fac_scale2 = dglap%isr_kinematics%fac_scale**2
    allocate (evaluated(dglap%reg_data%get_n_flv_born (), dglap%reg_data%get_n_flv_real (), &
         dglap%reg_data%n_in))
    evaluated = .false.
    do alr = 1, dglap%reg_data%n_regions
       sqme_alr = zero
       emitter = dglap%reg_data%regions(alr)%emitter
       if (emitter > dglap%reg_data%n_in) cycle
       associate (i_flv_born => dglap%reg_data%regions(alr)%uborn_index, &
               i_flv_real => dglap%reg_data%regions(alr)%real_index)
          if (emitter == 0) then
             do emitter = 1, 2
                if (evaluated(i_flv_born, i_flv_real, emitter)) cycle
                call evaluate_alr (alr, emitter, i_flv_born, i_flv_real, sqme_alr, evaluated)
             end do
          else if (emitter > 0) then
             if (evaluated(i_flv_born, i_flv_real, emitter)) cycle
             call evaluate_alr (alr, emitter, i_flv_born, i_flv_real, sqme_alr, evaluated)
          end if
       end associate
       if (separate_alrs) then
          sqme_dglap(alr) = sqme_dglap(alr) + alpha_s / twopi * sqme_alr
       else
          sqme_dglap(1) = sqme_dglap(1) + alpha_s / twopi * sqme_alr
       end if
    end do

  contains
    function p_hat_gg (z)
      real(default) :: p_hat_gg
      real(default), intent(in) :: z
      real(default) :: onemz
      onemz = one - z

      p_hat_gg = two * CA * (z + onemz**2 / z + z * onemz**2)
    end function p_hat_gg

    function p_hat_qg (z)
      real(default) :: p_hat_qg
      real(default), intent(in) :: z
      real(default) :: onemz
      onemz = one - z

      p_hat_qg = CF * onemz / z * (one + onemz**2)
    end function p_hat_qg

    function p_hat_gq (z)
      real(default) :: p_hat_gq
      real(default), intent(in) :: z
      real(default) :: onemz
      onemz = one - z

      p_hat_gq = TR * (onemz - two * z * onemz**2)
    end function p_hat_gq

    function p_hat_qq (z)
      real(default) :: p_hat_qq
      real(default), intent(in) :: z
      p_hat_qq = CF * (one + z**2)
    end function p_hat_qq

    function p_derived_gg (z)
      real(default) :: p_derived_gg
      real(default), intent(in) :: z
      p_derived_gg = zero
    end function p_derived_gg

    function p_derived_qg (z)
      real(default) :: p_derived_qg
      real(default), intent(in) :: z
      p_derived_qg = -CF * z
    end function p_derived_qg

    function p_derived_gq (z)
      real(default) :: p_derived_gq
      real(default), intent(in) :: z
      real(default) :: onemz
      onemz = one - z

      p_derived_gq = -two * TR * z * onemz
    end function p_derived_gq

    function p_derived_qq (z)
      real(default) :: p_derived_qq
      real(default), intent(in) :: z
      real(default) :: onemz
      onemz = one - z

      p_derived_qq = -CF * onemz
    end function p_derived_qq

  subroutine evaluate_alr (alr, emitter, i_flv_born, i_flv_real, sqme_alr, evaluated)
    integer, intent(in) :: alr, emitter, i_flv_born, i_flv_real
    real(default), intent(inout) :: sqme_alr
    logical, intent(inout), dimension(:,:,:) :: evaluated
    real(default) :: z, jac
    real(default) :: factor, factor_soft, plus_dist_remnant
    real(default) :: xb, onemz
    real(default) :: sqme_scaled
    integer :: flv_em, flv_rad
    associate (template => dglap%settings%fks_template)
       z = dglap%isr_kinematics%z(emitter)
       flv_rad = dglap%reg_data%regions(alr)%flst_real%flst(dglap%reg_data%n_legs_real)
       flv_em = dglap%reg_data%regions(alr)%flst_real%flst(emitter)
       jac = dglap%isr_kinematics%jacobian(emitter)
       onemz = one - z
       factor = log (sb * template%delta_i / two / z / fac_scale2) / &
            onemz + two * log (onemz) / onemz
       factor_soft = log (sb * template%delta_i / two / fac_scale2) / &
            onemz + two * log (onemz) / onemz
       xb = dglap%isr_kinematics%x(emitter)
       plus_dist_remnant = log ((one - xb) / template%xi_cut) * log (sb * template%delta_i / &
            two / fac_scale2) + (log (one - xb)**2 - log (template%xi_cut)**2)
    end associate
    if (is_massless_vector (flv_em) .and. is_massless_vector (flv_rad)) then
       sqme_scaled = dglap%sqme_born(i_flv_born) * dglap%sf_factors(alr, emitter)
       sqme_alr = sqme_alr + p_hat_gg(z) * factor / z * sqme_scaled * jac &
            - p_hat_gg(one) * factor_soft * dglap%sqme_born(i_flv_born) * jac &
            + p_hat_gg(one) * plus_dist_remnant * dglap%sqme_born(i_flv_born)
    else if (is_fermion (flv_em) .and. is_massless_vector (flv_rad)) then
       sqme_scaled = dglap%sqme_born(i_flv_born) * dglap%sf_factors(alr, emitter)
       sqme_alr = sqme_alr + p_hat_qq(z) * factor / z * sqme_scaled * jac &
            - p_derived_qq(z) / z * sqme_scaled * jac &
            - p_hat_qq(one) * factor_soft * dglap%sqme_born(i_flv_born) * jac &
            + p_hat_qq(one) * plus_dist_remnant * dglap%sqme_born(i_flv_born)
    else if (is_fermion (flv_em) .and. is_fermion (flv_rad)) then
       sqme_alr = sqme_alr + (p_hat_qg(z) * factor - p_derived_qg(z)) / z * jac * &
            dglap%sqme_born(i_flv_born) * dglap%sf_factors(alr, emitter)
    else if (is_massless_vector (flv_em) .and. is_fermion (flv_rad)) then
       sqme_scaled = dglap%sqme_born(i_flv_born) * dglap%sf_factors(alr, emitter)
       sqme_alr = sqme_alr + (p_hat_gq(z) * factor - p_derived_gq(z)) / z * sqme_scaled * jac
    else
       sqme_alr = sqme_alr + zero
    end if
    evaluated(i_flv_born, i_flv_real, emitter) = .true.
  end subroutine evaluate_alr
  end subroutine dglap_remnant_evaluate

  subroutine dglap_remnant_final (dglap)
    class(dglap_remnant_t), intent(inout) :: dglap
    if (associated (dglap%isr_kinematics)) nullify (dglap%isr_kinematics)
    if (associated (dglap%reg_data)) nullify (dglap%reg_data)
    if (associated (dglap%settings)) nullify (dglap%settings)
    if (allocated (dglap%sqme_born)) deallocate (dglap%sqme_born)
    if (allocated (dglap%sf_factors)) deallocate (dglap%sf_factors)
  end subroutine dglap_remnant_final


end module dglap_remnant

