! WHIZARD 2.2.8 Nov 22 2015
! 
! Copyright (C) 1999-2015 by 
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!     
!     with contributions from
!     Fabian Bach <fabian.bach@t-online.de>
!     Bijan Chokoufe <bijan.chokoufe@desy.de>
!     Christian Speckner <cnspeckn@googlemail.com> 
!     Soyoung Shim <soyoung.shim@desy.de>
!     Florian Staub <florian.staub@cern.ch>  
!     Christian Weiss <christian.weiss@desy.de>
!     and Hans-Werner Boschmann, Felix Braam, 
!     Sebastian Schmidt, So-young Shim, Daniel Wiesler 
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by 
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'

module simulations

  use kinds, only: default
  use iso_varying_string, string_t => varying_string
  use io_units
  use format_utils, only: write_separator
  use format_defs, only: FMT_19
  use unit_tests, only: vanishes
  use diagnostics
  use sm_qcd
  use md5
  use variables
  use eval_trees
  use model_data
  use flavors
  use particles
  use state_matrices, only: FM_IGNORE_HELICITY
  use beam_structures, only: beam_structure_t
  use beams
  use rng_base
  use selectors
  use process_libraries, only: process_library_t
  use prc_core
  use processes
  use event_base
  use events
  use event_transforms
  use shower
  use eio_data
  use eio_base
  use rt_data

  use dispatch, only: dispatch_qcd
  use dispatch, only: dispatch_rng_factory
  use dispatch, only: dispatch_core_update, dispatch_core_restore
  use dispatch, only: dispatch_evt_decay
  use dispatch, only: dispatch_evt_shower
  use dispatch, only: dispatch_evt_hadrons

  use integrations
  use event_streams

  use evt_nlo
  use dispatch, only: dispatch_evt_nlo

  implicit none
  private

  public :: simulation_t
  public :: pacify

  type :: counter_t
     integer :: total = 0
     integer :: generated = 0
     integer :: read = 0
     integer :: positive = 0
     integer :: negative = 0
     integer :: zero = 0
     integer :: excess = 0
     real(default) :: max_excess = 0
     real(default) :: sum_excess = 0
   contains
     procedure :: write => counter_write
     procedure :: show_excess => counter_show_excess
     procedure :: record => counter_record
  end type counter_t
  
  type :: mci_set_t
     private
     integer :: n_components = 0
     integer, dimension(:), allocatable :: i_component
     type(string_t), dimension(:), allocatable :: component_id
     logical :: has_integral = .false.
     real(default) :: integral = 0
     real(default) :: error = 0
     real(default) :: weight_mci = 0
     type(counter_t) :: counter
   contains
     procedure :: write => mci_set_write
     procedure :: init => mci_set_init
  end type mci_set_t
     
  type :: core_safe_t
     class(prc_core_t), allocatable :: core
  end type core_safe_t
  
  type, extends (event_t) :: entry_t
     private
     type(string_t) :: process_id
     type(string_t) :: library
     type(string_t) :: run_id
     logical :: has_integral = .false.
     real(default) :: integral = 0
     real(default) :: error = 0
     real(default) :: process_weight = 0
     logical :: valid = .false.
     type(counter_t) :: counter
     integer :: n_in = 0
     integer :: n_mci = 0
     type(mci_set_t), dimension(:), allocatable :: mci_set
     type(selector_t) :: mci_selector
     type(core_safe_t), dimension(:), allocatable :: core_safe
     class(model_data_t), pointer :: model => null ()
     type(qcd_t) :: qcd
     type(entry_t), pointer :: first => null ()
     type(entry_t), pointer :: next => null ()
     class(evt_t), pointer :: evt_powheg => null ()
   contains
     procedure :: write_config => entry_write_config
     procedure :: final => entry_final
     procedure :: copy_entry => entry_copy_entry
     procedure :: init => entry_init
     procedure :: set_active_real_component => entry_set_active_real_component
     procedure, private :: import_process_characteristics &
          => entry_import_process_characteristics
     procedure, private :: import_process_def_characteristics &
          => entry_import_process_def_characteristics
     procedure, private :: import_process_results &
          => entry_import_process_results
     procedure, private :: prepare_expressions &
          => entry_prepare_expressions
     procedure :: setup_additional_entries => entry_setup_additional_entries
     procedure :: get_first => entry_get_first
     procedure :: get_next => entry_get_next
     procedure :: count_nlo_entries => entry_count_nlo_entries
     procedure :: reset_nlo_counter => entry_reset_nlo_counter 
     procedure :: determine_if_powheg_matching => entry_determine_if_powheg_matching
     procedure, private :: setup_event_transforms &
          => entry_setup_event_transforms
     procedure :: init_mci_selector => entry_init_mci_selector
     procedure :: select_mci => entry_select_mci
     procedure :: record => entry_record
     procedure :: update_process => entry_update_process
     procedure :: restore_process => entry_restore_process
     procedure :: connect_qcd => entry_connect_qcd
  end type entry_t

  type, extends (entry_t) :: alt_entry_t
   contains
     procedure :: init_alt => alt_entry_init
     procedure :: fill_particle_set => entry_fill_particle_set
  end type alt_entry_t
  
  type :: simulation_t
     private
     type(rt_data_t), pointer :: local => null ()
     type(string_t) :: sample_id
     logical :: unweighted = .true.
     logical :: negative_weights = .false.
     logical :: respect_selection = .true.
     integer :: norm_mode = NORM_UNDEFINED
     logical :: update_sqme = .false.
     logical :: update_weight = .false.
     logical :: update_event = .false.
     logical :: recover_beams = .false.
     logical :: pacify = .false.
     integer :: n_max_tries = 10000
     integer :: n_prc = 0
     integer :: n_alt = 0
     logical :: has_integral = .false.
     logical :: valid = .false.
     real(default) :: integral = 0
     real(default) :: error = 0
     integer :: version = 1
     character(32) :: md5sum_prc = ""
     character(32) :: md5sum_cfg = ""
     character(32), dimension(:), allocatable :: md5sum_alt
     type(entry_t), dimension(:), allocatable :: entry
     type(alt_entry_t), dimension(:,:), allocatable :: alt_entry
     type(selector_t) :: process_selector
     integer :: n_evt_requested = 0
     integer :: split_n_evt = 0
     integer :: split_n_kbytes = 0
     integer :: split_index = 0
     type(counter_t) :: counter
     class(rng_t), allocatable :: rng
     integer :: i_prc = 0
     integer :: i_mci = 0
     real(default) :: weight = 0
     real(default) :: excess = 0
   contains
     procedure :: write => simulation_write
     generic :: write_event => write_event_unit
     procedure :: write_event_unit => simulation_write_event_unit
     procedure :: write_alt_event => simulation_write_alt_event
     procedure :: final => simulation_final
     procedure :: init => simulation_init
     procedure :: compute_n_events => simulation_compute_n_events
     procedure :: show_efficiency => simulation_show_efficiency
     procedure :: get_n_nlo_entries => simulation_get_n_nlo_entries
     procedure :: compute_md5sum => simulation_compute_md5sum
     procedure :: init_process_selector => simulation_init_process_selector
     procedure :: select_prc => simulation_select_prc
     procedure :: select_mci => simulation_select_mci
     procedure :: generate => simulation_generate
     procedure :: calculate_alt_entries => simulation_calculate_alt_entries
     procedure :: rescan => simulation_rescan
     procedure :: update_processes => simulation_update_processes
     procedure :: restore_processes => simulation_restore_processes
     generic :: write_event => write_event_eio
     procedure :: write_event_eio => simulation_write_event_eio
     generic :: read_event => read_event_eio
     procedure :: read_event_eio => simulation_read_event_eio
     generic :: write_event => write_event_es_array
     procedure :: write_event_es_array => simulation_write_event_es_array
     generic :: read_event => read_event_es_array
     procedure :: read_event_es_array => simulation_read_event_es_array
     procedure :: recalculate => simulation_recalculate
     procedure :: get_md5sum_prc => simulation_get_md5sum_prc
     procedure :: get_md5sum_cfg => simulation_get_md5sum_cfg
     procedure :: get_md5sum_alt => simulation_get_md5sum_alt
     procedure :: get_data => simulation_get_data
     procedure :: get_default_sample_name => simulation_get_default_sample_name
     procedure :: is_valid => simulation_is_valid
     procedure :: evaluate_expressions => simulation_evaluate_expressions
  end type simulation_t
  

  interface pacify
     module procedure pacify_simulation
  end interface

contains

  subroutine counter_write (object, unit)
    class(counter_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit)
1   format (3x,A,I0)
2   format (5x,A,I0)
3   format (5x,A,ES19.12)
    write (u, 1)  "Events total      = ", object%total
    write (u, 2)  "generated       = ", object%generated
    write (u, 2)  "read            = ", object%read
    write (u, 2)  "positive weight = ", object%positive
    write (u, 2)  "negative weight = ", object%negative
    write (u, 2)  "zero weight     = ", object%zero
    write (u, 2)  "excess weight   = ", object%excess
    if (object%excess /= 0) then
       write (u, 3)  "max excess      = ", object%max_excess
       write (u, 3)  "avg excess      = ", object%sum_excess / object%total
    end if
  end subroutine counter_write

  subroutine counter_show_excess (counter)
    class(counter_t), intent(in) :: counter
    if (counter%excess > 0) then
       write (msg_buffer, "(A,1x,I0,1x,A,1x,'(',F7.3,' %)')") &
            "Encountered events with excess weight:", counter%excess, &
            "events", 100 * counter%excess / real (counter%total)
       call msg_warning ()
       write (msg_buffer, "(A,ES10.3)") &
            "Maximum excess weight =", counter%max_excess
       call msg_message ()
       write (msg_buffer, "(A,ES10.3)") &
            "Average excess weight =", counter%sum_excess / counter%total
       call msg_message ()
    end if
  end subroutine counter_show_excess
    
  subroutine counter_record (counter, weight, excess, from_file)
    class(counter_t), intent(inout) :: counter
    real(default), intent(in), optional :: weight, excess
    logical, intent(in), optional :: from_file
    counter%total = counter%total + 1
    if (present (from_file)) then
       if (from_file) then
          counter%read = counter%read + 1
       else
          counter%generated = counter%generated + 1
       end if
    else
       counter%generated = counter%generated + 1
    end if
    if (present (weight)) then
       if (weight > 0) then
          counter%positive = counter%positive + 1
       else if (weight < 0) then
          counter%negative = counter%negative + 1
       else
          counter%zero = counter%zero + 1
       end if
    else
       counter%positive = counter%positive + 1
    end if
    if (present (excess)) then
       if (excess > 0) then
          counter%excess = counter%excess + 1
          counter%max_excess = max (counter%max_excess, excess)
          counter%sum_excess = counter%sum_excess + excess
       end if
    end if
  end subroutine counter_record
    
  subroutine mci_set_write (object, unit)
    class(mci_set_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u, i
    u = given_output_unit (unit)
    write (u, "(3x,A)")  "Components:"
    do i = 1, object%n_components
       write (u, "(5x,I0,A,A,A)")  object%i_component(i), &
            ": '", char (object%component_id(i)), "'"
    end do
    if (object%has_integral) then
       write (u, "(3x,A," // FMT_19 // ")")  "Integral  = ", object%integral
       write (u, "(3x,A," // FMT_19 // ")")  "Error     = ", object%error
       write (u, "(3x,A,F13.10)")  "Weight    =", object%weight_mci
    else
       write (u, "(3x,A)")  "Integral  = [undefined]"
    end if
    call object%counter%write (u)
  end subroutine mci_set_write
  
  subroutine mci_set_init (object, i_mci, process)
    class(mci_set_t), intent(out) :: object
    integer, intent(in) :: i_mci
    type(process_t), intent(in), target :: process
    integer :: i
    call process%get_i_component (i_mci, object%i_component)
    object%n_components = size (object%i_component)
    allocate (object%component_id (object%n_components))
    do i = 1, size (object%component_id)
       object%component_id(i) = &
            process%get_component_id (object%i_component(i))
    end do
    if (process%has_integral (i_mci)) then
       object%integral = process%get_integral (i_mci)
       object%error = process%get_error (i_mci)
       object%has_integral = .true.
    end if
  end subroutine mci_set_init
    
  subroutine prepare_process &
       (process, process_id, use_process, integrate, local, global)
    type(process_t), pointer, intent(out) :: process
    type(string_t), intent(in) :: process_id
    logical, intent(in) :: use_process, integrate
    type(rt_data_t), intent(inout), target :: local
    type(rt_data_t), intent(inout), optional, target :: global
    if (present (global)) then
       process => global%process_stack%get_process_ptr (process_id)
    else
       process => local%process_stack%get_process_ptr (process_id)
    end if
    if (use_process .and. .not. associated (process)) then
       if (integrate) then
          call msg_message ("Simulate: process '" &
               // char (process_id) // "' needs integration")
       else
          call msg_message ("Simulate: process '" &
               // char (process_id) // "' needs initialization")
       end if
       if (present (global)) then
          call integrate_process (process_id, local, global, &
            init_only = .not. integrate)
       else
          call integrate_process (process_id, local, local_stack=.true., &
            init_only = .not. integrate)
       end if
       if (signal_is_pending ())  return
       process => global%process_stack%get_process_ptr (process_id)
       if (associated (process)) then
          if (integrate) then
             call msg_message ("Simulate: integration done")
             call global%process_stack%fill_result_vars (process_id)
          else
             call msg_message ("Simulate: process initialization done")
          end if
       else
          call msg_fatal ("Simulate: process '" &
               // char (process_id) // "' could not be initialized: aborting")
       end if
    else if (.not. associated (process)) then
       call msg_message &
            ("Simulate: process '" &
               // char (process_id) // "': enabled for rescan only")
    end if
  end subroutine prepare_process
    
  subroutine entry_write_config (object, unit)
    class(entry_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u, i
    u = given_output_unit (unit)
    write (u, "(3x,A,A,A)")  "Process   = '", char (object%process_id), "'"
    write (u, "(3x,A,A,A)")  "Library   = '", char (object%library), "'"
    write (u, "(3x,A,A,A)")  "Run       = '", char (object%run_id), "'"
    write (u, "(3x,A,L1)")   "is valid  = ", object%valid
    if (object%has_integral) then
       write (u, "(3x,A," // FMT_19 // ")")  "Integral  = ", object%integral
       write (u, "(3x,A," // FMT_19 // ")")  "Error     = ", object%error
       write (u, "(3x,A,F13.10)")  "Weight    =", object%process_weight
    else
       write (u, "(3x,A)")  "Integral  = [undefined]"
    end if
    write (u, "(3x,A,I0)")   "MCI sets  = ", object%n_mci
    call object%counter%write (u)
    do i = 1, size (object%mci_set)
       write (u, "(A)")
       write (u, "(1x,A,I0,A)")  "MCI set #", i, ":"
       call object%mci_set(i)%write (u)
    end do
    if (allocated (object%core_safe)) then
       do i = 1, size (object%core_safe)
          write (u, "(1x,A,I0,A)")  "Saved process-component core #", i, ":"
          call object%core_safe(i)%core%write (u)
       end do
    end if
  end subroutine entry_write_config
  
  subroutine entry_final (object)
    class(entry_t), intent(inout) :: object
    integer :: i
    if (associated (object%instance)) then
       do i = 1, object%n_mci
          call object%instance%final_simulation (i)
       end do
       call object%instance%final ()
       deallocate (object%instance)
    end if
    call object%event_t%final ()
  end subroutine entry_final
  
  subroutine entry_copy_entry (entry1, entry2)
    class(entry_t), intent(in) :: entry1
    type(entry_t), intent(inout) :: entry2
    entry2%event_t = entry1%event_t
    entry2%process_id = entry1%process_id
    entry2%library = entry1%library
    entry2%run_id = entry1%run_id 
    entry2%has_integral = entry1%has_integral
    entry2%integral = entry1%integral
    entry2%error = entry1%error
    entry2%process_weight = entry1%process_weight
    entry2%valid = entry1%valid
    entry2%counter = entry1%counter
    entry2%n_in = entry1%n_in
    entry2%n_mci = entry1%n_mci
    if (allocated (entry1%mci_set)) then
       allocate (entry2%mci_set (size (entry1%mci_set)))
       entry2%mci_set = entry1%mci_set
    end if
    entry2%mci_selector = entry1%mci_selector
    if (allocated (entry1%core_safe)) then
       allocate (entry2%core_safe (size (entry1%core_safe)))
       entry2%core_safe = entry1%core_safe
    end if
    entry2%model => entry1%model
    entry2%qcd = entry1%qcd
!    entry2%first => entry1%first
  end subroutine entry_copy_entry

  subroutine entry_init &
       (entry, process_id, &
       use_process, integrate, generate, update_sqme, &
       local, global, n_alt)
    class(entry_t), intent(inout), target :: entry
    type(string_t), intent(in) :: process_id
    logical, intent(in) :: use_process, integrate, generate, update_sqme
    type(rt_data_t), intent(inout), target :: local
    type(rt_data_t), intent(inout), optional, target :: global
    integer, intent(in), optional :: n_alt
    type(process_t), pointer :: process, master_process
    type(process_instance_t), pointer :: process_instance
    integer :: i
    logical :: combined_integration

    call prepare_process &
         (master_process, process_id, use_process, integrate, local, global)
    if (signal_is_pending ())  return

    if (associated (master_process)) then
       if (.not. master_process%has_matrix_element ()) then
          entry%has_integral = .true.
          entry%process_id = process_id
          entry%valid = .false.          
          return
       end if
    else
       call entry%basic_init (local%var_list)
       entry%has_integral = .false.
       entry%process_id = process_id
       call entry%import_process_def_characteristics (local%prclib, process_id)
       entry%valid = .true.
       return
    end if
    
    call entry%basic_init (local%var_list, n_alt)

    entry%process_id = process_id
    if (generate .or. integrate) then
       entry%run_id = master_process%get_run_id ()
       process => master_process
    else
       call local%set_log (var_str ("?rebuild_phase_space"), &
            .false., is_known = .true.)
       call local%set_log (var_str ("?check_phs_file"), &
            .false., is_known = .true.)
       call local%set_log (var_str ("?rebuild_grids"), &
            .false., is_known = .true.)
       entry%run_id = &
            local%var_list%get_sval (var_str ("$run_id"))
       if (update_sqme) then
          call prepare_local_process (process, process_id, local)
       else
          process => master_process
       end if
    end if

    call entry%import_process_characteristics (process)

    allocate (entry%mci_set (entry%n_mci))
    do i = 1, size (entry%mci_set)
       call entry%mci_set(i)%init (i, master_process)
    end do
    call entry%set_nlo_event (local%get_lval (var_str ("?nlo_fixed_order")))
    if (entry%is_nlo_event()) then
       call entry%init_sample_formats ()
       call entry%check_supported_sample_formats (local%sample_fmt(1))
    end if

    call entry%import_process_results (master_process)
    call entry%prepare_expressions (local)

    combined_integration = local%get_lval (var_str ("?combined_nlo_integration"))
    call prepare_process_instance (process_instance, process, local%model, &
       combined_integration = combined_integration, local = local)
    if (generate) then
       do i = 1, entry%n_mci
          call process%prepare_simulation (i)
          call process_instance%init_simulation (i, entry%config%safety_factor)
       end do
    end if
    call entry%setup_event_transforms (process, local)
    call dispatch_qcd (entry%qcd, local)

    call entry%connect_qcd ()

    if (entry%is_nlo_event ()) then
       select type (pcm => process_instance%pcm)
       class is (pcm_instance_nlo_t)
          call pcm%controller%set_fixed_order_event_mode ()
       end select
    end if

    if (present (global)) then
       call entry%connect (process_instance, local%model, global%process_stack)
    else
       call entry%connect (process_instance, local%model, local%process_stack)
    end if
    call entry%setup_expressions ()

    entry%model => process%get_model_ptr ()
    entry%valid = .true.
    
  end subroutine entry_init
    
  subroutine entry_set_active_real_component (entry, i_mci)
    class(entry_t), intent(inout) :: entry
    integer, intent(in) :: i_mci
    class(evt_t), pointer :: current_transform
    integer :: i
    select type (pcm => entry%instance%pcm)
    class is (pcm_instance_nlo_t)
       pcm%active_real_component = &
            entry%instance%process%get_associated_real_component (i_mci)
       i = pcm%active_real_component
       if (associated (entry%evt_powheg)) then
          select type (evt => entry%evt_powheg)
          type is (evt_shower_t)
             if (entry%instance%component(i)%get_component_type() &
                  == COMP_REAL_FIN) then
                call evt%disable_powheg_matching ()
             else
                call evt%enable_powheg_matching ()
             end if
          class default
             call msg_fatal ("powheg-evt should be evt_shower_t!")
          end select
       end if
    end select
  end subroutine entry_set_active_real_component

  subroutine prepare_local_process (process, process_id, local)
    type(process_t), pointer, intent(inout) :: process
    type(string_t), intent(in) :: process_id
    type(rt_data_t), intent(inout), target :: local
    type(integration_t) :: intg
    call intg%create_process (process_id)
    call intg%init_process (local)
    call intg%setup_process (local, verbose=.false.)
    process => intg%get_process_ptr ()
  end subroutine prepare_local_process
  
  subroutine prepare_process_instance (process_instance, process, model, combined_integration, local)
    type(process_instance_t), pointer, intent(inout) :: process_instance
    type(process_t), intent(inout), target :: process
    class(model_data_t), intent(in), optional :: model
    logical, intent(in), optional :: combined_integration
    type(rt_data_t), intent(in), optional, target :: local
    allocate (process_instance)
    if (process%is_nlo_calculation ()) then
       call process_instance%init (process, combined_integration = combined_integration)
       if (process_instance%has_blha_component () .and. present (local)) then
          call process_instance%create_blha_interface (local%beam_structure)
          call process_instance%load_blha_libraries (local%os_data)
       end if
       call setup_nlo_component_cores (process)
    else
       call process_instance%init (process)
    end if
    call process_instance%setup_event_data (model)
  end subroutine prepare_process_instance 

  subroutine entry_import_process_characteristics (entry, process)
    class(entry_t), intent(inout) :: entry
    type(process_t), intent(in), target :: process
    entry%library = process%get_library_name ()
    entry%n_in = process%get_n_in ()
    entry%n_mci = process%get_n_mci ()
  end subroutine entry_import_process_characteristics

  subroutine entry_import_process_def_characteristics (entry, prclib, id)
    class(entry_t), intent(inout) :: entry
    type(process_library_t), intent(in), target :: prclib
    type(string_t), intent(in) :: id
    entry%library = prclib%get_name ()
    entry%n_in = prclib%get_n_in (id)
  end subroutine entry_import_process_def_characteristics

  subroutine entry_import_process_results (entry, process)
    class(entry_t), intent(inout) :: entry
    type(process_t), intent(in), target :: process
    if (process%has_integral ()) then
       entry%integral = process%get_integral ()
       entry%error = process%get_error ()
       call entry%set_sigma (entry%integral)
       entry%has_integral = .true.
    end if
  end subroutine entry_import_process_results

  subroutine entry_prepare_expressions (entry, local)
    class(entry_t), intent(inout) :: entry
    type(rt_data_t), intent(in), target :: local
    type(eval_tree_factory_t) :: expr_factory
    call expr_factory%init (local%pn%selection_lexpr)
    call entry%set_selection (expr_factory)
    call expr_factory%init (local%pn%reweight_expr)
    call entry%set_reweight (expr_factory)
    call expr_factory%init (local%pn%analysis_lexpr)
    call entry%set_analysis (expr_factory)
  end subroutine entry_prepare_expressions

  subroutine entry_setup_additional_entries (entry)
    class(entry_t), intent(inout), target :: entry
    type(entry_t), pointer :: current_entry
    integer :: i, n_alr
    integer, dimension(:), allocatable :: emitters
    type(evt_nlo_t), pointer :: evt
    evt => null ()
    select type (pcm => entry%instance%pcm)
    class is (pcm_instance_nlo_t)
       n_alr = pcm%controller%reg_data%n_regions 
       emitters = pcm%controller%reg_data%emitters 
    end select
    select type (entry)
    type is (entry_t)
       current_entry => entry
       current_entry%first => entry
       evt => get_nlo_evt_ptr (current_entry)
       allocate (evt%emitters (n_alr))
       allocate (evt%particle_set_radiated (n_alr+1))
       select type (pcm => entry%instance%pcm)
       class is (pcm_instance_nlo_t)
          evt%emitters = pcm%controller%reg_data%get_emitter_list ()
       end select
       evt%qcd => entry%qcd
       do i = 1, n_alr
          allocate (current_entry%next)
          current_entry%next%first => current_entry%first
          current_entry => current_entry%next
          call entry%copy_entry (current_entry)
          current_entry%i_event = i
       end do
    end select
  contains
    function get_nlo_evt_ptr (entry) result (evt)
      type(entry_t), intent(in), target :: entry
      type(evt_nlo_t), pointer :: evt
      class(evt_t), pointer :: current_evt
      evt => null ()
      current_evt => entry%transform_first
      do
         select type (current_evt)
         type is (evt_nlo_t)
            evt => current_evt 
            exit
         end select
         if (associated (current_evt%next)) then 
            current_evt => current_evt%next
         else
            call msg_fatal ("evt_nlo not in list of event transforms")
         end if
      end do
    end function get_nlo_evt_ptr
  end subroutine entry_setup_additional_entries

  function entry_get_first (entry) result (entry_out)
    class(entry_t), intent(in), target :: entry
    type(entry_t), pointer :: entry_out
    entry_out => null ()
    select type (entry)
    type is (entry_t)
       if (entry%is_nlo_event()) then
          entry_out => entry%first
       else
          entry_out => entry
       end if
    end select
  end function entry_get_first

  function entry_get_next (entry) result (next_entry)
     class(entry_t), intent(in) :: entry
     type(entry_t), pointer :: next_entry
     next_entry => null ()
     if (associated (entry%next)) then
        next_entry => entry%next
     else
        call msg_fatal ("Get next entry: No next entry")
     end if
  end function entry_get_next 

  function entry_count_nlo_entries (entry) result (n)
    class(entry_t), intent(in), target :: entry
    integer :: n
    type(entry_t), pointer :: current_entry
    n = 1
    if (.not. associated (entry%next)) then
       return
    else
       current_entry => entry%next
       do
          n = n+1
          if (.not. associated (current_entry%next)) exit
          current_entry => current_entry%next
       end do
    end if
  end function entry_count_nlo_entries

  subroutine entry_reset_nlo_counter (entry)
    class(entry_t), intent(inout) :: entry
    class(evt_t), pointer :: evt
    evt => entry%transform_first
    do 
       select type (evt)
       type is (evt_nlo_t)
          evt%i_evaluation = 0
          exit
       end select
       if (associated (evt%next)) evt => evt%next
   end do
  end subroutine entry_reset_nlo_counter

  subroutine entry_determine_if_powheg_matching (entry)
     class(entry_t), intent(inout) :: entry
     class(evt_t), pointer :: current_transform
     if (associated (entry%transform_first)) then
        current_transform => entry%transform_first
        do
           select type (current_transform)
           type is (evt_shower_t)
              if (current_transform%contains_powheg_matching ()) &
                  entry%evt_powheg => current_transform
              exit
           end select
           if (associated (current_transform%next)) then
              current_transform => current_transform%next
           else
              exit
           end if
        end do
     end if
  end subroutine entry_determine_if_powheg_matching
  
  subroutine entry_setup_event_transforms (entry, process, local)
    class(entry_t), intent(inout) :: entry
    type(process_t), intent(inout), target :: process
    type(rt_data_t), intent(in), target :: local
    class(evt_t), pointer :: evt
    logical :: enable_fixed_order, enable_shower
    if (process%contains_unstable (local%model)) then
       call dispatch_evt_decay (evt, local)
       if (associated (evt))  call entry%import_transform (evt)
    end if
    enable_fixed_order = local%get_lval (var_str ("?nlo_fixed_order"))
    if (enable_fixed_order) then
       if (local%get_lval (var_str ("?unweighted"))) &
          call msg_fatal ("NLO Fixed Order events have to be generated with &
                          &?unweighted = false")
       call dispatch_evt_nlo (evt)
       call entry%import_transform (evt)
    end if
    enable_shower = local%get_lval (var_str ("?allow_shower")) .and. &
            (local%get_lval (var_str ("?ps_isr_active")) &
            .or. local%get_lval (var_str ("?ps_fsr_active")) &
            .or. local%get_lval (var_str ("?muli_active")) &
            .or. local%get_lval (var_str ("?mlm_matching")) &
            .or. local%get_lval (var_str ("?ckkw_matching")) &
            .or. local%get_lval (var_str ("?powheg_matching")))
    if (enable_shower) then
       call dispatch_evt_shower (evt, local, process)
       call entry%import_transform (evt)
    end if
    if (local%get_lval (var_str ("?hadronization_active"))) then
       call dispatch_evt_hadrons (evt, local, process)
       call entry%import_transform (evt)
    end if
  end subroutine entry_setup_event_transforms

  subroutine entry_init_mci_selector (entry, negative_weights)
    class(entry_t), intent(inout), target :: entry
    logical, intent(in), optional :: negative_weights
    type(entry_t), pointer :: current_entry
    integer :: i, j
    if (entry%has_integral) then
       select type (entry)
       type is (entry_t)
          current_entry => entry
          do j = 1, current_entry%count_nlo_entries ()
             if (j > 1) current_entry => current_entry%get_next ()
             call current_entry%mci_selector%init &
                  (current_entry%mci_set%integral, negative_weights)
             do i = 1, current_entry%n_mci
                current_entry%mci_set(i)%weight_mci = &
                   current_entry%mci_selector%get_weight (i)
             end do
          end do
       end select
    end if
  end subroutine entry_init_mci_selector

  function entry_select_mci (entry) result (i_mci)
    class(entry_t), intent(inout) :: entry
    integer :: i_mci
    call entry%mci_selector%generate (entry%rng, i_mci)
  end function entry_select_mci
  
  subroutine entry_record (entry, i_mci, from_file)
    class(entry_t), intent(inout) :: entry
    integer, intent(in) :: i_mci
    logical, intent(in), optional :: from_file
    real(default) :: weight, excess
    weight = entry%get_weight_prc ()
    excess = entry%get_excess_prc ()
    call entry%counter%record (weight, excess, from_file)
    if (i_mci > 0) then
       call entry%mci_set(i_mci)%counter%record (weight, excess)
    end if
  end subroutine entry_record
    
  subroutine entry_update_process (entry, model, qcd, helicity_selection)
    class(entry_t), intent(inout) :: entry
    class(model_data_t), intent(in), optional, target :: model
    type(qcd_t), intent(in), optional :: qcd
    type(helicity_selection_t), intent(in), optional :: helicity_selection
    type(process_t), pointer :: process
    class(prc_core_t), allocatable :: core
    integer :: i, n_components
    class(model_data_t), pointer :: model_local
    type(qcd_t) :: qcd_local
    if (present (model)) then
       model_local => model
    else
       model_local => entry%model
    end if
    if (present (qcd)) then
       qcd_local = qcd
    else
       qcd_local = entry%qcd
    end if
    process => entry%get_process_ptr ()
    n_components = process%get_n_components ()
    allocate (entry%core_safe (n_components))
    do i = 1, n_components
       if (process%has_matrix_element (i)) then
          call process%extract_component_core (i, core)
          call dispatch_core_update (core, &
               model_local, helicity_selection, qcd_local, &
               entry%core_safe(i)%core)
          call process%restore_component_core (i, core)
       end if
    end do
  end subroutine entry_update_process
  
  subroutine entry_restore_process (entry)
    class(entry_t), intent(inout) :: entry
    type(process_t), pointer :: process
    class(prc_core_t), allocatable :: core
    integer :: i, n_components
    process => entry%get_process_ptr ()
    n_components = process%get_n_components ()
    do i = 1, n_components
       if (process%has_matrix_element (i)) then
          call process%extract_component_core (i, core)
          call dispatch_core_restore (core, entry%core_safe(i)%core)
          call process%restore_component_core (i, core)
       end if
    end do
    deallocate (entry%core_safe)
  end subroutine entry_restore_process
  
  subroutine entry_connect_qcd (entry)
    class(entry_t), intent(inout), target :: entry
    class(evt_t), pointer :: evt
    evt => entry%transform_first
    do while (associated (evt))
       select type (evt)
       type is (evt_shower_t)
          evt%qcd => entry%qcd
          if (allocated (evt%matching)) then
             evt%matching%qcd => entry%qcd
          end if
       end select
       evt => evt%next
    end do
  end subroutine entry_connect_qcd

  subroutine alt_entry_init (entry, process_id, master_process, local)
    class(alt_entry_t), intent(inout), target :: entry
    type(string_t), intent(in) :: process_id
    type(process_t), intent(in), target :: master_process
    type(rt_data_t), intent(inout), target :: local
    type(process_t), pointer :: process
    type(process_instance_t), pointer :: process_instance
    type(string_t) :: run_id
    integer :: i

    call msg_message ("Simulate: initializing alternate process setup ...")

    run_id = &
         local%var_list%get_sval (var_str ("$run_id"))
    call local%set_log (var_str ("?rebuild_phase_space"), &
         .false., is_known = .true.)
    call local%set_log (var_str ("?check_phs_file"), &
         .false., is_known = .true.)
    call local%set_log (var_str ("?rebuild_grids"), &
         .false., is_known = .true.)
    
    call entry%basic_init (local%var_list)
    
    call prepare_local_process (process, process_id, local)
    entry%process_id = process_id
    entry%run_id = run_id

    call entry%import_process_characteristics (process)
    
    allocate (entry%mci_set (entry%n_mci))
    do i = 1, size (entry%mci_set)
       call entry%mci_set(i)%init (i, master_process)
    end do

    call entry%import_process_results (master_process)
    call entry%prepare_expressions (local)

    call prepare_process_instance (process_instance, process, local%model)
    call entry%setup_event_transforms (process, local)

    call entry%connect (process_instance, local%model, local%process_stack)
    call entry%setup_expressions ()

    entry%model => process%get_model_ptr ()

    call msg_message ("...  alternate process setup complete.")

  end subroutine alt_entry_init

  subroutine entry_fill_particle_set (alt_entry, entry)
    class(alt_entry_t), intent(inout) :: alt_entry
    class(entry_t), intent(in), target :: entry
    type(particle_set_t) :: pset
    call entry%get_hard_particle_set (pset)
    call alt_entry%set_hard_particle_set (pset)
    call pset%final ()
  end subroutine entry_fill_particle_set
    
  subroutine simulation_write (object, unit)
    class(simulation_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u, i
    u = given_output_unit (unit)
    call write_separator (u, 2)
    write (u, "(1x,A,A,A)")  "Event sample: '", char (object%sample_id), "'"
    write (u, "(3x,A,I0)")  "Processes    = ", object%n_prc
    if (object%n_alt > 0) then
       write (u, "(3x,A,I0)")  "Alt.wgts     = ", object%n_alt
    end if
    write (u, "(3x,A,L1)")  "Unweighted   = ", object%unweighted
    write (u, "(3x,A,A)")   "Event norm   = ", &
         char (event_normalization_string (object%norm_mode))
    write (u, "(3x,A,L1)")  "Neg. weights = ", object%negative_weights
    write (u, "(3x,A,L1)")  "Respect sel. = ", object%respect_selection
    write (u, "(3x,A,L1)")  "Update sqme  = ", object%update_sqme
    write (u, "(3x,A,L1)")  "Update wgt   = ", object%update_weight
    write (u, "(3x,A,L1)")  "Update event = ", object%update_event
    write (u, "(3x,A,L1)")  "Recov. beams = ", object%recover_beams
    write (u, "(3x,A,L1)")  "Pacify       = ", object%pacify
    write (u, "(3x,A,I0)")  "Max. tries   = ", object%n_max_tries
    if (object%has_integral) then
       write (u, "(3x,A," // FMT_19 // ")")  "Integral     = ", object%integral
       write (u, "(3x,A," // FMT_19 // ")")  "Error        = ", object%error
    else
       write (u, "(3x,A)")  "Integral     = [undefined]"
    end if
    write (u, "(3x,A,L1)")  "Sim. valid   = ", object%valid
    write (u, "(3x,A,I0)")  "Ev.file ver. = ", object%version
    if (object%md5sum_prc /= "") then
       write (u, "(3x,A,A,A)")  "MD5 sum (proc)   = '", object%md5sum_prc, "'"
    end if
    if (object%md5sum_cfg /= "") then
       write (u, "(3x,A,A,A)")  "MD5 sum (config) = '", object%md5sum_cfg, "'"
    end if
    write (u, "(3x,A,I0)")  "Events requested  = ", object%n_evt_requested
    if (object%split_n_evt > 0 .or. object%split_n_kbytes > 0) then
       write (u, "(3x,A,I0)")  "Events per file   = ", object%split_n_evt
       write (u, "(3x,A,I0)")  "KBytes per file   = ", object%split_n_kbytes
       write (u, "(3x,A,I0)")  "First file index  = ", object%split_index
    end if
    call object%counter%write (u)
    call write_separator (u)
    if (object%i_prc /= 0) then
       write (u, "(1x,A)")  "Current event:"
       write (u, "(3x,A,I0,A,A)")  "Process #", &
            object%i_prc, ": ", &
            char (object%entry(object%i_prc)%process_id)
       write (u, "(3x,A,I0)")  "MCI set #", object%i_mci
       write (u, "(3x,A," // FMT_19 // ")")  "Weight    = ", object%weight
       if (.not. vanishes (object%excess)) &
            write (u, "(3x,A," // FMT_19 // ")")  "Excess    = ", object%excess
    else
       write (u, "(1x,A,I0,A,A)")  "Current event: [undefined]"
    end if
    call write_separator (u)
    if (allocated (object%rng)) then
       call object%rng%write (u)
    else
       write (u, "(3x,A)")  "Random-number generator: [undefined]"
    end if
    if (allocated (object%entry)) then
       do i = 1, size (object%entry)
          if (i == 1) then
             call write_separator (u, 2)
          else
             call write_separator (u)
          end if
          write (u, "(1x,A,I0,A)") "Process #", i, ":"
          call object%entry(i)%write_config (u)
       end do
    end if
    call write_separator (u, 2)
  end subroutine simulation_write
  
  subroutine simulation_write_event_unit &
       (object, unit, i_prc, verbose, testflag)
    class(simulation_t), intent(in) :: object
    integer, intent(in), optional :: unit
    logical, intent(in), optional :: verbose
    integer, intent(in), optional :: i_prc
    logical, intent(in), optional :: testflag
    logical :: pacified
    integer :: current
    pacified = .false.;  if (present(testflag)) pacified = testflag
    pacified = pacified .or. object%pacify
    if (present (i_prc)) then
       current = i_prc
    else
       current = object%i_prc
    end if
    if (current > 0) then
       call object%entry(current)%write (unit, verbose = verbose, &
            testflag = pacified)
    else
       call msg_fatal ("Simulation: write event: no process selected")
    end if
  end subroutine simulation_write_event_unit

  subroutine simulation_write_alt_event (object, unit, j_alt, i_prc, &
       verbose, testflag)
    class(simulation_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer, intent(in), optional :: j_alt
    integer, intent(in), optional :: i_prc
    logical, intent(in), optional :: verbose
    logical, intent(in), optional :: testflag
    integer :: i, j
    if (present (j_alt)) then
       j = j_alt
    else
       j = 1
    end if
    if (present (i_prc)) then
       i = i_prc
    else
       i = object%i_prc
    end if
    if (i > 0) then
       if (j> 0 .and. j <= object%n_alt) then
          call object%alt_entry(i,j)%write (unit, verbose = verbose, &
               testflag = testflag)
       else
          call msg_fatal ("Simulation: write alternate event: out of range")
       end if
    else
       call msg_fatal ("Simulation: write alternate event: no process selected")
    end if
  end subroutine simulation_write_alt_event

  subroutine simulation_final (object)
    class(simulation_t), intent(inout) :: object
    integer :: i, j
    if (allocated (object%entry)) then
       do i = 1, size (object%entry)
          call object%entry(i)%final ()
       end do
    end if
    if (allocated (object%alt_entry)) then
       do j = 1, size (object%alt_entry, 2)
          do i = 1, size (object%alt_entry, 1)
             call object%alt_entry(i,j)%final ()
          end do
       end do
    end if
    if (allocated (object%rng))  call object%rng%final ()
  end subroutine simulation_final
  
  subroutine simulation_init (simulation, &
       process_id, integrate, generate, local, global, alt_env)
    class(simulation_t), intent(out), target :: simulation
    type(string_t), dimension(:), intent(in) :: process_id
    logical, intent(in) :: integrate, generate
    type(rt_data_t), intent(inout), target :: local
    type(rt_data_t), intent(inout), optional, target :: global
    type(rt_data_t), dimension(:), intent(inout), optional, target :: alt_env
    class(rng_factory_t), allocatable :: rng_factory
    type(string_t) :: norm_string, version_string
    logical :: use_process
    integer :: i, j
    simulation%local => local
    simulation%sample_id = &
         local%get_sval (var_str ("$sample"))
    simulation%unweighted = &
         local%get_lval (var_str ("?unweighted"))
    simulation%negative_weights = &
         local%get_lval (var_str ("?negative_weights"))
    simulation%respect_selection = &
         local%get_lval (var_str ("?sample_select"))
    version_string = &
         local%get_sval (var_str ("$event_file_version"))
    norm_string = &
         local%get_sval (var_str ("$sample_normalization"))
    simulation%norm_mode = &
         event_normalization_mode (norm_string, simulation%unweighted)
    simulation%pacify = &
         local%get_lval (var_str ("?sample_pacify"))
    simulation%n_max_tries = &
         local%get_ival (var_str ("sample_max_tries"))
    simulation%split_n_evt = &
         local%get_ival (var_str ("sample_split_n_evt"))
    simulation%split_n_kbytes = &
         local%get_ival (var_str ("sample_split_n_kbytes"))
    simulation%split_index = &
         local%get_ival (var_str ("sample_split_index"))
    simulation%update_sqme = &
         local%get_lval (var_str ("?update_sqme"))
    simulation%update_weight = &
         local%get_lval (var_str ("?update_weight"))
    simulation%update_event = &
         local%get_lval (var_str ("?update_event"))
    simulation%recover_beams = &
         local%get_lval (var_str ("?recover_beams"))
    use_process = &
         integrate .or. generate &
         .or. simulation%update_sqme &
         .or. simulation%update_weight &
         .or. simulation%update_event &
         .or. present (alt_env)
    select case (size (process_id))
    case (0)
       call msg_error ("Simulation: no process selected")
    case (1)
       write (msg_buffer, "(A,A,A)") &
            "Starting simulation for process '", &
            char (process_id(1)), "'"
       call msg_message ()
    case default
       write (msg_buffer, "(A,A,A)") &
            "Starting simulation for processes '", &
            char (process_id(1)), "' etc."
       call msg_message ()
    end select
    select case (char (version_string))
    case ("", "2.2.4")
       simulation%version = 2
    case ("2.2")
       simulation%version = 1
    case default
       simulation%version = 0
    end select
    if (simulation%version == 0) then
       call msg_fatal ("Event file format '" &
            // char (version_string) &
            // "' is not compatible with this version.")
    end if        
    simulation%n_prc = size (process_id)
    allocate (simulation%entry (simulation%n_prc))
    if (present (alt_env)) then
       simulation%n_alt = size (alt_env)
       do i = 1, simulation%n_prc
          call simulation%entry(i)%init (process_id(i), &
               use_process, integrate, generate, &
               simulation%update_sqme, &
               local, global, simulation%n_alt)
          if (signal_is_pending ())  return
       end do
       simulation%valid = any (simulation%entry%valid)
       if (.not. simulation%valid) then
          call msg_error ("Simulate: no process has a valid matrix element.")
          return
       end if
       call simulation%update_processes ()
       allocate (simulation%alt_entry (simulation%n_prc, simulation%n_alt))
       allocate (simulation%md5sum_alt (simulation%n_alt))
       simulation%md5sum_alt = ""
       do j = 1, simulation%n_alt
          do i = 1, simulation%n_prc
             call simulation%alt_entry(i,j)%init_alt (process_id(i), &
                  simulation%entry(i)%get_process_ptr (), alt_env(j))
             if (signal_is_pending ())  return
          end do
       end do
       call simulation%restore_processes ()
    else       
       do i = 1, simulation%n_prc
          call simulation%entry(i)%init &
               (process_id(i), &
               use_process, integrate, generate, simulation%update_sqme, &
               local, global)
          call simulation%entry(i)%determine_if_powheg_matching ()
          if (signal_is_pending ())  return          
          if (simulation%entry(i)%is_nlo_event()) &
             call simulation%entry(i)%setup_additional_entries ()
       end do
       simulation%valid = any (simulation%entry%valid)
       if (.not. simulation%valid) then
          call msg_error ("Simulate: " &
               // "no process has a valid matrix element.") 
          return
       end if
    end if
!!! if this becomes conditional, some ref files will need update (seed change)
!    if (generate) then
       call dispatch_rng_factory (rng_factory, local)
       call rng_factory%make (simulation%rng)
!    end if
    if (all (simulation%entry%has_integral)) then
       simulation%integral = sum (simulation%entry%integral)
       simulation%error = sqrt (sum (simulation%entry%error ** 2))
       simulation%has_integral = .true.
       if (integrate .and. generate) then
          do i = 1, simulation%n_prc
             if (simulation%entry(i)%integral < 0 .and. .not. &
                  simulation%negative_weights) then
                call msg_fatal ("Integral of process '" // &
                     char (process_id (i)) // "'is negative.")
             end if
          end do
       end if
    else 
       if (integrate .and. generate) &
            call msg_error ("Simulation contains undefined integrals.")
    end if
    if (simulation%integral > 0 .or. &
         (simulation%integral < 0 .and. simulation%negative_weights)) then
       simulation%valid = .true.
    else if (generate) then
       call msg_error ("Simulate: " &
            // "sum of process integrals must be positive; skipping.")
       simulation%valid = .false.
    else
       simulation%valid = .true.
    end if
    if (simulation%valid)  call simulation%compute_md5sum ()
  end subroutine simulation_init

  subroutine simulation_compute_n_events (simulation, n_events, var_list)
    class(simulation_t), intent(in) :: simulation
    integer, intent(out) :: n_events
    type(var_list_t) :: var_list
    real(default) :: lumi, x_events_lumi
    integer :: n_events_lumi
    logical :: is_scattering
    n_events = &
         var_list%get_ival (var_str ("n_events"))
    lumi = &
         var_list%get_rval (var_str ("luminosity"))
    if (simulation%unweighted) then
       is_scattering = simulation%entry(1)%n_in == 2
       if (is_scattering) then
          x_events_lumi = abs (simulation%integral * lumi)
          if (x_events_lumi < huge (n_events)) then
             n_events_lumi = nint (x_events_lumi)
          else
             call msg_message ("Simulation: luminosity too large, &
                  &limiting number of events")
             n_events_lumi = huge (n_events)
          end if
          if (n_events_lumi > n_events) then
             call msg_message ("Simulation: using n_events as computed from &
                  &luminosity value")
             n_events = n_events_lumi
          else
             write (msg_buffer, "(A,1x,I0)") &
                  "Simulation: requested number of events =", n_events
             call msg_message ()
             if (.not. vanishes (simulation%integral)) then
                write (msg_buffer, "(A,1x,ES11.4)") &
                     "            corr. to luminosity [fb-1] = ", &
                     n_events / simulation%integral        
                call msg_message ()
             end if
          end if
       end if
    end if
  end subroutine simulation_compute_n_events

  subroutine simulation_show_efficiency (simulation)
    class(simulation_t), intent(inout) :: simulation
    integer :: n_events, n_calls
    real(default) :: eff
    n_events = simulation%counter%generated
    n_calls = sum (simulation%entry%get_actual_calls_total ())
    if (n_calls > 0) then
       eff = real (n_events, kind=default) / n_calls
       write (msg_buffer, "(A,1x,F6.2,1x,A)") &
            "Events: actual unweighting efficiency =", 100 * eff, "%"
       call msg_message ()
    end if
  end subroutine simulation_show_efficiency
  
  function simulation_get_n_nlo_entries (simulation, i_prc) result (n_extra)
    class(simulation_t), intent(in) :: simulation
    integer, intent(in) :: i_prc
    integer :: n_extra
    n_extra = simulation%entry(i_prc)%count_nlo_entries ()
  end function simulation_get_n_nlo_entries

  subroutine simulation_compute_md5sum (simulation)
    class(simulation_t), intent(inout) :: simulation
    type(process_t), pointer :: process
    type(string_t) :: buffer
    integer :: j, i, n_mci, i_mci, n_component, i_component
    if (simulation%md5sum_prc == "") then
       buffer = ""
       do i = 1, simulation%n_prc
          if (.not. simulation%entry(i)%valid) cycle
          process => simulation%entry(i)%get_process_ptr ()
          if (associated (process)) then
             n_component = process%get_n_components ()
             do i_component = 1, n_component
                if (process%has_matrix_element (i_component)) then
                   buffer = buffer // process%get_md5sum_prc (i_component)
                end if
             end do
          end if
       end do
       simulation%md5sum_prc = md5sum (char (buffer))
    end if
    if (simulation%md5sum_cfg == "") then
       buffer = ""
       do i = 1, simulation%n_prc
          if (.not. simulation%entry(i)%valid) cycle          
          process => simulation%entry(i)%get_process_ptr ()
          if (associated (process)) then
             n_mci = process%get_n_mci ()
             do i_mci = 1, n_mci
                buffer = buffer // process%get_md5sum_mci (i_mci)
             end do
          end if
       end do
       simulation%md5sum_cfg = md5sum (char (buffer))
    end if
    do j = 1, simulation%n_alt
       if (simulation%md5sum_alt(j) == "") then
          buffer = ""
          do i = 1, simulation%n_prc
             process => simulation%alt_entry(i,j)%get_process_ptr ()
             if (associated (process)) then
                buffer = buffer // process%get_md5sum_cfg ()
             end if
          end do
          simulation%md5sum_alt(j) = md5sum (char (buffer))
       end if
    end do
  end subroutine simulation_compute_md5sum

  subroutine simulation_init_process_selector (simulation)
    class(simulation_t), intent(inout) :: simulation
    integer :: i
    if (simulation%has_integral) then
       call simulation%process_selector%init (simulation%entry%integral, &
            negative_weights = simulation%negative_weights)
       do i = 1, simulation%n_prc
          associate (entry => simulation%entry(i))
            if (.not. entry%valid) then
               call msg_warning ("Process '" // char (entry%process_id) // &
                    "': matrix element vanishes, no events can be generated.")
               cycle
            end if
            call entry%init_mci_selector (simulation%negative_weights)
            entry%process_weight = simulation%process_selector%get_weight (i)
          end associate
       end do
    end if
  end subroutine simulation_init_process_selector
    
  function simulation_select_prc (simulation) result (i_prc)
    class(simulation_t), intent(inout) :: simulation
    integer :: i_prc
    call simulation%process_selector%generate (simulation%rng, i_prc)
  end function simulation_select_prc

  function simulation_select_mci (simulation) result (i_mci)
    class(simulation_t), intent(inout) :: simulation
    integer :: i_mci
    i_mci = 0
    if (simulation%i_prc /= 0) then
       i_mci = simulation%entry(simulation%i_prc)%select_mci ()
    end if
  end function simulation_select_mci

  subroutine simulation_generate (simulation, n, es_array)
    class(simulation_t), intent(inout), target :: simulation
    integer, intent(in) :: n
    type(event_stream_array_t), intent(inout), optional :: es_array
    type(string_t) :: str1, str2, str3
    logical :: generate_new, passed
    integer :: i, j, k
    type(entry_t), pointer :: current_entry
    integer :: n_events
    simulation%n_evt_requested = n
    n_events = n * simulation%get_n_nlo_entries (1)
    call simulation%entry%set_n (n)
    if (simulation%n_alt > 0)  call simulation%alt_entry%set_n (n)
    str1 = "Events: generating"
    if (present (es_array)) then
       if (es_array%has_input ())  str1 = "Events: reading"
    end if
    if (simulation%entry(1)%config%unweighted) then
       str2 = "unweighted"
    else
       str2 = "weighted"
    end if
    if (simulation%entry(1)%config%factorization_mode == &
         FM_IGNORE_HELICITY) then
       str3 = ", unpolarized"
    else 
       str3 = ", polarized"
    end if    
    if (n_events == n) then
       write (msg_buffer, "(A,1x,I0,1x,A,1x,A)")  char (str1), n, &
            char (str2) // char(str3), "events ..."
    else
       write (msg_buffer, "(A,1x,I0,1x,A,1x,A)") char (str1), n_events, &
            char (str2) // char(str3), "NLO events ..."
    end if 
    call msg_message ()
    write (msg_buffer, "(A,1x,A)") "Events: event normalization mode", &
         char (event_normalization_string (simulation%norm_mode))
    call msg_message ()
    do i = 1, n
       if (present (es_array)) then
          call simulation%read_event (es_array, .true., generate_new)
       else
          generate_new = .true.
       end if
       if (generate_new) then
          simulation%i_prc = simulation%select_prc ()
          simulation%i_mci = simulation%select_mci ()
          associate (entry => simulation%entry(simulation%i_prc))
            call entry%set_active_real_component (simulation%i_mci)
            current_entry => entry%get_first ()
            do k = 1, current_entry%count_nlo_entries ()
               if (k > 1) then
                  current_entry => current_entry%get_next ()
                  current_entry%particle_set => current_entry%first%particle_set
                  current_entry%particle_set_is_valid &
                     = current_entry%first%particle_set_is_valid
               end if
               do j = 1, simulation%n_max_tries
                  if (.not. current_entry%valid)  call msg_warning &
                          ("Process '" // char (current_entry%process_id) // "': " // &
                          "matrix element vanishes, no events can be generated.")
                  call current_entry%generate (simulation%i_mci, i_nlo=k)
                  if (signal_is_pending ()) return
                  if (current_entry%has_valid_particle_set ())  exit
               end do
            end do
            if (entry%is_nlo_event()) call entry%reset_nlo_counter ()
            if (.not. entry%has_valid_particle_set ()) then
               write (msg_buffer, "(A,I0,A)")  "Simulation: failed to &
                    &generate valid event after ", &
                    simulation%n_max_tries, " tries (sample_max_tries)"
               call msg_fatal ()
            end if
            current_entry => entry%get_first ()
            do k = 1, current_entry%count_nlo_entries ()
               if (k > 1) current_entry => current_entry%get_next ()
               call current_entry%evaluate_expressions ()
            end do
            if (signal_is_pending ()) return
            if (entry%passed_selection ()) then
               simulation%weight = entry%get_weight_ref ()
               simulation%excess = entry%get_excess_prc ()
            end if
            call simulation%counter%record &
                 (simulation%weight, simulation%excess)
            call entry%record (simulation%i_mci)
          end associate
       else
          associate (entry => simulation%entry(simulation%i_prc))
            call entry%accept_sqme_ref ()
            call entry%accept_weight_ref ()
            !!! JRR: WK please check: why commented out
            ! call entry%evaluate_transforms ()  ! doesn't activate
            call entry%check ()
            call entry%evaluate_expressions ()
            if (signal_is_pending ()) return
            if (entry%passed_selection ()) then
               simulation%weight = entry%get_weight_ref ()
               simulation%excess = entry%get_excess_prc ()
            end if
            call simulation%counter%record &
                 (simulation%weight, simulation%excess, from_file=.true.)
            call entry%record (simulation%i_mci, from_file=.true.)
          end associate
       end if
       call simulation%calculate_alt_entries ()
       if (signal_is_pending ()) return
       if (simulation%pacify)  call pacify (simulation)
       if (simulation%respect_selection) then
          passed = simulation%entry(simulation%i_prc)%passed_selection ()
       else
          passed = .true.
       end if
       if (present (es_array)) then
          call simulation%write_event (es_array, passed)
       end if
    end do
    call msg_message ("        ... event sample complete.")
    if (simulation%unweighted)  call simulation%show_efficiency ()
    call simulation%counter%show_excess ()
  end subroutine simulation_generate
  
  subroutine simulation_calculate_alt_entries (simulation)
    class(simulation_t), intent(inout) :: simulation
    real(default) :: factor
    real(default), dimension(:), allocatable :: sqme_alt, weight_alt
    integer :: n_alt, i, j
    i = simulation%i_prc
    n_alt = simulation%n_alt
    if (n_alt == 0)  return
    allocate (sqme_alt (n_alt), weight_alt (n_alt))
    associate (entry => simulation%entry(i))
      do j = 1, n_alt
         if (signal_is_pending ())  return
         factor = entry%get_kinematical_weight ()
         associate (alt_entry => simulation%alt_entry(i,j))
           call alt_entry%update_process ()
           call alt_entry%select &
                (entry%get_i_mci (), entry%get_i_term (), entry%get_channel ())
           call alt_entry%fill_particle_set (entry)
           call alt_entry%recalculate &
                (update_sqme = .true., weight_factor = factor)
           if (signal_is_pending ())  return
           call alt_entry%accept_sqme_prc ()
           call alt_entry%update_normalization ()
           call alt_entry%accept_weight_prc ()
           call alt_entry%check ()
           call alt_entry%evaluate_expressions ()
           if (signal_is_pending ())  return
           call alt_entry%restore_process ()
           sqme_alt(j) = alt_entry%get_sqme_ref ()
           if (alt_entry%passed_selection ()) then
              weight_alt(j) = alt_entry%get_weight_ref ()
           end if
         end associate
      end do
      call entry%set (sqme_alt = sqme_alt, weight_alt = weight_alt)
      call entry%check ()
      call entry%store_alt_values ()
    end associate
  end subroutine simulation_calculate_alt_entries
       
  subroutine simulation_rescan (simulation, n, es_array, global)
    class(simulation_t), intent(inout) :: simulation
    integer, intent(in) :: n
    type(event_stream_array_t), intent(inout) :: es_array
    type(rt_data_t), intent(inout) :: global
    type(qcd_t) :: qcd
    type(string_t) :: str1, str2, str3
    logical :: complete
    str1 = "Rescanning"
    if (simulation%entry(1)%config%unweighted) then
       str2 = "unweighted"
    else
       str2 = "weighted"
    end if
    simulation%n_evt_requested = n
    call simulation%entry%set_n (n)
    if (simulation%update_sqme .or. simulation%update_weight) then
       call dispatch_qcd (qcd, global)
       call simulation%update_processes &
            (global%model, qcd, global%get_helicity_selection ())
       str3 = "(process parameters updated) "
    else
       str3 = ""
    end if
    write (msg_buffer, "(A,1x,A,1x,A,A,A)")  char (str1), char (str2), &
         "events ", char (str3), "..."
    call msg_message ()
    do
       call simulation%read_event (es_array, .false., complete)
       if (complete)  exit
       if (simulation%update_event &
            .or. simulation%update_sqme &
            .or. simulation%update_weight) then
          call simulation%recalculate ()
          if (signal_is_pending ())  return
          associate (entry => simulation%entry(simulation%i_prc))
            call entry%update_normalization ()
            if (simulation%update_event) then
               call entry%evaluate_transforms ()
            end if
            call entry%check ()
            call entry%evaluate_expressions ()
            if (signal_is_pending ())  return
            simulation%weight = entry%get_weight_prc ()
            call simulation%counter%record (simulation%weight, from_file=.true.)
            call entry%record (simulation%i_mci, from_file=.true.)
          end associate
       else
          associate (entry => simulation%entry(simulation%i_prc))
            call entry%accept_sqme_ref ()
            call entry%accept_weight_ref ()
            call entry%check ()
            call entry%evaluate_expressions ()
            if (signal_is_pending ())  return
            simulation%weight = entry%get_weight_ref ()
            call simulation%counter%record (simulation%weight, from_file=.true.)
            call entry%record (simulation%i_mci, from_file=.true.)
          end associate
       end if
       call simulation%calculate_alt_entries ()
       if (signal_is_pending ())  return
       call simulation%write_event (es_array)
    end do
    if (simulation%update_sqme .or. simulation%update_weight) then
       call simulation%restore_processes ()
    end if
  end subroutine simulation_rescan
  
  subroutine simulation_update_processes (simulation, &
       model, qcd, helicity_selection)
    class(simulation_t), intent(inout) :: simulation
    class(model_data_t), intent(in), optional, target :: model
    type(qcd_t), intent(in), optional :: qcd
    type(helicity_selection_t), intent(in), optional :: helicity_selection
    integer :: i
    do i = 1, simulation%n_prc
       call simulation%entry(i)%update_process (model, qcd, helicity_selection)
    end do
  end subroutine simulation_update_processes
  
  subroutine simulation_restore_processes (simulation)
    class(simulation_t), intent(inout) :: simulation
    integer :: i
    do i = 1, simulation%n_prc
       call simulation%entry(i)%restore_process ()
    end do
  end subroutine simulation_restore_processes
  
  subroutine simulation_write_event_eio (object, eio, i_prc)
    class(simulation_t), intent(in) :: object
    class(eio_t), intent(inout) :: eio
    integer, intent(in), optional :: i_prc
    logical :: increased
    integer :: current
    if (present (i_prc)) then
       current = i_prc
    else
       current = object%i_prc
    end if
    if (current > 0) then
       if (object%split_n_evt > 0 &
            .and. object%counter%total > 1 &
            .and. mod (object%counter%total, object%split_n_evt) == 1) then
          call eio%split_out ()
       else if (object%split_n_kbytes > 0) then
          call eio%update_split_count (increased)
          if (increased)  call eio%split_out ()
       end if
       call eio%output (object%entry(current)%event_t, current, pacify = object%pacify)
    else
       call msg_fatal ("Simulation: write event: no process selected")
    end if
  end subroutine simulation_write_event_eio

  subroutine simulation_read_event_eio (object, eio)
    class(simulation_t), intent(inout) :: object
    class(eio_t), intent(inout) :: eio
    integer :: iostat, current
    call eio%input_i_prc (current, iostat)
    select case (iostat)
    case (0)
       object%i_prc = current
       call eio%input_event (object%entry(current)%event_t, iostat)
    end select
    select case (iostat)
    case (:-1)
       object%i_prc = 0
       object%i_mci = 0
    case (1:)
       call msg_error ("Reading events: I/O error, aborting read")
       object%i_prc = 0
       object%i_mci = 0
    case default
       object%i_mci = object%entry(current)%get_i_mci ()
    end select
  end subroutine simulation_read_event_eio

  subroutine simulation_write_event_es_array (object, es_array, passed)
    class(simulation_t), intent(in), target :: object
    class(event_stream_array_t), intent(inout) :: es_array
    logical, intent(in), optional :: passed
    integer :: i_prc, event_index
    integer :: i
    type(entry_t), pointer :: current_entry
    i_prc = object%i_prc
    if (i_prc > 0) then
       event_index = object%counter%total
       current_entry => object%entry(i_prc)%get_first ()
       do i = 1, current_entry%count_nlo_entries ()
          if (i > 1) current_entry => current_entry%get_next ()
          call es_array%output (current_entry%event_t, i_prc, &
             event_index, passed = passed, pacify = object%pacify)
       end do
    else
       call msg_fatal ("Simulation: write event: no process selected")
    end if
  end subroutine simulation_write_event_es_array

  subroutine simulation_read_event_es_array (object, es_array, enable_switch, &
       fail)
    class(simulation_t), intent(inout), target :: object
    class(event_stream_array_t), intent(inout), target :: es_array
    logical, intent(in) :: enable_switch
    logical, intent(out) :: fail
    integer :: iostat, i_prc
    type(entry_t), pointer :: current_entry => null ()
    integer :: i
    if (es_array%has_input ()) then
       fail = .false.
       call es_array%input_i_prc (i_prc, iostat)
       select case (iostat)
       case (0)
          object%i_prc = i_prc
          current_entry => object%entry(i_prc)
          do i = 1, current_entry%count_nlo_entries ()
             if (i > 1) then
                call es_array%skip_eio_entry (iostat)
                current_entry => current_entry%get_next ()
             end if
             call es_array%input_event (current_entry%event_t, iostat)
          end do
       case (:-1)
          write (msg_buffer, "(A,1x,I0,1x,A)")  &
               "... event file terminates after", &
               object%counter%read, "events."
          call msg_message ()
          if (enable_switch) then
             call es_array%switch_inout ()
             write (msg_buffer, "(A,1x,I0,1x,A)")  &
                  "Generating remaining ", &
                  object%n_evt_requested - object%counter%read, "events ..."
             call msg_message ()
          end if
          fail = .true.
          return
       end select
       select case (iostat)
       case (0)
          object%i_mci = object%entry(i_prc)%get_i_mci ()
       case default
          write (msg_buffer, "(A,1x,I0,1x,A)")  &
               "Reading events: I/O error, aborting read after", &
               object%counter%read, "events."
          call msg_error ()
          object%i_prc = 0
          object%i_mci = 0
          fail = .true.
       end select
    else
       fail = .true.
    end if
  end subroutine simulation_read_event_es_array

  subroutine simulation_recalculate (simulation)
    class(simulation_t), intent(inout) :: simulation
    integer :: i_prc
    i_prc = simulation%i_prc
    associate (entry => simulation%entry(i_prc))
      if (simulation%update_weight) then
         call simulation%entry(i_prc)%recalculate &
              (update_sqme = simulation%update_sqme, &
              recover_beams = simulation%recover_beams, &
              weight_factor = entry%get_kinematical_weight ())
      else
         call simulation%entry(i_prc)%recalculate &
              (update_sqme = simulation%update_sqme, &
              recover_beams = simulation%recover_beams)
      end if
    end associate
  end subroutine simulation_recalculate

  function simulation_get_md5sum_prc (simulation) result (md5sum)
    class(simulation_t), intent(in) :: simulation
    character(32) :: md5sum
    md5sum = simulation%md5sum_prc
  end function simulation_get_md5sum_prc
    
  function simulation_get_md5sum_cfg (simulation) result (md5sum)
    class(simulation_t), intent(in) :: simulation
    character(32) :: md5sum
    md5sum = simulation%md5sum_cfg
  end function simulation_get_md5sum_cfg
    
  function simulation_get_md5sum_alt (simulation, i) result (md5sum)
    class(simulation_t), intent(in) :: simulation
    integer, intent(in) :: i
    character(32) :: md5sum
    md5sum = simulation%md5sum_alt(i)
  end function simulation_get_md5sum_alt
    
  function simulation_get_data (simulation, alt) result (sdata)
    class(simulation_t), intent(in) :: simulation
    logical, intent(in), optional :: alt
    type(event_sample_data_t) :: sdata
    type(process_t), pointer :: process
    type(beam_data_t), pointer :: beam_data
    type(beam_structure_t), pointer :: beam_structure
    type(flavor_t), dimension(:), allocatable :: flv
    integer :: n, i
    logical :: enable_alt, construct_beam_data
    real(default) :: sqrts
    class(model_data_t), pointer :: model
    logical :: decay_rest_frame
    type(string_t) :: process_id
    enable_alt = .true.;  if (present (alt))  enable_alt = alt    
    if (enable_alt) then
       call sdata%init (simulation%n_prc, simulation%n_alt)
       do i = 1, simulation%n_alt
          sdata%md5sum_alt(i) = simulation%get_md5sum_alt (i)
       end do
    else
       call sdata%init (simulation%n_prc)
    end if
    sdata%unweighted = simulation%unweighted
    sdata%negative_weights = simulation%negative_weights
    sdata%norm_mode = simulation%norm_mode
    process => simulation%entry(1)%get_process_ptr ()
    if (associated (process)) then
       beam_data => process%get_beam_data_ptr ()
       construct_beam_data = .false.
    else
       n = simulation%entry(1)%n_in
       sqrts = simulation%local%get_sqrts ()
       beam_structure => simulation%local%beam_structure
       call beam_structure%check_against_n_in (n, construct_beam_data)
       if (construct_beam_data) then
          allocate (beam_data)
          model => simulation%local%model
          decay_rest_frame = &
               simulation%local%get_lval (var_str ("?decay_rest_frame"))    
          call beam_data%init_structure (beam_structure, &
               sqrts, model, decay_rest_frame)
       else
          beam_data => null ()
       end if
    end if
    if (associated (beam_data)) then
       n = beam_data%get_n_in ()
       sdata%n_beam = n
       allocate (flv (n))
       flv = beam_data%get_flavor ()
       sdata%pdg_beam(:n) = flv%get_pdg ()
       sdata%energy_beam(:n) = beam_data%get_energy ()
       if (construct_beam_data)  deallocate (beam_data)
    else
       n = simulation%entry(1)%n_in
       sdata%n_beam = n
       process_id = simulation%entry(1)%process_id
       call simulation%local%prclib%get_pdg_in_1 &
            (process_id, sdata%pdg_beam(:n))
       sdata%energy_beam(:n) = sqrts / n
    end if
    do i = 1, simulation%n_prc
       if (.not. simulation%entry(i)%valid) cycle
       process => simulation%entry(i)%get_process_ptr ()
       if (associated (process)) then
          sdata%proc_num_id(i) = process%get_num_id ()
       else
          process_id = simulation%entry(i)%process_id
          sdata%proc_num_id(i) = simulation%local%prclib%get_num_id (process_id)
       end if
       if (sdata%proc_num_id(i) == 0)  sdata%proc_num_id(i) = i
       if (simulation%entry(i)%has_integral) then
          sdata%cross_section(i) = simulation%entry(i)%integral
          sdata%error(i) = simulation%entry(i)%error
       end if
    end do
    sdata%total_cross_section = sum (sdata%cross_section)
    sdata%md5sum_prc = simulation%get_md5sum_prc ()
    sdata%md5sum_cfg = simulation%get_md5sum_cfg ()
    if (simulation%split_n_evt > 0 .or. simulation%split_n_kbytes > 0) then
       sdata%split_n_evt = simulation%split_n_evt
       sdata%split_n_kbytes = simulation%split_n_kbytes
       sdata%split_index = simulation%split_index
    end if
  end function simulation_get_data
    
  function simulation_get_default_sample_name (simulation) result (sample)
    class(simulation_t), intent(in) :: simulation
    type(string_t) :: sample
    type(process_t), pointer :: process
    sample = "whizard"
    if (simulation%n_prc > 0) then
       process => simulation%entry(1)%get_process_ptr ()
       if (associated (process)) then
          sample = process%get_id ()
       end if
    end if
  end function simulation_get_default_sample_name

  function simulation_is_valid (simulation) result (valid)
    class(simulation_t), intent(inout) :: simulation
    logical :: valid
    valid = simulation%valid
  end function simulation_is_valid

  subroutine pacify_simulation (simulation)
    class(simulation_t), intent(inout) :: simulation
    integer :: i, j
    i = simulation%i_prc
    if (i > 0) then
       call pacify (simulation%entry(i))
       do j = 1, simulation%n_alt
          call pacify (simulation%alt_entry(i,j))
       end do
    end if
  end subroutine pacify_simulation
  
  subroutine simulation_evaluate_expressions (simulation)
    class(simulation_t), intent(inout) :: simulation
    call simulation%entry(simulation%i_prc)%evaluate_expressions ()
  end subroutine simulation_evaluate_expressions
  

end module simulations
