! WHIZARD 2.2.8 Nov 22 2015
! 
! Copyright (C) 1999-2015 by 
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!     
!     with contributions from
!     Fabian Bach <fabian.bach@t-online.de>
!     Bijan Chokoufe <bijan.chokoufe@desy.de>
!     Christian Speckner <cnspeckn@googlemail.com> 
!     Soyoung Shim <soyoung.shim@desy.de>
!     Florian Staub <florian.staub@cern.ch>  
!     Christian Weiss <christian.weiss@desy.de>
!     and Hans-Werner Boschmann, Felix Braam, 
!     Sebastian Schmidt, So-young Shim, Daniel Wiesler 
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by 
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'

module prc_openloops

  use, intrinsic :: iso_c_binding !NODEP!

  use kinds
  use io_units
  use iso_varying_string, string_t => varying_string
  use constants
  use unit_tests, only: vanishes
  use system_defs, only: TAB
  use diagnostics
  use system_dependencies
  use physics_defs
  use variables
  use os_interface
  use lorentz
  use interactions
  use sm_qcd
  use model_data

  use prclib_interfaces
  use prc_core_def
  use prc_core

  use blha_config
  use blha_olp_interfaces

  implicit none
  private

  public :: openloops_def_t
  public :: prc_openloops_t

  real(default), parameter :: openloops_default_bmass = 0._default
  real(default), parameter :: openloops_default_topmass = 172._default
  real(default), parameter :: openloops_default_topwidth = 0._default
  real(default), parameter :: openloops_default_wmass = 80.399_default
  real(default), parameter :: openloops_default_wwidth = 0._default
  real(default), parameter :: openloops_default_zmass = 91.1876_default
  real(default), parameter :: openloops_default_zwidth = 0._default
  real(default), parameter :: openloops_default_higgsmass = 125._default
  real(default), parameter :: openloops_default_higgswidth = 0._default

  integer :: N_EXTERNAL = 0


  type, extends (prc_blha_writer_t) :: openloops_writer_t
  contains
    procedure, nopass :: type_name => openloops_writer_type_name
  end type openloops_writer_t

  type, extends (blha_def_t) :: openloops_def_t
  contains
    procedure :: init => openloops_def_init
    procedure, nopass :: type_string => openloops_def_type_string
    procedure :: write => openloops_def_write
    procedure :: read => openloops_def_read
    procedure :: allocate_driver => openloops_def_allocate_driver
  end type openloops_def_t

  type, extends (blha_driver_t) :: openloops_driver_t 
    integer :: n_external = 0
    type(string_t) :: olp_file
    procedure(ol_evaluate_sc), nopass, pointer :: &
       evaluate_spin_correlations => null ()
  contains
    procedure :: init_dlaccess_to_library => openloops_driver_init_dlaccess_to_library
    procedure :: set_alpha_s => openloops_driver_set_alpha_s
    procedure :: set_alpha_qed => openloops_driver_set_alpha_qed
    procedure :: set_GF => openloops_driver_set_GF
    procedure :: set_weinberg_angle => openloops_driver_set_weinberg_angle
    procedure :: print_alpha_s => openloops_driver_print_alpha_s
    procedure, nopass :: type_name => openloops_driver_type_name
    procedure :: load_sc_procedure => openloops_driver_load_sc_procedure
  end type openloops_driver_t 

  type, extends (blha_state_t) :: openloops_state_t
  contains
    procedure :: write => openloops_state_write
  end type openloops_state_t

  type, extends (prc_blha_t) :: prc_openloops_t
  contains
    procedure :: allocate_workspace => prc_openloops_allocate_workspace
    procedure :: init_driver => prc_openloops_init_driver
    procedure :: write => prc_openloops_write
    procedure :: prepare_library => prc_openloops_prepare_library
    procedure :: load_driver => prc_openloops_load_driver
    procedure :: start => prc_openloops_start
    procedure :: set_n_external => prc_openloops_set_n_external
    procedure :: reset_parameters => prc_openloops_reset_parameters
    procedure :: set_verbosity => prc_openloops_set_verbosity
    procedure :: compute_sqme_born => prc_openloops_compute_sqme_born
    procedure :: compute_sqme_real => prc_openloops_compute_sqme_real
    procedure :: compute_sqme_sc => prc_openloops_compute_sqme_sc
  end type prc_openloops_t


  abstract interface
     subroutine omega_update_alpha_s (alpha_s) bind(C)
       import
       real(c_default_float), intent(in) :: alpha_s
     end subroutine omega_update_alpha_s
  end interface

  abstract interface
     subroutine ol_evaluate_sc (id, pp, emitter, polvect, res) bind(C)
       import
       integer(kind=c_int), value :: id, emitter
       real(kind=c_double), intent(in) :: pp(5*N_EXTERNAL), polvect(4)
       real(kind=c_double), intent(out) :: res(N_EXTERNAL)
     end subroutine ol_evaluate_sc
  end interface


contains

  function openloops_writer_type_name () result (string)
    type(string_t) :: string
    string = "openloops"
  end function openloops_writer_type_name

  subroutine openloops_def_init (object, basename, model_name, &
                                 prt_in, prt_out, nlo_type)
    class(openloops_def_t), intent(inout) :: object
    type(string_t), intent(in) :: basename, model_name
    type(string_t), dimension(:), intent(in) :: prt_in, prt_out
    integer :: nlo_type
    object%basename = basename
    allocate (openloops_writer_t :: object%writer)
    select case (nlo_type)
    case (BORN)
       object%suffix = '_BORN'
    case (NLO_REAL)
       object%suffix = '_REAL'
    case (NLO_VIRTUAL)
       object%suffix = '_LOOP'
    case (NLO_SUBTRACTION)
       object%suffix = '_SUB'
    end select
    select type (writer => object%writer)
    class is (prc_blha_writer_t)
       call writer%init (model_name, prt_in, prt_out)
    end select
  end subroutine openloops_def_init

  function openloops_def_type_string () result (string)
    type(string_t) :: string
    string = "openloops"
  end function openloops_def_type_string

  subroutine openloops_def_write (object, unit)
    class(openloops_def_t), intent(in) :: object
    integer, intent(in) :: unit
    select type (writer => object%writer)
    type is (openloops_writer_t)
       call writer%write (unit)
    end select
  end subroutine openloops_def_write

  subroutine openloops_driver_init_dlaccess_to_library &
     (object, os_data, dlaccess, success)
    class(openloops_driver_t), intent(in) :: object
    type(os_data_t), intent(in) :: os_data
    type(dlaccess_t), intent(out) :: dlaccess
    logical, intent(out) :: success
    type(string_t) :: ol_library, msg_buffer
    ol_library = OPENLOOPS_DIR // '/lib/libopenloops.' // &
         os_data%shrlib_ext
    msg_buffer = "One-Loop-Provider: Using OpenLoops"
    call msg_message (char(msg_buffer))
    msg_buffer = "Loading library: " // ol_library
    call msg_message (char(msg_buffer))
    if (os_file_exist (ol_library)) then
       call dlaccess_init (dlaccess, var_str (""), ol_library, os_data)
    else
       call msg_fatal ("Link OpenLoops: library not found")
    end if
    success = .not. dlaccess_has_error (dlaccess)
  end subroutine openloops_driver_init_dlaccess_to_library 

  subroutine openloops_driver_set_alpha_s (driver, alpha_s)
    class(openloops_driver_t), intent(inout) :: driver
    real(default), intent(in) :: alpha_s
    integer :: ierr
    call driver%blha_olp_set_parameter &
       (c_char_'alphas'//c_null_char, &
        dble (alpha_s), 0._double, ierr)
    if (ierr == 0) call parameter_error_message (var_str ('alphas'))
  end subroutine openloops_driver_set_alpha_s

  subroutine openloops_driver_set_alpha_qed (driver, alpha)
    class(openloops_driver_t), intent(inout) :: driver
    real(default), intent(in) :: alpha
    integer :: ierr
    call driver%blha_olp_set_parameter &
       (c_char_'alpha_qed'//c_null_char, &
        dble (alpha), 0._double, ierr) 
    if (ierr == 0) call parameter_error_message (var_str ('alpha_qed'))
  end subroutine openloops_driver_set_alpha_qed

  subroutine openloops_driver_set_GF (driver, GF)
    class(openloops_driver_t), intent(inout) :: driver
    real(default), intent(in) :: GF
    integer :: ierr 
    call driver%blha_olp_set_parameter &
       (c_char_'GF'//c_null_char, &
        dble(GF), 0._double, ierr)
    if (ierr == 0) call parameter_error_message (var_str ('GF'))
  end subroutine openloops_driver_set_GF

  subroutine openloops_driver_set_weinberg_angle (driver, sw2)
    class(openloops_driver_t), intent(inout) :: driver
    real(default), intent(in) :: sw2
    integer :: ierr 
    call driver%blha_olp_set_parameter &
       (c_char_'sw2'//c_null_char, &
        dble(sw2), 0._double, ierr)
    if (ierr == 0) call parameter_error_message (var_str ('sw2'))
  end subroutine openloops_driver_set_weinberg_angle

  subroutine openloops_driver_print_alpha_s (object)
    class(openloops_driver_t), intent(in) :: object
    call object%blha_olp_print_parameter (c_char_'alphas'//c_null_char)
  end subroutine openloops_driver_print_alpha_s

  function openloops_driver_type_name () result (type)
    type(string_t) :: type
    type = "OpenLoops"
  end function openloops_driver_type_name

  subroutine openloops_driver_load_sc_procedure (object, os_data, success)
    class(openloops_driver_t), intent(inout) :: object
    type(os_data_t), intent(in) :: os_data
    logical, intent(out) :: success
    type(dlaccess_t) :: dlaccess
    type(c_funptr) :: c_fptr
    logical :: init_success

    call object%init_dlaccess_to_library (os_data, dlaccess, init_success)
  
    c_fptr = dlaccess_get_c_funptr (dlaccess, var_str ("ol_evaluate_sc"))
    call c_f_procpointer (c_fptr, object%evaluate_spin_correlations)
    if (dlaccess_has_error (dlaccess)) &
       call msg_fatal ("Could not load Openloops spin correlations!")

    success = .true.
  end subroutine openloops_driver_load_sc_procedure 

  subroutine openloops_def_read (object, unit)
    class(openloops_def_t), intent(out) :: object
    integer, intent(in) :: unit
  end subroutine openloops_def_read

  subroutine openloops_def_allocate_driver (object, driver, basename)
    class(openloops_def_t), intent(in) :: object
    class(prc_core_driver_t), intent(out), allocatable :: driver
    type(string_t), intent(in) :: basename
    if (.not. allocated (driver)) allocate (openloops_driver_t :: driver)
  end subroutine openloops_def_allocate_driver

  subroutine openloops_state_write (object, unit)
    class(openloops_state_t), intent(in) :: object
    integer, intent(in), optional :: unit
  end subroutine openloops_state_write

  subroutine prc_openloops_allocate_workspace (object, core_state)
    class(prc_openloops_t), intent(in) :: object
    class(prc_core_state_t), intent(inout), allocatable :: core_state
    allocate (openloops_state_t :: core_state)
  end subroutine prc_openloops_allocate_workspace

  subroutine prc_openloops_init_driver (object, os_data)
    class(prc_openloops_t), intent(inout) :: object
    type(os_data_t), intent(in) :: os_data
    type(string_t) :: olp_file, olc_file

    select type (def => object%def)
    type is (openloops_def_t)
       olp_file = def%basename // def%suffix // '.olp'
       olc_file = def%basename // def%suffix // '.olc'
    class default
       call msg_bug ("prc_openloops_init_driver: core_def should be openloops-type")
    end select

    select type (driver => object%driver)
    type is (openloops_driver_t)
       driver%olp_file = olp_file
       driver%contract_file = olc_file
    end select
  end subroutine prc_openloops_init_driver

  subroutine prc_openloops_write (object, unit)
    class(prc_openloops_t), intent(in) :: object
    integer, intent(in), optional :: unit
    call msg_message (unit = unit, string = "OpenLoops")
  end subroutine prc_openloops_write

  subroutine prc_openloops_prepare_library (object, os_data, model, var_list)
    class(prc_openloops_t), intent(inout) :: object
    type(os_data_t), intent(in) :: os_data
    type(model_data_t), intent(in), target :: model
    type(var_list_t), intent(in) :: var_list
    integer :: verbosity
    call object%load_driver (os_data)
    call object%reset_parameters ()
    call object%set_particle_properties (model)
    !!!call object%set_alpha_qed (model)
    call object%set_electroweak_parameters (model)
    verbosity = var_list%get_ival (var_str ("openloops_verbosity"))
    call object%set_verbosity (verbosity)
  end subroutine prc_openloops_prepare_library

  subroutine prc_openloops_load_driver (object, os_data)
    class(prc_openloops_t), intent(inout) :: object
    type(os_data_t), intent(in) :: os_data
    logical :: success
    select type (driver => object%driver)
    type is (openloops_driver_t)
       call driver%load (os_data, success)
       call driver%load_sc_procedure (os_data, success)
    end select
  end subroutine prc_openloops_load_driver

  subroutine prc_openloops_start (object)
    class(prc_openloops_t), intent(inout) :: object
    integer :: ierr
    select type (driver => object%driver)
    type is (openloops_driver_t) 
       call driver%blha_olp_start (char (driver%olp_file)//c_null_char, ierr)
    end select
  end subroutine prc_openloops_start

  subroutine prc_openloops_set_n_external (object, n)
    class(prc_openloops_t), intent(inout) :: object
    integer, intent(in) :: n
    N_EXTERNAL = n
  end subroutine prc_openloops_set_n_external

  subroutine prc_openloops_reset_parameters (object)
    class(prc_openloops_t), intent(inout) :: object
    integer :: ierr
    select type (driver => object%driver)
    type is (openloops_driver_t)
       call driver%blha_olp_set_parameter ('mass(5)'//c_null_char, &
          dble(openloops_default_bmass), 0._double, ierr) 
       call driver%blha_olp_set_parameter ('mass(6)'//c_null_char, &
          dble(openloops_default_topmass), 0._double, ierr)        
       call driver%blha_olp_set_parameter ('width(6)'//c_null_char, &
          dble(openloops_default_topwidth), 0._double, ierr) 
       call driver%blha_olp_set_parameter ('mass(23)'//c_null_char, &
          dble(openloops_default_zmass), 0._double, ierr)
       call driver%blha_olp_set_parameter ('width(23)'//c_null_char, &
          dble(openloops_default_zwidth), 0._double, ierr) 
       call driver%blha_olp_set_parameter ('mass(24)'//c_null_char, &
          dble(openloops_default_wmass), 0._double, ierr) 
       call driver%blha_olp_set_parameter ('width(24)'//c_null_char, &
          dble(openloops_default_wwidth), 0._double, ierr) 
       call driver%blha_olp_set_parameter ('mass(25)'//c_null_char, &
          dble(openloops_default_higgsmass), 0._double, ierr) 
       call driver%blha_olp_set_parameter ('width(25)'//c_null_char, &
          dble(openloops_default_higgswidth), 0._double, ierr) 
    end select
  end subroutine prc_openloops_reset_parameters

  subroutine prc_openloops_set_verbosity (object, verbose)
    class(prc_openloops_t), intent(inout) :: object
    integer, intent(in) :: verbose
    integer :: ierr
    select type (driver => object%driver)
    type is (openloops_driver_t)
       call driver%blha_olp_set_parameter ('verbose'//c_null_char, &
          dble(verbose), 0._double, ierr)   
    end select
  end subroutine prc_openloops_set_verbosity

  subroutine prc_openloops_compute_sqme_born &
         (object, i_born, p, mu, sqme, bad_point)
    class(prc_openloops_t), intent(inout) :: object
    integer, intent(in) :: i_born
    type(vector4_t), dimension(:), intent(in) :: p
    real(default), intent(in) :: mu
    real(default), intent(out) :: sqme
    logical, intent(out) :: bad_point
    real(double), dimension(5*object%n_particles) :: mom
    real(default) :: acc_born 
    real(double), dimension(blha_result_array_size (object%n_particles, &
                                                        BLHA_AMP_TREE)) :: r
    real(double) :: mu_dble
    real(double) :: acc_dble
    real(default) :: alpha_s
    
    mom = object%create_momentum_array (p) 
    mu_dble = dble(mu)    
    alpha_s = object%qcd%alpha%get (mu)

    select type (driver => object%driver)
    type is (openloops_driver_t)
       call driver%set_alpha_s (alpha_s)
       if (allocated (object%i_born)) then
          call driver%blha_olp_eval2 (object%i_born(i_born), mom, mu_dble, r, acc_dble)
          sqme = r(1)
       else
          sqme = 0._default
          acc_dble = 0._default
       end if
    end select
    acc_born = acc_dble
    bad_point = acc_born > object%maximum_accuracy
  end subroutine prc_openloops_compute_sqme_born

  subroutine prc_openloops_compute_sqme_real &
         (object, i_flv, p, ren_scale, sqme, bad_point)
    class(prc_openloops_t), intent(inout) :: object
    integer, intent(in) :: i_flv
    type(vector4_t), intent(in), dimension(:) :: p
    real(default), intent(in) :: ren_scale
    real(default), intent(out) :: sqme
    logical, intent(out) :: bad_point
    real(default) :: mu
    real(double), dimension(5*object%n_particles) :: mom
    real(double), dimension(blha_result_array_size (object%n_particles, &
                                                    BLHA_AMP_TREE)) :: r
    real(double) :: mu_dble
    real(double) :: acc_dble
    real(default) :: acc
    real(default) :: alpha_s
 
    mom = object%create_momentum_array (p)
    if (vanishes (ren_scale)) then
       mu = sqrt (two * p(1) * p(2))
    else
       mu = ren_scale
    end if
    mu_dble = dble (mu)

    alpha_s = object%qcd%alpha%get (mu)
    select type (driver => object%driver)
    type is (openloops_driver_t)
       call driver%set_alpha_s (alpha_s)
       call driver%blha_olp_eval2 (object%i_real(i_flv), mom, &
                                    mu_dble, r, acc_dble)
       sqme = r(1)
    end select
    acc = acc_dble
    if (acc > object%maximum_accuracy) bad_point = .true.
  end subroutine prc_openloops_compute_sqme_real

  subroutine prc_openloops_compute_sqme_sc (object, &
                i_flv, em, p, ren_scale_in, pol_vects, &
            me_sc, bad_point)
    class(prc_openloops_t), intent(inout) :: object
    integer, intent(in) :: i_flv
    integer, intent(in) :: em
    type(vector4_t), intent(in), dimension(:) :: p
    real(default), intent(in) :: ren_scale_in
    type(vector4_t), dimension(:) :: pol_vects
    complex(default), intent(out) :: me_sc
    logical, intent(out) :: bad_point
    real(double), dimension(5*N_EXTERNAL) :: mom
    real(double), dimension(N_EXTERNAL) :: r
    real(default) :: ren_scale, alpha_s
    real(double), dimension(4) :: polvect
    integer :: i
    
    mom = object%create_momentum_array (p)
    me_sc = zero
    if (vanishes (ren_scale_in)) then
       ren_scale = sqrt (two * p(1) * p(2))
    else
       ren_scale = ren_scale_in
    end if
    alpha_s = object%qcd%alpha%get (ren_scale)

    forall(i=1:4) polvect(i) = pol_vects(em)%p(i-1)

    select type (driver => object%driver)
    type is (openloops_driver_t)
       call driver%set_alpha_s (alpha_s)
       call driver%evaluate_spin_correlations (1, mom, em, polvect, r)
    end select
    do i = 1, N_EXTERNAL
       if (i /= em) me_sc = me_sc + r(i)
    end do

    me_sc = me_sc / CA
    bad_point = .false.

  end subroutine prc_openloops_compute_sqme_sc


end module prc_openloops
