#!/usr/bin/perl -w
######################################################################
#
# $Id: WebJob-TimeRoutines.t,v 1.9 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2007-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Tests for WebJob::TimeRoutines.
#
######################################################################

use 5.008;
use strict;
use Test;

BEGIN
{
  my %hPlan =
  (
    'tests' => 1 + 1 + 1 + 1 + 14 + 57 + 57,
  );
  plan(%hPlan);
};

######################################################################
#
# Tests
#
######################################################################

  ####################################################################
  #
  # Test: The target module must load without error.
  #
  ####################################################################

  my (@aPairs, $sActualResult, $sDateTime, $sDhms, $sInput, $sSeconds);

  use WebJob::TimeRoutines;
  ok(1);

  ####################################################################
  #
  # Test: The UNIX epoch yields 0.
  #
  ####################################################################

  $sSeconds = DateTimeToSeconds("1970-01-01 00:00:00", 1);
  ok((defined($sSeconds) && $sSeconds == 0));

  ####################################################################
  #
  # Test: 0 yields the UNIX epoch.
  #
  ####################################################################

  $sDateTime = SecondsToDateTime(0, 1);
  ok((defined($sDateTime) && $sDateTime =~ /^1970-01-01 00:00:00$/));

  ####################################################################
  #
  # Test: undef yields "?".
  #
  ####################################################################

  $sDhms = SecondsToDhms(undef);
  ok((defined($sDhms) && $sDhms =~ /^\?$/));

  ####################################################################
  #
  # Test: Various integer inputs yield expected results.
  #
  ####################################################################

  @aPairs =
  (
    "-2:?",
    "-1:?",
    "0:0s",
    "1:1s",
    "2:2s",
    "60:1m0s",
    "61:1m1s",
    "3600:1h0m0s",
    "3660:1h1m0s",
    "3661:1h1m1s",
    "86400:1d0h0m0s",
    "90000:1d1h0m0s",
    "90060:1d1h1m0s",
    "90061:1d1h1m1s",
  );

  foreach my $sPair (@aPairs)
  {
    my ($sSeconds, $sExpectedResult) = split(/:/, $sPair);
    $sDhms = SecondsToDhms($sSeconds);
    ok((defined($sDhms) && $sDhms =~ /^$sExpectedResult$/));
  }

  ####################################################################
  #
  # Test: Various seconds to ISO 8601-ish duration conversions.
  #
  ####################################################################

  @aPairs =
  (
    "0:PT0S",
    "1:PT1S",
    "2:PT2S",
    "60:PT1M",
    "61:PT1M1S",
    "3600:PT1H",
    "3601:PT1H1S",
    "3660:PT1H1M",
    "3661:PT1H1M1S",
    "86400:P1D",
    "86401:P1DT1S",
    "86460:P1DT1M",
    "86461:P1DT1M1S",
    "90000:P1DT1H",
    "90001:P1DT1H1S",
    "90060:P1DT1H1M",
    "90061:P1DT1H1M1S",
    "2592000:P1M",
    "2592001:P1MT1S",
    "2592060:P1MT1M",
    "2592061:P1MT1M1S",
    "2595600:P1MT1H",
    "2595601:P1MT1H1S",
    "2595660:P1MT1H1M",
    "2595661:P1MT1H1M1S",
    "2678400:P1M1D",
    "2678401:P1M1DT1S",
    "2678460:P1M1DT1M",
    "2678461:P1M1DT1M1S",
    "2682000:P1M1DT1H",
    "2682001:P1M1DT1H1S",
    "2682060:P1M1DT1H1M",
    "2682061:P1M1DT1H1M1S",
    "31536000:P1Y",
    "31536001:P1YT1S",
    "31536060:P1YT1M",
    "31536061:P1YT1M1S",
    "31539600:P1YT1H",
    "31539601:P1YT1H1S",
    "31539660:P1YT1H1M",
    "31539661:P1YT1H1M1S",
    "34128000:P1Y1M",
    "34128001:P1Y1MT1S",
    "34128060:P1Y1MT1M",
    "34128061:P1Y1MT1M1S",
    "34131600:P1Y1MT1H",
    "34131601:P1Y1MT1H1S",
    "34131660:P1Y1MT1H1M",
    "34131661:P1Y1MT1H1M1S",
    "34214400:P1Y1M1D",
    "34214401:P1Y1M1DT1S",
    "34214460:P1Y1M1DT1M",
    "34214461:P1Y1M1DT1M1S",
    "34218000:P1Y1M1DT1H",
    "34218001:P1Y1M1DT1H1S",
    "34218060:P1Y1M1DT1H1M",
    "34218061:P1Y1M1DT1H1M1S",
  );

  foreach my $sPair (@aPairs)
  {
    my ($sInput, $sExpectedResult) = split(/:/, $sPair);
    $sActualResult = SecondsToDuration($sInput);
    ok((defined($sActualResult) && $sActualResult =~ /^$sExpectedResult$/));
  }

  ####################################################################
  #
  # Test: Various ISO 8601-ish duration to seconds conversions.
  #
  ####################################################################

  foreach my $sPair (@aPairs)
  {
    my ($sExpectedResult, $sInput) = split(/:/, $sPair);
    $sActualResult = DurationToSeconds($sInput);
    ok((defined($sActualResult) && $sActualResult == $sExpectedResult));
  }

