######################################################################
#
# $Id: VersionRoutines.pm,v 1.6 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2010-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Version routines
#
######################################################################

package WebJob::VersionRoutines;

require Exporter;

use 5.008;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use WebJob::ValidationRoutines;

@EXPORT = qw(VersionIsInRange VersionNumberToString VersionParseOutput VersionStringToNumber);

@EXPORT_OK = ();
@ISA = qw(Exporter);
$VERSION = do { my @r = (q$Revision: 1.6 $ =~ /(\d+)/g); sprintf("%d."."%03d" x $#r, @r); };

######################################################################
#
# Version Scheme:
#
#   The version scheme allocates 4 bits to the major number, 8 bits
#   to the minor number, 8 bits to the patch number, 2 bits to the
#   state number, and 10 bits to the build number.
#
#   +----+--------+--------+--+----------+
#   |3322|22222222|11111111|11|          |
#   |1098|76543210|98765432|10|9876543210|
#   |----+--------+--------|--|----------+
#   |MMMM|mmmmmmmm|pppppppp|ss|bbbbbbbbbb|
#   +----+--------+--------|--+----------+
#    ^^^^ ^^^^^^^^ ^^^^^^^^ ^^ ^^^^^^^^^^
#       |        |       |   |          |
#       |        |       |   |          +-----> b - build (0...1023)
#       |        |       |   +----------------> s - state (0......3)
#       |        |       +--------------------> p - patch (0....255)
#       |        +----------------------------> m - minor (0....255)
#       +-------------------------------------> M - major (0.....15)
#
# State Numbers:
#
#     00 = ds --> Development Snapshot
#     01 = rc --> Release Candidate
#     10 = sr --> Standard Release
#     11 = xs --> eXtended Snapshot
#
######################################################################

######################################################################
#
# VersionIsInRange
#
######################################################################

sub VersionIsInRange
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phPArgs,
    'Keys' =>
    [
#     'MaxVersion', # Optional
#     'MinVersion', # Optional
      'VersionString',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'} if (defined($phPArgs));
    return undef;
  }

  ####################################################################
  #
  # Assign default values where necessary.
  #
  ####################################################################

  my ($sMaxVersion, $sMinVersion);

  $sMaxVersion = $$phPArgs{'MaxVersion'} || 0xffffffff;

  $sMinVersion = $$phPArgs{'MinVersion'} || 0x00000000;

  ####################################################################
  #
  # Make sure we have the necessary prerequisites.
  #
  ####################################################################

  if ($sMaxVersion < 0x00000000 || $sMaxVersion > 0xffffffff)
  {
    $$phPArgs{'Error'} = "Max value is out of range.";
    return undef;
  }

  if ($sMinVersion < 0x00000000 || $sMinVersion > 0xffffffff)
  {
    $$phPArgs{'Error'} = "Min value is out of range.";
    return undef;
  }

  if ($sMinVersion > $sMaxVersion)
  {
    $$phPArgs{'Error'} = "Min value exceeds max value.";
    return undef;
  }

  ####################################################################
  #
  # Parse the version string.
  #
  ####################################################################

  my ($sVersionNumber);

  %hLArgs =
  (
    'VersionString' => $$phPArgs{'VersionString'},
  );
  $sVersionNumber = VersionStringToNumber(\%hLArgs);
  if (!defined($sVersionNumber))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'};
    return undef;
  }

  ####################################################################
  #
  # Make sure the value falls within the defined limits.
  #
  ####################################################################

  if ($sVersionNumber >= $sMinVersion && $sVersionNumber <= $sMaxVersion)
  {
    return 1;
  }
  else
  {
    $$phPArgs{'Error'} = "Value not within the defined limits.";
    return 0;
  }
}


######################################################################
#
# VersionNumberToString
#
######################################################################

sub VersionNumberToString
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phPArgs,
    'Keys' =>
    [
#     'Template',    # Optional
#     'StateFormat', # Optional
      'VersionNumber',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'} if (defined($phPArgs));
    return undef;
  }

  ####################################################################
  #
  # Assign default values where necessary.
  #
  ####################################################################

  my (%hStateStrings, $sStateFormat, $sVersionNumber, $sVersionString);

  $sStateFormat = (exists($$phPArgs{'StateFormat'}) && $$phPArgs{'StateFormat'} =~ /^(lower|number|upper)$/) ? $1 : "lower";

  %hStateStrings =
  (
    0 => 'ds',
    1 => 'rc',
    2 => 'sr',
    3 => 'xs',
  );

  $sVersionNumber = $$phPArgs{'VersionNumber'};

  $sVersionString = $$phPArgs{'Template'} || "%major.%minor.%patch (%state%build)";

  ####################################################################
  #
  # Make sure we have the necessary prerequisites.
  #
  ####################################################################

  if ($sVersionNumber < 0 || $sVersionNumber > 0xffffffff)
  {
    $$phPArgs{'Error'} = "Value out of range.";
    return undef;
  }

  ####################################################################
  #
  # Break the version number into its elements.
  #
  ####################################################################

  my (%hVersionStrings, $sStateNumber);

  $hVersionStrings{'major'} = ($sVersionNumber >> 28) & 0x0f;
  $hVersionStrings{'minor'} = ($sVersionNumber >> 20) & 0xff;
  $hVersionStrings{'patch'} = ($sVersionNumber >> 12) & 0xff;
  $sStateNumber = ($sVersionNumber >> 10) & 0x03;
  if ($sStateFormat eq "number")
  {
    $hVersionStrings{'state'} = $sStateNumber;
  }
  elsif ($sStateFormat eq "upper")
  {
    $hVersionStrings{'state'} = uc($hStateStrings{$sStateNumber});
  }
  else
  {
    $hVersionStrings{'state'} = $hStateStrings{$sStateNumber};
  }
  $hVersionStrings{'build'} = $sVersionNumber & 0x3ff;

  ####################################################################
  #
  # Format the version string according to the specified template.
  #
  ####################################################################

  my ($sTokenCount);

  foreach my $sString (reverse(sort(keys(%hVersionStrings))))
  {
    $sVersionString =~ s/%$sString/$hVersionStrings{$sString}/g;
  }
  $sTokenCount = 0;
  foreach my $sCharacter (split(//, $sVersionString))
  {
    $sTokenCount++ if ($sCharacter eq "%");
  }
  if ($sTokenCount % 2)
  {
    $$phPArgs{'Error'} = "One or more unresolved tokens.";
    return undef;
  }
  $sVersionString =~ s/%%/%/g;

  return $sVersionString;
}


######################################################################
#
# VersionParseOutput
#
######################################################################

sub VersionParseOutput
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phPArgs,
    'Keys' =>
    [
      'VersionOutput',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'} if (defined($phPArgs));
    return undef;
  }

  ####################################################################
  #
  # Assign default values where necessary.
  #
  ####################################################################

  # EMPTY

  ####################################################################
  #
  # Make sure we have the necessary prerequisites.
  #
  ####################################################################

  # EMPTY

  ####################################################################
  #
  # Parse the version string. This routine recognizes version output
  # produced by running the following command:
  #
  #   webjob --version
  #
  ####################################################################

  my ($sVersionNumber);

  if ($$phPArgs{'VersionOutput'} =~ /^(webjob) (\d+[.]\d+[.]\d+(?: [(](?:ds|rc|sr|xs)\d+[)])?) (32|64)-bit(?: ([\w (),.-]+))?$/)
  {
    $$phPArgs{'Program'} = $1;
    $$phPArgs{'VersionString'} = $2;
    $$phPArgs{'Bits'} = $3;
    $$phPArgs{'CapabilityList'} = $4;
    foreach my $sCapability (split(",", $$phPArgs{'CapabilityList'}))
    {
      if (defined($sCapability) && $sCapability =~ /^([\w-]+)(?:[(]([^()]+)[)])?$/)
      {
        $$phPArgs{'Capabilities'}{lc($1)} = (defined($2)) ? $2 : "";
      }
    }
  }
  else
  {
    $$phPArgs{'Error'} = "Unrecognized version output.";
    return undef;
  }

  %hLArgs =
  (
    'VersionString' => $$phPArgs{'VersionString'},
  );
  $sVersionNumber = VersionStringToNumber(\%hLArgs);
  if (!defined($sVersionNumber))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'};
    return undef;
  }
  $$phPArgs{'VersionNumber'} = $sVersionNumber;
  $$phPArgs{'Major'} = $hLArgs{'Major'};
  $$phPArgs{'Minor'} = $hLArgs{'Minor'};
  $$phPArgs{'Patch'} = $hLArgs{'Patch'};
  $$phPArgs{'State'} = $hLArgs{'State'};
  $$phPArgs{'Build'} = $hLArgs{'Build'};

  return $sVersionNumber;
}


######################################################################
#
# VersionStringToNumber
#
######################################################################

sub VersionStringToNumber
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phPArgs,
    'Keys' =>
    [
      'VersionString',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'} if (defined($phPArgs));
    return undef;
  }

  ####################################################################
  #
  # Assign default values where necessary.
  #
  ####################################################################

  my (%hStateNumbers, $sVersionString);

  %hStateNumbers =
  (
    'ds' => 0,
    'rc' => 1,
    'sr' => 2,
    'xs' => 3,
  );

  $sVersionString = $$phPArgs{'VersionString'};

  ####################################################################
  #
  # Make sure we have the necessary prerequisites.
  #
  ####################################################################

  # EMPTY

  ####################################################################
  #
  # Parse the version string. This routine recognizes the following
  # version formats:
  #
  #   N.N.N (SSN) --> EXE format (e.g., "webjob 1.9.0 (ds9)")
  #   N.N.N.SSN   --> TAR format (e.g., "webjob-1.9.0.ds9.tgz")
  #   N_N_N_SSN   --> CVS format (e.g., "V1_9_0_DS9")
  #
  ####################################################################

  my ($sMajor, $sMinor, $sPatch, $sState, $sBuild);

  $sVersionString =~ s/[)(]//g; # Drop parentheses.
  if ($sVersionString =~ /(\d+)([._])(\d+)\2(\d+)(?:[ ._](ds|rc|sr|xs)(\d+))?/i)
  {
    $sMajor = $1;
    $sMinor = $3;
    $sPatch = $4;
    # NOTE: If the state/build numbers are missing, assume it's a standard release version.
    $sState = (defined($5) && exists($hStateNumbers{lc($5)})) ? $hStateNumbers{lc($5)} : 2;
    $sBuild = (defined($6)) ? $6 : 0;
  }
  else
  {
    $$phPArgs{'Error'} = "Unrecognized version string.";
    return undef;
  }

  ####################################################################
  #
  # Construct the version number.
  #
  ####################################################################

  my ($sVersionNumber);

  $sVersionNumber = 0;
  $sVersionNumber |= $sMajor << 28;
  $sVersionNumber |= $sMinor << 20;
  $sVersionNumber |= $sPatch << 12;
  $sVersionNumber |= $sState << 10;
  $sVersionNumber |= $sBuild;

  $$phPArgs{'Major'} = $sMajor;
  $$phPArgs{'Minor'} = $sMinor;
  $$phPArgs{'Patch'} = $sPatch;
  $$phPArgs{'State'} = $sState;
  $$phPArgs{'Build'} = $sBuild;

  return $sVersionNumber;
}

1;

__END__

=pod

=head1 NAME

WebJob::VersionRoutines - Version routines

=head1 SYNOPSIS

    use WebJob::VersionRoutines;

=head1 DESCRIPTION

This module is a collection of version routines designed to support
various WebJob server-side utilities. As such, minimal effort was put
into supporting this code for general consumption. In other words, use
at your own risk and don't expect the interface to remain the same or
backwards compatible from release to release. This module does not
provide an OO interface, nor will it do so anytime soon.

=head1 AUTHOR

Klayton Monroe

=head1 LICENSE

All documentation and code are distributed under same terms and
conditions as B<WebJob>.

=cut
