#!/usr/bin/perl
######################################################################
#
# $Id: nph-webjob.cgi,v 1.15 2004/03/29 22:07:01 mavrik Exp $
#
######################################################################
#
# Copyright 2001-2004 Klayton Monroe, All Rights Reserved.
#
######################################################################

use strict;

######################################################################
#
# Main Routine
#
######################################################################

  my (%hProperties, %hReturnCodes, $sLocalError);

  %hReturnCodes =
  (
    '200' => "OK",
    '251' => "Link Test OK",
    '404' => "Not Found",
    '405' => "Method Not Allowed",
    '450' => "Invalid Query",
    '451' => "File Already Exists",
    '452' => "Username Undefined",
    '453' => "Username-ClientId Mismatch",
    '454' => "Content-Length Undefined",
    '455' => "Content-Length Exceeds Limit",
    '456' => "Content-Length Mismatch",
    '457' => "File Not Available",
    '500' => "Internal Server Error",
    '550' => "Internal Server Initialization Error",
  );

  ####################################################################
  #
  # Punch in and go to work.
  #
  ####################################################################

  $hProperties{'StartTime'} = time;

  ####################################################################
  #
  # Create/Verify run time environment, and process GET/PUT requests.
  #
  ####################################################################

  if (!defined(CreateRunTimeEnvironment(\%hProperties, \$sLocalError)))
  {
    $hProperties{'ReturnStatus'} = 550;
    $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
    $hProperties{'ErrorMessage'} = $sLocalError;
  }
  else
  {
    if ($hProperties{'RequestMethod'} eq "GET")
    {
      $hProperties{'ReturnStatus'} = ProcessGetRequest(\%hProperties, \$sLocalError);
      $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
      $hProperties{'ErrorMessage'} = $sLocalError;
    }
    elsif ($hProperties{'RequestMethod'} eq "PUT")
    {
      $hProperties{'ReturnStatus'} = ProcessPutRequest(\%hProperties, \$sLocalError);
      $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
      $hProperties{'ErrorMessage'} = $sLocalError;
    }
    else
    {
      $hProperties{'ReturnStatus'} = 405;
      $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
      $hProperties{'ErrorMessage'} = "Method ($hProperties{'RequestMethod'}) not allowed";
    }
  }
  $hProperties{'ServerContentLength'} = SendResponse(\%hProperties);

  ####################################################################
  #
  # Clean up and go home.
  #
  ####################################################################

  $hProperties{'StopTime'} = time;

  if ($hProperties{'EnableLogging'} =~ /^[Yy]$/)
  {
    LogMessage(\%hProperties);
  }

  1;


######################################################################
#
# CreateRunTimeEnvironment
#
######################################################################

sub CreateRunTimeEnvironment
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Put input/output streams in binary mode.
  #
  ####################################################################

  foreach my $sHandle (\*STDIN, \*STDOUT, \*STDERR)
  {
    binmode($sHandle);
  }

  ####################################################################
  #
  # Initialize regex variables.
  #
  ####################################################################

  my %hRegexes =
  (
    'SOL'       => qq(^),
    'Version'   => qq(VERSION=(webjob[\\w .]{1,64})),
    'System'    => qq(&SYSTEM=([\\w- ()+,./:]{1,64})),
    'ClientId'  => qq(&CLIENTID=([\\w-]{1,64})),
    'Filename'  => qq(&FILENAME=([\\w-+.:]{1,1024})),
    'RunType'   => qq(&RUNTYPE=(linktest|snapshot)),
    'OutLength' => qq(&STDOUT_LENGTH=(\\d{1,20})), # 18446744073709551615
    'ErrLength' => qq(&STDERR_LENGTH=(\\d{1,20})), # 18446744073709551615
    'EnvLength' => qq(&STDENV_LENGTH=(\\d{1,20})), # 18446744073709551615
    'EOL'       => qq(\$),
  );

  $$phProperties{'GETRegex'} =
    $hRegexes{'SOL'} .
    $hRegexes{'Version'} .
    $hRegexes{'System'} .
    $hRegexes{'ClientId'} .
    $hRegexes{'Filename'} .
    $hRegexes{'EOL'}
    ;

  $$phProperties{'PUTRegex'} =
    $hRegexes{'SOL'} .
    $hRegexes{'Version'} .
    $hRegexes{'System'} .
    $hRegexes{'ClientId'} .
    $hRegexes{'Filename'} .
    $hRegexes{'RunType'} .
    $hRegexes{'OutLength'} .
    $hRegexes{'ErrLength'} .
    $hRegexes{'EnvLength'} .
    $hRegexes{'EOL'}
    ;

  $$phProperties{'JobIdRegex'} = qq([\\w-]{1,64}_\\d{10}_\\d{5});

  ####################################################################
  #
  # Initialize environment-specific variables.
  #
  ####################################################################

  $$phProperties{'ContentLength'}  = $ENV{'CONTENT_LENGTH'};
  $$phProperties{'QueryString'}    = $ENV{'QUERY_STRING'};
  $$phProperties{'RemoteAddress'}  = $ENV{'REMOTE_ADDR'};
  $$phProperties{'RemoteUser'}     = $ENV{'REMOTE_USER'};
  $$phProperties{'RequestMethod'}  = $ENV{'REQUEST_METHOD'};
  $$phProperties{'ServerSoftware'} = $ENV{'SERVER_SOFTWARE'};
  $$phProperties{'PropertiesFile'} = $ENV{'WEBJOB_PROPERTIES_FILE'};

  ####################################################################
  #
  # Initialize platform-specific variables.
  #
  ####################################################################

  if ($^O =~ /MSWin32/i)
  {
    $$phProperties{'OSClass'} = "WINDOWS";
    $$phProperties{'Newline'} = "\r\n";
  }
  else
  {
    $$phProperties{'OSClass'} = "UNIX";
    $$phProperties{'Newline'} = "\n";
    umask(022);
  }

  ####################################################################
  #
  # Initialize site-specific variables.
  #
  ####################################################################

  my (%hSiteProperties, $sLocalError);

  %hSiteProperties =
  (
    'BaseDirectory'     => qq(^(?:[A-Za-z]:)?/[\\w-./]+\$),
    'CapContentLength'  => qq(^[YyNn]\$),
    'EnableJobIds'      => qq(^[YyNn]\$),
    'EnableLogging'     => qq(^[YyNn]\$),
    'FolderList'        => qq(^[\\w-.]+(?::[\\w-.]+)*\$),
    'UseGMT'            => qq(^[YyNn]\$),
    'MaxContentLength'  => qq(^\\d{1,20}\$), # 18446744073709551615
    'PutNameFormat'     => qq(^[\\w-%+./:]+\$),
    'RequireMatch'      => qq(^[YyNn]\$),
    'RequireUser'       => qq(^[YyNn]\$),
    'ServerId'          => qq(^[\\w-]{1,64}\$)
  );

  GetSiteProperties($phProperties, \%hSiteProperties, \$sLocalError);

  ####################################################################
  #
  # Initialize derived variables.
  #
  ####################################################################

  $$phProperties{'IncomingDirectory'} = $$phProperties{'BaseDirectory'} . "/incoming";
  $$phProperties{'LogfilesDirectory'} = $$phProperties{'BaseDirectory'} . "/logfiles";
  $$phProperties{'ProfilesDirectory'} = $$phProperties{'BaseDirectory'} . "/profiles";
  $$phProperties{'LogFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-webjob.log";

  ####################################################################
  #
  # Verify run time environment.
  #
  ####################################################################

  if (!defined(VerifyRunTimeEnvironment($phProperties, \%hSiteProperties, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Conditionally, initialize and verify the job id.
  #
  ####################################################################

  if ($$phProperties{'EnableJobIds'} =~ /^[Yy]$/)
  {
    if ($$phProperties{'RequestMethod'} eq "GET")
    {
      $$phProperties{'JobId'} = sprintf("%s_%010u_%05d", $$phProperties{'ServerId'}, $$phProperties{'StartTime'}, $$);
    }
    else
    {
      if (exists($ENV{'HTTP_JOB_ID'}))
      {
        if ($ENV{'HTTP_JOB_ID'} =~ /^($$phProperties{'JobIdRegex'})$/)
        {
          $$phProperties{'JobId'} = $1;
        }
        else
        {
          $$psError = "HTTP_JOB_ID ($ENV{'HTTP_JOB_ID'}) is undefined or invalid";
          return undef;
        }
      }
      else
      {
        $$phProperties{'JobId'} = "NA"; # Not Assigned.
      }
    }
  }
  else
  {
    $$phProperties{'JobId'} = "NR"; # Not Required.
  }
  if (!defined($$phProperties{'JobId'}) || $$phProperties{'JobId'} !~ /^(NA|NR|$$phProperties{'JobIdRegex'})$/)
  {
    $$psError = "JobId ($$phProperties{'JobId'}) is undefined or invalid";
    return undef;
  }

  1;
}


######################################################################
#
# GetKeysAndValues
#
######################################################################

sub GetKeysAndValues
{
  my ($sFile, $phValidKeys, $phKeyValuePairs, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if (!defined($sFile) || !defined($phValidKeys) || !defined($phKeyValuePairs))
  {
    $$psError = "Unable to proceed due to missing or undefined inputs" if (defined($psError));
    return undef;
  }

  ####################################################################
  #
  # Open properties file.
  #
  ####################################################################

  if (!open(FH, "<$sFile"))
  {
    $$psError = "File ($sFile) could not be opened ($!)" if (defined($psError));
    return undef;
  }

  ####################################################################
  #
  # Read properties file. Ignore comments and blank lines.
  #
  ####################################################################

  while (my $sLine = <FH>)
  {
    $sLine =~ s/[\r\n]+$//; # Remove CRs and LFs.
    $sLine =~ s/#.*$//; # Remove comments.
    if ($sLine !~ /^\s*$/)
    {
      my ($sKey, $sValue) = ($sLine =~ /^([^=]*)=(.*)$/);
      $sKey =~ s/^\s+//; # Remove leading whitespace.
      $sKey =~ s/\s+$//; # Remove trailing whitespace.
      $sValue =~ s/^\s+//; # Remove leading whitespace.
      $sValue =~ s/\s+$//; # Remove trailing whitespace.
      if (defined($sKey) && length($sKey))
      {
        foreach my $sKnownKey (keys(%$phValidKeys))
        {
          if ($sKey =~ /^$sKnownKey$/i)
          {
            $$phKeyValuePairs{$sKnownKey} = $sValue;
          }
        }
      }
    }
  }
  close(FH);

  1;
}


######################################################################
#
# GetSiteProperties
#
######################################################################

sub GetSiteProperties
{
  my ($phProperties, $phSiteProperties, $psError) = @_;

  ####################################################################
  #
  # BaseDirectory is the epicenter of activity.
  #
  ####################################################################

  $$phProperties{'BaseDirectory'} = "/integrity";

  ####################################################################
  #
  # FolderList specifies locations where shared programs can be found.
  # If a requested file does not exist in a given client's commands
  # directory, the FolderList is searched according to the order given
  # here. The list delimiter is a colon (e.g., "common:shared").
  #
  ####################################################################

  $$phProperties{'FolderList'} = "common";

  ####################################################################
  #
  # PutNameFormat controls how files are named/saved in the incoming
  # directory. In other words, it controls the directory's layout.
  # Basically, PutNameFormat is a format string consisting of one or
  # more conversion specifications optionally interspersed with zero
  # or more plain text characters. The following conversion
  # specifications are supported:
  #
  #   %CID = Client ID as a string
  #   %CMD = Command as a string
  #   %d   = Day of the month as a decimal number (01-31)
  #   %H   = Hour as a decimal number (00-23)
  #   %IP  = IP address as a dotted quad string
  #   %M   = Minute as a decimal number (00-59)
  #   %m   = Month as a decimal number (01-12)
  #   %PID = Process ID of server-side CGI script
  #   %S   = Second as a decimal number (00-60)
  #   %s   = Number of seconds since the Epoch
  #   %Y   = Year with century as a decimal number
  #
  # For example, the following format string:
  #
  #   "%CMD/%IP_%Y-%m-%d_%H.%M.%S"
  #
  # will cause uploaded files to be stored in sub-directories that
  # correspond to the name of the command executed, and each output
  # filename will consist of an IP address, date, and time.
  #
  # The added flexibility provided by this scheme means that it is
  # possible to create format strings that are problematic. Consider
  # the following string:
  #
  #   "%CID/%CMD"
  #
  # While this is a legal format string, it is likely to cause name
  # collisions (e.g., the same client runs the same command two or
  # more times). Therefore, it is important to create format strings
  # that contain enough job specific information to distinguish one
  # set of uploaded files from another.
  #
  ####################################################################

  $$phProperties{'PutNameFormat'} = "%CID_%Y%m%d%H%M%S_%PID_%CMD";

  ####################################################################
  #
  # RequireUser forces the script to abort unless RemoteUser has been
  # set.
  #
  ####################################################################

  $$phProperties{'RequireUser'} = "Y"; # [Y|N]

  ####################################################################
  #
  # RequireMatch forces the script to abort unless ClientId matches
  # RemoteUser. When this value is disabled, any authenticated user
  # will be allowed to issue requests for a given client. Disabling
  # RequireUser implicitly disables RequireMatch.
  #
  ####################################################################

  $$phProperties{'RequireMatch'} = "Y"; # [Y|N]

  ####################################################################
  #
  # CapContentLength forces the script to abort when ContentLength
  # exceeds MaxContentLength.
  #
  ####################################################################

  $$phProperties{'CapContentLength'} = "N"; # [Y|N]
  $$phProperties{'MaxContentLength'} = 100000000; # 100 MB

  ####################################################################
  #
  # When active, EnableLogging forces the script to generate a log
  # message for each request. If the designated LogFile can not be
  # opened, the log message will be written to STDERR.
  #
  ####################################################################

  $$phProperties{'EnableLogging'} = "Y"; # [Y|N]

  ####################################################################
  #
  # When active, UseGMT forces the script to convert all time values
  # to GMT. Otherwise, time values are converted to local time.
  #
  ####################################################################

  $$phProperties{'UseGMT'} = "N"; # [Y|N]

  ####################################################################
  #
  # When active, EnableJobIds forces the script to generate a job id
  # for each GET request. EnableJobIds will also force the script to
  # abort if a PUT request does not contain a valid job id.
  #
  ####################################################################

  $$phProperties{'EnableJobIds'} = "Y"; # [Y|N]

  ####################################################################
  #
  # ServerId specifies the identity assigned to the WebJob server.
  #
  ####################################################################

  $$phProperties{'ServerId'} = "server_1";

  ####################################################################
  #
  # Pull in any externally defined properties. These properties trump
  # internally defined properties.
  #
  ####################################################################

  GetKeysAndValues($$phProperties{'PropertiesFile'}, $phSiteProperties, $phProperties, undef);

  1;
}


######################################################################
#
# LogMessage
#
######################################################################

sub LogMessage
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Create date/time stamp and calculate duration.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings
  ) = ($$phProperties{'UseGMT'} =~ /^[Yy]$/) ? gmtime($$phProperties{'StopTime'}) : localtime($$phProperties{'StopTime'});

  $$phProperties{'DateTime'} = sprintf("%04s-%02s-%02s %02s:%02s:%02s",
    $sYear + 1900,
    $sMonth + 1,
    $sMonthDay,
    $sHour,
    $sMinute,
    $sSecond
    );

  $$phProperties{'Duration'} = $$phProperties{'StopTime'} - $$phProperties{'StartTime'};

  ####################################################################
  #
  # Construct log message.
  #
  ####################################################################

  my (@aLogFields, @aOutputFields, $sLogMessage);

  @aLogFields =
  (
    'DateTime',
    'JobId',
    'RemoteUser',
    'RemoteAddress',
    'RequestMethod',
    'ClientId',
    'ClientFilename',
    'ContentLength',
    'ServerContentLength',
    'Duration',
    'ReturnStatus',
    'ErrorMessage'
  );

  foreach my $sField (@aLogFields)
  {
    my $sValue = $$phProperties{$sField};
    if ($sField =~ /^ErrorMessage$/)
    {
      push(@aOutputFields, ((defined($sValue) && length($sValue)) ? "-- $sValue" : "--"));
    }
    else
    {
      push(@aOutputFields, ((defined($sValue) && length($sValue)) ? "$sValue" : "-"));
    }
  }
  $sLogMessage = join(" ", @aOutputFields);

  ####################################################################
  #
  # Deliver log message.
  #
  ####################################################################

  if (!open(LH, ">>" . $$phProperties{'LogFile'}))
  {
    print STDERR $sLogMessage, $$phProperties{'Newline'};
    return undef;
  }
  binmode(LH);
  print LH $sLogMessage, $$phProperties{'Newline'};
  close(LH);

  1;
}


######################################################################
#
# MakePutName
#
######################################################################

sub MakePutName
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if
  (
    (!exists($$phProperties{'PutNameFormat'})  || !defined($$phProperties{'PutNameFormat'})) ||
    (!exists($$phProperties{'StartTime'})      || !defined($$phProperties{'StartTime'}))     ||
    (!exists($$phProperties{'ClientId'})       || !defined($$phProperties{'ClientId'}))      ||
    (!exists($$phProperties{'RemoteAddress'})  || !defined($$phProperties{'RemoteAddress'})) ||
    (!exists($$phProperties{'ClientFilename'}) || !defined($$phProperties{'ClientFilename'}))
  )
  {
    $$psError = "Unable to proceed due to missing or undefined inputs";
    return undef;
  }

  ####################################################################
  #
  # Create conversion values.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings
  ) = ($$phProperties{'UseGMT'} =~ /^[Yy]$/) ? gmtime($$phProperties{'StartTime'}) : localtime($$phProperties{'StartTime'});

  my %hFormatValues =
  (
    'CID' => $$phProperties{'ClientId'},
    'CMD' => $$phProperties{'ClientFilename'},
    'IP'  => $$phProperties{'RemoteAddress'},
    'Y'   => sprintf("%04d", $sYear + 1900),
    'm'   => sprintf("%02d", $sMonth + 1),
    'd'   => sprintf("%02d", $sMonthDay),
    'H'   => sprintf("%02d", $sHour),
    'M'   => sprintf("%02d", $sMinute),
    'PID' => sprintf("%05d", $$),
    'S'   => sprintf("%02d", $sSecond),
    's'   => sprintf("%010u", $$phProperties{'StartTime'})
  );

  ####################################################################
  #
  # Verify conversion values.
  #
  ####################################################################

  my %hFormatChecks =
  (
    'CID' => qq([\\w-]{1,64}),
    'CMD' => qq([\\w-+.:]{1,1024}),
    'IP'  => qq((?:\\d{1,3}\\.){3}\\d{1,3}),
    'Y'   => qq(\\d{4}),
    'm'   => qq(\\d{2}),
    'd'   => qq(\\d{2}),
    'H'   => qq(\\d{2}),
    'M'   => qq(\\d{2}),
    'PID' => qq(\\d{5}),
    'S'   => qq(\\d{2}),
    's'   => qq(\\d{10})
  );

  foreach my $sKey (keys(%hFormatChecks))
  {
    if ($hFormatValues{$sKey} !~ /^$hFormatChecks{$sKey}$/)
    {
      $$psError = "Conversion value ($hFormatValues{$sKey}) for corresponding specification ($sKey) is not valid";
      return undef;
    }
  }

  ####################################################################
  #
  # Create PutName.
  #
  ####################################################################

  my ($sPutName, $sTokenList);

  $sTokenList = join('|', sort(keys(%hFormatValues)));
  $sPutName = $$phProperties{'PutNameFormat'};
  $sPutName =~ s/%($sTokenList)/$hFormatValues{$1}/ge;

  ####################################################################
  #
  # Verify that at least one conversion took place.
  #
  ####################################################################

  if ($sPutName eq $$phProperties{'PutNameFormat'})
  {
    $$psError = "Format string ($$phProperties{'PutNameFormat'}) must contain at least one conversion specification";
    return undef;
  }

  return $sPutName;
}


######################################################################
#
# MakePutTree
#
######################################################################

sub MakePutTree
{
  my ($sIncomingDirectory, $sPutName, $sMode, $sPopCount, $psError) = @_;

  ####################################################################
  #
  # Pop the specified number of elements from PutName. Normally, only
  # the trailing filename is removed (i.e. PopCount = 1).
  #
  ####################################################################

  my (@aElements);

  @aElements = split(/[\/\\]/, $sPutName);

  while (defined($sPopCount) && $sPopCount-- > 0)
  {
    pop(@aElements);
  }

  ####################################################################
  #
  # Create the tree -- one element at a time.
  #
  ####################################################################

  my ($sPath);

  $sPath = $sIncomingDirectory;

  foreach my $sElement (@aElements)
  {
    $sPath .= "/$sElement";
    if (!-d $sPath)
    {
      if (!mkdir($sPath, $sMode))
      {
        $$psError = "Directory ($sPath) could not be created ($!)";
        return undef;
      }
    }
  }

  1;
}


######################################################################
#
# ProcessGetRequest
#
######################################################################

sub ProcessGetRequest
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Proceed only if QueryString matches GETRegex.
  #
  ####################################################################

  my $sQueryString = URLDecode($$phProperties{'QueryString'});

  if ($sQueryString =~ /$$phProperties{'GETRegex'}/)
  {
    $$phProperties{'ClientVersion'}  = $1;
    $$phProperties{'ClientSystem'}   = $2;
    $$phProperties{'ClientId'}       = $3 || "nobody";
    $$phProperties{'ClientFilename'} = $4;

    ##################################################################
    #
    # Do username and client ID checks.
    #
    ##################################################################

    if ($$phProperties{'RequireUser'} =~ /^[Yy]$/ && (!defined($$phProperties{'RemoteUser'}) || !length($$phProperties{'RemoteUser'})))
    {
      $$psError = "Remote user is undefined or null";
      return 452;
    }

    if ($$phProperties{'RequireUser'} =~ /^[Yy]$/ && $$phProperties{'RequireMatch'} =~ /^[Yy]$/ && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
    {
      $$psError = "Remote user ($$phProperties{'RemoteUser'}) does not match client ID ($$phProperties{'ClientId'})";
      return 453;
    }

    ##################################################################
    #
    # Do content length checks.
    #
    ##################################################################

    if (!defined($$phProperties{'ContentLength'}) || !length($$phProperties{'ContentLength'}))
    {
      $$psError = "Content length is undefined or null";
      return 454;
    }

    if ($$phProperties{'CapContentLength'} =~ /^[Yy]$/ && $$phProperties{'ContentLength'} > $$phProperties{'MaxContentLength'})
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) exceeds maximum allowed length ($$phProperties{'MaxContentLength'})";
      return 455;
    }

    ##################################################################
    #
    # Locate the requested file and serve it up. Start by searching
    # the client's commands directory. Then, move on to the shared
    # folders.
    #
    ##################################################################

    my $sEffectiveFolderList = $$phProperties{'ClientId'} . ":" . $$phProperties{'FolderList'};

    foreach my $sFolder (split(/:/, $sEffectiveFolderList))
    {
      my $sGetFile = $$phProperties{'ProfilesDirectory'} . "/" . $sFolder . "/" . "commands" . "/" . $$phProperties{'ClientFilename'};
      if (-e $sGetFile)
      {
        if (!open(FH, "<$sGetFile"))
        {
          $$psError = "Requested file ($sGetFile) could not be opened ($!)";
          return 457;
        }
        binmode(FH);
        $$phProperties{'ReturnHandle'} = \*FH;
        $$psError = "Success";
        return 200;
      }
    }
    $$psError = "Requested file ($$phProperties{'ClientFilename'}) was not found in effective folder list ($sEffectiveFolderList)";
    return 404;
  }
  else
  {
    $$psError = "Invalid query string ($$phProperties{'QueryString'})";
    return 450;
  }
}


######################################################################
#
# ProcessPutRequest
#
######################################################################

sub ProcessPutRequest
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Proceed only if QueryString matches PUTRegex.
  #
  ####################################################################

  my $sQueryString = URLDecode($$phProperties{'QueryString'});

  if ($sQueryString =~ /$$phProperties{'PUTRegex'}/)
  {
    my ($sEnvLength, $sErrLength, $sOutLength);

    $$phProperties{'ClientVersion'}   = $1;
    $$phProperties{'ClientSystem'}    = $2;
    $$phProperties{'ClientId'}        = $3 || "nobody";
    $$phProperties{'ClientFilename'}  = $4;
    $$phProperties{'ClientRunType'}   = $5;
    $$phProperties{'ClientOutLength'} = $sOutLength = $6;
    $$phProperties{'ClientErrLength'} = $sErrLength = $7;
    $$phProperties{'ClientEnvLength'} = $sEnvLength = $8;

    ##################################################################
    #
    # Do username and client ID checks.
    #
    ##################################################################

    if ($$phProperties{'RequireUser'} =~ /^[Yy]$/ && (!defined($$phProperties{'RemoteUser'}) || !length($$phProperties{'RemoteUser'})))
    {
      $$psError = "Remote user is undefined or null";
      return 452;
    }

    if ($$phProperties{'RequireUser'} =~ /^[Yy]$/ && $$phProperties{'RequireMatch'} =~ /^[Yy]$/ && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
    {
      $$psError = "Remote user ($$phProperties{'RemoteUser'}) does not match client ID ($$phProperties{'ClientId'})";
      return 453;
    }

    ##################################################################
    #
    # Do content length checks.
    #
    ##################################################################

    if (!defined($$phProperties{'ContentLength'}) || !length($$phProperties{'ContentLength'}))
    {
      $$psError = "Content length is undefined or null";
      return 454;
    }

    if ($$phProperties{'CapContentLength'} =~ /^[Yy]$/ && $$phProperties{'ContentLength'} > $$phProperties{'MaxContentLength'})
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) exceeds maximum allowed length ($$phProperties{'MaxContentLength'})";
      return 455;
    }

    if ($$phProperties{'ContentLength'} != ($sOutLength + $sErrLength + $sEnvLength))
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) does not equal sum of individual stream lengths ($sOutLength + $sErrLength + $sEnvLength)";
      return 456;
    }

    ##################################################################
    #
    # If this is a link test, dump the data and return success.
    #
    ##################################################################

    if ($$phProperties{'ClientRunType'} eq "linktest")
    {
      SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
      $$psError = "Success";
      return 251;
    }

    ##################################################################
    #
    # Make output filenames and directories.
    #
    ##################################################################

    my ($sLocalError, $sEnvFile, $sErrFile, $sOutFile, $sPutName, $sRdyFile);

    $sPutName = MakePutName($phProperties, \$sLocalError);
    if (!defined($sPutName))
    {
      SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
      $$psError = $sLocalError;
      return 500;
    }
    $sOutFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".out";
    $sErrFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".err";
    $sEnvFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".env";
    $sRdyFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".rdy";

    if (!defined(MakePutTree($$phProperties{'IncomingDirectory'}, $sPutName, 0755, 1, \$sLocalError)))
    {
      SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
      $$psError = $sLocalError;
      return 500;
    }

    ##################################################################
    #
    # Make sure that none of the output files exist.
    #
    ##################################################################

    foreach my $sPutFile ($sOutFile, $sErrFile, $sEnvFile, $sRdyFile)
    {
      if (-e $sPutFile)
      {
        SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
        $$psError = "File ($sPutFile) already exists";
        return 451;
      }
    }

    ##################################################################
    #
    # Write output files (.out, .err, .env, .rdy) to disk.
    #
    ##################################################################

    my (%hStreamLengths);

    $hStreamLengths{$sOutFile} = $sOutLength;
    $hStreamLengths{$sErrFile} = $sErrLength;
    $hStreamLengths{$sEnvFile} = $sEnvLength;

    foreach my $sPutFile ($sOutFile, $sErrFile, $sEnvFile, $sRdyFile)
    {
      if (!open(FH, ">$sPutFile"))
      {
        SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
        $$psError = "File ($sPutFile) could not be opened ($!)";
        return 500;
      }
      binmode(FH);
      if ($sPutFile eq $sRdyFile)
      {
        print FH "Jid=", $$phProperties{'JobId'}, $$phProperties{'Newline'};
      }
      else
      {
        my $sByteCount = SysReadWrite(\*STDIN, \*FH, $hStreamLengths{$sPutFile}, \$sLocalError);
        if (!defined($sByteCount))
        {
          close(FH);
          $$psError = $sLocalError;
          return 500;
        }
        if ($sByteCount != $hStreamLengths{$sPutFile})
        {
          close(FH);
          $$psError = "Stream length ($hStreamLengths{$sPutFile}) does not equal number of bytes processed ($sByteCount) for output file ($sPutFile)";
          return 456;
        }
      }
      close(FH);
    }
    $$psError = "Success";
    return 200;
  }
  else
  {
    $$psError = "Invalid query string ($$phProperties{'QueryString'})";
    return 450;
  }
}


######################################################################
#
# SendResponse
#
######################################################################

sub SendResponse
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Send response header.
  #
  ####################################################################

  my ($sHandle, $sHeader, $sLength, $sReason, $sServer, $sStatus);

  $sHandle = $$phProperties{'ReturnHandle'};
  $sStatus = $$phProperties{'ReturnStatus'};
  $sReason = $$phProperties{'ReturnReason'};
  $sServer = $$phProperties{'ServerSoftware'};
  $sLength = (defined($sHandle)) ? -s $sHandle : 0;

  $sHeader  = "HTTP/1.1 $sStatus $sReason\r\n";
  $sHeader .= "Server: $sServer\r\n";
  $sHeader .= "Content-Type: application/octet-stream\r\n";
  $sHeader .= "Content-Length: $sLength\r\n";
  if ($$phProperties{'RequestMethod'} eq 'GET' && $$phProperties{'EnableJobIds'} =~ /^[Yy]$/)
  {
    if (defined($$phProperties{'JobId'}) && $$phProperties{'JobId'} =~ /^$$phProperties{'JobIdRegex'}$/)
    {
      $sHeader .= "Job-Id: $$phProperties{'JobId'}\r\n";
    }
  }
  $sHeader .= "\r\n";

  syswrite(STDOUT, $sHeader, length($sHeader));

  ####################################################################
  #
  # Send content if any.
  #
  ####################################################################

  if (defined($sHandle))
  {
    SysReadWrite($sHandle, \*STDOUT, $sLength, undef);
    close($sHandle);
  }

  return $sLength;
}


######################################################################
#
# SysReadWrite
#
######################################################################

sub SysReadWrite
{
  my ($sReadHandle, $sWriteHandle, $sLength, $psError) = @_;

  ####################################################################
  #
  # Read/Write data, but discard data if write handle is undefined.
  #
  ####################################################################

  my ($sData, $sEOF, $sNRead, $sNProcessed, $sNWritten);

  for ($sEOF = $sNRead = $sNProcessed = 0; !$sEOF && $sLength > 0; $sLength -= $sNRead)
  {
    $sNRead = sysread($sReadHandle, $sData, ($sLength > 0x4000) ? 0x4000 : $sLength);
    if (!defined($sNRead))
    {
      $$psError = "Error reading from input stream ($!)" if (defined($psError));
      return undef;
    }
    elsif ($sNRead == 0)
    {
      $sEOF = 1;
    }
    else
    {
      if (defined($sWriteHandle))
      {
        $sNWritten = syswrite($sWriteHandle, $sData, $sNRead);
        if (!defined($sNWritten))
        {
          $$psError = "Error writing to output stream ($!)" if (defined($psError));
          return undef;
        }
      }
      else
      {
        $sNWritten = $sNRead;
      }
      $sNProcessed += $sNWritten;
    }
  }

  return $sNProcessed;
}


######################################################################
#
# URLDecode
#
######################################################################

sub URLDecode
{
  my ($sData) = @_;

  $sData =~ s/\+/ /sg;
  $sData =~ s/%([0-9a-fA-F]{2})/pack('C', hex($1))/seg;

  return $sData;
}


######################################################################
#
# VerifyRunTimeEnvironment
#
######################################################################

sub VerifyRunTimeEnvironment
{
  my ($phProperties, $phRequiredProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure all required properties are defined and valid.
  #
  ####################################################################

  foreach my $sProperty (keys(%$phRequiredProperties))
  {
    my $sValue = $$phProperties{$sProperty};
    if (!defined($sValue) || $sValue !~ /$$phRequiredProperties{$sProperty}/)
    {
      $$psError = "$sProperty property ($sValue) is undefined or invalid";
      return undef;
    }
  }

  ####################################################################
  #
  # Make sure the logfiles directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'LogfilesDirectory'} || !-R _)
  {
    $$psError = "Logfiles directory ($$phProperties{'LogfilesDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the profiles directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'ProfilesDirectory'} || !-R _)
  {
    $$psError = "Profiles directory ($$phProperties{'ProfilesDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the incoming directory is writable.
  #
  ####################################################################

  if (!-d $$phProperties{'IncomingDirectory'} || !-W _)
  {
    $$psError = "Incoming directory ($$phProperties{'IncomingDirectory'}) does not exist or is not writeable";
    return undef;
  }

  1;
}
