// Copyright (c) NetXS Group.
// Licensed under the MIT license.

/**
 * unidata.hpp, autogenerated on 2023-05-08 01:31:45.632846
 *
 * Provides fastest access to the Unicode Character Database.
 * Properties of a single Unicode character are accessed by its code point value.
 *
 * Format conventions
 *  https://www.unicode.org/reports/tr44/
 *
 * Character presentation width rules
 *  EAW:    https://www.unicode.org/reports/tr11
 *  Emoji:  https://www.unicode.org/reports/tr51
 *
 * Grapheme clusterization
 *  https://www.unicode.org/reports/tr29
 *
 * Unicode Character Database properties
 *  https://www.unicode.org/reports/tr44/#Property_Index
 *
 * All emoji characters are treated 'East Asian Wide'
 *  Current practice is for emoji to have a square aspect ratio,
 *  deriving from their origin in Japanese.
 *  https://www.unicode.org/reports/tr51/#Display
 *
 * VS16/15
 *  Emoji presentation sequences behave as though they were East Asian Wide,
 *  regardless of their assigned East_Asian_Width property value.
 *  https://www.unicode.org/reports/tr11/#Recommendations
 *
 * Ambiguous characters
 *  Ambiguous characters behave like wide or narrow characters
 *  depending on the context.
 *  If the context cannot be established reliably,
 *  they should be treated as narrow characters by default.
 *  https://www.unicode.org/reports/tr11/#Recommendations
 *
 * Categories of the character width
 *  0 - Non-printable
 *  1 - Halfwidth
 *  2 - Fullwidth
 *
 * C0 controls 0x00..0x1F
 *  Since C0 codes are based on bytes, they are excluded from
 *  the property list of controls.
 *  They are analyzed in static tables before decoding UTF-8.
 *
 * C1 controls 0x80..0x9F
 *  The C1 codes require two bytes to be encoded in UTF-8
 *  (for instance CSI at U+009B is encoded as the bytes 0xC2, 0x9B in UTF-8),
 *  so there is no advantage to using them rather than the equivalent
 *  two-byte ESC+letter sequence, so the C1 controls are represented
 *  as C0 escaped sequences.
 *  The C1 characters appear outdated now.
 *  https://en.wikipedia.org/wiki/C0_and_C1_control_codes#Unicode
 *
 * Soft Hyphen
 *  Two ways:
 *    1. Interpret it as a command and divide the text
 *       strings into two independent once.
 *    2. Append it to the last grapheme cluster.
 *
 * Printable format characters
 *  A 'Prepend' characters always have the width 'Narrow' to be
 *  the basis of the grapheme cluster.
 *  https://www.unicode.org/reports/tr29/#Table_Combining_Char_Sequences_and_Grapheme_Clusters
 *  https://www.unicode.org/reports/tr29/#GB9b
 *
 * Names and Aliases
 *  Character name aliases are immutable.
 *  https://www.unicode.org/versions/Unicode12.0.0/ch04.pdf#page=24
 *
 *  -del- Invisible math operators
 *  -del- All of science and technology uses formulas,
 *  -del- equations, and mathematical notation as part of
 *  -del- the language of the subject.
 *  -del- Nevertheless, I suppose that invisible math operators should be dropped,
 *  -del- because there is no way to apply them using a cellular/monospaced display.
 *  -del- https://unicode.org/reports/tr25/#page=23
 *
 * https://www.unicode.org/cldr/utility/bidi.jsp
 *
 * Unicode Bidirectional Algorithm
 *  https://unicode.org/reports/tr9/
 *
 *
 * Controls (should be enumerated in ascending order)
 *    command: <NON_CONTROL (paragraph break possible)
 *            c0
 *            c1
 *            \u2029 PARAGRAPH SEPARATOR
 *    visible: =NON_CONTROL - non control chars
 *     format: >NON_CONTROL
 *             all other enumarated controls
 *
 **/

#pragma once

#include <cstdint>
#include <vector>
#include <iterator>

namespace netxs::unidata
{
    namespace widths
    {
        enum type : unsigned char
        {
            zero,  // non-printable
            slim,  // narrow
            wide,  // fullwidth
        };
    }

    namespace gbreak
    {
        enum type : unsigned char
        {
            any     ,  // Other
            cr      ,  // CR
            lf      ,  // LF
            ctrl    ,  // Control
            ext     ,  // Extend or Emoji_Modifier_Base
            l       ,  // HANGUL CHOSEONG
            v       ,  // HANGUL JUNGSEONG
            t       ,  // HANGUL JUNGSEONG
            lv      ,  // HANGUL SYLLABLE
            lvt     ,  // HANGUL SYLLABLE
            ri      ,  // Regional_Indicator
            sm      ,  // SpacingMark
            prep    ,  // Prepend
            zwj     ,  // ZERO WIDTH JOINER
            ep      ,  // Extended_Pictographic
            combo   ,  // EP + ZWJ
        };
    }

    namespace cntrls
    {
        enum type : unsigned char
        {
            nul                                       ,  //   0     0 NULL
            soh                                       ,  //   1     1 START OF HEADING
            stx                                       ,  //   2     2 START OF TEXT
            etx                                       ,  //   3     3 END OF TEXT
            eot                                       ,  //   4     4 END OF TRANSMISSION
            enq                                       ,  //   5     5 ENQUIRY
            ack                                       ,  //   6     6 ACKNOWLEDGE
            bel                                       ,  //   7     7 ALERT
            bs                                        ,  //   8     8 BACKSPACE
            tab                                       ,  //   9     9 CHARACTER TABULATION
            eol                                       ,  //  10     A LINE FEED
            vt                                        ,  //  11     B LINE TABULATION
            ff                                        ,  //  12     C FORM FEED
            cr                                        ,  //  13     D CARRIAGE RETURN
            so                                        ,  //  14     E SHIFT OUT
            si                                        ,  //  15     F SHIFT IN
            dle                                       ,  //  16    10 DATA LINK ESCAPE
            dc1                                       ,  //  17    11 DEVICE CONTROL ONE
            dc2                                       ,  //  18    12 DEVICE CONTROL TWO
            dc3                                       ,  //  19    13 DEVICE CONTROL THREE
            dc4                                       ,  //  20    14 DEVICE CONTROL FOUR
            nak                                       ,  //  21    15 NEGATIVE ACKNOWLEDGE
            syn                                       ,  //  22    16 SYNCHRONOUS IDLE
            etb                                       ,  //  23    17 END OF TRANSMISSION BLOCK
            can                                       ,  //  24    18 CANCEL
            eom                                       ,  //  25    19 END OF MEDIUM
            sub                                       ,  //  26    1A SUBSTITUTE
            esc                                       ,  //  27    1B ESCAPE
            fs                                        ,  //  28    1C INFORMATION SEPARATOR FOUR
            gs                                        ,  //  29    1D INFORMATION SEPARATOR THREE
            rs                                        ,  //  30    1E INFORMATION SEPARATOR TWO
            us                                        ,  //  31    1F INFORMATION SEPARATOR ONE
            del                                       ,  //  32    7F DELETE
            pad                                       ,  //  33    80 PADDING CHARACTER
            hop                                       ,  //  34    81 HIGH OCTET PRESET
            bph                                       ,  //  35    82 BREAK PERMITTED HERE
            nbh                                       ,  //  36    83 NO BREAK HERE
            ind                                       ,  //  37    84 INDEX
            nel                                       ,  //  38    85 NEXT LINE
            ssa                                       ,  //  39    86 START OF SELECTED AREA
            esa                                       ,  //  40    87 END OF SELECTED AREA
            hts                                       ,  //  41    88 CHARACTER TABULATION SET
            htj                                       ,  //  42    89 CHARACTER TABULATION WITH JUSTIFICATION
            vts                                       ,  //  43    8A LINE TABULATION SET
            pld                                       ,  //  44    8B PARTIAL LINE FORWARD
            plu                                       ,  //  45    8C PARTIAL LINE BACKWARD
            ri                                        ,  //  46    8D REVERSE LINE FEED
            ss2                                       ,  //  47    8E SINGLE SHIFT TWO
            ss3                                       ,  //  48    8F SINGLE SHIFT THREE
            dcs                                       ,  //  49    90 DEVICE CONTROL STRING
            pu1                                       ,  //  50    91 PRIVATE USE ONE
            pu2                                       ,  //  51    92 PRIVATE USE TWO
            sts                                       ,  //  52    93 SET TRANSMIT STATE
            cch                                       ,  //  53    94 CANCEL CHARACTER
            mw                                        ,  //  54    95 MESSAGE WAITING
            spa                                       ,  //  55    96 START OF GUARDED AREA
            epa                                       ,  //  56    97 END OF GUARDED AREA
            sos                                       ,  //  57    98 START OF STRING
            sgc                                       ,  //  58    99 SINGLE GRAPHIC CHARACTER INTRODUCER
            sci                                       ,  //  59    9A SINGLE CHARACTER INTRODUCER
            csi                                       ,  //  60    9B CONTROL SEQUENCE INTRODUCER
            st                                        ,  //  61    9C STRING TERMINATOR
            osc                                       ,  //  62    9D OPERATING SYSTEM COMMAND
            pm                                        ,  //  63    9E PRIVACY MESSAGE
            apc                                       ,  //  64    9F APPLICATION PROGRAM COMMAND
            paragraph_separator                       ,  //  65  2029 PARAGRAPH SEPARATOR
            non_control                               ,  //  66    -1 NON CONTROL
            shy                                       ,  //  67    AD SOFT HYPHEN
            alm                                       ,  //  68   61C ARABIC LETTER MARK
            mvs                                       ,  //  69  180E MONGOLIAN VOWEL SEPARATOR
            zwsp                                      ,  //  70  200B ZERO WIDTH SPACE
            zwnj                                      ,  //  71  200C ZERO WIDTH NON-JOINER
            lrm                                       ,  //  72  200E LEFT-TO-RIGHT MARK
            rlm                                       ,  //  73  200F RIGHT-TO-LEFT MARK
            line_separator                            ,  //  74  2028 LINE SEPARATOR
            lre                                       ,  //  75  202A LEFT-TO-RIGHT EMBEDDING
            rle                                       ,  //  76  202B RIGHT-TO-LEFT EMBEDDING
            pdf                                       ,  //  77  202C POP DIRECTIONAL FORMATTING
            lro                                       ,  //  78  202D LEFT-TO-RIGHT OVERRIDE
            rlo                                       ,  //  79  202E RIGHT-TO-LEFT OVERRIDE
            wj                                        ,  //  80  2060 WORD JOINER
            function_application                      ,  //  81  2061 FUNCTION APPLICATION
            invisible_times                           ,  //  82  2062 INVISIBLE TIMES
            invisible_separator                       ,  //  83  2063 INVISIBLE SEPARATOR
            invisible_plus                            ,  //  84  2064 INVISIBLE PLUS
            lri                                       ,  //  85  2066 LEFT-TO-RIGHT ISOLATE
            rli                                       ,  //  86  2067 RIGHT-TO-LEFT ISOLATE
            fsi                                       ,  //  87  2068 FIRST STRONG ISOLATE
            pdi                                       ,  //  88  2069 POP DIRECTIONAL ISOLATE
            inhibit_symmetric_swapping                ,  //  89  206A INHIBIT SYMMETRIC SWAPPING
            activate_symmetric_swapping               ,  //  90  206B ACTIVATE SYMMETRIC SWAPPING
            inhibit_arabic_form_shaping               ,  //  91  206C INHIBIT ARABIC FORM SHAPING
            activate_arabic_form_shaping              ,  //  92  206D ACTIVATE ARABIC FORM SHAPING
            national_digit_shapes                     ,  //  93  206E NATIONAL DIGIT SHAPES
            nominal_digit_shapes                      ,  //  94  206F NOMINAL DIGIT SHAPES
            zwnbsp                                    ,  //  95  FEFF ZERO WIDTH NO-BREAK SPACE
            interlinear_annotation_anchor             ,  //  96  FFF9 INTERLINEAR ANNOTATION ANCHOR
            interlinear_annotation_separator          ,  //  97  FFFA INTERLINEAR ANNOTATION SEPARATOR
            interlinear_annotation_terminator         ,  //  98  FFFB INTERLINEAR ANNOTATION TERMINATOR
            egyptian_hieroglyph_vertical_joiner       ,  //  99 13430 EGYPTIAN HIEROGLYPH VERTICAL JOINER
            egyptian_hieroglyph_horizontal_joiner     ,  // 100 13431 EGYPTIAN HIEROGLYPH HORIZONTAL JOINER
            egyptian_hieroglyph_insert_at_top_start   ,  // 101 13432 EGYPTIAN HIEROGLYPH INSERT AT TOP START
            egyptian_hieroglyph_insert_at_bottom_start,  // 102 13433 EGYPTIAN HIEROGLYPH INSERT AT BOTTOM START
            egyptian_hieroglyph_insert_at_top_end     ,  // 103 13434 EGYPTIAN HIEROGLYPH INSERT AT TOP END
            egyptian_hieroglyph_insert_at_bottom_end  ,  // 104 13435 EGYPTIAN HIEROGLYPH INSERT AT BOTTOM END
            egyptian_hieroglyph_overlay_middle        ,  // 105 13436 EGYPTIAN HIEROGLYPH OVERLAY MIDDLE
            egyptian_hieroglyph_begin_segment         ,  // 106 13437 EGYPTIAN HIEROGLYPH BEGIN SEGMENT
            egyptian_hieroglyph_end_segment           ,  // 107 13438 EGYPTIAN HIEROGLYPH END SEGMENT
            shorthand_format_letter_overlap           ,  // 108 1BCA0 SHORTHAND FORMAT LETTER OVERLAP
            shorthand_format_continuing_overlap       ,  // 109 1BCA1 SHORTHAND FORMAT CONTINUING OVERLAP
            shorthand_format_down_step                ,  // 110 1BCA2 SHORTHAND FORMAT DOWN STEP
            shorthand_format_up_step                  ,  // 111 1BCA3 SHORTHAND FORMAT UP STEP
            musical_symbol_begin_beam                 ,  // 112 1D173 MUSICAL SYMBOL BEGIN BEAM
            musical_symbol_end_beam                   ,  // 113 1D174 MUSICAL SYMBOL END BEAM
            musical_symbol_begin_tie                  ,  // 114 1D175 MUSICAL SYMBOL BEGIN TIE
            musical_symbol_end_tie                    ,  // 115 1D176 MUSICAL SYMBOL END TIE
            musical_symbol_begin_slur                 ,  // 116 1D177 MUSICAL SYMBOL BEGIN SLUR
            musical_symbol_end_slur                   ,  // 117 1D178 MUSICAL SYMBOL END SLUR
            musical_symbol_begin_phrase               ,  // 118 1D179 MUSICAL SYMBOL BEGIN PHRASE
            musical_symbol_end_phrase                 ,  // 119 1D17A MUSICAL SYMBOL END PHRASE
            count                                     ,  // 120
        };
    }

    struct unidata;
    inline unidata const& select(ui32 cp);

    struct unidata
    {
        widths::type  ucwidth;
        gbreak::type  brgroup;
        cntrls::type  control;
        unsigned char padding = {};

        constexpr unidata()
            : ucwidth{ widths::slim },
              brgroup{ gbreak::any },
              control{ cntrls::non_control }
        { }

        constexpr unidata(widths::type ucwidth, gbreak::type brgroup, cntrls::type control)
            : ucwidth{ ucwidth },
              brgroup{ brgroup },
              control{ control }
        { }

        unidata(ui32 cp)
            : unidata{ select(cp) }
        { }

        unidata(unidata const&) = default;

        bool is_cmd()
        {
            return control < cntrls::non_control;
        }

        // Unicode 13.0.0 UAX #29 https://www.unicode.org/reports/tr29/#Grapheme_Cluster_Boundary_Rules
        bool allied(gbreak::type& l) const
        {
            auto const& r = brgroup;
            auto result =
                (  l == gbreak::cr    &&  r == gbreak::lf   )  ? true: // GB3

                (  l >= gbreak::cr    &&  l <= gbreak::ctrl )  ? faux: // GB4

                (  r >= gbreak::cr    &&  r <= gbreak::ctrl )  ? faux: // GB5

                (  l == gbreak::l     && (r == gbreak::l
                                      ||  r == gbreak::v
                                      ||  r == gbreak::lv
                                      ||  r == gbreak::lvt  )) ? true: // GB6

                (( l == gbreak::lv    ||  l == gbreak::v    )
              && ( r == gbreak::v     ||  r == gbreak::t    )) ? true: // GB7

                (( l == gbreak::lvt   ||  l == gbreak::t    )
                                      &&  r == gbreak::t    )  ? true: // GB8

                (  l == gbreak::prep  ||  r == gbreak::zwj
                                      ||  r == gbreak::sm
                                      ||  r == gbreak::ext  )  ? true: // GB9,a,b

                (  l == gbreak::combo &&  r == gbreak::ep   )  ? true: // GB11

                (  l == gbreak::ri    &&  r == gbreak::ri   )  ? true: // GB12,13
                                                                 faux; // GB999
            if (l == gbreak::ep)
            {
                l = (r == gbreak::ext) ? gbreak::ep    :
               	    (r == gbreak::zwj) ? gbreak::combo : r;
            }
            else
            {
                l = (l == gbreak::ri && r == gbreak::ri) ? gbreak::any : r;
            }
            return result;
        }
    };

    struct base
    {
        static constexpr auto blocks_size = sz_t{ 4352 };
        static constexpr si32 blocks_pack[] =
        {
              0,  -2, 256, 512, 768, 1024, 1280, 1536, 1792, 2048,
            2304, 2560, 2816, 3072, 3328, 3584, 3840, 4096, 4352, 4608,
             -2, 256, 4864, 5120, 5376, 5632, 5888, 6144, 6400, 6656,
            256, 6912, 7168, 7424, 256, 7680, 7936, 8192, 8448, 8704,
            256, 8960, 256, 9216, 9472, 9728, 9984, 10240, 10496, 10752,
            11008, -26, 11264, 11520, -86, 11264, 11776, 256, 12032, 12288,
            12544, 12800, 13056, 13312, 13568, 13824, 14080, 14336, 14592, 14848,
            15104, 13568, 13824, 14080, 14336, 14592, 14848, 15104, 13568, 13824,
            14080, 14336, 14592, 14848, 15104, 13568, 13824, 14080, 14336, 14592,
            14848, 15104, 13568, 13824, 14080, 14336, 14592, 14848, 15104, 13568,
            13824, 14080, 14336, 14592, 14848, 15104, 13568, 15360,  -8, 15616,
            -25, 256, 11264, 15872, 16128, 256, 16384, 16640, 16896, 17152,
            17408, 17664, 17920, 18176, 18432, 256, 18688, 18944, 19200, 19456,
            19712, 19968, 20224, 20480, 20736, 20992, 21248, 21504, 21760, 22016,
            22272, 22528, 22784, 23040, 23296, 23552, 15616, 23808, 24064, 24320,
            24576,  -3, 256, 24832, 25088, 25344,  -9, 15616, 25600,  -4,
            256, 25856, -15, 15616,  -2, 256, 26112, -33, 15616,  -2,
            256, 26368, 26624,  -2, 15616, 26880, 27136, -23, 11264, 27392,
             -4, 11264, 27648, 27904, -33, 15616, 28160, 11264, 28416, 28672,
             -9, 15616, 28928, -18, 15616, 29184, 29440, 29696, 29952, 30208,
            30464, 30720, 30976, 31232,  -2, 256, 31488,  -4, 15616, 31744,
            32000, 32256, 32512,  -4, 15616, 32768, 33024, 33280,  -2, 15616,
            33536, 33792, 34048, 15616, 34304, 34560, 34816, 35072, 35328, 35584,
            35840, 36096, 36352, 36608, 36864, 37120,  -3, 37376, 37632, -166,
            11264, 37888, -16, 11264, 38144, 38400, -21, 11264, 38656, -28,
            11264, 38912, -12, 15616,  -2, 11264, 39168,  -5, 15616, -19,
            11264, 39424, -2796, 15616, 39680, 39936, -14, 40192, -240, 15616,
            -255, 256, 40448, -255, 256, 40448
        };

        static constexpr auto offset_size = sz_t{ 40704 };
        static constexpr si32 offset_pack[] =
        {
              1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16,
             17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32,
            -95,  0, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46,
             47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62,
             63, 64, 65, -9,  0, 66, -3,  0, 67, 66,-337,  0,-112, 68, -8,  0,
             -2, 69, -6,  0, -4, 69, -7,  0, 69,  0, 69,-20,  0, 69,-224,  0,
             -7, 68,-166,  0, 69,-38,  0, -2, 69,-50,  0, -2, 69, -3,  0, 69,
            -45, 68,  0, 68,  0, -2, 68,  0, -2, 68,  0, 68, -8, 69,-27,  0,
             -4, 69, -6,  0,-11, 69, -6, 70,-10,  0,-11, 68,  0, 71,-46,  0,
            -21, 68,-16,  0, 68,-101,  0, -7, 68, 70,  0, -6, 68, -2,  0, -2,
             68,  0, -4, 68,-32,  0, 69, 70,  0, 68,-30,  0,-27, 68, -2, 69,
            -89,  0,-11, 68,  0,-14, 69,-43,  0, -9, 68, -7,  0, -2, 69, 68,
            -24,  0, -4, 68,  0, -9, 68,  0, -3, 68,  0, -5, 68, -2, 69,-15,
              0, 69,-25,  0, -3, 68, -2, 69,  0, 69,-11,  0, -5, 69,-31,  0,
             69, -2, 70, -6, 69, -8, 68,-42,  0,-24, 68, 70,-32, 68, 72,-54,
              0, 68, 72, 68,  0, -3, 72, -8, 68, -4, 72, 68, -2, 72,  0, -7,

             68,-10,  0, -2, 68,-29,  0, 68, -2, 72, 69, -8,  0, -2, 69, -2,
              0, -2, 69,-22,  0, 69, -7,  0, 69,  0, -3, 69, -4,  0, -2, 69,
             68,  0, 68, -2, 72, -4, 68, -2, 69, -2, 72, -2, 69, -2, 72, 68,
              0, -8, 69, 68, -4, 69, -2,  0, 69, -3,  0, -2, 68, -2, 69,-24,
              0, 68, -2, 69, -2, 68, 72, 69, -6,  0, -4, 69, -2,  0, -2, 69,
            -22,  0, 69, -7,  0, 69, -2,  0, 69, -2,  0, 69, -2,  0, -2, 69,
             68, 69, -3, 72, -2, 68, -4, 69, -2, 68, -2, 69, -3, 68, -3, 69,
             68, -7, 69, -4,  0, 69,  0, -7, 69,-10,  0, -2, 68, -3,  0, 68,
              0,-10, 69, -2, 68, 72, 69, -9,  0, 69, -3,  0, 69,-22,  0, 69,
             -7,  0, 69, -2,  0, 69, -5,  0, -2, 69, 68,  0, -3, 72, -5, 68,
             69, -2, 68, 72, 69, -2, 72, 68, -2, 69,  0,-15, 69, -2,  0, -2,
             68, -2, 69,-12,  0, -7, 69,  0, -6, 68, 69, 68, -2, 72, 69, -8,
              0, -2, 69, -2,  0, -2, 69,-22,  0, 69, -7,  0, 69, -2,  0, 69,
             -5,  0, -2, 69, 68,  0, -2, 68, 72, -4, 68, -2, 69, -2, 72, -2,
             69, -2, 72, 68, -7, 69, -3, 68, -4, 69, -2,  0, 69, -3,  0, -2,
             68, -2, 69,-18,  0,-10, 69, 68,  0, 69, -6,  0, -3, 69, -3,  0,

             69, -4,  0, -3, 69, -2,  0, 69,  0, 69, -2,  0, -3, 69, -2,  0,
             -3, 69, -3,  0, -3, 69,-12,  0, -4, 69, 68, 72, 68, -2, 72, -3,
             69, -3, 72, 69, -3, 72, 68, -2, 69,  0, -6, 69, 68,-14, 69,-21,
              0, -5, 69, 68, -3, 72, 68, -8,  0, 69, -3,  0, 69,-23,  0, 69,
            -16,  0, -2, 69, 68,  0, -3, 68, -4, 72, 69, -3, 68, 69, -4, 68,
             -7, 69, -2, 68, 69, -3,  0, -2, 69,  0, -2, 69, -2,  0, -2, 68,
             -2, 69,-10,  0, -7, 69,-10,  0, 68, -2, 72, -9,  0, 69, -3,  0,
             69,-23,  0, 69,-10,  0, 69, -5,  0, -2, 69, 68,  0, 72, 68, -2,
             72, 68, -2, 72, 69, 68, -2, 72, 69, -2, 72, -2, 68, -7, 69, -2,
             68, -6, 69, -2,  0, 69, -2,  0, -2, 68, -2, 69,-10,  0, 69, -2,
              0,-13, 69, -2, 68, -2, 72, -9,  0, 69, -3,  0, 69,-41,  0, -2,
             68,  0, 68, -2, 72, -4, 68, 69, -3, 72, 69, -3, 72, 68, 70,  0,
             -4, 69, -3,  0, 68,-10,  0, -2, 68, -2, 69,-26,  0, 69, 68, -2,
             72, 69,-18,  0, -3, 69,-24,  0, 69, -9,  0, 69,  0, -2, 69, -7,
              0, -3, 69, 68, -4, 69, 68, -2, 72, -3, 68, 69, 68, 69, -7, 72,
             68, -6, 69,-10,  0, -2, 69, -2, 72,  0,-12, 69,-48,  0, 68,  0,

             73, -7, 68, -4, 69, -8,  0, -8, 68,-13,  0,-37, 69, -2,  0, 69,
              0, 69, -5,  0, 69,-24,  0, 69,  0, 69,-10,  0, 68,  0, 73, -9,
             68,  0, -2, 69, -5,  0, 69,  0, 69, -6, 68, -2, 69,-10,  0, -2,
             69, -4,  0,-32, 69,-24,  0, -2, 68,-27,  0, 68,  0, 68,  0, 68,
             -4,  0, -2, 72, -8,  0, 69,-36,  0, -4, 69,-14, 68, 72, -5, 68,
              0, -2, 68, -5,  0,-11, 68, 69,-36, 68, 69, -8,  0, 68, -6,  0,
             69,-13,  0,-37, 69,-43,  0, -2, 69, -4, 68, 72, -6, 68, 69, -2,
             68, -2, 72, -2, 68,-23,  0, -2, 72, -2, 68, -4,  0, -3, 68,  0,
             -3, 69, -2,  0, -7, 69, -3,  0, -4, 68,-13,  0, 68, 69, 72, -2,
             68, -6, 69, 68,  0, 69,-10,  0, -3, 69, 68,-40,  0, 69,  0, -5,
             69,  0, -2, 69,-48,  0,-96, 74,-72, 75,-88, 76,-73,  0, 69, -4,
              0, -2, 69, -7,  0, 69,  0, 69, -4,  0, -2, 69,-41,  0, 69, -4,
              0, -2, 69,-33,  0, 69, -4,  0, -2, 69, -7,  0, 69,  0, 69, -4,
              0, -2, 69,-15,  0, 69,-57,  0, 69, -4,  0, -2, 69,-67,  0, -2,
             69, -3, 68,-29,  0, -3, 69,-26,  0, -6, 69,-86,  0, -2, 69, -6,
              0, -2, 69,-157,  0, -3, 69,-89,  0, -7, 69,-18,  0, -3, 68, 72,

             -9, 69,-19,  0, -2, 68, 72, -2,  0, -9, 69,-18,  0, -2, 68,-12,
             69,-13,  0, 69, -3,  0, 69, -2, 68,-12, 69,-52,  0, -2, 68, 72,
             -7, 68, -8, 72, 68, -2, 72,-11, 68, -9,  0, 68, -2, 69,-10,  0,
             -6, 69,-10,  0, -6, 69,-11,  0, -3, 68, 77, 68,-10,  0, -6, 69,
            -89,  0, -7, 69, -5,  0, -2, 68,-34,  0, 68,  0, -5, 69,-70,  0,
            -10, 69,-31,  0, 69, -3, 68, -4, 72, -2, 68, -3, 72, -4, 69, -2,
             72, 68, -6, 72, -3, 68, -4, 69,  0, -3, 69,-42,  0, -2, 69, -5,
              0,-11, 69,-44,  0, -4, 69,-26,  0, -6, 69,-11,  0, -3, 69,-57,
              0, -2, 68, -2, 72, 68, -2, 69,-55,  0, 72, 68, 72, -7, 68, 69,
             68, 69, 68, -2, 69, -8, 68, -6, 72,-10, 68, -2, 69, 68,-10,  0,
             -6, 69,-10,  0, -6, 69,-14,  0, -2, 69,-31, 68,-49, 69, -4, 68,
             72,-47,  0, -7, 68, 72, 68, -5, 72, 68, -2, 72, -8,  0, -3, 69,
            -27,  0, -9, 68,-11,  0, 69, -2, 68, 72,-30,  0, 72, -4, 68, -2,
             72, -2, 68, 72, -3, 68,-56,  0, 68, 72, -2, 68, -3, 72, 68, 72,
             -3, 68, -2, 72, -8, 69,-40,  0, -8, 72, -8, 68, -2, 72, -2, 68,
             -3, 69,-15,  0, -3, 69,-60,  0, -7, 69,-43,  0, -2, 69,-11,  0,

             -8, 69, -3, 68,  0,-13, 68, 72, -7, 68, -4,  0, 68, -6,  0, 68,
             -2,  0, 72, -2, 68,  0, -5, 69,-192,  0,-64, 68,-22,  0, -2, 69,
             -6,  0, -2, 69,-38,  0, -2, 69, -6,  0, -2, 69, -8,  0, 69,  0,
             69,  0, 69,  0, 69,-31,  0, -2, 69,-53,  0, 69,-15,  0, 69,-14,
              0, -2, 69, -6,  0, 69,-19,  0, -2, 69, -3,  0, 69, -9,  0, 69,
            -11,  0, 78, 79, 80, 81, 82,-24,  0, 83, 84, 85, 86, 87, 88, 89,
            -13,  0, 66,-12,  0, 66,-22,  0, 90, 91, 92, 93, 94, 95, 96, 97,
             98, 99,100,101,102,103,104,105, -2,  0, -2, 69,-27,  0, 69,-13,
              0, -3, 69,-33,  0,-15, 69,-33, 68,-15, 69,-34,  0, 66,-22,  0,
             66,-82,  0, -4, 69, -4,  0, -6, 66,-15,  0, -2, 66,-111,  0, -2,
            106,-12,  0, 66, -2,107,-93,  0, 66,-70,  0, 66,-25,  0, -4,106,
             -3, 66,106, -2, 66,106, -4,  0, -3, 66,-44,  0,-25, 69,-11,  0,
            -21, 69,-98,  0, 66,-231,  0, -2, 66,-10,  0, 66, -9,  0, 66,-58,
              0, -2, 66, -2,106,  0, -6, 66,  0,-12, 66,  0, -2,106,-50, 66,
            -12,106,-43, 66,106, -6, 66,-10,  0, -3, 66,106,-13, 66,106, -8,
             66, -2,106,-17, 66, -2,106, -5, 66, -2,106, -8, 66,106, -5, 66,

            106,-21, 66,106, -7, 66, -2,106, 66,106, -4, 66,106, -2, 66,106,
             -7, 66,106, -2,  0, -2, 66, -2,106, -7, 66,  0, 66,  0, 66, -6,
              0, 66, -3,  0, 66, -6,  0,106,-10,  0, -2, 66,-15,  0, 66, -2,
              0, 66, -4,  0,106,  0,106, -4,  0, -3,106,  0,106,-11,  0, -5,
             66,-45,  0, -3,106, -9,  0, 66,-14,  0,106,-14,  0,106,-116,  0,
             -2, 66,-207,  0, -3, 66,-19,  0, -2,106,-51,  0,106, -4,  0,106,
            -30,  0, -2, 69,-32,  0, 69,-344,  0, -3, 68, -2,  0, -5, 69,-45,
              0, 69,  0, -5, 69,  0, -2, 69,-56,  0, -7, 69, -2,  0,-14, 69,
             68,-23,  0, -9, 69, -7,  0, 69, -7,  0, 69, -7,  0, 69, -7,  0,
             69, -7,  0, 69, -7,  0, 69, -7,  0, 69, -7,  0, 69,-32, 68,-94,
              0,-34, 69,-26,107, 69,-89,107,-12, 69,-214,107,-26, 69,-12,107,
             -4, 69,-42,107, -6, 68,106,-12,107,106,107,  0, 69,-86,107, -2,
             69, -2, 68,-101,107, -5, 69,-43,107, 69,-94,107, 69,-84,107,-12,
             69,-47,107, 69,-40,107, -8,  0,-71,107,106,107,106,-550,107,-64,
              0,-141,107, -3, 69,-55,107, -9, 69,-92,  0,-20, 69,-47,  0, -4,
             68,  0,-10, 68,-32,  0, -2, 68,-80,  0, -2, 68, -6,  0, -8, 69,

            -203,  0, -5, 69, -2,  0, 69,  0, 69, -5,  0,-24, 69,-16,  0, 68,
             -3,  0, 68, -4,  0, 68,-23,  0, -2, 72, -2, 68, 72, -4,  0, 68,
             -3, 69,-10,  0, -6, 69,-56,  0, -8, 69, -2, 72,-50,  0,-16, 72,
             -2, 68, -8, 69,-12,  0, -6, 69,-18, 68,-13,  0, 68,-38,  0, -8,
             68,-25,  0,-11, 68, -2, 72,-11, 69,  0,-29, 74, -3, 69, -3, 68,
             72,-47,  0, 68, -2, 72, -4, 68, -2, 72, -2, 68, -3, 72,-13,  0,
             69,-11,  0, -4, 69, -7,  0, 68,-25,  0, 69,-41,  0, -6, 68, -2,
             72, -2, 68, -2, 72, -2, 68, -9, 69, -3,  0, 68, -8,  0, 68, 72,
             -2, 69,-10,  0, -2, 69,-31,  0, 69, 68, 69,-50,  0, 68,  0, -3,
             68, -2,  0, -2, 68, -5,  0, -2, 68,  0, 68,  0,-24, 69,-16,  0,
             72, -2, 68, -2, 72, -5,  0, 72, 68,-10, 69, -6,  0, -2, 69, -6,
              0, -2, 69, -6,  0, -9, 69, -7,  0, 69, -7,  0, 69,-60,  0, -4,
             69,-115,  0, -2, 72, 68, -2, 72, 68, -2, 72,  0, 72, 68, -2, 69,
            -10,  0, -6, 69,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,

            109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,
            109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,
            109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,
            -27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,108,-27,
            109,108,-51,109,108,-27,109,108,-27,109,108,-27,109,108,-27,109,
            108,-27,109,-12, 69,-23, 75, -4, 69,-49, 76,-260, 69,-110,107, -2,
             69,-106,107,-38, 69, -7,  0,-12, 69, -5,  0, -5, 69,  0, 68,-24,
              0, 69, -5,  0, 69,  0, 69, -2,  0, 69, -2,  0, 69,-125,  0,-16,
             69,-189,  0, -2, 69,-54,  0, -7, 69,  0,-32, 69,-16,  0,-16, 68,
            -10,107, -6, 69,-16, 68,-35,107, 69,-19,107, 69, -4,107, -4, 69,
             -5,  0, 69,-135,  0, -2, 69,110, 69,-96,107,-61,  0, -2,111,-31,

              0, -3, 69, -6,  0, -2, 69, -6,  0, -2, 69, -6,  0, -2, 69, -3,
              0, -3, 69, -7,107, 69, -7,  0, 69, -9, 95,112,113,114, -2,  0,
             -2, 69,-12,  0, 69,-26,  0, 69,-19,  0, 69, -2,  0, 69,-15,  0,
             -2, 69,-14,  0,-34, 69,-123,  0, -5, 69, -3,  0, -4, 69,-45,  0,
             -3, 69,-88,  0, 69,-13,  0, -3, 69,  0,-47, 69,-45,  0, 68,-130,
             69,-29,  0, -3, 69,-49,  0,-15, 69, 68,-27,  0, -4, 69,-36,  0,
             -9, 69,-30,  0, -5, 69,-38,  0, -5, 68, -5, 69,-30,  0, 69,-37,
              0, -4, 69,-14,  0,-42, 69,-158,  0, -2, 69,-10,  0, -6, 69,-36,
              0, -4, 69,-36,  0, -4, 69,-40,  0, -8, 69,-52,  0,-11, 69,-12,
              0, 69,-15,  0, 69, -7,  0, 69, -2,  0, 69,-11,  0, 69,-15,  0,
             69, -7,  0, 69, -2,  0,-67, 69,-55,  0, -9, 69,-22,  0,-10, 69,
             -8,  0,-24, 69, -6,  0, 69,-42,  0, 69, -9,  0,-69, 69, -6,  0,
             -2, 69,  0, 69,-44,  0, 69, -2,  0, -3, 69,  0, -2, 69,-23,  0,
             69,-72,  0, -8, 69, -9,  0,-48, 69,-19,  0, 69, -2,  0, -5, 69,
            -33,  0, -3, 69,-27,  0, -5, 69,  0,-64, 69,-56,  0, -4, 69,-20,
              0, -2, 69,-47,  0, -3, 68, 69, -2, 68, -5, 69, -4, 68, -4,  0,

             69, -3,  0, 69,-29,  0, -2, 69, -3, 68, -4, 69, 68, -9,  0, -7,
             69, -9,  0, -7, 69,-64,  0,-32, 69,-37,  0, -2, 68, -4, 69,-12,
              0, -9, 69,-54,  0, -3, 69,-29,  0, -2, 69,-27,  0, -5, 69,-26,
              0, -7, 69, -4,  0,-12, 69, -7,  0,-80, 69,-73,  0,-55, 69,-51,
              0,-13, 69,-51,  0, -7, 69,-42,  0, -4, 68, -8, 69,-10,  0,-294,
             69,-31,  0, 69,-42,  0, 69, -2, 68,  0, -2, 69, -2,  0,-78, 69,
            -40,  0, -8, 69,-22,  0,-11, 68, -9,  0,-22, 69,-18,  0, -4, 68,
             -4,  0,-38, 69,-28,  0,-20, 69,-23,  0, -9, 69, 72, 68, 72,-53,
              0,-15, 68, -7,  0, -4, 69,-30,  0, 68, -2,  0, -2, 68,  0, -9,
             69, -3, 68, 72,-45,  0, -3, 72, -4, 68, -2, 72, -2, 68, -2,  0,
             70, -4,  0, 68,-10, 69, 70, -2, 69,-25,  0, -7, 69,-10,  0, -6,
             69, -3, 68,-36,  0, -5, 68, 72, -8, 68, 69,-15,  0, -2, 72,  0,
             -8, 69,-35,  0, 68, -3,  0, -9, 69, -2, 68, 72,-48,  0, -3, 72,
             -9, 68, -2, 72,  0, -2, 70, -5,  0, -4, 68,  0, 72, 68,-16,  0,
             69,-20,  0,-11, 69,-18,  0, 69,-25,  0, -3, 72, -3, 68, -2, 72,
             68, 72, -2, 68, -6,  0, 68,-65, 69, -7,  0, 69,  0, 69, -4,  0,

             69,-15,  0, 69,-11,  0, -6, 69,-47,  0, 68, -3, 72, -8, 68, -5,
             69,-10,  0, -6, 69, -2, 68, -2, 72, 69, -8,  0, -2, 69, -2,  0,
             -2, 69,-22,  0, 69, -7,  0, 69, -2,  0, 69, -5,  0, 69, -2, 68,
              0, 68, 72, 68, -4, 72, -2, 69, -2, 72, -2, 69, -3, 72, -2, 69,
              0, -6, 69, 68, -5, 69, -5,  0, -2, 72, -2, 69, -7, 68, -3, 69,
             -5, 68,-139, 69,-53,  0, -3, 72, -8, 68, -2, 72, -3, 68, 72, 68,
            -21,  0, 69,  0, 68, -3,  0,-30, 69,-48,  0, 68, -2, 72, -6, 68,
             72, 68, -2, 72, 68, 72, -2, 68, 72, -2, 68, -4,  0, -8, 69,-10,
              0,-166, 69,-47,  0, 68, -2, 72, -4, 68, -2, 69, -4, 72, -2, 68,
             72, -2, 68,-27,  0, -2, 68,-34, 69,-48,  0, -3, 72, -8, 68, -2,
             72, 68, 72, -2, 68, -4,  0,-11, 69,-10,  0, -6, 69,-13,  0,-19,
             69,-43,  0, 68, 72, 68, -2, 72, -6, 68, 72, 68, -2,  0, -6, 69,
            -10,  0,-54, 69,-27,  0, -2, 69, -3, 68, -2, 69, -4, 68, 72, -5,
             68, -4, 69,-23,  0,-185, 69,-44,  0, -3, 72, -9, 68, 72, -2, 68,
              0,-100, 69,-83,  0,-12, 69, -8,  0, -2, 69,  0, -2, 69, -8,  0,
             69, -2,  0, 69,-24,  0, 68, -5, 72, 69, -2, 72, -2, 69, -2, 68,

             72, 68, 70, 72, 70, 72, 68, -3,  0, -9, 69,-10,  0,-70, 69, -8,
              0, -2, 69,-39,  0, -3, 72, -4, 68, -2, 69, -2, 68, -4, 72, 68,
             -3,  0, 72,-27, 69,  0,-10, 68,-40,  0, -6, 68, 72, 70, -4, 68,
             -8,  0, 68, -8, 69,  0, -6, 68, -2, 72, -3, 68,-40,  0, -6, 70,
            -13, 68, 72, -2, 68, -9,  0,-13, 69,-73,  0, -7, 69, -9,  0, 69,
            -37,  0, 72, -7, 68, 69, -6, 68, 72, 68, -6,  0,-10, 69,-29,  0,
             -3, 69,-32,  0, -2, 69,-22, 68, 69, 72, -7, 68, 72, -2, 68, 72,
             -2, 68,-73, 69, -7,  0, 69, -2,  0, 69,-38,  0, -6, 68, -3, 69,
             68, 69, -2, 68, 69, -7, 68, 70, 68, -8, 69,-10,  0, -6, 69, -6,
              0, 69, -2,  0, 69,-32,  0, -5, 72, 69, -2, 68, 69, -2, 72, 68,
             72, 68,  0, -7, 69,-10,  0,-310, 69,-19,  0, -2, 68, -2, 72, -2,
              0,-183, 69,  0,-15, 69,-50,  0,-13, 69,-155,  0,-102, 69,-111,  0,
             69, -5,  0,-11, 69,-196,  0,-332, 69,-99,  0,-13, 69,-47,  0, 69,
            115,116,117,118,119,120,121,122,123,-199, 69,-71,  0,-185, 69,-57,
              0, -7, 69,-31,  0, 69,-10,  0, -4, 69,-81,  0, 69,-10,  0, -6,
             69,-30,  0, -2, 69, -5, 68,  0,-10, 69,-48,  0, -7, 68,-15,  0,

            -10, 69,-10,  0, 69, -7,  0, 69,-21,  0, -5, 69,-19,  0,-176, 69,
            -91,  0,-101, 69,-75,  0, -4, 69, 68,  0,-55, 72, -7, 69, -4, 68,
            -13,  0,-64, 69, -4,107, 68,-11, 69, -2, 72,-14, 69,-248,107, -8,
             69,-214,107,-42, 69, -9,107,-487, 69, -4,107, 69, -7,107, 69, -2,
            107, 69,-35,107,-45, 69, -3,107,-17, 69, -4,107, -8, 69,-396,107,
             -4, 69,-107,  0, -5, 69,-13,  0, -3, 69, -9,  0, -7, 69,-10,  0,
             -2, 69,  0, -2, 68,  0,124,125,126,127,-92, 69,-46, 68, -2, 69,
            -23, 68, -9, 69,-116,  0,-60, 69,-246,  0,-10, 69,-39,  0, -2, 69,
            -60,  0, 68, 72, -3, 68, -3,  0, 72, -5, 68,128,129,130,131,132,
            133,134,135, -8, 68, -2,  0, -7, 68,-30,  0, -4, 68,-61,  0,-21,
             69,-66,  0, -3, 68,  0,-154, 69,-20,  0,-12, 69,-87,  0, -9, 69,
            -25,  0,-135, 69,-85,  0, 69,-71,  0, 69, -2,  0, -2, 69,  0, -2,
             69, -2,  0, -2, 69, -4,  0, 69,-12,  0, 69,  0, 69, -7,  0, 69,
            -65,  0, 69, -4,  0, -2, 69, -8,  0, 69, -7,  0, 69,-28,  0, 69,
             -4,  0, 69, -5,  0, 69,  0, -3, 69, -7,  0, 69,-340,  0, -2, 69,
            -292,  0, -2, 69,-50,  0,-55, 68, -4,  0,-50, 68, -8,  0, 68,-14,

              0, 68, -7,  0,-15, 69, -5, 68, 69,-15, 68,-80, 69,-31,  0,-225,
             69, -7, 68, 69,-17, 68, -2, 69, -7, 68, 69, -2, 68, 69, -5, 68,
            -213, 69,-45,  0, -3, 69, -7, 68, -7,  0, -2, 69,-10,  0, -4, 69,
             -2,  0,-320, 69,-30,  0, 68,-17, 69,-44,  0, -4, 68,-10,  0, -5,
             69,  0,-224, 69, -7,  0, 69, -4,  0, 69, -2,  0, 69,-15,  0, 69,
            -197,  0, -2, 69, -9,  0, -7, 68,-41, 69,-68,  0, -7, 68,  0, -4,
             69,-10,  0, -4, 69, -2,  0,-273, 69,-68,  0,-76, 69,-61,  0,-194,
             69, -4,  0, 69,-27,  0, 69, -2,  0, 69,  0, -2, 69,  0, 69,-10,
              0, 69, -4,  0, 69,  0, 69,  0, -6, 69,  0, -4, 69,  0, 69,  0,
             69,  0, 69, -3,  0, 69, -2,  0, 69,  0, -2, 69,  0, 69,  0, 69,
              0, 69,  0, 69,  0, 69, -2,  0, 69,  0, -2, 69, -4,  0, 69, -7,
              0, 69, -4,  0, 69, -4,  0, 69,  0, 69,-10,  0, 69,-17,  0, -5,
             69, -3,  0, 69, -5,  0, 69,-17,  0,-52, 69, -2,  0,-14, 69, -4,
             66,106,-39, 66, -4,136,-100, 66,-12,136,-15, 66, -2,136,-15, 66,
            136,-14, 66,106,136,-37, 66,-10,136,-13,  0, -3, 66,-31,  0, 66,
            -60,  0, -6, 66,-12,  0, -2, 66,-14,  0,106, -2,  0,-10,106,-18,

              0, 66,-56,136,-26,137,107, -2,106,-13,136,-10,107,106,-20,107,
            106, -2,107, -9,106,107, -4,136, -9,107, -7,136, -2,106,-14,136,
             -6,106,-154,136,-33,106,-12, 66, -9,106, 66,-70,106, 66,-22,106,
            -12, 66,-43,106, -4, 66, -5,106,-12, 66,-17,106, -3, 66,106, -3,
             66, -3,106, -5,138,-63,106, 66,106, 66,-187,106, -2, 66,-63,106,
             -8,  0, -5, 66, -4,106, 66,-24,106,-18, 66,106,-26, 66, -2,106,
            -13, 66,106,-86, 66,-85,106,-48,  0,-70,106, -6, 66,106, -3, 66,
             -3,106, -2, 66, -3,106, -5,136, -3,106,-11, 66, -2,106, -3,136,
             -4, 66, -9,106, -3,136,-116,  0,-12,136,-85,  0, -4, 66, -7,136,
            -12,106, -4,136,106,-15,136,-12,  0, -4,136,-56,  0, -8,136,-10,
              0, -6,136,-40,  0, -8,136,-30,  0, -2,136, -2, 66,-78,136,-12,
              0,-47,106,  0,-10,106,  0,-185,106,-84, 66,-12,136,-14, 66, -2,
            136, -5,106, -3,136, -5,106, -3,136, -7,106, -9,136,-29,106, -3,
            136,-11,106, -5,136, -6,106,-10,136,-10,106, -6,136, -8,106, -8,
            136, -7,106, -9,136,-147,  0, 69,-55,  0,-37, 69,-10,  0, -6, 69,
            -510,136, -2, 69,-224,107,-32, 69,-57,107, -7, 69,-222,107, -2, 69,

            -386,107,-14, 69,-305,107,-31, 69,-30,107,-226, 69,-75,107,-181, 69,
            -32, 95,-96, 68,-128, 95,-240, 68,-272, 95,-254,  0, -2, 69
        };

        static constexpr unidata ucspec[] =
        {
            { widths::slim, gbreak::any , cntrls::non_control            },  //   0
            { widths::zero, gbreak::ctrl, cntrls::non_control            },  //   1
            { widths::zero, gbreak::ctrl, cntrls::soh                    },  //   2
            { widths::zero, gbreak::ctrl, cntrls::stx                    },  //   3
            { widths::zero, gbreak::ctrl, cntrls::etx                    },  //   4
            { widths::zero, gbreak::ctrl, cntrls::eot                    },  //   5
            { widths::zero, gbreak::ctrl, cntrls::enq                    },  //   6
            { widths::zero, gbreak::ctrl, cntrls::ack                    },  //   7
            { widths::zero, gbreak::ctrl, cntrls::bel                    },  //   8
            { widths::zero, gbreak::ctrl, cntrls::bs                     },  //   9
            { widths::zero, gbreak::ctrl, cntrls::tab                    },  //  10
            { widths::zero, gbreak::lf  , cntrls::eol                    },  //  11
            { widths::zero, gbreak::ctrl, cntrls::vt                     },  //  12
            { widths::zero, gbreak::ctrl, cntrls::ff                     },  //  13
            { widths::zero, gbreak::cr  , cntrls::cr                     },  //  14
            { widths::zero, gbreak::ctrl, cntrls::so                     },  //  15
            { widths::zero, gbreak::ctrl, cntrls::si                     },  //  16
            { widths::zero, gbreak::ctrl, cntrls::dle                    },  //  17
            { widths::zero, gbreak::ctrl, cntrls::dc1                    },  //  18
            { widths::zero, gbreak::ctrl, cntrls::dc2                    },  //  19
            { widths::zero, gbreak::ctrl, cntrls::dc3                    },  //  20
            { widths::zero, gbreak::ctrl, cntrls::dc4                    },  //  21
            { widths::zero, gbreak::ctrl, cntrls::nak                    },  //  22
            { widths::zero, gbreak::ctrl, cntrls::syn                    },  //  23
            { widths::zero, gbreak::ctrl, cntrls::etb                    },  //  24
            { widths::zero, gbreak::ctrl, cntrls::can                    },  //  25
            { widths::zero, gbreak::ctrl, cntrls::eom                    },  //  26
            { widths::zero, gbreak::ctrl, cntrls::sub                    },  //  27
            { widths::zero, gbreak::ctrl, cntrls::esc                    },  //  28
            { widths::zero, gbreak::ctrl, cntrls::fs                     },  //  29
            { widths::zero, gbreak::ctrl, cntrls::gs                     },  //  30
            { widths::zero, gbreak::ctrl, cntrls::rs                     },  //  31
            { widths::zero, gbreak::ctrl, cntrls::us                     },  //  32
            { widths::zero, gbreak::ctrl, cntrls::del                    },  //  33
            { widths::zero, gbreak::ctrl, cntrls::pad                    },  //  34
            { widths::zero, gbreak::ctrl, cntrls::hop                    },  //  35
            { widths::zero, gbreak::ctrl, cntrls::bph                    },  //  36
            { widths::zero, gbreak::ctrl, cntrls::nbh                    },  //  37
            { widths::zero, gbreak::ctrl, cntrls::ind                    },  //  38
            { widths::zero, gbreak::ctrl, cntrls::nel                    },  //  39
            { widths::zero, gbreak::ctrl, cntrls::ssa                    },  //  40
            { widths::zero, gbreak::ctrl, cntrls::esa                    },  //  41
            { widths::zero, gbreak::ctrl, cntrls::hts                    },  //  42
            { widths::zero, gbreak::ctrl, cntrls::htj                    },  //  43
            { widths::zero, gbreak::ctrl, cntrls::vts                    },  //  44
            { widths::zero, gbreak::ctrl, cntrls::pld                    },  //  45
            { widths::zero, gbreak::ctrl, cntrls::plu                    },  //  46
            { widths::zero, gbreak::ctrl, cntrls::ri                     },  //  47
            { widths::zero, gbreak::ctrl, cntrls::ss2                    },  //  48
            { widths::zero, gbreak::ctrl, cntrls::ss3                    },  //  49
            { widths::zero, gbreak::ctrl, cntrls::dcs                    },  //  50
            { widths::zero, gbreak::ctrl, cntrls::pu1                    },  //  51
            { widths::zero, gbreak::ctrl, cntrls::pu2                    },  //  52
            { widths::zero, gbreak::ctrl, cntrls::sts                    },  //  53
            { widths::zero, gbreak::ctrl, cntrls::cch                    },  //  54
            { widths::zero, gbreak::ctrl, cntrls::mw                     },  //  55
            { widths::zero, gbreak::ctrl, cntrls::spa                    },  //  56
            { widths::zero, gbreak::ctrl, cntrls::epa                    },  //  57
            { widths::zero, gbreak::ctrl, cntrls::sos                    },  //  58
            { widths::zero, gbreak::ctrl, cntrls::sgc                    },  //  59
            { widths::zero, gbreak::ctrl, cntrls::sci                    },  //  60
            { widths::zero, gbreak::ctrl, cntrls::csi                    },  //  61
            { widths::zero, gbreak::ctrl, cntrls::st                     },  //  62
            { widths::zero, gbreak::ctrl, cntrls::osc                    },  //  63
            { widths::zero, gbreak::ctrl, cntrls::pm                     },  //  64
            { widths::zero, gbreak::ctrl, cntrls::apc                    },  //  65
            { widths::slim, gbreak::ep  , cntrls::non_control            },  //  66
            { widths::zero, gbreak::ctrl, cntrls::shy                    },  //  67
            { widths::zero, gbreak::ext , cntrls::non_control            },  //  68
            { widths::zero, gbreak::any , cntrls::non_control            },  //  69
            { widths::slim, gbreak::prep, cntrls::non_control            },  //  70
            { widths::zero, gbreak::ctrl, cntrls::alm                    },  //  71
            { widths::zero, gbreak::sm  , cntrls::non_control            },  //  72
            { widths::slim, gbreak::sm  , cntrls::non_control            },  //  73
            { widths::wide, gbreak::l   , cntrls::non_control            },  //  74
            { widths::slim, gbreak::v   , cntrls::non_control            },  //  75
            { widths::slim, gbreak::t   , cntrls::non_control            },  //  76
            { widths::zero, gbreak::ctrl, cntrls::mvs                    },  //  77
            { widths::zero, gbreak::ctrl, cntrls::zwsp                   },  //  78
            { widths::zero, gbreak::ext , cntrls::zwnj                   },  //  79
            { widths::zero, gbreak::zwj , cntrls::non_control            },  //  80
            { widths::zero, gbreak::ctrl, cntrls::lrm                    },  //  81
            { widths::zero, gbreak::ctrl, cntrls::rlm                    },  //  82
            { widths::zero, gbreak::ctrl, cntrls::line_separator         },  //  83
            { widths::zero, gbreak::ctrl, cntrls::paragraph_separator    },  //  84
            { widths::zero, gbreak::ctrl, cntrls::lre                    },  //  85
            { widths::zero, gbreak::ctrl, cntrls::rle                    },  //  86
            { widths::zero, gbreak::ctrl, cntrls::pdf                    },  //  87
            { widths::zero, gbreak::ctrl, cntrls::lro                    },  //  88
            { widths::zero, gbreak::ctrl, cntrls::rlo                    },  //  89
            { widths::zero, gbreak::ctrl, cntrls::wj                     },  //  90
            { widths::zero, gbreak::ctrl, cntrls::function_application   },  //  91
            { widths::zero, gbreak::ctrl, cntrls::invisible_times        },  //  92
            { widths::zero, gbreak::ctrl, cntrls::invisible_separator    },  //  93
            { widths::zero, gbreak::ctrl, cntrls::invisible_plus         },  //  94
            { widths::zero, gbreak::ctrl, cntrls::non_control            },  //  95
            { widths::zero, gbreak::ctrl, cntrls::lri                    },  //  96
            { widths::zero, gbreak::ctrl, cntrls::rli                    },  //  97
            { widths::zero, gbreak::ctrl, cntrls::fsi                    },  //  98
            { widths::zero, gbreak::ctrl, cntrls::pdi                    },  //  99
            { widths::zero, gbreak::ctrl, cntrls::inhibit_symmetric_swapping },  // 100
            { widths::zero, gbreak::ctrl, cntrls::activate_symmetric_swapping },  // 101
            { widths::zero, gbreak::ctrl, cntrls::inhibit_arabic_form_shaping },  // 102
            { widths::zero, gbreak::ctrl, cntrls::activate_arabic_form_shaping },  // 103
            { widths::zero, gbreak::ctrl, cntrls::national_digit_shapes  },  // 104
            { widths::zero, gbreak::ctrl, cntrls::nominal_digit_shapes   },  // 105
            { widths::wide, gbreak::ep  , cntrls::non_control            },  // 106
            { widths::wide, gbreak::any , cntrls::non_control            },  // 107
            { widths::wide, gbreak::lv  , cntrls::non_control            },  // 108
            { widths::wide, gbreak::lvt , cntrls::non_control            },  // 109
            { widths::zero, gbreak::ctrl, cntrls::zwnbsp                 },  // 110
            { widths::slim, gbreak::ext , cntrls::non_control            },  // 111
            { widths::zero, gbreak::ctrl, cntrls::interlinear_annotation_anchor },  // 112
            { widths::zero, gbreak::ctrl, cntrls::interlinear_annotation_separator },  // 113
            { widths::zero, gbreak::ctrl, cntrls::interlinear_annotation_terminator },  // 114
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_vertical_joiner },  // 115
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_horizontal_joiner },  // 116
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_top_start },  // 117
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_bottom_start },  // 118
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_top_end },  // 119
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_bottom_end },  // 120
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_overlay_middle },  // 121
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_begin_segment },  // 122
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_end_segment },  // 123
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_letter_overlap },  // 124
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_continuing_overlap },  // 125
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_down_step },  // 126
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_up_step },  // 127
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_beam },  // 128
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_beam },  // 129
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_tie },  // 130
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_tie },  // 131
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_slur },  // 132
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_slur },  // 133
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_phrase },  // 134
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_phrase },  // 135
            { widths::zero, gbreak::ep  , cntrls::non_control            },  // 136
            { widths::slim, gbreak::ri  , cntrls::non_control            },  // 137
            { widths::wide, gbreak::ext , cntrls::non_control            },  // 138
        };
    };

    template<class T, class D>
    auto unpack(D const& pack, sz_t size)
    {
        auto data = std::vector<T>{};
        data.reserve(size);
        auto iter = pack;
        auto tail = pack + std::size(pack);
        while (iter < tail)
        {
            auto n = *iter++;
            if (n < 0) data.insert(data.end(), -n, *iter++);
            else       data.push_back(n);
        }
        return data;
    }

    inline unidata const& select(ui32 cp)
    {
        static auto offset = unpack<byte>(base::offset_pack, base::offset_size);
        static auto blocks = unpack<ui16>(base::blocks_pack, base::blocks_size);

        return cp > 0x10FFFF
            ? base::ucspec[0]
            : base::ucspec[offset[blocks[cp >> 8] + (cp & 0xFF)]];
    }
}