// Copyright (c) NetXS Group.
// Licensed under the MIT license.

#pragma once

namespace netxs::app::tile
{
    using backups = std::list<sptr<ui::veer>>;
}

namespace netxs::events::userland
{
    struct tile
    {
        EVENTPACK( tile, ui::e2::extra::slot4 )
        {
            EVENT_XS( backup, app::tile::backups ),
            EVENT_XS( enlist, sptr<ui::base>     ),
            EVENT_XS( delist, bool               ),
            GROUP_XS( ui    , input::hids        ), // Window manager command pack.

            SUBSET_XS( ui )
            {
                EVENT_XS( create  , input::hids ),
                EVENT_XS( close   , input::hids ),
                EVENT_XS( toggle  , input::hids ), // toggle window size: maximize/restore.
                EVENT_XS( swap    , input::hids ),
                EVENT_XS( rotate  , input::hids ), // change nested objects order. See tilimg manager (ui::fork).
                EVENT_XS( equalize, input::hids ),
                EVENT_XS( select  , input::hids ),
                GROUP_XS( split   , input::hids ),

                SUBSET_XS( split )
                {
                    EVENT_XS( vt, input::hids ),
                    EVENT_XS( hz, input::hids ),
                };
            };
        };
    };
}

// tile: Tiling window manager.
namespace netxs::app::tile
{
    static constexpr auto id = "group";
    static constexpr auto desc = "Tiling Window Manager";
    static constexpr auto inheritance_limit = 30; // Tiling limits.

    using events = netxs::events::userland::tile;

    // tile: Right-side item list.
    class items
        : public pro::skill
    {
        using depth_t = decltype(e2::depth)::type;
        using skill::boss,
              skill::memo;

        sptr<ui::list> client;
        depth_t        depth;

    public:
        items(base&&) = delete;
        items(base& boss)
            : skill{ boss },
              depth{ 0    }
        {
            client = ui::list::ctor(axis::Y, ui::sort::reverse);

            client->SIGNAL(tier::release, e2::form::upon::vtree::attached, boss.This());

            boss.LISTEN(tier::release, e2::size::any, newsz, memo)
            {
                if (client)
                {
                    auto new_coor = twod{ newsz.x + 2/*todo resize grip width*/, 0 };
                    auto new_size = twod{ client->size().x, newsz.y };
                    client->base::moveto(new_coor);
                    client->base::resize(new_size);
                }
            };
            boss.LISTEN(tier::release, events::enlist, object, memo)
            {
                if (!client) return;
                auto label = [](auto data_src_sptr, auto header)
                {
                    auto highlight_color = skin::color(tone::highlight);
                    auto c3 = highlight_color;
                    auto x3 = cell{ c3 }.alpha(0x00);

                    return ui::pads::ctor(dent{ 1, 1, 0, 0 }, dent{})
                        ->plugin<pro::fader>(x3, c3, skin::globals().fader_time)
                        ->branch(ui::item::ctor(header.empty() ? "- no title -" : header))
                        ->invoke([&](auto& boss)
                        {
                            auto update_focus = [](auto& boss, auto state)
                            {
                                auto highlight_color = skin::color(tone::highlight);
                                auto c3 = highlight_color;
                                auto x3 = cell{ c3 }.alpha(0x00);
                                boss.color(state ? 0xFF00ff00 : x3.fgc(), x3.bgc());
                            };
                            auto data_shadow = ptr::shadow(data_src_sptr);
                            boss.LISTEN(tier::release, e2::form::upon::vtree::attached, parent, boss.tracker, (data_shadow))
                            {
                                if (auto data_ptr = data_shadow.lock())
                                {
                                    data_ptr->RISEUP(tier::request, e2::form::state::keybd::focus::state, state, ());
                                    update_focus(boss, state);
                                }
                            };
                            data_src_sptr->LISTEN(tier::release, e2::form::state::keybd::focus::state, state, boss.tracker)
                            {
                                update_focus(boss, state);
                            };
                            data_src_sptr->LISTEN(tier::release, events::delist, object, boss.tracker)
                            {
                                boss.detach(); // Destroy itself.
                            };
                            boss.LISTEN(tier::release, hids::events::mouse::button::any, gear, boss.tracker, (data_shadow))
                            {
                                if (auto data_ptr = data_shadow.lock())
                                {
                                    auto deed = boss.bell::template protos<tier::release>(); //todo clang 13.0.0 requires template
                                    data_ptr->template signal<tier::release>(deed, gear); //todo "template" keyword is required by gcc version 11.3.0
                                    gear.dismiss();
                                }
                            };
                            boss.LISTEN(tier::release, e2::form::state::mouse, active, boss.tracker, (data_shadow))
                            {
                                if (auto data_ptr = data_shadow.lock())
                                {
                                    data_ptr->SIGNAL(tier::release, e2::form::state::highlight, active);
                                }
                            };
                        });
                };
                client->attach_element(e2::form::prop::ui::header, object, label);
            };
            boss.LISTEN(tier::release, e2::render::any, parent_canvas, memo)
            {
                //todo magic numbers
                if (depth < 4 && client)
                {
                    auto& basis = boss.base::coor();
                    auto canvas_view = parent_canvas.core::view();
                    auto canvas_area = parent_canvas.core::area();
                    canvas_area.coor = dot_00;
                    parent_canvas.core::view(canvas_area);
                    parent_canvas.render<faux>(client, basis);
                    parent_canvas.core::view(canvas_view);
                }
            };
            boss.LISTEN(tier::anycast, e2::form::upon::started, root, memo)
            {
                client->clear();
                depth = 0;
                boss.template riseup<tier::request>(e2::depth, depth, true);
                log("tile: start depth=", depth);
            };
        }
       ~items()
        {
            if (client)
            {
                auto lock = netxs::events::sync{};
                client->SIGNAL(tier::release, e2::form::upon::vtree::detached, empty, ());
            }
        }
    };

    namespace
    {
        auto anycasting = [](auto& boss)
        {
            boss.LISTEN(tier::release, e2::form::upon::vtree::attached, parent)
            {
                auto parent_memo = ptr::shared(subs{});
                parent->LISTEN(tier::anycast, app::tile::events::ui::any, gear, *parent_memo)
                {
                    boss.RISEUP(tier::request, e2::form::state::keybd::find, gear_test, (gear.id, 0));
                    if (gear_test.second)
                    {
                        if (auto parent = boss.parent())
                        if (auto deed = parent->bell::template protos<tier::anycast>()) //todo "template" keyword is required by clang 13.0.0
                        {
                            switch (deed)
                            {
                                case app::tile::events::ui::create.id:
                                    boss.RISEUP(tier::request, e2::form::proceed::createby, gear);
                                    break;
                                case app::tile::events::ui::close.id:
                                    boss.RISEUP(tier::preview, e2::form::quit, boss.This());
                                    break;
                                case app::tile::events::ui::toggle.id:
                                    if (boss.base::kind() == 0) // Only apps can be maximized.
                                    if (gear.countdown > 0)
                                    {
                                        gear.countdown--; // The only one can be maximized if several are selected.
                                        boss.RISEUP(tier::release, e2::form::maximize, gear);
                                    }
                                    break;
                                case app::tile::events::ui::swap.id:
                                    if (gear.countdown > 0)
                                    {
                                        boss.RISEUP(tier::release, app::tile::events::ui::swap, gear);
                                    }
                                    break;
                                case app::tile::events::ui::rotate.id:
                                    boss.RISEUP(tier::release, app::tile::events::ui::rotate, gear);
                                    break;
                                case app::tile::events::ui::equalize.id:
                                    boss.RISEUP(tier::release, app::tile::events::ui::equalize, gear);
                                    break;
                                case app::tile::events::ui::split::vt.id:
                                    boss.RISEUP(tier::release, app::tile::events::ui::split::vt, gear);
                                    break;
                                case app::tile::events::ui::split::hz.id:
                                    boss.RISEUP(tier::release, app::tile::events::ui::split::hz, gear);
                                    break;
                            }
                        }
                    }
                };
                boss.LISTEN(tier::release, e2::form::upon::vtree::detached, parent, *parent_memo, (parent_memo))
                {
                    parent_memo.reset();
                };
            };
        };
        auto mouse_subs = [](auto& boss)
        {
            boss.LISTEN(tier::release, hids::events::mouse::button::dblclick::left, gear)
            {
                boss.RISEUP(tier::release, e2::form::maximize, gear);
                gear.dismiss();
            };
            //boss.LISTEN(tier::release, hids::events::mouse::button::click::leftright, gear)
            //{
            //    boss.RISEUP(tier::release, e2::form::quit, boss.This());
            //    gear.dismiss();
            //};
            //boss.LISTEN(tier::release, hids::events::mouse::button::click::middle, gear)
            //{
            //    boss.RISEUP(tier::release, e2::form::quit, boss.This());
            //    gear.dismiss();
            //};
        };
        auto app_window = [](auto& what)
        {
            return ui::fork::ctor(axis::Y)
                    ->template plugin<pro::title>(what.header, what.footer, true, faux, true)
                    ->template plugin<pro::limit>(twod{ 10,-1 }, twod{ -1,-1 })
                    ->template plugin<pro::light>()
                    ->template plugin<pro::focus>()
                    ->isroot(true)
                    ->active()
                    ->invoke([&](auto& boss)
                    {
                        anycasting(boss);
                        mouse_subs(boss);

                        if (what.applet->size() != dot_00) boss.resize(what.applet->size() + dot_01/*approx title height*/);

                        auto master_shadow = ptr::shadow(boss.This());
                        auto applet_shadow = ptr::shadow(what.applet);
                        boss.LISTEN(tier::release, hids::events::mouse::button::drag::start::any, gear, -, (applet_shadow, master_shadow, menuid = what.menuid))
                        {
                            if (auto master_ptr = master_shadow.lock())
                            if (auto applet_ptr = applet_shadow.lock())
                            if (applet_ptr->area().hittest(gear.coord))
                            {
                                auto& master = *master_ptr;
                                auto& applet = *applet_ptr;

                                auto deed = master.bell::template protos<tier::release>();
                                if (deed != hids::events::mouse::button::drag::start::left.id
                                 && deed != hids::events::mouse::button::drag::start::leftright.id) return;

                                // Restore if maximized. Parent can be changed.
                                master.SIGNAL(tier::release, e2::form::restore, e2::form::restore.param());

                                // Take current title.
                                auto what = vtm::events::handoff.param({ .menuid = menuid });
                                master.SIGNAL(tier::request, e2::form::prop::ui::header, what.header);
                                master.SIGNAL(tier::request, e2::form::prop::ui::footer, what.footer);
                                if (what.header.empty()) what.header = menuid;

                                // Find creator.
                                master.RISEUP(tier::request, e2::config::creator, world_ptr, ());

                                // Take coor and detach from the tiling wm.
                                gear.coord -= applet.base::coor(); // Localize mouse coor.
                                what.square.size = applet.base::size();
                                applet.global(what.square.coor);
                                what.square.coor = -what.square.coor;
                                what.forced = true;
                                what.applet = applet_ptr;
                                master.SIGNAL(tier::preview, e2::form::proceed::detach, applet_ptr);
                                applet.moveto(dot_00);

                                if (auto parent_ptr = master.parent())
                                {
                                    auto gear_id_list = pro::focus::get(parent_ptr); // Expropriate all foci.
                                    world_ptr->SIGNAL(tier::request, vtm::events::handoff, what); // Attach to the world.
                                    pro::focus::set(what.applet, gear_id_list, pro::focus::solo::off, pro::focus::flip::off, true); // Refocus.
                                    master.RISEUP(tier::release, e2::form::quit, master_ptr); // Destroy placeholder.
                                }

                                // Redirect this mouse event to the new world's window.
                                gear.pass<tier::release>(what.applet, dot_00);
                            }
                        };
                        boss.LISTEN(tier::anycast, e2::form::upon::started, root)
                        {
                            boss.RISEUP(tier::release, events::enlist, boss.This());
                        };
                    })
                    ->branch(slot::_1, ui::postfx<cell::shaders::contrast>::ctor()
                        ->upload(what.header)
                        ->invoke([&](auto& boss)
                        {
                            boss.LISTEN(tier::release, e2::form::upon::vtree::attached, parent)
                            {
                                auto shadow = ptr::shadow(boss.This());
                                parent->LISTEN(tier::release, e2::form::prop::ui::title, head_foci, -, (shadow))
                                {
                                    if (auto boss_ptr = shadow.lock())
                                    {
                                        boss_ptr->upload(head_foci);
                                    }
                                };
                            };
                        }))
                    ->branch(slot::_2, what.applet);
        };
        auto built_node = [](auto tag, auto slot1, auto slot2, auto grip_width)
        {
            auto node = tag == 'h' ? ui::fork::ctor(axis::X, grip_width == -1 ? 2 : grip_width, slot1, slot2)
                                   : ui::fork::ctor(axis::Y, grip_width == -1 ? 1 : grip_width, slot1, slot2);
            node->isroot(faux, 1) // Set object kind to 1 to be different from others. See empty_slot::select.
                ->template plugin<pro::limit>(dot_00)
                ->template plugin<pro::focus>()
                ->invoke([&](auto& boss)
                {
                    mouse_subs(boss);
                    boss.LISTEN(tier::release, app::tile::events::ui::swap    , gear) { boss.swap();       };
                    boss.LISTEN(tier::release, app::tile::events::ui::rotate  , gear) { boss.rotate();     };
                    boss.LISTEN(tier::release, app::tile::events::ui::equalize, gear) { boss.config(1, 1); };
                });
                auto grip = node->attach(slot::_I,
                                ui::mock::ctor()
                                ->isroot(true)
                                ->template plugin<pro::mover>() //todo GCC 11 requires template keyword
                                ->template plugin<pro::focus>(pro::focus::mode::focusable)
                                ->template plugin<pro::track>(true)
                                ->template plugin<pro::shade<cell::shaders::xlight>>()
                                ->invoke([&](auto& boss)
                                {
                                    anycasting(boss);
                                    //todo implement keydb support
                                })
                                ->active());
            return node;
        };
        auto empty_pane = []
        {
            auto menu_black = skin::color(tone::menu_black);
            auto cC = menu_black;

            using namespace app::shared;
            auto [menu_block, cover, menu_data] = menu::mini(true, true, faux, true,
            menu::list
            {
                { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = " +", .notes = " New app " } }}),
                [](ui::pads& boss, auto& item)
                {
                    boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                    {
                        boss.RISEUP(tier::request, e2::form::proceed::createby, gear);
                        gear.dismiss(true);
                    };
                }},
                { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "│", .notes = " Split horizontally " } }}),
                [](ui::pads& boss, auto& item)
                {
                    boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                    {
                        boss.RISEUP(tier::release, app::tile::events::ui::split::hz, gear);
                        gear.dismiss(true);
                    };
                }},
                { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "──", .notes = " Split vertically " } }}),
                [](ui::pads& boss, auto& item)
                {
                    boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                    {
                        boss.RISEUP(tier::release, app::tile::events::ui::split::vt, gear);
                        gear.dismiss(true);
                    };
                }},
                { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "×", .notes = " Delete pane " } }}),
                [](ui::pads& boss, auto& item)
                {
                    boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                    {
                        boss.RISEUP(tier::release, e2::form::quit, boss.This());
                        gear.dismiss(true);
                    };
                }},
            });
            menu_data->colors(cC.fgc(), cC.bgc());
            auto menu_id = menu_block->id;
            cover->invoke([&](auto& boss)
            {
                boss.LISTEN(tier::release, e2::render::any, parent_canvas, -, (menu_id))
                {
                    parent_canvas.fill([&](cell& c) { c.txt(whitespace).link(menu_id); });
                };
            });

            return ui::park::ctor()
                ->isroot(true, 2)
                ->colors(cC.fgc(), cC.bgc())
                ->plugin<pro::limit>(dot_00, -dot_11)
                ->plugin<pro::focus>(pro::focus::mode::focusable)
                ->plugin<pro::track>(true)
                ->invoke([&](auto& boss)
                {
                    anycasting(boss);
                    mouse_subs(boss);
                    boss.LISTEN(tier::release, hids::events::mouse::button::click::right, gear)
                    {
                        boss.RISEUP(tier::request, e2::form::proceed::createby, gear);
                        gear.dismiss();
                    };
                })
                ->branch
                (
                    snap::center, snap::center,
                    ui::post::ctor()->upload("Empty Slot", 10)
                )
                ->branch
                (
                    snap::stretch, snap::head,
                    menu_block
                );
        };
        auto empty_slot = [](auto&& empty_slot) -> sptr<ui::veer>
        {
            return ui::veer::ctor()
                ->plugin<pro::focus>(pro::focus::mode::hub/*default*/, true/*default*/, true)
                ->invoke([&](auto& boss)
                {
                    auto highlight = [](auto& boss, auto state)
                    {
                        auto c = state ? cell{ skin::color(tone::highlight) }.alpha(0x70)
                                       : cell{ skin::color(tone::menu_black) };
                        boss.front()->color(c.fgc(), c.bgc());
                        boss.deface();
                    };
                    boss.LISTEN(tier::release, e2::config::plugins::sizer::alive, state)
                    {
                        // Block rising up this event: DTVT object fires this event on exit.
                    };
                    boss.LISTEN(tier::release, vtm::events::d_n_d::abort, target)
                    {
                        if (boss.count())
                        {
                            highlight(boss, faux);
                        }
                    };
                    boss.LISTEN(tier::release, vtm::events::d_n_d::ask, target)
                    {
                        if (boss.count() == 1) // Only empty slot available.
                        {
                            highlight(boss, true);
                            target = boss.This();
                        }
                    };
                    boss.LISTEN(tier::release, vtm::events::d_n_d::drop, what)
                    {
                        if (boss.count() == 1) // Only empty pane/slot available.
                        {
                            highlight(boss, faux);
                            pro::focus::off(boss.This());
                            auto gear_id_list = pro::focus::get(what.applet);
                            auto app = app_window(what);
                            boss.attach(app);
                            app->SIGNAL(tier::anycast, e2::form::upon::started, app);
                            pro::focus::set(what.applet, gear_id_list, pro::focus::solo::mix, pro::focus::flip::off, true);
                        }
                    };
                    boss.LISTEN(tier::release, e2::form::proceed::swap, item_ptr)
                    {
                        if (boss.count() == 1) // Only empty slot available.
                        {
                            log("tile: empty_slot swap: defective structure, count=", boss.count());
                        }
                        else if (boss.count() == 2)
                        {
                            auto gear_id_list = pro::focus::get(boss.This());
                            auto deleted_item = boss.pop_back();
                            if (item_ptr)
                            {
                                boss.attach(item_ptr);
                            }
                            else item_ptr = boss.This();
                            pro::focus::set(boss.back(), gear_id_list, pro::focus::solo::off, pro::focus::flip::off);
                        }
                        else log("tile: empty_slot swap: defective structure, count=", boss.count());
                    };
                    boss.LISTEN(tier::release, e2::form::upon::vtree::attached, parent)
                    {
                        auto parent_memo = ptr::shared<subs>();
                        parent->LISTEN(tier::request, e2::form::proceed::swap, item_ptr, *parent_memo)
                        {
                            if (item_ptr != boss.This())
                            {
                                if (boss.count() == 1) // Only empty slot available.
                                {
                                    item_ptr.reset();
                                }
                                else if (boss.count() == 2)
                                {
                                    auto gear_id_list = pro::focus::get(boss.This());
                                    item_ptr = boss.pop_back();
                                    pro::focus::set(boss.back(), gear_id_list, pro::focus::solo::off, pro::focus::flip::off);
                                }
                                else log("tile: empty_slot: defective structure, count=", boss.count());
                                if (auto parent = boss.parent())
                                {
                                    parent->bell::template expire<tier::request>();
                                }
                            }
                        };
                        boss.LISTEN(tier::request /*swap specific*/, e2::form::upon::vtree::detached, parent, *parent_memo, (parent_memo))
                        {
                            parent_memo.reset();
                        };
                    };
                    boss.LISTEN(tier::anycast, e2::form::upon::started, root)
                    {
                        if (auto item_ptr = boss.back())
                        {
                            auto& item = *item_ptr;
                            if (item.base::root())
                            {
                                item.SIGNAL(tier::anycast, e2::form::upon::started, item_ptr);
                            }
                        }
                    };
                    boss.LISTEN(tier::anycast, app::tile::events::ui::select, gear)
                    {
                        auto item_ptr = boss.back();
                        if (item_ptr->base::kind() != 1) pro::focus::set(item_ptr, gear.id, pro::focus::solo::off, pro::focus::flip::off);
                        else                             pro::focus::off(item_ptr, gear.id); // Exclude grips.
                    };
                    boss.LISTEN(tier::release, e2::form::maximize, gear, -, (oneoff = subs{}))
                    {
                        if (boss.count() > 2 || oneoff) // It is a root or is already maximized. See build_inst::slot::_2's e2::form::proceed::attach for details.
                        {
                            boss.RISEUP(tier::release, e2::form::proceed::attach, e2::form::proceed::attach.param());
                        }
                        else
                        {
                            if (boss.count() > 1) // Preventing the empty slot from maximizing.
                            if (boss.back()->base::kind() == 0) // Preventing the splitter from maximizing.
                            if (auto fullscreen_item = boss.pop_back())
                            {
                                auto gear_id_list = pro::focus::get(boss.This()); // Seize all foci.
                                fullscreen_item->LISTEN(tier::release, e2::form::restore, item_ptr, oneoff)
                                {
                                    if (item_ptr)
                                    {
                                        boss.attach(item_ptr);
                                        boss.base::reflow();
                                    }
                                    oneoff.reset();
                                };
                                fullscreen_item->LISTEN(tier::release, e2::dtor, item_ptr, oneoff)
                                {
                                    oneoff.reset();
                                };
                                auto just_copy = fullscreen_item;
                                boss.RISEUP(tier::release, e2::form::proceed::attach, fullscreen_item);
                                pro::focus::set(just_copy, gear_id_list, pro::focus::solo::off, pro::focus::flip::off); // Handover all foci.
                                boss.base::reflow();
                            }
                        }
                    };
                    boss.LISTEN(tier::release, app::tile::events::ui::split::any, gear)
                    {
                        if (auto deed = boss.bell::template protos<tier::release>())
                        {
                            auto depth = e2::depth.param();
                            boss.template riseup<tier::request>(e2::depth, depth, true);
                            if constexpr (debugmode) log("tile: depth=", depth);
                            if (depth > inheritance_limit) return;

                            auto heading = deed == app::tile::events::ui::split::vt.id;
                            auto newnode = built_node(heading ? 'v':'h', 1, 1, heading ? 1 : 2);
                            auto empty_1 = empty_slot(empty_slot);
                            auto empty_2 = empty_slot(empty_slot);
                            auto gear_id = pro::focus::get(boss.This()); // Seize all foci.
                            auto curitem = boss.pop_back();
                            if (boss.empty())
                            {
                                boss.attach(empty_pane());
                                empty_1->pop_back();
                            }
                            auto slot_1 = newnode->attach(slot::_1, empty_1->branch(curitem));
                            auto slot_2 = newnode->attach(slot::_2, empty_2);
                            boss.attach(newnode);
                            pro::focus::set(slot_1->back(), gear_id, pro::focus::solo::off, pro::focus::flip::off); // Handover all foci.
                            pro::focus::set(slot_2->back(), gear_id, pro::focus::solo::off, pro::focus::flip::off);
                        }
                    };
                    boss.LISTEN(tier::anycast, e2::form::quit, nested_item_ptr)
                    {
                        boss.SIGNAL(tier::preview, e2::form::quit, nested_item_ptr);
                    };
                    boss.LISTEN(tier::preview, e2::form::quit, nested_item_ptr)
                    {
                        if (boss.count() > 1 && boss.back()->base::kind() == 0)
                        {
                            boss.back()->SIGNAL(tier::anycast, e2::form::quit, nested_item_ptr);
                        }
                        else boss.SIGNAL(tier::release, e2::form::quit, nested_item_ptr);
                    };
                    boss.LISTEN(tier::release, e2::form::quit, nested_item_ptr)
                    {
                        if (auto parent = boss.base::parent())
                        if (nested_item_ptr)
                        {
                            if (boss.count() > 1 && boss.back()->base::kind() == 0) // Only apps can be deleted.
                            {
                                auto gear_id_list = pro::focus::get(boss.This());
                                auto deleted_item = boss.pop_back(); // Throw away.
                                pro::focus::set(boss.back(), gear_id_list, pro::focus::solo::off, pro::focus::flip::off);
                            }
                            else if (boss.count() == 1) // Remove empty slot, reorganize.
                            {
                                parent->SIGNAL(tier::request, e2::form::proceed::swap, item_ptr, (boss.This())); // sptr must be of the same type as the event argument. Casting kills all intermediaries when return.
                                if (item_ptr != boss.This()) // Parallel slot is not empty or both slots are empty (item_ptr == null).
                                {
                                    parent->RISEUP(tier::release, e2::form::proceed::swap, item_ptr);
                                }
                            }
                            boss.deface();
                        }
                    };
                    boss.LISTEN(tier::request, e2::form::proceed::createby, gear)
                    {
                        static auto insts_count = 0;

                        if (boss.count() != 1) return; // Create new apps at the empty slots only.
                        auto& gate = gear.owner;
                        gate.SIGNAL(tier::request, e2::data::changed, current_default, ());
                        gate.RISEUP(tier::request, vtm::events::newapp, config, ({ .menuid = current_default }));

                        auto app = app_window(config);
                        auto gear_id_list = pro::focus::get(boss.back());
                        boss.attach(app);
                        if (auto world_ptr = gate.parent()) // Finalize app creation.
                        {
                            app->SIGNAL(tier::anycast, vtm::events::attached, world_ptr);
                        }

                        insts_count++; //todo unify, demo limits
                        config.applet->LISTEN(tier::release, e2::dtor, id)
                        {
                            insts_count--;
                            log("tile: inst: detached: ", insts_count, " id=", id);
                        };

                        app->SIGNAL(tier::anycast, e2::form::upon::started, app);
                        if (std::find(gear_id_list.begin(), gear_id_list.end(), gear.id) == gear_id_list.end())
                        {
                            gear_id_list.push_back(gear.id);
                        }
                        pro::focus::set(app, gear_id_list, pro::focus::solo::off, pro::focus::flip::off);
                    };
                    boss.LISTEN(tier::release, events::backup, empty_slot_list)
                    {
                        if (boss.count())
                        if (auto item_ptr = boss.back())
                        if (item_ptr->base::root())
                        {
                            empty_slot_list.push_back(boss.This());
                        }
                    };
                })
                ->branch(empty_pane());
        };
        auto parse_data = [](auto&& parse_data, view& utf8) -> sptr<ui::veer>
        {
            auto slot = empty_slot(empty_slot);
            utf::trim_front(utf8, ", ");
            if (utf8.empty()) return slot;
            auto tag = utf8.front();
            if ((tag == 'h' || tag == 'v') && utf8.find('(') < utf8.find(','))
            {
                // add split
                utf8.remove_prefix(1);
                utf::trim_front(utf8, " ");
                auto s1 = si32{ 1 };
                auto s2 = si32{ 1 };
                auto w  = si32{-1 };
                if (auto v = utf::to_int(utf8)) // Left side ratio
                {
                    s1 = std::abs(v.value());
                    if (utf8.empty() || utf8.front() != ':') return slot;
                    utf8.remove_prefix(1);
                    if (auto v = utf::to_int(utf8)) // Right side ratio
                    {
                        s2 = std::abs(v.value());
                        utf::trim_front(utf8, " ");
                        if (!utf8.empty() && utf8.front() == ':') // Grip width.
                        {
                            utf8.remove_prefix(1);
                            if (auto v = utf::to_int(utf8))
                            {
                                w = std::abs(v.value());
                                utf::trim_front(utf8, " ");
                            }
                        }
                    }
                    else return slot;
                }
                if (utf8.empty() || utf8.front() != '(') return slot;
                utf8.remove_prefix(1);
                auto node = built_node(tag, s1, s2, w);
                auto slot1 = node->attach(ui::slot::_1, parse_data(parse_data, utf8));
                auto slot2 = node->attach(ui::slot::_2, parse_data(parse_data, utf8));
                slot->attach(node);

                utf::trim_front(utf8, ") ");
            }
            else  // Add application.
            {
                utf::trim_front(utf8, " ");
                auto menuid = text{ utf::get_tail(utf8, " ,)") };
                if (menuid.empty()) return slot;

                utf::trim_front(utf8, " ,");
                if (utf8.size() && utf8.front() == ')') utf8.remove_prefix(1); // pop ')';

                auto oneoff = ptr::shared(hook{});
                slot->LISTEN(tier::anycast, vtm::events::attached, world_ptr, *oneoff, (oneoff, menuid, slot))
                {
                    world_ptr->SIGNAL(tier::request, vtm::events::newapp, what, ({ .menuid = menuid }));
                    auto inst = app_window(what);
                    slot->attach(inst);
                    inst->SIGNAL(tier::anycast, vtm::events::attached, world_ptr);
                    oneoff.reset();
                };
            }
            return slot;
        };
        auto build_inst = [](text cwd, view param, xmls& config, text patch) -> sptr<base>
        {
            auto menu_white = skin::color(tone::menu_white);
            auto cB = menu_white;

            auto object = ui::fork::ctor(axis::Y)
                ->plugin<items>()
                ->invoke([&](auto& boss)
                {
                    auto oneoff = ptr::shared(hook{});
                    boss.LISTEN(tier::anycast, e2::form::upon::created, gear, *oneoff, (oneoff))
                    {
                        auto& gate = gear.owner;
                        gate.SIGNAL(tier::request, e2::data::changed, menuid, ());
                        gate.RISEUP(tier::request, desk::events::menu, conf_list_ptr, ());
                        auto& conf_list = *conf_list_ptr;
                        auto& config = conf_list[menuid];
                        if (config.type == app::tile::id) // Reset the currently selected application to the previous one.
                        {
                            gate.SIGNAL(tier::preview, e2::data::changed, menuid); // Get previous default;
                            gate.SIGNAL(tier::release, e2::data::changed, menuid); // Set current  default;
                        }
                        oneoff.reset();
                    };
                });

            config.cd("/config/tile/", "/config/defapp/");

            using namespace app::shared;
            auto [menu_block, cover, menu_data] = menu::create(config,
                    menu::list
                    {
                        //  Green                                  ?Even    Red
                        // ┌────┐  ┌────┐  ┌─┬──┐  ┌────┐  ┌─┬──┐  ┌─┬──┐  ┌────┐  // ┌─┐  ┌─┬─┐  ┌─┬─┐  ┌─┬─┐  
                        // │Exec│  ├─┐  │  │ H  │  ├ V ─┤  │Swap│  │Fair│  │Shut│  // ├─┤  └─┴─┘  └<┴>┘  └>┴<┘  
                        // └────┘  └─┴──┘  └─┴──┘  └────┘  └─┴──┘  └─┴──┘  └────┘  // └─┘                       
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "  ┐└  ", .notes = " Maximize/restore active pane " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                gear.countdown = 1;
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::toggle, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "  +  ", .notes = " Create and run a new app in active panes " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::create, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = " ::: ", .notes = " Select all panes " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::select, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "  │  ", .notes = " Split active panes horizontally " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::split::hz, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = " ── ", .notes = " Split active panes vertically " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::split::vt, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "  ┌┘  ", .notes = " Change split orientation " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::rotate, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = " <-> ", .notes = " Swap two or more panes " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::swap, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = " >|< ", .notes = " Equalize split ratio " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::equalize, gear);
                                gear.dismiss(true);
                            };
                        }},
                        { ptr::shared(menu::item{ menu::item::type::Command, true, 0, std::vector<menu::item::look>{ { .label = "  ×  ", .notes = " Close active app or remove pane if there is no running app " } }}),
                        [](ui::pads& boss, auto& item)
                        {
                            boss.LISTEN(tier::release, hids::events::mouse::button::click::left, gear)
                            {
                                boss.SIGNAL(tier::anycast, app::tile::events::ui::close, gear);
                                gear.dismiss(true);
                            };
                        }},
                    });
            object->attach(slot::_1, menu_block)
                  ->invoke([](auto& boss)
                  {
                      boss.LISTEN(tier::anycast, e2::form::quit, item)
                      {
                          boss.RISEUP(tier::release, e2::form::quit, item);
                      };
                  });
            menu_data->colors(cB.fgc(), cB.bgc())
                     ->plugin<pro::track>()
                     ->plugin<pro::acryl>();
            auto menu_id = menu_block->id;
            cover->invoke([&](auto& boss)
            {
                auto bar = cell{ "▀"sv }.link(menu_id);
                boss.LISTEN(tier::release, e2::render::any, parent_canvas, -, (bar))
                {
                    auto menu_white = skin::color(tone::menu_white);
                    auto fgc = menu_white.bgc();
                    parent_canvas.fill([&](cell& c) { c.fgc(fgc).txt(bar).link(bar); });
                };
            });
            if (cwd.size())
            {
                auto err = std::error_code{};
                fs::current_path(cwd, err);
                if (err) log("tile: failed to change current working directory to '", cwd, "', error code: ", err.value());
                else     log("tile: change current working directory to '", cwd, "'");
            }

            object->attach(slot::_2, parse_data(parse_data, param))
                ->invoke([&](auto& boss)
                {
                    boss.LISTEN(tier::release, e2::form::proceed::attach, fullscreen_item, -, (foci_list = gear_id_list_t{}))
                    {
                        if (boss.count() > 2)
                        {
                            auto gear_id_list = pro::focus::get(boss.This()); // Seize all foci.
                            auto item_ptr = boss.pop_back();
                            item_ptr->SIGNAL(tier::release, e2::form::restore, item_ptr);
                            pro::focus::set(boss.back(), foci_list, pro::focus::solo::off, pro::focus::flip::off, true); // Restore saved foci.
                            pro::focus::set(item_ptr, gear_id_list, pro::focus::solo::off, pro::focus::flip::off); // Apply item's foci.
                            foci_list.clear();
                        }

                        if (fullscreen_item)
                        {
                            foci_list = pro::focus::get(boss.This()); // Save all foci.
                            boss.attach(fullscreen_item);
                            fullscreen_item.reset();
                        }
                        else log("tile: fullscreen_item is empty");
                    };
                    boss.LISTEN(tier::anycast, app::tile::events::ui::any, gear)
                    {
                        if (auto deed = boss.bell::template protos<tier::anycast>()) //todo "template" keyword is required by clang 13.0.0
                        {
                            if (boss.count() > 2 && deed != app::tile::events::ui::toggle.id) // Restore the window before any action if maximized.
                            {
                                boss.RISEUP(tier::release, e2::form::proceed::attach, e2::form::proceed::attach.param());
                            }

                            if (deed == app::tile::events::ui::swap.id)
                            {
                                auto empty_slot_list = backups{};
                                auto proc = e2::form::proceed::functor.param([&](sptr<base> item_ptr)
                                {
                                    item_ptr->SIGNAL(tier::request, e2::form::state::keybd::find, gear_test, (gear.id, 0));
                                    if (gear_test.second)
                                    {
                                        item_ptr->RISEUP(tier::release, events::backup, empty_slot_list);
                                    }
                                });
                                boss.SIGNAL(tier::general, e2::form::proceed::functor, proc);
                                auto slots_count = empty_slot_list.size();
                                log("tile: slots count=", slots_count);
                                if (slots_count >= 2) // Swap selected panes cyclically.
                                {
                                    using slot = sptr<base>;
                                    log("tile: Swap slots cyclically");
                                    auto i = 0;
                                    auto emp_slot = slot{};
                                    auto app_slot = slot{};
                                    auto emp_next = slot{};
                                    auto app_next = slot{};
                                    for (auto& s : empty_slot_list)
                                    {
                                        if (s->count() == 1) // empty only
                                        {
                                            app_next.reset();
                                            emp_next = s->pop_back();
                                        }
                                        else if (s->count() == 2) // empty + app
                                        {
                                            if (auto app = s->back())
                                            {
                                                app->SIGNAL(tier::release, events::delist, true);
                                            }
                                            app_next = s->pop_back();
                                            emp_next = s->pop_back();
                                        }
                                        if (emp_slot) s->attach(emp_slot);
                                        if (app_slot)
                                        {
                                            s->attach(app_slot);
                                            app_slot->RISEUP(tier::release, events::enlist, app_slot);
                                        }
                                        std::swap(emp_slot, emp_next);
                                        std::swap(app_slot, app_next);
                                    }
                                    auto& s = empty_slot_list.front();
                                    if (emp_slot) s->attach(emp_slot);
                                    if (app_slot)
                                    {
                                        s->attach(app_slot);
                                        app_slot->RISEUP(tier::release, events::enlist, app_slot);
                                    }
                                    gear.countdown = 0; // Interrupt swapping.
                                }
                                else // Swap panes in split.
                                {
                                    gear.countdown = 1;
                                }
                            }
                        }
                    };
                });
            return object;
        };
    }

    app::shared::initialize builder{ app::tile::id, build_inst };
}