// -*- coding: utf-8 -*-
// Copyright (C) 2011, 2013 Laboratoire de Recherche et
// Developpement de l'Epita (LRDE).
// Copyright (C) 2004  Laboratoire d'Informatique de Paris 6 (LIP6),
// département Systèmes Répartis Coopératifs (SRC), Université Pierre
// et Marie Curie.
//
// This file is part of Spot, a model checking library.
//
// Spot is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// Spot is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
// License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef SPOT_TGBAALGOS_DOTTYDEC_HH
# define SPOT_TGBAALGOS_DOTTYDEC_HH

# include "misc/common.hh"
# include <string>

namespace spot
{
  class state;
  class tgba;
  class tgba_succ_iterator;

  /// \addtogroup tgba_dotty Decorating the dot output
  /// \ingroup tgba_io

  /// \ingroup tgba_dotty
  /// \brief Choose state and link styles for spot::dotty_reachable.
  class SPOT_API dotty_decorator
  {
  public:
    virtual ~dotty_decorator();

    /// \brief Compute the style of a state.
    ///
    /// This function should output a string of the form
    /// <code>[label="foo", style=bar, ...]</code>.  The
    /// default implementation will simply output <code>[label="LABEL"]</code>
    /// with <code>LABEL</code> replaced by the value of \a label.
    ///
    /// \param a the automaton being drawn
    /// \param s the state being drawn (owned by the caller)
    /// \param n a unique number for this state
    /// \param si an iterator over the successors of this state (owned by the
    ///           caller, but can be freely iterated)
    /// \param label the computed name of this state
    /// \param accepting whether the state is accepting (it makes sense only
    ///                  for state-acceptance automata)
    virtual std::string state_decl(const tgba* a, const state* s, int n,
				   tgba_succ_iterator* si,
				   const std::string& label,
				   bool accepting);

    /// \brief Compute the style of a link.
    ///
    /// This function should output a string of the form
    /// <code>[label="foo", style=bar, ...]</code>.  The
    /// default implementation will simply output <code>[label="LABEL"]</code>
    /// with <code>LABEL</code> replaced by the value of \a label.
    ///
    /// \param a the automaton being drawn
    /// \param in_s the source state of the transition being drawn
    ///             (owned by the caller)
    /// \param in the unique number associated to \a in_s
    /// \param out_s the destination state of the transition being drawn
    ///             (owned by the caller)
    /// \param out the unique number associated to \a out_s
    /// \param si an iterator over the successors of \a in_s, pointing to
    ///          the current transition (owned by the caller and cannot
    ///        be iterated)
    /// \param label the computed name of this state
    virtual std::string link_decl(const tgba* a,
				  const state* in_s, int in,
				  const state* out_s, int out,
				  const tgba_succ_iterator* si,
				  const std::string& label);

    /// Get the unique instance of the default dotty_decorator.
    static dotty_decorator* instance();
  protected:
    dotty_decorator();
  };

}

#endif // SPOT_TGBAALGOS_DOTTYDEC_HH
