#include <RooAddPdf.h>
#include <RooDataSet.h>
#include <RooGaussian.h>
#include <RooRealVar.h>
#include <RooVoigtian.h>
#include <RooFitResult.h>
#include <RooNumIntConfig.h>

#include <gtest/gtest.h>

namespace {

// Input for the ROOT_10615 test. Since the input in not that large, it is hardcoded in the source.
std::vector<double> dtVals{
   2.8266379833221436,    -5.922840595245361,     -2.550363063812256,    -0.19226482510566711,  -1.156746506690979,
   -0.9937527179718018,   -0.12522433698177338,   -3.5416343212127686,   0.8696247339248657,    -2.6176908016204834,
   -0.6434765458106995,   7.2987060546875,        -0.8709370493888855,   0.47036510705947876,   -2.382615089416504,
   1.6966360807418823,    -85.43426513671875,     -0.1115502119064331,   5.010952949523926,     0.6995347738265991,
   -1.766488790512085,    -12.13000774383545,     -5.561145305633545,    -0.8444245457649231,   -0.0728561282157898,
   10.132277488708496,    1.0803542137145996,     1.111778974533081,     -2.4715583324432373,   -2.719576120376587,
   -0.6878403425216675,   0.001776090357452631,   -4.763464450836182,    1.6973565816879272,    3.7190158367156982,
   1.0734740495681763,    -1.678210973739624,     -1.633002519607544,    -0.04392572119832039,  -7.136231422424316,
   2.2054636478424072,    -0.46009859442710876,   -3.8151028156280518,   0.02724119834601879,   -2.5054469108581543,
   3.931730031967163,     0.8433017730712891,     -2.0951616764068604,   -0.2110850065946579,   -0.7834380865097046,
   -2.5010712146759033,   -5.5043792724609375,    10.291595458984375,    0.5632269978523254,    0.07301951199769974,
   -0.49804824590682983,  -21.912492752075195,    -2.922745943069458,    2.2575149536132812,    -7.440819263458252,
   1.5310102701187134,    -0.2791447639465332,    -3.003908395767212,    0.46010252833366394,   0.22349806129932404,
   2.376739263534546,     -0.8524564504623413,    3.051076889038086,     -1.9290188550949097,   -0.2410321682691574,
   0.8958514332771301,    0.8146747946739197,     -0.394214391708374,    -0.9892944097518921,   -0.8951123952865601,
   -1.0830937623977661,   0.80262291431427,       0.5363758206367493,    -1.570073127746582,    0.9778718948364258,
   -1.6733402013778687,   -2.3714420795440674,    0.5871254801750183,    1.5519882440567017,    -0.7291706800460815,
   0.40138664841651917,   -0.4024689197540283,    1.8067342042922974,    -1.83309805393219,     -3.5900111198425293,
   -1.5600225925445557,   -5.132297515869141,     -5.641075134277344,    -3.7356505393981934,   0.7781169414520264,
   -0.7191025018692017,   -0.6919004321098328,    52.0797004699707,      -6.511929035186768,    -0.5654265880584717,
   5.117639064788818,     -0.7519070506095886,    -2.0677459239959717,   2.9614710807800293,    0.056742019951343536,
   -1.0319997072219849,   0.8772448897361755,     -0.5201661586761475,   -3.0049562454223633,   0.47352632880210876,
   1.8169718980789185,    0.038413990288972855,   0.21474145352840424,   0.21196597814559937,   3.089988946914673,
   -1.252010703086853,    0.29640865325927734,    1.1744259595870972,    0.7405098080635071,    3.599918842315674,
   -8.332734107971191,    1.9649077653884888,     3.0272293090820312,    -2.903008222579956,    4.064101219177246,
   -1.7432153224945068,   0.24674241244792938,    1.163784146308899,     0.4338519275188446,    1.7183319330215454,
   0.17795982956886292,   -0.789238452911377,     -0.8550190329551697,   -2.743443250656128,    -2.9563374519348145,
   -0.21050390601158142,  1.2634727954864502,     0.8288640379905701,    -3.3362762928009033,   -1.0165200233459473,
   2.7145466804504395,    -2.3097925186157227,    0.6325840950012207,    -4.138699054718018,    -0.43038612604141235,
   3.2873382568359375,    -1.5398205518722534,    -0.15655113756656647,  -1.7568402290344238,   -2.33644700050354,
   0.2161458283662796,    -2.95906662940979,      -0.9148457646369934,   1.6522555351257324,    -0.43657341599464417,
   -2.8738155364990234,   2.563164472579956,      0.015573251061141491,  -2.169109582901001,    -0.49365246295928955,
   0.5760107636451721,    -6.630191802978516,     -6.062448501586914,    -3.0835909843444824,   0.0455557182431221,
   -1.4176547527313232,   -20.11192512512207,     0.6139457821846008,    -0.6418310403823853,   0.22401411831378937,
   1.0052326917648315,    -0.43452516198158264,   -1.4577398300170898,   -0.4707539975643158,   -0.9795296788215637,
   -16.97727394104004,    -0.12272408604621887,   -3.4410831928253174,   -1.0799919366836548,   -2.695807695388794,
   0.18735012412071228,   -0.7020919919013977,    -1.092464804649353,    -3.8430287837982178,   -1.778789758682251,
   3.0498201847076416,    -1.020994782447815,     3.6186459064483643,    -3.514247417449951,    -0.8163723945617676,
   1.6989719867706299,    28.883350372314453,     -2.451812267303467,    1.6620678901672363,    -1.5374045372009277,
   -0.6689706444740295,   -3.056084394454956,     1.7914581298828125,    0.5941452383995056,    1.5185943841934204,
   -0.7340480089187622,   0.5841600894927979,     -0.35326051712036133,  -1.1610121726989746,   -1.8756312131881714,
   -0.5381126403808594,   -0.8679079413414001,    1.6819297075271606,    2.0014541149139404,    -0.467937171459198,
   2.972003698348999,     0.5106955766677856,     1.8488608598709106,    0.6056933999061584,    1.813750982284546,
   -4.160813808441162,    -4.067713737487793,     -2.523925542831421,    -1.5932868719100952,   -2.714006185531616,
   16.12789535522461,     0.03500473499298096,    0.9919832944869995,    0.14147064089775085,   1.721386194229126,
   -5.856757640838623,    -5.303279399871826,     1.2236930131912231,    1.927039384841919,     -1.117587924003601,
   0.918979287147522,     -0.615105926990509,     -0.018047841265797615, 2.190584659576416,     -0.803921639919281,
   4.962000846862793,     0.1306103765964508,     -0.7851812243461609,   0.19922390580177307,   -0.2644040584564209,
   0.13281972706317902,   -2.9220573902130127,    -0.23227094113826752,  -2.956996202468872,    0.2897966206073761,
   2.066275119781494,     68.9346923828125,       -3.101546049118042,    -1.0338068008422852,   -1.3424428701400757,
   -0.21454459428787231,  3.439805269241333,      -1.9863464832305908,   -4.102566719055176,    -0.5688574314117432,
   0.10946884751319885,   1.6048563718795776,     -3.1592068672180176,   6.110762596130371,     -0.7900059223175049,
   -0.7384921908378601,   1.8494611978530884,     1.0367982387542725,    -1.2035980224609375,   -3.8609354496002197,
   -1.5734144449234009,   -1.44398832321167,      -1.7853071689605713,   0.4306163489818573,    -55.47184371948242,
   -4.197507381439209,    0.4317173361778259,     13.710407257080078,    -2.897394895553589,    -0.6417321562767029,
   -2.4179952144622803,   -3.1099514961242676,    -18.880661010742188,   0.736578106880188,     4.328418254852295,
   0.3849676549434662,    0.19863088428974152,    -1.8348406553268433,   -0.028759099543094635, 0.00844759400933981,
   34.13146209716797,     0.29596054553985596,    -0.6970908641815186,   -0.1457819938659668,   0.5187986493110657,
   1.9436503648757935,    0.7974480390548706,     -1.9456027746200562,   3.0615129470825195,    2.420133590698242,
   0.26841551065444946,   -1.0943926572799683,    -3.193056106567383,    -2.3370206356048584,   4.647980690002441,
   0.7149807810783386,    2.1656644344329834,     -2.5500168800354004,   1.1677683591842651,    1.312869906425476,
   2.1911091804504395,    3.067445993423462,      -0.09058808535337448,  -2.698469638824463,    0.43482691049575806,
   -2.1604936122894287,   -1.2120847702026367,    -1.2060377597808838,   0.02373606339097023,   -2.655150890350342,
   1.918819785118103,     -0.8632082343101501,    2.4873251914978027,    -2.8296403884887695,   -0.5668258666992188,
   -0.036190200597047806, 4.359323024749756,      -3.39522123336792,     3.0655605792999268,    -1.6216464042663574,
   0.10085298120975494,   -1.4343738555908203,    1.629604458808899,     0.3287490904331207,    1.3653404712677002,
   -0.43771693110466003,  0.2630595564842224,     -3.2509243488311768,   -4.602761268615723,    -0.955985426902771,
   -1.9656364917755127,   -0.7165999412536621,    0.653108537197113,     -0.31122228503227234,  -2.0641541481018066,
   -4.262363910675049,    1.2590492963790894,     0.5570814609527588,    -1.183885931968689,    -3.7674102783203125,
   -1.8920069932937622,   -0.0456564798951149,    6.110235214233398,     -0.5294173955917358,   -0.047908250242471695,
   1.3371351957321167,    -0.7059128880500793,    -1.334913730621338,    -0.6085501909255981,   1.1620155572891235,
   0.42975500226020813,   0.4144030213356018,     2.1295318603515625,    -1.542677640914917,    0.5336112380027771,
   -5.6661176681518555,   1.9979193210601807,     -4.378593444824219,    0.548524022102356,     -1.0859588384628296,
   -1.280246615409851,    0.6730429530143738,     -0.30895134806632996,  -0.4144652187824249,   -2.9721436500549316,
   0.07800689339637756,   1.6902259588241577,     -0.15527556836605072,  -5.621766567230225,    -0.2285587042570114,
   -3.9352805614471436,   -1.0912928581237793,    -1.3998512029647827,   -0.09776410460472107,  1.2070242166519165,
   5.950258731842041,     0.2621818482875824,     -1.015973687171936,    -1.2341476678848267,   1.0973775386810303,
   1.4210201501846313,    0.6451131105422974,     3.7340331077575684,    -0.4524728059768677,   0.23967613279819489,
   -1.7517019510269165,   -1.5562810897827148,    -2.0684239864349365,   1.9971668720245361,    0.9888216853141785,
   -0.2786630094051361,   3.521080493927002,      -1.2520416975021362,   -5.741950988769531,    0.4906579256057739,
   0.008913683705031872,  1.00362229347229,       -1.6393157243728638,   -1.4653292894363403,   0.1405085027217865,
   -0.4173285663127899,   0.0019979598000645638,  -0.1846299171447754,   -1.963813066482544,    0.768883228302002,
   0.3042655289173126,    -6.606032848358154,     -0.13850121200084686,  1.0398000478744507,    3.1176137924194336,
   0.2531544864177704,    -2.1711816787719727,    3.34458065032959,      -0.6482598185539246,   -0.9298532605171204,
   1.0177197456359863,    -2.1260740756988525,    -3.2033774852752686,   -0.2472154051065445,   -1.7773890495300293,
   -1.0052404403686523,   -3.6802327632904053,    -3.0570733547210693,   -0.6141660809516907,   2.3998067378997803,
   -0.9492958784103394,   -0.7456861734390259,    3.244791030883789,     -2.152898073196411,    -0.03450721129775047,
   0.7297899723052979,    -2.582406759262085,     1.9502114057540894,    1.551411509513855,     24.913331985473633,
   1.173578143119812,     2.2933189868927,        -0.0441926084458828,   0.8407466411590576,    -0.6475434303283691,
   1.2887327671051025,    0.5652068853378296,     -1.0873559713363647,   -4.349416732788086,    -2.6013450622558594,
   -3.2782557010650635,   3.574897050857544,      -4.509507656097412,    -1.45671546459198,     -0.20600983500480652,
   5.598803997039795,     5.39168119430542,       0.25083890557289124,   0.23568107187747955,   0.39629626274108887,
   2.2019131183624268,    1.6220026016235352,     -3.179694652557373,    -1.4805322885513306,   3.874481439590454,
   1.9394983053207397,    -0.07736560702323914,   1.7453492879867554,    1.9610720872879028,    0.5634517073631287,
   -1.8436990976333618,   -1.9274146556854248,    9.607552528381348,     -1.101751685142517,    0.7912783026695251,
   -0.7635292410850525,   -4.715912342071533,     2.5583322048187256,    -2.363145112991333,    -1.0401372909545898,
   -8.265142440795898,    -1.2847415208816528,    1.2997654676437378,    0.29620760679244995,   -1.2075390815734863,
   -0.15197056531906128,  -1.3155065774917603,    -4.516768932342529,    -1.1480783224105835,   -0.43007344007492065,
   0.7342324256896973,    -0.8108710646629333,    0.2708045244216919,    -0.499053418636322,    -1.1171764135360718,
   0.07932508736848831,   -1.1393648386001587,    -1.97962486743927,     2.460477590560913,     -5.33926248550415,
   1.6168500185012817,    0.026065349578857422,   -1.6841216087341309,   -1.2605044841766357,   0.3105122148990631,
   4.798562526702881,     -5.641758441925049,     0.045763447880744934,  -0.2240714579820633,   -3.284721851348877,
   -1.6287548542022705,   -0.08540278673171997,   -3.5407774448394775,   0.40387067198753357,   -1.0487786531448364,
   -0.6759702563285828,   0.15470941364765167,    0.633712649345398,     2.2192819118499756,    -0.8343179225921631,
   0.19003483653068542,   -1.0185774564743042,    -0.604653000831604,    0.4040887951850891,    -4.235228061676025,
   7.0822577476501465,    2.6259167194366455,     4.613339424133301,     9.784403800964355,     -1.0613282918930054,
   -3.0188703536987305,   -0.4729401171207428,    -4.543302536010742,    2.39048171043396,      -2.4233133792877197,
   0.21090160310268402,   -5.788041114807129,     -3.5900611877441406,   2.7094314098358154,    -5.067966938018799,
   -1.186936855316162,    -7.126315116882324,     -4.906425476074219,    -5.0304365158081055,   5.525407791137695,
   0.8349937200546265,    -0.625641942024231,     1.8191490173339844,    -0.16909310221672058,  2.5061326026916504,
   4.750252723693848,     -0.15736263990402222,   -1.9335176944732666,   -1.1558775901794434,   0.07357990741729736,
   -2.278942584991455,    -1.4982175827026367,    0.177586168050766,     -0.49100759625434875,  -0.01884038932621479,
   -1.0934052467346191,   -3.287818670272827,     -2.2426350116729736,   -0.3371116816997528,   -0.4218229353427887,
   0.5909421443939209,    1.6734846830368042,     -0.5110045075416565,   -4.5083513259887695,   2.4603195190429688,
   4.28099250793457,      -1.565510869026184,     4.9759345054626465,    -1.2872886657714844,   0.11388003826141357,
   1.310612440109253,     -1.4997074604034424,    2.0875613689422607,    2.623558759689331,     0.7289532423019409,
   1.315991997718811,     -2.446355104446411,     -1.809743881225586,    -1.919993281364441,    19.25080680847168,
   -3.379228353500366,    0.6025159955024719,     -0.09128012508153915,  -0.31812506914138794,  -0.11418932676315308,
   -1.0425817966461182,   2.772944927215576,      1.044714093208313,     1.3640260696411133,    -0.7180152535438538,
   -0.37938156723976135,  4.399036407470703,      3.120701313018799,     -0.18312327563762665,  -10.938495635986328,
   0.2630477845668793,    -0.7842718958854675,    -0.6023545265197754,   -0.5608141422271729,   -2.157660722732544,
   -1.7979357242584229,   0.27133917808532715,    2.483478546142578,     0.21124869585037231,   -0.7324768304824829,
   1.5650004148483276,    -3.3837578296661377,    -1.6038181781768799,   -0.9670430421829224,   2.7546374797821045,
   -0.4951609671115875,   -3.809602737426758,     5.141548156738281,     -5.016628265380859,    -1.3966799974441528,
   -1.7679316997528076,   3.2074904441833496,     1.9173043966293335,    0.17117945849895477,   -2.018186330795288,
   -11.60806941986084,    1.3798491954803467,     0.7680304050445557,    -1.067282795906067,    -5.362876892089844,
   3.5064899921417236,    0.8493854403495789,     -1.2587461471557617,   1.10347580909729,      2.3785793781280518,
   -14.21773624420166,    1.352631688117981,      0.2499239593744278,    1.1228320598602295,    -0.5589932799339294,
   1.6254792213439941,    0.603907585144043,      -1.5968588590621948,   0.23238533735275269,   -3.235860824584961,
   -0.6562446355819702,   5.542327404022217,      -3.7162277698516846,   -2.5599982738494873,   -2.2996537685394287,
   -0.16704322397708893,  -1.1631553173065186,    -4.392309188842773,    1.5351265668869019,    3.104694366455078,
   -4.075592994689941,    1.4513535499572754,     1.6950253248214722,    -0.1417434811592102,   3.0004074573516846,
   -4.9498443603515625,   1.291295051574707,      -0.5954280495643616,   0.36493057012557983,   -1.91603684425354,
   1.5865285396575928,    1.197903037071228,      0.9245343208312988,    -2.053290843963623,    -1.8147518634796143,
   1.4570043087005615,    3.904966354370117,      -2.3308141231536865,   -1.3332265615463257,   -0.501945972442627,
   -2.132948875427246,    0.5763935446739197,     -0.4482395052909851,   -1.4528112411499023,   -1.8538944721221924,
   -0.8771736025810242,   -1.6269510984420776,    1.8402249813079834,    2.953080177307129,     0.9795225262641907,
   -0.1429866999387741,   -2.9807662963867188,    -1.6724199056625366,   1.088204264640808,     -0.12841005623340607,
   -7.785520553588867,    -0.8455806374549866,    1.3301483392715454,    2.114962339401245,     -1.4084621667861938,
   -1.4308973550796509,   0.8167319893836975,     -8.076427459716797,    -0.2838226854801178,   -0.6522688865661621,
   -1.1607680320739746,   0.5803080201148987,     2.2861905097961426,    -0.6850014328956604,   0.9160593152046204,
   2.7312049865722656,    -0.38588204979896545,   0.8755513429641724,    0.09339257329702377,   0.001790353562682867,
   0.03430848941206932,   -0.2627887427806854,    -4.144521236419678,    0.8610742688179016,    1.1403565406799316,
   -1.7078615427017212,   -1.1400506496429443,    -1.6991355419158936,   -3.0027005672454834,   -1.3584794998168945,
   0.2458818405866623,    0.6979472637176514,     -0.8577844500541687,   2.4016926288604736,    -1.5818814039230347,
   -5.05034065246582,     -15.246307373046875,    -3.779719114303589,    -1.3386844396591187,   -0.8418460488319397,
   -1.145071268081665,    2.284898281097412,      -5.293917179107666,    -0.7687748074531555,   -3.8685367107391357,
   0.26262038946151733,   -1.8972108364105225,    -0.015978194773197174, 1.5912684202194214,    5.551901340484619,
   -2.054844856262207,    -0.4796889126300812,    -0.195564866065979,    1.1348748207092285,    0.3600684702396393,
   1.711858868598938,     1.5110238790512085,     0.5614420175552368,    -1.8429672718048096,   0.8336097598075867,
   1.4369890689849854,    3.067124366760254,      -2.6001076698303223,   -1.8377865552902222,   -6.626893043518066,
   5.390143394470215,     -0.17130538821220398,   -8.752378463745117,    0.7828227281570435,    1.8327789306640625,
   -0.4848863184452057,   -1.8559913635253906,    -1.9137557744979858,   21.88369369506836,     -1.5696673393249512,
   -0.7675939202308655,   1.4796955585479736,     -0.028250327333807945, -1.6112840175628662,   0.28635072708129883,
   -2.6856675148010254,   -1.0087695121765137,    -6.564421653747559,    -2.693781852722168,    -7.447775363922119,
   -8.603723526000977,    1.4492334127426147,     -0.23503145575523376,  1.4299144744873047,    2.0401291847229004,
   5.026698112487793,     -0.0001406308583682403, 3.848947763442993,     0.4900077283382416,    4.139863014221191,
   1.8836357593536377,    0.41689059138298035,    2.9303739070892334,    1.4516701698303223,    -3.078670024871826,
   1.2903386354446411,    3.5369744300842285,     -4.244988918304443,    1.045324683189392,     -0.896399974822998,
   0.8332988023757935,    -0.33581820130348206,   0.9497665166854858,    0.16599951684474945,   1.9956121444702148,
   -2.3651130199432373,   -4.426061153411865,     -0.27434778213500977,  -1.3050986528396606,   -4.8599534034729,
   -0.8398025631904602,   0.5910659432411194,     1.47472083568573,      2.707857608795166,     0.0875067412853241,
   -2.1632447242736816,   0.6850031018257141,     0.015737704932689667,  0.941871166229248,     -2.962428331375122,
   1.4660701751708984,    -3.337707042694092,     0.07806733250617981,   -0.5001533627510071,   0.3980059325695038,
   -0.9553836584091187,   0.7261481881141663,     0.2001577913761139,    2.092745304107666,     -1.1848784685134888,
   1.8700026273727417,    -0.5544077754020691,    1.0924134254455566,    -0.5703863501548767,   4.728804588317871,
   0.2050211876630783,    -1.6632870435714722,    0.9240105152130127,    2.623292922973633,     14.80235481262207,
   -14.959753036499023,   -0.0818018987774849,    -0.11649259179830551,  0.4001704156398773,    -1.6020524501800537,
   50.91973876953125,     -5.688167572021484,     -0.5608856678009033,   -3.8215391635894775,   0.35167384147644043,
   -0.6042256951332092,   -1.155979037284851,     0.21037401258945465,   -0.023829136043787003, -12.188655853271484,
   -0.9223014712333679,   -0.6385392546653748,    -2.3142616748809814,   0.22120551764965057,   -2.796555995941162,
   0.7723158001899719,    -2.621422052383423,     -0.059495288878679276, 4.519052028656006,     -2.932147264480591,
   3.290726661682129,     -0.515079915523529,     -4.669440269470215,    5.683082580566406,     -0.496140718460083,
   1.529366374015808,     0.6719560027122498,     0.08404224365949631,   -0.5199785232543945,   0.06808090209960938,
   -0.4366801679134369,   -0.19866152107715607,   -2.8763041496276855,   -0.728960394859314,    0.5790876746177673,
   -0.1897372454404831,   -0.5004426836967468,    -0.21950168907642365,  0.013107957318425179,  0.7823934555053711,
   0.7038758397102356,    0.46807968616485596,    -3.467349052429199,    -1.5869717597961426,   -0.9547117948532104,
   -1.9873496294021606,   1.1786143779754639,     -1.518337368965149,    69.35067749023438,     1.4057104587554932,
   -32.6806640625,        -0.8058973550796509,    -1.478585958480835,    -0.5064217448234558,   0.08844950050115585,
   1.5458966493606567,    0.7000851631164551,     -3.5903666019439697,   -0.701282262802124,    0.9106563925743103,
   -0.6314098834991455,   -0.23639625310897827,   -3.449226140975952,    -1.8732339143753052,   -3.7602334022521973,
   1.0252889394760132,    5.427237033843994,      -0.3864818811416626,   -1.6850941181182861,   -5.09268045425415,
   -0.5401142835617065,   0.5462165474891663,     -9.775838851928711,    2.2050647735595703,    -1.942358136177063,
   -1.7503684759140015,   -4.578794956207275,     6.711188316345215,     -1.2038476467132568,   -0.7885754108428955,
   0.3234657049179077,    -9.135645866394043,     4.365988731384277,     -0.03301597386598587,  0.06169114261865616,
   -1.718961477279663,    -0.03358859196305275,   1.9855905771255493,    4.8856425285339355,    -1.2554705142974854,
   1.4390904903411865,    -15.797120094299316,    -2.5773675441741943,   -0.8722683787345886,   2.2055869102478027,
   -0.5232475996017456,   0.4223300814628601,     1.759389042854309,     2.1619112491607666,    -0.08437497913837433,
   -2.0475881099700928,   0.6844083666801453,     1.0975701808929443,    -2.525238513946533,    -0.3481937646865845,
   0.7654094099998474,    -0.768379807472229,     -0.7882152795791626,   0.24742840230464935,   -0.08882451057434082,
   -1.9270350933074951,   -0.897261381149292,     2.662902355194092,     2.150233745574951,     -0.07051065564155579,
   0.3822738826274872,    3.1293723583221436,     1.447133183479309,     -1.5371366739273071,   -0.49387797713279724,
   -1.8288229703903198,   0.9707151651382446,     0.15354667603969574,   0.744452178478241,     -1.470110535621643,
   -2.5066866874694824,   4.9847612380981445,     0.6415032744407654,    -4.971923351287842,    1.9800420999526978,
   0.058141451328992844,  -1.9296661615371704,    1.857109546661377,     -1.8201278448104858,   1.724632740020752,
   5.314121723175049,     0.5843242406845093,     -1.675420880317688,    -1.5032830238342285,   -5.313798427581787,
   3.600694179534912,     -3.1727843284606934,    -2.279006242752075,    1.3116523027420044,    -0.24790002405643463,
   0.29177772998809814,   -4.675588607788086,     0.4942968189716339,    -4.357414722442627,    0.6956951022148132,
   7.874487400054932,     -1.2131048440933228,    2.102613925933838,     3.802980661392212,     -2.339444398880005,
   1.6110705137252808,    -1.1728529930114746,    0.4244329631328583,    0.06247270107269287,   -1.5968064069747925};

} // namespace

// Test that a simple fit with Gauss and Voigt yields the same values as with
// ROOT 6.18. The fit requires numeric integration for the Voigt. To suppress
// false positives, the precision for numeric integrals was increased by 1000x.
// This test uses Minuit2.
//
// Previously in: roottest/root/roofitstats/stabilityTests.cxx
TEST(Stability, ROOT_10615)
{

   RooRealVar dt("dt", "#Deltat [ps]", -10, 10);

   RooRealVar bkg_dt_voigt_mu("bkg_dt_voigt_mu", "v_{#mu}", -0.303, -1, 1);
   RooRealVar bkg_dt_voigt_sigma("bkg_dt_voigt_sigma", "v_{#sigma}", 2.323, 0, 10);
   RooRealVar bkg_dt_voigt_width("bkg_dt_voigt_width", "v_{w}", 0.851, 0, 10);
   RooVoigtian bkg_dt_voigt("bkg_dt_voigt", "bkg_dt_voigt", dt, bkg_dt_voigt_mu, bkg_dt_voigt_width,
                            bkg_dt_voigt_sigma);
   RooNumIntConfig *voigtIntegrator = bkg_dt_voigt.specialIntegratorConfig(true);
   voigtIntegrator->setEpsAbs(1.E-10);
   voigtIntegrator->setEpsRel(1.E-10);
   voigtIntegrator->getConfigSection("RooIntegrator1D").setRealValue("maxSteps", 30);

   RooRealVar bkg_dt_gaus_mu_("bkg_dt_gaus_mu", "g_{#mu}", -0.161 - 1, 1);
   RooRealVar bkg_dt_gaus_sigma_("bkg_dt_gaus_sigma", "g_{#sigma}", 1.096, 1.E-6, 10);
   RooGaussian bkg_dt_gaus("bkg_dt_gaus", "bkg_dt_gaus", dt, bkg_dt_gaus_mu_, bkg_dt_gaus_sigma_);

   RooRealVar bkg_dt_f("bkg_dt_f", "f_{v/g}", 0.631, 0, 1);

   RooAddPdf bkg_dt_model_("bkg_dt_model", "bkg_dt_model", RooArgList(bkg_dt_voigt, bkg_dt_gaus), RooArgList(bkg_dt_f));

   RooArgSet dataVars{dt};
   RooDataSet dataset("dataset", "dataset", dataVars);

   for (double dtVal : dtVals) {
      if (dtVal >= dt.getMin() && dtVal <= dt.getMax()) {
         dt.setVal(dtVal);
         dataset.add(dataVars);
      }
   }

   auto fitResult = bkg_dt_model_.fitTo(dataset, RooFit::Minimizer("Minuit2"), RooFit::Save(), RooFit::Hesse(false),
                                        RooFit::PrintLevel(-1));

   EXPECT_NEAR(bkg_dt_f.getVal(), 0.5863848853184781, 1.E-9);
   EXPECT_NEAR(bkg_dt_f.getError(), 0.097259271172036643, 2.E-9); // Between Mac and linux, the values differ by 1.61E-9
   EXPECT_NEAR(fitResult->minNll(), 2200.69610626874464, 1.E-9);
   EXPECT_NEAR(fitResult->edm(), 0.00025734795846655238, 1.E-11);
}
