/* wild.c
 *
 * Match a wildcard string against a regular string.  Supports *, ?,
 * [...], and \.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <ctype.h>
#include <rbmake/rbfile.h>

static bool backslashMatch(char *bscp, char ch);

bool
Wild_EQ(const char *wild, const char *str)
{
    const char *lastWild = NULL, *nextStr = NULL;
    char ch, lower;
    bool match, negclass, gotdash;

    for ( ; *wild; wild++) {
	switch (*wild) {
	  case '*':
	    if (!wild[1])
		return true;
	    lastWild = wild;
	    nextStr = str+1;
	    continue;		/* start loop over, matching *str again */
	  case '?':
	    match = true;
	    break;
	  case '[':
	    if (*++wild == '^') {
		negclass = true;
		wild++;
	    }
	    else
		negclass = false;
	    match = true;
	    gotdash = false;
	    if (*wild == ']') {
		if (*str == *wild++)
		    goto classdone;
	    }
	    for (lower = '\0'; (ch = *wild) != '\0' && ch != ']'; wild++) {
		if (ch == '-' && lower && !gotdash && wild[1] != ']') {
		    gotdash = true;
		    continue;
		}
		if (ch == '\\' && wild[1]) {
		    ch = *++wild;
		    if (backslashMatch(&ch, *str))
			goto classdone;
		    if (!ch) {
			if (gotdash) {
			    if (*str == '-')
				goto classdone;
			    gotdash = false;
			}
			lower = '\0';
			continue;
		    }
		}
		if (gotdash) {
		    if (uc(str,0) >= uc(&lower,0) && uc(str,0) <= uc(&ch,0))
			goto classdone;
		    gotdash = false;
		    lower = '\0';
		}
		else if (*str == ch)
		    goto classdone;
		else
		    lower = ch;
	    }
	    match = false;
	  classdone:
	    match ^= negclass;
	    if (match) {
		while (*wild != ']') {
		    if (!*wild) {
			wild--;
			break;
		    }
		    wild++;
		}
	    }
	    break;
	  case '\\':
	    if (wild[1])
		wild++;
	    ch = *wild;
	    match = backslashMatch(&ch, *str);
	    break;
	  default:
	    match = *str == *wild;
	    break;
	}
	if (match && *str)
	    str++;
	else {
	    if (!lastWild || !*(str = nextStr++))
		return false;
	    wild = lastWild;
	}
    }
    return *str == '\0';
}

static bool
backslashMatch(char *bscp, char ch)
{
    char bsc = *bscp;
    switch (bsc) {
      case 's':
	*bscp = '\0';
	return !!ISSPACE(ch);
      case 'S':
	*bscp = '\0';
	return !ISSPACE(ch);
      case 'd':
	*bscp = '\0';
	return !!ISDIGIT(ch);
      case 'D':
	*bscp = '\0';
	return !ISDIGIT(ch);
      case 'w':
	*bscp = '\0';
	return ISALNUM(ch) || ch == '_';
      case 'W':
	*bscp = '\0';
	return !ISALNUM(ch) && ch != '_';
      case 'n':
	bsc = *bscp = '\n';
	break;
      case 'r':
	bsc = *bscp = '\r';
	break;
      case 't':
	bsc = *bscp = '\t';
	break;
      case 'e':
	bsc = *bscp = '\033';
	break;
      case 'f':
	bsc = *bscp = '\f';
	break;
      case 'a':
	bsc = *bscp = '\007';
	break;
      case 'b':
	bsc = *bscp = '\010';
	break;
    }
    return ch == bsc;
}

MBuf *
Wild_escapeWildcards(const char *str)
{
    MBuf *mb = MBuf_new(32, 0);

    MBuf_setUpcomingLength(mb, strlen(str));
    while (*str) {
	switch (*str) {
	  case '*':
	  case '?':
	  case '[':
	  case '\\':
	    MBuf_putc(mb, '\\');
	    break;
	}
	MBuf_putc(mb, *str++);
    }
    return mb;
}
