/* rburls.c
 *
 * Some simple utility routines.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <limits.h>
#include <ctype.h>
#include <rbmake/rbfile.h>

/* If the URL is for a local file, return a pointer to the file's path,
 * otherwise return NULL. */
char *
rbGetUrlFn(const char *url)
{
    char *cp;
    if (strnNE(url, "file:", 5))
	return NULL;
    url += 5;
    if (*url == '/')
	while (url[1] == '/') url++;
    cp = rbUrlUnescape(url);
#ifdef DOS_FILENAMES
    /* Transform /mnt_C/path into C:/path */
    cp[0] = cp[5], cp[1] = ':', strcpy(cp+2, cp+6);
#endif
    return cp;
}

char *
rbUrlEscape(const char *url)
{
    MBuf *tmp = MBuf_new(128, 0);
    int ch;

    while ((ch = uc(url++,0)) != '\0') {
#ifdef DOS_FILENAMES
	if (ch == '\\')
	    ch = '/';
#endif
	if (!ISALNUM(ch) && strchr("#/-+._,?&:;!~'(*)%@=$", ch) == NULL) {
	    char buf[8];
	    int hi = ch / 16, lo = ch % 16;
	    sprintf(buf, "%%%c%c", hi + (hi > 9? 'A'-10 : '0'),
				   lo + (lo > 9? 'A'-10 : '0'));
	    MBuf_puts(tmp, buf);
	}
	else
	    MBuf_putc(tmp, ch);
    }

    return MBuf_toBuffer(tmp, NULL);
}

static char ishexdigit_array[256];

#define ISHEXDIGIT(c) ishexdigit_array[(unsigned char)(c)]

char *
rbUrlUnescape(const char *url)
{
    char *t, *bp = Mem_alloc(strlen(url)+1);
    const char *f;
    int ch;

    if (!ishexdigit_array['0']) {
	for (ch = '0'; ch <= '9'; ch++)
	    ishexdigit_array[ch] = 1;
	for (ch = 'A'; ch <= 'F'; ch++)
	    ishexdigit_array[ch] = ishexdigit_array[TOLOWER(ch)] = 1;
    }

    for (t = bp, f = url; (ch = *f++) != '\0'; ) {
	if (ch == '%' && ISHEXDIGIT(*f) && ISHEXDIGIT(f[1])) {
	    int val;
	    ch = *f++;
	    if (!ISDIGIT(ch))
		ch = (ch & 0x1F) + 9 + '0';
	    val = ch - '0';
	    ch = *f++;
	    if (!ISDIGIT(ch))
		ch = (ch & 0x1F) + 9 + '0';
	    val = val * 16 + ch - '0';
	    *t++ = val;
	}
	else
	    *t++ = ch;
    }
    *t = '\0';

    return bp;
}

const char *
rbGetCwdURL()
{
    static char *cwdUrl;
    if (!cwdUrl) {
	char buf[PATH_MAX];
	char *cwd = getcwd(buf, sizeof buf);
	char *cp;
#ifdef DOS_FILENAMES
	cp = rbUrlEscape(cwd+2);
	cwdUrl = Mem_alloc(13 + strlen(cp) + 1 + 1);
	sprintf(cwdUrl, "file:///mnt_%c%s/", *cwd, cp);
#else
	cp = rbUrlEscape(cwd);
	cwdUrl = Mem_alloc(7 + strlen(cp) + 1 + 1);
	sprintf(cwdUrl, "file://%s/", cp);
#endif
	Mem_free(cp);
    }
    return cwdUrl;
}

const char *
rbStripCwd(const char *url)
{
    const char *cwd = rbGetCwdURL();
    int len = strlen(cwd);
    if (strnEQ(url, cwd, len))
	url += len;
    return url;
}
