/* marray.c
 *
 * This derived "class" is based on an MBuf, and it stores and retrieves
 * ints (large enough to store a pointer) into an expandable array.  You
 * can choose to store multiple chunks of ints/pointers, and merge MArray
 * objects together as you see fit, but if you need to access the data in
 * one big chunk (e.g. for use with qsort()), you'll need to choose to use
 * just one realloc'ed object when you create the array.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <rbmake/rbfile.h>
#include "mbuf.h"

static int ptrSizeVar = PTRSIZE;

MArray *
MArray_new(int growBy, int maxChunkSize)
{
    return (MArray*)MBuf_new(growBy * PTRSIZE, maxChunkSize * PTRSIZE);
}

void
MArray_delete(MArray *me)
{
    MBuf_delete((MBuf*)me);
}

int
MArray_append(MArray *me, int i)
{
    return MBuf_write((MBuf*)me, (char*)&i, PTRSIZE);
}

int
MArray_appendPtr(MArray *me, const void *ptr)
{
    return MBuf_write((MBuf*)me, (char*)&ptr, PTRSIZE);
}

int
MArray_storeAt(MArray *me, int pos, int i)
{
    return MBuf_overwrite((MBuf*)me, pos * PTRSIZE, (char*)&i, PTRSIZE);
}

int
MArray_storePtrAt(MArray *me, int pos, const void *ptr)
{
    return MBuf_overwrite((MBuf*)me, pos * PTRSIZE, (char*)&ptr, PTRSIZE);
}

int
MArray_truncate(MArray *me, int pos)
{
    return MBuf_truncate((MBuf*)me, pos * PTRSIZE);
}

int
MArray_setFetchPos(MArray *me, int pos)
{
    int ret = MBuf_setReadPos((MBuf*)me,pos * PTRSIZE, 0);
    if (ret > 0)
	ret /= PTRSIZE;
    return ret;
}

int
MArray_fetch(MArray *me)
{
    int *dp = (int*)MBuf_dataPtr((MBuf*)me, &ptrSizeVar);
    return dp? *dp : 0;
}

void *
MArray_fetchPtr(MArray *me)
{
    void **dp = (void**)MBuf_dataPtr((MBuf*)me, &ptrSizeVar);
    return dp? *dp : NULL;
}

int
MArray_fetchAt(MArray *me, int pos)
{
    int *dp = (int*)MArray_dataPtrAt(me, pos);
    return dp? *dp : 0;
}

void *
MArray_fetchPtrAt(MArray *me, int pos)
{
    void **dp = (void**)MArray_dataPtrAt(me, pos);
    return dp? *dp : NULL;
}

void *
MArray_dataPtrAt(MArray *me, int pos)
{
    return (void*)MBuf_dataPtrAt((MBuf*)me, pos * PTRSIZE, NULL);
}

int
MArray_itemCnt(MArray *me)
{
    return me->totalLen / PTRSIZE;
}
