/* gifimage.c
 *
 * Modified by Wayne Davison, October 2000.
 */

/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, 1991, 1993, David Koblas.  (koblas@netcom.com)    | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */


#include <config.h>
#ifdef HAVE_LIBPNG
#include <png.h>
#include <rbmake/rbmake.h>
#include "rbimage.h"

#define MAXCOLORMAPSIZE 256

#define CM_RED		0
#define CM_GREEN	1
#define CM_BLUE		2

#define MAX_LWZ_BITS	12

#define INTERLACE	0x40
#define LOCALCOLORMAP	0x80
#define BitSet(byte, bit)	(((byte) & (bit)) == (bit))

#define ReadOK(mb,buf,len)	(MBuf_read(mb, buf, len) > 0)

#define LM_TO_UINT(p,o1,o2) (((unsigned int)uc(p,o2)<<8) | uc(p,o1))
#ifdef __GNUC__
static inline unsigned int LM_to_uint(char *ptr, int o1, int o2)
{ return LM_TO_UINT(ptr, o1, o2); }
#undef LM_TO_UINT
#else
#define LM_to_uint(p,o1,o2) LM_TO_UINT(p,o1,o2)
#endif

static struct {
    unsigned int Width;
    unsigned int Height;
    unsigned char ColorMap[3][MAXCOLORMAPSIZE];
    unsigned int BitPixel;
    unsigned int ColorResolution;
    unsigned int Background;
    unsigned int AspectRatio;
    int GrayScale;
} GifScreen;

static struct {
    int transparent;
    int delayTime;
    int inputFlag;
    int disposal;
} Gif89 = { -1, -1, -1, 0 };

static int ReadColorMap(MBuf *mb, int number,
			unsigned char buffer[3][MAXCOLORMAPSIZE]);
static void DoExtension(MBuf *mb, int label);
static int GetDataBlock(MBuf *mb, char *buf);
static int GetCode(MBuf *mb, int code_size, int flag);
static int LWZReadByte(MBuf *mb, int flag, int input_code_size);
static int ReadImage(MBuf *mb, RbImage *img,
		     unsigned char cmap[3][MAXCOLORMAPSIZE], int interlace);

bool
gifCheckSig(char *bp)
{
    return strnEQ(bp,"GIF",3) && (strnEQ(bp+3,"87a",3) || strnEQ(bp+3,"89a",3));
}

RbImage *
gifToImage(MBuf *mb)
{
    char buf[16];
    char c;
    unsigned char localColorMap[3][MAXCOLORMAPSIZE];
    int useGlobalColormap, bitPixel;
    RbImage *img = NULL;

    if (!ReadOK(mb, buf, 6)) {
	RbImage_setError("error reading magic number");
	return NULL;
    }

    if (!gifCheckSig(buf)) {
	RbImage_setError("not a GIF file");
	return NULL;
    }

    if (!ReadOK(mb, buf, 7)) {
	RbImage_setError("failed to read screen descriptor");
	return NULL;
    }

    GifScreen.Width = LM_to_uint(buf,0,1);
    GifScreen.Height = LM_to_uint(buf,2,3);
    GifScreen.BitPixel = 2 << (uc(buf,4)&0x07);
    GifScreen.ColorResolution = ((uc(buf,4)&0x70) >> 3) + 1;
    GifScreen.Background = uc(buf,5);
    GifScreen.AspectRatio = uc(buf,6);

    if (BitSet(uc(buf,4), LOCALCOLORMAP)) {    /* Global Colormap */
	GifScreen.GrayScale = ReadColorMap(mb, GifScreen.BitPixel,
					   GifScreen.ColorMap);
	if (GifScreen.GrayScale < 0) {
	    RbImage_setError("error reading global colormap");
	    return NULL;
	}
    }

    for (;;) {
	if (!ReadOK(mb, &c,1)) {
	    RbImage_setError("EOF / read error on image data");
	    return NULL;
	}

	if (c == ';') {	/* GIF terminator */
	    RbImage_setError("no image in file");
	    return NULL;
	}

	if (c == '!') {	/* Extension */
	    if (!ReadOK(mb, &c,1)) {
		RbImage_setError("OF / read error on extention function code");
		return NULL;
	    }
	    DoExtension(mb, uc(&c,0));
	    continue;
	}

	if (c != ',')	/* Not a valid start character */
	    continue;

	if (!ReadOK(mb, buf, 9)) {
	    RbImage_setError("couldn't read left/top/width/height");
	    return NULL;
	}

	useGlobalColormap = !BitSet(uc(buf,8), LOCALCOLORMAP);
	bitPixel = 1 << ((uc(buf,8)&0x07)+1);

	img = Mem_alloc(sizeof *img);
	img->width = LM_to_uint(buf,4,5);
	img->height = LM_to_uint(buf,6,7);
	img->row_pointers = Mem_alloc(img->height * sizeof (char*));
	img->bit_depth = 8;

	if (!useGlobalColormap) {
	    int grayScale = ReadColorMap(mb, bitPixel, localColorMap);
	    if (grayScale < 0) {
		RbImage_setError("error reading local colormap");
		return NULL;
	    }
	    img->color_type = grayScale? PNG_COLOR_TYPE_GRAY
				       : PNG_COLOR_TYPE_RGB;
	    ReadImage(mb, img, localColorMap, BitSet(uc(buf,8), INTERLACE));
	    break;
	}
	else {
	    img->color_type = GifScreen.GrayScale? PNG_COLOR_TYPE_GRAY
						 : PNG_COLOR_TYPE_RGB;
	    ReadImage(mb, img, GifScreen.ColorMap, BitSet(uc(buf,8), INTERLACE));
	    break;
	}
    }
    return img;
}

static int
ReadColorMap(MBuf *mb, int number, unsigned char buffer[3][MAXCOLORMAPSIZE])
{
    int i;
    char rgb[3];
    int flag = true;

    for (i = 0; i < number; i++) {
	if (!ReadOK(mb, rgb, sizeof rgb))
	    return -1;

	buffer[CM_RED][i] = uc(rgb,0);
	buffer[CM_GREEN][i] = uc(rgb,1);
	buffer[CM_BLUE][i] = uc(rgb,2);

	flag &= (rgb[0] == rgb[1] && rgb[1] == rgb[2]);
    }

    return flag;
}

static void
DoExtension(MBuf *mb, int label)
{
    static char buf[256];
    char *str;

    switch (label) {
      case 0x01:	/* Plain Text Extension */
	str = "Plain Text Extension";
	break;
      case 0xff:	/* Application Extension */
	str = "Application Extension";
	break;
      case 0xfe:	/* Comment Extension */
	str = "Comment Extension";
	while (GetDataBlock(mb, buf) > 0) {}
	return;
      case 0xf9:	/* Graphic Control Extension */
	str = "Graphic Control Extension";
	(void) GetDataBlock(mb, buf);
	Gif89.disposal    = (buf[0] >> 2) & 0x7;
	Gif89.inputFlag   = (buf[0] >> 1) & 0x1;
	Gif89.delayTime   = LM_to_uint(buf,1,2);
	if ((buf[0] & 0x1) != 0)
	    Gif89.transparent = uc(buf,3);

	while (GetDataBlock(mb, buf) > 0) {}
	return;
      default:
	str = buf;
	sprintf(buf, "UNKNOWN (0x%02x)", label);
	break;
    }

    while (GetDataBlock(mb, buf) > 0) {}
}

int ZeroDataBlock = false;

static int
GetDataBlock(MBuf *mb, char *buf)
{
    char ch;
    int count;

    if (!ReadOK(mb, &ch, 1)) {
	RbImage_setError("error in getting DataBlock size");
	return -1;
    }
    count = uc(&ch,0);

    ZeroDataBlock = count == 0;

    if (count != 0 && !ReadOK(mb, buf, count)) {
	RbImage_setError("error in reading DataBlock");
	return -1;
    }

    return count;
}

static int
GetCode(MBuf *mb, int code_size, int flag)
{
    static char buf[280];
    static int curbit, lastbit, done, last_byte;
    int i, j, ret;
    unsigned char count;

    if (flag) {
	curbit = 0;
	lastbit = 0;
	done = false;
	return 0;
    }

    if (curbit + code_size >= lastbit) {
	if (done) {
	    if (curbit >= lastbit)
		RbImage_setError("ran off the end of my bits");
	    return -1;
	}
	buf[0] = buf[last_byte-2];
	buf[1] = buf[last_byte-1];

	if ((count = GetDataBlock(mb, &buf[2])) == 0)
	    done = true;

	last_byte = 2 + count;
	curbit = (curbit - lastbit) + 16;
	lastbit = (2+count)*8;
    }

    ret = 0;
    for (i = curbit, j = 0; j < code_size; i++, j++)
	ret |= ((uc(buf, i / 8 ) & (1 << (i % 8))) != 0) << j;

    curbit += code_size;

    return ret;
}

static int
LWZReadByte(MBuf *mb, int flag, int input_code_size)
{
    static int fresh = false;
    static int code_size, set_code_size;
    static int max_code, max_code_size;
    static int firstcode, oldcode;
    static int clear_code, end_code;
    static int table[2][(1<< MAX_LWZ_BITS)];
    static int stack[(1<<(MAX_LWZ_BITS))*2], *sp;
    int code, incode;
    register int i;

    if (flag) {
	set_code_size = input_code_size;
	code_size = set_code_size+1;
	clear_code = 1 << set_code_size;
	end_code = clear_code + 1;
	max_code_size = 2*clear_code;
	max_code = clear_code+2;

	GetCode(mb, 0, true);
	   
	fresh = true;

	for (i = 0; i < clear_code; i++) {
	    table[0][i] = 0;
	    table[1][i] = i;
	}
	for (; i < (1<<MAX_LWZ_BITS); i++)
	    table[0][i] = table[1][0] = 0;

	sp = stack;

	return 0;
    }

    if (fresh) {
	fresh = false;
	do {
	    firstcode = oldcode = GetCode(mb, code_size, false);
	} while (firstcode == clear_code);
	return firstcode;
    }

    if (sp > stack)
	return *--sp;

    while ((code = GetCode(mb, code_size, false)) >= 0) {
	if (code == clear_code) {
	    for (i = 0; i < clear_code; i++) {
		table[0][i] = 0;
		table[1][i] = i;
	    }
	    for (; i < (1<<MAX_LWZ_BITS); i++)
		table[0][i] = table[1][i] = 0;
	    code_size = set_code_size+1;
	    max_code_size = 2*clear_code;
	    max_code = clear_code+2;
	    sp = stack;
	    firstcode = oldcode = GetCode(mb, code_size, false);
	    return firstcode;
	}

	if (code == end_code) {
	    int count;
	    char buf[260];

	    if (ZeroDataBlock)
		return -2;

	    while ((count = GetDataBlock(mb, buf)) > 0) {}

	    return -2;
	}

	incode = code;

	if (code >= max_code) {
	    *sp++ = firstcode;
	    code = oldcode;
	}

	while (code >= clear_code) {
	    *sp++ = table[1][code];
	    if (code == table[0][code]) {
		RbImage_setError("circular table entry BIG ERROR");
		return -1;
	    }
	    code = table[0][code];
	}

	*sp++ = firstcode = table[1][code];

	if ((code = max_code) <(1<<MAX_LWZ_BITS)) {
	    table[0][code] = oldcode;
	    table[1][code] = firstcode;
	    max_code++;
	    if ((max_code >= max_code_size) &&
		(max_code_size < (1<<MAX_LWZ_BITS))) {
		max_code_size *= 2;
		code_size++;
	    }
	}

	oldcode = incode;

	if (sp > stack)
	    return *--sp;
    }
    return code;
}

static int
ReadImage(MBuf *mb, RbImage *img, unsigned char cmap[3][MAXCOLORMAPSIZE],
	  int interlace)
{
    char c;
    int count, xpos, ypos, pass, v;
    png_bytep t;

    /* Initialize the Compression routines */
    if (!ReadOK(mb, &c, 1)) {
	RbImage_setError("EOF / read error on image data");
	return -1;
    }
    count = uc(&c,0);

    if (LWZReadByte(mb, true, count) < 0) {
	RbImage_setError("error reading image");
	return -1;
    }

    v = img->width * (img->color_type? 3 : 1);
    for (ypos = 0; ypos < img->height; ypos++)
	img->row_pointers[ypos] = Mem_alloc(v);

    xpos = ypos = pass = 0;
    t = img->row_pointers[ypos];
    while ((v = LWZReadByte(mb, false, count)) >= 0) {
	if (img->color_type == PNG_COLOR_TYPE_GRAY)
	    *t++ = cmap[CM_RED][v];
	else {
	    *t++ = cmap[CM_RED][v];
	    *t++ = cmap[CM_GREEN][v];
	    *t++ = cmap[CM_BLUE][v];
	}

	xpos++;
	if (xpos == img->width) {
	    xpos = 0;
	    if (interlace) {
		switch (pass) {
		  case 0:
		  case 1:
		    ypos += 8; break;
		  case 2:
		    ypos += 4; break;
		  case 3:
		    ypos += 2; break;
		}

		if (ypos >= img->height) {
		    pass++;
		    switch (pass) {
		      case 1:
			ypos = 4; break;
		      case 2:
			ypos = 2; break;
		      case 3:
			ypos = 1; break;
		      default:
			return 0;
		    }
		}
	    }
	    else if (++ypos >= img->height)
		break;
	    t = img->row_pointers[ypos];
	}
    }

    return 0;
}

#endif
