// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

cr.define('cr.icon', function() {
  /**
   * @return {!Array<number>} The scale factors supported by this platform for
   *     webui resources.
   */
  function getSupportedScaleFactors() {
    var supportedScaleFactors = [];
    if (!cr.isIOS) {
      // This matches the code in ResourceBundle::InitSharedInstance() that
      // supports SCALE_FACTOR_100P on all non-iOS platforms.
      supportedScaleFactors.push(1);
    }
    if (cr.isMac || cr.isChromeOS || cr.isWindows || cr.isLinux) {
      // All desktop platforms support zooming which also updates the renderer's
      // device scale factors (a.k.a devicePixelRatio), and these platforms have
      // high DPI assets for 2x.  Let the renderer pick the closest image for
      // the current device scale factor.
      supportedScaleFactors.push(2);
    } else {
      // For other platforms that use fixed device scale factor, use
      // the window's device pixel ratio.
      // TODO(oshima): Investigate corresponding to
      // ResourceBundle::InitSharedInstance() more closely.
      supportedScaleFactors.push(window.devicePixelRatio);
    }
    return supportedScaleFactors;
  }

  /**
   * Generates a CSS -webkit-image-set for a chrome:// url.
   * An entry in the image set is added for each of getSupportedScaleFactors().
   * The scale-factor-specific url is generated by replacing the first instance
   * of 'scalefactor' in |path| with the numeric scale factor.
   *
   * @param {string} path The URL to generate an image set for.
   *     'scalefactor' should be a substring of |path|.
   * @return {string} The CSS -webkit-image-set.
   */
  function getImageSet(path) {
    var supportedScaleFactors = getSupportedScaleFactors();

    var replaceStartIndex = path.indexOf('scalefactor');
    if (replaceStartIndex < 0)
      return url(path);

    var s = '';
    for (var i = 0; i < supportedScaleFactors.length; ++i) {
      var scaleFactor = supportedScaleFactors[i];
      var pathWithScaleFactor = path.substr(0, replaceStartIndex) +
          scaleFactor + path.substr(replaceStartIndex + 'scalefactor'.length);

      s += url(pathWithScaleFactor) + ' ' + scaleFactor + 'x';

      if (i != supportedScaleFactors.length - 1)
        s += ', ';
    }
    return '-webkit-image-set(' + s + ')';
  }

  /**
   * Returns the URL of the image, or an image set of URLs for the provided
   * path.  Resources in chrome://theme have multiple supported scale factors.
   *
   * @param {string} path The path of the image.
   * @return {string} The url, or an image set of URLs.
   */
  function getImage(path) {
    var chromeThemePath = 'chrome://theme';
    var isChromeThemeUrl =
        (path.slice(0, chromeThemePath.length) == chromeThemePath);
    return isChromeThemeUrl ? getImageSet(path + '@scalefactorx') : url(path);
  }

  /**
   * A regular expression for identifying favicon URLs.
   * @const {!RegExp}
   */
  var FAVICON_URL_REGEX = /\.ico$/i;

  /**
   * Creates a CSS -webkit-image-set for a favicon request.
   *
   * @param {string} url Either the URL of the original page or of the favicon
   *     itself.
   * @return {string} -webkit-image-set for the favicon.
   */
  function getFavicon(url) {
    return getImageSet(
        'chrome://favicon/size/16@scalefactorx/' +
        // Note: Literal 'iconurl' must match |kIconURLParameter| in
        // components/favicon_base/favicon_url_parser.cc.
        (FAVICON_URL_REGEX.test(url) ? 'iconurl/' : '') + url);
  }

  return {
    getImage: getImage,
    getFavicon: getFavicon,
  };
});
