# -*- coding: utf-8 -*-
"""QGIS Unit tests for QgsNumericFormat

.. note:: This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
"""
__author__ = 'Nyall Dawson'
__date__ = '6/01/2020'
__copyright__ = 'Copyright 2020, The QGIS Project'

import qgis  # NOQA

from qgis.core import (QgsFallbackNumericFormat,
                       QgsBasicNumericFormat,
                       QgsNumericFormatContext,
                       QgsBearingNumericFormat,
                       QgsPercentageNumericFormat,
                       QgsScientificNumericFormat,
                       QgsCurrencyNumericFormat,
                       QgsNumericFormatRegistry,
                       QgsNumericFormat,
                       QgsReadWriteContext)
from qgis.testing import start_app, unittest
from qgis.PyQt.QtXml import QDomDocument

start_app()


class TestFormat(QgsNumericFormat):

    def id(self):
        return 'test'

    def formatDouble(self, value, context):
        return 'xxx' + str(value)

    def visibleName(self):
        return 'Test'

    def clone(self):
        return TestFormat()

    def create(self, configuration, context):
        return TestFormat()

    def configuration(self, context):
        return {}


class TestQgsNumericFormat(unittest.TestCase):

    def testFallbackFormat(self):
        """ test fallback formatter """
        f = QgsFallbackNumericFormat()
        context = QgsNumericFormatContext()
        self.assertEqual(f.formatDouble(5, context), '5')
        self.assertEqual(f.formatDouble(5.5, context), '5.5')
        self.assertEqual(f.formatDouble(-5, context), '-5')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5')

        f2 = f.clone()
        self.assertIsInstance(f2, QgsFallbackNumericFormat)

        doc = QDomDocument("testdoc")
        elem = doc.createElement("test")
        f2.writeXml(elem, doc, QgsReadWriteContext())

        f3 = QgsNumericFormatRegistry().createFromXml(elem, QgsReadWriteContext())
        self.assertIsInstance(f3, QgsFallbackNumericFormat)

    def testEquality(self):
        f = QgsBasicNumericFormat()
        f2 = QgsBasicNumericFormat()
        self.assertEqual(f, f2)
        f2.setShowPlusSign(True)
        self.assertNotEqual(f, f2)
        f.setShowPlusSign(True)
        self.assertEqual(f, f2)
        self.assertNotEqual(f, QgsCurrencyNumericFormat())

    def testBasicFormat(self):
        """ test basic formatter """
        f = QgsBasicNumericFormat()
        context = QgsNumericFormatContext()
        self.assertEqual(f.formatDouble(0, context), '0')
        self.assertEqual(f.formatDouble(5, context), '5')
        self.assertEqual(f.formatDouble(5.5, context), '5.5')
        self.assertEqual(f.formatDouble(-5, context), '-5')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55,555,555.5')
        context.setDecimalSeparator('☕')
        self.assertEqual(f.formatDouble(0, context), '0')
        self.assertEqual(f.formatDouble(-5.5, context), '-5☕5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55,555,555☕5')
        context.setThousandsSeparator('⚡')
        self.assertEqual(f.formatDouble(-5.5, context), '-5☕5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55⚡555⚡555☕5')
        f.setShowThousandsSeparator(False)
        self.assertEqual(f.formatDouble(-5.5, context), '-5☕5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555☕5')
        context.setDecimalSeparator('.')
        f.setDecimalSeparator('⛹')
        self.assertEqual(f.formatDouble(-5.5, context), '-5⛹5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555⛹5')
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.formatDouble(0, context), '0')
        self.assertEqual(f.formatDouble(5.5, context), '6')
        self.assertEqual(f.formatDouble(55555555.5, context), '55555556')
        self.assertEqual(f.formatDouble(55555555.123456, context), '55555555')
        self.assertEqual(f.formatDouble(-5.5, context), '-6')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555556')
        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '0')
        self.assertEqual(f.formatDouble(5.5, context), '5⛹5')
        self.assertEqual(f.formatDouble(55555555.5, context), '55555555⛹5')
        self.assertEqual(f.formatDouble(55555555.123456, context), '55555555⛹123')
        self.assertEqual(f.formatDouble(-5.5, context), '-5⛹5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555⛹5')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '0⛹000')
        self.assertEqual(f.formatDouble(5, context), '5⛹000')
        self.assertEqual(f.formatDouble(-5, context), '-5⛹000')
        self.assertEqual(f.formatDouble(5.5, context), '5⛹500')
        self.assertEqual(f.formatDouble(55555555.5, context), '55555555⛹500')
        self.assertEqual(f.formatDouble(55555555.123456, context), '55555555⛹123')
        self.assertEqual(f.formatDouble(-5.5, context), '-5⛹500')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555⛹500')
        f.setShowPlusSign(True)
        self.assertEqual(f.formatDouble(0, context), '0⛹000')
        self.assertEqual(f.formatDouble(5, context), '+5⛹000')
        self.assertEqual(f.formatDouble(-5, context), '-5⛹000')
        self.assertEqual(f.formatDouble(5.5, context), '+5⛹500')
        self.assertEqual(f.formatDouble(55555555.5, context), '+55555555⛹500')
        self.assertEqual(f.formatDouble(55555555.123456, context), '+55555555⛹123')
        self.assertEqual(f.formatDouble(-5.5, context), '-5⛹500')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555⛹500')
        context.setPositiveSign('w')
        self.assertEqual(f.formatDouble(5, context), 'w5⛹000')
        self.assertEqual(f.formatDouble(-5, context), '-5⛹000')
        self.assertEqual(f.formatDouble(5.5, context), 'w5⛹500')

        f.setShowPlusSign(False)
        f.setRoundingType(QgsBasicNumericFormat.SignificantFigures)
        self.assertEqual(f.formatDouble(0, context), '0⛹00')
        self.assertEqual(f.formatDouble(5, context), '5⛹00')
        self.assertEqual(f.formatDouble(-5, context), '-5⛹00')
        self.assertEqual(f.formatDouble(5.5, context), '5⛹50')
        self.assertEqual(f.formatDouble(1231.23123123123123, context), '1230')
        self.assertEqual(f.formatDouble(123.123123123123123, context), '123')
        self.assertEqual(f.formatDouble(12.3123123123123123, context), '12⛹3')
        self.assertEqual(f.formatDouble(1.23123123123123123, context), '1⛹23')
        self.assertEqual(f.formatDouble(-1231.23123123123123, context), '-1230')
        self.assertEqual(f.formatDouble(-123.123123123123123, context), '-123')
        self.assertEqual(f.formatDouble(-12.3123123123123123, context), '-12⛹3')
        self.assertEqual(f.formatDouble(-1.23123123123123123, context), '-1⛹23')
        self.assertEqual(f.formatDouble(100, context), '100')
        self.assertEqual(f.formatDouble(1000, context), '1000')
        self.assertEqual(f.formatDouble(1001, context), '1000')
        self.assertEqual(f.formatDouble(9999, context), '10000')
        self.assertEqual(f.formatDouble(10, context), '10⛹0')
        self.assertEqual(f.formatDouble(1, context), '1⛹00')
        self.assertEqual(f.formatDouble(0.00000123456, context), '0⛹00000123')
        self.assertEqual(f.formatDouble(55555555.5, context), '55600000')
        self.assertEqual(f.formatDouble(55555555.123456, context), '55600000')
        self.assertEqual(f.formatDouble(-5.5, context), '-5⛹50')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55600000')

        f.setThousandsSeparator('✅')
        f.setShowThousandsSeparator(True)
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55✅600✅000')
        f.setShowThousandsSeparator(False)

        f.setShowPlusSign(True)

        f2 = f.clone()
        self.assertIsInstance(f2, QgsBasicNumericFormat)

        self.assertEqual(f2.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f2.showPlusSign(), f.showPlusSign())
        self.assertEqual(f2.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f2.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f2.roundingType(), f.roundingType())
        self.assertEqual(f2.thousandsSeparator(), f.thousandsSeparator())
        self.assertEqual(f2.decimalSeparator(), f.decimalSeparator())

        doc = QDomDocument("testdoc")
        elem = doc.createElement("test")
        f2.writeXml(elem, doc, QgsReadWriteContext())

        f3 = QgsNumericFormatRegistry().createFromXml(elem, QgsReadWriteContext())
        self.assertIsInstance(f3, QgsBasicNumericFormat)

        self.assertEqual(f3.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f3.showPlusSign(), f.showPlusSign())
        self.assertEqual(f3.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f3.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f3.roundingType(), f.roundingType())
        self.assertEqual(f3.thousandsSeparator(), f.thousandsSeparator())
        self.assertEqual(f3.decimalSeparator(), f.decimalSeparator())

    def testCurrencyFormat(self):
        """ test currency formatter """
        f = QgsCurrencyNumericFormat()
        f.setPrefix('$')
        context = QgsNumericFormatContext()
        f.setShowTrailingZeros(False)
        self.assertEqual(f.formatDouble(0, context), '$0')
        self.assertEqual(f.formatDouble(5, context), '$5')
        self.assertEqual(f.formatDouble(5.5, context), '$5.5')
        self.assertEqual(f.formatDouble(-5, context), '-$5')
        self.assertEqual(f.formatDouble(-5.5, context), '-$5.5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55,555,555.5')
        context.setDecimalSeparator('x')
        self.assertEqual(f.formatDouble(0, context), '$0')
        self.assertEqual(f.formatDouble(-5.5, context), '-$5x5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55,555,555x5')
        context.setThousandsSeparator('y')
        self.assertEqual(f.formatDouble(-5.5, context), '-$5x5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55y555y555x5')
        f.setShowThousandsSeparator(False)
        self.assertEqual(f.formatDouble(-5.5, context), '-$5x5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55555555x5')
        context.setDecimalSeparator('.')
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.formatDouble(0, context), '$0')
        self.assertEqual(f.formatDouble(5.5, context), '$6')
        self.assertEqual(f.formatDouble(55555555.5, context), '$55555556')
        self.assertEqual(f.formatDouble(55555555.123456, context), '$55555555')
        self.assertEqual(f.formatDouble(-5.5, context), '-$6')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55555556')
        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '$0')
        self.assertEqual(f.formatDouble(5.5, context), '$5.5')
        self.assertEqual(f.formatDouble(55555555.5, context), '$55555555.5')
        self.assertEqual(f.formatDouble(55555555.123456, context), '$55555555.123')
        self.assertEqual(f.formatDouble(-5.5, context), '-$5.5')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55555555.5')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '$0.000')
        self.assertEqual(f.formatDouble(5, context), '$5.000')
        self.assertEqual(f.formatDouble(-5, context), '-$5.000')
        self.assertEqual(f.formatDouble(5.5, context), '$5.500')
        self.assertEqual(f.formatDouble(55555555.5, context), '$55555555.500')
        self.assertEqual(f.formatDouble(55555555.123456, context), '$55555555.123')
        self.assertEqual(f.formatDouble(-5.5, context), '-$5.500')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55555555.500')
        f.setShowPlusSign(True)
        self.assertEqual(f.formatDouble(0, context), '$0.000')
        self.assertEqual(f.formatDouble(5, context), '+$5.000')
        self.assertEqual(f.formatDouble(-5, context), '-$5.000')
        self.assertEqual(f.formatDouble(5.5, context), '+$5.500')
        self.assertEqual(f.formatDouble(55555555.5, context), '+$55555555.500')
        self.assertEqual(f.formatDouble(55555555.123456, context), '+$55555555.123')
        self.assertEqual(f.formatDouble(-5.5, context), '-$5.500')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55555555.500')

        f.setSuffix('AUD')
        self.assertEqual(f.formatDouble(0, context), '$0.000AUD')
        self.assertEqual(f.formatDouble(5, context), '+$5.000AUD')
        self.assertEqual(f.formatDouble(-5, context), '-$5.000AUD')
        self.assertEqual(f.formatDouble(5.5, context), '+$5.500AUD')
        self.assertEqual(f.formatDouble(55555555.5, context), '+$55555555.500AUD')
        self.assertEqual(f.formatDouble(55555555.123456, context), '+$55555555.123AUD')
        self.assertEqual(f.formatDouble(-5.5, context), '-$5.500AUD')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-$55555555.500AUD')

        f2 = f.clone()
        self.assertIsInstance(f2, QgsCurrencyNumericFormat)

        self.assertEqual(f2.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f2.showPlusSign(), f.showPlusSign())
        self.assertEqual(f2.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f2.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f2.prefix(), f.prefix())
        self.assertEqual(f2.suffix(), f.suffix())

        doc = QDomDocument("testdoc")
        elem = doc.createElement("test")
        f2.writeXml(elem, doc, QgsReadWriteContext())

        f3 = QgsNumericFormatRegistry().createFromXml(elem, QgsReadWriteContext())
        self.assertIsInstance(f3, QgsCurrencyNumericFormat)

        self.assertEqual(f3.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f3.showPlusSign(), f.showPlusSign())
        self.assertEqual(f3.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f3.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f3.prefix(), f.prefix())
        self.assertEqual(f3.suffix(), f.suffix())

    def testBearingFormat(self):
        """ test bearing formatter """
        f = QgsBearingNumericFormat()
        f.setDirectionFormat(QgsBearingNumericFormat.UseRange0To180WithEWDirectionalSuffix)
        context = QgsNumericFormatContext()
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(90, context), '90°E')
        self.assertEqual(f.formatDouble(180, context), '180°')
        self.assertEqual(f.formatDouble(270, context), '90°W')
        self.assertEqual(f.formatDouble(300, context), '60°W')
        self.assertEqual(f.formatDouble(5, context), '5°E')
        self.assertEqual(f.formatDouble(5.5, context), '5.5°E')
        self.assertEqual(f.formatDouble(-5, context), '5°W')
        self.assertEqual(f.formatDouble(-5.5, context), '5.5°W')
        context.setDecimalSeparator('x')
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(-5.5, context), '5x5°W')
        self.assertEqual(f.formatDouble(180, context), '180°')
        context.setDecimalSeparator('.')
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(5.5, context), '6°E')
        self.assertEqual(f.formatDouble(-5.5, context), '6°W')
        self.assertEqual(f.formatDouble(180, context), '180°')
        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(5.5, context), '5.5°E')
        self.assertEqual(f.formatDouble(-5.5, context), '5.5°W')
        self.assertEqual(f.formatDouble(180, context), '180°')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '0.000°E') # todo - fix and avoid E
        self.assertEqual(f.formatDouble(5, context), '5.000°E')
        self.assertEqual(f.formatDouble(-5, context), '5.000°W')
        self.assertEqual(f.formatDouble(5.5, context), '5.500°E')
        self.assertEqual(f.formatDouble(-5.5, context), '5.500°W')
        self.assertEqual(f.formatDouble(180, context), '180.000°E') # todo fix and avoid E

        f = QgsBearingNumericFormat()
        f.setDirectionFormat(QgsBearingNumericFormat.UseRangeNegative180ToPositive180)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(90, context), '90°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        self.assertEqual(f.formatDouble(270, context), '-90°')
        self.assertEqual(f.formatDouble(5, context), '5°')
        self.assertEqual(f.formatDouble(5.5, context), '5.5°')
        self.assertEqual(f.formatDouble(-5, context), '-5°')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5°')
        context.setDecimalSeparator('x')
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(-5.5, context), '-5x5°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        context.setDecimalSeparator('.')
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(5.5, context), '6°')
        self.assertEqual(f.formatDouble(-5.5, context), '-6°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(5.5, context), '5.5°')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '0.000°')
        self.assertEqual(f.formatDouble(5, context), '5.000°')
        self.assertEqual(f.formatDouble(-5, context), '-5.000°')
        self.assertEqual(f.formatDouble(5.5, context), '5.500°')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.500°')
        self.assertEqual(f.formatDouble(180, context), '180.000°')

        f = QgsBearingNumericFormat()
        f.setDirectionFormat(QgsBearingNumericFormat.UseRange0To360)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(90, context), '90°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        self.assertEqual(f.formatDouble(270, context), '270°')
        self.assertEqual(f.formatDouble(5, context), '5°')
        self.assertEqual(f.formatDouble(5.5, context), '5.5°')
        self.assertEqual(f.formatDouble(-5, context), '355°')
        self.assertEqual(f.formatDouble(-5.5, context), '354.5°')
        context.setDecimalSeparator('x')
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(-5.5, context), '354x5°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        context.setDecimalSeparator('.')
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(5.5, context), '6°')
        self.assertEqual(f.formatDouble(-5.4, context), '355°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '0°')
        self.assertEqual(f.formatDouble(5.5, context), '5.5°')
        self.assertEqual(f.formatDouble(-5.5, context), '354.5°')
        self.assertEqual(f.formatDouble(180, context), '180°')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '0.000°')
        self.assertEqual(f.formatDouble(5, context), '5.000°')
        self.assertEqual(f.formatDouble(-5, context), '355.000°')
        self.assertEqual(f.formatDouble(5.5, context), '5.500°')
        self.assertEqual(f.formatDouble(-5.5, context), '354.500°')
        self.assertEqual(f.formatDouble(180, context), '180.000°')

        f2 = f.clone()
        self.assertIsInstance(f2, QgsBearingNumericFormat)

        self.assertEqual(f2.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f2.showPlusSign(), f.showPlusSign())
        self.assertEqual(f2.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f2.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f2.directionFormat(), f.directionFormat())

        doc = QDomDocument("testdoc")
        elem = doc.createElement("test")
        f2.writeXml(elem, doc, QgsReadWriteContext())

        f3 = QgsNumericFormatRegistry().createFromXml(elem, QgsReadWriteContext())
        self.assertIsInstance(f3, QgsBearingNumericFormat)

        self.assertEqual(f3.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f3.showPlusSign(), f.showPlusSign())
        self.assertEqual(f3.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f3.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f3.directionFormat(), f.directionFormat())

    def testPercentageFormat(self):
        """ test percentage formatter """
        f = QgsPercentageNumericFormat()
        f.setInputValues(QgsPercentageNumericFormat.ValuesArePercentage)
        context = QgsNumericFormatContext()
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(5, context), '5%')
        self.assertEqual(f.formatDouble(5.5, context), '5.5%')
        self.assertEqual(f.formatDouble(-5, context), '-5%')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55,555,555.5%')
        context.setDecimalSeparator('x')
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(-5.5, context), '-5x5%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55,555,555x5%')
        context.setThousandsSeparator('y')
        self.assertEqual(f.formatDouble(-5.5, context), '-5x5%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55y555y555x5%')
        f.setShowThousandsSeparator(False)
        self.assertEqual(f.formatDouble(-5.5, context), '-5x5%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555x5%')
        context.setDecimalSeparator('.')
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(5.5, context), '6%')
        self.assertEqual(f.formatDouble(55555555.5, context), '55555556%')
        self.assertEqual(f.formatDouble(55555555.123456, context), '55555555%')
        self.assertEqual(f.formatDouble(-5.5, context), '-6%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555556%')
        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(5.5, context), '5.5%')
        self.assertEqual(f.formatDouble(55555555.5, context), '55555555.5%')
        self.assertEqual(f.formatDouble(55555555.123456, context), '55555555.123%')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555.5%')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '0.000%')
        self.assertEqual(f.formatDouble(5, context), '5.000%')
        self.assertEqual(f.formatDouble(-5, context), '-5.000%')
        self.assertEqual(f.formatDouble(5.5, context), '5.500%')
        self.assertEqual(f.formatDouble(55555555.5, context), '55555555.500%')
        self.assertEqual(f.formatDouble(55555555.123456, context), '55555555.123%')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.500%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555.500%')
        f.setShowPlusSign(True)
        self.assertEqual(f.formatDouble(0, context), '0.000%')
        self.assertEqual(f.formatDouble(5, context), '+5.000%')
        self.assertEqual(f.formatDouble(-5, context), '-5.000%')
        self.assertEqual(f.formatDouble(5.5, context), '+5.500%')
        self.assertEqual(f.formatDouble(55555555.5, context), '+55555555.500%')
        self.assertEqual(f.formatDouble(55555555.123456, context), '+55555555.123%')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.500%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-55555555.500%')

        f = QgsPercentageNumericFormat()
        f.setInputValues(QgsPercentageNumericFormat.ValuesAreFractions)
        context = QgsNumericFormatContext()
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(5, context), '500%')
        self.assertEqual(f.formatDouble(5.5, context), '550%')
        self.assertEqual(f.formatDouble(-5, context), '-500%')
        self.assertEqual(f.formatDouble(-5.5, context), '-550%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5,555,555,550%')
        context.setDecimalSeparator('x')
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(-5.5, context), '-550%')
        self.assertEqual(f.formatDouble(-0.005, context), '-0x5%')
        context.setThousandsSeparator('y')
        self.assertEqual(f.formatDouble(-5.5, context), '-550%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5y555y555y550%')
        f.setShowThousandsSeparator(False)
        self.assertEqual(f.formatDouble(-5.5, context), '-550%')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5555555550%')
        context.setDecimalSeparator('.')
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(5.5, context), '550%')
        self.assertEqual(f.formatDouble(55555555.5, context), '5555555550%')
        self.assertEqual(f.formatDouble(0.123456, context), '12%')
        self.assertEqual(f.formatDouble(-5.5, context), '-550%')
        self.assertEqual(f.formatDouble(-0.123456, context), '-12%')
        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '0%')
        self.assertEqual(f.formatDouble(5.5, context), '550%')
        self.assertEqual(f.formatDouble(55555555.5, context), '5555555550%')
        self.assertEqual(f.formatDouble(0.123456, context), '12.346%')
        self.assertEqual(f.formatDouble(-5.5, context), '-550%')
        self.assertEqual(f.formatDouble(-0.123456, context), '-12.346%')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '0.000%')
        self.assertEqual(f.formatDouble(5, context), '500.000%')
        self.assertEqual(f.formatDouble(-5, context), '-500.000%')
        self.assertEqual(f.formatDouble(0.5, context), '50.000%')
        self.assertEqual(f.formatDouble(55555555.5, context), '5555555550.000%')
        self.assertEqual(f.formatDouble(0.123456, context), '12.346%')
        self.assertEqual(f.formatDouble(-5.5, context), '-550.000%')
        self.assertEqual(f.formatDouble(-1234.5, context), '-123450.000%')
        f.setShowPlusSign(True)
        self.assertEqual(f.formatDouble(0, context), '0.000%')
        self.assertEqual(f.formatDouble(5, context), '+500.000%')
        self.assertEqual(f.formatDouble(-5, context), '-500.000%')
        self.assertEqual(f.formatDouble(5.5, context), '+550.000%')
        self.assertEqual(f.formatDouble(-5.5, context), '-550.000%')

        context.setPercent('p')
        self.assertEqual(f.formatDouble(0, context), '0.000p')
        self.assertEqual(f.formatDouble(5, context), '+500.000p')
        self.assertEqual(f.formatDouble(-5, context), '-500.000p')

        f2 = f.clone()
        self.assertIsInstance(f2, QgsPercentageNumericFormat)

        self.assertEqual(f2.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f2.showPlusSign(), f.showPlusSign())
        self.assertEqual(f2.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f2.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f2.inputValues(), f.inputValues())

        doc = QDomDocument("testdoc")
        elem = doc.createElement("test")
        f2.writeXml(elem, doc, QgsReadWriteContext())

        f3 = QgsNumericFormatRegistry().createFromXml(elem, QgsReadWriteContext())
        self.assertIsInstance(f3, QgsPercentageNumericFormat)

        self.assertEqual(f3.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f3.showPlusSign(), f.showPlusSign())
        self.assertEqual(f3.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f3.showThousandsSeparator(), f.showThousandsSeparator())
        self.assertEqual(f3.inputValues(), f.inputValues())

    def testScientificFormat(self):
        """ test scientific formatter """
        f = QgsScientificNumericFormat()
        context = QgsNumericFormatContext()
        self.assertEqual(f.formatDouble(0, context), '0e+00')
        self.assertEqual(f.formatDouble(5, context), '5e+00')
        self.assertEqual(f.formatDouble(5.5, context), '5.5e+00')
        self.assertEqual(f.formatDouble(-5, context), '-5e+00')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5e+00')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5.555556e+07')
        context.setDecimalSeparator('x')
        self.assertEqual(f.formatDouble(0, context), '0e+00')
        self.assertEqual(f.formatDouble(-5.5, context), '-5x5e+00')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5x555556e+07')
        context.setDecimalSeparator('.')

        # places must be at least 1 for scientific notation!
        f.setNumberDecimalPlaces(0)
        self.assertEqual(f.numberDecimalPlaces(), 1)
        self.assertEqual(f.formatDouble(0, context), '0e+00')
        self.assertEqual(f.formatDouble(5.5, context), '5.5e+00')
        self.assertEqual(f.formatDouble(55555555.5, context), '5.6e+07')
        self.assertEqual(f.formatDouble(55555555.123456, context), '5.6e+07')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5e+00')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5.6e+07')

        f.setNumberDecimalPlaces(3)
        self.assertEqual(f.formatDouble(0, context), '0e+00')
        self.assertEqual(f.formatDouble(5.5, context), '5.5e+00')
        self.assertEqual(f.formatDouble(55555555.5, context), '5.556e+07')
        self.assertEqual(f.formatDouble(55555555.123456, context), '5.556e+07')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.5e+00')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5.556e+07')
        f.setShowTrailingZeros(True)
        self.assertEqual(f.formatDouble(0, context), '0.000e+00')
        self.assertEqual(f.formatDouble(5, context), '5.000e+00')
        self.assertEqual(f.formatDouble(-5, context), '-5.000e+00')
        self.assertEqual(f.formatDouble(5.5, context), '5.500e+00')
        self.assertEqual(f.formatDouble(55555555.5, context), '5.556e+07')
        self.assertEqual(f.formatDouble(55555555.123456, context), '5.556e+07')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.500e+00')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5.556e+07')
        f.setShowPlusSign(True)
        self.assertEqual(f.formatDouble(0, context), '0.000e+00')
        self.assertEqual(f.formatDouble(5, context), '+5.000e+00')
        self.assertEqual(f.formatDouble(-5, context), '-5.000e+00')
        self.assertEqual(f.formatDouble(5.5, context), '+5.500e+00')
        self.assertEqual(f.formatDouble(55555555.5, context), '+5.556e+07')
        self.assertEqual(f.formatDouble(55555555.123456, context), '+5.556e+07')
        self.assertEqual(f.formatDouble(-5.5, context), '-5.500e+00')
        self.assertEqual(f.formatDouble(-55555555.5, context), '-5.556e+07')

        f2 = f.clone()
        self.assertIsInstance(f2, QgsScientificNumericFormat)

        self.assertEqual(f2.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f2.showPlusSign(), f.showPlusSign())
        self.assertEqual(f2.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f2.showThousandsSeparator(), f.showThousandsSeparator())

        doc = QDomDocument("testdoc")
        elem = doc.createElement("test")
        f2.writeXml(elem, doc, QgsReadWriteContext())

        f3 = QgsNumericFormatRegistry().createFromXml(elem, QgsReadWriteContext())
        self.assertIsInstance(f3, QgsScientificNumericFormat)

        self.assertEqual(f3.showTrailingZeros(), f.showTrailingZeros())
        self.assertEqual(f3.showPlusSign(), f.showPlusSign())
        self.assertEqual(f3.numberDecimalPlaces(), f.numberDecimalPlaces())
        self.assertEqual(f3.showThousandsSeparator(), f.showThousandsSeparator())

    def testRegistry(self):
        registry = QgsNumericFormatRegistry()
        self.assertTrue(registry.formats())
        for f in registry.formats():
            self.assertEqual(registry.format(f).id(), f)

        self.assertIn('default', registry.formats())
        registry.addFormat(TestFormat())
        self.assertIn('test', registry.formats())
        self.assertTrue(isinstance(registry.format('test'), TestFormat))
        self.assertTrue(isinstance(registry.create('test', {}, QgsReadWriteContext()), TestFormat))

        registry.removeFormat('test')

        self.assertNotIn('test', registry.formats())
        self.assertTrue(isinstance(registry.format('test'), QgsFallbackNumericFormat))
        self.assertTrue(isinstance(registry.create('test', {}, QgsReadWriteContext()), QgsFallbackNumericFormat))

        self.assertTrue(isinstance(registry.fallbackFormat(), QgsFallbackNumericFormat))

        self.assertEqual(registry.visibleName('default'), 'General')
        self.assertEqual(registry.visibleName('basic'), 'Number')

        self.assertEqual(registry.sortKey('default'), 0)
        self.assertEqual(registry.sortKey('basic'), 1)
        self.assertEqual(registry.sortKey('currency'), 100)


if __name__ == '__main__':
    unittest.main()
