#!/usr/local/bin/python

# ========================================================================
#  Classes which define devices
#
#  Brian Landers <blanders@sapient.com>
#  07.07.2001
#
#  $Id: devices.py,v 1.5 2001/10/24 18:42:57 bluecoat93 Exp $
# ========================================================================

class Device:
    def __init__( self ):
        """ Constructor """
        self._class             = "BASE CLASS"
        
        self._needsEnable       = 1
        self._commands          = { 'disablePaging' : '',
                                    'enablePaging'  : '',
                                    'getConfig'     : '',
                                    'enable'        : '',
                                    'disable'       : '' }

        self._prompts           = { 'login'           : '',
                                    'username'        : '',
                                    'password'        : '',
                                    'command'         : '',
                                    'enable'          : '',
                                    'enabledIndicator': ''}

        # These are the default commands and prompts
        self.setCommand( 'enable',        'enable' )
        self.setCommand( 'disable',       'disable' )
        self.setPrompt( 'login',    '[Ll]ogin[:\s]*$' )
        self.setPrompt( 'username', '[Uu]sername[:\s]*$' )
        self.setPrompt( 'password', '[Pp]assw(?:or)?d[:\s]*$' )
        self.setPrompt( 'command',  '[\w().-]*[\$#>]\s?(?:\(enable\))?\s*$' )
        self.setPrompt( 'enable',   '[Pp]assword[:\s]*$' )
        self.setPrompt( 'enabledIndicator', '\#|enable' )

    def getPrompt( self, inKey=None ):
        """ Get the RE to match a given prompt on the device """
        assert inKey != None
        
        try:
            return self._prompts[inKey]
        except KeyError:
            return ''

    def setPrompt( self, inKey=None, inValue=None ):
        """ Set the RE to match a given prompt on the device """
        assert inKey   != None
        assert inValue != None

        self._prompts[inKey] = inValue

    def getCommand( self, inKey=None ):
        """ Get the command to perform a given function on the device """
        assert inKey != None
        
        try:
            return self._commands[inKey]
        except KeyError:
            return ''

    def setCommand( self, inKey=None, inValue=None ):
        """ Set the command to perform a given function on the device """
        assert inKey   != None
        assert inValue != None

        self._commands[inKey] = inValue

    def needsEnable( self ):
        return self._needsEnable

class IOSDevice( Device ):
    def __init__( self ):
        """ Constructor"""
        Device.__init__( self )
        
        self._class       = "IOS"
        self._needsEnable = 1
        
        self.setCommand( 'disablePaging', 'terminal length 0' )
        self.setCommand( 'enablePaging',  'terminal length 24' )
        self.setCommand( 'getConfig',     'show running-config' )
        
class CatOSDevice( Device ):
    def __init__( self ):
        """ Constructor """
        Device.__init__( self )
        
        self._class       = "CatOS"
        self._needsEnable = 1
        
        self.setCommand( 'disablePaging', 'set length 0' )
        self.setCommand( 'enablePaging',  'set length 24' )
        self.setCommand( 'getConfig',     'write term' )
        
class PixDevice( Device ):
    def __init__( self ):
        """ Constructor """
        Device.__init__( self )
        
        self._class       = "Pix"
        self._needsEnable = 1
        
        self.setCommand( 'disablePaging', 'no pager' )
        self.setCommand( 'enablePaging',  'pager' )
        self.setCommand( 'getConfig',     'write term' )

class DeviceFactory:
    def createDevice( self, inClass=None ):
        assert inClass != None

        if inClass == 'IOS':
            return IOSDevice()
        elif inClass == 'CatOS':
            return CatOSDevice()
        elif inClass == 'Pix':
            return PixDevice()
        else:
            raise RuntimeError( "Class '" + inClass + "' not supported" )

# ------------------------------------------------------------------------

if __name__ == "__main__":
    foo = DeviceFactory().createDevice( "IOS" )
