// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0

#include <gtest/gtest.h>
#include <openssl/base.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/mem.h>
#include <openssl/obj.h>
#include "../test/test_util.h"

#include <vector>
#include "../crypto/evp_extra/internal.h"
#include "../fipsmodule/evp/internal.h"
#include "../internal.h"

#ifdef ENABLE_DILITHIUM

#include "sig_dilithium.h"

static const uint8_t kPublicKey[] = { 0xBB, 0x0A, 0x41, 0x1A, 0x53, 0x91, 0x4E,
    0x67, 0x77, 0x78, 0xD0, 0xBC, 0xF0, 0xFD, 0x8A, 0x39, 0x65, 0x96, 0x48,
    0x54, 0xF5, 0x91, 0xCE, 0xDB, 0x3B, 0x92, 0xBF, 0x2A, 0xC4, 0x01, 0xCD,
    0xBE, 0x9E, 0x60, 0x05, 0x45, 0x68, 0x3E, 0xC0, 0xC6, 0xDC, 0xBB, 0x76,
    0x42, 0x2D, 0x91, 0x47, 0xC2, 0x7C, 0x96, 0xAA, 0x5B, 0x28, 0x0D, 0x3B,
    0x24, 0x50, 0x12, 0x4E, 0x05, 0x0F, 0xBD, 0x09, 0xB5, 0x51, 0xF5, 0xD6,
    0x12, 0x93, 0xF8, 0x62, 0x69, 0x92, 0x25, 0x3B, 0x2E, 0x1B, 0xC8, 0x47,
    0xFA, 0x06, 0x2C, 0x21, 0x8A, 0x62, 0xA2, 0x0A, 0x3F, 0x1B, 0xD3, 0x6D,
    0x2D, 0x82, 0x4D, 0x7D, 0xDE, 0x80, 0xBB, 0x08, 0x79, 0x53, 0xD8, 0xDB,
    0xCD, 0x73, 0xF8, 0x8F, 0x85, 0xA8, 0xB4, 0xB5, 0x63, 0xF5, 0x6A, 0x9C,
    0xD5, 0x58, 0xAC, 0xD7, 0x9C, 0x96, 0x3C, 0xEA, 0x5A, 0x66, 0x69, 0xD6,
    0x5E, 0xC3, 0xD5, 0x81, 0xFE, 0x6B, 0x42, 0xA0, 0xB6, 0x58, 0x54, 0xFA,
    0x2F, 0xBE, 0x92, 0x9A, 0x41, 0xF8, 0x32, 0x10, 0x10, 0x86, 0x77, 0x01,
    0xD6, 0xEF, 0x81, 0x14, 0x1B, 0xA3, 0x8B, 0xF4, 0x33, 0x94, 0x4E, 0x85,
    0xDC, 0x94, 0x7C, 0x01, 0x9E, 0x6D, 0x9E, 0x7A, 0x26, 0x0F, 0xCF, 0xA0,
    0x7D, 0x92, 0xB0, 0x84, 0x95, 0x76, 0xDF, 0x40, 0x85, 0x87, 0xEF, 0xF5,
    0x33, 0x52, 0x2E, 0xF6, 0x73, 0x9E, 0xDA, 0x27, 0x6C, 0xEE, 0x08, 0xFE,
    0x7A, 0x67, 0x43, 0x88, 0xC6, 0xF0, 0xF1, 0x0C, 0x08, 0xA5, 0x67, 0xC4,
    0xB0, 0x01, 0xFB, 0x5B, 0x59, 0xFC, 0xAE, 0xCF, 0x07, 0x14, 0x52, 0x45,
    0xD0, 0xEB, 0x97, 0xAB, 0x0E, 0x52, 0x65, 0x12, 0x41, 0xA2, 0x03, 0xB1,
    0xE5, 0x1E, 0xA6, 0x33, 0xCE, 0x9A, 0x5C, 0x8F, 0xDF, 0x1A, 0x42, 0x6C,
    0xB6, 0xCB, 0x78, 0xDD, 0x02, 0x78, 0xD9, 0xD8, 0xAE, 0x31, 0x5F, 0x13,
    0xEB, 0x71, 0xD3, 0xAD, 0x6B, 0xD0, 0x98, 0x18, 0x7A, 0x5B, 0x0D, 0xA9,
    0x5D, 0x35, 0xB4, 0x26, 0x5E, 0xCA, 0xF6, 0x7D, 0xC3, 0x9F, 0xE1, 0xBD,
    0xFB, 0x73, 0xA6, 0xF1, 0xE0, 0x26, 0x0F, 0xCA, 0x1B, 0xC3, 0xA9, 0xD0,
    0x27, 0x46, 0x82, 0x49, 0x73, 0xED, 0x3A, 0x6B, 0x61, 0x05, 0xD8, 0xA7,
    0x7F, 0x2B, 0x36, 0x1C, 0x69, 0x31, 0x73, 0x29, 0xDE, 0x6D, 0x28, 0xB6,
    0xEA, 0xFE, 0x37, 0xC2, 0xC0, 0x94, 0x4B, 0x3E, 0xD4, 0x21, 0x3F, 0xE0,
    0x02, 0xFA, 0x5F, 0x16, 0xCE, 0x16, 0x66, 0xB5, 0xC9, 0xF6, 0x3F, 0x2A,
    0x5C, 0x83, 0x00, 0x5F, 0xDA, 0x0B, 0x6E, 0xE7, 0xB6, 0x55, 0x55, 0xBF,
    0x50, 0x29, 0x8E, 0x39, 0xA2, 0xB7, 0x85, 0x79, 0xC3, 0xCE, 0x93, 0x63,
    0x57, 0xD1, 0xF2, 0x03, 0x6E, 0x42, 0x78, 0xB3, 0x5F, 0x72, 0x5C, 0x7D,
    0xB0, 0xE0, 0x55, 0x96, 0xE1, 0x34, 0x0B, 0x5C, 0x75, 0xA9, 0xEF, 0x17,
    0x47, 0x5B, 0xDC, 0x4F, 0x01, 0x5D, 0x80, 0x60, 0x14, 0xA7, 0x9A, 0xD3,
    0x38, 0x07, 0xDA, 0x61, 0x0D, 0x7F, 0x37, 0xEC, 0x0A, 0xD3, 0x54, 0x91,
    0x53, 0x16, 0x3D, 0xAD, 0xC8, 0x2C, 0xAC, 0xBB, 0x1B, 0x03, 0xD7, 0x96,
    0x8D, 0x6D, 0x2D, 0x81, 0xBB, 0x57, 0xDB, 0x10, 0x3E, 0xFD, 0xCF, 0xA9,
    0xA3, 0xA4, 0xAF, 0xF7, 0xA1, 0x9F, 0x4B, 0xB0, 0x3E, 0x3D, 0x51, 0x15,
    0xEA, 0x63, 0x8C, 0x75, 0x59, 0x78, 0xBB, 0xEC, 0x94, 0x47, 0xA8, 0x81,
    0x7B, 0x98, 0xF1, 0x6E, 0x0A, 0xAD, 0xE6, 0x42, 0x58, 0x37, 0x76, 0x8E,
    0x06, 0xA5, 0x43, 0xBF, 0xE9, 0xFA, 0x53, 0x2E, 0xAC, 0x2E, 0xBC, 0x6A,
    0x99, 0x08, 0x51, 0x98, 0xAC, 0x3A, 0x23, 0x52, 0x19, 0x07, 0x81, 0x14,
    0x4C, 0x62, 0x3F, 0x38, 0x2C, 0xB6, 0x70, 0x33, 0xD8, 0xB3, 0x97, 0x5B,
    0x9C, 0x54, 0x62, 0x63, 0x03, 0xA8, 0xAD, 0xC1, 0xB3, 0xC6, 0x3A, 0x74,
    0x5D, 0x6E, 0x9C, 0x3F, 0xCE, 0xE8, 0x79, 0xF4, 0x02, 0x90, 0x3A, 0x2B,
    0x51, 0x50, 0x9F, 0x88, 0xF5, 0x98, 0xA7, 0xC4, 0xD2, 0xE9, 0x4F, 0xCA,
    0x52, 0xE9, 0xE1, 0xB2, 0xB3, 0x78, 0xB0, 0xA2, 0x33, 0x9F, 0xA8, 0xC3,
    0xD9, 0xB6, 0xDD, 0x6A, 0x05, 0x5B, 0xC7, 0x52, 0xC3, 0x68, 0xC2, 0x30,
    0x7F, 0x74, 0x1E, 0x07, 0x66, 0xC3, 0xA4, 0x2D, 0x84, 0xAA, 0x17, 0xD6,
    0x49, 0x89, 0x60, 0x31, 0x0E, 0x4E, 0xAE, 0xCF, 0x56, 0x93, 0x40, 0x3C,
    0x9B, 0xC9, 0x13, 0x5C, 0x5D, 0x38, 0x94, 0xAE, 0x6F, 0xE9, 0x04, 0xB7,
    0xA6, 0x1E, 0xAB, 0xB9, 0xD4, 0x04, 0xFF, 0x3C, 0x99, 0x2F, 0xE4, 0x80,
    0x69, 0xDB, 0x50, 0x5B, 0xAC, 0x34, 0x48, 0x77, 0xF6, 0xB2, 0x00, 0x02,
    0x8D, 0x75, 0xA0, 0x2E, 0x93, 0x0A, 0x20, 0x22, 0x2F, 0x73, 0x15, 0xEA,
    0xEA, 0xBD, 0x8B, 0xE6, 0x85, 0xFD, 0x30, 0x97, 0xF7, 0x24, 0xB9, 0xD5,
    0x30, 0xA0, 0x7A, 0x36, 0x0E, 0xA4, 0xE1, 0xEA, 0xFD, 0xE6, 0xDD, 0xB6,
    0xFF, 0xC6, 0xFC, 0x20, 0xD6, 0x71, 0x36, 0x1B, 0x5D, 0xF2, 0x63, 0xCA,
    0xC6, 0xDE, 0x55, 0xF2, 0x39, 0x05, 0x9A, 0x88, 0x0C, 0x7C, 0xAC, 0x5C,
    0xC0, 0x50, 0x05, 0x1F, 0xDE, 0xBB, 0x6E, 0xF2, 0x06, 0x77, 0x9A, 0xC7,
    0x03, 0x14, 0x26, 0xD4, 0x5D, 0xE0, 0xA9, 0x02, 0xC6, 0xA1, 0xCA, 0x4B,
    0x8D, 0xAC, 0x0B, 0x54, 0xC1, 0xF7, 0x91, 0xBF, 0xD0, 0xE6, 0xEE, 0x4B,
    0x21, 0x82, 0xE2, 0x33, 0xD2, 0xB4, 0xC0, 0x14, 0x64, 0xC8, 0x6B, 0xD0,
    0x50, 0xE5, 0x4A, 0x84, 0xF2, 0x25, 0x23, 0x2E, 0xD2, 0x65, 0x3B, 0x4A,
    0x72, 0x03, 0xAA, 0x0C, 0x82, 0x98, 0x14, 0x11, 0x8E, 0x4A, 0x1F, 0x77,
    0xE1, 0x82, 0xBE, 0xB4, 0x5C, 0xE4, 0x7A, 0x74, 0xFE, 0x29, 0x42, 0xCD,
    0x9D, 0xED, 0x40, 0xC8, 0x88, 0x85, 0xFA, 0x98, 0xBA, 0xBC, 0x45, 0xC7,
    0xCE, 0x5B, 0xB8, 0x36, 0x1B, 0x2A, 0x69, 0x55, 0xF5, 0x98, 0xCF, 0xD7,
    0x4E, 0x7F, 0x2F, 0x59, 0x4D, 0x26, 0x33, 0x54, 0xD2, 0x5F, 0x51, 0xFD,
    0xD2, 0xB9, 0xBB, 0xE0, 0x99, 0x5A, 0xDD, 0xD0, 0x11, 0xEA, 0xFC, 0x84,
    0xD2, 0x21, 0x06, 0x26, 0xEA, 0x2E, 0xBA, 0x52, 0xEB, 0x39, 0xF0, 0x97,
    0x5B, 0xCC, 0x76, 0xAF, 0x87, 0x94, 0x9F, 0xB0, 0xA0, 0xD1, 0x59, 0x9F,
    0x73, 0x83, 0x69, 0x50, 0x76, 0x03, 0x0D, 0x44, 0x13, 0xE1, 0x2B, 0x31,
    0xF9, 0xF4, 0xE2, 0x6D, 0xCD, 0x31, 0xBF, 0x39, 0xFC, 0x61, 0xA1, 0x40,
    0x52, 0x56, 0xF5, 0xD7, 0x12, 0x12, 0xAB, 0x5E, 0x81, 0x48, 0x8A, 0x2E,
    0xE8, 0x85, 0xDC, 0xE7, 0x65, 0x16, 0x6B, 0xBE, 0x28, 0x93, 0x95, 0x0B,
    0x0E, 0x20, 0xFE, 0x4F, 0xF7, 0xEB, 0x0E, 0xEF, 0x2D, 0x41, 0x0C, 0x24,
    0x0C, 0x3F, 0xA6, 0x62, 0x61, 0xB3, 0x7A, 0x5E, 0x9A, 0xEF, 0xF6, 0xAE,
    0x93, 0x51, 0xD5, 0xD7, 0x94, 0xBB, 0xA0, 0x9E, 0xDF, 0x90, 0x87, 0xBD,
    0xD3, 0xF8, 0x4B, 0x86, 0x9E, 0xFF, 0xC5, 0xD7, 0xAF, 0x32, 0x72, 0xC4,
    0xB8, 0x4D, 0x4C, 0x3E, 0xCD, 0xB1, 0xEE, 0xAE, 0xDD, 0x31, 0x12, 0xC6,
    0xD8, 0xC8, 0xFA, 0x20, 0x90, 0xB1, 0x83, 0xAA, 0x33, 0x1F, 0x07, 0xA6,
    0xC1, 0xA1, 0x31, 0x05, 0x3F, 0xE3, 0xA5, 0x4D, 0x41, 0x8B, 0x1C, 0x5B,
    0x51, 0x10, 0x03, 0x0B, 0x0B, 0x2C, 0x3F, 0x97, 0xC5, 0xA2, 0x38, 0x0D,
    0xCB, 0xEF, 0x53, 0x45, 0xBE, 0x28, 0x35, 0x6E, 0x02, 0x10, 0xE8, 0x29,
    0x23, 0xE7, 0xE4, 0xC8, 0x90, 0x99, 0x72, 0xDB, 0xB9, 0x32, 0xB7, 0x87,
    0x52, 0x85, 0x7C, 0x92, 0x47, 0xAF, 0x71, 0x0B, 0x51, 0xC7, 0x8F, 0x24,
    0xE4, 0x7C, 0xE2, 0xF3, 0x33, 0x05, 0xFF, 0x92, 0x0C, 0xC0, 0x01, 0xF0,
    0xF5, 0xEF, 0x52, 0xA1, 0x69, 0xA3, 0x58, 0xFC, 0x48, 0x58, 0x28, 0xBA,
    0x53, 0xA2, 0x3D, 0x03, 0x89, 0x34, 0x12, 0x78, 0xBC, 0x31, 0x5A, 0x93,
    0x62, 0x32, 0xE5, 0xE2, 0x10, 0x56, 0xC0, 0xCA, 0xED, 0x1E, 0x2E, 0x6E,
    0x23, 0x6B, 0xA1, 0x46, 0x6D, 0x7D, 0x58, 0x0B, 0xCC, 0xB0, 0x5B, 0x7B,
    0xA1, 0x48, 0x89, 0xF6, 0x10, 0xBE, 0x8C, 0x30, 0x10, 0xE6, 0xE1, 0x87,
    0x3C, 0xD0, 0x27, 0x47, 0x06, 0xAA, 0xB9, 0x73, 0x26, 0xED, 0x7F, 0x38,
    0xCA, 0xC5, 0xD0, 0x3F, 0x27, 0xEE, 0xA9, 0x5F, 0xF0, 0x31, 0xA9, 0x88,
    0x26, 0x7D, 0x32, 0x34, 0x13, 0xD1, 0xDC, 0x39, 0x7F, 0xAC, 0x3B, 0xB6,
    0xC6, 0x97, 0x5E, 0x44, 0xCA, 0x68, 0xA4, 0x84, 0x90, 0xD0, 0x90, 0x6F,
    0x93, 0xB4, 0xDA, 0x61, 0x43, 0x23, 0xE8, 0x31, 0x34, 0xCD, 0x2F, 0xCA,
    0xB8, 0x63, 0x5F, 0x59, 0x71, 0xE2, 0x2A, 0x11, 0x4F, 0xD0, 0x12, 0xF6,
    0x70, 0x5F, 0xEE, 0xA8, 0xB2, 0x98, 0x89, 0x6C, 0xED, 0x76, 0xCD, 0xE3,
    0x81, 0x1E, 0x5C, 0xEC, 0x7D, 0xB6, 0x21, 0xCE, 0x49, 0x25, 0x7F, 0x97,
    0x8F, 0x1F, 0xBA, 0x81, 0x0B, 0x9F, 0x58, 0xA9, 0x16, 0x19, 0xA2, 0x56,
    0x2F, 0x89, 0x59, 0xA6, 0x8D, 0x97, 0x7C, 0xF7, 0x02, 0xB3, 0x39, 0x37,
    0x4B, 0x79, 0xCF, 0x84, 0x6B, 0xDE, 0x40, 0x6D, 0x1D, 0x17, 0x5E, 0x48,
    0x62, 0x6C, 0x07, 0xE8, 0xBC, 0x0E, 0xD6, 0xD4, 0xEC, 0xA7, 0x55, 0x16,
    0x57, 0x05, 0xC2, 0x5B, 0xA8, 0xBB, 0xA1, 0x8B, 0x63, 0x04, 0x63, 0x99,
    0xC2, 0x11, 0x29, 0xCC, 0x37, 0xC7, 0xB1, 0x70, 0x34, 0x27, 0x1F, 0xD7,
    0xB4, 0x4D, 0xD6, 0xF3, 0x4A, 0x92, 0xF6, 0xA6, 0xAA, 0x8F, 0xC6, 0xC8,
    0x6B, 0xE1, 0xFC, 0x97, 0xAF, 0xC9, 0x6F, 0x68, 0xF1, 0x41, 0xA1, 0xB2,
    0x8C, 0x9E, 0xE2, 0xD7, 0x64, 0xA3, 0xD9, 0x47, 0xBD, 0xBA, 0xFC, 0x84,
    0x14, 0x6C, 0x4B, 0xCC, 0xE6, 0xB4, 0x93, 0x54, 0x03, 0xA6, 0x14, 0xD5,
    0xB1, 0xFB, 0xBC, 0x8E, 0x96, 0x88, 0xB4, 0xD6, 0x86, 0xE5, 0xEE, 0x5B,
    0x1A, 0xA1, 0x7C, 0x43, 0x46, 0x6B, 0x80, 0x2C, 0xAD, 0xDE, 0x4B, 0xA9,
    0x75, 0xA1, 0x42, 0xC7, 0x0B, 0x72, 0xCF, 0x10, 0xA7, 0x5A, 0xD8, 0x96,
    0x0D, 0xC1, 0xB2, 0xAB, 0x5D, 0xD2, 0xA5, 0x5E, 0xE6, 0x72, 0xE6, 0x46,
    0x72, 0x85, 0xAF, 0xC3, 0xD0, 0x09, 0x81, 0xF8, 0x9F, 0xA2, 0xD6, 0x40,
    0xB4, 0x50, 0x0D, 0x35, 0x6A, 0xEB, 0x9F, 0xC2, 0xF1, 0x41, 0x5A, 0x2B,
    0x87, 0xE5, 0x1D, 0xA3, 0x36, 0x6F, 0x5A, 0x95, 0x49, 0xD5, 0x67, 0x1E,
    0xD5, 0x6D, 0xCF, 0x8A, 0xF7, 0xF9, 0x53, 0x9C, 0xCC, 0x7E, 0xD1, 0x3A,
    0xBD, 0xB9, 0x36, 0xEB, 0x82, 0xC5, 0xB5, 0xAC, 0xE7, 0x60, 0x4B, 0x1F,
    0x96, 0xF8, 0xF5, 0x6F, 0x4D, 0x11, 0x76, 0xAF, 0xD0, 0x79, 0x86, 0xD7,
    0xCF, 0x49, 0x61, 0x93, 0x2E, 0xDA, 0xFD, 0x64, 0x4E, 0xB3, 0x27, 0xFA,
    0x38, 0x91, 0x27, 0xA0, 0x94, 0x35, 0xD6, 0x8E, 0x03, 0x6E, 0xB8, 0x67,
    0x2B, 0x35, 0x8E, 0x56, 0x49, 0x52, 0xF1, 0x82, 0x9A, 0x6A, 0x9C, 0xE4,
    0x4E, 0x56, 0xD1, 0x2E, 0xCB, 0xD4, 0x1E, 0x27, 0x69, 0xA9, 0xA4, 0xFE,
    0xB4, 0x33, 0x0C, 0xCD, 0x7A, 0xA9, 0xC2, 0x60, 0x64, 0xB2, 0x4A, 0x34,
    0xBF, 0x8F, 0x67, 0x9A, 0x82, 0x1E, 0x03, 0x0B, 0x85, 0x6F, 0xA6, 0xA0,
    0x2A, 0x7B, 0x27, 0xC6, 0xEA, 0x67, 0x8C, 0xD9, 0xB4, 0x57, 0xAB, 0xBA,
    0x97, 0x6E, 0x51, 0x94, 0x20, 0x00, 0x8B, 0x0C, 0x2B, 0xF9, 0x9D, 0xEA,
    0xB6, 0xAC, 0x56, 0x6F, 0xEF, 0x29, 0x6D, 0x15, 0xC5, 0x30, 0xAB, 0x10,
    0xA7, 0x2F, 0x82, 0x95, 0xAC, 0x78, 0xE1, 0x91, 0x61, 0xE4, 0xA0, 0x40,
    0x98, 0x84, 0x91, 0xDF, 0xD9, 0x6C, 0x78, 0x2F, 0xC3, 0x37, 0x81, 0xC7,
    0x4E, 0xE0, 0x74, 0x0A, 0xD9, 0xBB, 0xBE, 0x30, 0x2C, 0x18, 0x14, 0x88,
    0xAF, 0xDB, 0xD1, 0x56, 0x3C, 0x3D, 0x8E, 0xA7, 0xD8, 0x4E, 0x4C, 0xFE,
    0x29, 0x59, 0x44, 0x0B, 0x39, 0x90, 0x3C, 0x7A, 0xE2, 0xDE, 0x74, 0xB3,
    0xAD, 0xFC, 0xBD, 0x69, 0xAD, 0x23, 0x85, 0x61, 0x49, 0x36, 0x88, 0x26,
    0xED, 0x3A, 0x5A, 0xB0, 0x68, 0x11, 0x34, 0x5A, 0x91, 0xD1, 0x6F, 0x88,
    0xDE, 0x9A, 0x11, 0x43, 0xCD, 0x26, 0xDB, 0x14, 0x80, 0x17, 0xF7, 0xFD,
    0x78, 0x87, 0xA5, 0x14, 0x5A, 0x59, 0x2C, 0x45, 0x54, 0xE7, 0x6D, 0x30,
    0x77, 0x06, 0x2F, 0xF9, 0xD4, 0xDB, 0xCC, 0xDE, 0x60, 0x82, 0x45, 0x14,
    0x19, 0xB7, 0x38, 0x22, 0xDA, 0xC1, 0x6E, 0x51, 0xA4, 0xD6, 0xD1, 0xBF,
    0x6F, 0xA0, 0x47, 0x95, 0x0C, 0x76, 0x50, 0xBA, 0x07, 0xF4, 0xC4, 0x4A,
    0xD0, 0x43, 0x86, 0x3E, 0x40, 0x1B, 0xFB, 0xCB, 0xB4, 0x56, 0x83, 0xD5,
    0xD1, 0xEE, 0x24, 0xB7, 0xB5, 0x40, 0x78, 0x0B, 0xA0, 0x9E, 0x4B, 0x83,
    0xBE, 0x0C, 0x26, 0x76, 0xBF, 0x5E, 0xF2, 0xE9, 0x1B, 0x1D, 0x29, 0x70,
    0x62, 0x38, 0x97, 0xD2, 0xC5, 0x37, 0xFA, 0xE8, 0xF1, 0xE6, 0x79, 0xC2,
    0x20, 0x1C, 0xC7, 0x27, 0x65, 0xA6, 0xD1, 0x7F, 0xE7, 0x7B, 0x28, 0x84,
    0x0A, 0x29, 0x28, 0x6B, 0xD2, 0xDD, 0x1B, 0xF9, 0x12, 0xDE, 0xFC, 0xA9,
    0x09, 0x61, 0xD9, 0xF6, 0x63, 0x7C, 0x8A, 0x00, 0xD8, 0x73, 0xA7, 0x98,
    0xD8, 0xB2, 0xB4, 0x80, 0x98, 0xCA, 0x7D, 0x40, 0xC6, 0x4E, 0x51, 0x41,
    0x84, 0x25, 0x57, 0xD6, 0xED, 0x67, 0x4A, 0x76, 0xC3, 0xFA, 0x4D, 0x37,
    0xA5, 0x4E, 0xCF, 0x19, 0x38, 0xDA, 0x34, 0x02, 0x5D, 0xD9, 0x66, 0xA9,
    0xDF, 0x78, 0xE1, 0x4F, 0xD7, 0xB1, 0x37, 0xC6, 0x60, 0x94, 0x09, 0xD3,
    0xE6, 0xC4, 0xDA, 0x03, 0x84, 0xE6, 0x6F, 0xAB, 0x26, 0xBA, 0xDF, 0xF4,
    0x23, 0xD2, 0x00, 0xCE, 0x85, 0x6E, 0x8C, 0xE9, 0x17, 0xB2, 0x8D, 0x81,
    0x32, 0x01, 0xCC, 0x21, 0xE9, 0x47, 0x43, 0x6F, 0x47, 0xF4, 0x5B, 0x6F,
    0x2B, 0x31, 0xE5, 0x4A, 0x0B, 0xF2, 0x77, 0x28, 0x0A, 0xAC, 0xAF, 0x7A,
    0xF0, 0xF0, 0x33, 0x59, 0xBB, 0xCC, 0xB9, 0xF8, 0x08, 0x94, 0x3D, 0x25,
    0x4E, 0x0C, 0x92, 0x68, 0xDE, 0x7F, 0x82, 0x15, 0xBB, 0x42, 0xF2, 0x94,
    0x8C, 0xE0, 0x72, 0x90, 0x9E, 0x58, 0xF0, 0x6E, 0x29, 0x1E, 0xAD, 0xFC,
    0x35, 0x87, 0xD9, 0x88, 0xDD, 0xA6, 0xFE, 0xAF, 0x3B, 0x2E, 0xD9, 0x0C,
    0x9B, 0x0A, 0x91, 0xE6, 0x0E, 0xD4, 0xD0, 0xFF, 0xD9, 0x64, 0x7E, 0xCF,
    0xF9, 0x7D, 0x10, 0x10, 0x16, 0x5D, 0xE2, 0x9F, 0xD4, 0x54, 0x47, 0xC1,
    0xCF, 0xF1, 0x6D, 0xED, 0x42, 0xCB, 0x1B, 0xA1, 0x87, 0xED, 0xB1, 0x15,
    0xD2,
};

static const uint8_t kPublicKeySPKI[] = {
    0x30, 0x82, 0x07, 0xb4, 0x30, 0x0d, 0x06, 0x0b, 0x2b, 0x06, 0x01, 0x04,
    0x01, 0x02, 0x82, 0x0b, 0x07, 0x06, 0x05, 0x03, 0x82, 0x07, 0xa1, 0x00,
    0xbb, 0x0a, 0x41, 0x1a, 0x53, 0x91, 0x4e, 0x67, 0x77, 0x78, 0xd0, 0xbc,
    0xf0, 0xfd, 0x8a, 0x39, 0x65, 0x96, 0x48, 0x54, 0xf5, 0x91, 0xce, 0xdb,
    0x3b, 0x92, 0xbf, 0x2a, 0xc4, 0x01, 0xcd, 0xbe, 0x9e, 0x60, 0x05, 0x45,
    0x68, 0x3e, 0xc0, 0xc6, 0xdc, 0xbb, 0x76, 0x42, 0x2d, 0x91, 0x47, 0xc2,
    0x7c, 0x96, 0xaa, 0x5b, 0x28, 0x0d, 0x3b, 0x24, 0x50, 0x12, 0x4e, 0x05,
    0x0f, 0xbd, 0x09, 0xb5, 0x51, 0xf5, 0xd6, 0x12, 0x93, 0xf8, 0x62, 0x69,
    0x92, 0x25, 0x3b, 0x2e, 0x1b, 0xc8, 0x47, 0xfa, 0x06, 0x2c, 0x21, 0x8a,
    0x62, 0xa2, 0x0a, 0x3f, 0x1b, 0xd3, 0x6d, 0x2d, 0x82, 0x4d, 0x7d, 0xde,
    0x80, 0xbb, 0x08, 0x79, 0x53, 0xd8, 0xdb, 0xcd, 0x73, 0xf8, 0x8f, 0x85,
    0xa8, 0xb4, 0xb5, 0x63, 0xf5, 0x6a, 0x9c, 0xd5, 0x58, 0xac, 0xd7, 0x9c,
    0x96, 0x3c, 0xea, 0x5a, 0x66, 0x69, 0xd6, 0x5e, 0xc3, 0xd5, 0x81, 0xfe,
    0x6b, 0x42, 0xa0, 0xb6, 0x58, 0x54, 0xfa, 0x2f, 0xbe, 0x92, 0x9a, 0x41,
    0xf8, 0x32, 0x10, 0x10, 0x86, 0x77, 0x01, 0xd6, 0xef, 0x81, 0x14, 0x1b,
    0xa3, 0x8b, 0xf4, 0x33, 0x94, 0x4e, 0x85, 0xdc, 0x94, 0x7c, 0x01, 0x9e,
    0x6d, 0x9e, 0x7a, 0x26, 0x0f, 0xcf, 0xa0, 0x7d, 0x92, 0xb0, 0x84, 0x95,
    0x76, 0xdf, 0x40, 0x85, 0x87, 0xef, 0xf5, 0x33, 0x52, 0x2e, 0xf6, 0x73,
    0x9e, 0xda, 0x27, 0x6c, 0xee, 0x08, 0xfe, 0x7a, 0x67, 0x43, 0x88, 0xc6,
    0xf0, 0xf1, 0x0c, 0x08, 0xa5, 0x67, 0xc4, 0xb0, 0x01, 0xfb, 0x5b, 0x59,
    0xfc, 0xae, 0xcf, 0x07, 0x14, 0x52, 0x45, 0xd0, 0xeb, 0x97, 0xab, 0x0e,
    0x52, 0x65, 0x12, 0x41, 0xa2, 0x03, 0xb1, 0xe5, 0x1e, 0xa6, 0x33, 0xce,
    0x9a, 0x5c, 0x8f, 0xdf, 0x1a, 0x42, 0x6c, 0xb6, 0xcb, 0x78, 0xdd, 0x02,
    0x78, 0xd9, 0xd8, 0xae, 0x31, 0x5f, 0x13, 0xeb, 0x71, 0xd3, 0xad, 0x6b,
    0xd0, 0x98, 0x18, 0x7a, 0x5b, 0x0d, 0xa9, 0x5d, 0x35, 0xb4, 0x26, 0x5e,
    0xca, 0xf6, 0x7d, 0xc3, 0x9f, 0xe1, 0xbd, 0xfb, 0x73, 0xa6, 0xf1, 0xe0,
    0x26, 0x0f, 0xca, 0x1b, 0xc3, 0xa9, 0xd0, 0x27, 0x46, 0x82, 0x49, 0x73,
    0xed, 0x3a, 0x6b, 0x61, 0x05, 0xd8, 0xa7, 0x7f, 0x2b, 0x36, 0x1c, 0x69,
    0x31, 0x73, 0x29, 0xde, 0x6d, 0x28, 0xb6, 0xea, 0xfe, 0x37, 0xc2, 0xc0,
    0x94, 0x4b, 0x3e, 0xd4, 0x21, 0x3f, 0xe0, 0x02, 0xfa, 0x5f, 0x16, 0xce,
    0x16, 0x66, 0xb5, 0xc9, 0xf6, 0x3f, 0x2a, 0x5c, 0x83, 0x00, 0x5f, 0xda,
    0x0b, 0x6e, 0xe7, 0xb6, 0x55, 0x55, 0xbf, 0x50, 0x29, 0x8e, 0x39, 0xa2,
    0xb7, 0x85, 0x79, 0xc3, 0xce, 0x93, 0x63, 0x57, 0xd1, 0xf2, 0x03, 0x6e,
    0x42, 0x78, 0xb3, 0x5f, 0x72, 0x5c, 0x7d, 0xb0, 0xe0, 0x55, 0x96, 0xe1,
    0x34, 0x0b, 0x5c, 0x75, 0xa9, 0xef, 0x17, 0x47, 0x5b, 0xdc, 0x4f, 0x01,
    0x5d, 0x80, 0x60, 0x14, 0xa7, 0x9a, 0xd3, 0x38, 0x07, 0xda, 0x61, 0x0d,
    0x7f, 0x37, 0xec, 0x0a, 0xd3, 0x54, 0x91, 0x53, 0x16, 0x3d, 0xad, 0xc8,
    0x2c, 0xac, 0xbb, 0x1b, 0x03, 0xd7, 0x96, 0x8d, 0x6d, 0x2d, 0x81, 0xbb,
    0x57, 0xdb, 0x10, 0x3e, 0xfd, 0xcf, 0xa9, 0xa3, 0xa4, 0xaf, 0xf7, 0xa1,
    0x9f, 0x4b, 0xb0, 0x3e, 0x3d, 0x51, 0x15, 0xea, 0x63, 0x8c, 0x75, 0x59,
    0x78, 0xbb, 0xec, 0x94, 0x47, 0xa8, 0x81, 0x7b, 0x98, 0xf1, 0x6e, 0x0a,
    0xad, 0xe6, 0x42, 0x58, 0x37, 0x76, 0x8e, 0x06, 0xa5, 0x43, 0xbf, 0xe9,
    0xfa, 0x53, 0x2e, 0xac, 0x2e, 0xbc, 0x6a, 0x99, 0x08, 0x51, 0x98, 0xac,
    0x3a, 0x23, 0x52, 0x19, 0x07, 0x81, 0x14, 0x4c, 0x62, 0x3f, 0x38, 0x2c,
    0xb6, 0x70, 0x33, 0xd8, 0xb3, 0x97, 0x5b, 0x9c, 0x54, 0x62, 0x63, 0x03,
    0xa8, 0xad, 0xc1, 0xb3, 0xc6, 0x3a, 0x74, 0x5d, 0x6e, 0x9c, 0x3f, 0xce,
    0xe8, 0x79, 0xf4, 0x02, 0x90, 0x3a, 0x2b, 0x51, 0x50, 0x9f, 0x88, 0xf5,
    0x98, 0xa7, 0xc4, 0xd2, 0xe9, 0x4f, 0xca, 0x52, 0xe9, 0xe1, 0xb2, 0xb3,
    0x78, 0xb0, 0xa2, 0x33, 0x9f, 0xa8, 0xc3, 0xd9, 0xb6, 0xdd, 0x6a, 0x05,
    0x5b, 0xc7, 0x52, 0xc3, 0x68, 0xc2, 0x30, 0x7f, 0x74, 0x1e, 0x07, 0x66,
    0xc3, 0xa4, 0x2d, 0x84, 0xaa, 0x17, 0xd6, 0x49, 0x89, 0x60, 0x31, 0x0e,
    0x4e, 0xae, 0xcf, 0x56, 0x93, 0x40, 0x3c, 0x9b, 0xc9, 0x13, 0x5c, 0x5d,
    0x38, 0x94, 0xae, 0x6f, 0xe9, 0x04, 0xb7, 0xa6, 0x1e, 0xab, 0xb9, 0xd4,
    0x04, 0xff, 0x3c, 0x99, 0x2f, 0xe4, 0x80, 0x69, 0xdb, 0x50, 0x5b, 0xac,
    0x34, 0x48, 0x77, 0xf6, 0xb2, 0x00, 0x02, 0x8d, 0x75, 0xa0, 0x2e, 0x93,
    0x0a, 0x20, 0x22, 0x2f, 0x73, 0x15, 0xea, 0xea, 0xbd, 0x8b, 0xe6, 0x85,
    0xfd, 0x30, 0x97, 0xf7, 0x24, 0xb9, 0xd5, 0x30, 0xa0, 0x7a, 0x36, 0x0e,
    0xa4, 0xe1, 0xea, 0xfd, 0xe6, 0xdd, 0xb6, 0xff, 0xc6, 0xfc, 0x20, 0xd6,
    0x71, 0x36, 0x1b, 0x5d, 0xf2, 0x63, 0xca, 0xc6, 0xde, 0x55, 0xf2, 0x39,
    0x05, 0x9a, 0x88, 0x0c, 0x7c, 0xac, 0x5c, 0xc0, 0x50, 0x05, 0x1f, 0xde,
    0xbb, 0x6e, 0xf2, 0x06, 0x77, 0x9a, 0xc7, 0x03, 0x14, 0x26, 0xd4, 0x5d,
    0xe0, 0xa9, 0x02, 0xc6, 0xa1, 0xca, 0x4b, 0x8d, 0xac, 0x0b, 0x54, 0xc1,
    0xf7, 0x91, 0xbf, 0xd0, 0xe6, 0xee, 0x4b, 0x21, 0x82, 0xe2, 0x33, 0xd2,
    0xb4, 0xc0, 0x14, 0x64, 0xc8, 0x6b, 0xd0, 0x50, 0xe5, 0x4a, 0x84, 0xf2,
    0x25, 0x23, 0x2e, 0xd2, 0x65, 0x3b, 0x4a, 0x72, 0x03, 0xaa, 0x0c, 0x82,
    0x98, 0x14, 0x11, 0x8e, 0x4a, 0x1f, 0x77, 0xe1, 0x82, 0xbe, 0xb4, 0x5c,
    0xe4, 0x7a, 0x74, 0xfe, 0x29, 0x42, 0xcd, 0x9d, 0xed, 0x40, 0xc8, 0x88,
    0x85, 0xfa, 0x98, 0xba, 0xbc, 0x45, 0xc7, 0xce, 0x5b, 0xb8, 0x36, 0x1b,
    0x2a, 0x69, 0x55, 0xf5, 0x98, 0xcf, 0xd7, 0x4e, 0x7f, 0x2f, 0x59, 0x4d,
    0x26, 0x33, 0x54, 0xd2, 0x5f, 0x51, 0xfd, 0xd2, 0xb9, 0xbb, 0xe0, 0x99,
    0x5a, 0xdd, 0xd0, 0x11, 0xea, 0xfc, 0x84, 0xd2, 0x21, 0x06, 0x26, 0xea,
    0x2e, 0xba, 0x52, 0xeb, 0x39, 0xf0, 0x97, 0x5b, 0xcc, 0x76, 0xaf, 0x87,
    0x94, 0x9f, 0xb0, 0xa0, 0xd1, 0x59, 0x9f, 0x73, 0x83, 0x69, 0x50, 0x76,
    0x03, 0x0d, 0x44, 0x13, 0xe1, 0x2b, 0x31, 0xf9, 0xf4, 0xe2, 0x6d, 0xcd,
    0x31, 0xbf, 0x39, 0xfc, 0x61, 0xa1, 0x40, 0x52, 0x56, 0xf5, 0xd7, 0x12,
    0x12, 0xab, 0x5e, 0x81, 0x48, 0x8a, 0x2e, 0xe8, 0x85, 0xdc, 0xe7, 0x65,
    0x16, 0x6b, 0xbe, 0x28, 0x93, 0x95, 0x0b, 0x0e, 0x20, 0xfe, 0x4f, 0xf7,
    0xeb, 0x0e, 0xef, 0x2d, 0x41, 0x0c, 0x24, 0x0c, 0x3f, 0xa6, 0x62, 0x61,
    0xb3, 0x7a, 0x5e, 0x9a, 0xef, 0xf6, 0xae, 0x93, 0x51, 0xd5, 0xd7, 0x94,
    0xbb, 0xa0, 0x9e, 0xdf, 0x90, 0x87, 0xbd, 0xd3, 0xf8, 0x4b, 0x86, 0x9e,
    0xff, 0xc5, 0xd7, 0xaf, 0x32, 0x72, 0xc4, 0xb8, 0x4d, 0x4c, 0x3e, 0xcd,
    0xb1, 0xee, 0xae, 0xdd, 0x31, 0x12, 0xc6, 0xd8, 0xc8, 0xfa, 0x20, 0x90,
    0xb1, 0x83, 0xaa, 0x33, 0x1f, 0x07, 0xa6, 0xc1, 0xa1, 0x31, 0x05, 0x3f,
    0xe3, 0xa5, 0x4d, 0x41, 0x8b, 0x1c, 0x5b, 0x51, 0x10, 0x03, 0x0b, 0x0b,
    0x2c, 0x3f, 0x97, 0xc5, 0xa2, 0x38, 0x0d, 0xcb, 0xef, 0x53, 0x45, 0xbe,
    0x28, 0x35, 0x6e, 0x02, 0x10, 0xe8, 0x29, 0x23, 0xe7, 0xe4, 0xc8, 0x90,
    0x99, 0x72, 0xdb, 0xb9, 0x32, 0xb7, 0x87, 0x52, 0x85, 0x7c, 0x92, 0x47,
    0xaf, 0x71, 0x0b, 0x51, 0xc7, 0x8f, 0x24, 0xe4, 0x7c, 0xe2, 0xf3, 0x33,
    0x05, 0xff, 0x92, 0x0c, 0xc0, 0x01, 0xf0, 0xf5, 0xef, 0x52, 0xa1, 0x69,
    0xa3, 0x58, 0xfc, 0x48, 0x58, 0x28, 0xba, 0x53, 0xa2, 0x3d, 0x03, 0x89,
    0x34, 0x12, 0x78, 0xbc, 0x31, 0x5a, 0x93, 0x62, 0x32, 0xe5, 0xe2, 0x10,
    0x56, 0xc0, 0xca, 0xed, 0x1e, 0x2e, 0x6e, 0x23, 0x6b, 0xa1, 0x46, 0x6d,
    0x7d, 0x58, 0x0b, 0xcc, 0xb0, 0x5b, 0x7b, 0xa1, 0x48, 0x89, 0xf6, 0x10,
    0xbe, 0x8c, 0x30, 0x10, 0xe6, 0xe1, 0x87, 0x3c, 0xd0, 0x27, 0x47, 0x06,
    0xaa, 0xb9, 0x73, 0x26, 0xed, 0x7f, 0x38, 0xca, 0xc5, 0xd0, 0x3f, 0x27,
    0xee, 0xa9, 0x5f, 0xf0, 0x31, 0xa9, 0x88, 0x26, 0x7d, 0x32, 0x34, 0x13,
    0xd1, 0xdc, 0x39, 0x7f, 0xac, 0x3b, 0xb6, 0xc6, 0x97, 0x5e, 0x44, 0xca,
    0x68, 0xa4, 0x84, 0x90, 0xd0, 0x90, 0x6f, 0x93, 0xb4, 0xda, 0x61, 0x43,
    0x23, 0xe8, 0x31, 0x34, 0xcd, 0x2f, 0xca, 0xb8, 0x63, 0x5f, 0x59, 0x71,
    0xe2, 0x2a, 0x11, 0x4f, 0xd0, 0x12, 0xf6, 0x70, 0x5f, 0xee, 0xa8, 0xb2,
    0x98, 0x89, 0x6c, 0xed, 0x76, 0xcd, 0xe3, 0x81, 0x1e, 0x5c, 0xec, 0x7d,
    0xb6, 0x21, 0xce, 0x49, 0x25, 0x7f, 0x97, 0x8f, 0x1f, 0xba, 0x81, 0x0b,
    0x9f, 0x58, 0xa9, 0x16, 0x19, 0xa2, 0x56, 0x2f, 0x89, 0x59, 0xa6, 0x8d,
    0x97, 0x7c, 0xf7, 0x02, 0xb3, 0x39, 0x37, 0x4b, 0x79, 0xcf, 0x84, 0x6b,
    0xde, 0x40, 0x6d, 0x1d, 0x17, 0x5e, 0x48, 0x62, 0x6c, 0x07, 0xe8, 0xbc,
    0x0e, 0xd6, 0xd4, 0xec, 0xa7, 0x55, 0x16, 0x57, 0x05, 0xc2, 0x5b, 0xa8,
    0xbb, 0xa1, 0x8b, 0x63, 0x04, 0x63, 0x99, 0xc2, 0x11, 0x29, 0xcc, 0x37,
    0xc7, 0xb1, 0x70, 0x34, 0x27, 0x1f, 0xd7, 0xb4, 0x4d, 0xd6, 0xf3, 0x4a,
    0x92, 0xf6, 0xa6, 0xaa, 0x8f, 0xc6, 0xc8, 0x6b, 0xe1, 0xfc, 0x97, 0xaf,
    0xc9, 0x6f, 0x68, 0xf1, 0x41, 0xa1, 0xb2, 0x8c, 0x9e, 0xe2, 0xd7, 0x64,
    0xa3, 0xd9, 0x47, 0xbd, 0xba, 0xfc, 0x84, 0x14, 0x6c, 0x4b, 0xcc, 0xe6,
    0xb4, 0x93, 0x54, 0x03, 0xa6, 0x14, 0xd5, 0xb1, 0xfb, 0xbc, 0x8e, 0x96,
    0x88, 0xb4, 0xd6, 0x86, 0xe5, 0xee, 0x5b, 0x1a, 0xa1, 0x7c, 0x43, 0x46,
    0x6b, 0x80, 0x2c, 0xad, 0xde, 0x4b, 0xa9, 0x75, 0xa1, 0x42, 0xc7, 0x0b,
    0x72, 0xcf, 0x10, 0xa7, 0x5a, 0xd8, 0x96, 0x0d, 0xc1, 0xb2, 0xab, 0x5d,
    0xd2, 0xa5, 0x5e, 0xe6, 0x72, 0xe6, 0x46, 0x72, 0x85, 0xaf, 0xc3, 0xd0,
    0x09, 0x81, 0xf8, 0x9f, 0xa2, 0xd6, 0x40, 0xb4, 0x50, 0x0d, 0x35, 0x6a,
    0xeb, 0x9f, 0xc2, 0xf1, 0x41, 0x5a, 0x2b, 0x87, 0xe5, 0x1d, 0xa3, 0x36,
    0x6f, 0x5a, 0x95, 0x49, 0xd5, 0x67, 0x1e, 0xd5, 0x6d, 0xcf, 0x8a, 0xf7,
    0xf9, 0x53, 0x9c, 0xcc, 0x7e, 0xd1, 0x3a, 0xbd, 0xb9, 0x36, 0xeb, 0x82,
    0xc5, 0xb5, 0xac, 0xe7, 0x60, 0x4b, 0x1f, 0x96, 0xf8, 0xf5, 0x6f, 0x4d,
    0x11, 0x76, 0xaf, 0xd0, 0x79, 0x86, 0xd7, 0xcf, 0x49, 0x61, 0x93, 0x2e,
    0xda, 0xfd, 0x64, 0x4e, 0xb3, 0x27, 0xfa, 0x38, 0x91, 0x27, 0xa0, 0x94,
    0x35, 0xd6, 0x8e, 0x03, 0x6e, 0xb8, 0x67, 0x2b, 0x35, 0x8e, 0x56, 0x49,
    0x52, 0xf1, 0x82, 0x9a, 0x6a, 0x9c, 0xe4, 0x4e, 0x56, 0xd1, 0x2e, 0xcb,
    0xd4, 0x1e, 0x27, 0x69, 0xa9, 0xa4, 0xfe, 0xb4, 0x33, 0x0c, 0xcd, 0x7a,
    0xa9, 0xc2, 0x60, 0x64, 0xb2, 0x4a, 0x34, 0xbf, 0x8f, 0x67, 0x9a, 0x82,
    0x1e, 0x03, 0x0b, 0x85, 0x6f, 0xa6, 0xa0, 0x2a, 0x7b, 0x27, 0xc6, 0xea,
    0x67, 0x8c, 0xd9, 0xb4, 0x57, 0xab, 0xba, 0x97, 0x6e, 0x51, 0x94, 0x20,
    0x00, 0x8b, 0x0c, 0x2b, 0xf9, 0x9d, 0xea, 0xb6, 0xac, 0x56, 0x6f, 0xef,
    0x29, 0x6d, 0x15, 0xc5, 0x30, 0xab, 0x10, 0xa7, 0x2f, 0x82, 0x95, 0xac,
    0x78, 0xe1, 0x91, 0x61, 0xe4, 0xa0, 0x40, 0x98, 0x84, 0x91, 0xdf, 0xd9,
    0x6c, 0x78, 0x2f, 0xc3, 0x37, 0x81, 0xc7, 0x4e, 0xe0, 0x74, 0x0a, 0xd9,
    0xbb, 0xbe, 0x30, 0x2c, 0x18, 0x14, 0x88, 0xaf, 0xdb, 0xd1, 0x56, 0x3c,
    0x3d, 0x8e, 0xa7, 0xd8, 0x4e, 0x4c, 0xfe, 0x29, 0x59, 0x44, 0x0b, 0x39,
    0x90, 0x3c, 0x7a, 0xe2, 0xde, 0x74, 0xb3, 0xad, 0xfc, 0xbd, 0x69, 0xad,
    0x23, 0x85, 0x61, 0x49, 0x36, 0x88, 0x26, 0xed, 0x3a, 0x5a, 0xb0, 0x68,
    0x11, 0x34, 0x5a, 0x91, 0xd1, 0x6f, 0x88, 0xde, 0x9a, 0x11, 0x43, 0xcd,
    0x26, 0xdb, 0x14, 0x80, 0x17, 0xf7, 0xfd, 0x78, 0x87, 0xa5, 0x14, 0x5a,
    0x59, 0x2c, 0x45, 0x54, 0xe7, 0x6d, 0x30, 0x77, 0x06, 0x2f, 0xf9, 0xd4,
    0xdb, 0xcc, 0xde, 0x60, 0x82, 0x45, 0x14, 0x19, 0xb7, 0x38, 0x22, 0xda,
    0xc1, 0x6e, 0x51, 0xa4, 0xd6, 0xd1, 0xbf, 0x6f, 0xa0, 0x47, 0x95, 0x0c,
    0x76, 0x50, 0xba, 0x07, 0xf4, 0xc4, 0x4a, 0xd0, 0x43, 0x86, 0x3e, 0x40,
    0x1b, 0xfb, 0xcb, 0xb4, 0x56, 0x83, 0xd5, 0xd1, 0xee, 0x24, 0xb7, 0xb5,
    0x40, 0x78, 0x0b, 0xa0, 0x9e, 0x4b, 0x83, 0xbe, 0x0c, 0x26, 0x76, 0xbf,
    0x5e, 0xf2, 0xe9, 0x1b, 0x1d, 0x29, 0x70, 0x62, 0x38, 0x97, 0xd2, 0xc5,
    0x37, 0xfa, 0xe8, 0xf1, 0xe6, 0x79, 0xc2, 0x20, 0x1c, 0xc7, 0x27, 0x65,
    0xa6, 0xd1, 0x7f, 0xe7, 0x7b, 0x28, 0x84, 0x0a, 0x29, 0x28, 0x6b, 0xd2,
    0xdd, 0x1b, 0xf9, 0x12, 0xde, 0xfc, 0xa9, 0x09, 0x61, 0xd9, 0xf6, 0x63,
    0x7c, 0x8a, 0x00, 0xd8, 0x73, 0xa7, 0x98, 0xd8, 0xb2, 0xb4, 0x80, 0x98,
    0xca, 0x7d, 0x40, 0xc6, 0x4e, 0x51, 0x41, 0x84, 0x25, 0x57, 0xd6, 0xed,
    0x67, 0x4a, 0x76, 0xc3, 0xfa, 0x4d, 0x37, 0xa5, 0x4e, 0xcf, 0x19, 0x38,
    0xda, 0x34, 0x02, 0x5d, 0xd9, 0x66, 0xa9, 0xdf, 0x78, 0xe1, 0x4f, 0xd7,
    0xb1, 0x37, 0xc6, 0x60, 0x94, 0x09, 0xd3, 0xe6, 0xc4, 0xda, 0x03, 0x84,
    0xe6, 0x6f, 0xab, 0x26, 0xba, 0xdf, 0xf4, 0x23, 0xd2, 0x00, 0xce, 0x85,
    0x6e, 0x8c, 0xe9, 0x17, 0xb2, 0x8d, 0x81, 0x32, 0x01, 0xcc, 0x21, 0xe9,
    0x47, 0x43, 0x6f, 0x47, 0xf4, 0x5b, 0x6f, 0x2b, 0x31, 0xe5, 0x4a, 0x0b,
    0xf2, 0x77, 0x28, 0x0a, 0xac, 0xaf, 0x7a, 0xf0, 0xf0, 0x33, 0x59, 0xbb,
    0xcc, 0xb9, 0xf8, 0x08, 0x94, 0x3d, 0x25, 0x4e, 0x0c, 0x92, 0x68, 0xde,
    0x7f, 0x82, 0x15, 0xbb, 0x42, 0xf2, 0x94, 0x8c, 0xe0, 0x72, 0x90, 0x9e,
    0x58, 0xf0, 0x6e, 0x29, 0x1e, 0xad, 0xfc, 0x35, 0x87, 0xd9, 0x88, 0xdd,
    0xa6, 0xfe, 0xaf, 0x3b, 0x2e, 0xd9, 0x0c, 0x9b, 0x0a, 0x91, 0xe6, 0x0e,
    0xd4, 0xd0, 0xff, 0xd9, 0x64, 0x7e, 0xcf, 0xf9, 0x7d, 0x10, 0x10, 0x16,
    0x5d, 0xe2, 0x9f, 0xd4, 0x54, 0x47, 0xc1, 0xcf, 0xf1, 0x6d, 0xed, 0x42,
    0xcb, 0x1b, 0xa1, 0x87, 0xed, 0xb1, 0x15, 0xd2,
};

TEST(Dilithium3Test, KeyGeneration) {
  // Basic key generation tests for Dilithium3
  // Generate a Dilithium3 key
  EVP_PKEY_CTX *dilithium_pkey_ctx = EVP_PKEY_CTX_new_id(EVP_PKEY_DILITHIUM3, nullptr);
  ASSERT_NE(dilithium_pkey_ctx, nullptr);

  EVP_PKEY *dilithium_pkey = EVP_PKEY_new();
  ASSERT_NE(dilithium_pkey, nullptr);

  EXPECT_TRUE(EVP_PKEY_keygen_init(dilithium_pkey_ctx));
  EXPECT_TRUE(EVP_PKEY_keygen(dilithium_pkey_ctx, &dilithium_pkey));
  ASSERT_NE(dilithium_pkey->pkey.ptr, nullptr);

  const DILITHIUM3_KEY *dilithium3Key = (DILITHIUM3_KEY *)(dilithium_pkey->pkey.ptr);
  EXPECT_TRUE(dilithium3Key->has_private);

  // Extract public key and check it is of the correct size
  uint8_t *buf = nullptr;
  size_t buf_size;
  EXPECT_TRUE(EVP_PKEY_get_raw_public_key(dilithium_pkey, buf, &buf_size));
  EXPECT_EQ((size_t)DILITHIUM3_PUBLIC_KEY_BYTES, buf_size);

  buf = (uint8_t *)OPENSSL_malloc(buf_size);
  ASSERT_NE(buf, nullptr);
  EXPECT_TRUE(EVP_PKEY_get_raw_public_key(dilithium_pkey, buf, &buf_size));

  buf_size = 0;
  EXPECT_FALSE(EVP_PKEY_get_raw_public_key(dilithium_pkey, buf, &buf_size));

  uint32_t err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_BUFFER_TOO_SMALL, ERR_GET_REASON(err));
  OPENSSL_free(buf);
  buf = nullptr;

  // Extract private key and check it is of the correct size
  EXPECT_TRUE(EVP_PKEY_get_raw_private_key(dilithium_pkey, buf, &buf_size));
  EXPECT_EQ((size_t)DILITHIUM3_PRIVATE_KEY_BYTES, buf_size);

  buf = (uint8_t *)OPENSSL_malloc(buf_size);
  ASSERT_NE(buf, nullptr);
  EXPECT_TRUE(EVP_PKEY_get_raw_private_key(dilithium_pkey, buf, &buf_size));

  buf_size = 0;
  EXPECT_FALSE(EVP_PKEY_get_raw_private_key(dilithium_pkey, buf, &buf_size));
  err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_BUFFER_TOO_SMALL, ERR_GET_REASON(err));
  OPENSSL_free(buf);

  EVP_PKEY_CTX_free(dilithium_pkey_ctx);
  EVP_PKEY_free(dilithium_pkey);
}

TEST(Dilithium3Test, KeyComparison) {
  // Generate two Dilithium3 keys are check that they are not equal.
  EVP_PKEY_CTX *dilithium_pkey_ctx1 = EVP_PKEY_CTX_new_id(EVP_PKEY_DILITHIUM3, nullptr);
  ASSERT_NE(dilithium_pkey_ctx1, nullptr);

  EVP_PKEY *dilithium_pkey1 = EVP_PKEY_new();
  ASSERT_NE(dilithium_pkey1, nullptr);

  EXPECT_TRUE(EVP_PKEY_keygen_init(dilithium_pkey_ctx1));
  EXPECT_TRUE(EVP_PKEY_keygen(dilithium_pkey_ctx1, &dilithium_pkey1));
  ASSERT_NE(dilithium_pkey1->pkey.ptr, nullptr);

  EVP_PKEY_CTX *dilithium_pkey_ctx2 = EVP_PKEY_CTX_new_id(EVP_PKEY_DILITHIUM3, nullptr);
  ASSERT_NE(dilithium_pkey_ctx2, nullptr);

  EVP_PKEY *dilithium_pkey2 = EVP_PKEY_new();
  ASSERT_NE(dilithium_pkey2, nullptr);

  EXPECT_TRUE(EVP_PKEY_keygen_init(dilithium_pkey_ctx2));
  EXPECT_TRUE(EVP_PKEY_keygen(dilithium_pkey_ctx2, &dilithium_pkey2));
  ASSERT_NE(dilithium_pkey2->pkey.ptr, nullptr);

  EXPECT_EQ(0, EVP_PKEY_cmp(dilithium_pkey1, dilithium_pkey2));

  EVP_PKEY_free(dilithium_pkey1);
  EVP_PKEY_free(dilithium_pkey2);
  EVP_PKEY_CTX_free(dilithium_pkey_ctx1);
  EVP_PKEY_CTX_free(dilithium_pkey_ctx2);
}

TEST(Dilithium3Test, NewKeyFromBytes) {
  // Test the generation of a Dilithium3 key from bytes
  // Source key
  EVP_PKEY_CTX *dilithium_pkey_ctx = EVP_PKEY_CTX_new_id(EVP_PKEY_DILITHIUM3, nullptr);
  ASSERT_NE(dilithium_pkey_ctx, nullptr);

  EVP_PKEY *dilithium_pkey = EVP_PKEY_new();
  ASSERT_NE(dilithium_pkey, nullptr);

  EXPECT_TRUE(EVP_PKEY_keygen_init(dilithium_pkey_ctx));
  EXPECT_TRUE(EVP_PKEY_keygen(dilithium_pkey_ctx, &dilithium_pkey));
  ASSERT_NE(dilithium_pkey->pkey.ptr, nullptr);
  const DILITHIUM3_KEY *dilithium3Key = (DILITHIUM3_KEY *)(dilithium_pkey->pkey.ptr);

  // New raw public key
  EVP_PKEY *new_public = EVP_PKEY_new_raw_public_key(EVP_PKEY_DILITHIUM3,
                                                     NULL,
                                                     dilithium3Key->pub,
                                                     DILITHIUM3_PUBLIC_KEY_BYTES);
  ASSERT_NE(new_public, nullptr);

  uint8_t *buf = nullptr;
  size_t buf_size;
  EXPECT_FALSE(EVP_PKEY_get_raw_private_key(new_public, buf, &buf_size));
  uint32_t err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_NOT_A_PRIVATE_KEY, ERR_GET_REASON(err));

  // EVP_PKEY_cmp just compares the public keys so this should return 1
  EXPECT_EQ(1, EVP_PKEY_cmp(dilithium_pkey, new_public));

  // New raw private key
  EVP_PKEY *new_private = EVP_PKEY_new_raw_private_key(EVP_PKEY_DILITHIUM3,
                                                       NULL,
                                                       dilithium3Key->priv,
                                                       DILITHIUM3_PRIVATE_KEY_BYTES);
  ASSERT_NE(new_private, nullptr);
  const DILITHIUM3_KEY *newDilithium3Key = (DILITHIUM3_KEY *)(new_private->pkey.ptr);
  EXPECT_EQ(0, OPENSSL_memcmp(dilithium3Key->priv, newDilithium3Key->priv,
                              DILITHIUM3_PRIVATE_KEY_BYTES));

  EVP_PKEY_CTX_free(dilithium_pkey_ctx);
  EVP_PKEY_free(new_public);
  EVP_PKEY_free(new_private);
  EVP_PKEY_free(dilithium_pkey);
}

TEST(Dilithium3Test, KeySize) {
  // Test the key size of Dilithium3 key is as expected
  EVP_PKEY_CTX *dilithium_pkey_ctx = EVP_PKEY_CTX_new_id(EVP_PKEY_DILITHIUM3, nullptr);
  ASSERT_NE(dilithium_pkey_ctx, nullptr);

  EVP_PKEY *dilithium_pkey = EVP_PKEY_new();
  ASSERT_NE(dilithium_pkey, nullptr);

  EXPECT_TRUE(EVP_PKEY_keygen_init(dilithium_pkey_ctx));
  EXPECT_TRUE(EVP_PKEY_keygen(dilithium_pkey_ctx, &dilithium_pkey));

  EXPECT_EQ(DILITHIUM3_SIGNATURE_BYTES, EVP_PKEY_size(dilithium_pkey));
  EXPECT_EQ(8*(DILITHIUM3_PUBLIC_KEY_BYTES), EVP_PKEY_bits(dilithium_pkey));

  EVP_PKEY_CTX_free(dilithium_pkey_ctx);
  EVP_PKEY_free(dilithium_pkey);
}

TEST(Dilithium3Test, Encoding) {
  // Test Dilithium keypairs are extractable, and encode/parse correctly.
  //generate dilithium key
  EVP_PKEY_CTX *dilithium_pkey_ctx = EVP_PKEY_CTX_new_id(EVP_PKEY_DILITHIUM3, nullptr);
  EVP_PKEY *dilithium_pkey = EVP_PKEY_new();
  EVP_PKEY_keygen_init(dilithium_pkey_ctx);
  EVP_PKEY_keygen(dilithium_pkey_ctx, &dilithium_pkey);
  const DILITHIUM3_KEY *dilithium3Key = (DILITHIUM3_KEY *)(dilithium_pkey->pkey.ptr);

  //Create a public key.
  bssl::UniquePtr<EVP_PKEY> pubkey(EVP_PKEY_new_raw_public_key(EVP_PKEY_DILITHIUM3,
                                                               NULL,
                                                               dilithium3Key->pub,
                                                               DILITHIUM3_PUBLIC_KEY_BYTES));
  ASSERT_TRUE(pubkey);
  EXPECT_EQ(EVP_PKEY_DILITHIUM3, EVP_PKEY_id(pubkey.get()));

  // The public key must be extractable.
  uint8_t pub_buf[1952];
  size_t pub_len;
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(pubkey.get(), nullptr, &pub_len));
  EXPECT_EQ(pub_len, 1952u);
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(pubkey.get(), pub_buf, &pub_len));

  // The public key must encode properly.
  bssl::ScopedCBB cbb;
  uint8_t *der;
  size_t der_len;
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_public_key(cbb.get(), pubkey.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  bssl::UniquePtr<uint8_t> free_der(der);

  // The public key must parse properly.
  CBS cbs;
  CBS_init(&cbs, der, der_len);
  EVP_PKEY *dilithium_pkey_from_der = EVP_parse_public_key(&cbs);
  ASSERT_TRUE(dilithium_pkey_from_der);
  EXPECT_EQ(1, EVP_PKEY_cmp(dilithium_pkey, dilithium_pkey_from_der));

  // Create a private key.
  bssl::UniquePtr<EVP_PKEY> privkey(EVP_PKEY_new_raw_private_key(EVP_PKEY_DILITHIUM3,
                                                                 NULL,
                                                                 dilithium3Key->priv,
                                                                 DILITHIUM3_PRIVATE_KEY_BYTES));
  ASSERT_TRUE(privkey);
  EXPECT_EQ(EVP_PKEY_DILITHIUM3, EVP_PKEY_id(privkey.get()));

  // The private key must be extractable.
  uint8_t priv_buf[4000];
  size_t priv_len;
  ASSERT_TRUE(EVP_PKEY_get_raw_private_key(privkey.get(), nullptr, &priv_len));
  EXPECT_EQ(priv_len, 4000u);
  ASSERT_TRUE(EVP_PKEY_get_raw_private_key(privkey.get(), priv_buf, &priv_len));

  // The private key must encode properly.
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_private_key(cbb.get(), privkey.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  free_der.reset(der);

  // The private key must parse properly.
  CBS_init(&cbs, der, der_len);
  EVP_PKEY *dilithium_priv_from_der = EVP_parse_private_key(&cbs);
  ASSERT_TRUE(dilithium_priv_from_der);
  const DILITHIUM3_KEY *dilithium3Key_from_der = (DILITHIUM3_KEY *)(dilithium_priv_from_der->pkey.ptr);
  // The private key dilithium3Key_from_der must be equal to the original key
  EXPECT_EQ(Bytes(dilithium3Key->priv, DILITHIUM3_PRIVATE_KEY_BYTES),
            Bytes(dilithium3Key_from_der->priv, DILITHIUM3_PRIVATE_KEY_BYTES));

  EVP_PKEY_CTX_free(dilithium_pkey_ctx);
  EVP_PKEY_free(dilithium_pkey);
  EVP_PKEY_free(dilithium_pkey_from_der);
  EVP_PKEY_free(dilithium_priv_from_der);
}

TEST(Dilithium3Test, Decoding) {
  // generate a Dilithium3 public key based on the public key bytes
  bssl::UniquePtr<EVP_PKEY> pubkey(EVP_PKEY_new_raw_public_key(
      EVP_PKEY_DILITHIUM3, nullptr, kPublicKey, sizeof(kPublicKey)));
  ASSERT_TRUE(pubkey);
  EXPECT_EQ(EVP_PKEY_DILITHIUM3, EVP_PKEY_id(pubkey.get()));

  // encode the public key as DER
  bssl::ScopedCBB cbb;
  uint8_t *der;
  size_t der_len;
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_public_key(cbb.get(), pubkey.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  bssl::UniquePtr<uint8_t> free_der(der);

  // test that the encoded public key encodes as expected
  EXPECT_EQ(Bytes(kPublicKeySPKI), Bytes(der, der_len));

  // we now decode the DER structure, then parse as a PKEY.
  CBS cbs;
  CBS_init(&cbs, der, der_len);
  EVP_PKEY *dilithium_pkey_from_der = EVP_parse_public_key(&cbs);
  ASSERT_TRUE(dilithium_pkey_from_der);

  // extract the public key and check it is equivalent to original key
  uint8_t pub_buf[1952];
  size_t pub_len;
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(dilithium_pkey_from_der, nullptr, &pub_len));
  EXPECT_EQ(pub_len, 1952u);
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(dilithium_pkey_from_der, pub_buf, &pub_len));
  EXPECT_EQ(Bytes(kPublicKey), Bytes(pub_buf, pub_len));

  EVP_PKEY_free(dilithium_pkey_from_der);
}

TEST(Dilithium3Test, SIGOperations) {
  // Test basic functionality for Dilithium3
  // Generate a Dilithium3 key
  EVP_PKEY_CTX *dilithium_pkey_ctx = EVP_PKEY_CTX_new_id(EVP_PKEY_DILITHIUM3, nullptr);
  ASSERT_NE(dilithium_pkey_ctx, nullptr);

  EVP_PKEY *dilithium_pkey = EVP_PKEY_new();
  ASSERT_NE(dilithium_pkey, nullptr);

  EXPECT_TRUE(EVP_PKEY_keygen_init(dilithium_pkey_ctx));
  EXPECT_TRUE(EVP_PKEY_keygen(dilithium_pkey_ctx, &dilithium_pkey));

  // sign a message
  bssl::ScopedEVP_MD_CTX md_ctx;
  uint8_t signature[DILITHIUM3_SIGNATURE_BYTES];
  size_t signature_len = DILITHIUM3_SIGNATURE_BYTES;
  std::vector<uint8_t> msg = {
      0x4a, 0x41, 0x4b, 0x45, 0x20, 0x4d, 0x41, 0x53, 0x53, 0x49,
      0x4d, 0x4f, 0x20, 0x41, 0x57, 0x53, 0x32, 0x30, 0x32, 0x32, 0x2e};
  std::vector<uint8_t> badmsg = {
      0x4a, 0x41, 0x4b, 0x45, 0x20, 0x4d, 0x41, 0x53, 0x53, 0x49,
      0x4d, 0x4f, 0x20, 0x41, 0x57, 0x53, 0x32, 0x30, 0x32, 0x31, 0x2e};

  ASSERT_TRUE(EVP_DigestSignInit(md_ctx.get(), NULL, NULL, NULL, dilithium_pkey));
  ASSERT_TRUE(EVP_DigestSign(md_ctx.get(), signature, &signature_len,
                             msg.data(), msg.size()));

  // verify the correct signed message
  ASSERT_TRUE(EVP_DigestVerify(md_ctx.get(), signature, signature_len,
                               msg.data(), msg.size()));

  // verify the signed message fails upon a bad message
  ASSERT_FALSE(EVP_DigestVerify(md_ctx.get(), signature, signature_len,
                                badmsg.data(), badmsg.size()));

  // sign the bad message
  uint8_t signature1[DILITHIUM3_SIGNATURE_BYTES];
  ASSERT_TRUE(EVP_DigestSign(md_ctx.get(), signature1, &signature_len,
                             badmsg.data(), badmsg.size()));

  // check that the two signatures are not equal
  EXPECT_NE(0, OPENSSL_memcmp(signature, signature1, signature_len));

  // verify the signed message fails upon a bad signature
  ASSERT_FALSE(EVP_DigestVerify(md_ctx.get(), signature1, signature_len,
                                msg.data(), msg.size()));

  EVP_PKEY_free(dilithium_pkey);
  EVP_PKEY_CTX_free(dilithium_pkey_ctx);
  md_ctx.Reset();
}

#else

TEST(Dilithium3Test, EvpDisabled) {
  ASSERT_EQ(nullptr, EVP_PKEY_CTX_new_id(NID_DILITHIUM3_R3, nullptr));

  EVP_PKEY *pkey = EVP_PKEY_new();
  ASSERT_FALSE(EVP_PKEY_set_type(pkey, NID_DILITHIUM3_R3));
  EVP_PKEY_free(pkey);
}

#endif
