#ifndef URLINFO_H
#define URLINFO_H

#include <filesystem>
#include <optional>
#include <string>
#include <vector>
#include <boost/json.hpp>
#include "media.h"
#include "videocodec.h"

namespace Nickvision::TubeConverter::Shared::Models
{
    class UrlInfo
    {
    public:
        /**
         * @brief Constructs a UrlInfo.
         * @param url The Url
         * @param info The media information json object from yt-dlp
         * @param includeAutoGeneratedSubtitles Whether or not to include auto generated subtitles
         * @param preferredVideoCodec The preferred video codec
         */
        UrlInfo(const std::string& url, boost::json::object info, bool includeAutoGeneratedSubtitles, VideoCodec preferredVideoCodec);
        /**
         * @brief Constructs a UrlInfo.
         * @brief This method constructs a single playlist UrlInfo from a batch file.
         * @param batchFile The path to the batch file
         * @param urlInfos The UrlInfos of the batch file
         */
        UrlInfo(const std::filesystem::path& batchFile, const std::vector<std::optional<UrlInfo>>& urlInfos);
        /**
         * @brief Gets the title.
         * @return The title
         */
        const std::string& getTitle() const;
        /**
         * @brief Gets the Url.
         * @return The Url
         */
        const std::string& getUrl() const;
        /**
         * @brief Gets whether the Url is a playlist.
         * @return True if playlist, false otherwise
         */
        bool isPlaylist() const;
        /**
         * @brief Gets the number of media belonging to the Url.
         * @return The number of media
         */
        size_t count() const;
        /**
         * @brief Gets the media at the specified index.
         * @param index The index
         */
        Media& get(size_t index);
        /**
         * @brief Gets the media at the specified index.
         * @param index The index
         */
        const Media& get(size_t index) const;
        /**
         * @brief Gets the media at the specified index.
         * @param index The index
         */
        Media& operator[](size_t index);
        /**
         * @brief Gets the media at the specified index.
         * @param index The index
         */
        const Media& operator[](size_t index) const;

    private:
        std::string m_url;
        std::string m_title;
        bool m_isPlaylist;
        std::vector<Media> m_media;
    };
}

#endif //URLINFO_H