!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

! ---------------------------------------------------------
!> Orthonormalizes nst orbitals in mesh (honours state parallelization).
subroutine X(states_orthogonalization_full)(st, mesh, ik)
  type(states_t),         intent(inout) :: st
  type(mesh_t),           intent(in)    :: mesh
  integer,                intent(in)    :: ik

  R_TYPE, allocatable :: ss(:, :)
  type(profile_t), save :: prof
  integer :: nst

#ifdef HAVE_SCALAPACK
!pgi$r opt=0
!This is a pragma for the PGI compiler, forcing optimization -O0 for this subroutine
!With PGI 10.9 and ScaLAPACK, at -O2 and higher optimization levels, the test finite_systems_3d/10-fullerene fails in
!states_orthogonalization_full.par_gs with error message
!glibc detected *** octopus_mpi: malloc(): memory corruption
#endif

  call profiling_in(prof, "ORTHOGONALIZATION_FULL")
  PUSH_SUB(X(states_orthogonalization_full))

  nst = st%nst

  select case(st%d%orth_method)
  case(OPTION__STATESORTHOGONALIZATION__CHOLESKY_SERIAL)
    call cholesky_serial()

  case(OPTION__STATESORTHOGONALIZATION__CHOLESKY_PARALLEL)
    call cholesky_parallel()

  case(OPTION__STATESORTHOGONALIZATION__MGS)
    call mgs()

  case default
    write(message(1),'(a,i6)') "Internal error from states_orthogonalization_full: orth_method has illegal value ", &
      st%d%orth_method
    call messages_fatal(1)
  end select

  call profiling_out(prof)
  POP_SUB(X(states_orthogonalization_full))

contains
  
  subroutine cholesky_serial()

    integer :: ierr
    logical :: bof

    PUSH_SUB(X(states_orthogonalization_full).cholesky_serial)

    SAFE_ALLOCATE(ss(1:nst, 1:nst))

    ss = M_ZERO

    call X(states_calc_overlap)(st, mesh, ik, ss)

    bof = .false.
    ! calculate the Cholesky decomposition
    call lalg_cholesky(nst, ss, bof = bof, err_code = ierr)

    if(bof) then
      write(message(1),'(a,i6)') "The cholesky_serial orthogonalization failed with error code ", ierr
      message(2) = "There may be a linear dependence, a zero vector, or maybe a library problem."
      call messages_warning(2)
    end if

    call X(states_trsm)(st, mesh, ik, ss)

    SAFE_DEALLOCATE_A(ss)

    POP_SUB(X(states_orthogonalization_full).cholesky_serial)
  end subroutine cholesky_serial


  ! -----------------------------------------------------------------------------------------------
  subroutine cholesky_parallel()

    R_TYPE, allocatable :: psi(:, :, :)
    integer             :: psi_block(1:2), total_np
#ifdef HAVE_SCALAPACK
    integer             :: info, nbl, nrow, ncol
    integer             :: psi_desc(BLACS_DLEN), ss_desc(BLACS_DLEN)
    type(profile_t), save :: prof_cholesky, prof_trsm, prof_herk
#endif

    PUSH_SUB(X(states_orthogonalization_full).cholesky_parallel)

! some checks
#ifndef HAVE_MPI
    message(1) = 'The cholesky_parallel orthogonalizer can only be used in parallel.'
    call messages_fatal(1)
#else
#ifndef HAVE_SCALAPACK
    message(1) = 'The cholesky_parallel orthogonalizer requires ScaLAPACK.'
    call messages_fatal(1, only_root_writes = .true.)
#endif
    if(st%dom_st_mpi_grp%size == 1) then
      message(1) = 'The cholesky_parallel orthogonalizer is designed to be used with domain or state parallelization.'
      call messages_warning(1)
    end if
#endif

    call states_parallel_blacs_blocksize(st, mesh, psi_block, total_np)

    SAFE_ALLOCATE(psi(1:mesh%np_part, 1:st%d%dim, st%st_start:st%st_end))

    call states_get_state(st, mesh, ik, psi)
    
    ! We need to set to zero some extra parts of the array
    if(st%d%dim == 1) then
     psi(mesh%np + 1:psi_block(1), 1:st%d%dim, st%st_start:st%st_end) = M_ZERO
    else
     psi(mesh%np + 1:mesh%np_part, 1:st%d%dim, st%st_start:st%st_end) = M_ZERO
    end if

#ifdef HAVE_SCALAPACK
    
    call descinit(psi_desc(1), total_np, st%nst, psi_block(1), psi_block(2), 0, 0, st%dom_st_proc_grid%context, &
      st%d%dim*ubound(psi, dim = 1), info)

    if(info /= 0) then
      write(message(1),'(3a,i6)') "descinit for psi failed in ", TOSTRING(X(states_orthogonalization_full)), &
        ".cholesky_parallel with error ", info
      call messages_fatal(1)
    end if

    nbl = min(32, st%nst)
    nrow = max(1, numroc(st%nst, nbl, st%dom_st_proc_grid%myrow, 0, st%dom_st_proc_grid%nprow))
    ncol = max(1, numroc(st%nst, nbl, st%dom_st_proc_grid%mycol, 0, st%dom_st_proc_grid%npcol))

    SAFE_ALLOCATE(ss(1:nrow, 1:ncol))

    call descinit(ss_desc(1), st%nst, st%nst, nbl, nbl, 0, 0, st%dom_st_proc_grid%context, ubound(ss, dim = 1), info)

    if(info /= 0) then
      write(message(1),'(3a,i6)') "descinit for ss failed in ", TOSTRING(X(states_orthogonalization_full)), &
        ".cholesky_parallel with error ", info
      call messages_fatal(1)
    end if

    ss = M_ZERO

    call profiling_in(prof_herk, "SCALAPACK_HERK")
    call pblas_herk(uplo = 'U', trans = 'C', n = st%nst, k = total_np, &
      alpha = R_TOTYPE(mesh%vol_pp(1)), a = psi(1, 1, st%st_start), ia = 1, ja = 1, desca = psi_desc(1), &
      beta = R_TOTYPE(M_ZERO), c = ss(1, 1), ic = 1, jc = 1, descc = ss_desc(1))
    call profiling_count_operations(dble(mesh%np)*dble(nst)**2*(R_ADD + R_MUL))
    call profiling_out(prof_herk)

    call profiling_in(prof_cholesky, "SCALAPACK_CHOLESKY")
    ! calculate the Cholesky decomposition
    call scalapack_potrf(uplo = 'U', n = st%nst, a = ss(1, 1), ia = 1, ja = 1, desca = ss_desc(1), info = info)
    call profiling_out(prof_cholesky)

    if(info /= 0) then
      write(message(1),'(3a,i6)') "cholesky_parallel orthogonalization with ", TOSTRING(pX(potrf)), &
        " failed with error ", info
      call messages_fatal(1)
    end if

    call profiling_in(prof_trsm, "SCALAPACK_TRSM")
    call pblas_trsm(side = 'R', uplo = 'U', transa = 'N', diag = 'N', m = total_np, n = st%nst, &
      alpha = R_TOTYPE(M_ONE), a = ss(1, 1), ia = 1, ja = 1, desca = ss_desc(1), &
      b = psi(1, 1, st%st_start), ib = 1, jb = 1, descb = psi_desc(1))
    call profiling_out(prof_trsm)
#endif

    SAFE_DEALLOCATE_A(ss)

    call states_set_state(st, mesh, ik, psi)
    
    POP_SUB(X(states_orthogonalization_full).cholesky_parallel)
  end subroutine cholesky_parallel

  ! ----------------------------------------------------------------------------------

  subroutine mgs()

    integer :: ist, jst, idim
    FLOAT   :: cc
    R_TYPE, allocatable :: aa(:), psii(:, :), psij(:, :)
    
    PUSH_SUB(X(states_orthogonalization_full).mgs)

    if(st%parallel_in_states) then
      message(1) = 'The mgs orthogonalization method cannot work with state-parallelization.'
      call messages_fatal(1, only_root_writes = .true.)
    end if

    SAFE_ALLOCATE(psii(1:mesh%np, 1:st%d%dim))
    SAFE_ALLOCATE(psij(1:mesh%np, 1:st%d%dim))

    SAFE_ALLOCATE(aa(1:nst))

    do ist = 1, nst

      call states_get_state(st, mesh, ist, ik, psii)
      
      ! calculate the projections
      do jst = 1, ist - 1
        call states_get_state(st, mesh, jst, ik, psij)
        aa(jst) = X(mf_dotp)(mesh, st%d%dim, psij, psii, reduce = .false.)
      end do

      if(mesh%parallel_in_domains .and. ist > 1) call comm_allreduce(mesh%mpi_grp%comm, aa, dim = ist - 1)

      ! subtract the projections
      do jst = 1, ist - 1
        call states_get_state(st, mesh, jst, ik, psij)
        do idim = 1, st%d%dim
          call lalg_axpy(mesh%np, -aa(jst), psij(:, idim), psii(:, idim))
        end do
      end do

      ! renormalize
      cc = TOFLOAT(X(mf_dotp)(mesh, st%d%dim, psii, psii))

      do idim = 1, st%d%dim
        call lalg_scal(mesh%np, M_ONE/sqrt(cc), psii(:, idim))
      end do

      call states_set_state(st, mesh, ist, ik, psii)
      
    end do

    SAFE_DEALLOCATE_A(psii)
    SAFE_DEALLOCATE_A(psij)
    SAFE_DEALLOCATE_A(aa)

    POP_SUB(X(states_orthogonalization_full).mgs)
  end subroutine mgs

end subroutine X(states_orthogonalization_full)


! ---------------------------------------------------------

subroutine X(states_trsm)(st, mesh, ik, ss)
  type(states_t),         intent(inout) :: st
  type(mesh_t),           intent(in)    :: mesh
  integer,                intent(in)    :: ik
  R_TYPE,                 intent(in)    :: ss(:, :)

  integer :: idim, block_size, ib, size, sp
  R_TYPE, allocatable :: psicopy(:, :, :)
  type(accel_mem_t) :: psicopy_buffer, ss_buffer
  type(profile_t), save :: prof_copy
  type(profile_t), save :: prof

  PUSH_SUB(X(states_trsm))
  call profiling_in(prof, "STATES_TRSM")

  if(.not. (states_are_packed(st) .and. accel_is_enabled())) then

#ifdef R_TREAL  
    block_size = max(40, hardware%l2%size/(2*8*st%nst))
#else
    block_size = max(20, hardware%l2%size/(2*16*st%nst))
#endif

    SAFE_ALLOCATE(psicopy(1:st%nst, 1:st%d%dim, 1:block_size))

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)

      do ib = st%group%block_start, st%group%block_end
        call batch_get_points(st%group%psib(ib, ik), sp, sp + size - 1, psicopy)
      end do

      if(st%parallel_in_states) call states_parallel_gather(st, (/st%d%dim, size/), psicopy)      
      
      do idim = 1, st%d%dim
        
        call blas_trsm(side = 'L', uplo = 'U', transa = 'T', diag = 'N', &
          m = st%nst, n = size, &
          alpha = R_TOTYPE(M_ONE), a = ss(1, 1), lda = ubound(ss, dim = 1), &
          b = psicopy(1, idim, 1), ldb = ubound(psicopy, dim = 1)*st%d%dim)

      end do
      
      do ib = st%group%block_start, st%group%block_end
        call batch_set_points(st%group%psib(ib, ik), sp, sp + size - 1, psicopy)
      end do

    end do 
    
    SAFE_DEALLOCATE_A(psicopy)

  else

    if(st%d%dim > 1) call messages_not_implemented('Opencl states_trsm for spinors')

    block_size = batch_points_block_size(st%group%psib(st%group%block_start, ik))

    call accel_create_buffer(psicopy_buffer, ACCEL_MEM_READ_WRITE, R_TYPE_VAL, st%nst*block_size)

    call accel_create_buffer(ss_buffer, ACCEL_MEM_READ_ONLY, R_TYPE_VAL, product(ubound(ss)))

    call profiling_in(prof_copy, 'STATES_TRSM_COPY')
    call accel_write_buffer(ss_buffer, product(ubound(ss)), ss)
    call profiling_count_transfers(product(ubound(ss)), ss(1, 1))

    call profiling_out(prof_copy)

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)
      
      do ib = st%group%block_start, st%group%block_end
        ASSERT(R_TYPE_VAL == batch_type(st%group%psib(ib, ik)))
        call batch_get_points(st%group%psib(ib, ik), sp, sp + size - 1, psicopy_buffer, st%nst)
      end do

      call X(accel_trsm)(side = ACCEL_BLAS_LEFT, uplo = ACCEL_BLAS_UPPER, &
        trans = ACCEL_BLAS_T, diag = ACCEL_BLAS_DIAG_NON_UNIT, &
        M = int(st%nst, 8), N = int(size, 8), alpha = R_TOTYPE(M_ONE), &
        A = ss_buffer, offA = 0_8, lda = int(ubound(ss, dim = 1), 8), &
        B = psicopy_buffer, offB = 0_8, ldb = int(st%nst, 8))
      
      do ib = st%group%block_start, st%group%block_end
        call batch_set_points(st%group%psib(ib, ik), sp, sp + size - 1, psicopy_buffer, st%nst)
      end do
    end do

    
    call accel_release_buffer(ss_buffer)
    call accel_release_buffer(psicopy_buffer)

  end if

  call profiling_count_operations(mesh%np*dble(st%nst)*(st%nst + 1)*CNST(0.5)*(R_ADD + R_MUL))


  call profiling_out(prof)
  POP_SUB(X(states_trsm))
end subroutine X(states_trsm)

! ---------------------------------------------------------
subroutine X(states_orthogonalize_single)(st, mesh, nst, iqn, phi, normalize, mask, overlap, norm, Theta_fi, beta_ij)
  type(states_t),    intent(in)    :: st
  type(mesh_t),      intent(in)    :: mesh
  integer,           intent(in)    :: nst
  integer,           intent(in)    :: iqn
  R_TYPE,            intent(inout) :: phi(:,:)     !< phi(mesh%np_part, dim)
  logical, optional, intent(in)    :: normalize
  logical, optional, intent(inout) :: mask(:)      !< mask(nst)
  R_TYPE,  optional, intent(out)   :: overlap(:) 
  R_TYPE,  optional, intent(out)   :: norm
  FLOAT,   optional, intent(in)    :: theta_fi
  R_TYPE,  optional, intent(in)    :: beta_ij(:)   !< beta_ij(nst)

  integer :: ist, idim
  R_TYPE  :: nrm2
  R_TYPE, allocatable  :: ss(:), psi(:, :)
  type(profile_t), save :: prof
  type(profile_t), save :: reduce_prof
  
  call profiling_in(prof, "GRAM_SCHMIDT")
  PUSH_SUB(X(states_orthogonalize_single))

  ASSERT(nst <= st%nst)
  ASSERT(.not. st%parallel_in_states)

  SAFE_ALLOCATE(psi(1:mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(ss(1:nst))

  ss = M_ZERO

  do ist = 1, nst
    if(present(mask)) then
      if(mask(ist)) cycle
    end if
          
    call states_get_state(st, mesh, ist, iqn, psi)
    ss(ist) = X(mf_dotp)(mesh, st%d%dim, psi, phi, reduce = .false., dotu = st%cmplxscl%space)
  end do
    
  if(mesh%parallel_in_domains) then
    call profiling_in(reduce_prof, "GRAM_SCHMIDT_REDUCE")
    call comm_allreduce(mesh%mpi_grp%comm, ss, dim = nst)
    call profiling_out(reduce_prof)
  end if

  if(present(mask)) then
    do ist = 1, nst
      mask(ist) = (abs(ss(ist)) <= M_EPSILON)
    end do
  end if

  if(present(beta_ij)) ss(1:nst) = ss(1:nst)*beta_ij(1:nst)
  
  if(present(theta_fi)) then
    if(theta_fi /= M_ONE) phi(1:mesh%np, 1:st%d%dim) = theta_fi*phi(1:mesh%np, 1:st%d%dim)
  end if

  do ist = 1, nst
    if(present(mask)) then
      if(mask(ist)) cycle
    end if
    
    call states_get_state(st, mesh, ist, iqn, psi)
    do idim = 1, st%d%dim
      call blas_axpy(mesh%np, -ss(ist), psi(1, idim), 1, phi(1, idim), 1)
    end do
  end do

  if(optional_default(normalize, .false.)) then
    if (st%cmplxscl%space) then 
      nrm2 = sqrt(X(mf_dotp)(mesh, st%d%dim, phi, phi, dotu = .true.))
    else
      nrm2 = X(mf_nrm2)(mesh, st%d%dim, phi)
    end if
    if(abs(nrm2) == M_ZERO) then
      message(1) = "Wavefunction has zero norm after states_orthogonalize_single; cannot normalize."
      call messages_fatal(1)
    end if
    do idim = 1, st%d%dim
      call lalg_scal(mesh%np, M_ONE/nrm2, phi(:, idim))
    end do
  end if

  if(present(overlap)) then
    overlap(1:nst) = ss(1:nst)
  end if

  if(present(norm)) then
    ASSERT(present(normalize))
    ASSERT(normalize)
    norm = nrm2
  end if

  SAFE_DEALLOCATE_A(ss)
  SAFE_DEALLOCATE_A(psi)

  POP_SUB(X(states_orthogonalize_single))
  call profiling_out(prof)
end subroutine X(states_orthogonalize_single)

! ---------------------------------------------------------
!> Orthonormalizes phi to the nst orbitals psi.
!! It also permits doing only the orthogonalization (no normalization).
!! And one can pass an extra optional argument, mask, which:
!!  - on input, if mask(p) = .true., the p-orbital is not used.
!!  - on output, mask(p) = .true. if p was already orthogonal (to within 1e-12).
!! If Theta_Fi and beta_ij are present, it performs the generalized orthogonalization
!!   (Theta_Fi - sum_j beta_ij |j><j|Phi> as in De Gironcoli PRB 51, 6774 (1995).
!! This is used in response for metals
subroutine X(states_orthogonalization)(mesh, nst, dim, psi, phi,  &
  normalize, mask, overlap, norm, Theta_fi, beta_ij)
  type(mesh_t),      intent(in)    :: mesh
  integer,           intent(in)    :: nst
  integer,           intent(in)    :: dim
  R_TYPE,            intent(in)    :: psi(:,:,:)   !< psi(mesh%np_part, dim, nst)
  R_TYPE,            intent(inout) :: phi(:,:)     !< phi(mesh%np_part, dim)
  logical, optional, intent(in)    :: normalize
  logical, optional, intent(inout) :: mask(:)      !< mask(nst)
  R_TYPE,  optional, intent(out)   :: overlap(:) 
  R_TYPE,  optional, intent(out)   :: norm
  FLOAT,   optional, intent(in)    :: Theta_Fi
  R_TYPE,  optional, intent(in)    :: beta_ij(:)   !< beta_ij(nst)

  logical :: normalize_
  integer :: ist, idim
  FLOAT   :: nrm2
  R_TYPE, allocatable  :: ss(:)
  integer :: block_size, size, sp, ep
  type(profile_t), save :: prof
  type(profile_t), save :: reduce_prof

  call profiling_in(prof, "GRAM_SCHMIDT")
  PUSH_SUB(X(states_orthogonalization))

  ! This routine uses blocking to optimize cache usage. One block of
  ! |phi> is loaded in cache L1 and then then we calculate the dot
  ! product of it with the corresponding blocks of |psi_k>, next we
  ! load another block and do the same. This way we only have to load
  ! |psi> from the L2 or memory.
  block_size = hardware%X(block_size)

  SAFE_ALLOCATE(ss(1:nst))

  ss = M_ZERO

  if(.not. mesh%use_curvilinear) then

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)
      do ist = 1, nst
        do idim = 1, dim
        
          if(present(mask)) then
            if(mask(ist)) cycle
          end if

          ss(ist) = ss(ist) + blas_dot(size, psi(sp, idim, ist), 1, phi(sp, idim), 1)
        end do
      end do
    end do

    ss = ss * mesh%vol_pp(1)

    call profiling_count_operations((R_ADD + R_MUL) * mesh%np * dim * nst)

  else

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)
      ep = sp - 1 + size
      do ist = 1, nst
        do idim = 1, dim

          if(present(mask)) then
            if(mask(ist)) cycle
          end if

          ss(ist) = ss(ist) + sum(mesh%vol_pp(sp:ep)*R_CONJ(psi(sp:ep, idim, ist))*phi(sp:ep, idim))

        end do
      end do
    end do

    call profiling_count_operations((R_ADD + 2 * R_MUL) * mesh%np * dim * nst)

  end if

  if(mesh%parallel_in_domains) then
    call profiling_in(reduce_prof, "GRAM_SCHMIDT_REDUCE")
    call comm_allreduce(mesh%mpi_grp%comm, ss, dim = nst)
    call profiling_out(reduce_prof)
  end if

  if(present(mask)) then
    do ist = 1, nst
      mask(ist) = (abs(ss(ist)) <= M_EPSILON)
    end do
  end if

  if(present(beta_ij))  &
    ss(:) = ss(:) * beta_ij(:)

  do sp = 1, mesh%np, block_size
    size = min(block_size, mesh%np - sp + 1)

    if(present(Theta_Fi)) then
      if(Theta_Fi /= M_ONE) &
        call blas_scal(size, R_TOTYPE(Theta_Fi), phi(sp, idim), 1)
    end if

    do idim = 1, dim

      do ist = 1, nst

        if(present(mask)) then
          if(mask(ist)) cycle
        end if

        call blas_axpy(size, -ss(ist), psi(sp, idim, ist), 1, phi(sp, idim), 1)

      end do
    end do
  end do

  call profiling_count_operations((R_ADD + R_MUL) * mesh%np * dim * nst)

  ! the following ifs cannot be given as a single line (without the
  ! then) to avoid a bug in xlf 10.1

  normalize_ = .false.
  if(present(normalize)) then
    normalize_ = normalize
  end if

  if(normalize_) then
    nrm2 = X(mf_nrm2)(mesh, dim, phi)
    do idim = 1, dim
      call lalg_scal(mesh%np, M_ONE / nrm2, phi(:, idim))
    end do
  end if

  if(present(overlap)) then
    overlap(1:nst) = ss(1:nst)
  end if

  if(present(norm)) then
    ASSERT(normalize)
    norm = nrm2
  end if

  SAFE_DEALLOCATE_A(ss)

  POP_SUB(X(states_orthogonalization))
  call profiling_out(prof)
end subroutine X(states_orthogonalization)


! ---------------------------------------------------------
subroutine X(states_normalize_orbital)(mesh, dim, psi)
  type(mesh_t),    intent(in)    :: mesh
  integer,         intent(in)    :: dim
  R_TYPE,          intent(inout) :: psi(:,:)

  FLOAT   :: norm
  integer :: idim, ip

  PUSH_SUB(X(states_normalize_orbital))

  norm = X(mf_nrm2) (mesh, dim, psi)

  forall (idim = 1:dim, ip = 1:mesh%np) psi(ip, idim) = psi(ip, idim)/norm
  
  POP_SUB(X(states_normalize_orbital))
end subroutine X(states_normalize_orbital)


! ---------------------------------------------------------
FLOAT function X(states_residue)(mesh, dim, hf, ee, ff) result(rr)
  type(mesh_t),      intent(in)  :: mesh
  integer,           intent(in)  :: dim
  R_TYPE,            intent(in)  :: hf(:,:)
  FLOAT,             intent(in)  :: ee
  R_TYPE,            intent(in)  :: ff(:,:)

  R_TYPE, allocatable :: res(:,:)
  type(profile_t), save :: prof
  integer :: ip, idim

  PUSH_SUB(X(states_residue))

  call profiling_in(prof, "RESIDUE")

  SAFE_ALLOCATE(res(1:mesh%np, 1:dim))

  forall (idim = 1:dim, ip = 1:mesh%np) res(ip, idim) = hf(ip, idim) - ee*ff(ip, idim)

  call profiling_count_operations(dim*mesh%np*(R_ADD + R_MUL))

  rr = X(mf_nrm2)(mesh, dim, res)
  SAFE_DEALLOCATE_A(res)

  call profiling_out(prof)

  POP_SUB(X(states_residue))

end function X(states_residue)


! ---------------------------------------------------------
!> The routine calculates the expectation value of the momentum 
!! operator
!! <p> = < phi*(ist, k) | -i \nabla | phi(ist, ik) >
!!
! ---------------------------------------------------------
subroutine X(states_calc_momentum)(st, der, momentum)
  type(states_t),      intent(inout) :: st
  type(derivatives_t), intent(inout) :: der
  FLOAT,               intent(out)   :: momentum(:,:,:)

  integer             :: idim, ist, ik, idir
  CMPLX               :: expect_val_p
  R_TYPE, allocatable :: psi(:, :), grad(:,:,:)
  FLOAT               :: kpoint(1:MAX_DIM)  
#if defined(HAVE_MPI)
  integer             :: tmp
  FLOAT, allocatable  :: lmomentum(:), gmomentum(:)
  FLOAT, allocatable  :: lmom(:, :, :)
  integer             :: kstart, kend, kn, ndim
#endif

  PUSH_SUB(X(states_calc_momentum))

  SAFE_ALLOCATE(psi(1:der%mesh%np_part, 1:st%d%dim))
  SAFE_ALLOCATE(grad(1:der%mesh%np, 1:st%d%dim, 1:der%mesh%sb%dim))

  do ik = st%d%kpt%start, st%d%kpt%end
    do ist = st%st_start, st%st_end

      call states_get_state(st, der%mesh, ist, ik, psi)

      do idim = 1, st%d%dim
        call X(derivatives_grad)(der, psi(:, idim), grad(:, idim, 1:der%mesh%sb%dim))
      end do

      do idir = 1, der%mesh%sb%dim
        ! since the expectation value of the momentum operator is real
        ! for square integrable wfns this integral should be purely imaginary 
        ! for complex wfns but real for real wfns (see case distinction below)
        expect_val_p = X(mf_dotp)(der%mesh, st%d%dim, psi, grad(:, :, idir))

        ! In the case of real wavefunctions we do not include the 
        ! -i prefactor of p = -i \nabla
        if (states_are_real(st)) then
          momentum(idir, ist, ik) = real( expect_val_p )
        else
          momentum(idir, ist, ik) = real( -M_zI*expect_val_p )
        end if
      end do

      ! have to add the momentum vector in the case of periodic systems, 
      ! since psi contains only u_k
      kpoint = M_ZERO
      kpoint(1:der%mesh%sb%dim) = kpoints_get_point(der%mesh%sb%kpoints, states_dim_get_kpoint_index(st%d, ik))
      forall(idir = 1:der%mesh%sb%periodic_dim) momentum(idir, ist, ik) = momentum(idir, ist, ik) + kpoint(idir)

    end do

    ! Exchange momenta in the parallel case.
#if defined(HAVE_MPI)
    if(st%parallel_in_states) then
      SAFE_ALLOCATE(lmomentum(1:st%lnst))
      SAFE_ALLOCATE(gmomentum(1:st%nst))

      do idir = 1, der%mesh%sb%dim
        lmomentum(1:st%lnst) = momentum(idir, st%st_start:st%st_end, ik)
        call lmpi_gen_allgatherv(st%lnst, lmomentum, tmp, gmomentum, st%mpi_grp)
        momentum(idir, 1:st%nst, ik) = gmomentum(1:st%nst)
      end do

      SAFE_DEALLOCATE_A(lmomentum)
      SAFE_DEALLOCATE_A(gmomentum)
    end if
#endif
  end do

#if defined(HAVE_MPI)
  if(st%d%kpt%parallel) then
    kstart = st%d%kpt%start
    kend = st%d%kpt%end
    kn = st%d%kpt%nlocal
    ndim = ubound(momentum, dim = 1)
    
    ASSERT(.not. st%parallel_in_states)
    
    SAFE_ALLOCATE(lmom(1:ndim, 1:st%nst, 1:kn))
    
    lmom(1:ndim, 1:st%nst, 1:kn) = momentum(1:ndim, 1:st%nst, kstart:kend)
    
    call MPI_Allgatherv(lmom(1, 1, 1), ndim*st%nst*kn, MPI_FLOAT, &
      momentum, st%d%kpt%num(:)*st%nst*ndim, (st%d%kpt%range(1, :) - 1)*st%nst*ndim, MPI_FLOAT, &
      st%d%kpt%mpi_grp%comm, mpi_err)
    
    SAFE_DEALLOCATE_A(lmom)
  end if
#endif  

  SAFE_DEALLOCATE_A(psi)
  SAFE_DEALLOCATE_A(grad)

  POP_SUB(X(states_calc_momentum))
end subroutine X(states_calc_momentum)


! ---------------------------------------------------------
!> It calculates the expectation value of the angular
!! momentum of the states. If l2 is passed, it also
!! calculates the expectation value of the square of the
!! angular momentum of the state phi.
! ---------------------------------------------------------
subroutine X(states_angular_momentum)(st, gr, ll, l2)
  type(states_t),  intent(in)     :: st
  type(grid_t),    intent(inout)  :: gr
  FLOAT,           intent(out)    :: ll(:, :, :) !< (st%nst, st%d%nik, 1 or 3)
  FLOAT, optional, intent(out)    :: l2(:, :)    !< (st%nst, st%d%nik)

  integer :: idim, ist, ik
  R_TYPE, allocatable :: psi(:), lpsi(:, :)

  PUSH_SUB(X(states_angular_momemtum))

  ASSERT(gr%mesh%sb%dim /= 1)

  SAFE_ALLOCATE(psi(1:gr%mesh%np_part))

  select case(gr%mesh%sb%dim)
  case(3)
    SAFE_ALLOCATE(lpsi(1:gr%mesh%np_part, 1:3))
  case(2)
    SAFE_ALLOCATE(lpsi(1:gr%mesh%np_part, 1:1))
  end select

  ll = M_ZERO
  if(present(l2)) l2 = M_ZERO

  do ik = st%d%kpt%start, st%d%kpt%end
    do ist = st%st_start, st%st_end
      do idim = 1, st%d%dim
        call states_get_state(st, gr%mesh, idim, ist, ik, psi)

#if defined(R_TREAL)
        ll = M_ZERO
#else
        call X(physics_op_L)(gr%der, psi, lpsi)

        ll(ist, ik, 1) = ll(ist, ik, 1) + TOFLOAT(X(mf_dotp)(gr%mesh, psi, lpsi(:, 1)))
        if(gr%mesh%sb%dim == 3) then
          ll(ist, ik, 2) = ll(ist, ik, 2) + TOFLOAT(X(mf_dotp)(gr%mesh, psi, lpsi(:, 2)))
          ll(ist, ik, 3) = ll(ist, ik, 3) + TOFLOAT(X(mf_dotp)(gr%mesh, psi, lpsi(:, 3)))
        end if
#endif
        if(present(l2)) then
          call X(physics_op_L2)(gr%der, psi(:), lpsi(:, 1))
          l2(ist, ik) = l2(ist, ik) + TOFLOAT(X(mf_dotp)(gr%mesh, psi(:), lpsi(:, 1)))
        end if
      end do
    end do
  end do

  SAFE_DEALLOCATE_A(psi)
  SAFE_DEALLOCATE_A(lpsi)
  POP_SUB(X(states_angular_momemtum))
end subroutine X(states_angular_momentum)


! ---------------------------------------------------------
subroutine X(states_matrix)(mesh, st1, st2, aa)
  type(mesh_t),   intent(in)  :: mesh
  type(states_t), intent(in)  :: st1, st2
  R_TYPE,         intent(out) :: aa(:, :, :)

  integer :: ii, jj, dim, ik
  R_TYPE, allocatable :: psi1(:, :), psi2(:, :)
#if defined(HAVE_MPI)
  R_TYPE, allocatable :: phi2(:, :)
  integer :: kk, ll, ist
  integer :: status(MPI_STATUS_SIZE)
  integer :: request
#endif

  PUSH_SUB(X(states_matrix))

  dim = st1%d%dim

  SAFE_ALLOCATE(psi1(1:mesh%np, 1:st1%d%dim))
  SAFE_ALLOCATE(psi2(1:mesh%np, 1:st1%d%dim))

  do ik = st1%d%kpt%start, st1%d%kpt%end

    if(st1%parallel_in_states) then

#if defined(HAVE_MPI)
      call MPI_Barrier(st1%mpi_grp%comm, mpi_err)
      ! Each process sends the states in st2 to the rest of the processes.
      do ist = st1%st_start, st1%st_end
        call states_get_state(st2, mesh, ist, ik, psi2)
        do jj = 0, st1%mpi_grp%size - 1
          if(st1%mpi_grp%rank /= jj) then
            call MPI_Isend(psi2(1, 1), st1%d%dim*mesh%np, R_MPITYPE, jj, ist, st1%mpi_grp%comm, request, mpi_err)
          end if
        end do
      end do

      ! Processes are received, and then the matrix elements are calculated.
      SAFE_ALLOCATE(phi2(1:mesh%np, 1:st1%d%dim))
      do jj = 1, st2%nst

        ll = st1%node(jj)

        if(ll /= st1%mpi_grp%rank) then
          call MPI_Irecv(phi2(1, 1), st1%d%dim*mesh%np, R_MPITYPE, ll, jj, st1%mpi_grp%comm, request, mpi_err)
          call MPI_Wait(request, status, mpi_err)
        else
          call states_get_state(st2, mesh, jj, ik, phi2)
        end if

        do ist = st1%st_start, st1%st_end
          call states_get_state(st1, mesh, ist, ik, psi1)
          aa(ist, jj, ik) = X(mf_dotp)(mesh, dim, psi1, phi2)
        end do

      end do
      SAFE_DEALLOCATE_A(phi2)

      ! Each process holds some lines of the matrix. So it is broadcasted (All processes
      ! should get the whole matrix)
      call MPI_Barrier(st1%mpi_grp%comm, mpi_err)
      do ii = 1, st1%nst
        kk = st1%node(ii)
        do jj = 1, st2%nst
          call MPI_Bcast(aa(ii, jj, ik), 1, R_MPITYPE, kk, st1%mpi_grp%comm, mpi_err)
        end do
      end do
#else
      write(message(1), '(a)') 'Internal error at Xstates_matrix'
      call messages_fatal(1)
#endif

    else

      do ii = st1%st_start, st1%st_end

        call states_get_state(st1, mesh, ii, ik, psi1)

        do jj = st2%st_start, st2%st_end

          call states_get_state(st2, mesh, jj, ik, psi2)

          aa(ii, jj, ik) = X(mf_dotp)(mesh, dim, psi1, psi2)

        end do
      end do

    end if

  end do

  SAFE_DEALLOCATE_A(psi1)
  SAFE_DEALLOCATE_A(psi2)    

  POP_SUB(X(states_matrix))
end subroutine X(states_matrix)

! -----------------------------------------------------------

subroutine X(states_calc_orth_test)(st, mesh)
  type(states_t),    intent(inout) :: st
  type(mesh_t),      intent(in)    :: mesh
  
  PUSH_SUB(X(states_calc_orth_test))

  call states_allocate_wfns(st, mesh, wfs_type = R_TYPE_VAL)

  call states_generate_random(st, mesh)

  message(1) = 'Info: Orthogonalizing random wavefunctions.'
  message(2) = ''
  call messages_info(2)

  if(st%d%pack_states) call states_pack(st)

  call X(states_orthogonalization_full)(st, mesh, 1)

  if(st%d%pack_states) call states_unpack(st)

  call print_results()
  
  call states_deallocate_wfns(st)

  POP_SUB(X(states_calc_orth_test))

contains

  subroutine print_results()

    integer :: ist, jst
    R_TYPE :: dd
    R_TYPE, allocatable :: psi1(:, :), psi2(:, :)
    R_TYPE, allocatable :: spsi1(:, :), spsi2(:, :)
#ifdef HAVE_MPI
    integer :: req(1:4), nreq
#endif

    PUSH_SUB(X(states_calc_orth_test).print_results)

    SAFE_ALLOCATE(psi1(1:mesh%np, 1:st%d%dim))
    SAFE_ALLOCATE(psi2(1:mesh%np, 1:st%d%dim))
#ifdef HAVE_MPI
    if(st%parallel_in_states) then
      SAFE_ALLOCATE(spsi1(1:mesh%np, 1:st%d%dim))
      SAFE_ALLOCATE(spsi2(1:mesh%np, 1:st%d%dim))
    end if
#endif

    message(1) = 'Residuals:'
    call messages_info(1)
    
    do ist = 1, st%nst
      do jst = ist, st%nst
        if(.not. st%parallel_in_states) then
          call states_get_state(st, mesh, ist, 1, psi1)
          call states_get_state(st, mesh, jst, 1, psi2)
        end if

#ifdef HAVE_MPI
        if(st%parallel_in_states) then

          ! we bring the two vectors to node 0 and calculate the dot
          ! product, this is very simple and very slow, we only do it for testing
          
          nreq = 0

          ! post the receptions
          if(st%mpi_grp%rank == 0) then
            call MPI_Irecv(psi1(1, 1), mesh%np*st%d%dim, R_MPITYPE, st%node(ist), ist, &
              st%mpi_grp%comm, req(nreq + 1), mpi_err)
            call MPI_Irecv(psi2(1, 1), mesh%np*st%d%dim, R_MPITYPE, st%node(jst), jst, &
              st%mpi_grp%comm, req(nreq + 2), mpi_err)
            INCR(nreq, 2)
          end if

          ! if I have the wave function, I send it (note: a node could be sending to itself, this is by design)
          if(st%node(ist)  == st%mpi_grp%rank) then
            INCR(nreq, 1)
            call states_get_state(st, mesh, ist, 1, spsi1)
            call MPI_Isend(spsi1(1, 1), mesh%np*st%d%dim, R_MPITYPE, 0, ist, st%mpi_grp%comm, req(nreq), mpi_err)
          end if
          
          if(st%node(jst) == st%mpi_grp%rank) then
            INCR(nreq, 1)
            call states_get_state(st, mesh, jst, 1, spsi2)
            call MPI_Isend(spsi2(1, 1), mesh%np*st%d%dim, R_MPITYPE, 0, jst, st%mpi_grp%comm, req(nreq), mpi_err)
          end if

          if(nreq > 0) call MPI_Waitall(nreq, req, MPI_STATUSES_IGNORE, mpi_err)

          if(st%mpi_grp%rank /= 0) cycle

        end if
#endif
        dd = X(mf_dotp)(mesh, st%d%dim, psi1, psi2)
        write (message(1), '(2i7, e16.6)') ist, jst, abs(dd)
        call messages_info(1)

      end do
    end do
    
    message(1) = ''
    call messages_info(1)

    SAFE_DEALLOCATE_A(psi1)
    SAFE_DEALLOCATE_A(psi2)
    SAFE_DEALLOCATE_A(spsi1)
    SAFE_DEALLOCATE_A(spsi2)

    POP_SUB(X(states_calc_orth_test).print_results)
  end subroutine print_results

end subroutine X(states_calc_orth_test)

! ---------------------------------------------------------

subroutine X(states_rotate)(mesh, st, uu, ik)
  type(mesh_t),      intent(in)    :: mesh
  type(states_t),    intent(inout) :: st
  R_TYPE,            intent(in)    :: uu(:, :)
  integer,           intent(in)    :: ik
  
  integer       :: block_size, sp, idim, size, ib
  R_TYPE, allocatable :: psinew(:, :, :), psicopy(:, :, :)
  type(accel_mem_t) :: psinew_buffer, psicopy_buffer, uu_buffer
  type(profile_t), save :: prof

  PUSH_SUB(X(states_rotate))

  call profiling_in(prof, "STATES_ROTATE")

  ASSERT(R_TYPE_VAL == states_type(st))
  
  if(.not. states_are_packed(st) .or. .not. accel_is_enabled()) then
    
#ifdef R_TREAL  
    block_size = max(40, hardware%l2%size/(2*8*st%nst))
#else
    block_size = max(20, hardware%l2%size/(2*16*st%nst))
#endif

    SAFE_ALLOCATE(psinew(1:st%nst, 1:st%d%dim, 1:block_size))
    SAFE_ALLOCATE(psicopy(1:st%nst, 1:st%d%dim, 1:block_size))

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)
      
      do ib = st%group%block_start, st%group%block_end
        call batch_get_points(st%group%psib(ib, ik), sp, sp + size - 1, psicopy)
      end do

      if(st%parallel_in_states) call states_parallel_gather(st, (/st%d%dim, size/), psicopy)
      
      do idim = 1, st%d%dim
        
        call blas_gemm(transa = 't', transb = 'n',        &
          m = st%nst, n = size, k = st%nst,               &
          alpha = R_TOTYPE(M_ONE),                        &
          a = uu(1, 1), lda = ubound(uu, dim = 1),        &
          b = psicopy(1, idim, 1), ldb = st%nst*st%d%dim, &
          beta = R_TOTYPE(M_ZERO),                        & 
          c = psinew(1, idim, 1), ldc = st%nst*st%d%dim)
        
      end do
      
      do ib = st%group%block_start, st%group%block_end
        call batch_set_points(st%group%psib(ib, ik), sp, sp + size - 1, psinew)
      end do

    end do

    call profiling_count_operations((R_ADD + R_MUL)*st%nst*(st%nst - CNST(1.0))*mesh%np)

    SAFE_DEALLOCATE_A(psinew)
    SAFE_DEALLOCATE_A(psicopy)

  else

    if(st%d%dim > 1) call messages_not_implemented('Opencl states_rotate for spinors')

    block_size = batch_points_block_size(st%group%psib(st%group%block_start, ik))

    call accel_create_buffer(uu_buffer, ACCEL_MEM_READ_ONLY, R_TYPE_VAL, product(ubound(uu)))
    call accel_write_buffer(uu_buffer, product(ubound(uu)), uu)

    call accel_create_buffer(psicopy_buffer, ACCEL_MEM_READ_WRITE, R_TYPE_VAL, st%nst*block_size)
    call accel_create_buffer(psinew_buffer, ACCEL_MEM_READ_WRITE, R_TYPE_VAL, st%nst*block_size)

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)
      
      do ib = st%group%block_start, st%group%block_end
        call batch_get_points(st%group%psib(ib, ik), sp, sp + size - 1, psicopy_buffer, st%nst)
      end do

      call X(accel_gemm)(transA = CUBLAS_OP_T, transB = CUBLAS_OP_N, &
        M = int(st%nst, 8), N = int(size, 8), K = int(st%nst, 8), alpha = R_TOTYPE(M_ONE), &
        A = uu_buffer, offA = 0_8, lda = int(ubound(uu, dim = 1), 8), &
        B = psicopy_buffer, offB = 0_8, ldb = int(st%nst, 8), beta = R_TOTYPE(M_ZERO), &
        C = psinew_buffer, offC = 0_8, ldc = int(st%nst, 8))
      
      call accel_finish()

      do ib = st%group%block_start, st%group%block_end
        call batch_set_points(st%group%psib(ib, ik), sp, sp + size - 1, psinew_buffer, st%nst)
      end do
    end do

    call profiling_count_operations((R_ADD + R_MUL)*st%nst*(st%nst - CNST(1.0))*mesh%np)
   
    call accel_release_buffer(uu_buffer)
    call accel_release_buffer(psicopy_buffer)
    call accel_release_buffer(psinew_buffer)

  end if

  call profiling_out(prof)
  POP_SUB(X(states_rotate))
end subroutine X(states_rotate)

! ---------------------------------------------------------

subroutine X(states_calc_overlap)(st, mesh, ik, overlap)
  type(states_t),    intent(inout) :: st
  type(mesh_t),      intent(in)    :: mesh
  integer,           intent(in)    :: ik
  R_TYPE,            intent(out)   :: overlap(:, :)

  integer :: ip, ib, jb, block_size, sp, size
#ifndef R_TREAL
  integer :: ist, jst
#endif
  type(profile_t), save :: prof
  FLOAT :: vol
  R_TYPE, allocatable :: psi(:, :, :)
  type(accel_mem_t) :: psi_buffer, overlap_buffer

  PUSH_SUB(X(states_calc_overlap))

  call profiling_in(prof, "STATES_OVERLAP")

  if(.not. states_are_packed(st) .or. .not. accel_is_enabled()) then

#ifdef R_TREAL  
    block_size = max(80, hardware%l2%size/(8*st%nst))
#else
    block_size = max(40, hardware%l2%size/(16*st%nst))
#endif

    SAFE_ALLOCATE(psi(1:st%nst, 1:st%d%dim, 1:block_size))

    overlap(1:st%nst, 1:st%nst) = CNST(0.0)

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)

      do ib = st%group%block_start, st%group%block_end
        call batch_get_points(st%group%psib(ib, ik), sp, sp + size - 1, psi)
      end do

      if(st%parallel_in_states) call states_parallel_gather(st, (/st%d%dim, size/), psi)
      
      if(mesh%use_curvilinear) then
        do ip = sp, sp + size - 1
          vol = sqrt(mesh%vol_pp(ip))
          psi(1:st%nst, 1:st%d%dim, ip) = psi(1:st%nst, 1:st%d%dim, ip)*vol
        end do
      end if

      call blas_herk(uplo = 'u', trans = 'n',              &
        n = st%nst, k = size*st%d%dim,                     &
        alpha = mesh%volume_element,                       &
        a = psi(1, 1, 1), lda = ubound(psi, dim = 1),      &
        beta = CNST(1.0),                                  & 
        c = overlap(1, 1), ldc = ubound(overlap, dim = 1))

    end do

#ifndef R_TREAL
    do jst = 1, st%nst
      do ist = 1, jst
        overlap(ist, jst) = conjg(overlap(ist, jst))
      end do
    end do
#endif

    call profiling_count_operations((R_ADD + R_MUL)*CNST(0.5)*st%nst*(st%nst - CNST(1.0))*mesh%np)

    if(mesh%parallel_in_domains) call comm_allreduce(mesh%mpi_grp%comm, overlap, dim = (/st%nst, st%nst/))

    SAFE_DEALLOCATE_A(psi)


  else if(accel_is_enabled()) then

    ASSERT(ubound(overlap, dim = 1) == st%nst)
    ASSERT(.not. st%parallel_in_states)
    
    call accel_create_buffer(overlap_buffer, ACCEL_MEM_READ_WRITE, R_TYPE_VAL, st%nst*st%nst)
    call accel_set_buffer_to_zero(overlap_buffer, R_TYPE_VAL, st%nst*st%nst)

    ! we need to use a temporary array

    block_size = batch_points_block_size(st%group%psib(st%group%block_start, ik))

    call accel_create_buffer(psi_buffer, ACCEL_MEM_READ_WRITE, R_TYPE_VAL, st%nst*block_size)

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)

      do ib = st%group%block_start, st%group%block_end
        ASSERT(R_TYPE_VAL == batch_type(st%group%psib(ib, ik)))
        call batch_get_points(st%group%psib(ib, ik), sp, sp + size - 1, psi_buffer, st%nst)
      end do

      call X(accel_herk)(uplo = ACCEL_BLAS_UPPER, trans = ACCEL_BLAS_N, &
        n = int(st%nst, 8), k = int(size, 8), &
        alpha = mesh%volume_element, &
        A = psi_buffer, offa = 0_8, lda = int(st%nst, 8), &
        beta = 1.0_8, &
        C = overlap_buffer, offc = 0_8, ldc = int(st%nst, 8))
    end do

    call accel_finish()

    call accel_release_buffer(psi_buffer)

    call profiling_count_operations((R_ADD + R_MUL)*CNST(0.5)*st%nst*(st%nst - CNST(1.0))*mesh%np)

    call accel_read_buffer(overlap_buffer, st%nst*st%nst, overlap)

    call accel_finish()

#ifndef R_TREAL
    do jst = 1, st%nst
      do ist = 1, jst
        overlap(ist, jst) = conjg(overlap(ist, jst))
      end do
    end do
#endif

    if(mesh%parallel_in_domains) call comm_allreduce(mesh%mpi_grp%comm, overlap, dim = (/st%nst, st%nst/))

    call accel_release_buffer(overlap_buffer)

  else

    do ib = st%group%block_start, st%group%block_end
      do jb = ib, st%group%block_end
        if(ib == jb) then
          call X(mesh_batch_dotp_self)(mesh, st%group%psib(ib, ik), overlap)
        else
          call X(mesh_batch_dotp_matrix)(mesh, st%group%psib(ib, ik), st%group%psib(jb, ik), overlap)
        end if
      end do
    end do

  end if

! Debug output
!#ifndef R_TREAL
!  do ist = 1, st%nst
!    do jst = 1, st%nst
!      write(12, '(e12.6,a,e12.6,a)', advance = 'no') real(overlap(ist, jst)), ' ',  aimag(overlap(ist, jst)), ' '
!    end do
!      write(12, *) ' ' 
!  end do
!#endif

  call profiling_out(prof)

  POP_SUB(X(states_calc_overlap))
end subroutine X(states_calc_overlap)

!> This routine computes the projection between two set of states
subroutine X(states_calc_projections)(mesh, st, gs_st, ik, proj)
  type(mesh_t),           intent(in)    :: mesh
  type(states_t),         intent(in)    :: st
  type(states_t),         intent(in)    :: gs_st
  integer,                intent(in)    :: ik
  R_TYPE,                 intent(out)   :: proj(:, :)

  integer       :: ib, ip
  R_TYPE, allocatable :: psi(:, :, :), gspsi(:, :, :)
  integer :: sp, size, block_size
  type(profile_t), save :: prof

  PUSH_SUB(X(states_calc_projections))
  call profiling_in(prof, "STATES_PROJECTIONS")

  if(states_are_packed(st) .and. accel_is_enabled()) then
   message(1) = "states_calc_projections is not implemented with packed states or accel."
   call messages_fatal(1) 
  else

#ifdef R_TREAL  
    block_size = max(40, hardware%l2%size/(2*8*st%nst))
#else
    block_size = max(20, hardware%l2%size/(2*16*st%nst))
#endif

    proj(1:gs_st%nst, 1:st%nst) = CNST(0.0)
    
    SAFE_ALLOCATE(psi(1:st%nst, 1:st%d%dim, 1:block_size))
    SAFE_ALLOCATE(gspsi(1:gs_st%nst, 1:gs_st%d%dim, 1:block_size))

    do sp = 1, mesh%np, block_size
      size = min(block_size, mesh%np - sp + 1)
      
      do ib = st%group%block_start, st%group%block_end
        call batch_get_points(st%group%psib(ib, ik),  sp, sp + size - 1, psi)
        call batch_get_points(gs_st%group%psib(ib,ik), sp, sp + size - 1, gspsi)
      end do

      if(st%parallel_in_states) then
        call states_parallel_gather(st, (/st%d%dim, size/), psi)
        call states_parallel_gather(gs_st, (/st%d%dim, size/), gspsi)
      end if
      
      if(mesh%use_curvilinear) then
        do ip = 1, size
          psi(1:st%nst, 1:st%d%dim, ip) = psi(1:st%nst, 1:st%d%dim, ip)*mesh%vol_pp(sp + ip - 1)
          gspsi(1:gs_st%nst, 1:st%d%dim, ip) = gspsi(1:gs_st%nst, 1:st%d%dim, ip)*mesh%vol_pp(sp + ip - 1)
        end do
      end if

      call blas_gemm(transa = 'n', transb = 'c',        &
        m = gs_st%nst, n = st%nst, k = size*st%d%dim,      &
        alpha = R_TOTYPE(mesh%volume_element),      &
        a = gspsi(1, 1, 1), lda = ubound(gspsi, dim = 1),   &
        b = psi(1, 1, 1), ldb = ubound(psi, dim = 1), &
        beta = R_TOTYPE(CNST(1.0)),                     & 
        c = proj(1, 1), ldc = ubound(proj, dim = 1))
    end do

  end if
  
  call profiling_count_operations((R_ADD + R_MUL)*gs_st%nst*(st%nst - CNST(1.0))*mesh%np)
  
  if(mesh%parallel_in_domains) call comm_allreduce(mesh%mpi_grp%comm, proj, dim = (/gs_st%nst, st%nst/))
  
  call profiling_out(prof)
  POP_SUB(X(states_calc_projections))

end subroutine X(states_calc_projections)

! ---------------------------------------------------------
subroutine X(states_me_one_body)(dir, gr, geo, st, nspin, vhxc, nint, iindex, jindex, oneint)

  character(len=*),    intent(in)    :: dir
  type(grid_t),        intent(inout) :: gr
  type(geometry_t),    intent(in)    :: geo
  type(states_t),      intent(inout) :: st
  integer,             intent(in)    :: nspin
  FLOAT,               intent(in)    :: vhxc(1:gr%mesh%np, nspin)
  integer,             intent(in)    :: nint
  integer,             intent(out)   :: iindex(1:nint)
  integer,             intent(out)   :: jindex(1:nint)
  R_TYPE,              intent(out)   :: oneint(1:nint)  
  
  integer ist, jst, np, iint
  R_TYPE :: me
  R_TYPE, allocatable :: psii(:, :), psij(:, :)

  PUSH_SUB(X(states_me_one_body))

  SAFE_ALLOCATE(psii(1:gr%mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(psij(1:gr%mesh%np_part, 1:st%d%dim))
  
  np = gr%mesh%np
  iint = 1

  do ist = 1, st%nst

    call states_get_state(st, gr%mesh, ist, 1, psii)
    
    do jst = 1, st%nst
      if(jst > ist) cycle
      
      call states_get_state(st, gr%mesh, jst, 1, psij)

      psij(1:np, 1) = R_CONJ(psii(1:np, 1))*vhxc(1:np, 1)*psij(1:np, 1)

      me = - X(mf_integrate)(gr%mesh, psij(:, 1))

      if(ist==jst) me = me + st%eigenval(ist,1)

      iindex(iint) = ist
      jindex(iint) = jst
      oneint(iint) = me
      iint = iint + 1

    end do
  end do

  SAFE_DEALLOCATE_A(psii)
  SAFE_DEALLOCATE_A(psij)

  POP_SUB(X(states_me_one_body))
end subroutine X(states_me_one_body)


! ---------------------------------------------------------
subroutine X(states_me_two_body) (gr, st, nint, iindex, jindex, kindex, lindex, twoint)
  type(grid_t),     intent(inout)           :: gr
  type(states_t),   intent(in)              :: st
  integer,          intent(in)              :: nint
  integer,          intent(out)             :: iindex(1:nint)
  integer,          intent(out)             :: jindex(1:nint)
  integer,          intent(out)             :: kindex(1:nint)
  integer,          intent(out)             :: lindex(1:nint)
  R_TYPE,           intent(out)             :: twoint(1:nint)  !

  integer :: ist, jst, kst, lst, ijst, klst
  integer :: iint
  R_TYPE  :: me
  R_TYPE, allocatable :: nn(:), vv(:)
  R_TYPE, allocatable :: psii(:, :), psij(:, :), psik(:, :), psil(:, :)

  PUSH_SUB(X(states_me_two_body))

  SAFE_ALLOCATE(nn(1:gr%mesh%np))
  SAFE_ALLOCATE(vv(1:gr%mesh%np))
  SAFE_ALLOCATE(psii(1:gr%mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(psij(1:gr%mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(psik(1:gr%mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(psil(1:gr%mesh%np, 1:st%d%dim))

  ijst = 0
  iint = 1

  do ist = 1, st%nst

    call states_get_state(st, gr%mesh, ist, 1, psii)

    do jst = 1, st%nst
      if(jst > ist) cycle
      ijst=ijst+1

      call states_get_state(st, gr%mesh, jst, 1, psij)

      nn(1:gr%mesh%np) = R_CONJ(psii(1:gr%mesh%np, 1))*psij(1:gr%mesh%np, 1)
      call X(poisson_solve)(psolver, vv, nn, all_nodes=.false.)

      klst=0
      do kst = 1, st%nst
 
        call states_get_state(st, gr%mesh, kst, 1, psik)

        do lst = 1, st%nst
          if(lst > kst) cycle
          klst=klst+1
          if(klst > ijst) cycle

          call states_get_state(st, gr%mesh, lst, 1, psil)

          psil(1:gr%mesh%np, 1) = vv(1:gr%mesh%np)*psik(1:gr%mesh%np, 1)*R_CONJ(psil(1:gr%mesh%np, 1))

          me = X(mf_integrate)(gr%mesh, psil(:, 1))

          iindex(iint) =  ist
          jindex(iint) =  jst
          kindex(iint) =  kst
          lindex(iint) =  lst
          twoint(iint) =  me
          iint = iint + 1

        end do
      end do
    end do
  end do

  SAFE_DEALLOCATE_A(nn)
  SAFE_DEALLOCATE_A(vv)
  SAFE_DEALLOCATE_A(psii)
  SAFE_DEALLOCATE_A(psij)
  SAFE_DEALLOCATE_A(psik)
  SAFE_DEALLOCATE_A(psil)

  POP_SUB(X(states_me_two_body))
end subroutine X(states_me_two_body)


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
