!! Copyright (C) 2002-2011 M. Marques, A. Castro, A. Rubio, G. Bertsch, M. Oliveira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!


! --------------------------------------------------------- 
!>  Writes a cube_function in VTK legacy format
!!  see http://www.vtk.org/VTK/img/file-formats.pdf
! ---------------------------------------------------------
subroutine X(vtk_out_cf)(filename, namespace, fieldname, ierr, cf_in, cube, spacing, unit)
  character(len=*),      intent(in) :: filename        !< the file name
  type(namespace_t),     intent(in) :: namespace       !< namespace for directory
  character(len=*),      intent(in) :: fieldname       !< the name of the field
  integer,               intent(out):: ierr            !< error message   
  type(cube_function_t), intent(in) :: cf_in           !< the cube_function to be written 
  type(cube_t),          intent(in) :: cube            !< the underlying cube mesh
  FLOAT,                 intent(in) :: spacing(:)      !< the mesh spacing already converted to units_out
  type(unit_t),          intent(in) :: unit            !< unit of data in cf

  integer :: iunit, np
  type(cube_function_t) :: cf_out
  integer :: i1,i2,i3

  PUSH_SUB(X(vtk_out_cf))

  np = cube%rs_n_global(1)*cube%rs_n_global(2)*cube%rs_n_global(3)

  iunit = io_open(trim(filename), namespace, action='write')

  write(iunit, '(1a)') '# vtk DataFile Version 2.0 '

  write(iunit, '(6a)') 'Generated by octopus ', trim(conf%version), ' -  git: ', & 
    trim(conf%git_commit), " build: ",  trim(conf%build_time)
  write(iunit, '(1a)') 'BINARY'
  write(iunit, '(1a)') 'DATASET STRUCTURED_POINTS'
  write(iunit, '(1a,3i5)') 'DIMENSIONS ', cube%rs_n_global(1), cube%rs_n_global(2), cube%rs_n_global(3) 
  write(iunit, '(1a,3f12.6)') 'ORIGIN ', - int(cube%rs_n_global(1)/2)* spacing(1),&
    - int(cube%rs_n_global(2)/2)* spacing(2),&
    - int(cube%rs_n_global(3)/2)* spacing(3)
  write(iunit, '(1a,3f12.6)') 'SPACING ', spacing(1), spacing(2), spacing(3)
  write(iunit, '(1a,1i9)') 'POINT_DATA ', np
#ifdef R_TCOMPLEX  
  write(iunit, '(3a)') 'SCALARS Re_'//trim(fieldname), ' double 1'
#else
  write(iunit, '(3a)') 'SCALARS ', trim(fieldname), ' double 1'
#endif
  write(iunit, '(1a)') 'LOOKUP_TABLE default'
  call io_close(iunit)


  call cube_function_null(cf_out)
  call X(cube_function_alloc_RS) (cube, cf_out)

  do i3 = 1, cube%rs_n_global(3)
    do i2 = 1, cube%rs_n_global(2)
      do i1 = 1, cube%rs_n_global(1)
        cf_out%X(RS)(i1, i2, i3) = units_from_atomic(unit, cf_in%X(RS)(i1, i2, i3))
      end do
    end do
  end do

  ! Paraview likes BigEndian binaries
  call io_binary_write(io_workpath(filename, namespace), np, R_REAL(cf_out%X(RS)(:,:,:)),&
    ierr, nohead = .true., fendian = is_little_endian())

#ifdef R_TCOMPLEX

  iunit = io_open(trim(filename), namespace, action='write', position='append')

  write(iunit, '(1a)') ' '
  write(iunit, '(3a)') 'SCALARS Im_'//trim(fieldname), ' double 1'
  write(iunit, '(1a)') 'LOOKUP_TABLE default'

  call io_close(iunit)

  call io_binary_write(io_workpath(filename, namespace), np, R_AIMAG(cf_out%X(RS)(:,:,:)),&
    ierr, nohead = .true., fendian = is_little_endian())

#endif

  call X(cube_function_free_RS)(cube, cf_out)


  POP_SUB(X(vtk_out_cf))


end subroutine X(vtk_out_cf)

! --------------------------------------------------------- 
!>  Writes a cube_function in VTK legacy format
!!  see http://www.vtk.org/VTK/img/file-formats.pdf
! ---------------------------------------------------------
subroutine X(vtk_out_cf_vector)(filename, namespace, fieldname, ierr, cf_in, vector_dim, cube, spacing, unit)
  character(len=*),      intent(in) :: filename        !< the file name
  type(namespace_t),     intent(in) :: namespace       !< namespace for directory
  character(len=*),      intent(in) :: fieldname       !< the name of the field
  integer,               intent(out):: ierr            !< error message   
  type(cube_function_t), intent(in) :: cf_in(:)        !< the cube_function to be written
  integer,               intent(in) :: vector_dim
  type(cube_t),          intent(in) :: cube            !< the underlying cube mesh
  FLOAT,                 intent(in) :: spacing(:)      !< the mesh spacing already converted to units_out
  type(unit_t),          intent(in) :: unit            !< unit of data in cf

  integer :: iunit, np
  integer :: i1, i2, i3, ivd
  FLOAT, allocatable :: cfout(:, :, :, :)

  PUSH_SUB(X(vtk_out_cf_vector))

  np = cube%rs_n_global(1)*cube%rs_n_global(2)*cube%rs_n_global(3)

  iunit = io_open(trim(filename), namespace, action='write')

  write(iunit, '(1a)') '# vtk DataFile Version 2.0 '

  write(iunit, '(6a)') 'Generated by octopus ', trim(conf%version), ' -  git: ', & 
    trim(conf%git_commit), " build: ",  trim(conf%build_time)
  write(iunit, '(1a)') 'BINARY'
  write(iunit, '(1a)') 'DATASET STRUCTURED_POINTS'
  write(iunit, '(1a,3i5)') 'DIMENSIONS ', cube%rs_n_global(1), cube%rs_n_global(2), cube%rs_n_global(3) 
  write(iunit, '(1a,3f12.6)') 'ORIGIN ', - int(cube%rs_n_global(1)/2)* spacing(1),&
    - int(cube%rs_n_global(2)/2)* spacing(2),&
    - int(cube%rs_n_global(3)/2)* spacing(3)
  write(iunit, '(1a,3f12.6)') 'SPACING ', spacing(1), spacing(2), spacing(3)
  write(iunit, '(1a,1i9)') 'POINT_DATA ', np

#ifdef R_TCOMPLEX  
  write(iunit, '(3a,i1)') 'SCALARS Re_'//trim(fieldname), ' double ', vector_dim
#else
  write(iunit, '(3a,i1)') 'SCALARS ', trim(fieldname), ' double ', vector_dim
#endif

  write(iunit, '(1a)') 'LOOKUP_TABLE default'
  call io_close(iunit)

  SAFE_ALLOCATE(cfout(1:vector_dim, 1:cube%rs_n_global(1), 1:cube%rs_n_global(2), 1:cube%rs_n_global(3)))

  do i3 = 1, cube%rs_n_global(3)
    do i2 = 1, cube%rs_n_global(2)
      do i1 = 1, cube%rs_n_global(1)
        do ivd = 1, vector_dim
          cfout(ivd, i1, i2, i3) = R_REAL(units_from_atomic(unit, cf_in(ivd)%X(RS)(i1, i2, i3)))
        end do
      end do
    end do
  end do

  ! Paraview likes BigEndian binaries
  call io_binary_write(io_workpath(filename, namespace), vector_dim*np, cfout(:, :, :, :), ierr, &
    nohead = .true., fendian = is_little_endian())

#ifdef R_TCOMPLEX

  iunit = io_open(trim(filename), namespace, action='write', position='append')

  write(iunit, '(1a)') ' '
  write(iunit, '(3a,i1)') 'SCALARS Im_'//trim(fieldname), ' double ', vector_dim
  write(iunit, '(1a)') 'LOOKUP_TABLE default'

  call io_close(iunit)

  do ivd = 1, vector_dim
    do i3 = 1, cube%rs_n_global(3)
      do i2 = 1, cube%rs_n_global(2)
        do i1 = 1, cube%rs_n_global(1)
          cfout(ivd, i1, i2, i3) = R_AIMAG(units_from_atomic(unit, cf_in(ivd)%X(RS)(i1, i2, i3)))
        end do
      end do
    end do
  end do

  call io_binary_write(io_workpath(filename, namespace), vector_dim*np, cfout(:, :, :, :), ierr, &
    nohead = .true., fendian = is_little_endian())

#endif

  SAFE_DEALLOCATE_A(cfout)

  POP_SUB(X(vtk_out_cf_vector))
end subroutine X(vtk_out_cf_vector)

! --------------------------------------------------------- 
!>  Writes a mesh_function in VTK legacy format
!!  for structured grid data (i.e. curvilinear coordinates).
!!  The data is defined on a mesh that is topologically 
!!  equivalent to a cubic grid. In other words, it must be 
!!  laid on a cubic array but the value of each point 
!!  coordinate is free in space. 
!!    
! ---------------------------------------------------------
subroutine X(vtk_out_cf_structured)(filename, namespace, fieldname, ierr, cf_in, cube, unit, points, ascii)
  character(len=*),      intent(in) :: filename        !< the file name
  type(namespace_t),     intent(in) :: namespace       !< namespace for directory
  character(len=*),      intent(in) :: fieldname       !< the name of the field
  integer,               intent(out):: ierr            !< error message   
  type(cube_function_t), intent(in) :: cf_in           !< the cube_function to be written 
  type(cube_t),          intent(in) :: cube            !< the underlying cube mesh
  type(unit_t),          intent(in) :: unit            !< unit of data in cf
  FLOAT,                 intent(in) :: points(:,:,:,:) !< points(x,y,z,1:3): point coordinates in units_out     
  logical, optional,     intent(in) :: ascii

  integer :: iunit
  type(cube_function_t) :: cf_out
  integer :: i1,i2,i3,np

  FLOAT, allocatable :: pnts1(:,:,:,:)

  PUSH_SUB(X(vtk_out_cf_structured))

  np =  product(cube%rs_n_global(1:3))

  iunit = io_open(trim(filename), namespace, action='write')

  if (optional_default(ascii, .false.)) then 
    ! ASCII
    write(iunit, '(1a)') '# vtk DataFile Version 2.0 '

    write(iunit, '(6a)') 'Generated by octopus ', trim(conf%version), ' -  git: ', & 
      trim(conf%git_commit), " build: ",  trim(conf%build_time)
    write(iunit, '(1a)') 'ASCII'
    write(iunit, '(1a)') 'DATASET STRUCTURED_GRID'
    write(iunit, '(1a,3i5)') 'DIMENSIONS', cube%rs_n_global(1), cube%rs_n_global(2), cube%rs_n_global(3) 
    write(iunit, '(1a,1i9,1a)') 'POINTS ', np , ' FLOAT'

    do i3 =1 , cube%rs_n_global(3)
      do i2 =1 , cube%rs_n_global(2)
        do i1 =1 , cube%rs_n_global(1)
          write(iunit, '(3f12.6)') points(i1,i2,i3,1), points(i1,i2,i3,2), points(i1,i2,i3,3)
        end do
      end do
    end do

    write(iunit, '(1a)') ' '
    write(iunit, '(1a,1i9)') 'POINT_DATA ', np
#ifdef R_TCOMPLEX
    write(iunit, '(3a)') 'SCALARS Re_'//trim(fieldname), ' double 1'
#else
    write(iunit, '(3a)') 'SCALARS ', trim(fieldname), ' double 1'
#endif
    write(iunit, '(1a)') 'LOOKUP_TABLE default'

    call cube_function_null(cf_out)
    call X(cube_function_alloc_RS) (cube, cf_out)

    do i3 = 1, cube%rs_n_global(3)
      do i2 = 1, cube%rs_n_global(2)
        do i1 = 1, cube%rs_n_global(1)
          cf_out%X(RS)(i1, i2, i3) = units_from_atomic(unit, cf_in%X(RS)(i1, i2, i3))
          write(iunit, '(1f12.6)') R_REAL(cf_out%X(RS)(i1, i2, i3))
        end do
      end do
    end do

    call io_close(iunit)

  else 
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    ! binary
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


    write(iunit, '(1a)') '# vtk DataFile Version 3.0 '

    write(iunit, '(6a)') 'Generated by octopus ', trim(conf%version), ' -  git: ', & 
      trim(conf%git_commit), " build: ",  trim(conf%build_time)
    write(iunit, '(1a)') 'BINARY'
    write(iunit, '(1a)') 'DATASET STRUCTURED_GRID'
    write(iunit, '(1a,3i5)') 'DIMENSIONS', cube%rs_n_global(1), cube%rs_n_global(2), cube%rs_n_global(3) 
    write(iunit, '(1a,1i9,1a)') 'POINTS ', np , ' double'
    call io_close(iunit)

    SAFE_ALLOCATE(pnts1(1:3,1:cube%rs_n_global(1),1:cube%rs_n_global(2),1:cube%rs_n_global(3)))
    do i3 =1 , cube%rs_n_global(3)
      do i2 =1 , cube%rs_n_global(2)
        do i1 =1 , cube%rs_n_global(1)
          pnts1(1:3,i1,i2,i3) = points(i1,i2,i3,1:3)
        end do
      end do
    end do

    call io_binary_write(io_workpath(filename, namespace), np*3, pnts1(:,:,:,:) ,&
      ierr, nohead = .true., fendian = is_little_endian())

    SAFE_DEALLOCATE_A(pnts1)

    iunit = io_open(trim(filename), namespace, action='write', position='append')
    write(iunit, '(1a)') ' '
    write(iunit, '(1a,1i9)') 'POINT_DATA ', np
#ifdef R_TCOMPLEX
    write(iunit, '(3a)') 'SCALARS Re_'//trim(fieldname), ' double 1'
#else
    write(iunit, '(3a)') 'SCALARS ', trim(fieldname), ' double 1'
#endif
    write(iunit, '(1a)') 'LOOKUP_TABLE default'
    call io_close(iunit)

    call cube_function_null(cf_out)
    call X(cube_function_alloc_RS) (cube, cf_out)

    do i3 = 1, cube%rs_n_global(3)
      do i2 = 1, cube%rs_n_global(2)
        do i1 = 1, cube%rs_n_global(1)
          cf_out%X(RS)(i1, i2, i3) = units_from_atomic(unit, cf_in%X(RS)(i1, i2, i3))
        end do
      end do
    end do


    call io_binary_write(io_workpath(filename, namespace), np, R_REAL(cf_out%X(RS)(:,:,:)),&
      ierr, nohead = .true., fendian = is_little_endian())


#ifdef R_TCOMPLEX

    iunit = io_open(trim(filename), namespace, action='write', position='append')
    write(iunit, '(1a)') ' '
    write(iunit, '(3a)') 'SCALARS Im_'//trim(fieldname), ' double 1'
    write(iunit, '(1a)') 'LOOKUP_TABLE default'
    call io_close(iunit)

    call io_binary_write(io_workpath(filename, namespace), np, R_AIMAG(cf_out%X(RS)(:,:,:)),&
      ierr, nohead = .true., fendian = is_little_endian())

#endif

  end if


  call X(cube_function_free_RS)(cube, cf_out)


  POP_SUB(X(vtk_out_cf_structured))


end subroutine X(vtk_out_cf_structured)

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
