!! Copyright (C) 2016 X. Andrade
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module debug_oct_m
  use global_oct_m
  use namespace_oct_m
  use parser_oct_m

  implicit none
  
  private
  public ::             &
    debug_t,            &
    debug_init,         &
    debug_enable,       &
    debug_disable

  type debug_t
    private
    logical, public :: info
    logical, public :: trace
    logical, public :: trace_term
    logical, public :: trace_file
    logical :: extra_checks
    logical, public :: interaction_graph
    integer :: bits    
  end type debug_t

contains
  
  subroutine debug_init(this, namespace)
    type(debug_t),     intent(out)   :: this
    type(namespace_t), intent(in)    :: namespace

    !%Variable Debug
    !%Type flag
    !%Default no
    !%Section Execution::Debug
    !%Description
    !% This variable controls the amount of debugging information
    !% generated by Octopus. You can use include more than one option
    !% with the + operator.    
    !%Option no 0
    !% (default) <tt>Octopus</tt> does not enter debug mode.
    !%Option info 1
    !% Octopus prints additional information to the terminal.
    !%Option trace 2
    !% Octopus generates a stack trace as it enters end exits
    !% subroutines. This information is reported if Octopus stops with
    !% an error.
    !%Option trace_term 4
    !% The trace is printed to the terminal as Octopus enters or exits subroutines. This slows down execution considerably.
    !%Option trace_file 8
    !% The trace is written to files in the <tt>debug</tt>
    !% directory. For each node (when running in parallel) there is a file called
    !% <tt>debug_trace.&lt;rank&gt;</tt>. Writing these files slows down the code by a huge factor and
    !% it is usually only necessary for parallel runs.
    !%Option extra_checks 16
    !% This enables Octopus to perform some extra checks, to ensure
    !% code correctness, that might be too costly for regular runs.
    !%Option interaction_graph 32
    !% Octopus generates a dot file containing the graph for a multisystem run.
    !%End
    call parse_variable(namespace, 'Debug', OPTION__DEBUG__NO, this%bits)

    call from_bits(this)
    
  end subroutine debug_init

  !--------------------------------------------------
  
  subroutine debug_enable(this)
    type(debug_t), intent(inout) :: this
    
    this%info       = .true.
    this%trace      = .true.
    this%trace_term = .true.
    this%trace_file = .true.
    this%interaction_graph = .true.
    
  end subroutine debug_enable

  !--------------------------------------------------
  
  subroutine debug_disable(this)
    type(debug_t), intent(inout) :: this
    
    call from_bits(this)
    
  end subroutine debug_disable

  !--------------------------------------------------

  subroutine from_bits(this)
    type(debug_t), intent(inout) :: this
    
    this%info         = (bitand(this%bits, OPTION__DEBUG__INFO)         /= 0)
    this%trace_term   = (bitand(this%bits, OPTION__DEBUG__TRACE_TERM)   /= 0)
    this%trace_file   = (bitand(this%bits, OPTION__DEBUG__TRACE_FILE)   /= 0)
    this%trace        = (bitand(this%bits, OPTION__DEBUG__TRACE)        /= 0) .or. this%trace_term .or. this%trace_file
    this%extra_checks = (bitand(this%bits, OPTION__DEBUG__EXTRA_CHECKS) /= 0) .or. this%trace_term .or. this%trace_file
    this%interaction_graph = (bitand(this%bits, OPTION__DEBUG__INTERACTION_GRAPH)   /= 0)

  end subroutine from_bits
  
end module debug_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
