//
// TM & (c) 2017 Lucasfilm Entertainment Company Ltd. and Lucasfilm Ltd.
// All rights reserved.  See LICENSE.txt for license.
//

#include <MaterialXTest/Catch/catch.hpp>

#include <MaterialXFormat/File.h>
#include <MaterialXFormat/Util.h>

namespace mx = MaterialX;

TEST_CASE("Syntactic operations", "[file]")
{
    using InputPair = std::pair<std::string, mx::FilePath::Format>;
    std::vector<InputPair> inputPairs =
    {
        {"D:\\Assets\\Materials\\Robot.mtlx", mx::FilePath::FormatWindows},
        {"\\\\Show\\Assets\\Materials\\Robot.mtlx", mx::FilePath::FormatWindows},
        {"Materials\\Robot.mtlx", mx::FilePath::FormatWindows},
        {"/Assets/Materials/Robot.mtlx", mx::FilePath::FormatPosix},
        {"Assets/Materials/Robot.mtlx", mx::FilePath::FormatPosix},
        {"Materials/Robot.mtlx", mx::FilePath::FormatPosix}
    };

    for (const InputPair& pair : inputPairs)
    {
        mx::FilePath path(pair.first);
        REQUIRE(path.asString(pair.second) == pair.first);
    }
}

TEST_CASE("File system operations", "[file]")
{
    mx::StringVec filenames =
    {
        "libraries/stdlib/stdlib_defs.mtlx",
        "resources/Materials/Examples/Syntax/MaterialBasic.mtlx",
        "resources/Materials/Examples/Syntax/PaintMaterials.mtlx",
    };

    for (const std::string& filename : filenames)
    {
        mx::FilePath path(filename);
        REQUIRE(path.exists());
        REQUIRE(mx::FileSearchPath().find(path).exists());
    }

    mx::FilePath currentPath = mx::FilePath::getCurrentPath();
    mx::FilePath modulePath = mx::FilePath::getModulePath();
    bool expectedPaths = currentPath == modulePath ||
                         currentPath == modulePath.getParentPath();
    REQUIRE(expectedPaths);
}

TEST_CASE("File search path operations", "[file]")
{
    mx::FileSearchPath searchPath = "libraries/stdlib" + 
                                    mx::PATH_LIST_SEPARATOR + 
                                    "resources/Materials/Examples/Syntax";

    mx::FilePathVec filenames =
    {
        "stdlib_defs.mtlx",
        "MaterialBasic.mtlx",
        "PaintMaterials.mtlx",
    };

    for (const mx::FilePath& filename : filenames)
    {
        REQUIRE(searchPath.find(filename).exists());
    }
}

TEST_CASE("Flatten filenames", "[file]")
{
    const mx::FilePath TEST_FILE_PREFIX_STRING("resources\\Images\\");
    const mx::FilePath TEST_IMAGE_STRING1("brass_roughness.jpg");
    const mx::FilePath TEST_IMAGE_STRING2("brass_color.jpg");

    mx::DocumentPtr doc1 = mx::createDocument();

    // Set up document
    mx::NodeGraphPtr nodeGraph = doc1->addNodeGraph();
    nodeGraph->setFilePrefix(TEST_FILE_PREFIX_STRING.asString() + "\\"); // Note this is required as filepath->string strips out last separator
    mx::NodePtr image1 = nodeGraph->addNode("image");
    image1->setInputValue("file", "brass_roughness.jpg", mx::FILENAME_TYPE_STRING);
    mx::NodePtr image2 = nodeGraph->addNode("image");
    image2->setInputValue("file", "brass_color.jpg", mx::FILENAME_TYPE_STRING);

    // 1. Test resolving fileprefix
    mx::flattenFilenames(doc1);
    REQUIRE(nodeGraph->getFilePrefix() == mx::EMPTY_STRING);
    mx::FilePath resolvedPath(image1->getInputValue("file")->getValueString());
    REQUIRE(resolvedPath == (TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING1));
    resolvedPath = image2->getInputValue("file")->getValueString();
    REQUIRE(resolvedPath == (TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING2));

    // Reset document
    nodeGraph->setFilePrefix(TEST_FILE_PREFIX_STRING.asString() + "\\");
    image1->setInputValue("file", "brass_roughness.jpg", mx::FILENAME_TYPE_STRING);
    image2->setInputValue("file", "brass_color.jpg", mx::FILENAME_TYPE_STRING);

    // 2. Test resolving to absolute paths
    mx::FilePath rootPath(mx::FilePath::getCurrentPath());

    mx::FileSearchPath searchPath;
    searchPath.append(rootPath);

    mx::flattenFilenames(doc1, searchPath);    
    CHECK(nodeGraph->getFilePrefix() == mx::EMPTY_STRING);
    resolvedPath = image1->getInputValue("file")->getValueString();
    CHECK(resolvedPath.asString() == (rootPath / TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING1).asString());
    resolvedPath = image2->getInputValue("file")->getValueString();
    CHECK(resolvedPath.asString() == (rootPath / TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING2).asString());

    // Reset document
    nodeGraph->setFilePrefix(TEST_FILE_PREFIX_STRING.asString() + "\\");
    image1->setInputValue("file", "brass_roughness.jpg", mx::FILENAME_TYPE_STRING);
    image2->setInputValue("file", "brass_color.jpg", mx::FILENAME_TYPE_STRING);

    // 3. Test with additional resolvers
    // - Create resolver to replace all Windows separators with POSIX ones
    mx::StringResolverPtr separatorReplacer = mx::StringResolver::create();
    separatorReplacer->setFilenameSubstitution("\\\\", "/");
    separatorReplacer->setFilenameSubstitution("\\", "/");

    mx::flattenFilenames(doc1, searchPath, separatorReplacer);
    CHECK(nodeGraph->getFilePrefix() == mx::EMPTY_STRING);
    std::string resolvedPathString = image1->getInputValue("file")->getValueString();
    CHECK(resolvedPathString == (rootPath / TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING1).asString(mx::FilePath::FormatPosix));
    resolvedPathString = image2->getInputValue("file")->getValueString();
    CHECK(resolvedPathString == (rootPath / TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING2).asString(mx::FilePath::FormatPosix));

    // 4. Test with pre-resolved filenames
    nodeGraph->setFilePrefix(TEST_FILE_PREFIX_STRING.asString() + "\\");
    mx::flattenFilenames(doc1, searchPath, separatorReplacer);
    CHECK(nodeGraph->getFilePrefix() == mx::EMPTY_STRING);
    resolvedPathString = image1->getInputValue("file")->getValueString();
    CHECK(resolvedPathString == (rootPath / TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING1).asString(mx::FilePath::FormatPosix));
    resolvedPathString = image2->getInputValue("file")->getValueString();
    CHECK(resolvedPathString == (rootPath / TEST_FILE_PREFIX_STRING / TEST_IMAGE_STRING2).asString(mx::FilePath::FormatPosix));
}
