#include <framecpp_config.h>

#include "framecpp/Common/IOStream.hh"
#include "framecpp/Common/Description.hh"
#include "framecpp/Common/SearchContainer.hh"
#include "framecpp/Common/FrameSpec.tcc"

#include "framecpp/Version8/FrEvent.hh"
#include "framecpp/Version8/FrSE.hh"
#include "framecpp/Version8/FrSH.hh"

#include "framecpp/Version8/PTR_STRUCT.hh"

#include "Common/ComparePrivate.hh"

using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;

#define LM_DEBUG 0

//=======================================================================
// Static
//=======================================================================

static const FrameSpec::Info::frame_object_types s_object_id
= FrameSpec::Info::FSI_FR_EVENT;

namespace FrameCPP
{
  namespace Version_8
  {
    //=======================================================================
    // FrEvent
    //=======================================================================
    FrEvent::
    FrEvent( )
      : object_type( s_object_id, StructDescription( ) )
    {
    }

    FrEvent::
    FrEvent( const FrEvent& Source )
      : object_type( s_object_id, StructDescription( ) ),
	FrEventStorage( Source ),
	FrEventRefs( Source ),
	Common::TOCInfo( Source )
    {
    }

    FrEvent::
    FrEvent( const name_type& name,
	     const comment_type& comment,
	     const inputs_type& inputs,
	     const time_type& time,
	     const timeBefore_type timeBefore,
	     const timeAfter_type timeAfter,
	     const eventStatus_type eventStatus,
	     const amplitude_type amplitude,
	     const probability_type prob,
	     const statistics_type& statistics,
	     const ParamList_type& parameters )
      : object_type( s_object_id, StructDescription( ) )
    {
      m_data.name = name;
      m_data.comment = comment;
      m_data.inputs = inputs;
      m_data.GTime = time;
      m_data.timeBefore = timeBefore;
      m_data.timeAfter = timeAfter;
      m_data.eventStatus = eventStatus;
      m_data.amplitude = amplitude;
      m_data.probability = prob;
      m_data.statistics = statistics;
      m_data.Params = parameters;
    }

    FrEvent::
    FrEvent( const Previous::FrEvent& Source,
	     istream_type* Stream )
      : object_type( s_object_id, StructDescription( ) ),
	FrEventStorage( Source )
    {
      if ( Stream )
      {
	//-------------------------------------------------------------------
	// Modify references
	//-------------------------------------------------------------------
	Stream->ReplaceRef( RefData( ), Source.RefData( ),
			    Previous::FrEvent::MAX_REF );
	Stream->ReplaceRef( RefTable( ), Source.RefTable( ),
			    Previous::FrEvent::MAX_REF );
      }
    }

    FrEvent::
    FrEvent( istream_type& Stream )
      : object_type( s_object_id, StructDescription( ) )
    {
      m_data( Stream );
      m_refs( Stream );

      Stream.Next( this );
    }

    const std::string& FrEvent::
    GetNameSlow( ) const
    {
      return GetName( );
    }

    FrEvent& FrEvent::
    Merge( const FrEvent& RHS )
    {
      throw
	Unimplemented( "FrEvent& FrEvent::Merge( const FrEvent& RHS )",
		       DATA_FORMAT_VERSION, __FILE__, __LINE__ );
      return *this;
    }

    const char* FrEvent::
    ObjectStructName( ) const
    {
      return StructName( );
    }

    const Description* FrEvent::
    StructDescription( )
    {
      static Description ret;

      if ( ret.size( ) == 0 )
      {
	ret( FrSH( FrEvent::StructName( ), s_object_id,
		   "Event Data Structure" ) );

	ret( FrSE( "name", "STRING",
		   "Name of event." ) );
	ret( FrSE( "comment", "STRING",
		   "Descriptor of event" ) );
	ret( FrSE( "inputs", "STRING",
		   "Input channels and filter parameters to event process." ) );
	ret( FrSE( "GTimeS", "INT_4U",
		   "GPS time in seconds corresponding to reference vale of event,"
		   " as defined by the search algorigthm.") );
	ret( FrSE( "GTimeN", "INT_4U",
		   "GPS time in residual nanoseconds relative to GTimeS") );
	ret( FrSE( "timeBefore", "REAL_4",
		   "Signal duration before (GTimeS.GTimeN)(seconds)") );
	ret( FrSE( "timeAfter", "REAL_4",
		   "Signal duration after (GTimeS.GTimeN)(seconds)" ) );
	ret( FrSE( "eventStatus", "INT_4U",
		   "Defined bby event search algorithm") );
	ret( FrSE( "amplitude", "REAL_4",
		   "Continuouis output amplitude returned by event" ) );
	ret( FrSE( "probability", "REAL_4",
		   "Likelihood estimate of event, if available"
		   " (probability = -1 if cannot be estimated)") );
	ret( FrSE( "statistics", "STRING",
		   "Statistical description of event, if relevant or available") );
	ret( FrSE( "nParam", "INT_2U",
		   "Number of additional event parameters" ) );
	ret( FrSE( "parameters", "REAL_8[nParam]",
		   "Array of additional event paraameters(size of nParam)") );
	ret( FrSE( "parameterNames", "STRING[nParam]",
		   "Array of parameter names (size of nParam)." ) );
	ret( FrSE( "data", PTR_STRUCT::Desc( FrVect::StructName( ) ),
		   "" ) );
	ret( FrSE( "table", PTR_STRUCT::Desc( FrTable::StructName() ),
		   "" ) );
	ret( FrSE( "next", PTR_STRUCT::Desc( FrEvent::StructName( ) ),
		   "" ) );

	ret( FrSE( "chkSum", CheckSumDataClass( ), CheckSumDataComment( ) ) );
      }

      return &ret;
    }

    void FrEvent::
#if WORKING_VIRTUAL_TOCQUERY
    TOCQuery( int InfoClass, ... ) const
#else /*  WORKING_VIRTUAL_TOCQUERY */
    vTOCQuery( int InfoClass, va_list vl ) const
#endif /*  WORKING_VIRTUAL_TOCQUERY */
    {
      using Common::TOCInfo;

#if WORKING_VIRTUAL_TOCQUERY
      va_list	vl;
      va_start( vl, InfoClass );
#endif /*  WORKING_VIRTUAL_TOCQUERY */

      while ( InfoClass != TOCInfo::IC_EOQ )
      {
	int data_type = va_arg( vl, int );
	switch( data_type )
	{
	case TOCInfo::DT_STRING_2:
	  {
	    STRING* data = va_arg( vl, STRING* );
	    switch( InfoClass )
	    {
	    case TOCInfo::IC_NAME:
	      *data = GetName( );
	      break;
	    default:
	      goto cleanup;
	      break;
	    }
	  }
	  break;
	case TOCInfo::DT_INT_4U:
	  {
	    INT_4U* data = va_arg( vl, INT_4U* );
	    switch( InfoClass )
	    {
	    case TOCInfo::IC_GTIME_S:
	      *data = GetGTime( ).GetSeconds( );
	      break;
	    case TOCInfo::IC_GTIME_N:
	      *data = GetGTime( ).GetNanoseconds( );
	      break;
	    default:
	      goto cleanup;
	      break;
	    }
	  }
	  break;
	case TOCInfo::DT_REAL_4:
	  {
	    REAL_4* data = va_arg( vl, REAL_4* );
	    switch( InfoClass )
	    {
	    case TOCInfo::IC_AMPLITUDE:
	      *data = GetAmplitude( );
	      break;
	    default:
	      goto cleanup;
	      break;
	    }
	  }
	  break;
	case TOCInfo::DT_REAL_8:
	  {
	    REAL_8* data = va_arg( vl, REAL_8* );
	    switch( InfoClass )
	    {
	    case TOCInfo::IC_AMPLITUDE:
	      *data = GetAmplitude( );
	      break;
	    default:
	      goto cleanup;
	      break;
	    }
	  }
	  break;
	default:
	  // Stop processing
	  goto cleanup;
	}
	InfoClass = va_arg( vl, int );
      }
    cleanup:
#if WORKING_VIRTUAL_TOCQUERY
      va_end( vl )
#endif /*  WORKING_VIRTUAL_TOCQUERY */
	;
    }

    bool FrEvent::
    operator==( const Common::FrameSpec::Object& Obj ) const
    {
      return Common::Compare( *this, Obj );
    }

    FrEvent::demote_ret_type FrEvent::
    demote( INT_2U Target,
	    demote_arg_type Obj,
	    istream_type* Stream ) const
    {
      if ( Target >= DATA_FORMAT_VERSION )
      {
	return Obj;
      }
      try
      {
	//-------------------------------------------------------------------
	// Copy non-reference information
	//-------------------------------------------------------------------
	Previous::FrEvent::ParamList_type params;
	for ( ParamList_type::const_iterator
		cur = GetParam( ).begin( ),
		last = GetParam( ).end( );
	      cur != last;
	      ++cur )
	{
	  params.push_back( Previous::FrEvent::Param_type( cur->first,
							   cur->second ) );
	}
	// Do actual down conversion
	LDASTools::AL::SharedPtr< Previous::FrEvent >
	  retval( new Previous::FrEvent( GetName( ),
					 GetComment( ),
					 GetInputs( ),
					 GetGTime( ),
					 GetTimeBefore( ),
					 GetTimeAfter( ),
					 GetEventStatus( ),
					 GetAmplitude( ),
					 GetProbability( ),
					 GetStatistics( ),
					 params
					 ) )
	  ;
	if ( Stream )
	{
	  //-----------------------------------------------------------------
	  // Modify references
	  //-----------------------------------------------------------------
	  Stream->ReplaceRef( retval->RefData( ), RefData( ), MAX_REF );
	  Stream->ReplaceRef( retval->RefTable( ), RefTable( ), MAX_REF );
	}
	//-------------------------------------------------------------------
	// Return demoted object
	//-------------------------------------------------------------------
	return retval;
      }
      catch( ... )
      {
      }
      throw
	Unimplemented( "Object* FrEvent::demote( Object* Obj ) const",
		       DATA_FORMAT_VERSION, __FILE__, __LINE__ );
    }

    FrEvent::promote_ret_type FrEvent::
    promote( INT_2U Target,
	     promote_arg_type Obj,
	     istream_type* Stream ) const
    {
      return Promote( Target, Obj, Stream );
    }

    inline FrameCPP::cmn_streamsize_type FrEvent::
    pBytes( const Common::StreamBase& Stream ) const
    {
      return
	m_data.Bytes( )
	+ m_refs.Bytes( Stream )
	+ Stream.PtrStructBytes( )	// next
	;
    }

    FrEvent* FrEvent::
    pCreate( istream_type& Stream ) const
    {
      return new FrEvent( Stream );
    }

    void FrEvent::
    pWrite( ostream_type& Stream ) const
    {
      m_data( Stream );
      m_refs( Stream );
      WriteNext( Stream );
    }
  } // namespace - Version_8
} // namespace - FrameCPP
