#ifndef FrameCPP_VERSION_4_FrVect_HH
#define FrameCPP_VERSION_4_FrVect_HH

#if ! defined(SWIGIMPORTED)
#include <memory>
#include <vector>
#include <string>
#endif /* ! defined(SWIGIMPORTED) */

#include "ldastoolsal/types.hh"
#include "ldastoolsal/SharedArray.hh"

#include "framecpp/Common/Compression.hh"
#include "framecpp/Common/FrVect.hh"

#include "framecpp/Version3/FrVect.hh"

#include "framecpp/Version4/FrameSpec.hh"
#include "framecpp/Version4/Dimension.hh"
#include "framecpp/Version4/GPSTime.hh"
#include "framecpp/Version4/STRING.hh"

namespace FrameCPP
{
  namespace Version_4
  {
    using Previous::FrVectDataTypes;

    class FrVectCompressionScheme
    {
    public:
      enum compression_scheme_type {

#if WORDS_BIGENDIAN
#define NATIVE_DECL(x) x = BIGENDIAN_##x
#else
#define NATIVE_DECL(x) x = LITTLEENDIAN_##x
#endif
#define CST_DECL(x,y) \
	BIGENDIAN_##x = y, \
	LITTLEENDIAN_##x = y + 256, \
	NATIVE_DECL(x)

	CST_DECL(RAW,0),
	CST_DECL(GZIP,1),
	CST_DECL(DIFF,2),
        CST_DECL(DIFF_GZIP,3),
	CST_DECL(ZERO_SUPPRESS_SHORT,5),

#undef CST_DECL
#undef NATIVE_DECL

	//---------------------------------------------------------
	// Declaration of meta modes
	//---------------------------------------------------------
	ZERO_SUPPRESS_SHORT_GZIP_OTHER = 6
      };
    };

    class FrVect
      : public Common::FrameSpec::Object,
        public Common::FrVect,
        public FrVectDataTypes,
	public FrVectCompressionScheme
    {
    public:
      // using FrameCPP::Compression;

      typedef INT_2U					compress_type;
      typedef INT_2U					type_type;
      typedef INT_4U					nData_type;
      typedef INT_4U					nBytes_type;
      typedef LDASTools::AL::SharedArray< CHAR_U >	data_type;
      typedef INT_4U					nDim_type;
      typedef std::unique_ptr< FrVect >			subfrvect_type;


      static const int DEFAULT_GZIP_LEVEL;

      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj, istream_type* Stream );

      static const char* StructName( );

      static const Common::Description* StructDescription( );

      //: Default constructor
      FrVect( );

      //-----------------------------------------------------------------
      //
      //: Copy Constructor.  
      //
      //!param: const Vect& vect - Copied vector.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      FrVect( const FrVect& vect );

      //-----------------------------------------------------------------
      //
      //: Constructor.
      //
      //!param: const std::string& name - The name of the data.
      //!param: INT_2U type - The data type.
      //!param: INT_4U nDim - The number of dimensions.
      //!param: const Dimension* dims - A pointer to 'nDim' Dimension objects
      //+       containing information about the dimensions for this data.
      //!param: const byte_order_type byte_order - Byte order of the data. Default
      //+       is BIGENDIAN.   
      //!param: const void* data - A pointer to the data.  Default: 0
      //!param: const std::string& unitY - Units for the data.  Default: ""
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //    
      FrVect( const std::string& name, type_type type, nDim_type nDim,
	      const Dimension* dims,
	      const byte_order_type byte_order = BYTE_ORDER_HOST,
	      const void* data = 0, 
	      const std::string& unitY = "" );

      //-----------------------------------------------------------------
      //
      //: Constructor.
      //
      //!param: const std::string& name - The name of the data.
      //!param: INT_2U type - The data type.
      //!param: INT_4U nDim - The number of dimensions.
      //!param: const Dimension* dims - A pointer to 'nDim' Dimension objects
      //+     containing information about the dimensions for this data.
      //!param: const byte_order_type byte_order - Byte order of the data.
      //+	Default is BIGENDIAN.      
      //!param: const void* data - A pointer to the data.  Default: 0
      //!param: const std::string& unitY - Units for the data.  Default: ""
      //!param: bool allocate - Whether to allocate data. Default: true.
      //!param: bool owns - Whether to destruct data in the destructor.
      //+	Default: true.
      //
      //!exc: std::bad_alloc - Memory allocation failed.
      //
      FrVect( const std::string& name, type_type type, nDim_type nDim,
	      const Dimension* dims,
	      const byte_order_type byte_order = BYTE_ORDER_HOST,
	      void* data = 0,
	      const std::string& unitY = "", 
	      bool allocate = true, bool owns = true );

      FrVect( const std::string& name,
	      compress_type Compress,
	      type_type type,
	      nDim_type nDim,
	      const Dimension* dims,
	      nData_type NData,
	      nBytes_type NBytes,
	      void* data,
	      const std::string& unitY = "", 
	      bool allocate = true,
	      bool owns = true );

      FrVect( const std::string& name,
	      compress_type Compress,
	      type_type type,
	      nDim_type nDim,
	      const Dimension* dims,
	      nData_type NData,
	      nBytes_type NBytes,
	      data_type data,
	      const std::string& unitY = "" );

      //!exc: std::bad_alloc - Memory allocation failed.   
      template< class T >
      FrVect( const std::string& name, nDim_type nDim = 1,
	      const Dimension* dims = 0,
	      const T* data = 0, const std::string& unitY = "" );

      //!exc: std::bad_alloc - Memory allocation failed.   
      template< class T >
      FrVect( const std::string& name, nDim_type nDim = 1,
	      const Dimension* dims = 0,
	      T* data = 0, const std::string& unitY = "",
	      bool allocate = true, bool owns = true );

      FrVect( const Previous::FrVect& Source,
	      istream_type* Stream );

      virtual ~FrVect( );

      virtual const char* ObjectStructName( ) const;

      //: Comparison operator
      bool operator==( const FrVect& RHS ) const;
      bool operator!=( const FrVect& RHS ) const;
      //: Concatination operator
      FrVect& operator+=( const FrVect& RHS );

      /// \brief comparison operator
      virtual bool operator==( const Common::FrameSpec::Object& Obj ) const;

      //: Number of bytes needed to write this structure
      FrameCPP::cmn_streamsize_type Bytes( const Common::StreamBase& Stream ) const;

      // Accessors
      const std::string& GetName( ) const;
      compress_type GetCompress( ) const;
      type_type GetType( ) const;
      nData_type GetNData( ) const;
      nBytes_type GetNBytes( ) const;
      data_type GetData( bool Decompress = true ) const;

      data_type GetData( bool Decompress = true );

      data_type GetDataRaw( ) const;

      data_type GetDataRaw( );

      template< class T > static INT_2U GetDataType();
      nDim_type GetNDim( ) const;
      const Dimension& GetDim( nDim_type Offset ) const;
      Dimension& GetDim( nDim_type Offset );
      const std::string GetUnitY( ) const;

      // Mutators
      void Compress( compression_scheme_type Scheme, int GZipLevel );
      void Uncompress( );

      virtual void CompressData( INT_4U Scheme, INT_2U GZipLevel );

      virtual Compression::compression_base_type
      Compression( ) const;

      virtual Common::FrameSpec::Object*
      CloneCompressed( cmn_compression_scheme_type Scheme,
		       cmn_compression_level_type Level ) const;

      //-----------------------------------------------------------------
      /// \brief Establish the channel name.
      ///
      /// \param[in] Name
      ///     The channel name
      //-----------------------------------------------------------------
      void SetName( const std::string& Name );

      void SetNData( INT_4U NData );

      size_t GetTypeSize( ) const;
      static size_t GetTypeSize( type_type type );

      //: Merge with another FrAdcData
      FrVect& Merge( const FrVect& RHS );

      subfrvect_type SubFrVect( INT_4U Start, INT_4U Stop ) const;

      virtual FrVect* Create( istream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Verify the integrity of the object
      ///
      /// \param Verifier
      ///     The handle to the Verify object used to store data that
      ///     needs to be accumulated before being processed.
      /// \param Stream
      ///     The input stream from which the object was created.
      ///
      /// \remarks
      ///     Upon failure, this method thows an exception.
      //-----------------------------------------------------------------
      virtual void VerifyObject( Common::Verify& Verifier,
				 Common::IFrameStream& Stream ) const;

      //: Write the structure to the stream
      virtual void Write( ostream_type& Stream ) const;

    protected:
      /// \brief Down grade an object
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      istream_type* Stream ) const;

      /// \brief Upgrade an object
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       istream_type* Stream ) const;

      void
      expandToBuffer( LDASTools::AL::AutoArray< CHAR_U >& Dest,
		      INT_8U& DestNBytes ) const;

    private:
      using Common::FrameSpec::Object::Create;

      struct data_container_type
      {
      public:

	STRING				name;
	compress_type			compress;
	type_type			type;
	nData_type			nData;
	nBytes_type			nBytes;
	data_type			data;
	std::vector< Dimension >	dimension;
	STRING				unitY;

        data_container_type();
        data_container_type( const std::string& n, type_type t, 
			     const byte_order_type byte_order, INT_4U ndim,
			     const Dimension* dims, const std::string& unit );

        data_container_type( const std::string& Name,
			     compress_type Compress,
			     type_type Type, 
			     nData_type NData,
			     nBytes_type NBytes,
			     void* Data,
			     nDim_type NDim,
			     const Dimension* Dims,
			     const std::string& UnitY,
			     bool Allocate );

        data_container_type( const std::string& Name,
			     compress_type Compress,
			     type_type Type, 
			     nData_type NData,
			     nBytes_type NBytes,
			     data_type Data,
			     nDim_type NDim,
			     const Dimension* Dims,
			     const std::string& UnitY );

	void copy_core( const data_container_type& Source );

	bool operator==( const data_container_type& RHS ) const;

      };
 
      data_container_type	m_data;

      FrVect( istream_type& Stream );

      void copy_data( const data_type::element_type* Data, INT_8U size );

    };

    inline Compression::compression_base_type FrVect::
    Compression( ) const
    {
      return ( 0x0FF & m_data.compress );
    }

    inline FrVect::promote_ret_type FrVect::
    Promote( INT_2U Source,
	     promote_arg_type Obj,
	     istream_type* Stream )
    {
      return Object::PromoteObject< Previous::FrVect, FrVect >
	( DATA_FORMAT_VERSION, Source, Obj, Stream );
    }

    inline const char* FrVect::
    StructName( )
    {
      static const CHAR* class_name( "FrVect" );
      return class_name;
    }

    inline FrVect::data_type FrVect::
    GetDataRaw( ) const
    {
      return m_data.data;
    }

    inline FrVect::data_type FrVect::
    GetDataRaw( )
    {
      return m_data.data;
    }

    inline FrVect::compress_type FrVect::
    GetCompress( ) const
    {
      return m_data.compress;
    }

    inline FrVect::type_type FrVect::
    GetType( ) const
    {
      return m_data.type;
    }

    inline size_t FrVect::
    GetTypeSize( ) const
    {
      return GetTypeSize( m_data.type );
    }

    inline FrVect::nBytes_type FrVect::
    GetNBytes( ) const
    {
      return m_data.nBytes;
    }

    inline FrVect::nDim_type FrVect::GetNDim( ) const
    {
      return m_data.dimension.size( );
    }

    inline const Dimension& FrVect::
    GetDim( nDim_type Offset ) const
    {
      //:TODO: Throw exception if value out of range
      return m_data.dimension[ Offset ];
    }

    inline Dimension& FrVect::
    GetDim( nDim_type Offset )
    {
      //:TODO: Throw exception if value out of range
      return m_data.dimension[ Offset ];
    }

    inline const std::string FrVect::
    GetUnitY( ) const
    {
      return m_data.unitY;
    }

    //-------------------------------------------------------------------
    /// Assign a new value for the channel name.
    //-------------------------------------------------------------------
    inline void FrVect::
    SetName( const std::string& Name )
    {
      m_data.name = Name;
    }

    inline void FrVect::
    SetNData( nData_type NData )
    {
      m_data.nData = NData;
    }
  } // namespace - Version_4
} // namespace FrameCPP

#if defined( __SUNPRO_CC ) && ( __SUNPRO_CC <= 0x550 )
#include "framecpp/Version4/FrVect.tcc"
#endif /* */

#endif /* FrameCPP_VERSION_4_FrVect_HH */
