#include <algorithm>

#include "framecpp/Common/Description.hh"
#include "framecpp/Common/TOCInfo.hh"

#include "framecpp/Version6/FrameSpec.hh"
#include "framecpp/Version6/FrSE.hh"
#include "framecpp/Version6/FrSH.hh"
#include "framecpp/Version6/FrTOC.hh"

#include "framecpp/Version6/STRING.hh"

using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;

template < typename T >
class sum
{
public:
  inline sum ( T i = 0 )
    : res( i )
  {
  }

  inline void operator( )( T x )
  {
    if ( x != FrameCPP::Version_6::FrTOC::NO_DATA_AVAILABLE )
    {
      res += x;
    }
  }

  inline T result( ) const
  {
    return res;
  }
private:
  T res;
};

namespace FrameCPP
{
  namespace Version_6
  {
    //===================================================================
    //===================================================================
    FrTOCSimEvent::
    FrTOCSimEvent( )
    {
    }

    FrTOCSimEvent::
    FrTOCSimEvent( Common::IStream& Stream )
    {
      nevent_type	nevent;
      Stream >> nevent;
      if ( nevent )
      {
	//---------------------------------------------------------------
	// Read in the information
	//---------------------------------------------------------------
	std::vector< name_type >	names( nevent );
	std::vector< nevent_type >	events( nevent );

	Stream >> names
	       >> events
	  ;

	sum< nevent_type >	nevent_sum;
	nevent_type		offset = 0;

	for ( std::vector< nevent_type >::const_iterator
		cur = events.begin( ),
		last = events.end( );
	      cur != last;
	      ++cur )
	{
	  nevent_sum( *cur );
	}

	std::vector< gtimesSim_type >
	  gtimes( nevent_sum.result( ) );
	std::vector< gtimenSim_type >
	  gtimen( nevent_sum.result( ) );
	std::vector< amplitudeSimEvent_type >
	  amplitude( nevent_sum.result( ) );
	std::vector< positionSimEvent_type >
	  position( nevent_sum.result( ) );
	
	Stream >> gtimes
	       >> gtimen
	       >> amplitude
	       >> position
	  ;
	std::vector< nevent_type >::const_iterator
	  cur_event_counter = events.begin( );

	for ( std::vector< name_type >::const_iterator
		cur = names.begin( ),
		last = names.end( );
	      cur != last;
	      ++cur )
	{
	  events_container_type&	cur_events = m_info[ *cur ];

	  if ( ( *cur_event_counter == 0 )
	       || ( *cur_event_counter == FrTOC::NO_DATA_AVAILABLE ) )
	  {
	    continue;
	  }
	  cur_events.resize( *cur_event_counter );
	  for ( nevent_type
		  x = 0,
		  x_last = *cur_event_counter;
		x != x_last;
		++x, ++offset )
	  {
	    cur_events[ x ].GTime =
	      GPSTime( gtimes[ offset ], gtimen[ offset ] );
	    cur_events[ x ].amplitudeSimEvent = amplitude[ offset ];
	    cur_events[ x ].positionSimEvent = position[ offset ];
	  }
	}
      }
    }

    void FrTOCSimEvent::
    QuerySimEvent( const Common::TOCInfo& Info,
		   INT_4U FrameOffset,
		   INT_8U Position )
    {
      STRING	name;
      INT_4U	sec;
      INT_4U	nsec;
      REAL_4	ampl;

      Info.TOCQuery( Common::TOCInfo::IC_NAME, Common::TOCInfo::DataType( name ), &name,
		     Common::TOCInfo::IC_GTIME_S, Common::TOCInfo::DataType( sec ), &sec,
		     Common::TOCInfo::IC_GTIME_N, Common::TOCInfo::DataType( nsec ), &nsec,
		     Common::TOCInfo::IC_AMPLITUDE, Common::TOCInfo::DataType( ampl ), &ampl,
		     Common::TOCInfo::IC_EOQ );

      events_container_type& i( m_info[ name ] );

      event_type	e;
      e.GTime = GPSTime( sec, nsec );
      e.amplitudeSimEvent = ampl;
      e.positionSimEvent = Position;

      i.push_back( e );
    }

    FrTOCSimEvent::positionSimEvent_type FrTOCSimEvent::
    positionSimEvent( INT_4U FrameIndex, const std::string& SimEvent ) const
    {
      const nameSimEvent_container_type::const_iterator
	event( m_info.find( SimEvent ) );
      //-----------------------------------------------------------------
      // Locate the event by name
      //-----------------------------------------------------------------
      if ( event == m_info.end( ) )
      {
	//---------------------------------------------------------------
	// SimEvent name does not exist.
	//---------------------------------------------------------------
	std::ostringstream	msg;

	msg << "No FrSimEvent structures with the name " << SimEvent
	  ;
	throw
	  std::out_of_range( msg.str( ) );
      }
      //-----------------------------------------------------------------
      // Verify that the index exists
      //-----------------------------------------------------------------
      if ( FrameIndex >= event->second.size( ) )
      {
	std::ostringstream	msg;

	msg << "Request for frame " << FrameIndex
	    << " exceeds the range of 0 through " << ( event->second.size( ) - 1 )
	  ;
	throw
	  std::out_of_range( msg.str( ) );
      }
      //-----------------------------------------------------------------
      // Return position information
      //-----------------------------------------------------------------
      return event->second[ FrameIndex ].positionSimEvent;
    }

    void FrTOCSimEvent::
    write( Common::OStream& Stream ) const
    {
      //-----------------------------------------------------------------
      // Flatten data so it is streamable
      //-----------------------------------------------------------------
      if ( m_info.size( ) > 0 ) {
	std::vector< name_type >	names( m_info.size( ) );
	std::vector< nevent_type >	nevent( m_info.size( ) );
	nevent_type			offset( 0 );
	nevent_type			eoffset( 0 );
	std::vector< gtimesSim_type >		gtimes;
	std::vector< gtimenSim_type >		gtimen;
	std::vector< amplitudeSimEvent_type >	amplitude;
	std::vector< positionSimEvent_type >	position;

	for ( nameSimEvent_container_type::const_iterator
		cur = m_info.begin( ),
		last = m_info.end( );
	      cur != last;
	      ++cur, ++offset )
	{
	  names[ offset ] = cur->first;
	  const nevent_type
	    c = ( cur->second.size( ) == 0 )
	    ? ( FrTOC::NO_DATA_AVAILABLE )
	    : ( cur->second.size( ) )
	    ;
	  nevent[ offset ] = c;
	  if ( c != FrTOC::NO_DATA_AVAILABLE )
	  {
	    const int ns( c + eoffset );

	    gtimes.resize( ns );
	    gtimen.resize( ns );
	    amplitude.resize( ns );
	    position.resize( ns );

	    for ( nevent_type x = 0;
		  x != c;
		  ++x, ++eoffset )
	    {
	      gtimes[ eoffset ] = cur->second[ x ].GTime.GetSeconds( );
	      gtimen[ eoffset ] = cur->second[ x ].GTime.GetNanoseconds( );
	      amplitude[ eoffset ] = cur->second[ x ].amplitudeSimEvent;
	      position[ eoffset ] = cur->second[ x ].positionSimEvent;
	    }
	  }
	}
	Stream
	  << nevent_type( m_info.size( ) )
	  << names
	  << nevent
	  << gtimes
	  << gtimen
	  << amplitude
	  << position
	  ;
      }
      else
      {
	Stream << nevent_type( 0 );
      }
    }

  } // namespace - Version_6
} // namespace - FrameCPP
