#ifndef FRAME_CPP_INTERFACE__IFrameStream_HH
#define FRAME_CPP_INTERFACE__IFrameStream_HH

#if !defined(SWIG)
#include "ldastoolsal/Memory.hh"

#include "framecpp/Common/FrameBuffer.hh"

#include "framecpp/FrameCPP.hh"
#include "framecpp/FrameStream.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/FrDetector.hh"
#include "framecpp/FrEvent.hh"
#include "framecpp/FrProcData.hh"
#include "framecpp/FrSerData.hh"
#include "framecpp/FrSimData.hh"
#include "framecpp/FrSimEvent.hh"
#include "framecpp/FrTOC.hh"

#endif /* !defined(SWIG) */


namespace FrameCPP
{
  typedef Version::IFrameStream IFrameStream;

  //---------------------------------------------------------------------
  // 
  //---------------------------------------------------------------------
  class IFrameFStream
  {
  public:
    typedef Common::IFrameStream::frame_library_type	frame_library_type;
    typedef Common::IFrameStream::library_revision_type	library_revision_type;
    typedef Common::IFrameStream::size_type		size_type;
    typedef Common::IFrameStream::version_type		version_type;
    typedef LDASTools::AL::SharedPtr< FrameH >		frame_h_type;
    typedef LDASTools::AL::SharedPtr< FrDetector >	fr_detector_type;
    typedef LDASTools::AL::SharedPtr< FrEvent >		fr_event_type;
    typedef LDASTools::AL::SharedPtr< FrAdcData >	fr_adc_data_type;
    typedef LDASTools::AL::SharedPtr< FrProcData >	fr_proc_data_type;
    typedef LDASTools::AL::SharedPtr< FrSerData >	fr_ser_data_type;
    typedef LDASTools::AL::SharedPtr< FrSimData >	fr_sim_data_type;
    typedef LDASTools::AL::SharedPtr< FrSimEvent >	fr_sim_event_type;
 
    //-------------------------------------------------------------------
    /// \brief Constructor
    ///
    /// \param[in] Filename
    ///     Name of the input frame file
    //-------------------------------------------------------------------
    IFrameFStream( const char* Filename );

    //-------------------------------------------------------------------
    /// \brief Constructor
    ///
    /// \param[in] Filename
    ///     Name of the input frame file
    ///
    /// \param[in] MemoryMappedIO
    ///     True if memory mapped io should be used, false otherwise.
    ///
    /// \param[in] BufferSize
    ///     Number of bytes to use for file buffering.
    //-------------------------------------------------------------------
    IFrameFStream( const char* Filename, bool MemoryMappedIO, size_t BufferSize );

    std::string Filename( ) const;

    frame_library_type FrameLibrary( ) const;

    std::string FrameLibraryName( ) const;

    size_type GetNumberOfFrames( ) const;

    const FrTOC& GetTOC( ) const;

    library_revision_type LibraryRevision( ) const;

    fr_detector_type ReadDetector( const std::string& Name );

    fr_event_type ReadFrEvent( INT_4U Frame, const std::string& Channel );

    fr_adc_data_type ReadFrAdcData( INT_4U Frame, const std::string& Channel );

    fr_proc_data_type ReadFrProcData( INT_4U Frame, const std::string& Channel );

    fr_ser_data_type ReadFrSerData( INT_4U Frame, const std::string& Channel );

    fr_sim_data_type ReadFrSimData( INT_4U Frame, const std::string& Channel );

    fr_sim_event_type ReadFrSimEvent( INT_4U Frame, const std::string& Channel );

    frame_h_type ReadFrameN( INT_4U Index );

    frame_h_type ReadNextFrame( );

    version_type Version( ) const;

    static const int			BUFFER_SIZE = 1024 * 1024;
    static const bool			MEMORY_MAPPED_IO = false;

  private:
    typedef LDASTools::AL::filebuf		filebuf_type;
    typedef Common::FrameBuffer< filebuf_type >	frame_buffer_type;

    LDASTools::AL::AutoArray< CHAR >		m_buffer;
    std::string				        m_filename;
    std::unique_ptr< frame_buffer_type >	m_frame_buffer;
    std::unique_ptr< IFrameStream >		m_frame_stream;
    mutable std::unique_ptr< FrTOC >		m_toc;
  };

  inline IFrameFStream::
  IFrameFStream( const char* Filename )
    : m_filename( Filename )
  {
    m_buffer.reset( new CHAR[ BUFFER_SIZE ] );

    m_frame_buffer.reset( new frame_buffer_type( std::ios::in ) );
    m_frame_buffer->pubsetbuf( m_buffer.get( ), BUFFER_SIZE );
    m_frame_buffer->UseMemoryMappedIO( MEMORY_MAPPED_IO );
    m_frame_buffer->open( Filename,
			  std::ios::in | std::ios::binary );

    m_frame_stream.reset( new IFrameStream( false, m_frame_buffer.get( ) ) );
  }

  inline std::string IFrameFStream::
  Filename( ) const
  {
    return m_filename;
  }

  inline IFrameFStream::
  IFrameFStream( const char* Filename, bool MemoryMappedIO, size_t BufferSize = IFrameFStream::BUFFER_SIZE )
  {
    m_buffer.reset( new CHAR[ BufferSize ] );

    m_frame_buffer.reset( new frame_buffer_type( std::ios::in ) );
    m_frame_buffer->pubsetbuf( m_buffer.get( ), BufferSize );
    m_frame_buffer->UseMemoryMappedIO( MemoryMappedIO );
    m_frame_buffer->open( Filename,
			  std::ios::in | std::ios::binary );

    m_frame_stream.reset( new IFrameStream( false, m_frame_buffer.get( ) ) );
  }

  inline IFrameFStream::frame_library_type IFrameFStream::
  FrameLibrary( ) const
  {
    return m_frame_stream->FrameLibrary( );
  }

  inline std::string IFrameFStream::
  FrameLibraryName( ) const
  {
    return m_frame_stream->FrameLibraryName( );
  }

  inline IFrameFStream::size_type IFrameFStream::
  GetNumberOfFrames( ) const
  {
    return m_frame_stream->GetNumberOfFrames( );
  }
  
  inline const FrTOC& IFrameFStream::
  GetTOC( ) const
  {
    if ( m_toc.get( ) == NULL )
    {
      m_toc.reset( new FrTOC( m_frame_stream->GetTOC( ) ) );
    }
    return *m_toc;
  }

  inline IFrameFStream::library_revision_type IFrameFStream::
  LibraryRevision( ) const
  {
    return m_frame_stream->LibraryRevision( );
  }

  inline IFrameFStream::fr_detector_type IFrameFStream::
  ReadDetector( const std::string& Name )
  {

    Common::FrameSpec::ObjectInterface::object_type
      od( m_frame_stream->ReadDetector( Name ) );

    fr_detector_type
      detector( LDASTools::AL::DynamicPointerCast< FrDetector >( od ) );
		    
    if ( ! detector )
    {
      throw std::range_error( "Unable to read detector" );
    }
    return detector;
  }

  inline IFrameFStream::fr_event_type IFrameFStream::
  ReadFrEvent( INT_4U Frame,
		 const std::string& Channel )
  {
    typedef fr_event_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadFrEvent( Frame, Channel ) );

    retval_type
      retval( LDASTools::AL::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read FrEvent" );
  }

  inline IFrameFStream::fr_adc_data_type IFrameFStream::
  ReadFrAdcData( INT_4U Frame,
		 const std::string& Channel )
  {
    typedef fr_adc_data_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadFrAdcData( Frame, Channel ) );

    retval_type
      retval( LDASTools::AL::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read FrAdcData" );
  }

  inline IFrameFStream::fr_proc_data_type IFrameFStream::
  ReadFrProcData( INT_4U Frame,
		 const std::string& Channel )
  {
    return m_frame_stream->ReadFrProcData( Frame, Channel );
  }

  inline IFrameFStream::fr_ser_data_type IFrameFStream::
  ReadFrSerData( INT_4U Frame,
		 const std::string& Channel )
  {
    return m_frame_stream->ReadFrSerData( Frame, Channel );
  }

  inline IFrameFStream::fr_sim_data_type IFrameFStream::
  ReadFrSimData( INT_4U Frame,
		 const std::string& Channel )
  {
    return m_frame_stream->ReadFrSimData( Frame, Channel );
  }

  inline IFrameFStream::fr_sim_event_type IFrameFStream::
  ReadFrSimEvent( INT_4U Frame,
		 const std::string& Channel )
  {
    return m_frame_stream->ReadFrSimEvent( Frame, Channel );
  }

  inline IFrameFStream::frame_h_type IFrameFStream::
  ReadFrameN( INT_4U Index )
  {
    typedef frame_h_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadFrameN( Index ) );

    retval_type
      retval( LDASTools::AL::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read next frame" );
  }

  inline IFrameFStream::frame_h_type IFrameFStream::
  ReadNextFrame( )
  {
    typedef frame_h_type retval_type;

    Common::FrameSpec::ObjectInterface::object_type
      o( m_frame_stream->ReadNextFrame( ) );

    retval_type
      retval( LDASTools::AL::DynamicPointerCast< retval_type::element_type >
	      ( o ) );

    if ( retval )
    {
      return retval;
    }
    throw std::range_error( "Unable to read next frame" );
  }

  inline IFrameFStream::version_type IFrameFStream::
  Version( ) const
  {
    return m_frame_stream->Version( );
  }
}

#endif /* FRAME_CPP_INTERFACE__IFrameStream_HH */
