#include <diskcache_config.h>

#include <sys/select.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <ctype.h>
#include <unistd.h>

#include <algorithm>
#include <list>
#include <map>

#include "ldastoolsal/SignalHandler.hh"

#include "genericAPI/Logging.hh"
#include "genericAPI/LDASplatform.hh"
#include "genericAPI/MonitorMemory.hh"
#include "genericAPI/Stat.hh"

#include "diskcacheAPI/Common/Variables.hh"
#include "diskcacheAPI/Cache/ExcludedDirectoriesSingleton.hh"
#include "diskcacheAPI/Cache/HotDirectory.hh"

#include "diskcacheAPI/IO.hh"
#include "diskcacheAPI/MetaCommands.hh"


#include "Commands.hh"
#include "diskcachecmd.hh"
#include "DumpCacheDaemon.hh"
#include "MountPointScanner.hh"
#include "ScanMountPointsDaemon.hh"

using LDASTools::AL::CommandLineOptions;

using diskCache::Commands::updateFileExtList;
using diskCache::Commands::updateMountPtList;

typedef diskCache::MetaCommand::client_type	client_type;
typedef diskCache::MetaCommand::ClientServerInterface::ServerInfo ServerInfo;

namespace
{
  CommandLineOptions
  convert_line_to_options( char* Buffer )
  {
    //-------------------------------------------------------------------
    // Create list of options.
    //-------------------------------------------------------------------
    char*	lb_opts[ 64 ];
    int		cur_opt = 0;
    char*	lb_pos = Buffer;
    
    lb_opts[ cur_opt ] = lb_pos;
    while( *lb_pos )
    {
      if ( ::isspace( *lb_pos ) )
      {
	*lb_pos = '\0';
	++lb_pos;
	while( *lb_pos && ::isspace( *lb_pos ) )
	{
	  ++lb_pos;
	}
	if ( *lb_pos )
	{
	  lb_opts[ ++cur_opt ] = lb_pos;
	}
	continue;
      }
      ++lb_pos;
    }
    lb_opts[ ++cur_opt ] = (char*)NULL;
      
    CommandLineOptions	retval( cur_opt, lb_opts );

    return retval;
  }

  void
  order_by_access( std::list< std::string >& Paths )
  {
    typedef std::multimap< time_t, std::string > op_type;
    typedef std::list< std::string > paths_type;

    op_type ordered_paths;

    struct stat		stat_buf;

    for ( paths_type::const_iterator
	    cur = Paths.begin( ),
	    last = Paths.end( );
	  cur != last;
	  ++cur )
    {
      GenericAPI::Stat( *cur, stat_buf );
      
      ordered_paths.insert( op_type::value_type( stat_buf.st_mtime, *cur ) );
    }

    paths_type	new_paths;

    for( op_type::const_reverse_iterator
	   cur = ordered_paths.rbegin( ),
	   last = ordered_paths.rend( );
	 cur != last;
	 ++cur )
    {
      new_paths.push_back( cur->second );
    }

    Paths.swap( new_paths );
  }

  
} // namespace - anonymous
namespace diskCache
{
  namespace MetaCommand
  {
    //===================================================================
    // Daemon::Config_
    //===================================================================
    Daemon::Config_::
    Config_( Daemon& Command )
      : command( Command ),
	state( NON_BLOCK )
    {
    }

    void Daemon::Config_::
    Parse( std::istream& Stream )
    {
      base_type::Parse( Stream );
    }

    void Daemon::Config_::
    ParseBlock( const std::string& Value )
    {
      if ( Value.compare( "DAEMON" ) == 0 )
      {
	state = BLOCK_DAEMON;
      }
      else if ( Value.compare( "EXTENSIONS" ) == 0 )
      {
	state = BLOCK_EXTENSIONS;
      }
      else if ( Value.compare( "EXCLUDED_DIRECTORIES" ) == 0 )
      {
	state = BLOCK_EXCLUDED_DIRECTORIES;
      }
      else if ( Value.compare( "MOUNT_POINTS" ) == 0 )
      {
	state = BLOCK_MOUNT_POINTS;
      }
      else
      {
	//---------------------------------------------------------------
	// Tried everything we know. Flag as unknown block
	//---------------------------------------------------------------
	state = BLOCK_UNKNOWN;
      }
    }

    void Daemon::Config_::
    ParseKeyValue( const std::string& Key,
		   const std::string& Value )
    {
      switch( state )
      {
      case NON_BLOCK:
      case BLOCK_DAEMON:
	try
	{
	  diskCache::Common::Variables::Cache( Key, Value );
	  return;
	}
	catch( const std::invalid_argument& Error )
	{
	  //-------------------------------------------------------------
	  // This is the case where the variable is NOT handled by
	  //   the generic routine.
	  //-------------------------------------------------------------
	}
	if ( Key.compare( "CACHE_WRITE_DELAY_SECS" ) == 0 )
	{
	  command.cache_write_delay( Value );
	}
	else if ( Key.compare( "LOG" ) == 0 )
	{
	  GenericAPI::LDASplatform::AppName( Value );
	}
	else if ( Key.compare( "LOG_DIRECTORY" ) == 0 )
	{
	  GenericAPI::LoggingInfo::LogDirectory( Value );
	}
	else if ( Key.compare( "LOG_ARCHIVE_DIRECTORY" ) == 0 )
	{
	  GenericAPI::LoggingInfo::ArchiveDirectory( Value );
	}
	else if ( Key.compare( "LOG_DEBUG_LEVEL" ) == 0 )
	{
	  int	debug_level;
	  std::istringstream	dbg_level_stream( Value );

	  dbg_level_stream >> debug_level;
	  GenericAPI::LoggingInfo::DebugLevel( debug_level );
	}
	else if ( Key.compare( "LOG_FORMAT" ) == 0 )
	{
	  GenericAPI::LoggingInfo::Format( Value );
	}
	else if ( Key.compare( "LOG_ROTATE_ENTRY_COUNT" ) == 0 )
	{
	  int			v;
	  std::istringstream	s( Value );

	  s >> v;
	  GenericAPI::LoggingInfo::RotationCount( v );
	}
	else if ( Key.compare( "OUTPUT_ASCII" ) == 0 )
	{
	  command.set_output_ascii( Value );
	}
	else if ( Key.compare( "OUTPUT_BINARY" ) == 0 )
	{
	  command.set_output_binary( Value );
	}
	else if ( Key.compare( "OUTPUT_BINARY_VERSION" ) == 0 )
	{
	  command.set_output_binary_version( Value );
	}
	else if ( Key.compare( "SCAN_INTERVAL" ) == 0 )
	{
	  command.set_scan_interval( Value );
	}
	else if ( Key.compare( "SERVER_PORT" ) == 0 )
	{
	  command.set_server_port( Value );
	}
	else
	{
	  if ( state == BLOCK_DAEMON )
	  {
	    //-----------------------------------------------------------
	    /// \todo
	    ///     Need to produce and exception regarding unknown
	    ///     keyword found
	    //-----------------------------------------------------------
	  }
	}
	break;
      case BLOCK_EXCLUDED_DIRECTORIES:
      case BLOCK_EXTENSIONS:
      case BLOCK_MOUNT_POINTS:
	//---------------------------------------------------------------
	/// \todo
	///     Need to produce an exception regarding parse error
	///     in configuration file as a key/value pair was encountered
	///     where a word value was expected.
	//---------------------------------------------------------------
	break;
      default:
	//---------------------------------------------------------------
	// Ignore all other cases
	//---------------------------------------------------------------
	break;
      }
    }

    void Daemon::Config_::
    ParseWord( const std::string& Value )
    {
      if ( Value.size( ) <= 0 )
      {
	//---------------------------------------------------------------
	// Ignore empty words
	//---------------------------------------------------------------
	return;
      }
      //-----------------------------------------------------------------
      // Process the word according to the block it was found in
      //-----------------------------------------------------------------
      switch( state )
      {
      case BLOCK_EXTENSIONS:
	command.push_extension( Value );
	break;
      case BLOCK_EXCLUDED_DIRECTORIES:
	command.push_excluded_directory( Value );
	break;
      case BLOCK_MOUNT_POINTS:
	command.push_mount_point( Value );
	break;
      case BLOCK_DAEMON:
      case BLOCK_UNKNOWN:
      case NON_BLOCK:
	// This shouild never happen.
	/// \todo
	/// Have Daemon::Config_::ParseWord throw exception
	/// if it reaches an unreachable state.
	break;
      }
    }

    //===================================================================
    // Daemon
    //===================================================================
    bool Daemon::daemon_mode;
    OptionSet& Daemon::m_options( Daemon::init_options( ) );

    OptionSet& Daemon::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: daemon" );

      retval.
	Summary( "The daemon sub command is intended to"
		 " provide a continuous scanning mode." );

      retval.Add( Option( OPT_CONCURRENCY,
			  "concurrency",
			  Option::ARG_REQUIRED,
			  "Number of mount points to scan concurrently",
			  "integer" ) );

      retval.Add( Option( OPT_CONFIGURATION_FILE,
			  "configuration-file",
			  Option::ARG_REQUIRED,
			  "Name of file containing additional configuration information.",
			  "filename" ) );

      retval.Add( Option( OPT_EXCLUDED_DIRECTORIES,
			  "excluded-directories",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of directories not to be searched",
			  "list" ) );

      retval.Add( Option( OPT_EXTENSIONS,
			  "extensions",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of file extensions",
			  "list" ) );

      retval.Add( Option( OPT_SCAN_INTERVAL,
			  "interval",
			  Option::ARG_REQUIRED,
			  "Number of milliseconds to pause between scans.",
			  "integer" ) );

      retval.Add( Option( OPT_STAT_TIMEOUT,
			  "stat-timeout",
			  Option::ARG_REQUIRED,
			  "Number of seconds to wait for system stat call to return.",
			  "integer" ) );

      retval.Add( Option( OPT_MOUNT_POINTS,
			  "mount-points",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of mount points to scan",
			  "list" ) );

      retval.Add( Option( OPT_OUTPUT_ASCII,
			  "output-ascii",
			  Option::ARG_REQUIRED,
			  "Filename for the ascii output; '-' to direct to standard output",
			  "filename" ) );

      retval.Add( Option( OPT_OUTPUT_BINARY,
			  "output-binary",
			  Option::ARG_REQUIRED,
			  "Filename for the binary output",
			  "filename" ) );

      retval.Add( Option( OPT_VERSION_ASCII,
			  "version-ascii",
			  Option::ARG_REQUIRED,
			  "Version of the ascii diskcache dump format to output",
			  "version" ) );

      retval.Add( Option( OPT_VERSION_BINARY,
			  "version-binary",
			  Option::ARG_REQUIRED,
			  "Version of the binary diskcache dump format to output",
			  "version" ) );
      return retval;
    }

    //-------------------------------------------------------------------
    /// Contruct a new instance of Daemon.
    //-------------------------------------------------------------------
    Daemon::
    Daemon( CommandLineOptions& Args,
	    const ClientServerInterface::ServerInfo& Server,
	    const std::string& DefaultConfigurationFilename,
	    bool Seedable )
      : m_args( Args ),
	cache_write_delay_( 120 ),
	m_concurrency( 4 ),
	scan_interval( 16000 ),
	version_binary( Streams::Interface::VERSION_NONE ),
	finished( false ),
	reset_requested( false ),
	server_info( Server )
    {
      {
	configuration_filename = DefaultConfigurationFilename;
	std::ifstream	stream( configuration_filename.c_str( ) );

	if ( stream.is_open( ) )
	{
	  Config_		cfg( *this );

	  cfg.Parse( stream );
	}
      }
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );

	while( parsing )
	{
	  switch( m_args.Parse( m_options, arg_name, arg_value ) )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_CONCURRENCY:
	    diskCache::Common::Variables::Cache( "CONCURRENCY", arg_value );
	    break;
	  case OPT_CONFIGURATION_FILE:
	    {
	      configuration_filename = arg_value;
	      std::ifstream	stream( configuration_filename.c_str( ) );

	      if ( stream.is_open( ) )
	      {
		Config_		cfg( *this );

		cfg.Parse( stream );
	      }
	    }
	    break;
	  case OPT_EXCLUDED_DIRECTORIES:
	    {
	      //-----------------------------------------------------------
	      // Generate list of directories
	      //-----------------------------------------------------------
	      Cache::ExcludedDirectoriesSingleton::directory_container_type
		directories;

	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		//-------------------------------------------------------
		// Extract each directory
		//-------------------------------------------------------
		end = arg_value.find_first_of( ",", pos );
		const size_t dend( ( end == std::string::npos )
				   ? end
				   : ( end - pos ) );
		push_excluded_directory( arg_value.substr( pos, dend ) );
		pos = end + 1;
	      }
	      //---------------------------------------------------------
	      // Update the list of directories to be excluded
	      //---------------------------------------------------------
	      Cache::ExcludedDirectoriesSingleton::Update( directories );
	    }
	    break;
	  case OPT_EXTENSIONS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of extensions
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		push_extension( arg_value.substr( pos,( ( end == std::string::npos )
							? end
							: end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_SCAN_INTERVAL:
	    set_scan_interval( arg_value );
	    break;
	  case OPT_MOUNT_POINTS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of mount points
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		push_mount_point( arg_value.substr( pos,( ( end == std::string::npos )
							  ? end
							  : end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_OUTPUT_ASCII:
	    set_output_ascii( arg_value );
	    break;
	  case OPT_OUTPUT_BINARY:
	    set_output_binary( arg_value );
	    break;
	  case OPT_STAT_TIMEOUT:
	    diskCache::Common::Variables::Cache( "STAT_TIMEOUT", arg_value );
	    break;
	  case OPT_VERSION_BINARY:
	    set_output_binary_version( arg_value );
	    break;
	  default:
	    break;
	  }
	}
      }
      //-----------------------------------------------------------------
      // Check for seeding
      //-----------------------------------------------------------------
      if ( ( Seedable )
	   && ( output_binary.empty( ) == false ) )
      {
	std::list< std::string >	paths;

	paths.push_back( output_binary );
	paths.push_back( ( output_binary + ".bak") );
	paths.push_back( ( output_binary + "#Resync#" ) );

	::order_by_access( paths );		 

	//---------------------------------------------------------------
	// Seed the cache with previous binary
	//---------------------------------------------------------------
	for ( std::list< std::string >::const_iterator
		cur_path = paths.begin( ),
		last_path = paths.end( );
	      cur_path != last_path;
	      ++cur_path )
	{
	  try
	  {
	    QUEUE_LOG_MESSAGE( "Trying to seed the cache with: " << *cur_path,
			       MT_NOTE,
			       0,
			       "Daemon::Daemon",
			       "CXX" );

	    LDASTools::AL::ifstream	ifs( cur_path->c_str( ) );
	    diskCache::Streams::IBinary	stream( ifs );

	    try
	    {
	      diskCache::Read( stream );
	    }
	    catch( ... )
	    {
	      ifs.close( );
	      continue;
	    }
	    ifs.close( );
	    //-----------------------------------------------------------
	    // Was able to seed with the given file so terminate
	    //   loop of files to try.
	    //-----------------------------------------------------------
	    QUEUE_LOG_MESSAGE( "Succeeded to seed the cache with: "
			       << *cur_path,
			       MT_NOTE,
			       0,
			       "Daemon::Daemon",
			       "CXX" );
	    break;
	  }
	  catch( ... )
	  {
	    //-----------------------------------------------------------
	    // Something wrong with reading so advance to next
	    //   file to try.
	    //-----------------------------------------------------------
	  }
	}
      }
      //-----------------------------------------------------------------
      // Set variables once cache is seeded (ticket #2167)
      //-----------------------------------------------------------------
      setup_variables( );
    }

    //-------------------------------------------------------------------
    /// Return resource back to the system
    //-------------------------------------------------------------------
    Daemon::
    ~Daemon( )
    {
      //-----------------------------------------------------------------
      // Remove the daemon from the list of signal handlers
      //-----------------------------------------------------------------
      ResetOnSignal( false );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    const OptionSet& Daemon::
    Options( )
    {
      return m_options;
    }

    //-------------------------------------------------------------------
    /// \brief Register signal handler
    //-------------------------------------------------------------------
    void Daemon::
    ResetOnSignal( bool Value )
    {
      using LDASTools::AL::SignalHandler;

      static const char* method = "Daemon::ResetOnSignal";

      if ( Value )
      {
	QUEUE_LOG_MESSAGE( "Registering SIGNAL_HANGUP handler",
			   MT_NOTE,
			   0,
			   method,
			   "CXX" );
	SignalHandler::Register( this, SignalHandler::SIGNAL_HANGUP );
	SignalCapture( SignalHandler::SIGNAL_HANGUP );
      }
      else
      {
	QUEUE_LOG_MESSAGE( "Unregistering SIGNAL_HANGUP handler",
			   MT_NOTE,
			   0,
			   method,
			   "CXX" );
	SignalIgnore( SignalHandler::SIGNAL_HANGUP );
	SignalHandler::Unregister( this, SignalHandler::SIGNAL_HANGUP );
      }
    }

    void Daemon::
    SignalCallback( signal_type Signal )
    {
      static const char* method = "Daemon::SignalCallback";
      QUEUE_LOG_MESSAGE( "Calling Daemon::SignalCallback"
			 << " with signal: " << Signal,
			 MT_NOTE,
			 0,
			 method,
			 "CXX" );
      if ( Signal == LDASTools::AL::SignalHandler::SIGNAL_HANGUP )
      {
	//---------------------------------------------------------------
	// Start the process of resetting
	//---------------------------------------------------------------
#if WORKING
	reset( );
#else
	reset_requested = true;
#endif /* WORKING */
      }
    }

    void Daemon::
    do_client_request( )
    {
      static const char* method_name = "diskCache::MetaCommand::Daemon::do_client_request";

      if ( reset_requested )
      {
	reset( );
      }
      if ( server->good( ) )
      {
	try
	{
	  fd_set		sset;
	  struct timespec	timeout;

	  timeout.tv_sec = 1;
	  timeout.tv_nsec = 0;

	  FD_ZERO( &sset );
	  FD_SET( server->handle( ), &sset );
	  
	  int stat = pselect( server->handle( ) + 1,
			      &sset, (fd_set*)NULL, (fd_set*)NULL,
			      &timeout, (const sigset_t*)NULL );

	  if ( stat < 0 )
	  {
	    return;
	  }
	  if ( FD_ISSET( server->handle( ), &sset ) )
	  {
	    QUEUE_LOG_MESSAGE( "Server: waiting for client request",
			       MT_DEBUG,
			       30,
			       method_name,
			       "CXX" );
	    client = server->accept( );

	    if ( client->good( ) )
	    {
	      Spawn( );
	      Join( );
	    }

	    QUEUE_LOG_MESSAGE( "Server: client request completed",
			       MT_DEBUG,
			       30,
			       method_name,
			       "CXX" );
	    client.reset( ( server_responce_type::element_type* )NULL );
	  }
	}
	catch(...)
	{
	}
      }
    }

    bool Daemon::
    process_cmd( CommandLineOptions& Options )
    {
      bool	retval = true;

      switch( CommandTable::Lookup( Options.ProgramName( ) ) )
      {
      case CommandTable::CMD_QUIT:
	{
	  //-------------------------------------------------------------
	  // Quit the current contents of the cache
	  //-------------------------------------------------------------
	  MetaCommand::Quit	cmd( Options, server_info );
	  std::ostringstream	msg;

	  cmd.ClientHandle( client );

	  if ( finished )
	  {
	    msg << "Daemon is already shutting down";
	  }
	  else
	  {
	    msg << "Requested daemon to shut down";
	  }
	  cmd.msg = msg.str( );
	  finished = true;
	}
	break;
      case CommandTable::CMD_DUMP:
	{
	  //-------------------------------------------------------------
	  // Dump the current contents of the cache
	  //-------------------------------------------------------------
	  MetaCommand::Dump	cmd( Options, server_info );

	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_FILENAMES:
	{
	  MetaCommand::Filenames	cmd( Options, server_info );

	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_FILENAMES_RDS:
	{
	  MetaCommand::FilenamesRDS	cmd( Options, server_info );
	  
	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_INTERVALS:
	{
	  MetaCommand::Intervals	cmd( Options, server_info );
	  
	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_STATUS:
	{
	  //-------------------------------------------------------------
	  // Retrieve status information
	  //-------------------------------------------------------------
	  MetaCommand::Status	cmd( Options, server_info );

	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      default:
	{
	  retval = false;
	}
	break;
      }
      return retval;
    }

    inline void Daemon::
    push_excluded_directory( const std::string& Directory )
    {
      excluded_directories.push_back( Directory );
    }
    
    inline void Daemon::
    push_extension( const std::string& Extension )
    {
      m_extensions.push_back( Extension );
    }
    
    inline void Daemon::
    push_mount_point( const std::string& MountPoint )
    {
      m_mount_points.push_back( MountPoint );
    }

    bool Daemon::
    read_command( char* Buffer, size_t BufferSize )
    {
      bool	retval = false;

      if ( ! server )
      {
	std::cin.clear( );
	std::cin.getline( Buffer, BufferSize, '\n' );
	retval = ( ( std::cin.good( ) )
		  ? true
		  : false );
      }
      return retval;
    }

    inline void Daemon::
    cache_write_delay( const std::string& Value )
    {
      std::istringstream	s( Value );

      s >> cache_write_delay_;
    }

    inline void Daemon::
    set_concurrency( const std::string& Value )
    {
      std::istringstream	s( Value );

      s >> m_concurrency;
    }

    inline void Daemon::
    set_scan_interval( const std::string& Value )
    {
      std::istringstream	s( Value );

      s >> scan_interval;
    }

    inline void Daemon::
    set_stat_timeout( const std::string& Value )
    {
      std::istringstream	s( Value );

      s >> stat_timeout;
    }

    inline void Daemon::
    set_output_ascii( const std::string& Value )
    {
      output_ascii = Value;
    }

    inline void Daemon::
    set_output_binary( const std::string& Value )
    {
      output_binary = Value;
    }

    inline void Daemon::
    set_output_binary_version( const std::string& Value )
    {
      if ( Value.substr( 0, 2 ).compare( "0x" ) == 0 )
      {
	std::istringstream	s( Value.substr( 2 ) );
	s >> std::hex >> version_binary;
      }
      else if ( Value[0] == '0' )
      {
	std::istringstream	s( Value.substr( 1 ) );
	s >> std::oct >> version_binary;
      }
      else
      {
	std::istringstream	s( Value );
	s >> std::hex >> version_binary;
      }
    }

    inline void Daemon::
    set_server_port( const std::string& Value )
    {
      std::istringstream	s( Value );

      ClientServerInterface::ServerInfo::port_type	p;
      s >> p;

      const_cast< ClientServerInterface::ServerInfo& >( server_info ).Port( p );
    }

    void Daemon::
    operator()( )
    {
      //-----------------------------------------------------------------
      // 
      //-----------------------------------------------------------------
      daemon_mode = true;

      ScanMountPointListContinuously( );
      DumpCacheDaemonStart( );

      //-----------------------------------------------------------------
      // Check to see if the server port has been requested
      //-----------------------------------------------------------------
      if ( server_info.Port( ) >= 0 )
      {
	server = server_type( new server_type::element_type( ) );
	if ( server )
	{
	  server->open( server_info.Port( ) );
	}
      }

      //-----------------------------------------------------------------
      // Read commands to be executed.
      //-----------------------------------------------------------------
      if ( server )
      {
	while ( ( finished == false )
		&& ( server->good( ) ) )
	{
	  do_client_request( );
	}
      }
      else
      {
	char	line_buffer[ 2048 ];

	std::fill( line_buffer, line_buffer + sizeof( line_buffer ), '\0' );
	while ( finished == false )
	{
	  if ( read_command( line_buffer, sizeof( line_buffer ) ) )
	  {
	    //-----------------------------------------------------------
	    // Switch on the command specified by the user
	    //-----------------------------------------------------------
	    CommandLineOptions
	      line_options( convert_line_to_options( line_buffer ) );

	    process_cmd( line_options);
	  }
	}
      }
    }

    void Daemon::
    action( )
    {
      char	line_buffer[ 2048 ];

      INT_4U	bytes;

      (*client) >> bytes;
      if ( bytes >= sizeof( line_buffer) )
      {
      }
      else
      {
	client->read( line_buffer, bytes );
	line_buffer[ bytes ] = '\0';

	//---------------------------------------------------------------
	// Create the command line option
	//---------------------------------------------------------------
	CommandLineOptions
	  line_options( convert_line_to_options( line_buffer ) );

	process_cmd( line_options);
      }
    }

    //-------------------------------------------------------------------
    /// Setup the variables according to the requested configuration.
    /// Some of the variables can be reset by modifying the
    /// configuration file and then signaling the daemon process.
    //-------------------------------------------------------------------
    void Daemon::
    setup_variables( int Mask )
    {
      if ( Mask & VAR_EXCLUDED_DIRECTORIES )
      {
	//---------------------------------------------------------------
	// Update the list of directories to be excluded
	//---------------------------------------------------------------
	Cache::ExcludedDirectoriesSingleton::Update( excluded_directories );
      }
      if ( ( Mask & VAR_EXTENSIONS )
	   && ( m_extensions.size( ) > 0 ) )
      {
	//---------------------------------------------------------------
	// Add the extensions
	//---------------------------------------------------------------
	{
	  QUEUE_LOG_MESSAGE( "Adding extension: " << m_extensions.size( ),
			     MT_NOTE,
			     0,
			     "Daemon::setup_variables",
			     "CXX" );
	}
	updateFileExtList( m_extensions );
      }

      if ( ( Mask & VAR_MOUNT_POINTS )
	   && ( m_mount_points.size( ) > 0 ) )
      {
	//---------------------------------------------------------------
	// Add the mount points
	//---------------------------------------------------------------
	{
	  QUEUE_LOG_MESSAGE( "Adding mount points: " << m_mount_points.size( ),
			     MT_NOTE,
			     0,
			     "Daemon::setup_variables",
			     "CXX" );
	}
	MountPointManagerSingleton::UpdateResults    status;

	updateMountPtList( status, m_mount_points, false );
	{
	  QUEUE_LOG_MESSAGE( "Added mount points status:"
			     << " added: " << status.s_added.size( )
			     << " deleted: " << status.s_deleted.size( )
			     << " errors: " << status.s_errors.size( ),
			     MT_NOTE,
			     0,
			     "Daemon::setup_variables",
			     "CXX" );
	}
      }

      if ( Mask & VAR_CONCURRENCY )
      {
	//---------------------------------------------------------------
	// Setup scanning parameters
	//---------------------------------------------------------------
	diskCache::Common::Variables::Set( "CONCURRENCY" );
      }

      if ( Mask & VAR_SCAN_INTERVAL )
      {
	//---------------------------------------------------------------
	// Setup scanning parameters
	//---------------------------------------------------------------
	diskCache::ScanMountPointsDaemon::Interval( scan_interval );
      }

      if ( Mask & VAR_STAT_TIMEOUT )
      {
	//---------------------------------------------------------------
	// Setup stat timeout value
	//---------------------------------------------------------------
	diskCache::Common::Variables::Set( "STAT_TIMEOUT" );
      }

      //-----------------------------------------------------------------
      // Update the cache files
      //-----------------------------------------------------------------
      {
	QUEUE_LOG_MESSAGE( "Checking to if cache delay should be set: "
			   << (Mask & VAR_CACHE_WRITE_DELAY),
			   MT_NOTE,
			   0,
			   "Daemon::setup_variables",
			   "CXX" );
      }
      if ( Mask & VAR_CACHE_WRITE_DELAY )
      {
	QUEUE_LOG_MESSAGE( "Setting CACHE_WRITE_DELAY: " << cache_write_delay_,
			   MT_NOTE,
			   0,
			   "Daemon::setup_variables",
			   "CXX" );
	diskCache::DumpCacheDaemon::Interval( cache_write_delay_ * 1000 );
      }

      if ( ( Mask & VAR_OUTPUT_ASCII ) && ( output_ascii.size( ) > 0 ) )
      {
	diskCache::DumpCacheDaemon::FilenameAscii( output_ascii );
      }
      if ( ( Mask & VAR_OUTPUT_BINARY ) && ( output_binary.size( ) > 0 ) )
      {
	diskCache::DumpCacheDaemon::FilenameBinary( output_binary );
      }
      if ( Mask & VAR_VERSION_ASCII )
      {
	
	/* && ( ver_ascii != Streams::Interface::VERSION_NONE )  ) */
	diskCache::Common::Variables::Set( "OUTPUT_ASCII_VERSION" );
      }
      if ( ( Mask & VAR_VERSION_BINARY )
	   && ( version_binary != Streams::Interface::VERSION_NONE ) )
      {
	diskCache::DumpCacheDaemon::BinaryVersion( version_binary );
      }
    }

    //-------------------------------------------------------------------
    /// Resetting of the daemon process forces the rereading of
    /// vital configuration information without having to restart
    /// the process.
    //-------------------------------------------------------------------
    void Daemon::
    reset( )
    {
      static const char* method_name = "diskCache::MetaCommand::Daemon::reset";

      reset_requested = false;
      //-----------------------------------------------------------------
      // Re-Read the configuration information
      //-----------------------------------------------------------------
      if ( ! configuration_filename.empty( ) )
      {
	QUEUE_LOG_MESSAGE( "Rereading configuration file: "
			   << configuration_filename
			   ,
			   MT_NOTE,
			   0,
			   method_name,
			   "CXX" );
	Config_		cfg( *this );
	std::ifstream	stream( configuration_filename.c_str( ) );
      
	cfg.Parse( stream );
      }
      setup_variables( HOT_VARIABLES );
    }
  } // namespace - MetaCommand
} // namespace - diskCache
