/* -*- mode: c++; c-basic-offset: 2; -*- */

#ifndef LDASTOOLSAL__TRI_STATE_POOL_HH
#define LDASTOOLSAL__TRI_STATE_POOL_HH

#include <stdexcept>
#include <list>

#include "ldastoolsal/Pool.hh"

namespace LDASTools
{
  namespace AL
  {
    //-------------------------------------------------------------------
    /// \brief Base class for objects to be stored in the TriStatePool
    ///
    /// This class describes the interface used by the TriStatePool
    /// class.
    //-------------------------------------------------------------------
    class TriStateInterface
    {
    protected:
      template< class Pool > friend class TriStatePool;

      //-----------------------------------------------------------------
      /// \brief Test if the object should be made immediate available
      ///
      /// Virtual method called to test if object should be made immedately
      /// available.
      ///
      /// \return True if the object can be immediately reused.
      //-----------------------------------------------------------------
      virtual bool makeAvailable( ) const;
    };

    inline bool TriStateInterface::
    makeAvailable( ) const
    {
      return( false );
    }

    //-------------------------------------------------------------
    /// \brief Maintain a collection system calls
    //-------------------------------------------------------------
    template< typename T >
    class TriStatePool
      : public Pool< T >
    {
    public:
      typedef typename Pool< T >::value_type value_type;

      TriStatePool( );

    protected:
      virtual void		relinquish( value_type Source );

      virtual value_type	request( value_type (*CreateFunc)( ) );

    private:
      typedef typename Pool< T >::pool_type pool_type;

      pool_type			tri_state_pool;
    };

    template< typename T >
    TriStatePool< T >::
    TriStatePool( )
    {
    }


    template< typename T >
    void TriStatePool< T >::
    relinquish( value_type Source )
    {
      if ( Source->makeAvailable( ) )
      {
	Pool< T >::relinquish( Source );
	return;
      }
      tri_state_pool.push_back( Source );
    }

    template< typename T >
    typename TriStatePool< T >::value_type TriStatePool< T >::
    request( value_type (*CreateFunc)( ) )
    {
      typename pool_type::iterator
	cur = tri_state_pool.begin( ),
	last = tri_state_pool.end( );
      while( cur != last )
      {
	if ( (*cur)->makeAvailable( ) )
	{
	  typename pool_type::iterator::value_type
	    source = *cur;

	  // Remove from tri-state queue
	  cur = tri_state_pool.erase( cur );
	  // and make available for use
	  Pool< T >::relinquish( source );
	  continue;
	}
	++cur;
      }
      return Pool< T >::request( CreateFunc );
    }

  } // namespace - AL
} // namespace - LDASTools

#endif /* LDASTOOLSAL__TRI_STATE_POOL_HH */
