#ifndef GENERAL__PROFILE_HH
#define GENERAL__PROFILE_HH

#if ! defined(SWIGIMPORTED)
#include <string>
#include <sstream>
#endif /* ! defined(SWIGIMPORTED) */

#include "ldastoolsal/gpstime.hh"

namespace LDASTools
{
  namespace AL
  {
    //---------------------------------------------------------------------
    /// \brief Provide profiling information
    ///
    /// This class is intended to be used for debugging purposes.
    /// It records the the time the instance was instantiated and then
    /// reports the differnce between the current time and the instantiated
    /// time so a developer can evaluate the speed of a routine.
    //---------------------------------------------------------------------
    class Profile
    {
    public:
      //-------------------------------------------------------------------
      /// \brief Constructor
      ///
      /// \param Header
      ///     This is a message which is to appear at the beginning of
      ///     each line of output.
      ///     It most commonly will be the fully qualified name of the
      ///     function or method that was invoked.
      //-------------------------------------------------------------------
      Profile( const char* Header );

      //-------------------------------------------------------------------
      /// \brief Output profiling information
      ///
      /// \param Filename
      ///     The name of the file being profiled.
      /// \param Line
      ///     The line number inside the file being profiled.
      /// \param Message
      ///     Text to appear immediately following the Header given
      ///     at instantiation.
      ///
      /// \return
      ///     The profiling data formatted as a string.
      //-------------------------------------------------------------------
      std::string operator()( const char* Filename, size_t Line,
			      const char* Message = ""  );

    private:
      const char*	m_header;
      GPSTime	m_start;
      GPSTime	m_previous;
      GPSTime	m_now;
    };

    inline Profile::
    Profile( const char* Header )
      : m_header( Header )
    {
      m_start.Now( );
      m_previous.Now( );
    }

    inline std::string Profile::
    operator()( const char* Filename, size_t Line, const char* Message )
    {
      std::ostringstream	msg;

      m_now.Now( );

      msg << m_header << Message
	  << " dt: " << ( m_now - m_previous )
	  << " cum: " << ( m_now - m_start )
	  << " [ line: " << Line
	  << " file: " << Filename << " ]"
	;
      m_previous = m_now;
    
      return msg.str( );
    }
  } // namespace - AL
} // namespace - LDASTools

#endif /* GENERAL__PROFILE_HH */
