/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 *
 * This file is part of GNU Radio
 *
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_non_blocker.h>
#include <gnuradio/io_signature.h>

#include <stdio.h>
//#include <typeinfo>

/*
 * Create a new instance of baz_non_blocker and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_non_blocker_sptr
baz_make_non_blocker (int item_size, /*gr::msg_queue::sptr queue, */bool blocking /*= false*/)
{
  return baz_non_blocker_sptr (new baz_non_blocker (item_size, /*queue, */blocking));
}

/*
 * The private constructor
 */
baz_non_blocker::baz_non_blocker (int item_size, /*gr::msg_queue::sptr queue, */bool blocking)
  : gr::block ("non_blocker",
		   gr::io_signature::make (1, 1, item_size),
		   gr::io_signature::make (1, 1, item_size))
  , d_item_size(item_size), /*d_queue(queue), */d_blocking(blocking)
  , d_blocking_forecasted(blocking)
{
  fprintf(stderr, "[%s] Blocking: %s\n", name().c_str(), (blocking ? "yes" : "no"));
}

/*
 * Our virtual destructor.
 */
baz_non_blocker::~baz_non_blocker ()
{
}

void baz_non_blocker::set_blocking(bool enable /*= true*/)
{
  if (d_blocking != enable)
    fprintf(stderr, "[%s] Changing blocking: %s -> %s\n", name().c_str(), (d_blocking ? "yes" :"no"), (enable ? "yes" : "no"));
  d_blocking = enable;
}

/*
    Problem is can't change from blocking back to non-blocking when upstream is blocked:
    - runtime will be waiting for more samples to arrive before calling 'forecast'
        (no way to kick runtime into calling this forecast/general_work when that happens)
    So this block will always be non-blocking (if set to blocking, will still pass samples through as if it was non-blocking, however through 'else' part)
*/

void baz_non_blocker::forecast(int noutput_items, gr_vector_int &ninput_items_required)
{
  d_blocking_forecasted = d_blocking;
  ninput_items_required[0] = 0; // This is the trick: never want any samples
/*  if (d_blocking_forecasted)
    gr::block::forecast(noutput_items, ninput_items_required);
  else {
    fprintf(stderr, "[%s] Not requiring any input items for %d output items\n", name().c_str(), noutput_items);
    ninput_items_required[0] = 0;
  }
*/}

int
baz_non_blocker::general_work (int noutput_items,
                              gr_vector_int &ninput_items,
			gr_vector_const_void_star &input_items,
			gr_vector_void_star &output_items)
{
  const char *in = (char*)input_items[0];
  char *out = (char*)output_items[0];

  //for (int i = 0; i < noutput_items; i++) {
  //  memcpy(out + (i * d_item_size), in + (i * d_item_size), d_item_size);
  //}

  if ((d_blocking_forecasted) && (noutput_items <= ninput_items[0]))
  {
    memcpy(out, in, d_item_size * noutput_items);
    consume(0, noutput_items);
  }
  else
  {
    //if (d_blocking_forecasted)    // This will happen when blocking but upstream is blocked
    //  fprintf(stderr, "[%s] Not enough items (%d requested, %d available)\n", name().c_str(), noutput_items, ninput_items[0]);
    int to_copy = std::min(noutput_items, ninput_items[0]); // Should always be the latter
    if (/*ninput_items[0]*/to_copy > 0) {  // Although forecast requires 0, this still continually consumes samples
      //fprintf(stderr, "[%s] Discarding %d unused items\n", name().c_str(), ninput_items[0]);  // FIXME: Could output these...
      memcpy(out, in, d_item_size * to_copy);
      consume(0, ninput_items[0]);
    }
    memset(out + (to_copy * d_item_size), 0x00, d_item_size * (noutput_items - to_copy));
  }

  return noutput_items;
}
