// Copyright 2022 - 2023 The nfp Authors. All rights reserved. Use of this
// source code is governed by a BSD-style license that can be found in the
// LICENSE file.
//
// This package NFP (Number Format Parser) produce syntax trees for number
// format expression. Excel Number format controls options such the number of
// decimal digits, the currency sign, commas to separate thousands, and
// display of negative numbers. The number format of an index applies wherever
// that index is used, including row or column headers of a table, or graph
// axis that uses that index.
//
// Implementation with Go language by Ri Xu: https://xuri.me

package nfp

import (
	"fmt"
	"testing"
)

var testStrings [][]string = [][]string{
	{"general", "[{Positive [{general General []}]}]"},
	{"0", "[{Positive [{0 ZeroPlaceHolder []}]}]"},
	{"0.00", "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"#,##0", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{"#,##0.00", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"0%", "[{Positive [{0 ZeroPlaceHolder []} {% Percent []}]}]"},
	{"0.00%", "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {% Percent []}]}]"},
	{"0.00e+00", "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {e+ Exponential []} {00 ZeroPlaceHolder []}]}]"},
	{"# ?/?", "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {? DigitalPlaceHolder []}]}]"},
	{"# ??/??", "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {?? DigitalPlaceHolder []}]}]"},
	{"mm-dd-yy", "[{Positive [{mm DateTimes []} {- Literal []} {dd DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{"d-mmm-yy", "[{Positive [{d DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{"d-mmm", "[{Positive [{d DateTimes []} {- Literal []} {mmm DateTimes []}]}]"},
	{"mmm-yy", "[{Positive [{mmm DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{"h:mm am/pm", "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {am/pm DateTimes []}]}]"},
	{"h:mm:ss am/pm", "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {am/pm DateTimes []}]}]"},
	{"hh:mm", "[{Positive [{hh DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{"hh:mm:ss", "[{Positive [{hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{"m/d/yy hh:mm", "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yy DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{"#,##0 ;(#,##0)", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []}]} {Negative [{( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"#,##0 ;[red](#,##0)", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []}]} {Negative [{red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"#,##0.00;(#,##0.00)", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"#,##0.00;[red](#,##0.00)", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`_(* #,##0_);_(* \(#,##0\);_(* "-"_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_("$"* #,##0_);_("$"* \(#,##0\);_("$"* "-"_);_(@_)`, "[{Positive [{$ Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{$ Literal []} {  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{$ Literal []} {  RepeatsChar []} {- Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0.00_);_(* \(#,##0.00\);_(* "-"??_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_("$"* #,##0.00_);_("$"* \(#,##0.00\);_("$"* "-"??_);_(@_)`, "[{Positive [{$ Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{$ Literal []} {  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]} {Zero [{$ Literal []} {  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{"mm:ss", "[{Positive [{mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{"[h]:mm:ss", "[{Positive [{h ElapsedDateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{"mmss.0", "[{Positive [{mm DateTimes []} {ss DateTimes []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]}]"},
	{"##0.0e+0", "[{Positive [{## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {e+ Exponential []} {0 ZeroPlaceHolder []}]}]"},
	{"@", "[{Text [{@ TextPlaceHolder []}]}]"},
	{`**x`, "[{Positive [{* RepeatsChar []} {x Literal []}]}]"},
	{`[$-]`, "[{Positive [{[$-] CurrencyLanguage []}]}]"},
	{"[$-ja-JP-x-gannen,80]", "[{Positive [{[$-ja-JP-x-gannen,80] CurrencyLanguage [{{ja-JP-x-gannen,80 LanguageInfo []} }]}]}]"},
	{"x0.00", "[{Positive [{x Literal []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"0.0bc00", "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {bc Literal []} {00 ZeroPlaceHolder []}]}]"},
	{"x00", "[{Positive [{x Literal []} {00 ZeroPlaceHolder []}]}]"},
	{"1.2", "[{Positive [{1 Literal []} {. DecimalPoint []} {2 Literal []}]}]"},
	{"#.2", "[{Positive [{# HashPlaceHolder []} {. DecimalPoint []} {2 Literal []}]}]"},
	{"1,2#", "[{Positive [{1 Literal []} {, ThousandsSeparator []} {2 Literal []} {# HashPlaceHolder []}]}]"},
	{`;;;;`, "[]"},
	{`;;;;@`, "[]"},
	{`;;; "xxx" @ ""`, "[{Positive []} {Negative []} {Zero []} {Text [{  Literal []} {xxx Literal []} {  Literal []} {@ TextPlaceHolder []} {  Literal []}]}]"},
	{`#''.#`, "[{Positive [{# HashPlaceHolder []} {. DecimalPoint []} {# HashPlaceHolder []}]}]"},
	{`#"''".#`, "[{Positive [{# HashPlaceHolder []} {'' Literal []} {. DecimalPoint []} {# HashPlaceHolder []}]}]"},
	{"[h]:m00m:ss", "[{Positive [{h ElapsedDateTimes []} {: Literal []} {m DateTimes []} {00 ZeroPlaceHolder []} {m DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{"bc[h]:mm:ss.0xxxx", "[{Positive [{bc Literal []} {h ElapsedDateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {xxxx Literal []}]}]"},
	{"[s].00.", "[{Positive [{s ElapsedDateTimes []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {. DecimalPoint []}]}]"},
	{"[s]\".00\"", "[{Positive [{s ElapsedDateTimes []} {. Literal []} {00 Literal []}]}]"},
	{"[s].##", "[{Positive [{s ElapsedDateTimes []} {. DecimalPoint []} {## HashPlaceHolder []}]}]"},
	{`上午/下午h"时"mm"分"ss"秒";@`, "[{Positive [{上午/下午 DateTimes []} {h DateTimes []} {时 Literal []} {mm DateTimes []} {分 Literal []} {ss DateTimes []} {秒 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`abcdefghijklmnopqrstuvwxyz`, "[{Positive [{a DateTimes []} {bc Literal []} {d DateTimes []} {e DateTimes []} {f Literal []} {g DateTimes []} {h DateTimes []} {ijkl Literal []} {m DateTimes []} {nopqr Literal []} {s DateTimes []} {tuvwx Literal []} {y DateTimes []} {z Literal []}]}]"},
	{`oeopwierop`, "[{Positive [{o Literal []} {e DateTimes []} {opwi Literal []} {e DateTimes []} {rop Literal []}]}]"},
	{`[DBNum1][$US-111]yyyy"年"m"月"d"日";@`, "[{Positive [{[DBNum1] SwitchArgument []} {[$US-111] CurrencyLanguage [{{US CurrencyString []} } {{111 LanguageInfo []} }]} {yyyy DateTimes []} {年 Literal []} {m DateTimes []} {月 Literal []} {d DateTimes []} {日 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`上午/下午 hh"時"mm"分"ss"秒"`, "[{Positive [{上午/下午 DateTimes []} {hh DateTimes []} {時 Literal []} {mm DateTimes []} {分 Literal []} {ss DateTimes []} {秒 Literal []}]}]"},
	{`am/pm hh"時"mm"分"ss"秒"`, "[{Positive [{am/pm DateTimes []} {  Literal []} {hh DateTimes []} {時 Literal []} {mm DateTimes []} {分 Literal []} {ss DateTimes []} {秒 Literal []}]}]"},
	{`[$-404]e/m/d`, "[{Positive [{[$-404] CurrencyLanguage [{{404 LanguageInfo []} }]} {e DateTimes []} {/ Literal []} {m DateTimes []} {/ Literal []} {d DateTimes []}]}]"},
	{`[$$-409]0#,##0`, "[{Positive [{[$$-409] CurrencyLanguage [{{$ CurrencyString []} } {{409 LanguageInfo []} }]} {0 ZeroPlaceHolder []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{`[$-412]AM/PM\ h"시"\ mm"분"\ ss"초";@`, "[{Positive [{[$-412] CurrencyLanguage [{{412 LanguageInfo []} }]} {AM/PM DateTimes []} {  Literal []} {h DateTimes []} {시 Literal []} {  Literal []} {mm DateTimes []} {분 Literal []} {  Literal []} {ss DateTimes []} {초 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`dddd,,, MMMM d,, yyyy,,,,`, "[{Positive [{dddd DateTimes []} {, Literal []} {  Literal []} {MMMM DateTimes []} {  Literal []} {d DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []} {, Literal []}]}]"},
	{`m/d/yyyy\ h:mm:ss AM/PM;@`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$SFr.-810] #,##0.00_);[Red]([$SFr.-810] #,##0.00\)`, "[{Positive [{[$SFr.-810] CurrencyLanguage [{{SFr. CurrencyString []} } {{810 LanguageInfo []} }]} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {( Literal []} {[$SFr.-810] CurrencyLanguage [{{SFr. CurrencyString []} } {{810 LanguageInfo []} }]} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`??/??`, "[{Positive [{?? DigitalPlaceHolder []} {/ Fraction []} {?? DigitalPlaceHolder []}]}]"},
	{"\" Excellent\"", "[{Positive [{ Excellent Literal []}]}]"},
	{"\" Fair\"", "[{Positive [{ Fair Literal []}]}]"},
	{"\" Good\"", "[{Positive [{ Good Literal []}]}]"},
	{"\" Poor\"", "[{Positive [{ Poor Literal []}]}]"},
	{"\" Very Good\"", "[{Positive [{ Very Good Literal []}]}]"},
	{"\"$\"#,##0", "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{"\"$\"#,##0.00", "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"\"$\"#,##0.00_,[Red]\\(\"$\"#,##0.00\\)", "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {Red Color []} {( Literal []} {$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"\"$\"#,##0.00_,\\(\"$\"#,##0.00\\)", "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {( Literal []} {$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"\"$\"#,##0;[Red]\\-\"$\"#,##0", "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {- Literal []} {$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{"\"$\"#,##0_,[Red]\\(\"$\"#,##0\\)", "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {Red Color []} {( Literal []} {$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"\"$\"#,##0_,\\(\"$\"#,##0\\)", "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {( Literal []} {$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"\"Haha!\"\\ @\\ \"Yeah!\"", "[{Text [{Haha! Literal []} {  Literal []} {@ TextPlaceHolder []} {  Literal []} {Yeah! Literal []}]}]"},
	{"\"TRUE\";\"TRUE\";\"FALSE\"", "[{Positive [{TRUE Literal []}]} {Negative [{TRUE Literal []}]} {Zero [{FALSE Literal []}]}]"},
	{"\"True\";\"True\";\"False\";@", "[{Positive [{True Literal []}]} {Negative [{True Literal []}]} {Zero [{False Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{"\"Years: \"0", "[{Positive [{Years:  Literal []} {0 ZeroPlaceHolder []}]}]"},
	{"\"Yes\";\"Yes\";\"No\";@", "[{Positive [{Yes Literal []}]} {Negative [{Yes Literal []}]} {Zero [{No Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{"\"kl \"hh:mm:ss;@", "[{Positive [{kl  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{"\"£\"#,##0.00", "[{Positive [{£ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"\"£\"#,##0;[Red]\\-\"£\"#,##0", "[{Positive [{£ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {- Literal []} {£ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{"\"€\"#,##0.00", "[{Positive [{€ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"\"€\"\\ #,##0.00_-", "[{Positive [{€ Literal []} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"\"上午/下午 \"hh\"時\"mm\"分\"ss\"秒 \"", "[{Positive [{上午/下午  Literal []} {hh DateTimes []} {時 Literal []} {mm DateTimes []} {分 Literal []} {ss DateTimes []} {秒  Literal []}]}]"},
	{"\"￥\"#,##0.00;\"￥\"\\-#,##0.00", "[{Positive [{￥ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{￥ Literal []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"#", "[{Positive [{# HashPlaceHolder []}]}]"},
	{"# ?/?", "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {? DigitalPlaceHolder []}]}]"},
	{"# ??/??", "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {?? DigitalPlaceHolder []}]}]"},
	{"#\" \"?/?", "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {? DigitalPlaceHolder []}]}]"},
	{"#\" \"??/??", "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {?? DigitalPlaceHolder []}]}]"},
	{"#\"abded\"\\ ??/??", "[{Positive [{# HashPlaceHolder []} {abded Literal []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {?? DigitalPlaceHolder []}]}]"},
	{"###0.00;-###0.00", "[{Positive [{### HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{- Literal []} {### HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"###0;-###0", "[{Positive [{### HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{- Literal []} {### HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{"##0.0E+0", "[{Positive [{## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {E+ Exponential []} {0 ZeroPlaceHolder []}]}]"},
	{"#,##0 ;(#,##0)", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []}]} {Negative [{( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"#,##0 ;[Red](#,##0)", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []}]} {Negative [{Red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{"#,##0\"p.\";[Red]\\-#,##0\"p.\"", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {p. Literal []}]} {Negative [{Red Color []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {p. Literal []}]}]"},
	{`#,##0.0`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]}]"},
	{`#,##0.00`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{"#,##0.00 \"\u0222\"", "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {\u0222 Literal []}]}]"},
	{`#,##0.00 €;-#,##0.00 €`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} { € Literal []}]} {Negative [{- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} { € Literal []}]}]"},
	{`#,##0.00"p.";[Red]\-#,##0.00"p."`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {p. Literal []}]} {Negative [{Red Color []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {p. Literal []}]}]"},
	{`#,##0.000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]}]"},
	{`#,##0.0000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []}]}]"},
	{`#,##0.00000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00000 ZeroPlaceHolder []}]}]"},
	{`#,##0.000000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000000 ZeroPlaceHolder []}]}]"},
	{`#,##0.0000000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0000000 ZeroPlaceHolder []}]}]"},
	{`#,##0.00000000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00000000 ZeroPlaceHolder []}]}]"},
	{`#,##0.000000000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000000000 ZeroPlaceHolder []}]}]"},
	{`#,##0.00000000;[Red]#,##0.00000000`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00000000 ZeroPlaceHolder []}]} {Negative [{Red Color []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00000000 ZeroPlaceHolder []}]}]"},
	{`#,##0.0000_ `, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []}]}]"},
	{`#,##0.000_ `, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]}]"},
	{`#,##0.000_,\(#,##0.000\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0.00;(#,##0.00)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0.00;(#,##0.00,0.00`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {, ThousandsSeparator []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`#,##0.00;[Red](#,##0.00)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0.00;[Red]\(#,##0.00\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0.00;\(#,##0.00\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0.00[$₹-449]_,\(#,##0.00[$₹-449]\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {[$₹-449] CurrencyLanguage [{{₹ CurrencyString []} } {{449 LanguageInfo []} }]} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {[$₹-449] CurrencyLanguage [{{₹ CurrencyString []} } {{449 LanguageInfo []} }]} {) Literal []}]}]"},
	{`#,##0.00\ "p."`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {p. Literal []}]}]"},
	{`#,##0.00\ "p.";[Red]\-#,##0.00\ "p."`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {p. Literal []}]} {Negative [{Red Color []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {p. Literal []}]}]"},
	{`#,##0.00\ [$€-407]`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {[$€-407] CurrencyLanguage [{{€ CurrencyString []} } {{407 LanguageInfo []} }]}]}]"},
	{`#,##0.00\ [$€-40C]`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {[$€-40C] CurrencyLanguage [{{€ CurrencyString []} } {{40C LanguageInfo []} }]}]}]"},
	{`#,##0.00_,\(#,##0.00\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0.00_р_.;[Red]\-#,##0.00_р_.`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`#,##0.00_р_.;\-#,##0.00_р_.`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`#,##0.0;[Red]#,##0.0`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]}]"},
	{`#,##0.0_ ;\-#,##0.0\ `, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]} {Negative [{- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} { Literal []}]}]"},
	{`#,##0.0_,[Red]\(#,##0.0\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {Red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0.0_,\(#,##0.0\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0;\-#,##0;0`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Zero [{0 ZeroPlaceHolder []}]}]"},
	{`#,##0\ "р.";[Red]\-#,##0\ "р."`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []} {р. Literal []}]} {Negative [{Red Color []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []} {р. Literal []}]}]"},
	{`#,##0\ "р.";\-#,##0\ "р."`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []} {р. Literal []}]} {Negative [{- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []} {р. Literal []}]}]"},
	{`#,##0\ ;[Red]\(#,##0\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []}]} {Negative [{Red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0\ ;\(#,##0\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []}]} {Negative [{( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0_ `, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{`#,##0_ ;[Red]\-#,##0\ `, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} { Literal []}]}]"},
	{`#,##0_,[Red]\(#,##0\)`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {Red Color []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`#,##0_р_.;[Red]\-#,##0_р_.`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{`#,##0_р_.;\-#,##0_р_.`, "[{Positive [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{`#.0000,,`, "[{Positive [{# HashPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []} {, ThousandsSeparator []} {, Literal []}]}]"},
	{`#0`, "[{Positive [{# HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{`#0.00`, "[{Positive [{# HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`#0.0000`, "[{Positive [{# HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []}]}]"},
	{`#\ ?/10`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {10 Denominator []}]}]"},
	{`#\ ?/2`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {2 Denominator []}]}]"},
	{`#\ ?/4`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {4 Denominator []}]}]"},
	{`#\ ?/8`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {8 Denominator []}]}]"},
	{`#\ ?/?`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {? DigitalPlaceHolder []} {/ Fraction []} {? DigitalPlaceHolder []}]}]"},
	{`#\ ??/100`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {100 Denominator []}]}]"},
	{`#\ ??/100;[Red]\(#\ ??/16\)`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {100 Denominator []}]} {Negative [{Red Color []} {( Literal []} {# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {16 Denominator []} {) Literal []}]}]"},
	{`#\ ??/16`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {16 Denominator []}]}]"},
	{`#\ ??/??`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {?? DigitalPlaceHolder []}]}]"},
	{`#\ ??/?????????`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {?? DigitalPlaceHolder []} {/ Fraction []} {????????? DigitalPlaceHolder []}]}]"},
	{`#\ ???/???`, "[{Positive [{# HashPlaceHolder []} {  Literal []} {??? DigitalPlaceHolder []} {/ Fraction []} {??? DigitalPlaceHolder []}]}]"},
	{`**\ #,###,#00,000.00,**`, "[{Positive [{* RepeatsChar []} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {### HashPlaceHolder []} {, ThousandsSeparator []} {# HashPlaceHolder []} {00 ZeroPlaceHolder []} {, ThousandsSeparator []} {000 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {, ThousandsSeparator []} {* RepeatsChar []}]}]"},
	{`0`, "[{Positive [{0 ZeroPlaceHolder []}]}]"},
	{`0"abcde".0"??"000E+00`, "[{Positive [{0 ZeroPlaceHolder []} {abcde Literal []} {. DecimalPoint []} {0 ZeroPlaceHolder []} { Literal []} {?? Literal []} {000 ZeroPlaceHolder []} {E+ Exponential []} {00 ZeroPlaceHolder []}]}]"},
	{`0%`, "[{Positive [{0 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`0.0`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]}]"},
	{`0.0%`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`0.00`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`0.00"°"`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {° Literal []}]}]"},
	{`0.00%`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`0.000`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]}]"},
	{`0.000%`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`0.0000`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []}]}]"},
	{`0.000000`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {000000 ZeroPlaceHolder []}]}]"},
	{`0.00000000`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00000000 ZeroPlaceHolder []}]}]"},
	{`0.000000000`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {000000000 ZeroPlaceHolder []}]}]"},
	{`0.000000000%`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {000000000 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`0.00000000000`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00000000000 ZeroPlaceHolder []}]}]"},
	{`0.000000000000000`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {000000000000000 ZeroPlaceHolder []}]}]"},
	{`0.00000000E+00`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00000000 ZeroPlaceHolder []} {E+ Exponential []} {00 ZeroPlaceHolder []}]}]"},
	{`0.0000E+00`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []} {E+ Exponential []} {00 ZeroPlaceHolder []}]}]"},
	{`0.00;[Red]0.00`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`0.00E+00`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {E+ Exponential []} {00 ZeroPlaceHolder []}]}]"},
	{`0.00_);[Red]\(0.00\)`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {( Literal []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`0.00_);\(0.00\)`, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{( Literal []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`0.0_ `, "[{Positive [{0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]}]"},
	{`00.00.00.000`, "[{Positive [{00 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]}]"},
	{`00.000%`, "[{Positive [{00 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`0000`, "[{Positive [{0000 ZeroPlaceHolder []}]}]"},
	{`00000`, "[{Positive [{00000 ZeroPlaceHolder []}]}]"},
	{`00000000`, "[{Positive [{00000000 ZeroPlaceHolder []}]}]"},
	{`000000000`, "[{Positive [{000000000 ZeroPlaceHolder []}]}]"},
	{`00000\-0000`, "[{Positive [{00000 ZeroPlaceHolder []} {- Literal []} {0000 ZeroPlaceHolder []}]}]"},
	{`00000\-00000`, "[{Positive [{00000 ZeroPlaceHolder []} {- Literal []} {00000 ZeroPlaceHolder []}]}]"},
	{`000\-00\-0000`, "[{Positive [{000 ZeroPlaceHolder []} {- Literal []} {00 ZeroPlaceHolder []} {- Literal []} {0000 ZeroPlaceHolder []}]}]"},
	{`0;[Red]0`, "[{Positive [{0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {0 ZeroPlaceHolder []}]}]"},
	{`0\-00000\-00000\-0`, "[{Positive [{0 ZeroPlaceHolder []} {- Literal []} {00000 ZeroPlaceHolder []} {- Literal []} {00000 ZeroPlaceHolder []} {- Literal []} {0 ZeroPlaceHolder []}]}]"},
	{`0_);[Red]\(0\)`, "[{Positive [{0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {( Literal []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`0_);\(0\)`, "[{Positive [{0 ZeroPlaceHolder []}]} {Negative [{( Literal []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`A/P`, "[{Positive [{A/P DateTimes []}]}]"},
	{`AM/PM`, "[{Positive [{AM/PM DateTimes []}]}]"},
	{`AM/PMh"時"mm"分"ss"秒";@`, "[{Positive [{AM/PM DateTimes []} {h DateTimes []} {時 Literal []} {mm DateTimes []} {分 Literal []} {ss DateTimes []} {秒 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`D`, "[{Positive [{D DateTimes []}]}]"},
	{`DD`, "[{Positive [{DD DateTimes []}]}]"},
	{`DD/MM/YY;@`, "[{Positive [{DD DateTimes []} {/ Literal []} {MM DateTimes []} {/ Literal []} {YY DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`DD/MM/YYYY`, "[{Positive [{DD DateTimes []} {/ Literal []} {MM DateTimes []} {/ Literal []} {YYYY DateTimes []}]}]"},
	{`DD/MM/YYYY;@`, "[{Positive [{DD DateTimes []} {/ Literal []} {MM DateTimes []} {/ Literal []} {YYYY DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`DDD`, "[{Positive [{DDD DateTimes []}]}]"},
	{`DDDD`, "[{Positive [{DDDD DateTimes []}]}]"},
	{`DDDD", "MMMM\ DD", "YYYY`, "[{Positive [{DDDD DateTimes []} {,  Literal []} {MMMM DateTimes []} {  Literal []} {DD DateTimes []} {,  Literal []} {YYYY DateTimes []}]}]"},
	{`GENERAL`, "[{Positive [{GENERAL General []}]}]"},
	{`General`, "[{Positive [{General General []}]}]"},
	{`H`, "[{Positive [{H DateTimes []}]}]"},
	{`H:MM:SS\ AM/PM`, "[{Positive [{H DateTimes []} {: Literal []} {MM DateTimes []} {: Literal []} {SS DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`HH:MM`, "[{Positive [{HH DateTimes []} {: Literal []} {MM DateTimes []}]}]"},
	{`HH:MM:SS\ AM/PM`, "[{Positive [{HH DateTimes []} {: Literal []} {MM DateTimes []} {: Literal []} {SS DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`HHM`, "[{Positive [{HH DateTimes []} {M DateTimes []}]}]"},
	{`HHMM`, "[{Positive [{HH DateTimes []} {MM DateTimes []}]}]"},
	{`HH[MM]`, "[{Positive [{HH DateTimes []} {MM ElapsedDateTimes []}]}]"},
	{`HH[M]`, "[{Positive [{HH DateTimes []} {M ElapsedDateTimes []}]}]"},
	{`M/D/YYYY`, "[{Positive [{M DateTimes []} {/ Literal []} {D DateTimes []} {/ Literal []} {YYYY DateTimes []}]}]"},
	{`M/D/YYYY\ H:MM`, "[{Positive [{M DateTimes []} {/ Literal []} {D DateTimes []} {/ Literal []} {YYYY DateTimes []} {  Literal []} {H DateTimes []} {: Literal []} {MM DateTimes []}]}]"},
	{`MM/DD/YY`, "[{Positive [{MM DateTimes []} {/ Literal []} {DD DateTimes []} {/ Literal []} {YY DateTimes []}]}]"},
	{`S`, "[{Positive [{S DateTimes []}]}]"},
	{`SS`, "[{Positive [{SS DateTimes []}]}]"},
	{`YY`, "[{Positive [{YY DateTimes []}]}]"},
	{`YYM`, "[{Positive [{YY DateTimes []} {M DateTimes []}]}]"},
	{`YYMM`, "[{Positive [{YY DateTimes []} {MM DateTimes []}]}]"},
	{`YYMMM`, "[{Positive [{YY DateTimes []} {MMM DateTimes []}]}]"},
	{`YYMMMM`, "[{Positive [{YY DateTimes []} {MMMM DateTimes []}]}]"},
	{`YYMMMMM`, "[{Positive [{YY DateTimes []} {MMMMM DateTimes []}]}]"},
	{`YYYY`, "[{Positive [{YYYY DateTimes []}]}]"},
	{`YYYY-MM-DD HH:MM:SS`, "[{Positive [{YYYY DateTimes []} {- Literal []} {MM DateTimes []} {- Literal []} {DD DateTimes []} {  Literal []} {HH DateTimes []} {: Literal []} {MM DateTimes []} {: Literal []} {SS DateTimes []}]}]"},
	{`YYYY\-MM\-DD`, "[{Positive [{YYYY DateTimes []} {- Literal []} {MM DateTimes []} {- Literal []} {DD DateTimes []}]}]"},
	{`[$$-409]#,##0`, "[{Positive [{[$$-409] CurrencyLanguage [{{$ CurrencyString []} } {{409 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]}]"},
	{`[$$-409]#,##0.00`, "[{Positive [{[$$-409] CurrencyLanguage [{{$ CurrencyString []} } {{409 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[$$-409]#,##0.00_);[Red]\([$$-409]#,##0.00\)`, "[{Positive [{[$$-409] CurrencyLanguage [{{$ CurrencyString []} } {{409 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {( Literal []} {[$$-409] CurrencyLanguage [{{$ CurrencyString []} } {{409 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`[$$-C09]#,##0.00`, "[{Positive [{[$$-C09] CurrencyLanguage [{{$ CurrencyString []} } {{C09 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[$-100042A]h:mm:ss\ AM/PM;@`, "[{Positive [{[$-100042A] CurrencyLanguage [{{100042A LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-1010409]0.000%`, "[{Positive [{[$-1010409] CurrencyLanguage [{{1010409 LanguageInfo []} }]} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`[$-1010409]General`, "[{Positive [{[$-1010409] CurrencyLanguage [{{1010409 LanguageInfo []} }]} {General General []}]}]"},
	{`[$-1010409]d/m/yyyy\ h:mm\ AM/PM;@`, "[{Positive [{[$-1010409] CurrencyLanguage [{{1010409 LanguageInfo []} }]} {d DateTimes []} {/ Literal []} {m DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-1010409]dddd, mmmm dd, yyyy`, "[{Positive [{[$-1010409] CurrencyLanguage [{{1010409 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {dd DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]}]"},
	{`[$-1010409]m/d/yyyy`, "[{Positive [{[$-1010409] CurrencyLanguage [{{1010409 LanguageInfo []} }]} {m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yyyy DateTimes []}]}]"},
	{`[$-1409]h:mm:ss\ AM/PM;@`, "[{Positive [{[$-1409] CurrencyLanguage [{{1409 LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-2000000]h:mm:ss;@`, "[{Positive [{[$-2000000] CurrencyLanguage [{{2000000 LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-2010401]d/mm/yyyy\ h:mm\ AM/PM;@`, "[{Positive [{[$-2010401] CurrencyLanguage [{{2010401 LanguageInfo []} }]} {d DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-4000439]h:mm:ss\ AM/PM;@`, "[{Positive [{[$-4000439] CurrencyLanguage [{{4000439 LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-4010439]d/m/yyyy\ h:mm\ AM/PM;@`, "[{Positive [{[$-4010439] CurrencyLanguage [{{4010439 LanguageInfo []} }]} {d DateTimes []} {/ Literal []} {m DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]AM/PM\ hh:mm:ss;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {AM/PM DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]d/m/yyyy\ hh:mm;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {d DateTimes []} {/ Literal []} {m DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]d\-mmm;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {d DateTimes []} {- Literal []} {mmm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]d\-mmm\-yy;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {d DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]d\-mmm\-yyyy;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {d DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]dd/mm/yyyy\ hh:mm;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]dd\-mmm\-yy;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {dd DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]h:mm:ss\ AM/PM;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]h:mm\ AM/PM;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]m/d/yy\ h:mm\ AM/PM;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]mmm\-yy;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]mmmm\ d\,\ yyyy;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmmm DateTimes []} {  Literal []} {d DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]mmmm\-yy;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]mmmmm;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmmmm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-409]mmmmm\-yy;@`, "[{Positive [{[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmmmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-40E]h\ "óra"\ m\ "perckor"\ AM/PM;@`, "[{Positive [{[$-40E] CurrencyLanguage [{{40E LanguageInfo []} }]} {h DateTimes []} {  Literal []} {óra Literal []} {  Literal []} {m DateTimes []} {  Literal []} {perckor Literal []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-412]AM/PM\ h"시"\ mm"분"\ ss"초";@`, "[{Positive [{[$-412] CurrencyLanguage [{{412 LanguageInfo []} }]} {AM/PM DateTimes []} {  Literal []} {h DateTimes []} {시 Literal []} {  Literal []} {mm DateTimes []} {분 Literal []} {  Literal []} {ss DateTimes []} {초 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-41C]h:mm:ss\.AM/PM;@`, "[{Positive [{[$-41C] CurrencyLanguage [{{41C LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {. Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-449]hh:mm:ss\ AM/PM;@`, "[{Positive [{[$-449] CurrencyLanguage [{{449 LanguageInfo []} }]} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-44E]hh:mm:ss\ AM/PM;@`, "[{Positive [{[$-44E] CurrencyLanguage [{{44E LanguageInfo []} }]} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-44F]hh:mm:ss\ AM/PM;@`, "[{Positive [{[$-44F] CurrencyLanguage [{{44F LanguageInfo []} }]} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-D000409]h:mm\ AM/PM;@`, "[{Positive [{[$-D000409] CurrencyLanguage [{{D000409 LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-D010000]d/mm/yyyy\ h:mm\ "น.";@`, "[{Positive [{[$-D010000] CurrencyLanguage [{{D010000 LanguageInfo []} }]} {d DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {น. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[$-F400]h:mm:ss\ AM/PM`, "[{Positive [{[$-F400] CurrencyLanguage [{{F400 LanguageInfo []} }]} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`[$-F800]dddd\,\ mmmm\ dd\,\ yyyy`, "[{Positive [{[$-F800] CurrencyLanguage [{{F800 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {dd DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]}]"},
	{`[$AUD]\ #,##0.00`, "[{Positive [{[$AUD] CurrencyLanguage [{{AUD CurrencyString []} }]} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[$RD$-1C0A]#,##0.00;[Red]\-[$RD$-1C0A]#,##0.00`, "[{Positive [{[$RD$-1C0A] CurrencyLanguage [{{RD$ CurrencyString []} } {{1C0A LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {- Literal []} {[$RD$-1C0A] CurrencyLanguage [{{RD$ CurrencyString []} } {{1C0A LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[$SFr.-810]\ #,##0.00_);[Red]\([$SFr.-810]\ #,##0.00\)`, "[{Positive [{[$SFr.-810] CurrencyLanguage [{{SFr. CurrencyString []} } {{810 LanguageInfo []} }]} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {( Literal []} {[$SFr.-810] CurrencyLanguage [{{SFr. CurrencyString []} } {{810 LanguageInfo []} }]} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`[$£-809]#,##0.00;[Red][$£-809]#,##0.00`, "[{Positive [{[$£-809] CurrencyLanguage [{{£ CurrencyString []} } {{809 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{Red Color []} {[$£-809] CurrencyLanguage [{{£ CurrencyString []} } {{809 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[$¥-411]#,##0.00`, "[{Positive [{[$¥-411] CurrencyLanguage [{{¥ CurrencyString []} } {{411 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[$¥-804]#,##0.00`, "[{Positive [{[$¥-804] CurrencyLanguage [{{¥ CurrencyString []} } {{804 LanguageInfo []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[<0]"";0%`, "[{Positive [{<0 Condition [{{< Operator []} } {{0 Operand []} }]} { Literal []}]} {Negative [{0 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`[<=9999999]###\-####;\(###\)\ ###\-####`, "[{Positive [{<=9999999 Condition [{{<= Operator []} } {{9999999 Operand []} }]} {### HashPlaceHolder []} {- Literal []} {#### HashPlaceHolder []}]} {Negative [{( Literal []} {### HashPlaceHolder []} {) Literal []} {  Literal []} {### HashPlaceHolder []} {- Literal []} {#### HashPlaceHolder []}]}]"},
	{`[=0]?;#,##0.00`, "[{Positive [{=0 Condition [{{= Operator []} } {{0 Operand []} }]} {? DigitalPlaceHolder []}]} {Negative [{# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[=0]?;0%`, "[{Positive [{=0 Condition [{{= Operator []} } {{0 Operand []} }]} {? DigitalPlaceHolder []}]} {Negative [{0 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`[=0]?;[<4.16666666666667][hh]:mm:ss;[hh]:mm`, "[{Positive [{=0 Condition [{{= Operator []} } {{0 Operand []} }]} {? DigitalPlaceHolder []}]} {Negative [{<4.16666666666667 Condition [{{< Operator []} } {{4.16666666666667 Operand []} }]} {hh ElapsedDateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]} {Zero [{hh ElapsedDateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{`[>999999]#,,"M";[>999]#,"K";#`, "[{Positive [{>999999 Condition [{{> Operator []} } {{999999 Operand []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {, Literal []} {M Literal []}]} {Negative [{>999 Condition [{{> Operator []} } {{999 Operand []} }]} {# HashPlaceHolder []} {, ThousandsSeparator []} {K Literal []}]} {Zero [{# HashPlaceHolder []}]}]"},
	{`[>999999]#.000,,"M";[>999]#.000,"K";#.000`, "[{Positive [{>999999 Condition [{{> Operator []} } {{999999 Operand []} }]} {# HashPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {, ThousandsSeparator []} {, Literal []} {M Literal []}]} {Negative [{>999 Condition [{{> Operator []} } {{999 Operand []} }]} {# HashPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {, ThousandsSeparator []} {K Literal []}]} {Zero [{# HashPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]}]"},
	{`[>=100000]0.000\ \";[Red]0.000\ \<\ \>\ \"\ \&\ \'\ `, "[{Positive [{>=100000 Condition [{{>= Operator []} } {{100000 Operand []} }]} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {  Literal []} {\" Literal []}]} {Negative [{Red Color []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {  Literal []} {< Literal []} {  Literal []} {> Literal []} {  Literal []} {\" Literal []} {  Literal []} {& Literal []} {  Literal []} {' Literal []} { Literal []}]}]"},
	{`[>=100000]0.000\ \<;[Red]0.000\ \>`, "[{Positive [{>=100000 Condition [{{>= Operator []} } {{100000 Operand []} }]} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {  Literal []} {< Literal []}]} {Negative [{Red Color []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {  Literal []} {> Literal []}]}]"},
	{`[BLACK]@`, "[{Text [{BLACK Color []} {@ TextPlaceHolder []}]}]"},
	{`[BLUE]GENERAL`, "[{Positive [{BLUE Color []} {GENERAL General []}]}]"},
	{`[Black]@`, "[{Text [{Black Color []} {@ TextPlaceHolder []}]}]"},
	{`[Blue]General`, "[{Positive [{Blue Color []} {General General []}]}]"},
	{`[CYAN]@`, "[{Text [{CYAN Color []} {@ TextPlaceHolder []}]}]"},
	{`[Cyan]@`, "[{Text [{Cyan Color []} {@ TextPlaceHolder []}]}]"},
	{`[DBNum1][$-804]AM/PMh"时"mm"分";@`, "[{Positive [{[DBNum1] SwitchArgument []} {[$-804] CurrencyLanguage [{{804 LanguageInfo []} }]} {AM/PM DateTimes []} {h DateTimes []} {时 Literal []} {mm DateTimes []} {分 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[DBNum1][$-804]General`, "[{Positive [{[DBNum1] SwitchArgument []} {[$-804] CurrencyLanguage [{{804 LanguageInfo []} }]} {General General []}]}]"},
	{`[DBNum1][$-804]h"时"mm"分";@`, "[{Positive [{[DBNum1] SwitchArgument []} {[$-804] CurrencyLanguage [{{804 LanguageInfo []} }]} {h DateTimes []} {时 Literal []} {mm DateTimes []} {分 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-1004]dddd\,\ d\ mmmm\,\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-1004] CurrencyLanguage [{{1004 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-101040D]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-101040D] CurrencyLanguage [{{101040D LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-101042A]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-101042A] CurrencyLanguage [{{101042A LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-140C]dddd\ "YeahWoo!"\ ddd\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-140C] CurrencyLanguage [{{140C LanguageInfo []} }]} {dddd DateTimes []} {  Literal []} {YeahWoo! Literal []} {  Literal []} {ddd DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-2C0A]dddd\ d" de "mmmm" de "yyyy;@`, "[{Positive [{ENG Unknown []} {[$-2C0A] CurrencyLanguage [{{2C0A LanguageInfo []} }]} {dddd DateTimes []} {  Literal []} {d DateTimes []} { de  Literal []} {mmmm DateTimes []} { de  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-402]dd\ mmmm\ yyyy\ "г.";@`, "[{Positive [{ENG Unknown []} {[$-402] CurrencyLanguage [{{402 LanguageInfo []} }]} {dd DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []} {  Literal []} {г. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-403]dddd\,\ d" / "mmmm" / "yyyy;@`, "[{Positive [{ENG Unknown []} {[$-403] CurrencyLanguage [{{403 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {d DateTimes []} { /  Literal []} {mmmm DateTimes []} { /  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-405]d\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-405] CurrencyLanguage [{{405 LanguageInfo []} }]} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-408]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-408] CurrencyLanguage [{{408 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-409]d\-mmm;@`, "[{Positive [{ENG Unknown []} {[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {d DateTimes []} {- Literal []} {mmm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-409]d\-mmm\-yy;@`, "[{Positive [{ENG Unknown []} {[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {d DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-409]d\-mmm\-yyyy;@`, "[{Positive [{ENG Unknown []} {[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {d DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-409]dd\-mmm\-yy;@`, "[{Positive [{ENG Unknown []} {[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {dd DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-409]mmm\-yy;@`, "[{Positive [{ENG Unknown []} {[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-409]mmmm\ d\,\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmmm DateTimes []} {  Literal []} {d DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-409]mmmm\-yy;@`, "[{Positive [{ENG Unknown []} {[$-409] CurrencyLanguage [{{409 LanguageInfo []} }]} {mmmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-40B]d\.\ mmmm\t\a\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-40B] CurrencyLanguage [{{40B LanguageInfo []} }]} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {t Literal []} {a Literal []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-40C]d/mmm/yyyy;@`, "[{Positive [{ENG Unknown []} {[$-40C] CurrencyLanguage [{{40C LanguageInfo []} }]} {d DateTimes []} {/ Literal []} {mmm DateTimes []} {/ Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-40E]yyyy/\ mmmm\ d\.;@`, "[{Positive [{ENG Unknown []} {[$-40E] CurrencyLanguage [{{40E LanguageInfo []} }]} {yyyy DateTimes []} {/ Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {d DateTimes []} {. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-40F]dd\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-40F] CurrencyLanguage [{{40F LanguageInfo []} }]} {dd DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-410]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-410] CurrencyLanguage [{{410 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-415]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-415] CurrencyLanguage [{{415 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-416]d\ \ mmmm\,\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-416] CurrencyLanguage [{{416 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {  Literal []} {mmmm DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-418]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-418] CurrencyLanguage [{{418 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-41A]d\.\ mmmm\ yyyy\.;@`, "[{Positive [{ENG Unknown []} {[$-41A] CurrencyLanguage [{{41A LanguageInfo []} }]} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []} {. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-41B]d\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-41B] CurrencyLanguage [{{41B LanguageInfo []} }]} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-41D]"den "\ d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-41D] CurrencyLanguage [{{41D LanguageInfo []} }]} {den  Literal []} {  Literal []} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-420]dddd\,\ dd\ mmmm\,\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-420] CurrencyLanguage [{{420 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {dd DateTimes []} {  Literal []} {mmmm DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-421]dd\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-421] CurrencyLanguage [{{421 LanguageInfo []} }]} {dd DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-424]dddd\,\ d\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-424] CurrencyLanguage [{{424 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-425]dddd\,\ d\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-425] CurrencyLanguage [{{425 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-426]dddd\,\ yyyy". gada "d\.\ mmmm;@`, "[{Positive [{ENG Unknown []} {[$-426] CurrencyLanguage [{{426 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []} {. gada  Literal []} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-427]yyyy\ "m."\ mmmm\ d\ "d.";@`, "[{Positive [{ENG Unknown []} {[$-427] CurrencyLanguage [{{427 LanguageInfo []} }]} {yyyy DateTimes []} {  Literal []} {m. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {d DateTimes []} {  Literal []} {d. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-42B]dddd\,\ d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-42B] CurrencyLanguage [{{42B LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-42C]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-42C] CurrencyLanguage [{{42C LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-42D]yyyy"(e)ko"\ mmmm"ren"\ d"a";@`, "[{Positive [{ENG Unknown []} {[$-42D] CurrencyLanguage [{{42D LanguageInfo []} }]} {yyyy DateTimes []} {(e)ko Literal []} {  Literal []} {mmmm DateTimes []} {ren Literal []} {  Literal []} {d DateTimes []} {a Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-42F]dddd\,\ dd\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-42F] CurrencyLanguage [{{42F LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {dd DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-437]yyyy\ \წ\ლ\ი\ს\ dd\ mm\,\ dddd;@`, "[{Positive [{ENG Unknown []} {[$-437] CurrencyLanguage [{{437 LanguageInfo []} }]} {yyyy DateTimes []} {  Literal []} {წ Literal []} {ლ Literal []} {ი Literal []} {ს Literal []} {  Literal []} {dd DateTimes []} {  Literal []} {mm DateTimes []} {, Literal []} {  Literal []} {dddd DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-438]d\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-438] CurrencyLanguage [{{438 LanguageInfo []} }]} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-43F]d\ mmmm\ yyyy\ "ж.";@`, "[{Positive [{ENG Unknown []} {[$-43F] CurrencyLanguage [{{43F LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []} {  Literal []} {ж. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-444]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-444] CurrencyLanguage [{{444 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-449]dd\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-449] CurrencyLanguage [{{449 LanguageInfo []} }]} {dd DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-44E]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-44E] CurrencyLanguage [{{44E LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-44F]dd\ mmmm\ yyyy\ dddd;@`, "[{Positive [{ENG Unknown []} {[$-44F] CurrencyLanguage [{{44F LanguageInfo []} }]} {dd DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []} {  Literal []} {dddd DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-457]dd\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-457] CurrencyLanguage [{{457 LanguageInfo []} }]} {dd DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-813]dddd\ d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-813] CurrencyLanguage [{{813 LanguageInfo []} }]} {dddd DateTimes []} {  Literal []} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-81A]dddd\,\ d\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-81A] CurrencyLanguage [{{81A LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {d DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-82C]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-82C] CurrencyLanguage [{{82C LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-843]yyyy\ "й""и""л"\ d/mmmm;@`, "[{Positive [{ENG Unknown []} {[$-843] CurrencyLanguage [{{843 LanguageInfo []} }]} {yyyy DateTimes []} {  Literal []} {й Literal []} {и Literal []} {л Literal []} {  Literal []} {d DateTimes []} {/ Literal []} {mmmm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-C07]dddd\,\ dd\.\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-C07] CurrencyLanguage [{{C07 LanguageInfo []} }]} {dddd DateTimes []} {, Literal []} {  Literal []} {dd DateTimes []} {. Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-FC19]yyyy\,\ dd\ mmmm;@`, "[{Positive [{ENG Unknown []} {[$-FC19] CurrencyLanguage [{{FC19 LanguageInfo []} }]} {yyyy DateTimes []} {, Literal []} {  Literal []} {dd DateTimes []} {  Literal []} {mmmm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-FC22]d\ mmmm\ yyyy" р.";@`, "[{Positive [{ENG Unknown []} {[$-FC22] CurrencyLanguage [{{FC22 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []} { р. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[ENG][$-FC23]d\ mmmm\ yyyy;@`, "[{Positive [{ENG Unknown []} {[$-FC23] CurrencyLanguage [{{FC23 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[GREEN]#,###`, "[{Positive [{GREEN Color []} {# HashPlaceHolder []} {, ThousandsSeparator []} {### HashPlaceHolder []}]}]"},
	{`[Green]#,###`, "[{Positive [{Green Color []} {# HashPlaceHolder []} {, ThousandsSeparator []} {### HashPlaceHolder []}]}]"},
	{`[HH]`, "[{Positive [{HH ElapsedDateTimes []}]}]"},
	{`[HIJ][$-2060401]d/mm/yyyy\ h:mm\ AM/PM;@`, "[{Positive [{HIJ Unknown []} {[$-2060401] CurrencyLanguage [{{2060401 LanguageInfo []} }]} {d DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[HIJ][$-2060401]d\ mmmm\ yyyy;@`, "[{Positive [{HIJ Unknown []} {[$-2060401] CurrencyLanguage [{{2060401 LanguageInfo []} }]} {d DateTimes []} {  Literal []} {mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[H]`, "[{Positive [{H ElapsedDateTimes []}]}]"},
	{`[JPN][$-411]gggyy"年"m"月"d"日"\ dddd;@`, "[{Positive [{JPN Unknown []} {[$-411] CurrencyLanguage [{{411 LanguageInfo []} }]} {ggg DateTimes []} {yy DateTimes []} {年 Literal []} {m DateTimes []} {月 Literal []} {d DateTimes []} {日 Literal []} {  Literal []} {dddd DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[MAGENTA]0.00`, "[{Positive [{MAGENTA Color []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[Magenta]0.00`, "[{Positive [{Magenta Color []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`[RED]#.##`, "[{Positive [{RED Color []} {# HashPlaceHolder []} {. DecimalPoint []} {## HashPlaceHolder []}]}]"},
	{`[Red]#.##`, "[{Positive [{Red Color []} {# HashPlaceHolder []} {. DecimalPoint []} {## HashPlaceHolder []}]}]"},
	{`[Red][<-25]General;[Blue][>25]General;[Green]General;[Yellow]General\ `, "[{Positive [{Red Color []} {<-25 Condition [{{< Operator []} } {{-25 Operand []} }]} {General General []}]} {Negative [{Blue Color []} {>25 Condition [{{> Operator []} } {{25 Operand []} }]} {General General []}]} {Zero [{Green Color []} {General General []}]} {Text [{Yellow Color []} {General General []} { Literal []}]}]"},
	{`[Red][<=-25]General;[Blue][>=25]General;[Green]General;[Yellow]General`, "[{Positive [{Red Color []} {<=-25 Condition [{{<= Operator []} } {{-25 Operand []} }]} {General General []}]} {Negative [{Blue Color []} {>=25 Condition [{{>= Operator []} } {{25 Operand []} }]} {General General []}]} {Zero [{Green Color []} {General General []}]} {Text [{Yellow Color []} {General General []}]}]"},
	{`[Red][<>50]General;[Blue]000`, "[{Positive [{Red Color []} {<>50 Condition [{{<> Operator []} } {{50 Operand []} }]} {General General []}]} {Negative [{Blue Color []} {000 ZeroPlaceHolder []}]}]"},
	{`[Red][=50]General;[Blue]000`, "[{Positive [{Red Color []} {=50 Condition [{{= Operator []} } {{50 Operand []} }]} {General General []}]} {Negative [{Blue Color []} {000 ZeroPlaceHolder []}]}]"},
	{`[SS]`, "[{Positive [{SS ElapsedDateTimes []}]}]"},
	{`[S]`, "[{Positive [{S ElapsedDateTimes []}]}]"},
	{`[TWN][DBNum1][$-404]y"年"m"月"d"日";@`, "[{Positive [{TWN Unknown []} {[DBNum1] SwitchArgument []} {[$-404] CurrencyLanguage [{{404 LanguageInfo []} }]} {y DateTimes []} {年 Literal []} {m DateTimes []} {月 Literal []} {d DateTimes []} {日 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[WHITE]0.0`, "[{Positive [{WHITE Color []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]}]"},
	{`[White]0.0`, "[{Positive [{White Color []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]}]"},
	{`[YELLOW]@`, "[{Text [{YELLOW Color []} {@ TextPlaceHolder []}]}]"},
	{`[Yellow]@`, "[{Text [{Yellow Color []} {@ TextPlaceHolder []}]}]"},
	{`[h]`, "[{Positive [{h ElapsedDateTimes []}]}]"},
	{`[h]:mm:ss`, "[{Positive [{h ElapsedDateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{`[h]:mm:ss;@`, "[{Positive [{h ElapsedDateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[h]\.mm" Uhr ";@`, "[{Positive [{h ElapsedDateTimes []} {. Literal []} {mm DateTimes []} { Uhr  Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`[hh]`, "[{Positive [{hh ElapsedDateTimes []}]}]"},
	{`[s]`, "[{Positive [{s ElapsedDateTimes []}]}]"},
	{`[ss]`, "[{Positive [{ss ElapsedDateTimes []}]}]"},
	{`\#\r\e\c`, "[{Positive [{# Literal []} {r Literal []} {e Literal []} {c Literal []}]}]"},
	{`\$#,##0_);[Red]"($"#,##0\)`, "[{Positive [{$ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{Red Color []} {($ Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]}]"},
	{`\$0.00`, "[{Positive [{$ Literal []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
	{`\C\O\B\ \o\n\ @`, "[{Text [{C Literal []} {O Literal []} {B Literal []} {  Literal []} {o Literal []} {n Literal []} {  Literal []} {@ TextPlaceHolder []}]}]"},
	{`\C\R\O\N\T\A\B\ \o\n\ @`, "[{Text [{C Literal []} {R Literal []} {O Literal []} {N Literal []} {T Literal []} {A Literal []} {B Literal []} {  Literal []} {o Literal []} {n Literal []} {  Literal []} {@ TextPlaceHolder []}]}]"},
	{`\R\e\s\u\l\t\ \o\n\ @`, "[{Text [{R Literal []} {e Literal []} {s Literal []} {u Literal []} {l Literal []} {t Literal []} {  Literal []} {o Literal []} {n Literal []} {  Literal []} {@ TextPlaceHolder []}]}]"},
	{`\S\Q\L\ \:\ @`, "[{Text [{S Literal []} {Q Literal []} {L Literal []} {  Literal []} {: Literal []} {  Literal []} {@ TextPlaceHolder []}]}]"},
	{`\S\Q\L\ \R\e\q\u\e\s\t\ \f\o\r\ @`, "[{Text [{S Literal []} {Q Literal []} {L Literal []} {  Literal []} {R Literal []} {e Literal []} {q Literal []} {u Literal []} {e Literal []} {s Literal []} {t Literal []} {  Literal []} {f Literal []} {o Literal []} {r Literal []} {  Literal []} {@ TextPlaceHolder []}]}]"},
	{`\c\c\c?????0"aaaa"0"bbbb"000000.00%`, "[{Positive [{c Literal []} {c Literal []} {c Literal []} {????? DigitalPlaceHolder []} {0 ZeroPlaceHolder []} {aaaa Literal []} {0 ZeroPlaceHolder []} {bbbb Literal []} {000000 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {% Percent []}]}]"},
	{`\u\n\t\i\l\ h:mm;@`, "[{Positive [{u Literal []} {n Literal []} {t Literal []} {i Literal []} {l Literal []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_ "￥"* #,##0.00_ "Positive";_ "￥"* \-#,##0.00_ ;_ "￥"* "-"??_ "Negative";_ @_ \ "Zero"`, "[{Positive [{￥ Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {Positive Literal []}]} {Negative [{￥ Literal []} {  RepeatsChar []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Zero [{￥ Literal []} {  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} {Negative Literal []}]} {Text [{@ TextPlaceHolder []} {  Literal []} {Zero Literal []}]}]"},
	{`_ * #,##0.00_)[$﷼-429]_ ;_ * \(#,##0.00\)[$﷼-429]_ ;_ * "-"??_)[$﷼-429]_ ;_ @_ `, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {[$﷼-429] CurrencyLanguage [{{﷼ CurrencyString []} } {{429 LanguageInfo []} }]}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []} {[$﷼-429] CurrencyLanguage [{{﷼ CurrencyString []} } {{429 LanguageInfo []} }]}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} {[$﷼-429] CurrencyLanguage [{{﷼ CurrencyString []} } {{429 LanguageInfo []} }]}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_ * #,##0_ ;_ * \-#,##0_ ;[Red]_ * "-"_ ;_ @_ `, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Zero [{Red Color []} {  RepeatsChar []} {- Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_("$"* #,##0.00_);_("$"* \(#,##0.00\);_("$"* "-"??_);_(@_)`, "[{Positive [{$ Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{$ Literal []} {  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]} {Zero [{$ Literal []} {  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_("$"* #,##0_);_("$"* \(#,##0\);_("$"* "-"??_);_(@_)`, "[{Positive [{$ Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{$ Literal []} {  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{$ Literal []} {  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_("$"* #,##0_);_("$"* \(#,##0\);_("$"* "-"_);_(@_)`, "[{Positive [{$ Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{$ Literal []} {  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{$ Literal []} {  RepeatsChar []} {- Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0.0000_);_(* \(#,##0.0000\);_(* "-"??_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0000 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0.000_);_(* \(#,##0.000\);_(* "-"??_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {000 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0.00_);_(* (#,##0.00);_(* "-"??_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0.00_);_(* \(#,##0.00\);_(* "-"??_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0.0_);_(* \(#,##0.0\);_(* "-"??_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0_);_(* \(#,##0\);_(* "-"??_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_(* #,##0_);_(* \(#,##0\);_(* "-"_);_(@_)`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{  RepeatsChar []} {- Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_([$ANG]\ * #,##0.0_);_([$ANG]\ * \(#,##0.0\);_([$ANG]\ * "-"?_);_(@_)`, "[{Positive [{[$ANG] CurrencyLanguage [{{ANG CurrencyString []} }]} {  Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]} {Negative [{[$ANG] CurrencyLanguage [{{ANG CurrencyString []} }]} {  Literal []} {  RepeatsChar []} {( Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {) Literal []}]} {Zero [{[$ANG] CurrencyLanguage [{{ANG CurrencyString []} }]} {  Literal []} {  RepeatsChar []} {- Literal []} {? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-"€"\ * #,##0.00_-;_-"€"\ * #,##0.00\-;_-"€"\ * "-"??_-;_-@_-`, "[{Positive [{€ Literal []} {  Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]} {Negative [{€ Literal []} {  Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {- Literal []}]} {Zero [{€ Literal []} {  Literal []} {  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0.00" TL"_-;\-* #,##0.00" TL"_-;_-* \-??" TL"_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} { TL Literal []}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} { TL Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} { TL Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0.00" €"_-;\-* #,##0.00" €"_-;_-* \-??" €"_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} { € Literal []}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} { € Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} { € Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0.00\ "р."_-;\-* #,##0.00\ "р."_-;_-* "-"??\ "р."_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {р. Literal []}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {р. Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} {  Literal []} {р. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0.00\ "€"_-;\-* #,##0.00\ "€"_-;_-* "-"??\ "€"_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {€ Literal []}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {€ Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} {  Literal []} {€ Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0.00\ [$€-407]_-;\-* #,##0.00\ [$€-407]_-;_-* \-??\ [$€-407]_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {[$€-407] CurrencyLanguage [{{€ CurrencyString []} } {{407 LanguageInfo []} }]}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []} {  Literal []} {[$€-407] CurrencyLanguage [{{€ CurrencyString []} } {{407 LanguageInfo []} }]}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} {  Literal []} {[$€-407] CurrencyLanguage [{{€ CurrencyString []} } {{407 LanguageInfo []} }]}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0.0\ _F_-;\-* #,##0.0\ _F_-;_-* "-"??\ _F_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {  Literal []}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {  Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []} {  Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0\ "€"_-;\-* #,##0\ "€"_-;_-* "-"\ "€"_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []} {€ Literal []}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {  Literal []} {€ Literal []}]} {Zero [{  RepeatsChar []} {- Literal []} {  Literal []} {€ Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-* #,##0_-;\-* #,##0_-;_-* "-"??_-;_-@_-`, "[{Positive [{  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Negative [{- Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []}]} {Zero [{  RepeatsChar []} {- Literal []} {?? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`_-\$* #,##0.0_ ;_-\$* \-#,##0.0\ ;_-\$* "-"?_ ;_-@_ `, "[{Positive [{$ Literal []} {  RepeatsChar []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]} {Negative [{$ Literal []} {  RepeatsChar []} {- Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {0 ZeroPlaceHolder []} {  Literal []}]} {Zero [{$ Literal []} {  RepeatsChar []} {- Literal []} {? DigitalPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`d`, "[{Positive [{d DateTimes []}]}]"},
	{`d-mmm`, "[{Positive [{d DateTimes []} {- Literal []} {mmm DateTimes []}]}]"},
	{`d-mmm-yy`, "[{Positive [{d DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{`d/m`, "[{Positive [{d DateTimes []} {/ Literal []} {m DateTimes []}]}]"},
	{`d/m/yy;@`, "[{Positive [{d DateTimes []} {/ Literal []} {m DateTimes []} {/ Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`d/m/yyyy;@`, "[{Positive [{d DateTimes []} {/ Literal []} {m DateTimes []} {/ Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`d/mm/yy;@`, "[{Positive [{d DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`d/mm/yyyy;@`, "[{Positive [{d DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`d\-mmm`, "[{Positive [{d DateTimes []} {- Literal []} {mmm DateTimes []}]}]"},
	{`d\-mmm\-yyyy`, "[{Positive [{d DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yyyy DateTimes []}]}]"},
	{`dd`, "[{Positive [{dd DateTimes []}]}]"},
	{`dd"-"mmm"-"yyyy`, "[{Positive [{dd DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yyyy DateTimes []}]}]"},
	{`dd/m/yyyy`, "[{Positive [{dd DateTimes []} {/ Literal []} {m DateTimes []} {/ Literal []} {yyyy DateTimes []}]}]"},
	{`dd/mm/yy`, "[{Positive [{dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yy DateTimes []}]}]"},
	{`dd/mm/yy;@`, "[{Positive [{dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`dd/mm/yy\ hh:mm`, "[{Positive [{dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yy DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{`dd/mm/yyyy`, "[{Positive [{dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []}]}]"},
	{`dd/mm/yyyy\ hh:mm:ss`, "[{Positive [{dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{`dd/mmm`, "[{Positive [{dd DateTimes []} {/ Literal []} {mmm DateTimes []}]}]"},
	{`dd\-mm\-yy`, "[{Positive [{dd DateTimes []} {- Literal []} {mm DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{`dd\-mmm\-yy`, "[{Positive [{dd DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{`dd\-mmm\-yyyy\ hh:mm:ss.000`, "[{Positive [{dd DateTimes []} {- Literal []} {mmm DateTimes []} {- Literal []} {yyyy DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]}]"},
	{`dd\/mm\/yy`, "[{Positive [{dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yy DateTimes []}]}]"},
	{`dd\/mm\/yyyy`, "[{Positive [{dd DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {yyyy DateTimes []}]}]"},
	{`ddd`, "[{Positive [{ddd DateTimes []}]}]"},
	{`dddd`, "[{Positive [{dddd DateTimes []}]}]"},
	{`dddd, mmmm dd, yyyy`, "[{Positive [{dddd DateTimes []} {, Literal []} {  Literal []} {mmmm DateTimes []} {  Literal []} {dd DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]}]"},
	{`h`, "[{Positive [{h DateTimes []}]}]"},
	{`h"时"mm"分"ss"秒";@`, "[{Positive [{h DateTimes []} {时 Literal []} {mm DateTimes []} {分 Literal []} {ss DateTimes []} {秒 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`h"時"mm"分"ss"秒";@`, "[{Positive [{h DateTimes []} {時 Literal []} {mm DateTimes []} {分 Literal []} {ss DateTimes []} {秒 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`h:mm`, "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{`h:mm AM/PM`, "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`h:mm:ss`, "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{`h:mm:ss AM/PM`, "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`h:mm:ss;@`, "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`h:mm;@`, "[{Positive [{h DateTimes []} {: Literal []} {mm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`h\.mm" Uhr ";@`, "[{Positive [{h DateTimes []} {. Literal []} {mm DateTimes []} { Uhr  Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`h\.mm" h";@`, "[{Positive [{h DateTimes []} {. Literal []} {mm DateTimes []} { h Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`h\.mm" u.";@`, "[{Positive [{h DateTimes []} {. Literal []} {mm DateTimes []} { u. Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`hh":"mm AM/PM`, "[{Positive [{hh DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`hh:mm:ss`, "[{Positive [{hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{`hh:mm:ss\ AM/PM`, "[{Positive [{hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`hh\.mm" h";@`, "[{Positive [{hh DateTimes []} {. Literal []} {mm DateTimes []} { h Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`hhm`, "[{Positive [{hh DateTimes []} {m DateTimes []}]}]"},
	{`hhmm`, "[{Positive [{hh DateTimes []} {mm DateTimes []}]}]"},
	{`m"月"d"日"`, "[{Positive [{m DateTimes []} {月 Literal []} {d DateTimes []} {日 Literal []}]}]"},
	{`m/d/yy`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yy DateTimes []}]}]"},
	{`m/d/yy h:mm`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{`m/d/yy;@`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`m/d/yy\ h:mm`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{`m/d/yy\ h:mm;@`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`m/d/yyyy`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yyyy DateTimes []}]}]"},
	{`m/d/yyyy;@`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yyyy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`m/d/yyyy\ h:mm:ss;@`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yyyy DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`m/d;@`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`m\/d\/yyyy`, "[{Positive [{m DateTimes []} {/ Literal []} {d DateTimes []} {/ Literal []} {yyyy DateTimes []}]}]"},
	{`mm/dd`, "[{Positive [{mm DateTimes []} {/ Literal []} {dd DateTimes []}]}]"},
	{`mm/dd/yy`, "[{Positive [{mm DateTimes []} {/ Literal []} {dd DateTimes []} {/ Literal []} {yy DateTimes []}]}]"},
	{`mm/dd/yy;@`, "[{Positive [{mm DateTimes []} {/ Literal []} {dd DateTimes []} {/ Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`mm/dd/yyyy`, "[{Positive [{mm DateTimes []} {/ Literal []} {dd DateTimes []} {/ Literal []} {yyyy DateTimes []}]}]"},
	{`mm:ss`, "[{Positive [{mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{`mm:ss.0;@`, "[{Positive [{mm DateTimes []} {: Literal []} {ss DateTimes []} {. DecimalPoint []} {0 ZeroPlaceHolder []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`mmm d, yyyy`, "[{Positive [{mmm DateTimes []} {  Literal []} {d DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]}]"},
	{`mmm" "d", "yyyy`, "[{Positive [{mmm DateTimes []} {  Literal []} {d DateTimes []} {,  Literal []} {yyyy DateTimes []}]}]"},
	{`mmm-yy`, "[{Positive [{mmm DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{`mmm-yy;@`, "[{Positive [{mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`mmm/yy`, "[{Positive [{mmm DateTimes []} {/ Literal []} {yy DateTimes []}]}]"},
	{`mmm\-yy`, "[{Positive [{mmm DateTimes []} {- Literal []} {yy DateTimes []}]}]"},
	{`mmm\-yy;@`, "[{Positive [{mmm DateTimes []} {- Literal []} {yy DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`mmm\-yyyy`, "[{Positive [{mmm DateTimes []} {- Literal []} {yyyy DateTimes []}]}]"},
	{`mmmm\ d\,\ yyyy`, "[{Positive [{mmmm DateTimes []} {  Literal []} {d DateTimes []} {, Literal []} {  Literal []} {yyyy DateTimes []}]}]"},
	{`mmmm\ yyyy`, "[{Positive [{mmmm DateTimes []} {  Literal []} {yyyy DateTimes []}]}]"},
	{`s`, "[{Positive [{s DateTimes []}]}]"},
	{`ss`, "[{Positive [{ss DateTimes []}]}]"},
	{`yy`, "[{Positive [{yy DateTimes []}]}]"},
	{`yy/mm/dd`, "[{Positive [{yy DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {dd DateTimes []}]}]"},
	{`yy\.mm\.dd`, "[{Positive [{yy DateTimes []} {. Literal []} {mm DateTimes []} {. Literal []} {dd DateTimes []}]}]"},
	{`yym`, "[{Positive [{yy DateTimes []} {m DateTimes []}]}]"},
	{`yymm`, "[{Positive [{yy DateTimes []} {mm DateTimes []}]}]"},
	{`yymmm`, "[{Positive [{yy DateTimes []} {mmm DateTimes []}]}]"},
	{`yymmmm`, "[{Positive [{yy DateTimes []} {mmmm DateTimes []}]}]"},
	{`yymmmmm`, "[{Positive [{yy DateTimes []} {mmmmm DateTimes []}]}]"},
	{`yyyy`, "[{Positive [{yyyy DateTimes []}]}]"},
	{`yyyy"년"\ m"월"\ d"일";@`, "[{Positive [{yyyy DateTimes []} {년 Literal []} {  Literal []} {m DateTimes []} {월 Literal []} {  Literal []} {d DateTimes []} {일 Literal []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`yyyy-m-d h:mm AM/PM`, "[{Positive [{yyyy DateTimes []} {- Literal []} {m DateTimes []} {- Literal []} {d DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []} {  Literal []} {AM/PM DateTimes []}]}]"},
	{`yyyy-mm-dd`, "[{Positive [{yyyy DateTimes []} {- Literal []} {mm DateTimes []} {- Literal []} {dd DateTimes []}]}]"},
	{`yyyy/mm/dd`, "[{Positive [{yyyy DateTimes []} {/ Literal []} {mm DateTimes []} {/ Literal []} {dd DateTimes []}]}]"},
	{`yyyy\-m\-d\ hh:mm:ss`, "[{Positive [{yyyy DateTimes []} {- Literal []} {m DateTimes []} {- Literal []} {d DateTimes []} {  Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []} {: Literal []} {ss DateTimes []}]}]"},
	{`yyyy\-mm\-dd`, "[{Positive [{yyyy DateTimes []} {- Literal []} {mm DateTimes []} {- Literal []} {dd DateTimes []}]}]"},
	{`yyyy\-mm\-dd;@`, "[{Positive [{yyyy DateTimes []} {- Literal []} {mm DateTimes []} {- Literal []} {dd DateTimes []}]} {Text [{@ TextPlaceHolder []}]}]"},
	{`yyyy\-mm\-dd\ h:mm`, "[{Positive [{yyyy DateTimes []} {- Literal []} {mm DateTimes []} {- Literal []} {dd DateTimes []} {  Literal []} {h DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{`yyyy\-mm\-dd\Thh:mm`, "[{Positive [{yyyy DateTimes []} {- Literal []} {mm DateTimes []} {- Literal []} {dd DateTimes []} {T Literal []} {hh DateTimes []} {: Literal []} {mm DateTimes []}]}]"},
	{`yyyy\-mm\-dd\Thhmmss.000`, "[{Positive [{yyyy DateTimes []} {- Literal []} {mm DateTimes []} {- Literal []} {dd DateTimes []} {T Literal []} {hh DateTimes []} {mm DateTimes []} {ss DateTimes []} {. DecimalPoint []} {000 ZeroPlaceHolder []}]}]"},
	{`[DBNum1][$-zh-CN]h"时"mm"分";"====="@@@"--"@"----"`, "[{Positive [{[DBNum1] SwitchArgument []} {[$-zh-CN] CurrencyLanguage [{{zh-CN LanguageInfo []} }]} {h DateTimes []} {时 Literal []} {mm DateTimes []} {分 Literal []}]} {Text [{===== Literal []} {@ TextPlaceHolder []} {@ TextPlaceHolder []} {@ TextPlaceHolder []} {-- Literal []} {@ TextPlaceHolder []} {---- Literal []}]}]"},
	{"[$ZWR]\\ #,##0.00", "[{Positive [{[$ZWR] CurrencyLanguage [{{ZWR CurrencyString []} }]} {  Literal []} {# HashPlaceHolder []} {, ThousandsSeparator []} {## HashPlaceHolder []} {0 ZeroPlaceHolder []} {. DecimalPoint []} {00 ZeroPlaceHolder []}]}]"},
}

func TestNFP(t *testing.T) {
	for _, item := range testStrings {
		p := NumberFormatParser()
		result := fmt.Sprint(p.Parse(item[0]))
		if result != item[1] {
			t.Errorf("not equal:\nexpected:%+v\nactual  :%+v\nmessages:%s", item[1], result, item[0])
		}
		p.PrettyPrint()
	}
}

// BenchmarkNumberFormatParser benchmarks the parser with all test strings.
func BenchmarkNumberFormatParser(b *testing.B) {
	for i := 0; i < b.N; i++ {
		for _, item := range testStrings {
			p := NumberFormatParser()
			p.Parse(item[0])
		}
	}
}
