package azure

import (
	"github.com/infracost/infracost/internal/resources"
	"github.com/infracost/infracost/internal/schema"
	"github.com/infracost/infracost/internal/usage"

	"github.com/shopspring/decimal"
)

type LinuxVirtualMachineScaleSet struct {
	Address         string
	SKU             string
	UltraSSDEnabled bool
	Region          string
	OSDiskData      *ManagedDiskData
	Instances       *int64       `infracost_usage:"instances"`
	OSDisk          *OSDiskUsage `infracost_usage:"os_disk"`
}

func (r *LinuxVirtualMachineScaleSet) CoreType() string {
	return "LinuxVirtualMachineScaleSet"
}

func (r *LinuxVirtualMachineScaleSet) UsageSchema() []*schema.UsageItem {
	return []*schema.UsageItem{
		{Key: "instances", ValueType: schema.Int64, DefaultValue: 0},
		{
			Key:          "os_disk",
			ValueType:    schema.SubResourceUsage,
			DefaultValue: &usage.ResourceUsage{Name: "os_disk", Items: OSDiskUsageSchema},
		},
	}
}

func (r *LinuxVirtualMachineScaleSet) PopulateUsage(u *schema.UsageData) {
	resources.PopulateArgsWithUsage(r, u)
}

func (r *LinuxVirtualMachineScaleSet) BuildResource() *schema.Resource {

	instanceType := r.SKU

	costComponents := []*schema.CostComponent{linuxVirtualMachineCostComponent(r.Region, instanceType, nil)}
	subResources := make([]*schema.Resource, 0)

	if r.UltraSSDEnabled {
		costComponents = append(costComponents, ultraSSDReservationCostComponent(r.Region))
	}

	var monthlyDiskOperations *decimal.Decimal
	if r.OSDisk != nil && r.OSDisk.MonthlyDiskOperations != nil {
		monthlyDiskOperations = decimalPtr(decimal.NewFromInt(*r.OSDisk.MonthlyDiskOperations))
	}

	if r.OSDiskData != nil {
		osDisk := osDiskSubResource(r.Region, r.OSDiskData.DiskType, r.OSDiskData.DiskSizeGB, r.OSDiskData.DiskIOPSReadWrite, r.OSDiskData.DiskMBPSReadWrite, monthlyDiskOperations)
		subResources = append(subResources, osDisk)
	}

	instanceCount := decimal.NewFromInt(*r.Instances)

	res := &schema.Resource{
		Name:           r.Address,
		CostComponents: costComponents,
		SubResources:   subResources,
		UsageSchema:    r.UsageSchema(),
	}

	schema.MultiplyQuantities(res, instanceCount)

	return res
}
