-- This file is part of the Grid LSC User Environment (GLUE)
-- 
-- GLUE is free software: you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation, either version 3 of the License, or (at your option) any later
-- version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
-- FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
-- details.
-- 
-- You should have received a copy of the GNU General Public License along
-- with this program.  If not, see <http://www.gnu.org/licenses/>.

CREATE TABLE process
(
-- This table contains information about a specific invocation of a program.

-- Database which created this entry
      creator_db         INTEGER NOT NULL WITH DEFAULT 1,

-- BASIC INFORMATION ABOUT THE PROGRAM
-- Program name
      program            VARCHAR(64) NOT NULL,
-- Version of the program
      version            VARCHAR(64),
-- Where the program is stored in the cvs repository
      cvs_repository     VARCHAR(256),
-- Time when the program was entered into the cvs repository (GPS seconds)
      cvs_entry_time     INTEGER,
-- User comment which describes the program
      comment            VARCHAR(255),

-- INFORMATION ABOUT THIS INVOCATION OF THE PROGRAM
-- Flag to indicate whether it was run on-line (1) or off-line (0)
      is_online          INTEGER NOT NULL WITH DEFAULT 0,
-- Node on which it was run
      node               VARCHAR(64) NOT NULL,
-- Unix username
      username           CHAR(16) NOT NULL,
-- Unix process ID
      unix_procid        INTEGER NOT NULL,
-- Start time (GPS seconds)
      start_time         INTEGER NOT NULL,
-- End time (GPS seconds); not filled initially, but filled when process
-- exits gracefully
      end_time           INTEGER,
-- LDAS job ID (numeric part only)
      jobid              INTEGER NOT NULL WITH DEFAULT 0,
-- LDAS site name (e.g. 'LHO', 'LLO', 'CIT', 'Dev', 'Test')
      domain	         VARCHAR(64) NOT NULL WITH DEFAULT '',

-- Unique identifier to this program, generated by DB2 (not unix process ID)
      process_id         CHAR(13) FOR BIT DATA NOT NULL,
-- Parameter set identifier.  Permits an association between multiple
-- invocations of a program which use the same set of input parameters.
-- Probably not filled initially (because of timing issues if multiple
-- processes start at about the same time), but updated later.
      param_set          INTEGER,

-- INFORMATION ABOUT THE DATA HANDLED BY THIS PROGRAM
-- Interferometer(s) from which data comes.  In general, a process
-- knows in advance what interferometer's data it is analyzing.
-- (This is necessary to retrieve frameset metadata, for instance.)
-- Make this variable long enough to indicate multiple interferometers.
-- (e.g. "H1H2L1G1V1T1")
      ifos               CHAR(12),
      
-- Insertion time (automatically assigned by the database)
      insertion_time     TIMESTAMP WITH DEFAULT CURRENT TIMESTAMP,

-- We constrain only on enforcing unique process_id and creator_db.
-- If two processes have the same process metadata, it is up to the
-- user to resolve this. Under condor we cannot enforce uniqueness
-- of start_time or node (since two processes may start on the same
-- dual cpu machine at the same time), unix_procid (since the value
-- returned by getpid() is invalid in the condor standard universe:
-- a checkpointed job may have multiple pids) and jobid (since the 
-- condor equaivalent of cluster.process is not guaranteed to be 
-- unique. If the condor history is deleted, it will reset to zero)
      CONSTRAINT process_pk
      PRIMARY KEY (process_id, creator_db)
)
-- The following line is needed for this table to be replicated to other sites
DATA CAPTURE CHANGES
;
-- Create an index based on program name
CREATE INDEX process_ind_prog ON process(program)
;
-- Create an index based on user name
CREATE INDEX process_ind_time ON process(username)
;
-- Create an index based on ifos
CREATE INDEX process_ind_ifos ON process(ifos)
;
