#!/bin/sh

# this script expects:
#
# lalapps build to be tested is installed is in ./opt/lscsoft
#
# $NMI_ligo_reference_xml defined (by Metronome, from our submit file)
# $NMI_component_version (git id) defined (by Metronome, from our submit file)
# $NMI_ligo_add_params defined (by Metronome, from our submit file)

# wrap this whole script in a block combining stdout & stderr
{

# exit immediately if any command exits with a non-zero status.
set -e
# treat unset variables as an error when performing parameter expansion.
set -u
# print (unexpanded) shell input lines as they are read
set -v
# print (expanded) commands before they're executed
set -x

# save our pwd
BASE_DIR=$(pwd)

# in the provided lalsuite install dir, rebase any now-incorrect
# static paths that were set at build time in *-user-env.sh, etc.
# NOTE: this has a bug if there is any whitespace in the CWD
find $BASE_DIR/opt \( -name \*-user-env.sh -or -name lscsoftrc \) -print0 | \
    xargs -0 \
    perl -p -i.orig -e 's!/\S+/execute/dir_\d+/userdir/!'${BASE_DIR}'/!g'

# although lalapps may be statically compiled, AFAICT many tools in
# lal, glue, etc. cannot be, and thus have runtime dependencies
# requiring env setup (boo)
set +u
. $BASE_DIR/opt/lscsoft/etc/lscsoftrc
set -u

# why isn't this done in lscsoftrc or *-user-env.sh?
for dir in $BASE_DIR/opt/lscsoft/*/lib; do
  export LD_LIBRARY_PATH=$dir:$LD_LIBRARY_PATH
  export DYLD_LIBRARY_PATH=$dir:$DYLD_LIBRARY_PATH
done
for dir in $BASE_DIR/opt/lscsoft/*/lib64/python*/site-packages; do
  export PYTHONPATH=$dir:$PYTHONPATH
done

# useful info to have logged for later debugging
$BASE_DIR/opt/lscsoft/lal/bin/lal-version

# create and work in a subdir named after the git id we're testing so
# that later diff tests can keep straight which results which
mkdir $NMI_component_version
cd $NMI_component_version

# create cwd symlinks to all our input files (as expected by lalapps);
# we assume they're in the same dir as the provided XML reference file
datadir=$(dirname $NMI_ligo_reference_xml)
ln -sv $datadir/cache
# *.tmp are generated by pre-execute.sh
for filename in $(< $BASE_DIR/input_files.tmp) $(< $BASE_DIR/symlinks.tmp); do

    # If necessary, add new columns (.e.g, spin1x) required by newer
    # lalsuite versions.  I don't know exactly when (at which git id)
    # these first became required -- but the existence of the
    # pylal_add_missing_sngl_inspiral_columns script from pylal is a
    # pretty close proxy for when it happened, so for now we just test
    # for that, and if it's there, we assume we need it.

    if [[ $filename == *.xml*  ]] && which pylal_add_missing_sngl_inspiral_columns; then
	pylal_add_missing_sngl_inspiral_columns -i $datadir/$filename -o $filename
    else
	ln -sv $datadir/$filename
    fi
done


# if we're trying to run a post-S6 version of lalapps_coire without
# the --glob option, we need to remove it from params and move any
# associated filename to the end of the argument list
if [[ $NMI_ligo_exe == lalapps_coire ]] && ! lalapps_coire --help 2>&1 | fgrep -qi glob; then
    mv $BASE_DIR/params.tmp $BASE_DIR/params-orig.tmp
    cat $BASE_DIR/params-orig.tmp | perl -pe 's/(.*)--glob(\s+\S+)\s+(.*)/$1$3$2/g' > $BASE_DIR/params.tmp
fi

# if we're trying to run a post-S6 version of lalapps_tmpltbank with
# the new candle mass arguments, we need to convert the old arguments
#
# I'm not sure what the general conversion should be, but I do know
# that --candle-mass1 1.000000e+00 --candle-mass2 1.000000e+00 should
# be --standard-candle --candle-minmass 1 --candle-maxmass 12, so we
# special-case that since it corresponds to the s6_20090722 arguments
# we care about most right now.
#
# TODO: we need a more general fix for this

if [[ $NMI_ligo_exe == lalapps_tmpltbank ]] && (lalapps_tmpltbank --help 2>&1 | fgrep -qi candle-minmass); then
    mv $BASE_DIR/params.tmp $BASE_DIR/params-orig.tmp
    cat $BASE_DIR/params-orig.tmp | perl -pe 's/(.*\s+)--candle-mass1 1.000000e\+00 --candle-mass2 1.000000e\+00(\s+.*)/$1 --candle-minmass 1 --candle-maxmass 12 $2/' > $BASE_DIR/params.tmp
fi

# do the deed
which $NMI_ligo_exe
time $(which $NMI_ligo_exe) $(< $BASE_DIR/params.tmp) $NMI_ligo_add_params $(< $BASE_DIR/input_files.tmp)

# end of stdout/stderr-combining block
} 2>&1
