module mrtindex_sec_calibration
  use gbl_message
  use gkernel_interfaces
  !---------------------------------------------------------------------
  ! Support module for calibration section in MRTCAL index
  !---------------------------------------------------------------------

  public :: sec_calib_id
  public :: sec_calib_t
  public :: sec_calib_array_t
  private
  !
  ! Section Calibration: reserved for calibration scans
  integer(kind=4), parameter :: sec_calib_id=2
! integer(kind=8), parameter :: sec_calib_len=unlimited (dynamic read/write)
  type sec_calib_t
    ! Save the calibration arrays for the calibration scan
    ! ZZZ Need a better definition
    integer(kind=4)               :: nfreq        ! [-----] Number of frequencies
    integer(kind=4)               :: nset         ! [-----] Number of setups
    integer(kind=4)               :: npix         ! [-----] Number of pixels
    character(len=8), allocatable :: frontend(:)  ! [-----] Receiver names[Nset]
    real(kind=8),     allocatable :: freq(:,:,:)  ! [MHz  ] Frequencies[Nfreq,Nset,Npix]
    real(kind=4),     allocatable :: atsys(:,:,:) ! [K    ] Attenuated Tsys[Nfreq,Nset,Npix]
    real(kind=4),     allocatable :: ztau(:,:,:)  ! [neper] Zenith tau[Nfreq,Nset,Npix]
  contains
    procedure, public :: variable => calib_variable
  end type sec_calib_t
  !
  type sec_calib_array_t
    integer(kind=4)              :: nent            ! Used size of 'ent' dimension (allocation may be larger)
    integer(kind=4)              :: mfreq           ! Strict size of allocation on 'freq' dimension
    integer(kind=4)              :: mset            ! Strict size of allocation on 'set' dimension
    integer(kind=4)              :: mpix            ! Strict size of allocation on 'pix' dimension
    integer(kind=4), allocatable :: nfreq(:)        ! [nent]
    integer(kind=4), allocatable :: nset(:)         ! [nent]
    integer(kind=4), allocatable :: npix(:)         ! [nent]
    !
    character(len=8), allocatable :: frontend(:,:)  ! [mset,nent]
    real(kind=8),     allocatable :: freq(:,:,:,:)  ! [mfreq,mset,mpix,nent]
    real(kind=4),     allocatable :: atsys(:,:,:,:) ! [mfreq,mset,mpix,nent]
    real(kind=4),     allocatable :: ztau(:,:,:,:)  ! [mfreq,mset,mpix,nent]
  contains
    procedure, public :: reallocate => calib_array_reallocate
    procedure, public :: set        => calib_array_set
    procedure, public :: variable   => calib_array_variable
  end type sec_calib_array_t
  !
contains
  !
  subroutine calib_variable(cal,struct,ro,error)
    !---------------------------------------------------------------------
    !
    !---------------------------------------------------------------------
    class(sec_calib_t), intent(in)    :: cal     !
    character(len=*),   intent(in)    :: struct  ! Structure name
    logical,            intent(in)    :: ro      ! Read-Only?
    logical,            intent(inout) :: error   ! Logical error flag
    !
    logical :: userreq
    character(len=32) :: str
    integer(kind=index_length) :: dims(sic_maxdims)
    !
    userreq = .false.
    str = trim(struct)//'%CAL'
    !
    call sic_delvariable(str,userreq,error)
    call sic_defstructure(str,.true.,error)
    if (error)  return
    !
    call sic_def_inte(trim(str)//'%NFREQ',cal%nfreq,0,0,ro,error)
    call sic_def_inte(trim(str)//'%NSET', cal%nset, 0,0,ro,error)
    call sic_def_inte(trim(str)//'%NPIX', cal%npix, 0,0,ro,error)
    if (cal%nfreq.le.0)  return
    !
    dims(1) = cal%nset
    call sic_def_charn(trim(str)//'%FRONTEND',cal%frontend,1,dims,ro,error)
    !
    dims(1) = cal%nfreq
    dims(2) = cal%nset
    dims(3) = cal%npix
    call sic_def_dble(trim(str)//'%FREQ',cal%freq,3,dims,ro,error)
    call sic_def_real(trim(str)//'%TSYS',cal%atsys,3,dims,ro,error)
    call sic_def_real(trim(str)//'%ZTAU',cal%ztau,3,dims,ro,error)
  end subroutine calib_variable
  !
  subroutine calib_array_reallocate(cal,mfreq,mset,mpix,nent,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(sec_calib_array_t), intent(inout) :: cal
    integer(kind=4),          intent(in)    :: mfreq
    integer(kind=4),          intent(in)    :: mset
    integer(kind=4),          intent(in)    :: mpix
    integer(kind=8),          intent(in)    :: nent
    logical,                  intent(inout) :: error
    !
    character(len=*), parameter :: rname='VARIABLE'
    logical :: realloc
    integer(kind=4) :: ier
    !
    cal%mfreq = mfreq
    cal%mset = mset
    cal%mpix = mpix
    cal%nent = nent
    !
    if (allocated(cal%freq)) then
      ! Strict equality on mfreq, mset, mpix dimensions, otherwise this means
      ! troubles when mapping SIC variables.
      ! Lower than is fine for trailing dimension
      realloc = ubound(cal%freq,1).ne.cal%mfreq .or.  &
                ubound(cal%freq,2).ne.cal%mset  .or.  &
                ubound(cal%freq,3).ne.cal%mpix  .or.  &
                ubound(cal%freq,4).lt.cal%nent
      if (cal%nent.le.0 .or. realloc) then
        deallocate(cal%nfreq,cal%nset,cal%npix)
        !
        deallocate(cal%frontend)
        deallocate(cal%freq,cal%atsys,cal%ztau)
      endif
    else
      realloc = cal%nent.gt.0
    endif
    !
    if (realloc) then
      allocate(cal%nfreq(cal%nent),cal%nset(cal%nent),cal%npix(cal%nent),stat=ier)
      !
      allocate(cal%frontend(cal%mset,cal%nent),stat=ier)
      allocate(cal%freq(cal%mfreq,cal%mset,cal%mpix,cal%nent),stat=ier)
      allocate(cal%atsys(cal%mfreq,cal%mset,cal%mpix,cal%nent),stat=ier)
      allocate(cal%ztau(cal%mfreq,cal%mset,cal%mpix,cal%nent),stat=ier)
      if (failed_allocate(rname,'MDX%HEAD%CAL arrays',ier,error))  return
    endif
  end subroutine calib_array_reallocate
  !
  subroutine calib_array_set(varcal,ient,sec,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(sec_calib_array_t), intent(inout) :: varcal
    integer(kind=8),          intent(in)    :: ient
    type(sec_calib_t),        intent(in)    :: sec
    logical,                  intent(inout) :: error
    !
    integer(kind=4) :: ifreq,iset,ipix
    !
    varcal%nfreq(ient) = sec%nfreq
    varcal%nset(ient) = sec%nset
    varcal%npix(ient) = sec%npix
    !
    ! Nullify the unused parts => play simple
    varcal%frontend(:,ient) = ':'
    varcal%freq(:,:,:,ient)  = 0.d0
    varcal%atsys(:,:,:,ient) = 0.
    varcal%ztau(:,:,:,ient)  = 0.
    !
    ! Fill the array
    do iset=1,sec%nset
      varcal%frontend(iset,ient) = sec%frontend(iset)
    enddo
    do ipix=1,sec%npix
      do iset=1,sec%nset
        do ifreq=1,sec%nfreq
          varcal%freq(ifreq,iset,ipix,ient)  = sec%freq(ifreq,iset,ipix)
          varcal%atsys(ifreq,iset,ipix,ient) = sec%atsys(ifreq,iset,ipix)
          varcal%ztau(ifreq,iset,ipix,ient)  = sec%ztau(ifreq,iset,ipix)
        enddo
      enddo
    enddo
  end subroutine calib_array_set
  !
  subroutine calib_array_variable(cal,str,error)
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(sec_calib_array_t), intent(in)    :: cal
    character(len=*),         intent(in)    :: str
    logical,                  intent(inout) :: error
    !
    logical, parameter :: ro=.true.
    character(len=32) :: topstr
    integer(kind=index_length) :: dims(sic_maxdims)
    !
    topstr = trim(str)//'%CAL'
    call sic_defstructure(topstr,.true.,error)
    if (error)  return
    dims(1) = cal%nent
    call sic_def_inte(trim(topstr)//'%NFREQ',cal%nfreq,1,dims,ro,error)
    call sic_def_inte(trim(topstr)//'%NSET', cal%nset, 1,dims,ro,error)
    call sic_def_inte(trim(topstr)//'%NPIX', cal%npix, 1,dims,ro,error)
    if (error)  return
    !
    dims(1) = cal%mset
    dims(2) = cal%nent
    call sic_def_charn(trim(topstr)//'%FRONTEND',cal%frontend,2,dims,ro,error)
    if (error)  return
    !
    dims(1) = cal%mfreq
    dims(2) = cal%mset
    dims(3) = cal%mpix
    dims(4) = cal%nent
    call sic_def_dble(trim(topstr)//'%FREQ',cal%freq,4,dims,ro,error)
    call sic_def_real(trim(topstr)//'%TSYS',cal%atsys,4,dims,ro,error)
    call sic_def_real(trim(topstr)//'%ZTAU',cal%ztau,4,dims,ro,error)
    if (error)  return
  end subroutine calib_array_variable
end module mrtindex_sec_calibration
