/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001-2004 National Institute of Water and Atmospheric
 * Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include "isocube.h"

#define DIAGONAL 0.866025403785
#define SLIGHTLY_LARGER 1.001

static gboolean plane_dont_cut_sphere (GfsGl2D * gl, FttVector * p, gdouble r)
{
  return (fabs ((p->x - gl->p[0].x)*gl->n.x + 
		(p->y - gl->p[0].y)*gl->n.y + 
		(p->z - gl->p[0].z)*gl->n.z)
	  > r);
}

static void cell_traverse_visible_plane_no_check (FttCell * root,
						  GfsFrustum * f,
						  GfsGl * gl,
						  FttCellTraverseFunc func,
						  gpointer data)
{
  gdouble r = ftt_cell_size (root)*SLIGHTLY_LARGER*DIAGONAL;
  FttVector p;

  ftt_cell_pos (root, &p);
  if (plane_dont_cut_sphere (GFS_GL2D (gl), &p, r))
    return;
  if (FTT_CELL_IS_LEAF (root) || ftt_cell_level (root) == gl->maxlevel)
    (* func) (root, data);
  else {
    gdouble r = ftt_cell_size (root)*SLIGHTLY_LARGER*DIAGONAL;
    FttVector p;
    
    ftt_cell_pos (root, &p);
    if (gfs_sphere_size (&p, r, f) < f->res)
      (* func) (root, data);
    else {
      struct _FttOct * children = root->children;
      guint n;
      
      for (n = 0; n < FTT_CELLS; n++) {
	FttCell * c = &(children->cell[n]);
	if (!FTT_CELL_IS_DESTROYED (c))
	  cell_traverse_visible_plane_no_check (c, f, gl, func, data);
      }
    }
  }
}

static void cell_traverse_visible_plane (FttCell * root,
					 GfsFrustum * f,
					 GfsGl * gl,
					 FttCellTraverseFunc func,
					 gpointer data)
{
  gdouble r = ftt_cell_size (root)*SLIGHTLY_LARGER*DIAGONAL;
  FttVector p;
  GtsIntersect i;

  ftt_cell_pos (root, &p);
  if (plane_dont_cut_sphere (GFS_GL2D (gl), &p, r))
    return;
  i = gfs_sphere_in_frustum (&p, r, f);
  if (i == GTS_OUT)
    return;
  if (FTT_CELL_IS_LEAF (root) ||
      ftt_cell_level (root) == gl->maxlevel || 
      gfs_sphere_size (&p, r, f) < f->res)
    (* func) (root, data);
  else if (i == GTS_IN)
    cell_traverse_visible_plane_no_check (root, f, gl, func, data);
  else {
    struct _FttOct * children = root->children;
    guint n;

    for (n = 0; n < FTT_CELLS; n++) {
      FttCell * c = &(children->cell[n]);
      if (!FTT_CELL_IS_DESTROYED (c))
	cell_traverse_visible_plane (c, f, gl, func, data);
    }
  }
}

static void box_traverse_visible_plane (GfsBox * b, gpointer * data)
{
  cell_traverse_visible_plane (b->root, data[0], data[3], data[1], data[2]);
}

/**
 * gfs_gl_cell_traverse_visible_plane:
 * @gl: a #GfsGl2D.
 * @f: a view frustum.
 * @func: a used-defined function.
 * @data: user data to pass to @func.
 *
 * Traverse the cells of @gl which are visible and whose bounding
 * sphere is intersected by the plane defined by @gl.
 */
static void gfs_gl_cell_traverse_visible_plane (GfsGl * gl,
						GfsFrustum * f,
						FttCellTraverseFunc func,
						gpointer data)
{
  gpointer dat[4];

  g_return_if_fail (gl != NULL);
  g_return_if_fail (f != NULL);
  g_return_if_fail (func != NULL);

  dat[0] = f;
  dat[1] = func;
  dat[2] = data;
  dat[3] = gl;
  gts_container_foreach (GTS_CONTAINER (gl->sim), (GtsFunc) box_traverse_visible_plane, dat);
}

/* GfsGlCells: Object */

static gdouble point_orientation (FttVector p[3], FttVector * c)
{
  gdouble adx, bdx, cdx;
  gdouble ady, bdy, cdy;
  gdouble adz, bdz, cdz;
  
  adx = p[0].x - c->x;
  bdx = p[1].x - c->x;
  cdx = p[2].x - c->x;
  ady = p[0].y - c->y;
  bdy = p[1].y - c->y;
  cdy = p[2].y - c->y;
  adz = p[0].z - c->z;
  bdz = p[1].z - c->z;
  cdz = p[2].z - c->z;
  
  return (adx * (bdy * cdz - bdz * cdy) +
	  bdx * (cdy * adz - cdz * ady) +
	  cdx * (ady * bdz - adz * bdy));
}

static gboolean plane_cuts_cell (FttVector plane[3], FttCell * cell)
{
  FttVector o;
  gdouble h = ftt_cell_size (cell)*SLIGHTLY_LARGER;
  guint i;

  ftt_cell_pos (cell, &o);
  o.x -= h/2.; o.y -= h/2.; o.z -= h/2.;
  for (i = 0; i < 12; i++) {
    FttVector e, d;
    gdouble a, b;
    d.x = o.x + h*edge[i][0].x; d.y = o.y + h*edge[i][0].y; d.z = o.z + h*edge[i][0].z;
    e.x = o.x + h*edge[i][1].x; e.y = o.y + h*edge[i][1].y; e.z = o.z + h*edge[i][1].z;
    a = point_orientation (plane, &e);
    b = point_orientation (plane, &d);
    if ((a <= 0. && b > 0.) || (a >= 0. && b < 0.))
      return TRUE;
  }
  return FALSE;
}

static void cube_plane_intersection (FttCell * cell,
				     FttVector plane[3],
				     FttVector p[12],
				     gint orient[12],
				     GfsVariable * var,
				     gdouble v[12],
				     gint max_level)
{
  FttVector o;
  gdouble h = ftt_cell_size (cell)*SLIGHTLY_LARGER, vc[8];
  guint i;

  if (var)
    for (i = 0; i < 8; i++)
      vc[i] = G_MAXDOUBLE;

  ftt_cell_pos (cell, &o);
  o.x -= h/2.; o.y -= h/2.; o.z -= h/2.;
  for (i = 0; i < 12; i++) {
    FttVector e, d;
    gdouble a, b;
    d.x = o.x + h*edge[i][0].x; d.y = o.y + h*edge[i][0].y; d.z = o.z + h*edge[i][0].z;
    e.x = o.x + h*edge[i][1].x; e.y = o.y + h*edge[i][1].y; e.z = o.z + h*edge[i][1].z;
    a = point_orientation (plane, &e);
    b = point_orientation (plane, &d);
    if ((a <= 0. && b > 0.) || (a >= 0. && b < 0.)) {
      gdouble c = a/(a - b);
      p[i].x = e.x + c*(d.x - e.x); p[i].y = e.y + c*(d.y - e.y); p[i].z = e.z + c*(d.z - e.z);
      orient[i] = (a > 0.);
      if (var) {
	guint j = edge1[i][0];
	if (vc[j] == G_MAXDOUBLE)
	  vc[j] = gfs_cell_corner_value (cell, corner[j], var, max_level);
	d.z = vc[j];
	j = edge1[i][1];
	if (vc[j] == G_MAXDOUBLE)
	  vc[j] = gfs_cell_corner_value (cell, corner[j], var, max_level);
	e.z = vc[j];
	v[i] = e.z + c*(d.z - e.z);
      }
    }
    else
      orient[i] = -1;
  }
}

#define cut_cube_vertices(cut,var,block) {\
  FttDirection d[12];\
  FttVector _p[12], * v[12];\
  gdouble val[12], _vv[12];\
  gint _orient[12];\
  guint _i;\
  cut = FALSE;\
  cube_plane_intersection (cell, GFS_GL2D (gl)->p, _p, _orient, var, _vv, GFS_GL (gl)->maxlevel);\
  for (_i = 0; _i < 12; _i++) {\
    guint nv = 0, _e = _i;\
    while (_orient[_e] >= 0) {\
      guint _m = 0, * _ne = connect[_e][_orient[_e]];\
      d[nv] = _ne[3];\
      val[nv] = _vv[_e];\
      v[nv++] = &(_p[_e]);\
      _orient[_e] = -1;\
      while (_m < 3 && _orient[_e] < 0)\
	_e = _ne[_m++];\
    }\
    if (nv > 2) {\
      block\
      cut = TRUE;\
    }\
  }\
}

static void gl_cell (FttCell * cell, GfsGl * gl)
{
  gboolean cut;
  
  cut_cube_vertices (cut, NULL, {
    guint i;
    glBegin (GL_LINE_LOOP);
    for (i = 0; i < nv; i++)
      glVertex3d (v[i]->x, v[i]->y, v[i]->z);
    glEnd ();
  });
  if (cut)
    gl->size++;
}

/* GfsGlSquares: Object */

static void gl_square (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  GtsColor c;
  gboolean cut;

  c = gfs_colormap_color (gls->cmap, gls->max > gls->min ?
			  (GFS_VARIABLE (cell, gls->v->i) - gls->min)/(gls->max - gls->min) :
			  0.5);
  glColor3f (c.r, c.g, c.b);
  cut_cube_vertices (cut, NULL, {
    guint i;
    glBegin (GL_POLYGON);
    for (i = 0; i < nv; i++)
      glVertex3d (v[i]->x, v[i]->y, v[i]->z);
    glEnd ();
  });
  if (cut)
    gl->size++;
}

static void gl_squares_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  gfs_gl_normal (gl);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_square, gl);
}

/* GfsGlLinear: Object */

#define param(v) (gls->max > gls->min ? ((v) - gls->min)/(gls->max - gls->min) : 0.5)
#define color(v) (gfs_colormap_color (gls->cmap, param (v)))

static void gl_linear_color (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  gboolean cut;

  cut_cube_vertices (cut, gls->v, {
    guint i;

    glBegin (GL_POLYGON);
    for (i = 0; i < nv; i++) {
      GtsColor c = color (val[i]);
      glColor3f (c.r, c.g, c.b);
      glVertex3d (v[i]->x, v[i]->y, v[i]->z);
    }
    glEnd ();
  });
  if (cut)
    gl->size++;
}

static void gl_linear_texture (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  gboolean cut;

  cut_cube_vertices (cut, gls->v, {
    guint i;
    FttVector c;
    gdouble vc = 0.;

    c.x = c.y = c.z = 0.;
    for (i = 0; i < nv; i++) {
      c.x += v[i]->x; c.y += v[i]->y; c.z += v[i]->z;
      vc += val[i];
    }

    glBegin (GL_TRIANGLE_FAN);
    glTexCoord1d (param (vc/nv));
    glVertex3d (c.x/nv, c.y/nv, c.z/nv);
    for (i = 0; i < nv; i++) {
      glTexCoord1d (param (val[i]));
      glVertex3d (v[i]->x, v[i]->y, v[i]->z);
    }
    glTexCoord1d (param (val[0]));
    glVertex3d (v[0]->x, v[0]->y, v[0]->z);
    glEnd ();
  });
  if (cut)
    gl->size++;
}

static void gl_linear_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glShadeModel (GL_SMOOTH);
  gfs_gl_normal (gl);
  if (gl->format == GL2PS_PS || gl->format == GL2PS_EPS || gl->format == GL2PS_PDF)
    gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_linear_color, gl);
  else {
    glEnable (GL_TEXTURE_1D);
    gfs_colormap_texture (GFS_GL_SCALAR (gl)->cmap);
    glColor3f (1., 1., 1.);
    gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_linear_texture, gl);
    glDisable (GL_TEXTURE_1D);
  }
}

/* GfsGlIsoline: Object */

static void gl_isoline (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  GArray * levels = GFS_GL_ISOLINE (gl)->levels;
  gboolean cut;
  
  cut_cube_vertices (cut, gls->v, {
    guint i;

    for (i = 0; i < levels->len; i++) {
      gdouble z = g_array_index (levels, gdouble, i);
      guint j;
      guint n = 0;

      val[nv] = val[0];
      v[nv] = v[0];
      for (j = 0; j < nv; j++)
	if ((val[j] > z && val[j + 1] <= z) || (val[j] <= z && val[j + 1] > z)) {
	  gdouble a = (z - val[j])/(val[j + 1] - val[j]);
	  glVertex3d (v[j]->x + a*(v[j + 1]->x - v[j]->x),
		      v[j]->y + a*(v[j + 1]->y - v[j]->y),
		      v[j]->z + a*(v[j + 1]->z - v[j]->z));
	  n++;
	}
      g_assert (n % 2 == 0);
    }
  });
  if (cut)
    gl->size++;
}

/* GfsGlSolid: Object */

static void cell_traverse_visible_cut (FttCell * root, GfsFrustum * f, 
				       GtsIntersect i,
				       GtsSurface * s, GfsGl * gl,
				       FttCellTraverseCutFunc func, gpointer data)
{
  gdouble r = ftt_cell_size (root)*SLIGHTLY_LARGER*DIAGONAL;
  FttVector p;
  GtsIntersect i1 = i;
  GtsSurface * s1;

  ftt_cell_pos (root, &p);
  if (i1 != GTS_IN) {
    i1 = gfs_sphere_in_frustum (&p, r, f);
    if (i1 == GTS_OUT)
      return;
  }
  if (!(s1 = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (root, GFS_GL_SOLID (gl)->s->i))))
    GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (root, GFS_GL_SOLID (gl)->s->i)) = s1 = 
      gfs_cell_is_cut (root, s, FALSE);
  if (s1 == NULL)
    return;
  if (FTT_CELL_IS_LEAF (root) ||
      ftt_cell_level (root) == gl->maxlevel ||
      gfs_sphere_size (&p, r, f) < f->res)
    (* func) (root, s1, data);
  else {
    struct _FttOct * children = root->children;
    guint n;

    for (n = 0; n < FTT_CELLS; n++) {
      FttCell * c = &(children->cell[n]);
      if (!FTT_CELL_IS_DESTROYED (c))
	cell_traverse_visible_cut (c, f, i1, s1, gl, func, data);
    }
  }
}

static void box_traverse_visible_cut (GfsBox * b, gpointer * data)
{
  cell_traverse_visible_cut (b->root, data[0], GTS_ON, data[4], data[3], data[1], data[2]);
}

/**
 * gfs_gl_cell_traverse_visible_cut:
 * @gl: a #GfsGl2D.
 * @f: a view frustum.
 * @s: a #GtsSurface.
 * @func: the function to call for each visited #FttCell.
 * @data: user data to pass to @func.
 *
 * Traverse the cells of @gl which are visible and which are cut by @s.
 */
static void gfs_gl_cell_traverse_visible_cut (GfsGl * gl,
					      GfsFrustum * f,
					      GtsSurface * s,
					      FttCellTraverseCutFunc func,
					      gpointer data)
{
  gpointer dat[5];

  g_return_if_fail (gl != NULL);
  g_return_if_fail (f != NULL);
  g_return_if_fail (s != NULL);
  g_return_if_fail (func != NULL);

  dat[0] = f;
  dat[1] = func;
  dat[2] = data;
  dat[3] = gl;
  dat[4] = s;
  gts_container_foreach (GTS_CONTAINER (gl->sim), (GtsFunc) box_traverse_visible_cut, dat);
}

typedef struct {
  guint m;
  FttVector * v;
  FttVector * n;
  gdouble * val;
} polygon;

typedef struct {
  polygon * p;
  guint n;
} polygons;

#define param(v) (gls->max > gls->min ? ((v) - gls->min)/(gls->max - gls->min) : 0.5)
#define color(v) (gfs_colormap_color (gls->cmap, param (v)))

static void polygon_draw (polygon * p, GfsGlSolid * gl)
{
  guint i;

  glBegin (GL_POLYGON);
  for (i = 0; i < p->m; i++) {
    if (gl->use_scalar) {
      GfsGlScalar * gls = GFS_GL_SCALAR (gl);
      guint format = GFS_GL (gl)->format;

      if (format == GL2PS_PS || format == GL2PS_EPS || format == GL2PS_PDF) {
	GtsColor c = color (p->val[i]);
	glColor3f (c.r, c.g, c.b);
      }
      else
	glTexCoord1d (param (p->val[i]));
    }
    if (gl->reversed)
      glNormal3d (-p->n[i].x, -p->n[i].y, -p->n[i].z);
    else
      glNormal3d (p->n[i].x, p->n[i].y, p->n[i].z);
    glVertex3d (p->v[i].x, p->v[i].y, p->v[i].z);
  }
  glEnd ();
}

static void polygons_draw (polygons * p, GfsGlSolid * gl)
{
  guint i;

  for (i = 0; i < p->n; i++)
    polygon_draw (&p->p[i], gl);
}

static void polygons_destroy (polygons * p)
{
  guint i;
  for (i = 0; i < p->n; i++) {
    g_free (p->p[i].v);
    g_free (p->p[i].n);
    g_free (p->p[i].val);
  }
  g_free (p->p);
  g_free (p);
}

static polygons * polygons_add (polygons * p, guint n)
{
  if (p == NULL) {
    p = g_malloc (sizeof (polygons));
    p->p = g_malloc (sizeof (polygon));
    p->n = 0;
  }
  else
    p->p = g_realloc (p->p, sizeof (polygon)*(p->n + 1));
  p->p[p->n].v = g_malloc (n*sizeof (FttVector)); 
  p->p[p->n].n = g_malloc (n*sizeof (FttVector));
  p->p[p->n].val = g_malloc (n*sizeof (gdouble));
  p->p[p->n].m = n;
  p->n++;
  return p;
}

static void free_polygons (FttCell * cell, GfsGlSolid * gl)
{
  gpointer po = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, gl->p->i));
  gpointer s = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, gl->s->i));

  if (po) {
    polygons_destroy (po);
    GFS_VARIABLE (cell, gl->p->i) = 0.;
  }
  if (s) {
    gts_object_destroy (s);
    GFS_VARIABLE (cell, gl->s->i) = 0.;
  }
}

void gfs_gl_solid_reset (GfsGlSolid * gl)
{
  g_return_if_fail (gl != NULL);
  
  if (gl->p && gl->s && GFS_GL (gl)->sim)
    gfs_domain_cell_traverse (GFS_DOMAIN (GFS_GL (gl)->sim), 
			      FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
			      (FttCellTraverseFunc) free_polygons, gl);
}

static void gl_solid_destroy (GtsObject * o)
{
  GfsGlSolid * gl = GFS_GL_SOLID (o);

  gfs_gl_solid_reset (gl);
  if (gl->p)
    gts_object_destroy (GTS_OBJECT (gl->p));
  if (gl->s)
    gts_object_destroy (GTS_OBJECT (gl->s));

  (* GTS_OBJECT_CLASS (gfs_gl_solid_class ())->parent_class->destroy) (o);
}

static void gl_solid_read (GtsObject ** o, GtsFile * fp)
{
  GfsGlSolid * gl = GFS_GL_SOLID (*o);
  GtsFileVariable var[] = {
    {GTS_INT, "reversed",   TRUE},
    {GTS_INT, "use_scalar", TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl_solid_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  var[0].data = &gl->reversed;
  var[1].data = &gl->use_scalar;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;
}

static void gl_solid_write (GtsObject * o, FILE * fp)
{
  GfsGlSolid * gl = GFS_GL_SOLID (o);

  (* GTS_OBJECT_CLASS (gfs_gl_solid_class ())->parent_class->write) (o, fp);

  fprintf (fp, " {\n"
	   "  reversed = %d\n"
	   "  use_scalar = %d\n"
	   "}",
	   gl->reversed, (gl->use_scalar != NULL));
}

typedef struct {
  GtsPoint p[8];
  gdouble x[12], v[12];
  guint n[12];
  gint inside[12];
  GtsVector n1[12];
} CellCube;

static gdouble segment_triangle_intersection (GtsPoint * E, GtsPoint * D,
					      GtsTriangle * t,
					      gboolean * inside)
{
  GtsVertex * Av, * Bv, * Cv;
  GtsPoint * A, * B, * C;
  gint ABCE, ABCD, ADCE, ABDE, BCDE;
  GtsEdge * AB, * BC, * CA;
  gdouble a, b;
  gboolean reversed = FALSE;

  gts_triangle_vertices_edges (t, NULL, &Av, &Bv, &Cv, &AB, &BC, &CA);
  A = GTS_POINT (Av);
  B = GTS_POINT (Bv);
  C = GTS_POINT (Cv);
  ABCE = gts_point_orientation_3d_sos (A, B, C, E);
  ABCD = gts_point_orientation_3d_sos (A, B, C, D);
  if (ABCE < 0 || ABCD > 0) {
    GtsPoint * tmpp;
    gint tmp;

    tmpp = E; E = D; D = tmpp;
    tmp = ABCE; ABCE = ABCD; ABCD = tmp;
    reversed = TRUE;
  }
  if (ABCE < 0 || ABCD > 0)
    return -1.;
  ADCE = gts_point_orientation_3d_sos (A, D, C, E);
  if (ADCE < 0)
    return -1.;
  ABDE = gts_point_orientation_3d_sos (A, B, D, E);
  if (ABDE < 0)
    return -1.;
  BCDE = gts_point_orientation_3d_sos (B, C, D, E);
  if (BCDE < 0)
    return -1.;
  *inside = reversed ? (ABCD < 0) : (ABCE < 0);
  a = gts_point_orientation_3d (A, B, C, E);
  b = gts_point_orientation_3d (A, B, C, D);
  if (a != b)
    return reversed ? 1. - a/(a - b) : a/(a - b);
  /* D and E are contained within ABC */
  g_assert (a == 0.);
  return 0.5;
}

static void triangle_cube_intersection (GtsTriangle * t, CellCube * cube)
{
  guint i;

  for (i = 0; i < 12; i++) {
    gboolean ins;
    gdouble x = segment_triangle_intersection (&cube->p[edge1[i][0]], &cube->p[edge1[i][1]], 
					       t, &ins);
    if (x != -1.) {
      GtsVector n;

      cube->x[i] += x; cube->n[i]++;
      cube->inside[i] += ins ? 1 : -1;
      gts_triangle_normal (t, &n[0], &n[1], &n[2]);
      cube->n1[i][0] -= n[0];
      cube->n1[i][1] -= n[1];
      cube->n1[i][2] -= n[2];
    }
  }
}

static void cell_size (FttCell * cell, FttVector * h)
{
  h->x = h->y = ftt_cell_size (cell);
#if FTT_2D3
  h->z = 1.;
#else  /* 3D */
  h->z = h->x;
#endif /* 3D */
}

static void cube_intersections (FttCell * cell,
				GtsSurface * s,
				GfsVariable * v,
				FttVector p[12],
				FttVector n[12],
				gint orient[12],
				gdouble val[12],
				gint max_level)
{
  CellCube cube;
  FttVector o, h;
  guint i;
  gint inside[8] = {0,0,0,0,0,0,0,0};

  ftt_cell_pos (cell, &o);
  cell_size (cell, &h);
  for (i = 0; i < FTT_DIMENSION; i++)
    (&o.x)[i] -= (&h.x)[i]/2.;
  for (i = 0; i < 12; i++) {
    cube.x[i] = 0.; cube.n[i] = 0; cube.inside[i] = 0;
    cube.n1[i][0] = cube.n1[i][1] = cube.n1[i][2] = 0.;
  }
  for (i = 0; i < 8; i++) { /* for each vertex of the cube */
    cube.p[i].x = o.x + h.x*vertex[i].x;
    cube.p[i].y = o.y + h.y*vertex[i].y;
    cube.p[i].z = o.z + h.z*vertex[i].z;
    if (v)
      cube.v[i] = gfs_cell_corner_value (cell, corner[i], v, max_level);
  }

  gts_surface_foreach_face (s, (GtsFunc) triangle_cube_intersection, &cube);

  for (i = 0; i < 12; i++) /* for each edge of the cube */
    if (cube.n[i] % 2 != 0) { /* only for odd number of intersections */
      guint j = edge1[i][0], k = edge1[i][1];

      /* intersection vertex position is the average of all the n[i] intersections */
      cube.x[i] /= cube.n[i];

      p[i].x = (1. - cube.x[i])*cube.p[j].x + cube.x[i]*cube.p[k].x;
      p[i].y = (1. - cube.x[i])*cube.p[j].y + cube.x[i]*cube.p[k].y;
      p[i].z = (1. - cube.x[i])*cube.p[j].z + cube.x[i]*cube.p[k].z;

      n[i].x = cube.n1[i][0]/cube.n[i];
      n[i].y = cube.n1[i][1]/cube.n[i];
      n[i].z = cube.n1[i][2]/cube.n[i];

      if (v)
	val[i] = (1. - cube.x[i])*cube.v[j] + cube.x[i]*cube.v[k];

      g_assert (inside[j] == 0 || inside[j] == cube.inside[i]);
      g_assert (inside[k] == 0 || inside[k] == - cube.inside[i]);
      inside[j] = cube.inside[i];
      inside[k] = - cube.inside[i];
      orient[i] = (inside[j] > 0);
    }
    else
      orient[i] = -1;
}

#define solid_cube_vertices(cut,s,var,max_level,block) {\
  FttVector _p[12], _n[12], * v[12], * n[12];\
  gdouble _val[12], val[12];\
  gint _orient[12];\
  guint _i;\
  cube_intersections (cell, s, var, _p, _n, _orient, _val, max_level);\
  for (_i = 0; _i < 12; _i++) {\
    guint nv = 0, _e = _i;\
    while (_orient[_e] >= 0) {\
      guint _m = 0, * _ne = connect[_e][_orient[_e]];\
      n[nv] = &(_n[_e]);\
      val[nv] = _val[_e];\
      v[nv++] = &(_p[_e]);\
      _orient[_e] = -1;\
      while (_m < 3 && _orient[_e] < 0)\
	_e = _ne[_m++];\
    }\
    if (nv > 2) {\
      block\
    }\
  }\
}

static void gl_solid (FttCell * cell, GtsSurface * s, GfsGl * gl)
{
  GfsGlSolid * gls = GFS_GL_SOLID (gl);
  polygons * p = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, gls->p->i));

  if (p) {
    polygons_draw (p, gls);
    gl->size++;
  }
  else {
    solid_cube_vertices (cut, s, gls->use_scalar ? GFS_GL_SCALAR (gl)->v : NULL, gl->maxlevel, {
      guint i;
      polygon * q;

      p = polygons_add (p, nv);
      q = &p->p[p->n - 1];

      for (i = 0; i < nv; i++) {
	q->v[i] = *(v[i]);
	q->n[i] = *(n[i]);
	q->val[i] = val[i];
      }
    });
    if (p) {
      polygons_draw (p, gls);
      GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, gls->p->i)) = p;
      gl->size++;
    }
  }
}

static void gl_solid_draw (GfsGl * gl)
{
  if (gl->sim->surface) {
    GfsFrustum f;
    GfsGlSolid * gls = GFS_GL_SOLID (gl);

    if (gls->use_scalar && gls->use_scalar != GFS_GL_SCALAR (gl)->v) {
      gls->use_scalar = GFS_GL_SCALAR (gl)->v;
      gfs_gl_solid_reset (gls);
    }
    
    gfs_gl_get_frustum (gl, &f);
    gl->size = 0;
    glShadeModel (GL_SMOOTH);
    glEnable (GL_NORMALIZE);
    if (gls->use_scalar && 
	gl->format != GL2PS_PS && gl->format != GL2PS_EPS && gl->format != GL2PS_PDF) {
      glEnable (GL_TEXTURE_1D);
      gfs_colormap_texture (GFS_GL_SCALAR (gl)->cmap);
      glColor3f (1., 1., 1.);
      gfs_gl_cell_traverse_visible_cut (gl, &f, gl->sim->surface,
					(FttCellTraverseCutFunc) gl_solid, gl);
      glDisable (GL_TEXTURE_1D);
    }
    else {
      glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
      gfs_gl_cell_traverse_visible_cut (gl, &f, gl->sim->surface,
					(FttCellTraverseCutFunc) gl_solid, gl);
    }
  }
}

static void reset_p_s (FttCell * cell, GfsGlSolid * gl)
{
  GFS_VARIABLE (cell, gl->p->i) = 0.;
  GFS_VARIABLE (cell, gl->s->i) = 0.;
}

static void gl_solid_set_simulation (GfsGl * object, GfsSimulation * sim)
{
  GfsDomain * domain = GFS_DOMAIN (sim);
  GfsGlSolid * gls = GFS_GL_SOLID (object);

  gfs_gl_solid_reset (gls);

  (*GFS_GL_CLASS (GTS_OBJECT_CLASS (gfs_gl_solid_class ())->parent_class)->set_simulation)
    (object, sim);

  if (gls->p)
    gts_object_destroy (GTS_OBJECT (gls->p));
  gls->p = gfs_temporary_variable (domain);
  if (gls->s)
    gts_object_destroy (GTS_OBJECT (gls->s));
  gls->s = gfs_temporary_variable (domain);

  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
			    (FttCellTraverseFunc) reset_p_s, gls);
}

static void gl_solid_class_init (GfsGlClass * klass)
{
  GTS_OBJECT_CLASS (klass)->destroy = gl_solid_destroy;
  GTS_OBJECT_CLASS (klass)->read = gl_solid_read;
  GTS_OBJECT_CLASS (klass)->write = gl_solid_write;
  klass->set_simulation = gl_solid_set_simulation;
  klass->draw = gl_solid_draw;
  klass->pick = NULL;
}

static void gl_solid_init (GfsGl * gl)
{
  GtsColor c = { 1., 1., 1. };

  gl->lc = c;
}

GfsGlClass * gfs_gl_solid_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_solid_info = {
      "GfsGlSolid",
      sizeof (GfsGlSolid),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_solid_class_init,
      (GtsObjectInitFunc) gl_solid_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_scalar_class ()),
				  &gfs_gl_solid_info);
  }

  return klass;
}

/* GfsGlFractions: Object */

static void gl_fractions (FttCell * cell, GfsGl * gl)
{
  GfsSolidVector * s = GFS_STATE (cell)->solid;
  FttCellFace f;
  gdouble h = ftt_cell_size (cell)/2.;

  f.cell = cell;
  for (f.d = 0; f.d < FTT_NEIGHBORS; f.d++) 
    if (s->s[f.d] > 0. && s->s[f.d] < 1.) {
      FttComponent c = f.d/2;
      static FttVector o[FTT_DIMENSION][4] = {
	{{0.,1.,-1.}, {0.,1.,1.}, {0.,-1.,1.}, {0.,-1.,-1.}},
	{{1.,0.,-1.}, {1.,0.,1.}, {-1.,0.,1.}, {-1.,0.,-1.}},
	{{1.,-1.,0.}, {1.,1.,0.}, {-1.,1.,0.}, {-1.,-1.,0.}}
      };
      static FttVector n[FTT_NEIGHBORS] = {
	{ 1., 0., 0. }, { -1., 0., 0.},
	{ 0., 1., 0. }, { 0., -1., 0.},
	{ 0., 0., 1. }, { 0., 0., -1.},
      };
      FttVector p;
      gdouble l = h*sqrt (s->s[f.d]);

      glNormal3d (n[f.d].x, n[f.d].y, n[f.d].z);
      ftt_face_pos (&f, &p);
      glVertex3d (p.x + l*o[c][0].x, p.y + l*o[c][0].y, p.z + l*o[c][0].z);
      glVertex3d (p.x + l*o[c][1].x, p.y + l*o[c][1].y, p.z + l*o[c][1].z);
      glVertex3d (p.x + l*o[c][2].x, p.y + l*o[c][2].y, p.z + l*o[c][2].z);
      glVertex3d (p.x + l*o[c][3].x, p.y + l*o[c][3].y, p.z + l*o[c][3].z);
    }
  gl->size++;
}

static void gl_fractions_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glShadeModel (GL_CONSTANT);
  glBegin (GL_QUADS);
  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_cell_traverse_visible_mixed (gl, &f, (FttCellTraverseFunc) gl_fractions, gl);
  glEnd ();
}

/* GfsGlBoundaries: Object */

static void gl_boundaries (FttCell * cell, GfsGl * gl)
{
  if (!GFS_IS_MIXED (cell)) {
    FttCellNeighbors n;
    gdouble h = ftt_cell_size (cell);
    FttVector p;
    guint i;

    ftt_cell_neighbors (cell, &n);
    ftt_cell_pos (cell, &p);
    p.x -= h/2.; p.y -= h/2.; p.z -= h/2.;
    for (i = 0; i < 12; i++) {
      static FttDirection edge2[12][2] = {
	{FTT_BOTTOM,FTT_BACK},{FTT_BOTTOM,FTT_FRONT},
	{FTT_TOP,FTT_FRONT},  {FTT_TOP,FTT_BACK},
	{FTT_LEFT,FTT_BACK},  {FTT_LEFT,FTT_FRONT},
	{FTT_RIGHT,FTT_FRONT},{FTT_RIGHT,FTT_BACK},
	{FTT_BOTTOM,FTT_LEFT},{FTT_BOTTOM,FTT_RIGHT},
	{FTT_TOP,FTT_RIGHT},   {FTT_TOP,FTT_LEFT}
      };
      guint j = edge2[i][0], k = edge2[i][1];

      if ((!n.c[j] || GFS_CELL_IS_BOUNDARY (n.c[j])) &&
	  (!n.c[k] || GFS_CELL_IS_BOUNDARY (n.c[k]))) {
	gl->size++;
	glVertex3d (p.x + edge[i][0].x*h, 
		    p.y + edge[i][0].y*h,
		    p.z + edge[i][0].z*h);
	glVertex3d (p.x + edge[i][1].x*h, 
		    p.y + edge[i][1].y*h,
		    p.z + edge[i][1].z*h);
      }
    }
  }
}

/* GfsGlLevels: Object */

static void gl_face (FttCell * cell, GfsGlLevels * gl)
{
  gboolean cut, drawn = FALSE;
  
  cut_cube_vertices (cut, NULL, {
    guint i;
    for (i = 0; i < nv - 1; i++) {
      FttCell * n = ftt_cell_neighbor (cell, d[i]);
      if (n && (floor (GFS_VARIABLE (cell, gl->v->i)) != 
		floor (GFS_VARIABLE (n, gl->v->i)))) {
	glVertex3d (v[i]->x, v[i]->y, v[i]->z);
	glVertex3d (v[i+1]->x, v[i+1]->y, v[i+1]->z);
	drawn = TRUE;
      }
    }
  });
  if (drawn)
    GFS_GL (gl)->size++;
}

/* GfsGlVectors: Object */

static void gl_vector (FttCell * cell, GfsGl * gl)
{
  if (plane_cuts_cell (GFS_GL2D (gl)->p, cell)) {
    GfsGl2D * gl2D = GFS_GL2D (gl);
    GfsGlVectors * gls = GFS_GL_VECTORS (gl);
    FttComponent c;
    FttVector pos, f;
    gdouble a;
    
    gl->size++;
    if (gls->use_scalar) {
      GfsGlScalar * gla = GFS_GL_SCALAR (gl);
      GtsColor c = gfs_colormap_color (gla->cmap, gla->max > gla->min ?
				       (GFS_VARIABLE (cell, gla->v->i) - gla->min)/
				       (gla->max - gla->min) :
				       0.5);
      glColor3f (c.r, c.g, c.b);
    }
    gfs_cell_cm (cell, &pos);
    a = ((gl2D->p[0].x - pos.x)*gl2D->n.x + 
	 (gl2D->p[0].y - pos.y)*gl2D->n.y + 
	 (gl2D->p[0].z - pos.z)*gl2D->n.z);
    pos.x += a*gl2D->n.x; pos.y += a*gl2D->n.y; pos.z += a*gl2D->n.z;
    for (c = 0; c < FTT_DIMENSION; c++)
      (&f.x)[c] = GFS_VARIABLE (cell, gls->v[c]->i);
    f.x *= gls->scale;
    f.y *= gls->scale;
    f.z *= gls->scale;
    glVertex3d (pos.x + f.x - (f.x - f.y/2.)/5., pos.y + f.y - (f.x/2. + f.y)/5., pos.z + f.z);
    glVertex3d (pos.x + f.x, pos.y + f.y, pos.z + f.z);
    glVertex3d (pos.x + f.x, pos.y + f.y, pos.z + f.z);
    glVertex3d (pos.x + f.x - (f.x + f.y/2.)/5., pos.y + f.y + (f.x/2. - f.y)/5., pos.z + f.z);
    glVertex3d (pos.x, pos.y, pos.z);
    glVertex3d (pos.x + f.x, pos.y + f.y, pos.z + f.z);
  }
}

/* GfsGlEllipses: Object */

static void gl_ellipse (FttCell * cell, GfsGl * gl)
{
  if (plane_cuts_cell (GFS_GL2D (gl)->p, cell)) {
    GfsGl2D * gl2D = GFS_GL2D (gl);
    GfsGlEllipses * gls = GFS_GL_ELLIPSES (gl);
    FttVector pos;
    gdouble a, t;
    
    gl->size++;
    if (gls->use_scalar) {
      GfsGlScalar * gla = GFS_GL_SCALAR (gl);
      GtsColor c = gfs_colormap_color (gla->cmap, gla->max > gla->min ?
				       (GFS_VARIABLE (cell, gla->v->i) - gla->min)/
				       (gla->max - gla->min) :
				       0.5);
      glColor3f (c.r, c.g, c.b);
    }
    gfs_cell_cm (cell, &pos);
    a = ((gl2D->p[0].x - pos.x)*gl2D->n.x + 
	 (gl2D->p[0].y - pos.y)*gl2D->n.y + 
	 (gl2D->p[0].z - pos.z)*gl2D->n.z);
    pos.x += a*gl2D->n.x; pos.y += a*gl2D->n.y; pos.z += a*gl2D->n.z;
    glBegin (GL_LINE_LOOP);
    for (t = 0.; t < 2.*M_PI; t += 2.*M_PI/20.) {
      gdouble cost = cos (t), sint = sin (t);
      
      glVertex3d (pos.x + gls->scale*(GFS_VARIABLE (cell, gls->v[0]->i)*cost + 
				      GFS_VARIABLE (cell, gls->v[2]->i)*sint),
		  pos.y + gls->scale*(GFS_VARIABLE (cell, gls->v[1]->i)*cost + 
				      GFS_VARIABLE (cell, gls->v[3]->i)*sint),
		  pos.z);
    }
    glEnd ();
  }
}

/* GfsGlStreamline: Object */

static GSList * circle_profile (GtsVertexClass * klass, 
				gdouble radius, guint np)
{
  GSList * lp = NULL;
  guint i;

  for (i = 0; i <= np; i++) {
    gdouble a = 2.*M_PI*i/(gdouble) np;
    gdouble cosa = cos (a), sina = sin (a);
    GtsPoint * p = gts_point_new (GTS_POINT_CLASS (klass), radius*cosa, radius*sina, 0.);
    gdouble * n = GTS_VERTEX_NORMAL (p)->n;

    n[0] = cosa; n[1] = sina; n[2] = 0.;
    lp = g_slist_prepend (lp, p);
  }
  return lp;
}

static void matrix_transpose (GtsMatrix * m)
{
  guint i, j;

  for (i = 1; i < 3; i++)
    for (j = 0; j < i; j++) {
      gdouble t = m[i][j];

      m[i][j] = m[j][i];
      m[j][i] = t;
    }
}

static void base (GtsMatrix * b, GtsPoint * p1, GtsPoint * p2)
{
  GtsVector x, y;

  x[0] = b[0][0];
  x[1] = b[1][0];
  x[2] = b[2][0];
  gts_vector_init (b[2], p2, p1);
  gts_vector_normalize (b[2]);
  gts_vector_cross (y, b[2], x);
  if (gts_vector_norm (y) > 1e-2) {
    b[1][0] = y[0];
    b[1][1] = y[1];
    b[1][2] = y[2];
    gts_vector_normalize (b[1]);
  }
  gts_vector_cross (b[0], b[1], b[2]);
  gts_vector_normalize (b[0]);
  matrix_transpose (b);
}

static void point_list (GtsMatrix * b, 
			GtsMatrix * c,
			GtsPoint * o,
			GSList * profile,
			GtsVertexNormal * pl, 
			guint np)
{
  guint i;
  gboolean colored = FALSE;

#if 0
  if (GTS_IS_COLORED_VERTEX (o) && 
      gts_object_class_is_from_class (GTS_OBJECT_CLASS (s->vertex_class),
				      GTS_OBJECT_CLASS (gts_colored_vertex_class ())))
    colored = TRUE;
#endif
  if (FALSE/*GFS_IS_TWISTED_VERTEX (o)*/) {
#if 0
    gdouble t = GFS_TWISTED_VERTEX (o)->theta;
    gdouble sint = sin (t), cost = cos (t);
    GtsMatrix * r = gts_matrix_new (cost, -sint, 0., 0.,
				    sint,  cost, 0., 0.,
				    0.,      0., 1., 0.,
				    0.,      0., 0., 0.);
    
    c = gts_matrix_product (b, r);
    gts_matrix_destroy (r);
#endif
  }
  else
    gts_matrix_assign (c,
		       b[0][0], b[0][1], b[0][2], 0.,
		       b[1][0], b[1][1], b[1][2], 0.,
		       b[2][0], b[2][1], b[2][2], 0.,
		       0., 0., 0., 0.);

  for (i = 0; i < np; i++, profile = profile->next, pl++) {
    GtsPoint * p = profile->data;
    gdouble * n = GTS_VERTEX_NORMAL (p)->n;
    GtsPoint n1;
    
    *GTS_POINT (pl) = *p;
#if 0
    if (colored)
      GTS_COLORED_VERTEX (v)->c = GTS_COLORED_VERTEX (o)->c;
#endif
    gts_point_transform (GTS_POINT (pl), c);
    GTS_POINT (pl)->x += o->x;
    GTS_POINT (pl)->y += o->y;
    GTS_POINT (pl)->z += o->z;

    n1.x = n[0]; n1.y = n[1]; n1.z = n[2];
    gts_point_transform (&n1, c);
    n = pl->n;
    n[0] = n1.x; n[1] = n1.y; n[2] = n1.z;
  }
}

static void draw_point (GtsPoint * p)
{
  glVertex3d (p->x, p->y, p->z);
}

static void draw_normal (GtsVertexNormal * v)
{
  glNormal3d (v->n[0], v->n[1], v->n[2]);
}

static void draw_faces (GtsVertexNormal * v1, GtsVertexNormal * v2, guint np)
{
  guint i;

  glBegin (GL_QUAD_STRIP);
  for (i = 0; i < np; i++, v1++, v2++) {
    draw_normal (v1);
    draw_point (GTS_POINT (v1));
    draw_normal (v2);
    draw_point (GTS_POINT (v2));    
  }
  glEnd ();
}

static GList * next_far_enough (GList * p, gdouble size)
{
  GtsPoint * ps;
  GList * pf = NULL;

  if (p == NULL)
    return NULL;
  ps = p->data;
  p = p->next;
  size *= size;
  while (p && !pf) {
    if (gts_point_distance2 (ps, p->data) > size)
      pf = p;
    p = p->next;
  }
  return pf;
}

static void extrude_profile (GSList * profile, GList * path)
{
  GtsMatrix * r, * c;
  GtsPoint * p0, * p1, * p2;
  GtsVertexNormal * pl1, * pl2, * tmp;
  GtsBBox * bb;
  gdouble size;
  guint np;

  g_return_if_fail (profile != NULL);
  g_return_if_fail (path != NULL);

  bb = gts_bbox_points (gts_bbox_class (), profile);
  size = bb->x2 - bb->x1;
  if (bb->y2 - bb->y1 > size)
    size = bb->y2 - bb->y1;
  gts_object_destroy (GTS_OBJECT (bb));

  size /= 4.;

  p0 = path->data;
  path = next_far_enough (path, size);
  if (path == NULL)
    return;
  p1 = path->data;
  r = gts_matrix_identity (NULL);
  c = gts_matrix_identity (NULL);
  np = g_slist_length (profile);
  pl1 = g_malloc (sizeof (GtsVertexNormal)*np);
  pl2 = g_malloc (sizeof (GtsVertexNormal)*np);

  base (r, p0, p1);
  point_list (r, c, p0, profile, pl1, np);
  do {
    path = next_far_enough (path, size);
    p2 = path ? path->data : NULL;
    if (p2)
      base (r, p0, p2);
    else
      base (r, p0, p1);
    point_list (r, c, p1, profile, pl2, np);
    draw_faces (pl1, pl2, np);
    tmp = pl1;
    pl1 = pl2;
    pl2 = tmp;
    p0 = p1;
    p1 = p2;
  } while (p1);

  g_free (pl1);
  g_free (pl2);
  gts_matrix_destroy (c);
  gts_matrix_destroy (r);
}

static void gl_streamline_draw (GfsGlStreamline * s,
				GfsGlStreamlines * gls)
{
  if (gls->radius > 0.) {
    GSList * profile = circle_profile (gts_vertex_normal_class (), gls->radius, 10);

    glShadeModel (GL_SMOOTH);
    extrude_profile (profile, s->l);

    g_slist_foreach (profile, (GFunc) gts_object_destroy, NULL);
    g_slist_free (profile);
  }
  else {
    glBegin (GL_LINE_STRIP);
    g_list_foreach (s->l, (GFunc) draw_point, NULL);
    glEnd ();
  }
}

