
#include "comb/perm-heap.h"
#include "comb/fact2perm.h"

#include "comb/comb-print.h"
#include "aux0/swap.h"

#include "fxtio.h"
#include "nextarg.h"
#include "fxttypes.h"
#include "jjassert.h"


//% Gray code for permutations, CAT algorithm.
//% Algorithm following
//% B. R. Heap: Permutations by Interchanges,
//% The Computer Journal, vol. 6, pp. 293-294, (1963).

// Cf. comb/perm-heap2-demo.cc
// Cf. comb/perm-heap-swaps-demo.cc

//#define TIMING  // uncomment to disable printing

#define INVERSE  // define to show inverse permutations

int
main(int argc, char **argv)
{
    ulong n = 4;
    NXARG(n, "Number of elements to permute, n > 0");
    jjassert( n > 0 );


    perm_heap P(n);
    P.first();

    ulong ct = 0;

#if defined TIMING

    do  { ++ct; }  while ( P.next() );

#else  // TIMING

    const bool dfz= true;  // whether to print dots for zeros
    const ulong *x = P.data();
    ulong *rfc = new ulong[n];  // rising factorial

#if defined INVERSE
    ulong *xi = new ulong[n];  // inverse permutations
    for (ulong k=0; k<n; ++k)  xi[k] = k;
#endif  // INVERSE

    do
    {
        cout << setw(4) << ct << ":";
        P.print("    ", dfz);

        ulong sw1, sw2;
        P.get_swap(sw1, sw2);
        cout << "     (" << sw1 << ", " << sw2 << ") ";

        print_mixedradix("    ", P.d_, n-1, dfz);


        perm2rfact(x, n, rfc);
        print_mixedradix("    ", rfc, n-1, dfz);


#if defined INVERSE
        cout << "    ";
        swap2( xi[x[sw1]], xi[x[sw2]]);  // update inverse permutation
        print_perm("    ", xi, n, dfz);
        perm2rfact(xi, n, rfc);
        print_mixedradix("    ", rfc, n-1, dfz);
#endif  // INVERSE

        cout << endl;
        ++ct;
    }
    while ( P.next() );

    delete [] rfc;
#if defined INVERSE
    delete [] xi;
#endif

#endif  // TIMING

    cout << " ct=" << ct << endl;

    return 0;
}
// -------------------------

/*
Timing: Intel(R) Core(TM) i7-8700K CPU @ 3.70GHz
GCC 12.2.0

 time ./bin 12
 arg 1: 12 == n  [Number of elements to permute, n > 0]  default=4
 ct=479001600
0.93user 0.00system 0:00.93elapsed 100%CPU
 ==> 479001600/0.93 == 515,055,483 per second

no swaps:
 time ./bin 12
 arg 1: 12 == n  [Number of elements to permute, n > 0]  default=4
 ct=479001600
0.81user 0.00system 0:00.81elapsed 100%CPU
 ==> 479001600/0.81 == 591,360,000 per second

*/

/*
Timing: (AMD Phenom II X4 945 3000MHz)

 time ./bin 12
arg 1: 12 == n  [Number of elements to permute]  default=4
 ct=479001600
./bin 12  3.13s user 0.00s system 99% cpu 3.136 total
 ==> 479001600/3.13 == 153,035,654 per second

 time ./bin 12
arg 1: 12 == n  [Number of elements to permute]  default=4
 ct=479001600
./bin 12  1.33s user 0.00s system 99% cpu 1.332 total
 ==> 479001600/1.33 == 360,151,578 per second

*/


/// Emacs:
/// Local Variables:
/// MyRelDir: "demo/comb"
/// makefile-dir: "../../"
/// make-target: "1demo DSRC=demo/comb/perm-heap-demo.cc"
/// make-target2: "1demo DSRC=demo/comb/perm-heap-demo.cc DEMOFLAGS=-DTIMING"
/// End:

