
#include "comb/dyck-gray.h"

#include "comb/comb-print.h"

#include "jjassert.h"
#include "fxtio.h"
#include "fxttypes.h"
#include "nextarg.h"


//% Gray code for k-ary Dyck words with homogeneous transitions.
//% Loopless algorithm following
//%   Dominique Roelants van Baronaigien:
//%   "A Loopless Gray-Code Algorithm for Listing k-ary Trees",
//%   Journal of Algorithms, vol.35, pp.100-107, (2000).

//#define TIMING  // uncomment to disable printing


int
main(int argc, char **argv)
{
    ulong n = 4;
    NXARG(n, "Number of ones in words.");
    ulong k = 3;
    NXARG(k, "k-ary Dyck words (k>=2).");
    jjassert( k >= 2 );

    dyck_gray D(n, k);
    D.first();

    ulong ct = 0;
    ulong ch = 0;
    do
    {
        ++ct;
#if !defined TIMING
//        const ulong *x = D.data();
        cout << setw(4) << ct << ":";
        print_sym_vec("  ", D.data(), n);
        print_sign_vec("    ", D.dr_+1, n);
//        print_vec("  np=", D.np_+1, n);
//        print_vec("  mx=", D.mx_+1, n);  // does not change

        print_set1_as_deltaset("    ", D.data(), n, k*n);

        cout << endl;
#endif
    }
    while ( (ch=D.next()) );

    cout << " ct=" << ct << endl;

    return 0;
}
// -------------------------

/*

Timing: Intel(R) Core(TM) i7-8700K CPU @ 3.70GHz
GCC 12.2.0

time ./bin 18 2
arg 1: 18 == n  [Number of ones in words.]  default=4
arg 2: 2 == k  [k-ary Dyck words (k>=2).]  default=3
 ct=477638700
1.23user 0.00system 0:01.23elapsed 100%CPU
 ==> 477638700/1.23 == 388,324,146 per second

time ./bin 13 3
arg 1: 13 == n  [Number of ones in words.]  default=4
arg 2: 3 == k  [k-ary Dyck words (k>=2).]  default=3
 ct=300830572
0.73user 0.00system 0:00.73elapsed 100%CPU
 ==> 300830572/0.73 == 412,096,673 per second

time ./bin 12 4
time ./bin 12 4
arg 1: 12 == n  [Number of ones in words.]  default=4
arg 2: 4 == k  [k-ary Dyck words (k>=2).]  default=3
 ct=1882933364
4.44user 0.00system 0:04.44elapsed 100%CPU
 ==> 1882933364/4.44 == 424,084,090 per second
*/

/*
Timing: (AMD Phenom II X4 945 3000MHz)

time ./bin 18 2
  ct=477638700
 ./bin 18 2  3.14s user 0.00s system 99% cpu 3.145 total
 ==> 477638700/3.14 == 152,114,235 per second

time ./bin 13 3
  ct=300830572
./bin 13 3  1.93s user 0.00s system 99% cpu 1.931 total
 ==> 300830572/1.93 == 155,870,762 per second

time ./bin 12 4
./bin 12 4  9.82s user 0.00s system 99% cpu 9.828 total
  ct=1882933364
 ==> 1882933364/9.82 == 191,744,741 per second

*/


/// Emacs:
/// Local Variables:
/// MyRelDir: "demo/comb"
/// makefile-dir: "../../"
/// make-target: "1demo DSRC=demo/comb/dyck-gray-demo.cc"
/// make-target2: "1demo DSRC=demo/comb/dyck-gray-demo.cc DEMOFLAGS=-DTIMING"
/// End:
