#if !defined HAVE_DICHOTOMY_H__
#define      HAVE_DICHOTOMY_H__
// This file is part of the FXT library.
// Copyright (C) 2023 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "fxttypes.h"
#include "fxtio.h"
#include "jjassert.h"


class dichotomy
// Partitioning of n elements into two classes, 'left' and 'right'.
// Elements can be moved (by value or index) between 'left' and 'right' in O(1).
{
private:
    ulong n_;        // number of elements
    ulong n_left_;   // number of elements in the left class
    ulong *x_;       // permutation of {0, 1, ..., n-1}
    ulong *xi_;      // inverse permutation

    dichotomy(const dichotomy&) = delete;
    dichotomy & operator = (const dichotomy&) = delete;

public:
    explicit dichotomy(ulong n, bool right_q=true)
    {
        n_ = n;
        x_ = new ulong[n_];
        xi_ = new ulong[n_];
        reset( right_q );
    }

    ~dichotomy()
    {
        delete [] x_;
        delete [] xi_;
    }

    void reset(bool right_q=true)
    {
        if ( right_q )  { n_left_ = 0; }  // everything in the right class
        else            { n_left_ = n_; }  // everything in the left class
        for (ulong j=0; j<n_; ++j)  xi_[j] = j;
        for (ulong j=0; j<n_; ++j)  x_[j] = j;
    }

public:
    ulong num_left()  const  { return n_left_; }
    const ulong * data_left()  const  { return x_; }

    ulong num_right()  const  { return n_ - n_left_; }
    const ulong * data_right()  const  { return x_ + n_left_; }

public:
    bool idx_right_q( ulong i )  const
    { return ( i >= n_left_ ); }

    bool idx_left_q( ulong i )  const
    { return  ! idx_right_q( i ); }

    bool elem_right_q( ulong e )  const
    { return  idx_right_q( xi_[e] ); }

    bool elem_left_q( ulong e )  const
    { return  ! elem_right_q( e ); }

//public:
//    void switch_elem( ulong e );
//    void switch_idx( ulong i );

public:
    bool move_elem_left( ulong e1 )
    {
//        cout << " :: move_elem_left(" << e1 << ")\n";
        const ulong i1 = xi_[e1];   // position of element e1
        if ( i1 < n_left_ )  { return false; }  // already at left
        const ulong i2 = n_left_ - 0;  // e1 will be moved to position i2
        const ulong e2 = x_[i2];  // element at i2
        xi_[e1] = i2;  xi_[e2] = i1;
        x_[i1] = e2;  x_[i2] = e1;
        n_left_ += 1;
        return true;
    }

    bool move_elem_right( ulong e1 )
    {
//        cout << " :: move_elem_right(" << e1 << ")\n";
        const ulong i1 = xi_[e1];   // position of element e1
        if ( i1 >= n_left_ )  { return false; }  // already at right
        const ulong i2 = n_left_ - 1;  // e1 will be moved to position i2
        const ulong e2 = x_[i2];  // element at i2
        xi_[e1] = i2;  xi_[e2] = i1;
        x_[i1] = e2;  x_[i2] = e1;
        n_left_ -= 1;
        return true;
    }

public:
    bool move_idx_left( ulong i1, ulong & e1 )
    // Move element at index i1 to the left.
    // Write moved element to e1.
    // Inverse operation is move_elem_right( e1 ).
    {
//        cout << " :: move_idx_right(" << i1 << ")\n";
        if ( i1 < n_left_ )  { return false; }
        e1 = x_[i1];   // element at position i1
        const ulong i2 = n_left_ - 0;  // e1 will be moved to position i2
        const ulong e2 = x_[i2];  // element at i2
        xi_[e1] = i2;  xi_[e2] = i1;
        x_[i1] = e2;  x_[i2] = e1;
        n_left_ += 1;
        return true;
    }

    bool move_idx_right( ulong i1, ulong & e1  )
    // Write moved element to e1.
    // Inverse operation is move_elem_left( e1 ).
    {
//        cout << " :: move_idx_right(" << i1 << ")\n";
        if ( i1 >= n_left_ )  { return false; }  // already at right
        e1 = x_[i1];   // element at position i1
        const ulong i2 = n_left_ - 1;  // e1 will be moved to position i2
        const ulong e2 = x_[i2];  // element at i2
        xi_[e1] = i2;  xi_[e2] = i1;
        x_[i1] = e2;  x_[i2] = e1;
        n_left_ -= 1;
        return true;
    }

//public:
//    void swap_idx( ulong i1, ulong i2 )
//    {
//        if ( i1 >= n_left_ )  { return false; }
//        if ( i2 < n_left_ )  { return false; }
//        const ulong e1 = x_[i1];  // element at i1
//        const ulong e2 = x_[i2];  // element at i2
//        xi_[e1] = i2;  xi_[e2] = i1;
//        x_[i1] = e2;  x_[i2] = e1;
//        return true;
//    }

public:
    bool move_first_left( ulong & e )
    // Move element at position n_left_ to left.
    // Done by simply incrementing n_left_.
    // Write moved element to e.
    // Inverse operation is move_elem_right( e ).
    {
        if ( n_left_ == n_ )  { return false; }
//        move_idx_left( n_left_ );  // nop
        e = x_[ n_left_ ];
        n_left_ += 1;
        return true;
    }

    bool move_last_right( ulong & e )
    // Move element at position n_left_ - 1 to right.
    // Done by simply decrementing n_left_.
    // Write moved element to e.
    // Inverse operation is move_elem_left( e ).
    {
        if ( n_left_ == 0 )  { return false; }
//        move_idx_right( n_left_ - 1 );  // nop
        n_left_ -= 1;
        e = x_[ n_left_ ];
        return true;
    }

public:
    void print()  const
    {
        ulong j;
        for (j=0; j<n_left_; ++j)  { cout << " " << x_[j]; }
        cout << " :";
        for (   ; j<n_; ++j)  { cout << " " << x_[j]; }
        cout << "  " << num_left() << ":" << num_right();
        cout << "\n";
    }

public:
    bool OK()  const
    {
        if ( n_left_ > n_ )  { return false; }
        for (ulong j=0; j<n_; ++j)
            if ( j != xi_[x_[j]] )  { return false; }
        return true;
    }
};
// -------------------------

#endif  // !defined HAVE_DICHOTOMY_H__
