// This file is part of the FXT library.
// Copyright (C) 2010, 2012, 2018 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "dctdst/dctdst.h"
#include "fht/fht-default.h"

#include "aux0/sincos.h"
#include "perm/ziprev.h"

#include "fxttypes.h"  // ulong

#include <cmath>  // sqrt()

void
dcth(double *x, ulong ldn, double *tmp/*=nullptr*/)
// transform wrt. basis:
//   cos( 2*Pi * 0.5 * k * (i+0.5)/n ) * (k==0?1.0:sqrt(2))
//   where k is the index of the basis
//
// the basis:
//  0: [* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *] ( 0)
//  1: [* * * * * * * * * * * * * * * *                                ] ( 1)
//  2: [* * * * * * * *                                 * * * * * * * *] ( 2)
//  3: [* * * * *                       * * * * * * * * * * *          ] ( 3)
//  4: [* * * *                 * * * * * * * *                 * * * *] ( 4)
//  5: [* * *               * * * * * *             * * * * * * *      ] ( 5)
//  6: [* * *           * * * * *             * * * * *           * * *] ( 6)
//  7: [* *           * * * *           * * * * *         * * * * *    ] ( 7)
//  8: [* *         * * * *         * * * *         * * * *         * *] ( 8)
//  9: [* *       * * * *       * * * *         * * *         * * *    ] ( 9)
// 10: [* *       * * *       * * *         * * *       * * *       * *] (10)
// 11: [*       * * *       * * *       * * *       * * *       * * *  ] (11)
// 12: [*       * * *     * * *       * *       * * *     * * *       *] (12)
// 13: [*       * *       * *       * *     * * *     * * *     * * *  ] (13)
// 14: [*     * * *     * *       * *     * *       * *     * * *     *] (14)
// 15: [*     * *     * * *     * *     * *     * *       * *     * *  ] (15)
// 16: [*     * *     * *     * *     * *     * *     * *     * *     *] (16)
// 17: [*     * *     *     * *     * *     * *     * *   * *     * *  ] (17)
// 18: [*     *     * *     * *   * *     * *   * *     * *     *     *] (18)
// 19: [*     *     * *   * *     *     * *   * *     *     * *   * *  ] (19)
// 20: [*   * *     *     *     * *   * *   * *     *     *     * *   *] (20)
// 21: [*   * *   * *   * *   * *   * *     *     *     *     *     *  ] (21)
// 22: [*   * *   * *   *     *     *     *     *     *   * *   * *   *] (22)
// 23: [*   *     *     *   * *   * *   *     *     *   * *   * *   *  ] (23)
// 24: [*   *     *   * *   *     *   * *   *     *   * *   *     *   *] (24)
// 25: [*   *   * *   *     *   *     *   * *   *   * *   *     *   *  ] (25)
// 26: [*   *   * *   *   *     *   *     *   *     *   *   * *   *   *] (26)
// 27: [*   *   *     *   *   *     *   *   * *   *   *   * *   *   *  ] (27)
// 28: [*   *   *   *     *   *   *   * *   *   *   *     *   *   *   *] (28)
// 29: [*   *   *   *   *   * *   *   *   *   *     *   *   *   *   *  ] (29)
// 30: [*   *   *   *   *   *   *   *     *   *   *   *   *   *   *   *] (30)
// 31: [*   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *  ] (31)
//
// H.Malvars algorithm: dct by fht
// ldn := base-2 logarithm of the array length
// tmp := (optional) pointer to scratch space
// if a (size-n) srcatch space is supplied (tmp!=0)
// then the slightly faster version of unzip_rev is used
{
    ulong n = (1UL<<ldn);

    double *y = x;
    if ( tmp )
    {
        y = tmp;
        unzip_rev(x, y, n);
    }
    else  unzip_rev(x, n);

    fht(y, ldn);

    cos_rot(y, x, n);
}
// -------------------------


void
idcth(double *x, ulong ldn, double *tmp/*=nullptr*/)
// inverse transform wrt. basis: cos(k*(i+0.5)*PI/n) * (k==0?1.0:sqrt(2))
// H.Malvars algorithm: idct by fht
// if a (size-n) srcatch space is supplied (tmp!=0)
// then the slightly faster version of zip_rev is used
{
    ulong n = (1UL<<ldn);

    double *y = x;
    if ( tmp )  y = tmp;

    cos_rot(x, y, n);

    fht(y, ldn);

    if ( tmp )  zip_rev(y, x, n);
    else        zip_rev(x, n);
}
// -------------------------


void
dcth_basis(double *f, ulong n, ulong k)
{
    double vv = (k==0 ? 1.0 : sqrt(2.0));
    const double ph = M_PI * (double)k / (double)n;
    for (ulong i=0; i<n; ++i)
    {
//        f[i] = vv * cos( M_PI * k * (i+0.5) / n );
        f[i] = vv * cos( ((double)i+0.5) * ph );
    }
}
// -------------------------


// basis functions produced by the inverse transform of delta peaks:
//
//   ++++++++++++++++++++++++++++++++++++   k = 0
//
//   norm = 1
//                             |---------------------------*             0.25   0
//                             |---------------------------*             0.25   1
//                             |---------------------------*             0.25   2
//                             |---------------------------*             0.25   3
//                             |---------------------------*             0.25   4
//                             |---------------------------*             0.25   5
//                             |---------------------------*             0.25   6
//                             |---------------------------*             0.25   7
//                             |---------------------------*             0.25   8
//                             |---------------------------*             0.25   9
//                             |---------------------------*             0.25   10
//                             |---------------------------*             0.25   11
//                             |---------------------------*             0.25   12
//                             |---------------------------*             0.25   13
//                             |---------------------------*             0.25   14
//                             |---------------------------*             0.25   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 1
//
//   norm = 1
//                              |--------------------------*         0.351851   0
//                              |-------------------------*           0.33833   1
//                              |-----------------------*            0.311806   2
//                              |--------------------*                 0.2733   3
//                              |----------------*                   0.224292   4
//                              |-----------*                        0.166664   5
//                              |------*                             0.102631   6
//                              |-*                                 0.0346543   7
//                           *--|                                  -0.0346543   8
//                      *-------|                                   -0.102631   9
//                 *------------|                                   -0.166664   10
//            *-----------------|                                   -0.224292   11
//        *---------------------|                                     -0.2733   12
//     *------------------------|                                   -0.311806   13
//   *--------------------------|                                    -0.33833   14
//  *---------------------------|                                   -0.351851   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 2
//
//   norm = 1
//                              |--------------------------*          0.34676   0
//                              |----------------------*             0.293969   1
//                              |--------------*                     0.196424   2
//                              |----*                              0.0689748   3
//                        *-----|                                  -0.0689748   4
//              *---------------|                                   -0.196424   5
//      *-----------------------|                                   -0.293969   6
//  *---------------------------|                                    -0.34676   7
//  *---------------------------|                                    -0.34676   8
//      *-----------------------|                                   -0.293969   9
//              *---------------|                                   -0.196424   10
//                        *-----|                                  -0.0689748   11
//                              |----*                              0.0689748   12
//                              |--------------*                     0.196424   13
//                              |----------------------*             0.293969   14
//                              |--------------------------*          0.34676   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 3
//
//   norm = 1
//                              |-------------------------*           0.33833   0
//                              |----------------*                   0.224292   1
//                              |-*                                 0.0346543   2
//                 *------------|                                   -0.166664   3
//     *------------------------|                                   -0.311806   4
//  *---------------------------|                                   -0.351851   5
//        *---------------------|                                     -0.2733   6
//                      *-------|                                   -0.102631   7
//                              |------*                             0.102631   8
//                              |--------------------*                 0.2733   9
//                              |--------------------------*         0.351851   10
//                              |-----------------------*            0.311806   11
//                              |-----------*                        0.166664   12
//                           *--|                                  -0.0346543   13
//            *-----------------|                                   -0.224292   14
//   *--------------------------|                                    -0.33833   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 4
//
//   norm = 1
//                             |---------------------------*         0.326641   0
//                             |-----------*                         0.135299   1
//                  *----------|                                    -0.135299   2
//  *--------------------------|                                    -0.326641   3
//  *--------------------------|                                    -0.326641   4
//                  *----------|                                    -0.135299   5
//                             |-----------*                         0.135299   6
//                             |---------------------------*         0.326641   7
//                             |---------------------------*         0.326641   8
//                             |-----------*                         0.135299   9
//                  *----------|                                    -0.135299   10
//  *--------------------------|                                    -0.326641   11
//  *--------------------------|                                    -0.326641   12
//                  *----------|                                    -0.135299   13
//                             |-----------*                         0.135299   14
//                             |---------------------------*         0.326641   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 5
//
//   norm = 1
//                              |-----------------------*            0.311806   0
//                              |-*                                 0.0346543   1
//        *---------------------|                                     -0.2733   2
//   *--------------------------|                                    -0.33833   3
//                      *-------|                                   -0.102631   4
//                              |----------------*                   0.224292   5
//                              |--------------------------*         0.351851   6
//                              |-----------*                        0.166664   7
//                 *------------|                                   -0.166664   8
//  *---------------------------|                                   -0.351851   9
//            *-----------------|                                   -0.224292   10
//                              |------*                             0.102631   11
//                              |-------------------------*           0.33833   12
//                              |--------------------*                 0.2733   13
//                           *--|                                  -0.0346543   14
//     *------------------------|                                   -0.311806   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 6
//
//   norm = 1
//                              |----------------------*             0.293969   0
//                        *-----|                                  -0.0689748   1
//  *---------------------------|                                    -0.34676   2
//              *---------------|                                   -0.196424   3
//                              |--------------*                     0.196424   4
//                              |--------------------------*          0.34676   5
//                              |----*                              0.0689748   6
//      *-----------------------|                                   -0.293969   7
//      *-----------------------|                                   -0.293969   8
//                              |----*                              0.0689748   9
//                              |--------------------------*          0.34676   10
//                              |--------------*                     0.196424   11
//              *---------------|                                   -0.196424   12
//  *---------------------------|                                    -0.34676   13
//                        *-----|                                  -0.0689748   14
//                              |----------------------*             0.293969   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 7
//
//   norm = 1
//                              |--------------------*                 0.2733   0
//                 *------------|                                   -0.166664   1
//   *--------------------------|                                    -0.33833   2
//                              |-*                                 0.0346543   3
//                              |--------------------------*         0.351851   4
//                              |------*                             0.102631   5
//     *------------------------|                                   -0.311806   6
//            *-----------------|                                   -0.224292   7
//                              |----------------*                   0.224292   8
//                              |-----------------------*            0.311806   9
//                      *-------|                                   -0.102631   10
//  *---------------------------|                                   -0.351851   11
//                           *--|                                  -0.0346543   12
//                              |-------------------------*           0.33833   13
//                              |-----------*                        0.166664   14
//        *---------------------|                                     -0.2733   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 8
//
//   norm = 1
//                             |---------------------------*             0.25   0
//  *--------------------------|                                        -0.25   1
//  *--------------------------|                                        -0.25   2
//                             |---------------------------*             0.25   3
//                             |---------------------------*             0.25   4
//  *--------------------------|                                        -0.25   5
//  *--------------------------|                                        -0.25   6
//                             |---------------------------*             0.25   7
//                             |---------------------------*             0.25   8
//  *--------------------------|                                        -0.25   9
//  *--------------------------|                                        -0.25   10
//                             |---------------------------*             0.25   11
//                             |---------------------------*             0.25   12
//  *--------------------------|                                        -0.25   13
//  *--------------------------|                                        -0.25   14
//                             |---------------------------*             0.25   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 9
//
//   norm = 1
//                              |----------------*                   0.224292   0
//     *------------------------|                                   -0.311806   1
//                      *-------|                                   -0.102631   2
//                              |--------------------------*         0.351851   3
//                           *--|                                  -0.0346543   4
//   *--------------------------|                                    -0.33833   5
//                              |-----------*                        0.166664   6
//                              |--------------------*                 0.2733   7
//        *---------------------|                                     -0.2733   8
//                 *------------|                                   -0.166664   9
//                              |-------------------------*           0.33833   10
//                              |-*                                 0.0346543   11
//  *---------------------------|                                   -0.351851   12
//                              |------*                             0.102631   13
//                              |-----------------------*            0.311806   14
//            *-----------------|                                   -0.224292   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 10
//
//   norm = 1
//                              |--------------*                     0.196424   0
//  *---------------------------|                                    -0.34676   1
//                              |----*                              0.0689748   2
//                              |----------------------*             0.293969   3
//      *-----------------------|                                   -0.293969   4
//                        *-----|                                  -0.0689748   5
//                              |--------------------------*          0.34676   6
//              *---------------|                                   -0.196424   7
//              *---------------|                                   -0.196424   8
//                              |--------------------------*          0.34676   9
//                        *-----|                                  -0.0689748   10
//      *-----------------------|                                   -0.293969   11
//                              |----------------------*             0.293969   12
//                              |----*                              0.0689748   13
//  *---------------------------|                                    -0.34676   14
//                              |--------------*                     0.196424   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 11
//
//   norm = 1
//                              |-----------*                        0.166664   0
//  *---------------------------|                                   -0.351851   1
//                              |----------------*                   0.224292   2
//                              |------*                             0.102631   3
//   *--------------------------|                                    -0.33833   4
//                              |--------------------*                 0.2733   5
//                              |-*                                 0.0346543   6
//     *------------------------|                                   -0.311806   7
//                              |-----------------------*            0.311806   8
//                           *--|                                  -0.0346543   9
//        *---------------------|                                     -0.2733   10
//                              |-------------------------*           0.33833   11
//                      *-------|                                   -0.102631   12
//            *-----------------|                                   -0.224292   13
//                              |--------------------------*         0.351851   14
//                 *------------|                                   -0.166664   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 12
//
//   norm = 1
//                             |-----------*                         0.135299   0
//  *--------------------------|                                    -0.326641   1
//                             |---------------------------*         0.326641   2
//                  *----------|                                    -0.135299   3
//                  *----------|                                    -0.135299   4
//                             |---------------------------*         0.326641   5
//  *--------------------------|                                    -0.326641   6
//                             |-----------*                         0.135299   7
//                             |-----------*                         0.135299   8
//  *--------------------------|                                    -0.326641   9
//                             |---------------------------*         0.326641   10
//                  *----------|                                    -0.135299   11
//                  *----------|                                    -0.135299   12
//                             |---------------------------*         0.326641   13
//  *--------------------------|                                    -0.326641   14
//                             |-----------*                         0.135299   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 13
//
//   norm = 1
//                              |------*                             0.102631   0
//        *---------------------|                                     -0.2733   1
//                              |--------------------------*         0.351851   2
//     *------------------------|                                   -0.311806   3
//                              |-----------*                        0.166664   4
//                              |-*                                 0.0346543   5
//            *-----------------|                                   -0.224292   6
//                              |-------------------------*           0.33833   7
//   *--------------------------|                                    -0.33833   8
//                              |----------------*                   0.224292   9
//                           *--|                                  -0.0346543   10
//                 *------------|                                   -0.166664   11
//                              |-----------------------*            0.311806   12
//  *---------------------------|                                   -0.351851   13
//                              |--------------------*                 0.2733   14
//                      *-------|                                   -0.102631   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 14
//
//   norm = 1
//                              |----*                              0.0689748   0
//              *---------------|                                   -0.196424   1
//                              |----------------------*             0.293969   2
//  *---------------------------|                                    -0.34676   3
//                              |--------------------------*          0.34676   4
//      *-----------------------|                                   -0.293969   5
//                              |--------------*                     0.196424   6
//                        *-----|                                  -0.0689748   7
//                        *-----|                                  -0.0689748   8
//                              |--------------*                     0.196424   9
//      *-----------------------|                                   -0.293969   10
//                              |--------------------------*          0.34676   11
//  *---------------------------|                                    -0.34676   12
//                              |----------------------*             0.293969   13
//              *---------------|                                   -0.196424   14
//                              |----*                              0.0689748   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 15
//
//   norm = 1
//                              |-*                                 0.0346543   0
//                      *-------|                                   -0.102631   1
//                              |-----------*                        0.166664   2
//            *-----------------|                                   -0.224292   3
//                              |--------------------*                 0.2733   4
//     *------------------------|                                   -0.311806   5
//                              |-------------------------*           0.33833   6
//  *---------------------------|                                   -0.351851   7
//                              |--------------------------*         0.351851   8
//   *--------------------------|                                    -0.33833   9
//                              |-----------------------*            0.311806   10
//        *---------------------|                                     -0.2733   11
//                              |----------------*                   0.224292   12
//                 *------------|                                   -0.166664   13
//                              |------*                             0.102631   14
//                           *--|                                  -0.0346543   15
//
//
