
#include "comb/balanced-ordered-tree-lev-seq.h"
#include "comb/word-stats.h"


#include "fxtio.h"
#include "fxttypes.h"
#include "jjassert.h"

#include "nextarg.h"

//% Statistics for (level sequences of) ordered balanced rooted trees.
//% Cf. the following OEIS sequences:
//% A079500 and A007059: all trees.
//% A048887: trees by height.


// Cf. comb/ordered-tree-lev-seq-stats-demo.cc for stats for all ordered trees
// Cf. comb/tree-lev-seq-stats-demo.cc for stats for unordered trees

//#define TIMING  // uncomment to disable printing


int
main(int argc, char **argv)
{
    ulong n = 8;
    NXARG(n, "Number of non-root nodes");

    ulong sq = 0;
    NXARG(sq, "Select stats:\n"
          "    0 ==> by height\n"
          "    1 ==> number of ascents\n"
          "    2 ==> number of descents\n"
          "    3 ==> number of ones\n"
          "    4 ==> number of fixed points\n"
          "    5 ==> number of maximal levels\n"
          "    6 ==> position of last occurrence of one\n"
          "    7 ==> position of first occurrence of the maximal value\n"
//          "    8 ==> position of last occurrence of the maximal value\n"
          "    9 ==> number of flat steps\n"
          "   10 ==> number of even values\n"
          "   11 ==> number of odd values\n"
          "   20 ==> number of flat steps\n"
          "   21 ==> number of non-flat steps\n"
          "   25 ==> number of peaks\n"
//          "   11 ==> number of \n"
          );

    bool aa = false;  // whether to print as ASCII art
    NXARG(aa,"Whether to render trees as ASCII art");

    balanced_ordered_tree_lev_seq T(n);
    const ulong N = T.size();  // == n + 1
    word_stats W(T.data(), N);

    ulong ct = 0;
    ulong * st = new ulong[N+1];  // stats
    for (ulong k=0; k < N+1; ++k)  st[k] = 0;


    ulong j = 0;
    do
    {
        ++ct;
        ulong s = 0;
        switch ( sq )
        {
        case 0:  // by height: A000000, cf. A048887 (transposed), A092921, A175331
            s = W.max_val();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 2, 1, 1, 0,
            // 0, 1, 3, 2, 1, 1, 0,
            // 0, 1, 5, 4, 2, 1, 1, 0,
            // 0, 1, 8, 7, 4, 2, 1, 1, 0,
            // 0, 1, 13, 13, 8, 4, 2, 1, 1, 0,
            // 0, 1, 21, 24, 15, 8, 4, 2, 1, 1, 0,
            // 0, 1, 34, 44, 29, 16, 8, 4, 2, 1, 1, 0,
            // 0, 1, 55, 81, 56, 31, 16, 8, 4, 2, 1, 1, 0,
            // 0, 1, 89, 149, 108, 61, 32, 16, 8, 4, 2, 1, 1, 0,
            // 0, 1, 144, 274, 208, 120, 63, 32, 16, 8, 4, 2, 1, 1, 0,


        case 1:  // by number of ascents: A000000, cf. A186807
            s = W.num_ascents();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 1, 2, 1, 0,
            // 0, 1, 1, 3, 2, 1, 0,
            // 0, 1, 1, 4, 4, 3, 1, 0,
            // 0, 1, 1, 5, 7, 6, 3, 1, 0,
            // 0, 1, 1, 6, 11, 11, 8, 4, 1, 0,
            // 0, 1, 1, 7, 16, 19, 17, 11, 4, 1, 0,
            // 0, 1, 1, 8, 22, 31, 32, 26, 13, 5, 1, 0,
            // 0, 1, 1, 9, 29, 48, 56, 54, 35, 17, 5, 1, 0,
            // 0, 1, 1, 10, 37, 71, 93, 102, 82, 48, 20, 6, 1, 0,
            // 0, 1, 1, 11, 46, 101, 148, 180, 172, 120, 63, 24, 6, 1, 0,


        case 2:  // by number of descents: A000000
            s = W.num_descents();
            break;
            // 1, 0,
            // 1, 0, 0,
            // 2, 0, 0, 0,
            // 3, 0, 0, 0, 0,
            // 4, 1, 0, 0, 0, 0,
            // 5, 3, 0, 0, 0, 0, 0,
            // 6, 7, 1, 0, 0, 0, 0, 0,
            // 7, 13, 4, 0, 0, 0, 0, 0, 0,
            // 8, 22, 12, 1, 0, 0, 0, 0, 0, 0,
            // 9, 34, 29, 5, 0, 0, 0, 0, 0, 0, 0,
            // 10, 50, 61, 18, 1, 0, 0, 0, 0, 0, 0, 0,
            // 11, 70, 116, 53, 6, 0, 0, 0, 0, 0, 0, 0, 0,
            // 12, 95, 205, 134, 25, 1, 0, 0, 0, 0, 0, 0, 0, 0,
            // 13, 125, 341, 302, 86, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,


        case 3:  // by number of ones: A000000
            s = W.num_val(1);
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 2, 0, 1, 0,
            // 0, 3, 1, 0, 1, 0,
            // 0, 5, 2, 0, 0, 1, 0,
            // 0, 8, 4, 1, 0, 0, 1, 0,
            // 0, 14, 6, 3, 0, 0, 0, 1, 0,
            // 0, 24, 11, 6, 1, 0, 0, 0, 1, 0,
            // 0, 43, 18, 11, 4, 0, 0, 0, 0, 1, 0,
            // 0, 77, 33, 18, 10, 1, 0, 0, 0, 0, 1, 0,
            // 0, 140, 60, 30, 20, 5, 0, 0, 0, 0, 0, 1, 0,
            // 0, 256, 112, 51, 36, 15, 1, 0, 0, 0, 0, 0, 1, 0,
            // 0, 472, 210, 90, 60, 35, 6, 0, 0, 0, 0, 0, 0, 1, 0,


        case 4:  // by number of fixed points: A000000, cf. A092921, A175331
            s = W.num_fixed_points();
            break;
            // 0, 1,
            // 0, 0, 1,
            // 0, 0, 1, 1,
            // 0, 0, 1, 1, 1,
            // 0, 0, 1, 2, 1, 1,
            // 0, 0, 1, 3, 2, 1, 1,
            // 0, 0, 1, 5, 4, 2, 1, 1,
            // 0, 0, 1, 8, 7, 4, 2, 1, 1,
            // 0, 0, 1, 13, 13, 8, 4, 2, 1, 1,
            // 0, 0, 1, 21, 24, 15, 8, 4, 2, 1, 1,
            // 0, 0, 1, 34, 44, 29, 16, 8, 4, 2, 1, 1,
            // 0, 0, 1, 55, 81, 56, 31, 16, 8, 4, 2, 1, 1,
            // 0, 0, 1, 89, 149, 108, 61, 32, 16, 8, 4, 2, 1, 1,
            // 0, 0, 1, 144, 274, 208, 120, 63, 32, 16, 8, 4, 2, 1, 1,


        case 5:  // by number of max levels: A184957, cf. A156041
            s = W.num_max_val();
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 2, 1, 1, 0,
            // 0, 1, 2, 3, 1, 1, 0,
            // 0, 1, 3, 4, 4, 1, 1, 0,
            // 0, 1, 3, 6, 7, 5, 1, 1, 0,
            // 0, 1, 4, 8, 11, 11, 6, 1, 1, 0,
            // 0, 1, 4, 11, 17, 19, 16, 7, 1, 1, 0,
            // 0, 1, 5, 13, 26, 32, 31, 22, 8, 1, 1, 0,
            // 0, 1, 5, 17, 35, 54, 56, 48, 29, 9, 1, 1, 0,
            // 0, 1, 6, 20, 48, 82, 102, 93, 71, 37, 10, 1, 1, 0,
            // 0, 1, 6, 24, 63, 120, 172, 180, 148, 101, 46, 11, 1, 1, 0,


        case 6:  // position of last one: A000000
            s = W.last_val_idx(1);
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 2, 0, 1, 0,
            // 0, 3, 0, 1, 1, 0,
            // 0, 5, 0, 1, 1, 1, 0,
            // 0, 8, 0, 1, 2, 2, 1, 0,
            // 0, 14, 0, 1, 2, 3, 3, 1, 0,
            // 0, 24, 0, 1, 3, 4, 5, 5, 1, 0,
            // 0, 43, 0, 1, 4, 5, 6, 9, 8, 1, 0,
            // 0, 77, 0, 1, 6, 7, 9, 11, 15, 13, 1, 0,
            // 0, 140, 0, 1, 9, 11, 12, 16, 19, 26, 21, 1, 0,
            // 0, 256, 0, 1, 14, 17, 19, 23, 28, 34, 45, 34, 1, 0,
            // 0, 472, 0, 1, 22, 28, 30, 36, 40, 51, 60, 78, 55, 1, 0,


        case 7:  // position of first occurrence of max: A092921, cf. A175331
            s = W.first_max_idx();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 2, 1, 1, 0,
            // 0, 1, 3, 2, 1, 1, 0,
            // 0, 1, 5, 4, 2, 1, 1, 0,
            // 0, 1, 8, 7, 4, 2, 1, 1, 0,
            // 0, 1, 13, 13, 8, 4, 2, 1, 1, 0,
            // 0, 1, 21, 24, 15, 8, 4, 2, 1, 1, 0,
            // 0, 1, 34, 44, 29, 16, 8, 4, 2, 1, 1, 0,
            // 0, 1, 55, 81, 56, 31, 16, 8, 4, 2, 1, 1, 0,
            // 0, 1, 89, 149, 108, 61, 32, 16, 8, 4, 2, 1, 1, 0,
            // 0, 1, 144, 274, 208, 120, 63, 32, 16, 8, 4, 2, 1, 1, 0,


//        case 8:  // position of last occurrence of max: trivially at end
//            s = W.last_max_idx();
//            break;
//            // 1, 0,
//            // 0, 1, 0,
//            // 0, 0, 2, 0,
//            // 0, 0, 0, 3, 0,


        case 9:  // number of flat steps: A000000
            s = W.num_flat_steps();
            break;
            // 1, 0,
            // 1, 0, 0,
            // 1, 1, 0, 0,
            // 1, 1, 1, 0, 0,
            // 2, 1, 1, 1, 0, 0,
            // 2, 3, 1, 1, 1, 0, 0,
            // 4, 3, 4, 1, 1, 1, 0, 0,
            // 4, 8, 4, 5, 1, 1, 1, 0, 0,
            // 8, 8, 13, 5, 6, 1, 1, 1, 0, 0,
            // 9, 20, 13, 19, 6, 7, 1, 1, 1, 0, 0,
            // 16, 23, 38, 19, 26, 7, 8, 1, 1, 1, 0, 0,
            // 21, 48, 44, 63, 26, 34, 8, 9, 1, 1, 1, 0, 0,
            // 34, 66, 104, 73, 96, 34, 43, 9, 10, 1, 1, 1, 0, 0,
            // 48, 119, 146, 192, 111, 138, 43, 53, 10, 11, 1, 1, 1, 0, 0,


        case 10:  // number of even values: A000000
            s = W.num_even_val();
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 1, 2, 1, 0,
            // 0, 1, 1, 2, 3, 1, 0,
            // 0, 1, 1, 4, 3, 4, 1, 0,
            // 0, 1, 1, 6, 3, 7, 5, 1, 0,
            // 0, 1, 1, 8, 9, 5, 12, 6, 1, 0,
            // 0, 1, 1, 10, 19, 5, 15, 18, 7, 1, 0,
            // 0, 1, 1, 12, 33, 19, 8, 32, 25, 8, 1, 0,
            // 0, 1, 1, 14, 51, 49, 8, 32, 57, 33, 9, 1, 0,
            // 0, 1, 1, 16, 73, 103, 38, 14, 82, 91, 42, 10, 1, 0,
            // 0, 1, 1, 18, 99, 189, 116, 14, 67, 170, 135, 52, 11, 1, 0,


        case 11:  // number of odd values: A000000
            s = W.num_odd_val();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 2, 1, 1, 0,
            // 0, 1, 3, 2, 1, 1, 0,
            // 0, 1, 4, 3, 4, 1, 1, 0,
            // 0, 1, 5, 7, 3, 6, 1, 1, 0,
            // 0, 1, 6, 12, 5, 9, 8, 1, 1, 0,
            // 0, 1, 7, 18, 15, 5, 19, 10, 1, 1, 0,
            // 0, 1, 8, 25, 32, 8, 19, 33, 12, 1, 1, 0,
            // 0, 1, 9, 33, 57, 32, 8, 49, 51, 14, 1, 1, 0,
            // 0, 1, 10, 42, 91, 82, 14, 38, 103, 73, 16, 1, 1, 0,
            // 0, 1, 11, 52, 135, 170, 67, 14, 116, 189, 99, 18, 1, 1, 0,


        case 20: // number of flat steps: A000000
            s = W.num_flat_steps();
            break;
            // 1, 0,
            // 1, 0, 0,
            // 1, 1, 0, 0,
            // 1, 1, 1, 0, 0,
            // 2, 1, 1, 1, 0, 0,
            // 2, 3, 1, 1, 1, 0, 0,
            // 4, 3, 4, 1, 1, 1, 0, 0,
            // 4, 8, 4, 5, 1, 1, 1, 0, 0,
            // 8, 8, 13, 5, 6, 1, 1, 1, 0, 0,
            // 9, 20, 13, 19, 6, 7, 1, 1, 1, 0, 0,
            // 16, 23, 38, 19, 26, 7, 8, 1, 1, 1, 0, 0,
            // 21, 48, 44, 63, 26, 34, 8, 9, 1, 1, 1, 0, 0,
            // 34, 66, 104, 73, 96, 34, 43, 9, 10, 1, 1, 1, 0, 0,
            // 48, 119, 146, 192, 111, 138, 43, 53, 10, 11, 1, 1, 1, 0, 0,


        case 21: // number of non-flat steps: A000000
            s = W.num_nonflat_steps();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 1, 1, 2, 0,
            // 0, 1, 1, 1, 3, 2, 0,
            // 0, 1, 1, 1, 4, 3, 4, 0,
            // 0, 1, 1, 1, 5, 4, 8, 4, 0,
            // 0, 1, 1, 1, 6, 5, 13, 8, 8, 0,
            // 0, 1, 1, 1, 7, 6, 19, 13, 20, 9, 0,
            // 0, 1, 1, 1, 8, 7, 26, 19, 38, 23, 16, 0,
            // 0, 1, 1, 1, 9, 8, 34, 26, 63, 44, 48, 21, 0,
            // 0, 1, 1, 1, 10, 9, 43, 34, 96, 73, 104, 66, 34, 0,
            // 0, 1, 1, 1, 11, 10, 53, 43, 138, 111, 192, 146, 119, 48, 0,


        case 25: // number of peaks: A000000
            s = W.num_peaks();
            break;
            // 1, 0,
            // 1, 0, 0,
            // 2, 0, 0, 0,
            // 3, 0, 0, 0, 0,
            // 4, 1, 0, 0, 0, 0,
            // 6, 2, 0, 0, 0, 0, 0,
            // 9, 4, 1, 0, 0, 0, 0, 0,
            // 14, 8, 2, 0, 0, 0, 0, 0, 0,
            // 22, 15, 5, 1, 0, 0, 0, 0, 0, 0,
            // 35, 28, 12, 2, 0, 0, 0, 0, 0, 0, 0,
            // 56, 53, 24, 6, 1, 0, 0, 0, 0, 0, 0, 0,
            // 91, 96, 50, 17, 2, 0, 0, 0, 0, 0, 0, 0, 0,
            // 148, 174, 106, 36, 7, 1, 0, 0, 0, 0, 0, 0, 0, 0,
            // 242, 317, 209, 81, 23, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,


        case 30: // max flat length: A000000
            s = W.max_flat_length();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 2, 1, 1, 1, 0,
            // 0, 2, 3, 1, 1, 1, 0,
            // 0, 4, 4, 3, 1, 1, 1, 0,
            // 0, 4, 9, 5, 3, 1, 1, 1, 0,
            // 0, 8, 13, 11, 5, 3, 1, 1, 1, 0,
            // 0, 9, 26, 19, 12, 5, 3, 1, 1, 1, 0,
            // 0, 16, 42, 38, 21, 12, 5, 3, 1, 1, 1, 0,
            // 0, 21, 76, 70, 44, 22, 12, 5, 3, 1, 1, 1, 0,
            // 0, 34, 131, 133, 83, 46, 22, 12, 5, 3, 1, 1, 1, 0,
            // 0, 48, 232, 249, 164, 89, 47, 22, 12, 5, 3, 1, 1, 1, 0,


        case 40: // max frequency in histogram (of level seq.): same as number of max level
            s = W.max_num_vals();
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 2, 1, 1, 0,
            // 0, 1, 2, 3, 1, 1, 0,
            // 0, 1, 3, 4, 4, 1, 1, 0,
            // 0, 1, 3, 6, 7, 5, 1, 1, 0,
            // 0, 1, 4, 8, 11, 11, 6, 1, 1, 0,
            // 0, 1, 4, 11, 17, 19, 16, 7, 1, 1, 0,
            // 0, 1, 5, 13, 26, 32, 31, 22, 8, 1, 1, 0,
            // 0, 1, 5, 17, 35, 54, 56, 48, 29, 9, 1, 1, 0,
            // 0, 1, 6, 20, 48, 82, 102, 93, 71, 37, 10, 1, 1, 0,
            // 0, 1, 6, 24, 63, 120, 172, 180, 148, 101, 46, 11, 1, 1, 0,


        case 50:  // max branching number (out-degree): A000000
            s = T.max_branching_number();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 1, 1, 0,
            // 0, 1, 2, 1, 1, 0,
            // 0, 1, 4, 1, 1, 1, 0,
            // 0, 1, 6, 4, 1, 1, 1, 0,
            // 0, 1, 8, 9, 3, 1, 1, 1, 0,
            // 0, 1, 12, 18, 6, 3, 1, 1, 1, 0,
            // 0, 1, 20, 28, 17, 5, 3, 1, 1, 1, 0,
            // 0, 1, 31, 48, 36, 13, 5, 3, 1, 1, 1, 0,
            // 0, 1, 47, 82, 75, 28, 12, 5, 3, 1, 1, 1, 0,
            // 0, 1, 70, 152, 136, 67, 23, 12, 5, 3, 1, 1, 1, 0,
            // 0, 1, 102, 285, 242, 145, 54, 22, 12, 5, 3, 1, 1, 1, 0,


        case 51:  // min branching number (out-degree) of internal nodes: A000000
            s = T.min_branching_number();
            // columns: A000000 (not: A018176)
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 2, 0, 1, 0,
            // 0, 4, 0, 0, 1, 0,
            // 0, 7, 0, 0, 0, 1, 0,
            // 0, 12, 1, 0, 0, 0, 1, 0,
            // 0, 21, 2, 0, 0, 0, 0, 1, 0,
            // 0, 39, 3, 0, 0, 0, 0, 0, 1, 0,
            // 0, 71, 5, 0, 0, 0, 0, 0, 0, 1, 0,
            // 0, 131, 8, 0, 0, 0, 0, 0, 0, 0, 1, 0,
            // 0, 243, 12, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
            // 0, 453, 17, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
            // 0, 846, 24, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,


        case 60:  // number of leaves: A000000
            s = T.num_leaves();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 2, 0, 0,
            // 0, 2, 1, 0, 0,
            // 0, 2, 2, 1, 0, 0,
            // 0, 2, 3, 2, 1, 0, 0,
            // 0, 2, 4, 5, 2, 1, 0, 0,
            // 0, 2, 5, 7, 7, 2, 1, 0, 0,
            // 0, 2, 6, 11, 12, 9, 2, 1, 0, 0,
            // 0, 2, 7, 15, 20, 19, 11, 2, 1, 0, 0,
            // 0, 2, 8, 20, 32, 34, 28, 13, 2, 1, 0, 0,
            // 0, 2, 9, 25, 48, 60, 55, 39, 15, 2, 1, 0, 0,
            // 0, 2, 10, 32, 66, 101, 104, 85, 52, 17, 2, 1, 0, 0,
            // 0, 2, 11, 38, 91, 154, 192, 171, 126, 67, 19, 2, 1, 0, 0,


        case 70:  // min limb length: A000000
            s = T.min_limb_length();
            break;
            // 1, 0,
            // 1, 0, 0,
            // 0, 1, 1, 0,
            // 0, 2, 0, 1, 0,
            // 0, 3, 1, 0, 1, 0,
            // 0, 6, 1, 0, 0, 1, 0,
            // 0, 10, 2, 1, 0, 0, 1, 0,
            // 0, 20, 2, 1, 0, 0, 0, 1, 0,
            // 0, 35, 5, 1, 1, 0, 0, 0, 1, 0,
            // 0, 68, 5, 2, 1, 0, 0, 0, 0, 1, 0,
            // 0, 124, 11, 2, 1, 1, 0, 0, 0, 0, 1, 0,
            // 0, 235, 14, 4, 1, 1, 0, 0, 0, 0, 0, 1, 0,
            // 0, 438, 24, 5, 2, 1, 1, 0, 0, 0, 0, 0, 1, 0,
            // 0, 826, 36, 7, 2, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0,


        case 71:  // max limb length: A000000
            s = T.max_limb_length();
            break;
            // 1, 0,
            // 1, 0, 0,
            // 0, 1, 1, 0,
            // 0, 2, 0, 1, 0,
            // 0, 3, 1, 0, 1, 0,
            // 0, 4, 3, 0, 0, 1, 0,
            // 0, 6, 6, 1, 0, 0, 1, 0,
            // 0, 9, 11, 3, 0, 0, 0, 1, 0,
            // 0, 14, 20, 7, 1, 0, 0, 0, 1, 0,
            // 0, 22, 37, 14, 3, 0, 0, 0, 0, 1, 0,
            // 0, 35, 69, 27, 7, 1, 0, 0, 0, 0, 1, 0,
            // 0, 56, 129, 52, 15, 3, 0, 0, 0, 0, 0, 1, 0,
            // 0, 91, 240, 102, 30, 7, 1, 0, 0, 0, 0, 0, 1, 0,
            // 0, 148, 447, 201, 59, 15, 3, 0, 0, 0, 0, 0, 0, 1, 0,


        default:
            cerr << "Invalid choice (exiting)." << endl;
            return 1;
        }


        jjassert( s <= N );
        st[ s ] += 1;


#ifndef TIMING
        cout << setw(4) << ct << ":";

        // print RGS:
        T.print("  ");

        cout << setw(4) << s;

        if ( aa )
        {
            cout << endl;
            T.print_aa("");
            cout << endl;
        }

        cout << endl;

        //        jjassert( T.OK() );
#endif  // TIMING
    }
    while ( (j=T.next()) );


    ulong sct = 0;
    for (ulong k=0; k<=N; ++k)
    {
        cout << st[k] << ", ";
        sct += st[k];
    }
    cout << endl;

    cout << " ct=" << ct;  // total: OEIS sequence A000108
    cout << endl;

    jjassert( sct == ct );

    delete [] st;

    return 0;
}
// -------------------------

/*

  Print triangle for stats q:


  OEIS example:
  q=0; for n in $(seq 0 11) ; do ./bin $n $q | grep ', $' ; done | nl -v0 -s':  ' -w2 -n rz

  C++ comment:
q=0;  for n in $(seq 0 11) ; do ./bin $n $q | grep ', $' ; done | sed 's/^/\/\/ /; s/ $//;'

Extract column c from stats q:
q=0;  c=0;  echo $(for n in $(seq 0 11) ; do ./bin $n $q | grep ', $' ; done | cut -d, -f$((c+1))) | sed 's/ /, /g;'

*/


/// Emacs:
/// Local Variables:
/// MyRelDir: "demo/comb"
/// makefile-dir: "../../"
/// make-target: "1demo DSRC=demo/comb/balanced-ordered-tree-lev-seq-stats-demo.cc"
/// make-target2: "1demo DSRC=demo/comb/balanced-ordered-tree-lev-seq-stats-demo.cc DEMOFLAGS=-DTIMING"
/// End:
