/*
 *  Copyright (c) 2018-present, Facebook, Inc.
 *  All rights reserved.
 *
 *  This source code is licensed under the BSD-style license found in the
 *  LICENSE file in the root directory of this source tree.
 */

#include <fizz/backend/openssl/OpenSSL.h>
#include <fizz/crypto/hpke/Hpke.h>
#include <fizz/crypto/hpke/Utils.h>
#include <fizz/crypto/hpke/test/Mocks.h>
#include <fizz/crypto/test/TestUtil.h>
#include <gtest/gtest.h>
#include <list>

using namespace fizz::test;

namespace fizz {
namespace hpke {
namespace test {

struct Params {
  Mode mode;
  NamedGroup group;
  HashFunction hash;
  CipherSuite suite;
  std::string sharedSecret;
  std::string info;
  // Key pair used for encryption
  std::string skE;
  std::string pkE;
  // Key pair used for decryption
  std::string skR;
  std::string pkR;
  // Sender keys for auth (Optional)
  std::string skS;
  std::string pkS;
  // PSKs (Optional)
  std::string psk;
  std::string pskId;
  // Expected traffic key
  std::string key;
  std::string iv;
  // Encryption/decryption
  std::string ciphertext;
  // Test exports
  std::string exporterSecret;
  std::vector<std::string> exportValues;
};

/***
 * Test vectors sourced from IETF's HPKE draft.
 * https://raw.githubusercontent.com/cfrg/draft-irtf-cfrg-hpke/5f503c564da00b0687b3de75f1dfbdfc4079ad31/test-vectors.json
 */
// clang-format off
static const std::list<Params> HPKETestCases = {
  Params{
      Mode::Base,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "fe0e18c9f024ce43799ae393c7e8fe8fce9d218875e8227b0187c04e7d2ea1fc",
      "4f6465206f6e2061204772656369616e2055726e",
      "52c4a758a802cd8b936eceea314432798d5baf2d7e9235dc084ab1b9cfa2f736",
      "37fda3567bdbd628e88668c3c8d7e97d1d1253b6d4ea6d44c150f741f1bf4431",
      "4612c550263fc8ad58375df3f557aac531d26850903e55a9f23f21d8534e8ac8",
      "3948cfe0ad1ddb695d780e59077195da6c56506b027329794ab02bca80815c4d",
      "",
      "",
      "",
      "",
      "4531685d41d65f03dc48f6b8302c05b0",
      "56d890e5accaaf011cff4b7d",
      "f938558b5d72f1a23810b4be2ab4f84331acc02fc97babc53a52ae8218a355a96d8770ac83d07bea87e13c512a",
      "45ff1c2e220db587171952c0592d5f5ebe103f1561a2614e38f2ffd47e99e3f8",
      {
          "3853fe2b4035195a573ffc53856e77058e15d9ea064de3e59f4961d0095250ee",
          "2e8f0b54673c7029649d4eb9d5e33bf1872cf76d623ff164ac185da9e88c21a5",
          "e9e43065102c3836401bed8c3c3c75ae46be1639869391d62c61f1ec7af54931",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "727699f009ffe3c076315019c69648366b69171439bd7dd0807743bde76986cd",
      "4f6465206f6e2061204772656369616e2055726e",
      "463426a9ffb42bb17dbe6044b9abd1d4e4d95f9041cef0e99d7824eef2b6f588",
      "0ad0950d9fb9588e59690b74f1237ecdf1d775cd60be2eca57af5a4b0471c91b",
      "c5eb01eb457fe6c6f57577c5413b931550a162c71a03ac8d196babbd4e5ce0fd",
      "9fed7e8c17387560e92cc6462a68049657246a09bfa8ade7aefe589672016366",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "15026dba546e3ae05836fc7de5a7bb26",
      "9518635eba129d5ce0914555",
      "e52c6fed7f758d0cf7145689f21bc1be6ec9ea097fef4e959440012f4feb73fb611b946199e681f4cfc34db8ea",
      "3d76025dbbedc49448ec3f9080a1abab6b06e91c0b11ad23c912f043a0ee7655",
      {
          "dff17af354c8b41673567db6259fd6029967b4e1aad13023c2ae5df8f4f43bf6",
          "6a847261d8207fe596befb52928463881ab493da345b10e1dcc645e3b94e2d95",
          "8aff52b45a1be3a734bc7a41e20b4e055ad4c4d22104b0c20285a7c4302401cd",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "2d6db4cf719dc7293fcbf3fa64690708e44e2bebc81f84608677958c0d4448a7",
      "4f6465206f6e2061204772656369616e2055726e",
      "ff4442ef24fbc3c1ff86375b0be1e77e88a0de1e79b30896d73411c5ff4c3518",
      "23fb952571a14a25e3d678140cd0e5eb47a0961bb18afcf85896e5453c312e76",
      "fdea67cf831f1ca98d8e27b1f6abeb5b7745e9d35348b80fa407ff6958f9137e",
      "1632d5c2f71c2b38d0a8fcc359355200caa8b1ffdf28618080466c909cb69b2e",
      "dc4a146313cce60a278a5323d321f051c5707e9c45ba21a3479fecdf76fc69dd",
      "8b0c70873dc5aecb7f9ee4e62406a397b350e57012be45cf53b7105ae731790b",
      "",
      "",
      "b062cb2c4dd4bca0ad7c7a12bbc341e6",
      "a1bc314c1942ade7051ffed0",
      "5fd92cc9d46dbf8943e72a07e42f363ed5f721212cd90bcfd072bfd9f44e06b80fd17824947496e21b680c141b",
      "ee1a093e6e1c393c162ea98fdf20560c75909653550540a2700511b65c88c6f1",
      {
          "28c70088017d70c896a8420f04702c5a321d9cbf0279fba899b59e51bac72c85",
          "25dfc004b0892be1888c3914977aa9c9bbaf2c7471708a49e1195af48a6f29ce",
          "5a0131813abc9a522cad678eb6bafaabc43389934adb8097d23c5ff68059eb64",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "f9d0e870aba28d04709b2680cb8185466c6a6ff1d6e9d1091d5bf5e10ce3a577",
      "4f6465206f6e2061204772656369616e2055726e",
      "14de82a5897b613616a00c39b87429df35bc2b426bcfd73febcb45e903490768",
      "820818d3c23993492cc5623ab437a48a0a7ca3e9639c140fe1e33811eb844b7c",
      "cb29a95649dc5656c2d054c1aa0d3df0493155e9d5da6d7e344ed8b6a64a9423",
      "1d11a3cd247ae48e901939659bd4d79b6b959e1f3e7d66663fbc9412dd4e0976",
      "fc1c87d2f3832adb178b431fce2ac77c7ca2fd680f3406c77b5ecdf818b119f4",
      "2bfb2eb18fcad1af0e4f99142a1c474ae74e21b9425fc5c589382c69b50cc57e",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "1364ead92c47aa7becfa95203037b19a",
      "99d8b5c54669807e9fc70df1",
      "a84c64df1e11d8fd11450039d4fe64ff0c8a99fca0bd72c2d4c3e0400bc14a40f27e45e141a24001697737533e",
      "f048d55eacbf60f9c6154bd4021774d1075ebf963c6adc71fa846f183ab2dde6",
      {
          "08f7e20644bb9b8af54ad66d2067457c5f9fcb2a23d9f6cb4445c0797b330067",
          "52e51ff7d436557ced5265ff8b94ce69cf7583f49cdb374e6aad801fc063b010",
          "a30c20370c026bbea4dca51cb63761695132d342bae33a6a11527d3e7679436d",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "3101c54c3a4f87439eaac080699ed9bbcc726ffe44e860c0424ccb7e3e2ead7b",
      "4f6465206f6e2061204772656369616e2055726e",
      "179d4b53b6365c45b600c4163b61d95cbc2f4d9e36f1695558dce265ab8bab11",
      "6c93e09869df3402d7bf231bf540fadd35cd56be14f97178f0954db94b7fc256",
      "497b4502664cfea5d5af0b39934dac72242a74f8480451e1aee7d6a53320333d",
      "430f4b9859665145a6b1ba274024487bd66f03a2dd577d7753c68d7d7d00c00c",
      "",
      "",
      "",
      "",
      "f50b0609186798729ed0564b36ef2ef8044f1f9d05636874d1f46c819c7a669f",
      "151d9929e2449747889bc923",
      "e5d84cd531cfb583096e7cfa9641bd3079cf3a91cda813c52deb5f512be9931980a41de125a925cdad859d5b7a",
      "86017151bbff6a1940e8abae2ac9e0e7032e33df1eaaecc02ca6259b130d62df",
      {
          "ded6cffafaea6b812cbf3e241e88332adbc077aca81512914213810ee291770a",
          "04d3cb6cc116b28ffd22ad5bc276c60d31fec71ceb87ae24db811c64b7507339",
          "7c5ded445732c14fe09727d29b4251c0fd38455fe8440571e687f0886aac94d2",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "cb095862cd41f4cb5be5f63e11d17728c84b4d0f66ebe6bcb1ed0ce8d895aa1d",
      "4f6465206f6e2061204772656369616e2055726e",
      "e24413c8dc5760ffbedbfbfb48d087f85ae448b62575db480763d430636663af",
      "4f3e44d4dde1d0d12a724242df8cef0a68ea53617dab8a6aade4239d404a5154",
      "d99132243a09c24a7497f3da8608f0ba808c21a575d33679f4b24603e96d27ad",
      "62a61ceb338540516edde460e27923a8df6749bc38e27b1001cd5b8b9102e44c",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "de08a0822c00994ffd1a4136a3caaf2703b4ce0c083c2656e598345fcd27510f",
      "02b1fe14a5b6ad526ccff550",
      "316d9b4214a33182212888e86f23005b0706c30db2b1052c4e28c2c100fcdb85cc934b0a64c8db0d7dd339b64c",
      "8bb2d1661275a9c505481682c41171dcec9d4c468276878d71c98a050bddd53c",
      {
          "c2dccc00e2dda4c34a38e25a9ec1c0a43338b2d3c08ab7a870a978839d64af98",
          "b0eba64b7c69140740872216442aebbfbdbb3c5acfcd394d2272ae8b5694c1a9",
          "83c8f8266bad56783567d44f9cd2a1c0070e1ea179d147e1424622037e7fb61c",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "6579475ca739247fad60b7713b0077f1e966e0eaf6f95bff8fa41e446db4b226",
      "4f6465206f6e2061204772656369616e2055726e",
      "805b278cabd22c9dbd461bf25771703eda4950ed3ef35b369163097899555356",
      "9e59f4b1fa5c876f684765290c34e51145894cc4f244342b9fb1a4bdfd8bb426",
      "47f1eee3670dfaaf27c30a83d06ee9f257af174727c17b35328ef730dfc1cd81",
      "3668d659cec6f338f4f8dc6da6733118d2a633f186a3c1415c895111a8eb7c7d",
      "98fdf9b9773578a79d4ba82fbe483c74cc2e3b8d9525d148a18969fd79a74876",
      "4a91c3d0893433f5e31a79fc520f885527a1bc60bf2b0c72693dd7f0b2e41a5a",
      "",
      "",
      "db0218adcafe73ee2e320bd08146d232cedfbd45c7e43d1fae3f1c79dc179b40",
      "41da94323642095905a34938",
      "10b964283ac2cc0bdc4c85ab617291b446bf3832e9359b2c3a0facc50ea75a3c1afd08aeaacd6041d02eb560ec",
      "ca56d3b4d84d60bc3cd4a0749adeb578ff9c19c9d49a5848632c23c5c912c5ea",
      {
          "8890c5615e5d6b0e1b212e26d80a7e8c0d03e796377f09e9377aa0497ccf89c9",
          "51f60f1d4505688a1aca99c9b789e44f38a5bfa177a6b4660ff57114bf50c6be",
          "25f7c731201fe73978b5c66405f17de3e59b7f1c4bbe21e9ff57541d152841ac",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "40d16ac46fa9b4c4c02937e106ecb5a67109ae60ebb66262cfc704880d907d58",
      "4f6465206f6e2061204772656369616e2055726e",
      "489982fb92e71f638c2957a971f4d635af14d725481bbf4db187006600a26557",
      "d38af616e071a4e3717ad1575fc8df781c541b4d0cc02cdf98f2d156a9eda15f",
      "a494cc9d803df57792c866f6ab716ba8ce953236e3ec71914908cd80fb721c15",
      "49823d14040d46e3d405e21f421a810a4968a361bc96c5abcf2f36e66b15a36e",
      "06d5b0b9a559a48588a2447b51f153ef5a03fae0c022c831e64ad85bb3d3ab41",
      "f94a4aad51983c18a48a960f2072c14818b9bf1eac2cc4575e32d8d029387a2e",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "501e5469a0814eb5e6be3c9711d884765835aaec5d15947054aa2b4c5a467efd",
      "1455fb0f644ca05dec2dc40e",
      "49d13e16bc1f0e45805ac211e0c2e6bf5d436ed00df5f02f16c4c8eaeda0418d3f614636e2f026949bbd6dd281",
      "23d5857f167856ec7d9200832e9ae284d046df2d9abf11aef698f3d6b6a2534e",
      {
          "0404bb6afcf9f3a2f8b10e0d2077b7829b5b90d97f799a3ebdefa3772e53137a",
          "b27b4d9756004ad06b8b57e680df80097ea5600796c1bf9235b8c3d9a28515ae",
          "d4a4033268f372ee2725be064512c4de92591f94740efdb1ed4be226c5d4e20f",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "0bbe78490412b4bbea4812666f7916932b828bba79942424abb65244930d69a7",
      "4f6465206f6e2061204772656369616e2055726e",
      "f4ec9b33b792c372c1d2c2063507b684ef925b8c75a42dbcbf57d63ccd381600",
      "1afa08d3dec047a643885163f1180476fa7ddb54c6a8029ea33f95796bf2ac4a",
      "8057991eef8f1f1af18f4a9491d16a1ce333f695d4db8e38da75975c4478e0fb",
      "4310ee97d88cc1f088a5576c77ab0cf5c3ac797f3d95139c6c84b5429c59662a",
      "",
      "",
      "",
      "",
      "ad2744de8e17f4ebba575b3f5f5a8fa1f69c2a07f6e7500bc60ca6e3e3ec1c91",
      "5c4d98150661b848853b547f",
      "1c5250d8034ec2b784ba2cfd69dbdb8af406cfe3ff938e131f0def8c8b60b4db21993c62ce81883d2dd1b51a28",
      "a3b010d4994890e2c6968a36f64470d3c824c8f5029942feb11e7a74b2921922",
      {
          "4bbd6243b8bb54cec311fac9df81841b6fd61f56538a775e7c80a9f40160606e",
          "8c1df14732580e5501b00f82b10a1647b40713191b7c1240ac80e2b68808ba69",
          "5acb09211139c43b3090489a9da433e8a30ee7188ba8b0a9a1ccf0c229283e53",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "4be079c5e77779d0215b3f689595d59e3e9b0455d55662d1f3666ec606e50ea7",
      "4f6465206f6e2061204772656369616e2055726e",
      "0c35fdf49df7aa01cd330049332c40411ebba36e0c718ebc3edf5845795f6321",
      "2261299c3f40a9afc133b969a97f05e95be2c514e54f3de26cbe5644ac735b04",
      "77d114e0212be51cb1d76fa99dd41cfd4d0166b08caa09074430a6c59ef17879",
      "13640af826b722fc04feaa4de2f28fbd5ecc03623b317834e7ff4120dbe73062",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "600d2fdb0313a7e5c86a9ce9221cd95bed069862421744cfb4ab9d7203a9c019",
      "112e0465562045b7368653e7",
      "4a177f9c0d6f15cfdf533fb65bf84aecdc6ab16b8b85b4cf65a370e07fc1d78d28fb073214525276f4a89608ff",
      "73b506dc8b6b4269027f80b0362def5cbb57ee50eed0c2873dac9181f453c5ac",
      {
          "813c1bfc516c99076ae0f466671f0ba5ff244a41699f7b2417e4c59d46d39f40",
          "2745cf3d5bb65c333658732954ee7af49eb895ce77f8022873a62a13c94cb4e1",
          "ad40e3ae14f21c99bfdebc20ae14ab86f4ca2dc9a4799d200f43a25f99fa78ae",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "d2d67828c8bc9fa661cf15a31b3ebf1febe0cafef7abfaaca580aaf6d471e3eb",
      "4f6465206f6e2061204772656369616e2055726e",
      "c94619e1af28971c8fa7957192b7e62a71ca2dcdde0a7cc4a8a9e741d600ab13",
      "f7674cc8cd7baa5872d1f33dbaffe3314239f6197ddf5ded1746760bfc847e0e",
      "3ca22a6d1cda1bb9480949ec5329d3bf0b080ca4c45879c95eddb55c70b80b82",
      "1a478716d63cb2e16786ee93004486dc151e988b34b475043d3e0175bdb01c44",
      "2def0cb58ffcf83d1062dd085c8aceca7f4c0c3fd05912d847b61f3e54121f05",
      "f0f4f9e96c54aeed3f323de8534fffd7e0577e4ce269896716bcb95643c8712b",
      "",
      "",
      "b071fd1136680600eb447a845a967d35e9db20749cdf9ce098bcc4deef4b1356",
      "d20577dff16d7cea2c4bf780",
      "ab1a13c9d4f01a87ec3440dbd756e2677bd2ecf9df0ce7ed73869b98e00c09be111cb9fdf077347aeb88e61bdf",
      "be2d93b82071318cdb88510037cf504344151f2f9b9da8ab48974d40a2251dd7",
      {
          "070cffafd89b67b7f0eeb800235303a223e6ff9d1e774dce8eac585c8688c872",
          "2852e728568d40ddb0edde284d36a4359c56558bb2fb8837cd3d92e46a3a14a8",
          "1df39dc5dd60edcbf5f9ae804e15ada66e885b28ed7929116f768369a3f950ee",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::x25519,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "86a6c0ed17714f11d2951747e660857a5fd7616c933ef03207808b7a7123fe67",
      "4f6465206f6e2061204772656369616e2055726e",
      "5e6dd73e82b856339572b7245d3cbb073a7561c0bee52873490e305cbb710410",
      "656a2e00dc9990fd189e6e473459392df556e9a2758754a09db3f51179a3fc02",
      "7b36a42822e75bf3362dfabbe474b3016236408becb83b859a6909e22803cb0c",
      "a5099431c35c491ec62ca91df1525d6349cb8aa170c51f9581f8627be6334851",
      "90761c5b0a7ef0985ed66687ad708b921d9803d51637c8d1cb72d03ed0f64418",
      "3ac5bd4dd66ff9f2740bef0d6ccb66daa77bff7849d7895182b07fb74d087c45",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "49c7e6d7d2d257aded2a746fe6a9bf12d4de8007c4862b1fdffe8c35fb65054c",
      "abac79931e8c1bcb8a23960a",
      "9aa52e29274fc6172e38a4461361d2342585d3aeec67fb3b721ecd63f059577c7fe886be0ede01456ebc67d597",
      "7c6cc1bb98993cd93e2599322247a58fd41fdecd3db895fb4c5fd8d6bbe606b5",
      {
          "c23ebd4e7a0ad06a5dddf779f65004ce9481069ce0f0e6dd51a04539ddcbd5cd",
          "ed7ff5ca40a3d84561067ebc8e01702bc36cf1eb99d42a92004642b9dfaadd37",
          "d3bae066aa8da27d527d85c040f7dd6ccb60221c902ee36a82f70bcd62a60ee4",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "3b5f8cba3b53c7d4711f5c6a5a0397bda23762e9a6a5319081443372a1c12e66",
      "4f6465206f6e2061204772656369616e2055726e",
      "b2ddee7e705637e56848f7d79722037df28ac5a4343502dd83a896c7133c1713",
      "8998da4c3d6ade83c53e861a022c046db909f1c31107196ab4c2f4dd37e1a949",
      "ddfbb71d7ea8ebd98fa9cc211aa7b535d258fe9ab4a08bc9896af270e35aad35",
      "adf16c696b87995879b27d470d37212f38a58bfe7f84e6d50db638b8f2c22340",
      "",
      "",
      "",
      "",
      "5470dd5c2a9dd27cc3afcc0a22db8b7f",
      "674e489fcfed0d05867cf633",
      "d3a676359d7db814f1f7a12cbe98ab334c834e14d61def40616dfc7e53dc5fc92e1e05d8c8139596dc8e7b04f5",
      "80af20f76b14d0b2a62f6c8f35a8dbfc5daeec7ac991a3cd44296e4f1dcd05b3a03b97c1701629ac5f5408a00244d2c769b83c07462b15ff1146d5a0bf040187",
      {
          "846a732d3dd7d974ec41c3b3dcc871ad2e6bcbd4da9235cb9775ec7278d4aac1",
          "74556ec046a23049f4c9d9ca36aecf195a27a780c53766ceedf81eaa15ea6dad",
          "8b9f09cc299227800f159c64a8026b27538f5be27c33789d511ecc0aaa1ad1ae",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "10a111d8208f53967c18f2ab4d9caf3281c96e31eb329a0318ff7d99e2d11be9",
      "4f6465206f6e2061204772656369616e2055726e",
      "2c8593887c023446e36e9027d2cac5e586c544da87360bdc70b9c794dbf64f18",
      "557f2ad9994ecd48e299947c7a609621bb48a3675f91f93c379c956e82fed744",
      "d16a548d4228623e62db73f4a1b3d1fe7dacdbc3ccaa99df9311afc15f2e7833",
      "a268e077bf5458cf2c1aaf7abc539598b32b7c4d22a9c9db18952b9a7182ed2e",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "c77cd5e8efef3b074662056ced6e4be5",
      "e849f28fc830cc8b4380b6d4",
      "b8a853057198e1d230b5708d9eb9861086a468ddf649e60f3c5d1ca9e50d1bef7be47151bd8c297bda37d4c279",
      "6d0c8d626d3f80e2910dbfd186ae10bf3d47b1c94668c6ba2b6286d048550eff9c6d1235be920142e1bc6994430a0d0e5271694b865dc4735b09778edcdabdc1",
      {
          "18c61daf1df392114311cbdc395fe433537a550dfd6411d4557a6ed0a6368173",
          "95e99529c6992276507e06cb7665b1d8a4af5367bfa0b04b3793200dbc39adf7",
          "456d3bb18092c49437c3f84d4a33f02df323e6494ae1eca4b04f1878015025af",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "8d75921a2cfd345a076ac2dc64dd2af08598322dd3aadb90a43395c13445c654",
      "4f6465206f6e2061204772656369616e2055726e",
      "899bcc666197a9a9629248daaf7b2cae2020f450b42e2aa633a5dab67031c021",
      "cbbf4bf8393f27f04cdbc5e67a449cadc22df22dcf0c14f61d17471c8b49687f",
      "1ea5548fb3412eca9ca9d5165a382bea32877415b12253fb2c594b0cfa4e8197",
      "9144025cd5cf5049cd429d95efefa7e7ba1a896054cdb1d6c93bac79134b1f5f",
      "bee14df75c1654067db5b7551d3ebd0a5e2e18495733639e6a054c91bde97a17",
      "4b65143baa4aaeae70c23e052972ca61467aa42883b1c3ef388821496f120717",
      "",
      "",
      "d9d173d39d6b281a0aec686097a9ebec",
      "8895a6427778c6d6219b1056",
      "4bf8568019638be84f424742a6fa07b29acaa39d0b56f67ab9dceaf5371f49bafccf6294f18da4d32a1a563175",
      "0f22ca936c399d0c4041ff33cfbfac1e7786f4718040afc4a173f866ea09331bf62e6076512f176840ee2d7a42aff59c5af739b9b9bf5423e414e5f168279110",
      {
          "3797c85ceed01733b5fbbd0a6cea8f11f7ab4aefb4b7efa5b0f6533c735be190",
          "9e9f8ba0d531498e8f9caedb9b51edec7285219f526b88a7b7aa5782922a2931",
          "b7f6b8b0755634589c47321fe3996ac102e76b41a0c79c8440b065670de7d044",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "4521e4db04361cb8c86b836ec49a0470f9bb6484bcff7ce27e602dcc956b9404",
      "4f6465206f6e2061204772656369616e2055726e",
      "4a9c54eb2bec2abf51d73b1debfe4c5c77706498ef41ea3d01e05d47002e8dec",
      "d16f9195a7ec9fa5bdae0492d8ba39af16170953cd0e14293b869f19248c511b",
      "2e88db2354b96b778742281a8b7ed4053ca87e5fc7182875d5fce63c34f970f8",
      "8a3ee49d145eeda1ce67c97719d1549ea3db1f6e1ddc08c5a96424cb626af40c",
      "d19c4ac7b0f6b25a86bccaafddc9e3e1e593cb4a54f517a545be8107633ce772",
      "29f9e969591e0dc2871e753bc917199865cd9c4777f5c02fcadc0116d0a26837",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "ca48fc901a9d2b5badb98aac9b63fe04",
      "34846c33e043809eac003484",
      "a0dd42c7babfcb6977040a71f1a387663f9904ac26ea8d8b9f7f42ec1d0c853449776887b76ea0c7a46bb19499",
      "ea7f1197df2007ce693f297e2010a6d81cf070330eab8bbd8bd14072430d14bb81836e26a1a268feea24105122baefb2e024cc89d4d8e5d3a689b6512bfd7e9b",
      {
          "8d720e83a445508d550edb28ddbe643351bfdbc45633ef73567b1fc2d17a8e5d",
          "c49895ffd569e451416e1e749fa19b47e9f8bfca505fc96c281aa95e4be82712",
          "7acb7cff7302ea5c5819fea2f0b69d6ebabc664a17476cb7771af1598eb5c8c6",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "d22ed5c53b896b89c11940993dbc6924a8f0e17f11ca0d095804060bf9909106",
      "4f6465206f6e2061204772656369616e2055726e",
      "5386934a3f61c6cdb2a70b18fb67106d7e7a77c8b4d4126c016a350be0ab3217",
      "473a5c15d5e0b488c7b321e99172e1663be514efe79387ffb1da4a53b806c461",
      "8dc885ddff9915dee8a360309675d770d4c9facb8f214d24f7baf130153e0a1a",
      "740730cdce9e8dab82ca0648a3cc2df40281d4c2166e9f6c3698e6aa666e4930",
      "ac9e7ab12c37daeaa9b2098502a7db2118d536e6b3b9e8385d79a52ee7f71541",
      "99ce50c3f04d367deac454e1c04c662fa2b398ea2fae15d93d163aa07d6dba49",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "d96b2d9043a9b875fc4b2b7079dccd0d6e2c7b431a0517065e73a349b625bb24",
      "7782f07d1ce3bd345b1de3da",
      "2ea1d1a353b0aba7bb38ed44f518adf446e08fc09f0957587ab42c16986ec2c673b0c1b4874b2ef68f1faaa67b",
      "b47dad6405736797e6583defa8ee9adab77fe62c3c0730ed6672a08c63fc10b8bc4fad3cb8c2016358419fc2266afd1856c81e9353baf32b007c5f7bbd55a9e0",
      {
          "243c7c7b1461cd6c8640e728b32ae1a6bf9ab58ffaaa21d3e048bc385dd54008",
          "a0e09de8c298866898cd022934a8c5e3c9cb4b35e483b40fea76518682b822a7",
          "cf3817737cfd63c25ff9fec3541fdc0ed2a7279dfc5cef3cdde9a18648644808",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "96fe0a805d100153533f0646095a652eecb19346db433089666ee539a796ffb2",
      "4f6465206f6e2061204772656369616e2055726e",
      "dc926085fd67a0338320c3b47944b56eec296981d646ab5e3492e3460bebaf51",
      "c12ba9fb91d7ebb03057d8bea4398688dcc1d1d1ff3b97f09b96b9bf89bd1e4a",
      "bdd8943c1e60191f3ea4e69fc4f322aa1086db9650f1f952fdce88395a4bd1af",
      "aa7bddcf5ca0b2c0cf760b5dffc62740a8e761ec572032a809bebc87aaf7575e",
      "",
      "",
      "",
      "",
      "f3354d286a48f67ca0c22029feb446938efb1b9b8a410852d7bdd3404acd0c09",
      "d654f65e557737ea2a0b5489",
      "186cbeffd80fd68862b09d968a944c9f1ecc1c3f5dbcd1e26973ec30a9856f006f7bb472c3e30fff57ced669fc",
      "74536eda135901a81409ab3f8f4767d2cf41933136bbd194427cec8e6fe2253f3ac0beae54180a7837dea9277a3290749777f65a874fdd2ca69c7ef5ee5bbcfe",
      {
          "e0c5b2c8c3af6ea743bf51b48f75d965f5eb71fce668c550863b14b75f61840c",
          "782f53407c273fdd8ffe55fe9540b5c209dcf74beeffb38a807948b354fca3b3",
          "af616a8dc3fa47900b8e68f878fba983134b4b608bcad9c0f743d2aa7c1a781b",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "360d4f9490b0822e944c012ce6dac05f3331a1ae2695a2e64d6f42e3ef63abb9",
      "4f6465206f6e2061204772656369616e2055726e",
      "245b6a48b7cf15a0d89b40b932804edb018b3a6de68e4f3f7c33f64ba3d8d2e6",
      "67867a1c41afa75cbce4f726304adda5062c2793c2e6b307dd0191a204a4db5b",
      "7ef44e93d5b9df2b8c7f7e3bec24a1581b98624a6c0d4f5df9fdb383fbca1750",
      "7891026ecbfe6339d804da654cdd6797e9bedf85f3abc56ae46a693eeef55743",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "0976c6d00ce1f600195b827db4d60232bda81c1f577d1de13e19ad00ebbc38ba",
      "fa603a394e9e6bd93d21cd52",
      "018c929f81250301f7839048f814448a679e94f0e19b944737b54ced9e623e535e5ebc439e6eb49ca00b04883e",
      "348e036205f78026df40a27b87f7e474015a20e5a8e9a828cd396f18aa3fa0e38a943bda9604865ce99481c93c481068f746ab7e87fd9842f2c12b07fc96f29f",
      {
          "75570a8d2eac7404054cd589d70987bbf69a7771a0cdefdc431fc97144085dd8",
          "b637f2a82362259126c2e3f955b3958b03d7c29561b825c79fd1b8f33e0f30a5",
          "202e2a37a076d0e683cdbc27c03eaeeb2d73519eb018d8bdabe467743d1d3bfb",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "039e572d8d6928e925dd19e3400d080dad8e469723897558bdc5694196556787",
      "4f6465206f6e2061204772656369616e2055726e",
      "c38ab7cc90dfb49776bc0f1137eda624e62371bead515cbc93c69000eff747c5",
      "3e276b60dab1aeddce9176e30201795fc7c32736912f670c8f09e1334008a354",
      "6ade1a44d2ee24ca4e44648119ccaf2e2f0de11fee18536f5b5b4ff543f1621c",
      "c05b1ec51b2ddb9f226074582fd6e259cc9ca35e92c73a24c7b5062e2ac3f712",
      "163665f9be4038f7f4b78bf097690ce1820afeca2d7502d6b342c4df9132bcac",
      "80ffae75685b9d176ad0ed7f721c64f3c274b50f5a1b113165c44915db7c5217",
      "",
      "",
      "948cd9484623c2e148e2294619ca39e99ebee2bd59494841458c45b99e09367d",
      "a46aebcafe409e3c97ed0970",
      "3866644bbf36102c2360070942108b1459b725a28c6bd3d4224deff4ae11c04b7bb484cc688395222c0287a010",
      "8534e883089b983739244d4b6dfb5409e7bc8664cde57937b0322d9ddfb0047a92508ebe5932355004dc1050136d52ec5d8c6f47581a16995bb2c05a0188f1b4",
      {
          "53e2ea7a4836acfed06560f2c3e9e4769c64c327ebb8b935dbe48545eae3bac2",
          "d16bdb8c2e89e98f01adb67b812a077be2a70ed601fe41d72fbd566792bb394c",
          "7080e8ab74a5c901cb4556cacb48570737ffb5acdf895c2c9e6e436cf865b773",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "7ca45a4b0fd3491569e88d54471bcc83777566e88b02244493720d412dddd03f",
      "4f6465206f6e2061204772656369616e2055726e",
      "76bb47b1f20139b5506a2f44fd80210e92a6fa32f8ecaf65a42c1e8060c8eb30",
      "1d38fc578d4209ea0ef3ee5f1128ac4876a9549d74dc2d2f46e75942a6188244",
      "fad15f488c09c167bd18d8f48f282e30d944d624c5676742ad820119de44ea91",
      "06aa193a5612d89a1935c33f1fda3109fcdf4b867da4c4507879f184340b0e0e",
      "",
      "",
      "",
      "",
      "855901be1fd77ee5e6ce4a44e74fd553fbf0940d090d3a3fdf913c723b84920d",
      "6a6a5c9d22e9c26961fd202d",
      "72da9627fd7eb3a8b7169c6d97419b80adefca751c6b52b39a2e084d35ce3eb4487aadaca5a9c590e0938c48b9",
      "3d29344e6384990232ec822334a97cb099714e3f778b604e919743010929280f8d1d8cc4fb13093ef6257abf17271097b9d2b9231639e69667a7e0d0fdc05994",
      {
          "5b6120165c82456080db3c730b886b07129e0aec9b5f7beae9e5bbd103c67f2d",
          "30890b81a37b14b818c462ae5b680b4273cdc7a1ce5ca86d30d482fbe4323e7a",
          "b0b5c19ae0daf8d005593f5755d6e8cab29bd3c5c8245823586d009d15aa5237",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "cbd7eeb81ca7cc4b76411df346291e840990b7f059e507b055158575e656ff7b",
      "4f6465206f6e2061204772656369616e2055726e",
      "eca9fe586030d7de742a2fc531685684d9c4cf32ea1af4aba1f85895d9c2f563",
      "bc441a64a700843a8efd5cd574c20e9909c3a2ff7d35e260f9328cbb8e555d56",
      "408882e1f5e554b270a1174ec38e6c647ad1394a408ebafc228c0410dbf98a24",
      "2b54cf0ed6c4ef3ef5c2303a85abd3db8f540a5c53a22f8bf9639921c81a324b",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "a6185e8133becdb0ee3acbc901c6085bd5d5a3e7cce9949c57647a7f81c437e3",
      "f4fee6a6f8e2f5657369f3bc",
      "65a46e483d921343f20cba85da69976b2e0e52f450db7919f7796604977d6708d884a40d5e4fd5b820211264aa",
      "bc3b934f4bba7bf8adb625c8cdf255d8db109aa16ef4a99f180cdd817a0c90e04b857a6a42d669b6f52eb1f2264495b45c827a0bb763656cd199a3bde2b3974f",
      {
          "722aa34bd26f69aa1763f46d7eae6cf461ce74b6952483f3ea7d490c88882982",
          "ea0c03bea28f6a22f5c93c52a999fdbd386572920a2838304e987d6f930d5fa4",
          "3a3980d8a63287c12db540669ded019a0643e236e25896f2f3197edda044b3ce",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "aadac9b340124ae5d0d0793b56fc50a9d3b7699fb44d8e583d4e863dfeacd406",
      "4f6465206f6e2061204772656369616e2055726e",
      "6ccbd501372c8976c2ecb9d69949311a23de77b6dd1cbd917566e28200f2ab8d",
      "331597d5612993d3cad921fc4ba43cef927b0e371b3a2881e6e7c45b10d6ea35",
      "b3e6af7ec768ad8afbf7d4b1686f055dc5607d4dfbfff43ef798ab7eb9225400",
      "f14842fb034d3725cd7c6a2fd86daaa1151b7d3f6e732d42d2fcd6cc90c11617",
      "cec1b09bc81db8f6087e86fe02586b09e5e68166cda9655d5221a7be1528d5e6",
      "679cebc8fe9b8b0e559e938fce8e91d52aa703de6a7b1ffc9ba968f587f08553",
      "",
      "",
      "fd6ef19ab54900b95d3dd5a524c53ee6abf7a2646265ef676c4138d6aad6e3fd",
      "256c397646960f5fe361c7f6",
      "adbd321208ae0bcda6521dcc01a1cd232aaab5b882730de597c580a9b6222d0e6038af6dfe09f3d46a1fdc7f8f",
      "987ba4ffced939f3d55945ff86bfe4beee4461fcfcc4dba0cc00d04b47629b926b255f8ddd15134ac538a1d7d81000f2e04b539ebfbf8e67af35e385ecf38484",
      {
          "2c0f19b5c89412626afe181c1d73655b138d9552b71a1903291d83db49439727",
          "f25f481149e39535f644fce32eff3b1faba30c83515f5c28a65656dda576cfc4",
          "2014260af052a892da042c3c5dd83743826660d84338c1d4bdf36e810fda3c90",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::x25519,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "3b38cd8e6540ef714a0b21a1cd82bb85af3159f1fa0eee44c3361d97e6f84cae",
      "4f6465206f6e2061204772656369616e2055726e",
      "e819eee9d9331740deaa7acb05789f106fb85fa9e5a1888387c715b1df3c8f82",
      "75f842965c219379c24a25dcc7985ef4fa23307de9ec96d8700b1990a907ff3a",
      "d791b71bd90aafed576683312da4f0d6b43bc026e614db1ab99590b5a8394772",
      "6a8e4ccc7a70b66b4682dae9fa35e4e53869e15bde9d21ac100f4efa1c099e6c",
      "5924132e9437a0728d80b8ecb9f0fd4bf9cb1af869deebf98ad125e6e704bd29",
      "50c0cf51b4a336fbc3bfc085112e87a41fc7a43d02795bac17d5348903029833",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "387a1a482c6b659c86f74c6bc5eb6dc67bbefe2a74173674af7279f535286e47",
      "4ec80a1044d5881196f55265",
      "1782237de6ce3dc25dde59dd1aeeb242d99f46a3b625f4ed83875df5ac029785a954f290663eb40913307109dc",
      "50ce7c982b0f0a9b9a986b26124d226202bf18b5182a7116751c0f6fe3b22e9e441bdc9105babfb8b75298fa43b63ffe81d8d833e8158c39345d1f7877a5f2e6",
      {
          "927a9af16036e67245bb2701c1c381be93687eecce24281c5ee23367e7d2c6d8",
          "fdfb03f3a9359ded10ad52954f432481fd1f7e64303be022fd5546972d20cc81",
          "ba08d8f7983e7256dd5b0d2cd9bd341524d70a01c1049696ed41deb507dd91a9",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "2e783ad86a1beae03b5749e0f3f5e9bb19cb7eb382f2fb2dd64c99f15ae0661b",
      "4f6465206f6e2061204772656369616e2055726e",
      "57427244f6cc016cddf1c19c8973b4060aa13579b4c067fd5d93a5d74e32a90f",
      "04305d35563527bce037773d79a13deabed0e8e7cde61eecee403496959e89e4d0ca701726696d1485137ccb5341b3c1c7aaee90a4a02449725e744b1193b53b5f",
      "438d8bcef33b89e0e9ae5eb0957c353c25a94584b0dd59c991372a75b43cb661",
      "040d97419ae99f13007a93996648b2674e5260a8ebd2b822e84899cd52d87446ea394ca76223b76639eccdf00e1967db10ade37db4e7db476261fcc8df97c5ffd1",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "55d9eb9d26911d4c514a990fa8d57048",
      "b595dc6b2d7e2ed23af529b1",
      "90c4deb5b75318530194e4bb62f890b019b1397bbf9d0d6eb918890e1fb2be1ac2603193b60a49c2126b75d0eb",
      "895a723a1eab809804973a53c0ee18ece29b25a7555a4808277ad2651d66d705",
      {
          "a115a59bf4dd8dc49332d6a0093af8efca1bcbfd3627d850173f5c4a55d0c185",
          "4517eaede0669b16aac7c92d5762dd459c301fa10e02237cd5aeb9be969430c4",
          "164e02144d44b607a7722e58b0f4156e67c0c2874d74cf71da6ca48a4cbdc5e0",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "d4aea336439aadf68f9348880aa358086f1480e7c167b6ef15453ba69b94b44f",
      "4f6465206f6e2061204772656369616e2055726e",
      "6b8de0873aed0c1b2d09b8c7ed54cbf24fdf1dfc7a47fa501f918810642d7b91",
      "042224f3ea800f7ec55c03f29fc9865f6ee27004f818fcbdc6dc68932c1e52e15b79e264a98f2c535ef06745f3d308624414153b22c7332bc1e691cb4af4d53454",
      "d929ab4be2e59f6954d6bedd93e638f02d4046cef21115b00cdda2acb2a4440e",
      "04423e363e1cd54ce7b7573110ac121399acbc9ed815fae03b72ffbd4c18b01836835c5a09513f28fc971b7266cfde2e96afe84bb0f266920e82c4f53b36e1a78d",
      "1120ac99fb1fccc1e8230502d245719d1b217fe20505c7648795139d177f0de9",
      "04a817a0902bf28e036d66add5d544cc3a0457eab150f104285df1e293b5c10eef8651213e43d9cd9086c80b309df22cf37609f58c1127f7607e85f210b2804f73",
      "",
      "",
      "19aa8472b3fdc530392b0e54ca17c0f5",
      "b390052d26b67a5b8a8fcaa4",
      "82ffc8c44760db691a07c5627e5fc2c08e7a86979ee79b494a17cc3405446ac2bdb8f265db4a099ed3289ffe19",
      "f152759972660eb0e1db880835abd5de1c39c8e9cd269f6f082ed80e28acb164",
      {
          "837e49c3ff629250c8d80d3c3fb957725ed481e59e2feb57afd9fe9a8c7c4497",
          "594213f9018d614b82007a7021c3135bda7b380da4acd9ab27165c508640dbda",
          "14fe634f95ca0d86e15247cca7de7ba9b73c9b9deb6437e1c832daf7291b79d5",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "d4c27698391db126f1612d9e91a767f10b9b19aa17e1695549203f0df7d9aebe",
      "4f6465206f6e2061204772656369616e2055726e",
      "36f771e411cf9cf72f0701ef2b991ce9743645b472e835fe234fb4d6eb2ff5a0",
      "046a1de3fc26a3d43f4e4ba97dbe24f7e99181136129c48fbe872d4743e2b131357ed4f29a7b317dc22509c7b00991ae990bf65f8b236700c82ab7c11a84511401",
      "bdf4e2e587afdf0930644a0c45053889ebcadeca662d7c755a353d5b4e2a8394",
      "04d824d7e897897c172ac8a9e862e4bd820133b8d090a9b188b8233a64dfbc5f725aa0aa52c8462ab7c9188f1c4872f0c99087a867e8a773a13df48a627058e1b3",
      "b0ed8721db6185435898650f7a677affce925aba7975a582653c4cb13c72d240",
      "049f158c750e55d8d5ad13ede66cf6e79801634b7acadcad72044eac2ae1d0480069133d6488bf73863fa988c4ba8bde1c2e948b761274802b4d8012af4f13af9e",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "4d567121d67fae1227d90e11585988fb",
      "67c9d05330ca21e5116ecda6",
      "b9f36d58d9eb101629a3e5a7b63d2ee4af42b3644209ab37e0a272d44365407db8e655c72e4fa46f4ff81b9246",
      "3f479020ae186788e4dfd4a42a21d24f3faabb224dd4f91c2b2e5e9524ca27b2",
      {
          "595ce0eff405d4b3bb1d08308d70a4e77226ce11766e0a94c4fdb5d90025c978",
          "110472ee0ae328f57ef7332a9886a1992d2c45b9b8d5abc9424ff68630f7d38d",
          "18ee4d001a9d83a4c67e76f88dd747766576cac438723bad0700a910a4d717e6",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "c0d26aeab536609a572b07695d933b589dcf363ff9d93c93adea537aeabb8cb8",
      "4f6465206f6e2061204772656369616e2055726e",
      "4995788ef4b9d6132b249ce59a77281493eb39af373d236a1fe415cb0c2d7beb",
      "04a92719c6195d5085104f469a8b9814d5838ff72b60501e2c4466e5e67b325ac98536d7b61a1af4b78e5b7f951c0900be863c403ce65c9bfcb9382657222d18c4",
      "f3ce7fdae57e1a310d87f1ebbde6f328be0a99cdbcadf4d6589cf29de4b8ffd2",
      "04fe8c19ce0905191ebc298a9245792531f26f0cece2460639e8bc39cb7f706a826a779b4cf969b8a0e539c7f62fb3d30ad6aa8f80e30f1d128aafd68a2ce72ea0",
      "",
      "",
      "",
      "",
      "868c066ef58aae6dc589b6cfdd18f97e",
      "4e0bc5018beba4bf004cca59",
      "5ad590bb8baa577f8619db35a36311226a896e7342a6d836d8b7bcd2f20b6c7f9076ac232e3ab2523f39513434",
      "14ad94af484a7ad3ef40e9f3be99ecc6fa9036df9d4920548424df127ee0d99f",
      {
          "5e9bc3d236e1911d95e65b576a8a86d478fb827e8bdfe77b741b289890490d4d",
          "6cff87658931bda83dc857e6353efe4987a201b849658d9b047aab4cf216e796",
          "d8f1ea7942adbba7412c6d431c62d01371ea476b823eb697e1f6e6cae1dab85a",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "48893fecd82f7c3456af6a42d8f56325d21e08c10fa81299986aaff54cde7b49",
      "4f6465206f6e2061204772656369616e2055726e",
      "90345e3a1d116c1dd39ae76d95ab858c142223a63e44f8f85318cfa91a84858e",
      "04c06b4f6bebc7bb495cb797ab753f911aff80aefb86fd8b6fcc35525f3ab5f03e0b21bd31a86c6048af3cb2d98e0d3bf01da5cc4c39ff5370d331a4f1f7d5a4e0",
      "317f915db7bc629c48fe765587897e01e282d3e8445f79f27f65d031a88082b2",
      "04abc7e49a4c6b3566d77d0304addc6ed0e98512ffccf505e6a8e3eb25c685136f853148544876de76c0f2ef99cdc3a05ccf5ded7860c7c021238f9e2073d2356c",
      "",
      "",
      "",
      "",
      "ee16802a936d5f544771131900ee6973d0551de9e852ece2ef34bf0d5f9e1d1d",
      "9bc50980832a7b4b58c40161",
      "58c61a45059d0c5704560e9d88b564a8b63f1364b8d1fcb3c4c6ddc1d291742465e902cd216f8908da49f8f96f",
      "a8e9a7e62621879fdc89cea7da8e6153458f463e2851baaf009a7461d699cfb6",
      {
          "7a4c2b89e1909fb0e3ca42d5040f4c2d8346dc0643d787b8474e804f8f72798e",
          "3ca0e7e10b601a32edd2f91c49bac766892c52bde2df01a6126320c6e6eb8af1",
          "76c6b4f404990ae362be3efe0d60d9669d87017f9dfe33b8c2ed9fd31d295182",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "aeb4e12a4b956e80588b330a6105a9158b580382427a40dc7c480472dfa346a7",
      "4f6465206f6e2061204772656369616e2055726e",
      "5171dce7db66a978110f345b97bfbdd836338c368d1b819bc125daffd90703db",
      "04a3cd1fd41bb0915973a14325a6c7612b336630e6c2fd3f3ae5a311bfe950d493155f446f3fc4a45d439073e998624fca9490ac7eca4c312271d8720f8e6d7a74",
      "dd70766222d5a88e72c247bd8ad9c28ea49125ee463a63902cc6db68c34f76a6",
      "04349f377dc7fcbb0d52d09e7caa97f53a1badc59aac6959f74a4f5a965f1015d4eeced4cd89f4b3d06c7a716e741d4a9863d8313843c987b96f756b111080f07c",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "2a3c038fe08ade60865e1ff54064471a20dcb4ef90bb692fff3d036f68c03b24",
      "2b272740b827c1e16070c32f",
      "1552f6db424acdef53728dbfab35b85266681af9f9c42fa60e30cc858da8eb1fe05437fea881290cdeaad317d0",
      "b24a488883ad4461ab2b218b48b82063038b5aa6d7d71fbc6612a32539c26fa2",
      {
          "7424d7da93e4b3a2f65b9a0779a827fe764c236ecc201ef4b88475afc692113d",
          "3c42c9b4238f1eeb9272e7fbed204cce2f6f77317d43053cb4241c7856c2e990",
          "86f23bd9b57d6fc2ca1501d9707b83ecb0309f629cfb5a3c8a98a8f0da6d5a0b",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "4b6e403bf494c60342caaa46b3738ee0423892720751607338034b0a067cc1db",
      "4f6465206f6e2061204772656369616e2055726e",
      "7a6cb29fab4e249d1796f95645288a6504d2167c7ff463bc447ab6022462af42",
      "04a7aeac79fda402674ef247c12d6f5fdfd21498d896b67ff04ec181382d4516b7662be32b4a2ae817c2d57104ecb6fcaa527438939810612d1b3d0af36ffc66ce",
      "d9f10996a02cd6c9dbda1d1f225f18f781ea3c893b8c2a6cb2e266e59f3cd9a9",
      "04cd38ef80923e26f157e06c9887f80177c97e1005a41104127271237f946df22eda13d40801bce6184f1a631c44b0807a1a5e8d039975ed0f6079fcbd2dfe6652",
      "6e7b14befe49443dc501def1cc2f0f293d9c5cfa045a23e9a2e0e7703b42705d",
      "04ece9b48cc98ee03ba742fe1218a3fbec960cc34b6e1defdcd3285276f39028e95b90f9526607565888766a1101f429dc3ec87364b5c8c613f0a081881950427f",
      "",
      "",
      "640064834667025be3ce7abf1eb42ccc0dea2db9782b9823519f474e054524e7",
      "29240057274f71e55bfcca28",
      "59b9890aabf94c1d502c39d8d356989ab0880ed43e984255db7b32a8d7b0ad5beba799a4ec326a0ddca3dd5e5d",
      "5b03fe338463543c9d4b195ef8f9c5a914a7503a2a490efc6b6a466f5f85f306",
      {
          "6c0386ae15b1b834a5247ca5595b4e102347cbcdc65de64832f36008ce9c9483",
          "3507f1d3914e96bf72447b5c2d227af2932c7978172085cb826a5ef7f25f74a3",
          "e04a3d5ec48b3729b57b61e02d66eb6f67f4bf013f2767ebd2281592ea3ccef8",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "03d3d0a77139bd73e237854a1a740c8b037101df499e88b1e5af17ccd82b43a6",
      "4f6465206f6e2061204772656369616e2055726e",
      "d593197688dc6d7b5c898368edaf017d625b2099ea76d685303a460a0409e793",
      "04060c9ead3a3787e8e84cfe055a5211c11fc228e661aee80dbe9b0daa76f3915e2a8084284618ff1c18b0cd4af90a6a2f901a09df7b1ba88957b4101c9391607c",
      "711abbbfd2c99aca70eb0f4f057c8bc1d32dfe09409a2d28a8d74da3b85e604d",
      "0436d96b06fc928e8ccebcaf62291265a2fab8c9a0bc27414fcf86ddd8fc47286caabe02a1fe4a9881984ab1abc8475cc5008fddec1eea72082d4854f190982f6f",
      "81dd6b76fe0fdd5871f75ac19c5008f12d6e6963645c02dda572f402d036135c",
      "048387ea40e9944a81e20ae3b8efe7abb3f5b89b1560179f55a8ea40b56a0341c9ef414590f4f9bf1f33a21d6f860c4d428ec2e6309f8bf1ee1816bb5746391491",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "7887c4773caf8a64c4d98505645db1fd7f6e5fcafe520d0f4862ea812442fe2a",
      "9d1500195f9750f4f42e34c4",
      "9b575da82843bf4561f9ba910e533d6991705e4abda231f62b6a3659ce2cdce44fc1240271727a58edc27f4c8d",
      "47f32a7f67c037f2168625ea1569baf4c9f96503e542d232514976a916befcd2",
      {
          "4fb1428cf96d008d0be04dab1c55bfef61d75fb4bd179db6c099113fa779930a",
          "8a005f4b798cee5bfa96f290fb4ab96175a8b1fb73ef464a584c14ae21bc0b3c",
          "a8fa1145e7439b054cf2ab7d45652b684d96fef8a45bbf74741c37f67b086029",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "87584311791036a3019bc36803cdd42e9a8931a98b13c88835f2f8a9036a4fd6",
      "4f6465206f6e2061204772656369616e2055726e",
      "11b7e4de2d919240616a31ab14944cced79bc2372108bb98f6792e3b645fe546",
      "043539917ee26f8ae0aa5f784a387981b13de33124a3cde88b94672030183110f331400115855808244ff0c5b6ca6104483ac95724481d41bdcd9f15b430ad16f6",
      "c29fc577b7e74d525c0043f1c27540a1248e4f2c8d297298e99010a92e94865c",
      "04d383fd920c42d018b9d57fd73a01f1eee480008923f67d35169478e55d2e8817068daf62a06b10e0aad4a9e429fa7f904481be96b79a9c231a33e956c20b81b6",
      "53541bd995f874a67f8bfd8038afa67fd68876801f42ff47d0dc2a4deea067ae",
      "0492cf8c9b144b742fe5a63d9a181a19d416f3ec8705f24308ad316564823c344e018bd7c03a33c926bb271b28ef5bf28c0ca00abff249fee5ef7f33315ff34fdb",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "31e140c8856941315d4067239fdc4ebe077fbf45a6fc78a61e7a6c8b3bacb10a",
      "75838a8010d2e4760254dd56",
      "9eadfa0f954835e7e920ffe56dec6b31a046271cf71fdda55db72926e1d8fae94cc6280fcfabd8db71eaa65c05",
      "600895965755db9c5027f25f039a6e3e506c35b3b7084ce33c4a48d59ee1f0e3",
      {
          "c52b4592cd33dd38b2a3613108ddda28dcf7f03d30f2a09703f758bfa8029c9a",
          "2f03bebc577e5729e148554991787222b5c2a02b77e9b1ac380541f710e5a318",
          "e01dd49e8bfc3d9216abc1be832f0418adf8b47a7b5a330a7436c31e33d765d7",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "806520f82ef0b03c823b7fc524b6b55a088f566b9751b89551c170f4113bd850",
      "4f6465206f6e2061204772656369616e2055726e",
      "7550253e1147aae48839c1f8af80d2770fb7a4c763afe7d0afa7e0f42a5b3689",
      "04c07836a0206e04e31d8ae99bfd549380b072a1b1b82e563c935c095827824fc1559eac6fb9e3c70cd3193968994e7fe9781aa103f5b50e934b5b2f387e381291",
      "a4d1c55836aa30f9b3fbb6ac98d338c877c2867dd3a77396d13f68d3ab150d3b",
      "04a697bffde9405c992883c5c439d6cc358170b51af72812333b015621dc0f40bad9bb726f68a5c013806a790ec716ab8669f84f6b694596c2987cf35baba2a006",
      "",
      "",
      "",
      "",
      "a8f45490a92a3b04d1dbf6cf2c3939ad8bfc9bfcb97c04bffe116730c9dfe3fc",
      "726b4390ed2209809f58c693",
      "6469c41c5c81d3aa85432531ecf6460ec945bde1eb428cb2fedf7a29f5a685b4ccb0d057f03ea2952a27bb458b",
      "4f9bd9b3a8db7d7c3a5b9d44fdc1f6e37d5d77689ade5ec44a7242016e6aa205",
      {
          "9b13c510416ac977b553bf1741018809c246a695f45eff6d3b0356dbefe1e660",
          "6c8b7be3a20a5684edecb4253619d9051ce8583baf850e0cb53c402bdcaf8ebb",
          "477a50d804c7c51941f69b8e32fe8288386ee1a84905fe4938d58972f24ac938",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "ac4f260dce4db6bf45435d9c92c0e11cfdd93743bd3075949975974cc2b3d79e",
      "4f6465206f6e2061204772656369616e2055726e",
      "7d6e4e006cee68af9b3fdd583a0ee8962df9d59fab029997ee3f456cbc857904",
      "04f336578b72ad7932fe867cc4d2d44a718a318037a0ec271163699cee653fa805c1fec955e562663e0c2061bb96a87d78892bff0cc0bad7906c2d998ebe1a7246",
      "12ecde2c8bc2d5d7ed2219c71f27e3943d92b344174436af833337c557c300b3",
      "041eb8f4f20ab72661af369ff3231a733672fa26f385ffb959fd1bae46bfda43ad55e2d573b880831381d9367417f554ce5b2134fbba5235b44db465feffc6189e",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "6d61cb330b7771168c8619498e753f16198aad9566d1f1c6c70e2bc1a1a8b142",
      "0de7655fb65e1cd51a38864e",
      "21433eaff24d7706f3ed5b9b2e709b07230e2b11df1f2b1fe07b3c70d5948a53d6fa5c8bed194020bd9df0877b",
      "754ca00235b245e72d1f722a7718e7145bd113050a2aa3d89586d4cb7514bfdb",
      {
          "530bbc2f68f078dccc89cc371b4f4ade372c9472bafe4601a8432cbb934f528d",
          "6e25075ddcc528c90ef9218f800ca3dfe1b8ff4042de5033133adb8bd54c401d",
          "6f6fbd0d1c7733f796461b3235a856cc34f676fe61ed509dfc18fa16efe6be78",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp256r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "1a45aa4792f4b166bfee7eeab0096c1a6e497480e2261b2a59aad12f2768d469",
      "4f6465206f6e2061204772656369616e2055726e",
      "085fd5d5e6ce6497c79df960cac93710006b76217d8bcfafbd2bb2c20ea03c42",
      "040d5176aedba55bc41709261e9195c5146bb62d783031280775f32e507d79b5cbc5748b6be6359760c73cfe10ca19521af704ca6d91ff32fc0739527b9385d415",
      "3cb2c125b8c5a81d165a333048f5dcae29a2ab2072625adad66dbb0f48689af9",
      "0444f6ee41818d9fe0f8265bffd016b7e2dd3964d610d0f7514244a60dbb7a11ece876bb110a97a2ac6a9542d7344bf7d2bd59345e3e75e497f7416cf38d296233",
      "39b19402e742d48d319d24d68e494daa4492817342e593285944830320912519",
      "04265529a04d4f46ab6fa3af4943774a9f1127821656a75a35fade898a9a1b014f64d874e88cddb24c1c3d79004d3a587db67670ca357ff4fba7e8b56ec013b98b",
      "",
      "",
      "cf292f8a4313280a462ce55cde05b5aa5744fe4ca89a5d81b0146a5eaca8092d",
      "7e45c21e20e869ae00492123",
      "25881f219935eec5ba70d7b421f13c35005734f3e4d959680270f55d71e2f5cb3bd2daced2770bf3d9d4916872",
      "dba6e307f71769ba11e2c687cc19592f9d436da0c81e772d7a8a9fd28e54355f",
      {
          "56c4d6c1d3a46c70fd8f4ecda5d27c70886e348efb51bd5edeaa39ff6ce34389",
          "d2d3e48ed76832b6b3f28fa84be5f11f09533c0e3c71825a34fb0f1320891b51",
          "eb0d312b6263995b4c7761e64b688c215ffd6043ff3bad2368c862784cbe6eff",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "02f584736390fc93f5b4ad039826a3fa08e9911bd1215a3db8e8791ba533cafd",
      "4f6465206f6e2061204772656369616e2055726e",
      "2292bf14bb6e15b8c81a0f45b7a6e93e32d830e48cca702e0affcfb4d07e1b5c",
      "0493ed86735bdfb978cc055c98b45695ad7ce61ce748f4dd63c525a3b8d53a15565c6897888070070c1579db1f86aaa56deb8297e64db7e8924e72866f9a472580",
      "3ac8530ad1b01885960fab38cf3cdc4f7aef121eaa239f222623614b4079fb38",
      "04085aa5b665dc3826f9650ccbcc471be268c8ada866422f739e2d531d4a8818a9466bc6b449357096232919ec4fe9070ccbac4aac30f4a1a53efcf7af90610edd",
      "",
      "",
      "",
      "",
      "090ca96e5f8aa02b69fac360da50ddf9",
      "9c995e621bf9a20c5ca45546",
      "d3cf4984931484a080f74c1bb2a6782700dc1fef9abe8442e44a6f09044c88907200b332003543754eb51917ba",
      "4a7abb2ac43e6553f129b2c5750a7e82d149a76ed56dc342d7bca61e26d494f4855dff0d0165f27ce57756f7f16baca006539bb8e4518987ba610480ac03efa8",
      {
          "a32186b8946f61aeead1c093fe614945f85833b165b28c46bf271abf16b57208",
          "84998b304a0ea2f11809398755f0abd5f9d2c141d1822def79dd15c194803c2a",
          "93fb9411430b2cfa2cf0bed448c46922a5be9beff20e2e621df7e4655852edbc",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "2912aacc6eaebd71ff715ea50f6ef3a6637856b2a4c58ea61e0c3fc159e3bc16",
      "4f6465206f6e2061204772656369616e2055726e",
      "a5901ff7d6931959c2755382ea40a4869b1dec3694ed3b009dda2d77dd488f18",
      "04a307934180ad5287f95525fe5bc6244285d7273c15e061f0f2efb211c35057f3079f6e0abae200992610b25f48b63aacfcb669106ddee8aa023feed301901371",
      "bc6f0b5e22429e5ff47d5969003f3cae0f4fec50e23602e880038364f33b8522",
      "043f5266fba0742db649e1043102b8a5afd114465156719cea90373229aabdd84d7f45dabfc1f55664b888a7e86d594853a6cccdc9b189b57839cbbe3b90b55873",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "0b910ba8d9cfa17e5f50c211cb32839a",
      "0c29e714eb52de5b7415a1b7",
      "57624b6e320d4aba0afd11f548780772932f502e2ba2a8068676b2a0d3b5129a45b9faa88de39e8306da41d4cc",
      "50c0a182b6f94b4c0bd955c4aa20df01f282cc12c43065a0812fe4d4352790171ed2b2c4756ad7f5a730ba336c8f1edd0089d8331192058c385bae39c7cc8b57",
      {
          "8158bea21a6700d37022bb7802866edca30ebf2078273757b656ef7fc2e428cf",
          "6a348ba6e0e72bb3ef22479214a139ef8dac57be34509a61087a12565473da8d",
          "2f6d4f7a18ec48de1ef4469f596aada4afdf6d79b037ed3c07e0118f8723bffc",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "1ed49f6d7ada333d171cd63861a1cb700a1ec4236755a9cd5f9f8f67a2f8e7b3",
      "4f6465206f6e2061204772656369616e2055726e",
      "93cddd5288e7ef4884c8fe321d075df01501b993ff49ffab8184116f39b3c655",
      "04fec59fa9f76f5d0f6c1660bb179cb314ed97953c53a60ab38f8e6ace60fd59178084d0dd66e0f79172992d4ddb2e91172ce24949bcebfff158dcc417f2c6e9c6",
      "1ea4484be482bf25fdb2ed39e6a02ed9156b3e57dfb18dff82e4a048de990236",
      "04378bad519aab406e04d0e5608bcca809c02d6afd2272d4dd03e9357bd0eee8adf84c8deba3155c9cf9506d1d4c8bfefe3cf033a75716cc3cc07295100ec96276",
      "02b266d66919f7b08f42ae0e7d97af4ca98b2dae3043bb7e0740ccadc1957579",
      "0404d3c1f9fca22eb4a6d326125f0814c35593b1da8ea0d11a640730b215a259b9b98a34ad17e21617d19fe1d4fa39a4828bfdb306b729ec51c543caca3b2d9529",
      "",
      "",
      "9d4b1c83129f3de6db95faf3d539dcf1",
      "ea4fd7a485ee5f1f4b62c1b7",
      "2480179d880b5f458154b8bfe3c7e8732332de84aabf06fc440f6b31f169e154157fa9eb44f2fa4d7b38a9236e",
      "ca2410672369aae1afd6c2639f4fe34ca36d35410c090608d2924f60def17f910d7928575434d7f991b1f19d3e8358b8278ff59ced0d5eed4774cec72e12766e",
      {
          "f03fbc82f321a0ab4840e487cb75d07aafd8e6f68485e4f7ff72b2f55ff24ad6",
          "1ce0cadec0a8f060f4b5070c8f8888dcdfefc2e35819df0cd559928a11ff0891",
          "70c405c707102fd0041ea716090753be47d68d238b111d542846bd0d84ba907c",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "02bee8be0dda755846115db45071c0cf59c25722e015bde1c124de849c0fea52",
      "4f6465206f6e2061204772656369616e2055726e",
      "778f2254ae5d661d5c7fca8c4a7495a25bd13f26258e459159f3899df0de76c1",
      "04801740f4b1b35823f7fb2930eac2efc8c4893f34ba111c0bb976e3c7d5dc0aef5a7ef0bf4057949a140285f774f1efc53b3860936b92279a11b68395d898d138",
      "00510a70fde67af487c093234fc4215c1cdec09579c4b30cc8e48cb530414d0e",
      "04a4ca7af2fc2cce48edbf2f1700983e927743a4e85bb5035ad562043e25d9a111cbf6f7385fac55edc5c9d2ca6ed351a5643de95c36748e11dbec98730f4d43e9",
      "d743b20821e6326f7a26684a4beed7088b35e392114480ca9f6c325079dcf10b",
      "04b59a4157a9720eb749c95f842a5e3e8acdccbe834426d405509ac3191e23f2165b5bb1f07a6240dd567703ae75e13182ee0f69fc102145cdb5abf681ff126d60",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "b68bb0e2fbf7431cedb46cc3b6f1fe9e",
      "76af62719d33d39a1cb6be9f",
      "840669634db51e28df54f189329c1b727fd303ae413f003020aff5e26276aaa910fc4296828cb9d862c2fd7d16",
      "7f72308ae68c9a2b3862e686cb547b16d33d00fe482c770c4717d8b54e9b1e547244c3602bdd86d5a788a8443befea0a7658002b23f1c96a62a64986fffc511a",
      {
          "c8c917e137a616d3d4e4c9fcd9c50202f366cb0d37862376bc79f9b72e8a8db9",
          "33a5d4df232777008a06d0684f23bb891cfaef702f653c8601b6ad4d08dddddf",
          "bed80f2e54f1285895c4a3f3b3625e6206f78f1ed329a0cfb5864f7c139b3c6a",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "fcc960a01d9bc0f30605eb29cbd3f9c2b9dab0c7083e88bb266fb17951876376",
      "4f6465206f6e2061204772656369616e2055726e",
      "109449871ed61c0fdd8cecdc56be12fd6f946e13c5c7a863903c592e022904cc",
      "0404dc39344526dbfa728afba96986d575811b5af199c11f821a0e603a4d191b25544a402f25364964b2c129cb417b3c1dab4dfc0854f3084e843f731654392726",
      "9648e8711e9b6cb12dc19abf9da350cf61c3669c017b1db17bb36913b54a051d",
      "0400f209b1bf3b35b405d750ef577d0b2dc81784005d1c67ff4f6d2860d7640ca379e22ac7fa105d94bc195758f4dfc0b82252098a8350c1bfeda8275ce4dd4262",
      "",
      "",
      "",
      "",
      "490666b45bd4aece6eaab989af2e1eb1800ca326955db2be0ce31343c72efc76",
      "ad23d477d0f9ec0c12282360",
      "949f58e87c39b3f55390b6a970de27dfac44aadc2fbc9d623dcde1a08b628c83ad07dbbee6aede7fcfbf955670",
      "073cabf2b9f230a76c75d63051f22c16d257e58d900f85aa650a4ab181bb5c222a43f576894c3bbf7f59a0bb3c435e185d72fbfff459c3310e8a5f7e347dd77e",
      {
          "c9d634be6e873105fc38fae1f86e195a0aa025c5cf1672acd2a358e7e2a84244",
          "d51a7dee4bb7da5e8d6271c5d6755967bbade71c4ceddab1acded3e6e5f642d0",
          "1a677fc144ec3f0df86cfebd6578a0a1a402beeb6f6c36235006369f1211edfa",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "27ad900ec494ed811a9f14087e816cbe85fa0b54bf0a652cad3efcf0802eb44d",
      "4f6465206f6e2061204772656369616e2055726e",
      "859601134f7ee1728c4a56fde416f2967cfea85b9387061d83593c93f7f07a2b",
      "048739ebbaea3156cbd5e39b4ef41ee7e3b52c8cb4958d087112b17b778897152c7e99307095b1cee54b807077f6f5092970a27fbb57ce2835263132c75e52e7e0",
      "564fc2a44c6961fcf0ef8eec0024ef50bcf31f43812114c975e8ffe87c17606f",
      "0480080438469055361f6ba695975ca3f0d14cfd61ae17c4a67886ab44e04ad86db30c5a6d90ea007e7d5ff3625a4c5156a6cfbfaee71da2dccf75ccd944d3039f",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "28b3e9411cd47cda728f7dea88faa449f103f90ca2afebbc5791e315bd355de6",
      "f2a9f537ec6d21162c70efbc",
      "351d83aa6f2ba77c4b9b89aa22fcb18aff3f792bb04e999de9f76f03f99e92c8d9203605cc0dcbb5eb08a9db6b",
      "1fcdfcfacccf116fc8808ce22e8983bcf1121d0a96ca8bae2af6b14ff707fd5c7c3126da658100b4ff8cf756765c4a9ae1b7d22f042a28d876e081aec8f44b58",
      {
          "850caf7336dd83d41fdee7cb133c7c12b62bf7111d3c5d3d60b20128484adada",
          "50121f10b5674e3dc46eed39616ff502ef0d6d7f356783808887a867f6a717c6",
          "32b9b0b8315cfc2415852b21e9353e79c233233f400def9623404e21657bdab5",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "d4e32a68e5e4f00c1eb737975c6d16f4c0d2a7e0406dd13139f39ca95b7ede2c",
      "4f6465206f6e2061204772656369616e2055726e",
      "27d7c7c966f345f7084d81e7a835c26e4ad01afbc41736975ac62a07e7d284f8",
      "048728fc2d342b8eba23e97b31731f85125ff14130829ba01a843d76487d1262fb8f1e67d9fd9f2fbcf8e0399968c21716be6b93c84134ba36b2529803f173c262",
      "1b18d5fa894ff8cc9682a3b540c56a93ed146711f1c7d4a7cf985bc2bf8bd20a",
      "04ba835cdff4e075ba97db2cf705f18471eff67d54039377be8a01fbe93a85bdde3265013c562b977969654d2dbf855b2cbe5950282f8226d94794eefb175bddab",
      "7f209ec8f791935eefe39fdbb2b8b574747c69e9e082660a4fa194f1fac28664",
      "04c5f644ac06da9242231782dca7f0753abb82f909deae17d3ac041a8df848075dd50ece4df6fcd98bafb69441600477c76cacc6cada8d4ca67a6208a7f6e278ce",
      "",
      "",
      "d9d10a6e718b8a230e259b97a7de54690f87d710623379021f60124e53fad1d8",
      "06ab4f04d6a36db110566315",
      "860171a270f1f02f3635047a054241c977878028491fb1dde6bf232e8c21b4e325a53d2f9816195f8563ceab3d",
      "f6e20045902aa6eb6aea9a2c5ce7f839b61cb50392d92db47f57d83de3b18ec6eea8ee547aadc59e1577aed5dd6452c8ed400d1d1fe88afd14f4554ff49da346",
      {
          "35361366275906f48d15493e2f3fbd02955dce15a2c7ef90663dd40ca1c31853",
          "02f1e9c4d41c18669f04d9f8436bbca817e8eac039e799812ec215c51ce94167",
          "7b602374c2ff1d79e029684721f6bdbb53c18c6c8eeab01ff7dc49399893732e",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "7c6c091d7cfff77a14cb4b84c7601d4cdeecce18289367fd399e5740e42063a9",
      "4f6465206f6e2061204772656369616e2055726e",
      "6d3a013f71ac08608e41c5730d32129f9a7bdcf46edfa6ecca7e9f167f87eee6",
      "04634dbc3ecb06a564d7703453b871b113f54e8343f4dcafd15759b56233291f564cc04defb5567534f2649687bb9ea92732ec4c08cadc027bb2637c3c6d43f310",
      "0ef201dfa67e8bebb4ec676766fdc50f491c8478b71d2bafdfa5b78fc9cff590",
      "04ba7f6721d4721645ae7ddc399a22aa28493443188abed3d0461739793134896a6f18f71d9f6b6f97b5e440a58ce863a13a7d230c7b115e26aedd5d5c94f2fd46",
      "4ba160d72272103fb74e880ab0a7c372a5009c910fb3c0914e19cb62e0eaed5c",
      "041ae41c99c53bcbdef12be6caba1ba534568512748bfb77f81a8ba945aad1595f65940f08b62b18de2c27852af6dadb1754225993494dc2d2efc7cc2a0cbda8d5",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "6c172fa64100161b14386d0496b98a350b1d3b8e9ccbda515cd09eacc980b447",
      "37d649d5002b2c68ebb689b8",
      "39b23f5c413932ee827c2214644439973869613c670b169d90b5d4ca304af1fd40a04dbc3cba713bf282ee748c",
      "9f481f55ce166ab296b786dc483b2b1eb2816d455797c31cc7664aa6e4ef494548b031af63cc28fde7efd4689286e888384a9af0fab1419c863133581f6e9958",
      {
          "de509982313c15ffccec8b2b1193632093319d6603f35942f166d025cb687d39",
          "e4d41b307b5ee49b7da8fdb01f1c19c556ce35e90961d8c1dfa36dcbc5da65f7",
          "67fd7c5a4cbf7d12f3f92876913a46578b9cdbca4f8031f61c11424991c20ef5",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "646e82a31c200d31ee3f4d6716fd4a1706b3fe94ccac9bb01a2cc602f04c2428",
      "4f6465206f6e2061204772656369616e2055726e",
      "2b6ea1342943b50473e43a5204b7a7eec0d411fd749c3aa2a235852b91d89610",
      "049fafd3c13356c526754bf9ac57d2875fb04814ff0feb446b1fd6dcf0bbd99c99bd2a362ac625e10659e199336f906acd7e42955f907f8ec80941d9cd76e009f7",
      "fa7e84221081c521fcae967681ec5e3f657306e846c926379024f34b07d41ae1",
      "043ebb4a2ee7a6d228f11c71f02dd3cf66698e61216691a3baaa6e8f9a7bd50b179a72a62056124797e2580b4fb81856f339bfc674d62feb7559e249629aace4ea",
      "d4b47742dc88c21a27e7e21486aefbbecd5de72ae85a3c03d65b15931a2e2a0c",
      "041863c08ca8b01735bb2514f4f38ab8e505873b2f2a706a1b8b76cba95c1589f67618688bea6b5f2cb001f0d4cee7deb72f4102b8bb0095a3a466a65817c5d4f1",
      "",
      "",
      "5c0e0156d0118f8c8565550c9af908af1377736a6266d34cca42c6f97a8a70ef",
      "862a93b766411f32b0e10f78",
      "b5ff8ee759239c6fa1810740c971bc35c708bc02901a0629e7bcbc4d69754629229cfb9fe95e70b8a82430ba6d",
      "3df3a047627d05fa1b0de290be6f87316d8da529be9f102ca8f1abd8e78e43135fe0fbf6d74eeb9614ac11cc7b4168d8ef2f54a1123fdf87c27523811cdf7b8d",
      {
          "d58edc871b9e9141e57393914186ed608ccbd30e19c3a64fed3fb7a670012829",
          "5ba3aea5722326c8248c05daa29e8d8256d664df57f864e7611e4484ede51dde",
          "9a7c1f201f0daf12e6a6f55d850cd6a0f552a00a4676fe6c452771517287047e",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "c2218b04e7e39da9fcafe21842d521416160e98160d2a7ad0de07cbc09363bca",
      "4f6465206f6e2061204772656369616e2055726e",
      "56dc4cb1e5fac070b762935f32411ab3c802b4fc9fb6ab030322cc2bdd4fb088",
      "0417d935019f332ebf4d3c2307f933368c49818518efbf71d14d860e226936e44c426b0469d8111feca002488512912b9a3625aac02bac7ac429113cf0c280cc0b",
      "bc29193d846d960edb11b846e80b43738d6341e5444dc0e69dcc5ae0d97de6cc",
      "049c331117efee994348cdcd7f7ec6d1cfde4ef9948ae7a1cbcba8d20fc4843a01e15c59ccbbf0af817207a88d564234f2f968bb19d79c12c69087f31f61a07fb3",
      "d780ccebb177b388675e9cd8f80a2fb105fc953a8243e4a1c9a6f7ece22c662d",
      "0431923a7243d37a2653263106c77504052a95d4b119d9b5de3bc76d7c150b591db6c2e4338bf3737efee490fdd7ba6950ab03a5a76e8c127a803ed7285687bf2c",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "dea3edc0191d27e7026e5046fe37e8f6cfe862cf580f3a4f5be198d844c23028",
      "a97875b9a3444d718ec08055",
      "5526be92443bb658fcaf9ca8a220ecf00d70888bffd88ffae51bfdcccd6e148ecc65f1e93b63a7523f40f4a76e",
      "344fa684a7524362e6eddca84a0532cfd296e769654354d95d66794e7de22ee8f6d2aa693b1ef8e4318577b7e3a4d21e5d1ee27dde1050e7008aa47beadab97d",
      {
          "99af05e72d6fe4d2976d4c7f6653ba7b224cc6a93525f4047722229687158ba0",
          "aa005005bfc0d0b8d69a8d172843757ce9af2a17d557b596e37e3f4bff8f6c0d",
          "c55181900ac721c1e1c7d88fd8d46ac5434f60239b1aa2dcff7d2f953f0fe23d",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "918406d83412cb2ae65becc752da66323801933dd73df81c4e4e7c747181574e",
      "4f6465206f6e2061204772656369616e2055726e",
      "9c00a6ecce7eac4a73094bfad06d17b2c195ce5d891a76c466d9ce17e2927aff",
      "044415d6537c2e9dd4c8b73f2868b5b9e7e8e3d836990dc2fd5b466d1324c88f2df8436bac7aa2e6ebbfd13bd09eaaa7c57c7495643bacba2121dca2f2040e1c5f",
      "ebedc3ca088ad03dfbbfcd43f438c4bb5486376b8ccaea0dc25fc64b2f7fc0da",
      "048fed808e948d46d95f778bd45236ce0c464567a1dc6f148ba71dc5aeff2ad52a43c71851b99a2cdbf1dad68d00baad45007e0af443ff80ad1b55322c658b7372",
      "",
      "",
      "",
      "",
      "a438e7fa5713046c634b7ebf36efe9175d2aa63164a430ad1871c21cbce28ef1",
      "80e67dfe703b591e18cdb04e",
      "81a1f54372913f6dd88f45d7889dab174942baef7b1f3a32ee42058bd4b5ca5e8323301420b9e3f3c7b56fa8b4",
      "c585a0c00032a14c67e7b4f6b1e02f1e9059415607e91db6a75fd09ecd239f87ed97c1e5cd6938aaff851b01a92319344ed6b01e82de3ca2aa43aea64f09f605",
      {
          "bf563e98d70c6daa0ef4d5f4b6144bc0eabf51b3dcfaf42dbee3556fbd0598eb",
          "cbd5221dfd7d5ad25beb6a516112cead025edc9040cf796cb6ddbfb9e15d5179",
          "62816ce52594cc9bdfa3abf9a72422b1a03b1abd0716741f0e7c6421617520ef",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp256r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "8424c8c9eb1a482a8b6dfefe729f5fe33ea6de7f07ba37a58fe30b256cf54e9d",
      "4f6465206f6e2061204772656369616e2055726e",
      "871b25f4b1e214da16ca38ae3d45d7f7d57b403a930b52a44d8427e5bf12aa09",
      "041422d399504a8c51e81dbba8ddda0a5b7e712c6305b5eb4a7dbb9b93f1ec82d9c3bcfb0d0b282ceb7c9950ef28742250e5e34a942e239bb0547629340afec33e",
      "eee2a31e38d131ee6172aa8409d0c920f002f63ee5aeefbadcd50720efb6630e",
      "044f44490804b7f3ec5a8da8eddc0a6b27c0dab0d7134c92144e3f99ec3dabecc657f6b54eabcfa05d60bac063a70db2125a7a16a051df4643dbaaa5076a25efa4",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "a122f5dbe80a805bb66929c084844c123538ead6fd44a0e3d7ba3dbe3b2f952c",
      "dc892fcb09fd090b4cfcd093",
      "0454bcbe4969734b80276bc16cf8fa2ce6e8f9f48d8a0724772cdbae5d7d49b2b74996274ed7bf45d973fd3bf2",
      "877fca15c1166285ac739430225c5df5ad93b404bcc4a3e333b63f1462b5d9be63164ad9aae04ddaa62e45823c79bc9218b0ad73149917541a5b878f1293753b",
      {
          "f1232ba252a0411b74f53701b14259f248de74a40ad39be2fa0faf2da464aabc",
          "f4711d74c4bbe0f2dc7e16631d6650179667c9c254fb6f5347419db8dead3783",
          "d2ac77a91477ba9e423c756545781370a5a03254deb31914e7d51b214cfe4cab",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "59501bad207bf432781371e7c9c26e908958301ad138a3332c6315e18215308dc13191d9c0258b88341569ce97dfb6e54f0a4ebf70d19166256c48343de6a9ff",
      "4f6465206f6e2061204772656369616e2055726e",
      "000ae237a3250c6365acb81ceb2c1d517404bc68e9d6ecbf0bc42cd2d02a18a2944e13d9b11830d632ce4a0348dcbcb479450d6e29c39f5784fb07df25e6573eb280",
      "0400bec215e31718cd2eff5ba61d55d062d723527ec2029d7679a9c867d5c68219c9b217a9d7f78562dc0af3242fef35d1d6f4a28ee75f0d4b31bc918937b559b70762004c4fd6ad7373db7e31da8735fbd6171bbdcfa770211420682c760a40a482cc24f4125edbea9cb31fe71d5d796cfe788dc408857697a52fef711fb921fa7c385218",
      "009227b4b91cf1eb6eecb6c0c0bae93a272d24e11c63bd4c34a581c49f9c3ca01c16bbd32a0a1fac22784f2ae985c85f183baad103b2d02aee787179dfc1a94fea11",
      "0400b81073b1612cf7fdb6db07b35cf4bc17bda5854f3d270ecd9ea99f6c07b46795b8014b66c523ceed6f4829c18bc3886c891b63fa902500ce3ddeb1fbec7e608ac70050b76a0a7fc081dbf1cb30b005981113e635eb501a973aba662d7f16fcc12897dd752d657d37774bb16197c0d9724eecc1ed65349fb6ac1f280749e7669766f8cd",
      "",
      "",
      "",
      "",
      "829f508524d2cf6fa51616d9ccd9f862",
      "f9ac336746772688d4d87ab0",
      "025404c525808e9087ae0f62204c31076cf5d6473f5d9b4e437e03c84158497341d2c941e8b94c8050190c8947",
      "81c6f475e112ea4139f032e6edc40e55e630d29438a3ab42dd2e92bcde147880",
      {
          "9b36d9cc29b33fa931e3065f4490b7a084f1c91ebe6541aab102305b5b8c9be6",
          "befb79721b20a53fdccd9af50e8f7e823dd3516a68c4357145b94412e96a2326",
          "2c1d9ac662c578e0739fdd44fc98dae7888816c3f779853fbee596a987e0ef9b",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "753ec759fa73213126a8d5eed5f9931fd70a80ae52626ed46f70d0b3d27725f8cadee6d6bdf3553804e03962ce66f659e12a294429efe6841ff475f4a2c6a8b3",
      "4f6465206f6e2061204772656369616e2055726e",
      "0195ebf694a7d579ddb5537b6ebc3c998f62ffc44deee53b73833765c2cc776cfae5a72f5b7d3fddbf854c6a419908c299f215cb212eff21ca6422a43f68689ce8c0",
      "0400557890041cccad0afae552ccc920f6e1242830df929fb0c552e299463471d16b5537c27c3627e46aa6decf5d0b600566592a7c4c315281798b37fa9874cdac3f050150b8429bf35a38250341eadfee6ecae5cd317dbc9262d0b3a6c44efaa555d26822bf7fc370e75dbf1db5ceeece20b5ae7ed8bd9f384226a4a43aa33093b15a8be3",
      "00e28b0281c417a1db047b20dab9eaab8c57fcde9f82becc94356ae168968107a7f9507e77a77f5946840ed5107b8a77eb53145815e942f4c01d251b91272a9864ea",
      "04012e8e7975a4bedd89c4536917c7696011ed70dff9d3743e92421e4c515d0bee54613b84a48fe6eb0dc5c397ecc8e10001ed3a52c508a32a556126944bbb04468024007555833b07bab58559ddfc0116ad8dbcadc2ebd54149140218a3042c0c916df7ca952f9061977d29150c51534c5a790230cae9df06e90fd4c5fba197f4f9414e62",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "674ceb6b6d927faaf7f6adfb8fc3c024",
      "cd67bab65c8acc84e73c2448",
      "5824d9da9f1cdfba1fd76bcaf5f80f65947b9d68dede981638a49d9a61256f3a0dfe77db6a4c9c8ab6d37e9952",
      "1549772bf8739a6fd35bacf3607b3ab636f1779905672f25e441b8819e3b0b24",
      {
          "6b8b9c434567c1fe2e78770380ffdc3fd837d7e85ed27a1ff7572ec6aaa2201a",
          "ff55be731174ba0652d7da58167318434c69652648c7d69d7d625e7ec6c00d57",
          "3a5a2a565a2ea22cb7ba1ca8757dca20d3af4512e20b64ec4ad34678b180a995",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "9f799a200a9be8def31a2e686bfe514a70e7935b90951bda4f7d56ae8c3ad7de5a0a1ccbf193a858b51ef22e7973fbaff8ba6816a03448293c09ed02860d9cdc",
      "4f6465206f6e2061204772656369616e2055726e",
      "007b25ed6e784d7abad90c5cfa48075e45a96a9b0232a1b54b209479b0a069e651d186ea05821e38e32379577721cd3f07b837f89dfc57ddeaa4c9af5dc76eeaccfb",
      "040167ad166ce1411e22e0ac24e70c5259e81de2689a05d838e6dcb894c6c372ec0636f3889c16a03dfef4ee399ac83f073483a13ac0966ebc8c21a7dc13d4f4de258601dff805c2254f447051674861a787e571f2cc19b45ccc09c20658cae8917d5acb92252ee81cafd420ab3cef7ba483208174e1764a94d7ca1299e6eb35607b43b8d3",
      "01d12cbd0eb8b421b5945d7f12c308b0554fed0040ebf279e51b1459597a4ce3e4705e7f06ec78ac076fe4f8df5a45094660510d55156f966fb6d326abd208e79f0e",
      "0401b3a70626fe69612cbf072bcc521577f78141e9eb2cfb3514ad9e160460976b5ab6c6e50740894b16929ed9774868f178d44f7e1b519b5dbaa9a19468c3d3d2c89a00d3e3ab413c3874b459eca453bd575e2268ca909e2a287d0d026d3499bdff7dcc6bdf1cfcd8eb3e328401a7daca8b20b721c0c2150f1367573abad488e6eac1ae8a",
      "01f8eb931a8c7cfd939008b2153c5ecacc375d7b8b4e77cb059af73a4c3f206ea5524b105f1e4f12f5dc641e6c3c883e85db6e89f42ed9dd5915b6624052d446e4fe",
      "0400ef22f755a8b24e272a773464dca9fc5026148375779135853c12b43457835dac6494379d01420b1697a8bd1b275956c32dc7938e0001d0b506a891de69f7826b8a004878cf3ff41c0d47150c61feec702eeaa9a1f29d5f35d4aef965b9a58989b3bc558f78cdb2c3320572ea5b5ce199c1f6d8adf4be80f55fa97252a55dcf25439ce2",
      "",
      "",
      "b4c1e183807099d092faa5a28377140b",
      "625b600a33be34bdd14b2476",
      "684863861429e719e3874931b126f3fefaa0b701e3d9f81f5928e1b04c1a7df136ec31c8823b205b104d0cd563",
      "5b7d30e90aadfb13362d0ecbe0ae0ed07df278a470673fd19c8d0f9078c25fd2",
      {
          "322039996f083e6364861a174056002b375bf30cae0e9f3180840997c7e03d66",
          "f131257cc50746ff2345ab42a61fde99e3eaae3930522d4c5d9031c8625b0228",
          "2df8fba4f83b8f2e3e501e6eb7642c688339173d3fe0fb00e0705638d6985c83",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "3b37201be7adf77d3e9c2cad6c40c7a202a171c6c3e494ae31ea2e3355c208ddb9ee1f4bc93b6a5d3457f104c2b1c693c1c1dc7b4bff41bf0e6629a2fcaa0b87",
      "4f6465206f6e2061204772656369616e2055726e",
      "01e1b006811a044a56ce62427cd2ea34b19ef6990c510f6e08ed5e1056c2ac39f61687134d292ae559fd070e31428ab2873b798908c3579e7a6f57e2e26d0dc532e7",
      "0401a514f452f316bda875c37ca40dd2ee5d93be7c80a81c423fb1500974d87314ffbe8d5aefd34e69d44f310cdf752519cad0a2ef1a240d67049e57222291aaffbb85004680e6232e8555c97eba731c7e0a47a1063e039d4c9e915da35f53ce5310ebdc0a9586b222ebad01ed9bbfb844c3fab4e49c06de034ef780bfc74b774cfabe93ac",
      "01b5ba66ae400d58a9e77b7cb924a2801fdecc849a0c059c29c665f1bc855e119d75e0ea7c693dc48a576c860637da2c9b4d595aaf6b33df78fe32087013c6d11f74",
      "04006bbee56eaa2fb413c0ae03cc3ce9adacc0cee742ddd3b2c147dc21a6b3124be6fa4ac3406d869b9b330ebbbcb6761e63d853cfad75bc73254b35c88e6e95a4171a017c53e5bfcd8818217abff317c03bf542eaa466a6f8f41be6cfbeae9b255f2361878cbe1fbe18efbebce0131e5bad132df514bae9e9154ef68c18074206b2f0db69",
      "005273388f9eb91d7266e53e859a601b2c4091f50d894c2ebfc252d047fab9e2c0cc7d1242ac81a959a55801211b0c378cc3a1d64becbe6d5e37213035e4e6b33b21",
      "040090a5544d64bea56f73d091ba0de8760f59f350852e533290afaf2fe4fcd12451f81889a6b53e30c495003b4483a620a2dc56f056756182eaa74db2b4d86b83e31b01a95d029e05524788257fccb07477073b5010ef95da7b41bc34188cd2355a2783c973e0e2999d9ba6ce8642c83abe78cd3ddc7991f5c444cb788a7fa625e46f4dbd",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "da50f1a50722933796f292d361a6da56",
      "90915fae644b85b3a550cf53",
      "3d2d1ac9e475e0d02d8f28c9d4dba172115a9051959c1a444b8c75d31b068b416f0ed314379b51e12040711b7f",
      "84a78abdbeea07967c522611183f194899fc7167c655b2df0d2316f0203911f8",
      {
          "008f5d0533730674833fc37c5d8013213c69853bba8cd02e9f83cc4bae81732e",
          "3c8a036540427b2e2d4b439fb8189461afb81772259bc7b33cef60f34088b6ac",
          "1d8d89638a1d3ab795003099f4a59560614b23116f3268081fa1be40d431b54a",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "6dd281daf38db958f858ed1a9c822d923c82d897007c8378e858647cffbccf5ee8af816cac5d6e43b4814b0002bf625580695fd622dc90adb603fbff60947917",
      "4f6465206f6e2061204772656369616e2055726e",
      "006c3ecd07198614d1c97c0c6874e522853efb5056f5272fb8d516db13d6767187b849bf59f0421b01af8f44b53f6bc7c73d608030086fd6fa08ee9a519c9b19cf1d",
      "0401c1cf49cafa9e26e24a9e20d7fa44a50a4e88d27236ef17358e79f3615a97f825899a985b3edb5195cad24a4fb64828701e81fbfd9a7ef673efde508e789509bd7c00fd5bfe053377bbee22e40ae5d64aa6fb47b314b5ab7d71b652db9259962dce742317d54084f0cf62a4b7e3f3caa9e6afb8efd6bf1eb8a2e13a7e73ec9213070d68",
      "01a27e65890d64a121cfe59b41484b63fd1213c989c00e05a049ac4ede1f5caeec52bf43a59bdc36731cb6f8a0b7d7724b047ff52803c421ee99d61d4ea2e569c825",
      "0400eb4010ca82412c044b52bdc218625c4ea797e061236206843e318882b3c1642e7e14e7cc1b4b171a433075ac0c8563043829eee51059a8b68197c8a7f6922465650075f40b6f440fdf525e2512b0c2023709294d912d8c68f94140390bff228097ce2d5f89b2b21f50d4c0892cfb955c380293962d5fe72060913870b61adc8b111953",
      "",
      "",
      "",
      "",
      "9e2062cb229bffa17e7ffcd25d30e3544391c2709eb6936f777ca5cfca69bb3c",
      "12cbc5e68d45d54c95ad63b5",
      "0d743e13c26783dfff2e2c7c33b7db67550980f8797556e2a4f9cdc7135fc85d0e1ed31bb1b6165729f724b95a",
      "6c62a3c65dc9bdaec6f45b2550fd90ed45a93cdb4c8c8d9839b5774d34791866",
      {
          "26d3ca5afc16beb8bfd2abe75126f8b29f78ce501943745cf6b8711e25545d5f",
          "b2cee665cd44ed9f93435dd3c24d9d3eaf4609b1260aa7210d9feb56e988d060",
          "23483d76811e31fbfed8cb718a4f10d64cb739347cb7e73d76ef2b2ba2bc731f",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "f34844ed2ffef87116a66d91bb381323529fad6f20f05201177bb319e3a0741ff990ffb1d0e21465ec1ca70832965a3c1696ed751666bf75a3d185aa1e525342",
      "4f6465206f6e2061204772656369616e2055726e",
      "018a197a6dceeb465b00eb46838c3bd6ab65a3ee30e678d8bcbf00629574ab8329260b8dea9a0aafe31e4a7830ab0937a32b21be30c3a0cdea681e29f20499a2d4b8",
      "040101c4c5d4a42f0e4e70f265a9f0fb14182f609b4f6eb5a6364b851258f16f1a01ec9456fc26df789f9f9d929af40506944d5008db42b4ebb80027a074165d70add50102c2b502ccbf139723014f7c409811d3f1fc84c77d3e4bf4b144b51eadbc156370b904fe76194b9eaf940973d21d6416ddb91067b9694fb631510d4e1c2218a542",
      "005517e1337af451eb4d3c145634525875ada40a250e463d24f901d78547f22991fe87d262cd3a2cda249a90b33515666cd01e58e742040d99c98a2314589e8cf282",
      "0401ce0f6e35b58a81f9da07980a8051e034f5ad9554985ecbb0e50502f2cd4f0dd1c7c003ed44b8dc4b4178453b81120aec0a30c97913add713f2eaac32a300ca575a01e68fc627924b920f1786e3520ab32acb2b8b65f63ee23bc06a8c42ff14b618175dd38de50a8ef1bf5a92af8d574e852550ff622bc6cb4c9480f353cd58c437188d",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "222f6bc59eaf5650a7f64e3fc993cb5d4da065025f301eb1dbc242511efb2b77",
      "519f891feadb8532857bd5a8",
      "a5501dd5d0e16f4ed33afc76edb6fdd737271c840ddabdfa4732354945cebc4d4fc870679d11e31770866892fc",
      "f117bba347d702df5c933551b79cd3857365c25704c11119a026f4a85fa66483",
      {
          "2f3d315c9931703a3abfc0ed38a51296ef70c14138cd64be8469dede3428444f",
          "7219515f51df0b7f88a7c202695a2bd30a7219390cefdeb5836f80b36ec61085",
          "adf0e43fe7a497f0452585f56e3453df84753a0597d48e886f3dcc6a08928433",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "4106dea2e745bfaa6eed969944796139b90b47c6cc56f177613d545688ba5bb1d4deafbb39147982e3d1d2ef652910f664b1f11afc8e36c2802c44b732a02b32",
      "4f6465206f6e2061204772656369616e2055726e",
      "0144cdba2de4ee23f0a3055f85d976340ce6db93c3a3ccec6f397f953d621e21c6645220a9cda5322e44980ebe25b4c4ddcf85784ef3ab9d484275aef71ecd52d13f",
      "0401c0407cd50c52d85dfc2da79838d2f6cc0edbe573db15bc3d459e16a7255feee1091be59d07bd41a1c1f2114ffc53767dc32c83d51dc00d7dafef0e93f0e96eba2100bc0ad8614d5cd5021e0fad6dbeb713e65045bca5cbc2332751580a25ee906da9c5ab9b83fee5c07121cd57b8f5a9b667911ef8c5c68f4b6f5f8c463a3fbd754ebc",
      "01af4ca8764d37e42d76ef87d8565669fe2e7a133b8e443d122153ecf9f2bc98a4c0a93d6c0e6c267d9e1f9702bfc4ae5cd07b8357709c0af85f6276284324552aa0",
      "04003f4ee80bb93b48744c5b020d929baf96a38457fb289ea1d19a9581a9fc157e85c9577e531a08dd74ed8990e2f90c795d4aa94134d45dbb966048cdb63625729c0701008c060684ac2f2fabcdc8286bf7f8fde3d3065c6b2c45429b666c993d0d3b74589f1dd5ae11d2377fb3b7098c60d24663b3653173a0368f18b7a2befb90b4d7c1",
      "01e73d20acd52cd2b05cb2b4421ccea7400d2b7704d14d3cb5bb9ff44a67651e965c49fa3b181a2ee650e6e65acfc43d0b74b64fac869130f6695ab40112204cb30a",
      "0401a6880df48ddfcef6dbc01073efdb0d4951983f8adbf949f9271a3b09a5fa417fb226b3f4dde9f22745f918c815d36bb88e8dd2eef35535cecad8769fc77f1dbca501bac4e3c599518cfafa9310c4ffc2b518d2ba2a0c72554ab7ca2929fa58b2eae7c83fd67f36149d78442c8c060433ab71320ff326f3edb8a07eb8599063fa45c605",
      "",
      "",
      "3e2d82420d5a8427a1e89802fd7626797149611ca25119b8cfdeea953264baa2",
      "c482bb57df0a9c4c0cf2ecd9",
      "70f68b3482bc302bce585df7d3d7373dff6566242e943e9c56349f7f8197d7823fbbfb77db69007dfb09024ddb",
      "eb90f6528bfa97ec831aa7583e331347fbaee11a66112619375f88c864c1359e",
      {
          "a7a8959282cbcea30fe48802014a7b60c1fd3fba742058a898d4e7fd5ae62257",
          "5e03be4f78e88b99aacfd04856a960412365712052f248b51bca733ab51a01d9",
          "81b8c13ad42f10a512eb97705ecbdc4e8c1ccfee9a867a89739c58adeedd61a9",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "75cde6105cef5b1178af64a6b0d007865b41a0c81a2382d746c31db74d6b21569fae8cb61106b6a68efbc2957c7eb3c7e253b814e304b8d6531fcc33d2f988b8",
      "4f6465206f6e2061204772656369616e2055726e",
      "01fb165240e92eff9a3945c1091cbaa10f78c0497e1468ca8532cd9b5377ae718d4ee676028da6dfaa6cca95807f4d14ca0406997c240cb1aeb08d55276aaca04c45",
      "04018f8f047f5cf53a77b45935b52b5b10f2da5a9389d76ea972114c44f6e011a3049bd45e27c7adaf02e25adc8ce199557dd75bcaaab53e7f91683c13edc2fc603a9801e689403f62005ccd3c7ff3d8ebfe94b37c68fa569787f47fda314439c934f6a01b52c9fe577682265106e0cfc883ddb874027f5fbd70e70848a2976c6b137e25ca",
      "00de3a538e7613215f792e61ac9c63381ab9995727d9b3430cc64f3da418992c3c5e74a5c4c35f42984a6d47d56500c7bd89a8cab30d4e7164dc99b6b11ca84e0500",
      "0400b7312b4f4dbbb221eb34fe21b56ad5f777a8d297666959819a356d6d2c2c494ce849ec0ba279ab692df1db4be7542e9fd230c5d5e5532ee4a5404d10b3a95cc58700f7cabfe05d845ab7fead770529d768c81c78e4f83a675fb35252459dc47facce676c82e94763cca3adf7de3a9e891fef60b0c0e8abb90e081b5950930c30beff60",
      "007a98f9c99ad3be564e1b87988feb0e9e5f2d3df50ae6a925770e310d598ae6cfffba08e6677f691c5ac706f904591bb0e30a159d48c4f3f9d8430576c19799c2f3",
      "040024f21bc628ff53946a172ace1f17effc65c2c99b202221a042a926012cbc564fbab64f82000c9aa7c7cba5f85934bbe8bf6d670708f1ab5e95c17204892948212f01f86c1b5e5dc20e1ec560fce0fd6de0dfdd5db1da6d28d516e6905c0e46176da98e2dfce940d46dcb952233bb9514791a37078af54e3d421b20394b9fbcaace2c69",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "8448bec9def51f26dadd558a64d0d6ec2bcff36acd2dcae33b516a4881b27098",
      "052ade27332de87413caf28e",
      "58fbb7f351e806e4fa2e5c865805c9334a1445e9a01eefabb0cbf7fb39b53cc32d0c323300e260382e314a8f3c",
      "20fc575028cfc8a86754d1fee57ecbca66ea1dd8ffdc824af1a9ed6223c8bf43",
      {
          "386a21a9854d775d62b151a0d79479befbec24020b9acb7b128a7b9418ec8702",
          "5763f7b24a89a972e1b7e3db40e56d3db4127489971941439f3753c214b912e9",
          "5276eac204dc27736052c619d0fdffc5da0b6a5d43a6a2baed8299c613a61c6e",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "e73f28d166cffc37eadd9f78c770d70c2007bacd02dd7a7b2390956401e89c0157d395fa13fe60fa8902578b8cca9a20ed00be644b7eb6f77e3332ecfc63b601",
      "4f6465206f6e2061204772656369616e2055726e",
      "0027d3b66bd43e58c81a0403bbd83a0a22f74080c7d2e03db9e6ca533e7fd440b581190f693e884e99c1976476411a751e05799e3eb633159988df7d403b378eab6e",
      "0401f828650ec526a647386324a31dadf75b54550b06707ae3e1fb83874b2633c935bb862bc4f07791ccfafbb08a1f00e18c531a34fec76f2cf3d581e7915fa40bbc3b010ab7c3d9162ea69928e71640ecff08b97f4fa9e8c66dfe563a13bf561cee7635563f91d387e2a38ee674ea28b24c633a988d1a08968b455e96307c64bda3f094b7",
      "0168c8bf969b30bd949e154bf2db1964535e3f230f6604545bc9a33e9cd80fb17f4002170a9c91d55d7dd21db48e687cea83083498768cc008c6adf1e0ca08a309bd",
      "040086b1a785a52af34a9a830332999896e99c5df0007a2ec3243ee3676ba040e60fde21bacf8e5f8db26b5acd42a2c81160286d54a2f124ca8816ac697993727431e50002aa5f5ebe70d88ff56445ade400fb979b466c9046123bbf5be72db9d90d1cde0bb7c217cff8ea0484445150eaf60170b039f54a5f6baeb7288bc62b1dedb59a1b",
      "",
      "",
      "",
      "",
      "f4ae46e447f44ddb7af4edf67b7bbce376d774b6e06a8d88571ecfefdec87745",
      "adbd83083d1c47d3d3c30bac",
      "7a0f34ffa87168b3308f5518e4046a538cc64dba1b704e24451478cb3a173599cf99f954138c0f384551548ca4",
      "65cad52078696c8602ca1b56fc7051ac5eec9958ae6862140f04545926f6f1b6",
      {
          "adedb5a830b8db684153c08f95481a35108ec46957b152d547b0aae7260cf8d5",
          "31385bdb10361801741b4cb5f84d6c7e57a63a8b7437a4e63b44d76a3797d153",
          "88d45aed98aeac9b4627805a5aafa8aeff81457a18dc211db691ef64c5b14a1d",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "7ff3f72d99113ce0667e6800829a3e6f07c4df79c34fb9d7a3394207fd23e1969d1dfd968711eee244772af20147929517d86cc9f6c1d2ef311f804622ee3fa4",
      "4f6465206f6e2061204772656369616e2055726e",
      "01c71d9b7e300c944ef7f65b97282f6a8d912477a9f7feb3a7eb7c4d9ae56a9edc30a536190c86375bd09d961fb59ba39541e9776b3ff4182e2fcd34b3384f0a0aaf",
      "04008c8deff5ecfc636ea8056b3f4187bed210ac4cf82bc3bb8045c514a3dd61863cea0218b0f0253624ea3c6a8d9195f2f17f5bcab5ab0d7140bcd4c40cab455707da01eed3c38fb1e0a1d1506b0fd25abea429f39113d7963a626243be616455337baacbf54b1c14c50e0ecfdf59e67574bde945d24f689bcb8680202afe6326b0174a89",
      "007c35842a7906baa88e0c4fc379de1568765d7db7381960b9ee36bd57e3938dca3a6dbfed7045e0fe43679e0528a7687dc23f8348bbba0aeb56330e39eda544781d",
      "0401f458bb82512325b1b1d43c800ad8ead076e9611d89f4758d9e219c670c011a0cbe855afd3eb26efda09267ae810e63bd74c8031de8137d25521f94840714d5ec6001f0282cd80999bccf62d33b77e772f7a39d6ea2724fa5b609b0a721d6a640b73c9caa49f861806d56a5b9659b0cd9f3ad2e15512d7ecc4354f272cce22d6294779a",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "c898d4bbf1832410da205971346124a84a0c12b3763a7c06a394166d21f5e1dd",
      "b26d9a2cf1357cae1e929442",
      "1a2a4d9dd2d72a08ab153c2b63d3265d3c380833bff40f1df8b407023a9a74bfafde8688096ad6e745e285d6d1",
      "25b8635587e67edf4a9b70ddaa922e0b6cef4b9bee83e948dd414947d0aae700",
      {
          "535616299a69f825d697c8cd8a0ca33de8d92e392e281f4ea724d738a8f389be",
          "74b46995a46b46e6dddea5d62ebefbb3144c1fd1924f9746fad743db5979369d",
          "10b098f36e0c0c3f62ab038d160c7da1e6207d7fdb72074308502c4a3721ce84",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "9198c6fee952700cf6e7ed7f0b1efbd375b5c5e6dfeda06ffa5e127c9ac9e6fbd02a1a547ee268aecfa19ced505cd91bb69c407380e1a741507e487a55c3c428",
      "4f6465206f6e2061204772656369616e2055726e",
      "01891ef8fcd1a2418fc2903e42e77383357483a4275baa930a74b534786454f6db0123ecf5322b1c9f9006d16c496328398021f421deb2894d74ff58dbb0a787fc7c",
      "0401043bf4020a8f010412a53856e1e142944badc3974337bd4f258ff8a5304d3b3878dbc4db63d9c0dff93c8fed5ca6adc5971ee8010b37db0fe4fd217bea144baf4301ad7d27dbdf711b951aba6ee0aafe8f0de942f8dd082c8377fc7b727da2f1d22a0871011640b73dd3a046ea64466a7b985d347bbe7662edd23626678a07207ac1f9",
      "01897555bcd43ee0157c13b31f850d8091db285b9c181e9bd4a056e2b77b732e9be5cea23d529cb4cae7d1421abfb62c410b1f897d41d9fc11e6dadcd832c4a73c41",
      "04011da0436077e26578b5a50dffd8d56832e6941e0465c4aab3875447ed6965ca10a4dcc19400170dca865592d483cb58fc28e59dbf9ebaaaf1ded87cf146ab1fbb1901581bd0e13600ea4d398dded9e899ba02109075e920751576ffdc9466a68a46549344d326f808eb1280dba9ad15e2ac71470cf4a627c62ae9bd74149023fb28a38b",
      "002c885bdee68225fadac861b86632a91f0d2cc3900fa576af2da27ae5f1e3fb9c8c641e342df80e612bae341fcfb6d5b14f2a84188d9fdbcd5e6a16fd371d87164c",
      "04003723436e3499ce249df96832287fd0fd377de596baaeb744cc2a1a06c989acef296f1d6d887e7ca1fb98b7a13e00146e2bf5e23d73c89b82cd898df126f898015a01d89ac13e4c88b93ca7d7d4ba4290d360f67ec3ba7c6a88afa51955c55609d9df091f091dde3632ae1f4abcb6f45f956f2587e948929558096e6abb65c0deaedf80",
      "",
      "",
      "eb9544e4423332342dc1784d086ec5789cee7776f97dc2df814b57535e51bf1b",
      "45da82d75544a7dae10e9831",
      "1f9aba3b0ee7bb6ad69ba428d1a09296ccc663238e9d26cd8b13b2a5ce3d4ba41baaca58ce37ebb84f2be057fc",
      "45f4cb758582f08b8b2b3074ba1bc3e9299153122913eeae305fd227887bc547",
      {
          "dbcb43aacd922fc610f7d344c0a85a12c778a98de01a94a8d9013c7b1adc1c5c",
          "8b30bd4113462e4b1294aed78c61b21cda0008a55967dcf5950b8ece1b532473",
          "23570916bb52bc4b83b98fbc640d521eee2244f42b75b6fd0b4ed7ffcfe6548c",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp521r1,
      HashFunction::Sha256,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "1259bdd55fa19a54f83e7ce8a76ac50fa4b3b806e4570a07a474e2e83b15434126df0c1a1061ba10b92721278293cec27b44c25329b67283a28391015272e78a",
      "4f6465206f6e2061204772656369616e2055726e",
      "00a4149da44e7a0bf65d02995521b18b18aa8f4eab11758e0ddb02e7be778bfc6e852e9e7fc0fd0b3c91b9905c0fae5368bee1e82b27fa2b73c51ec3430cac727428",
      "0400e5adef7e178bce0c908cd5122f337ee0f8dc5cede93b6e342d7de0a19f4487a13b63a7ef98131d356a8658ff8a42e0dec9bb7022187c282a032191609fd65dffa80024eb16e156af999055e7e11d842232e9e3d9be9eef33cd2ebd6c348d863e66f18701cd249c7ca907131ca98b775b3acacd1e0c5331ff574e8e8aa9bb2f204b4aef",
      "003de4f3598ff9c250efa793d0860a3b72926aa851e911e1350ab191a31d1c887cf56a5d5bac1278dd911e65e996906bfff166f440eb6d7b31c91e34feb06aa2398c",
      "0401155517c52eab850375772ad82adb0a829f6532953a4683e3618d74d1bc4bfa865f1ac8b45b3399764dbe399795aa91c8f14399747631286417ee1c1f82afa7dc2d01c4c4c11af9539f16e895891996df4f7a49226b543481612be56f8147d4b1f5e27537324bf1148b0c63defa7efdce3e2264a63dc2520ed173510f3be437cd8d548e",
      "015fe14b0d7e41d92cc4737dc12e460677ff250b90512dfc330ed16c567849ef75491cbd93e168543759dec5bb4857feded56a47089808a6a5c6be6af7b46aa6c18e",
      "04019217082e755b3ec4c6db3e05fb707020e1bfa3e739d304ff42c92fcceddf03ad2ecaea1181830078edc065c13d08d7ca611536e407fc5dcbe4763a098a6edfef65013aabc7ddcfc5bfce0dfabe31bbcede6728e3be66f783332903d3e1d54f1e7b1ed3632df7ca00c72d0439d4b14c17d52212b5999bbc92c6c9bde3a5da1343f13fea",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "6c92626445d925377438c758a713fa26e50eaabec852118b9b9eed7f54cce894",
      "3b29b8bb622ba14d9ff4d4dd",
      "d112aaf02c1d29a86f07dae92dc20d5756c8fdc2fee0d1516155bd717c8a90ea092271472f84ae45b136cf65cd",
      "45caa601cdb71555ca48db1b72e2ee49f22589d43d4ce5a54cf1c9fd05da24a3",
      {
          "2b01436ed180e15e3478c70470a7a2d234524d627a8b75fb3a6bc9e67a93b1c9",
          "756b4be8f269d5c4fa2127a29325404a4b317a595d8870949ef71c9836bb862e",
          "cfb6062516e1cae6235e147efde9ed51ff1d10e740cb5bc58f79dbbc7af8a286",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "e8d9d4ebf5911a6048e15638b2248753c5f5a76d4229fef34e905d7e60a320fe2a7b0a61ddc8aa7c3988dd439eded7be497c0f054d3788218c9c34febab2b445",
      "4f6465206f6e2061204772656369616e2055726e",
      "01bbe3a4e3be70cf3c589d340fcb0e221ea6db07af3ac673bdc2d74188ca70b9086aeba2e39e497461363419cd1691d9ae453f8092e4df26eaa63f866738310d9328",
      "0400ac8d1611948105f23cf5e6842b07bd39b352d9d1e7bff2c93ac063731d6372e2661eff2afce604d4a679b49195f15e4fa228432aed971f2d46c1beb51fb3e5812501fe199c3d94c1b199393642500443dd82ce1c01701a1279cc3d74e29773030e26a70d3512f761e1eb0d7882209599eb9acd295f5939311c55e737f11c19988878d6",
      "019db24a3e8b1f383436cd06997dd864eb091418ff561e3876cee2e4762a0cc0b69688af9a7a4963c90d394b2be579144af97d4933c0e6c2c2d13e7505ea51a06b0d",
      "0401e06b350786c48a60dfc50eed324b58ecafc4efba26242c46c14274bd97f0989487a6fae0626188fea971ae1cb53f5d0e87188c1c62af92254f17138bbcebf5acd0018e574ee1d695813ce9dc45b404d2cf9c04f27627c4c55da1f936d813fd39435d0713d4a3cdc5409954a1180eb2672bdfc4e0e79c04eda89f857f625e058742a1c8",
      "",
      "",
      "",
      "",
      "b7ae4daeaffb64e9de88beaa81aa4da3",
      "fb856a6033ee142b92d6eb63",
      "15eeadf40282492721baac39290f4ff45b85884fb72f5ae9f491ec3d9ba72c7e1cd73d73fa9c110b3dbf0d867c",
      "4b569893084d0e6467a0bd1cd3dce3c8f2d2bc146b175026e84f70ee9d05b4951993b51e769eec791f061af0ed3aba0757972d78d54f4c0ecc2dde01deebb195",
      {
          "937c4bca58dcf53229fe35a369a58f5bbdd669b9b6d48a31eb5e209f12397a25",
          "404ebf64752a554afac66b9894829d1e14ffff3fc6af0d85fe59079586482ff6",
          "e3be1ae143f77450427b7e3123d3323083902ff3e4600e8c6e070f383f4ef8dd",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "eb7e17024fcbf53d8120f14db3769651cf3d281b24d430d2b32568c643247625f3b8c58f3e3078958819af06644a6bd21287ff77dc87b934084da52ccb854521",
      "4f6465206f6e2061204772656369616e2055726e",
      "01a1ebd77adf3b478a030c69fe82f842be5986c54b8841e6775e7a6234dd990c715d054662c84f5ee419e9fc32434ae1b69e40339d4d8155e90eacb956f031730d4a",
      "0401fcd057ff1053a2ab2810de6941b64c0dd8139a208fc4808ea78353c4a1c36f772e53c7a26de7ed1f3184880db678a02937e3e40ca9aae17ef3371ee57ad48c1d2700471a52fcf4e95f57db377e82069d3757a02e98b588e935fab2604bc790eeb8b72067fd1b505b9feca5c5c86c62bdf80a3a3870429e545ecf3ab2f3e2f83bd8d67a",
      "0014baa1efbe9dfd4a61dc592455859defeed5f2b8e6492d942737fc2696745f585a71a82eeaf1f086a075a19ada572a37b7b2295f62a56537ed406ab3cf5b24aeaa",
      "04019f3b493f53634d1e44224f6af757b80e071ff26220e33fc1feb87bf68e2d40484a636c04be45a05f6d423cab3e9081f6799a03c22ad5d98f01401fa8303e5ebde7010c1c068404dabc80cdf3adab9e00e415e05a6935028858d9e5231d6c4ec3db83fdea587a35c6ea4fa5bd1edc702e026b7713af68cc16bda1591a250c25d7b22162",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "a97b660812a5caa28088fa2f491a9d9e",
      "dc98071f41d23172e43f33d8",
      "6b314d3918da44e15f1693cf1ca23584cd71fd6a9f9ed6733810a13709a1eccd8ae9c9f2e2a1b33f31c2ed03f8",
      "2bcd1d1816fe0ba14e9bccd9f813db78beec530ef70dd58d23725da8763b461ec3500f819ed34093c50e62585ce74942fe5ecf842d2f511d4ee5d8a5ffa69b4c",
      {
          "ae366e3cfbb9ac8240dcd3ce6588489db2a4c3e5be3bad55b70d1768f999d875",
          "a5d4e56d9cf8f567e00ad5598c520948d6c7330c82f966ffd815b74daf0b5a2e",
          "95a5fcc552ce75c2ae8a0575b540f9d15bbae266adab2dd11fc9f14b92005d2d",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "04d117f3e55a8cf97884b544a4182fccd747499e785199b3cc9379c2d43a7a5a6c392257cb97e5bcb978858477257a9bb992ac89e9295c7cd8ec29e7de7f9b07",
      "4f6465206f6e2061204772656369616e2055726e",
      "017d1ac3449371ed0f117f4086a8a0a7be5eb548f4b2e87203f4becb43e4c1801e07a49cbf02f9872e72f950d525c51f9ea82fba0c45e222b30de8188a7e3f077688",
      "04004631acc6884f44ca28527f8e92212709437e53e990cf855cdd910f4ca93e067d7611541b19a4c2c37e3ecf1d781b4838840d9d2bfb64338175802345138c245cec019ac62ab2dce06e584cc407b933e682eb6848611efbc9b6ce68c24d1ac91befd737f63021b93654fc5a8f4ca35b0899f42b78920a2def54f57bfd51ff8059074a87",
      "01f65e54fbfe298b8704595b2b6ed235f76284c21e669f3fc3e88f0423a7706cef6e060ae4078c436cd9a4aaf312787c08991a817ee14dc48c487c658580d4267881",
      "0401c42330bb25c88eace11f73d297f9e59cc8a956e6d3252b42f521dbe61915eb7f99086589fcc31414e97c59f2b03873300638806eaa2a107c25f3b0519ea0be13f50094d6b1ca47230bc95dc5a2a22e37d01ff12fb484f6e6b8ab99171a4b5b59000ed70d23315924cabf790c6c267f40d0c6e1072af93bc529edae30e27b1c2da14f8e",
      "00f3579410baac65c169bd06ed6cf516e9d289e49cd48cc9c352c6ab992f4104c8e5411b66efc2ec728da4ad8b8a9f052b632516c2e265e5985b9c6352a4ff141b5b",
      "0401a22556675e3a5cc3d1512023a39048491e6609ab1a0dcab6b91fdeb9ea709514e0955be23a93c37c0b8a00bf94fa61a15c27e0af39d8598b2168792d02000ecf0f00c48c856b0998a1d9dac0cedf9bdd694a9a0e2d95efc85362ca563dd0be6c4a1ba140b49f30fd97d9e07c4044fb60fb3784129b3ccfacccaf676b4090484dc98595",
      "",
      "",
      "78f10a425764ca95a2d51c6fe88909b2",
      "3211ef1fea85ca6c115d9c90",
      "259d273d16006a91072733bd69ff2a683422745d56a8aa5ebf96f3b58af9d51e19366f3d67e7bba007377fd4e4",
      "9a3e8e19d6fc389a8fae4e2ce008c0905233185f3239c9ade248195307cfb583d547c776f45748c72d2a788c8f7754b2693f5c4b229bc9cbebb29c0cf8f118ce",
      {
          "c5fd0e5e565b1a7eeb9d61ec5cf99f37f45f976fe0bc114fe7f43c12d977ae23",
          "b45a4fbbc48c2efdbf3657e9ea705bdb55e44eb9c6d43d75a4d55cb5e21a4f27",
          "c1d7f3417b5551f903a88ff004cc87a3e2ad0455ccf6d513422007a46ad121c3",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_128_GCM_SHA256,
      "184d2c20295675c3ac521597664ffef56d0e0dcd9c034ac149bcb58eee7e84631a58c64b705d8b84cbb1d24ab8063311d43bc6dfcd469ff9345bc06175a776a4",
      "4f6465206f6e2061204772656369616e2055726e",
      "01db124eff897bac7192fa70b925b4d7de5fc8df534a54b0ad6f67b2e73202615600879e396106a60bcc2f1897e99cd76ae8fe2a1d7650cbddf42173c0f33edc9cf9",
      "0400f30cea3e9069e2d74658751ffcd54578005c82bb253f6710e2873ac093f58dc19887bea69b6003e1e636d05f72ed9ecd38ff166a93e042efe57dca426ca223033c005e036d6e38eb9fa718434f35380942aea351ebb6473bace137fe792d241215ee7d145db452298615aabc3178550ed9a3b5ac12407780a9d57266a552b8452a4c52",
      "01d2082b6a2af4c8b3896f27d49b5aa1a29df49300111d4339d91438346e84611e6597ccfc864dcb4e7f5c26afea9f7ebaa599b62bd6b9baed5d39b8494bf59f8511",
      "04004e7fd8134b992f93661a949b11e93966a571f05d73a3535897a068a83379f19a348ab4f8bf3891d0c8e9ee3be87b9dc342c573116f5fadb9e694ae64269eb2597500b0b0b2d54c62a5e9beb1b3f6463fb6eae34ab32ff097d1a38ec72675ba042117b54850c0eb6a6c7594588f7cdd8a5ac3e0283890d125e2aec49e7427d105efae52",
      "01bae43ef5a5d2690c5b175dfd70b94b05857c4d5d34aefe8bdfe59c1a1e63140747533bbfd3dfaa3751ecce4df12468d37ab94e09d6f25637a9c64e4a55fbc72a43",
      "0401440a728d61959fb582a1e7c2978f0d173d5d4346368fa16af6cb94a2bd83a484d9766d1924f8265bdf99f2859f58b141ae2df528027b0859c4dfd0297fc3fc44f5004f173ca1a114d5b8a2fb394b1c19d8d084914ada66b885fed5679b2c0a8f6d5b48d629ab09473c755fd0a790154006b8c15a1d78d2e87ce01642ca272878106249",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "7dd39181d6245abf310006a567c31b74",
      "16e906b45a0515f7e542b5c5",
      "9153a7fcf8ec932b91bab63f777265ac545eb9a3f23eaac388a9143aa16c6915a27bfb3e97ea57fc6d829a8fcd",
      "fa8766377a9f11a2e7ef0fcc984365230a3239573da413290992fe48fc688c93c2830154828266da1393a6a662dd8c8633de940c425382c23e990c48fc93cf10",
      {
          "c921c5d289c146f6c3f6d1605f34eebc334a47ada58c4ee95658b1edb933a242",
          "1a55dc86b2399bdac7270edf371ab33deaf62b71c96214a0fbfb4e120d6f36af",
          "ba593ad3d22c0f3e8fadf4838e71c80727b358a28af718496c61317abe049022",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "776ab421302f6eff7d7cb5cb1adaea0cd50872c71c2d63c30c4f1d5e43653336fef33b103c67e7a98add2d3b66e2fda95b5b2a667aa9dac7e59cc1d46d30e818",
      "4f6465206f6e2061204772656369616e2055726e",
      "014784c692da35df6ecde98ee43ac425dbdd0969c0c72b42f2e708ab9d535415a8569bdacfcc0a114c85b8e3f26acf4d68115f8c91a66178cdbd03b7bcc5291e374b",
      "040138b385ca16bb0d5fa0c0665fbbd7e69e3ee29f63991d3e9b5fa740aab8900aaeed46ed73a49055758425a0ce36507c54b29cc5b85a5cee6bae0cf1c21f2731ece2013dc3fb7c8d21654bb161b463962ca19e8c654ff24c94dd2898de12051f1ed0692237fb02b2f8d1dc1c73e9b366b529eb436e98a996ee522aef863dd5739d2f29b0",
      "01462680369ae375e4b3791070a7458ed527842f6a98a79ff5e0d4cbde83c27196a3916956655523a6a2556a7af62c5cadabe2ef9da3760bb21e005202f7b2462847",
      "0401b45498c1714e2dce167d3caf162e45e0642afc7ed435df7902ccae0e84ba0f7d373f646b7738bbbdca11ed91bdeae3cdcba3301f2457be452f271fa6837580e661012af49583a62e48d44bed350c7118c0d8dc861c238c72a2bda17f64704f464b57338e7f40b60959480c0e58e6559b190d81663ed816e523b6b6a418f66d2451ec64",
      "",
      "",
      "",
      "",
      "751e346ce8f0ddb2305c8a2a85c70d5cf559c53093656be636b9406d4d7d1b70",
      "55ff7a7d739c69f44b25447b",
      "170f8beddfe949b75ef9c387e201baf4132fa7374593dfafa90768788b7b2b200aafcc6d80ea4c795a7c5b841a",
      "e4ff9dfbc732a2b9c75823763c5ccc954a2c0648fc6de80a58581252d0ee3215388a4455e69086b50b87eb28c169a52f42e71de4ca61c920e7bd24c95cc3f992",
      {
          "05e2e5bd9f0c30832b80a279ff211cc65eceb0d97001524085d609ead60d0412",
          "fca69744bb537f5b7a1596dbf34eaa8d84bf2e3ee7f1a155d41bd3624aa92b63",
          "f389beaac6fcf6c0d9376e20f97e364f0609a88f1bc76d7328e9104df8477013",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "0d52de997fdaa4797720e8b1bebd3df3d03c4cf38cc8c1398168d36c3fc7626428c9c254dd3f9274450909c64a5b3acbe45e2d850a2fd69ac0605fe5c8a057a5",
      "4f6465206f6e2061204772656369616e2055726e",
      "012e5cfe0daf5fe2a1cd617f4c4bae7c86f1f527b3207f115e262a98cc65268ec88cb8645aec73b7aa0a472d0292502d1078e762646e0c093cf873243d12c39915f6",
      "040085eff0835cc84351f32471d32aa453cdc1f6418eaaecf1c2824210eb1d48d0768b368110fab21407c324b8bb4bec63f042cfa4d0868d19b760eb4beba1bff793b30036d2c614d55730bd2a40c718f9466faf4d5f8170d22b6df98dfe0c067d02b349ae4a142e0c03418f0a1479ff78a3db07ae2c2e89e5840f712c174ba2118e90fdcb",
      "011bafd9c7a52e3e71afbdab0d2f31b03d998a0dc875dd7555c63560e142bde264428de03379863b4ec6138f813fa009927dc5d15f62314c56d4e7ff2b485753eb72",
      "04006917e049a2be7e1482759fb067ddb94e9c4f7f5976f655088dec45246614ff924ed3b385fc2986c0ecc39d14f907bf837d7306aada59dd5889086125ecd038ead400603394b5d81f89ebfd556a898cc1d6a027e143d199d3db845cb91c5289fb26c5ff80832935b0e8dd08d37c6185a6f77683347e472d1edb6daa6bd7652fea628fae",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "f764a5a4b17e5d1ffba6e699d65560497ebaea6eb0b0d9010a6d979e298a39ff",
      "479afdf3546ddba3a9841f38",
      "de69e9d943a5d0b70be3359a19f317bd9aca4a2ebb4332a39bcdfc97d5fe62f3a77702f4822c3be531aa7843a1",
      "5c3d4b65a13570502b93095ef196c42c8211a4a188c4590d35863665c705bb140ecba6ce9256be3fad35b4378d41643867454612adfd0542a684b61799bf293f",
      {
          "62691f0f971e34de38370bff24deb5a7d40ab628093d304be60946afcdb3a936",
          "76083c6d1b6809da088584674327b39488eaf665f0731151128452e04ce81bff",
          "0c7cfc0976e25ae7680cf909ae2de1859cd9b679610a14bec40d69b91785b2f6",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "26648fa2a2deb0bfc56349a590fd4cb7108a51797b634694fc02061e8d91b3576ac736a68bf848fe2a58dfb1956d266e68209a4d631e513badf8f4dcfc00f30a",
      "4f6465206f6e2061204772656369616e2055726e",
      "0185f03560de87bb2c543ef03607f3c33ac09980000de25eabe3b224312946330d2e65d192d3b4aa46ca92fc5ca50736b624402d95f6a80dc04d1f10ae9517137261",
      "04017de12ede7f72cb101dab36a111265c97b3654816dcd6183f809d4b3d111fe759497f8aefdc5dbb40d3e6d21db15bdc60f15f2a420761bcaeef73b891c2b117e9cf01e29320b799bbc86afdc5ea97d941ea1c5bd5ebeeac7a784b3bab524746f3e640ec26ee1bd91255f9330d974f845084637ee0e6fe9f505c5b87c86a4e1a6c3096dd",
      "013ef326940998544a899e15e1726548ff43bbdb23a8587aa3bef9d1b857338d87287df5667037b519d6a14661e9503cfc95a154d93566d8c84e95ce93ad05293a0b",
      "04007d419b8834e7513d0e7cc66424a136ec5e11395ab353da324e3586673ee73d53ab34f30a0b42a92d054d0db321b80f6217e655e304f72793767c4231785c4a4a6e008f31b93b7a4f2b8cd12e5fe5a0523dc71353c66cbdad51c86b9e0bdfcd9a45698f2dab1809ab1b0f88f54227232c858accc44d9a8d41775ac026341564a2d749f4",
      "001018584599625ff9953b9305849850d5e34bd789d4b81101139662fbea8b6508ddb9d019b0d692e737f66beae3f1f783e744202aaf6fea01506c27287e359fe776",
      "04015cc3636632ea9a3879e43240beae5d15a44fba819282fac26a19c989fafdd0f330b8521dff7dc393101b018c1e65b07be9f5fc9a28a1f450d6a541ee0d76221133001e8f0f6a05ab79f9b9bb9ccce142a453d59c5abebb5674839d935a3ca1a3fbc328539a60b3bc3c05fed22838584a726b9c176796cad0169ba4093332cbd2dc3a9f",
      "",
      "",
      "01fced239845e53f0ec616e71777883a1f9fcab22a50f701bdeee17ad040e44d",
      "9752b85fe8c73eda183f9e80",
      "0116aeb3a1c405c61b1ce47600b7ecd11d89b9c08c408b7e2d1e00a4d64696d12e6881dc61688209a8207427f9",
      "80466a9d9cc5112ddad297e817e038801e15fa18152bc4dc010a35d7f534089c87c98b4bacd7bbc6276c4002a74085adcd9019fca6139826b5292569cfb7fe47",
      {
          "8d78748d632f95b8ce0c67d70f4ad1757e61e872b5941e146986804b3990154b",
          "80a4753230900ea785b6c80775092801fe91183746479f9b04c305e1db9d1f4d",
          "620b176d737cf366bcc20d96adb54ec156978220879b67923689e6dca36210ed",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_AES_256_GCM_SHA384,
      "9e1d5f62cb38229f57f68948a0fbc1264499910cce50ec62cb24188c5b0a98868f3c1cfa8c5baa97b3f24db3cdd30df6e04eae83dc4347be8a981066c3b5b945",
      "4f6465206f6e2061204772656369616e2055726e",
      "003430af19716084efeced1241bb1a5625b6c826f11ef31649095eb27952619e36f62a79ea28001ac452fb20ddfbb66e62c6c0b1be03c0d28c97794a1fb638207a83",
      "04000a5096a6e6e002c83517b494bfc2e36bfb8632fae8068362852b70d0ff71e560b15aff96741ecffb63d8ac3090c3769679009ac59a99a1feb4713c5f090fc0dbed01ad73c45d29d369e36744e9ed37d12f80700c16d816485655169a5dd66e4ddf27f2acffe0f56f7f77ea2b473b4bf0518b975d9527009a3d14e5a4957e3e8a9074f8",
      "0053c0bc8c1db4e9e5c3e3158bfdd7fc716aef12db13c8515adf821dd692ba3ca53041029128ee19c8556e345c4bcb840bb7fd789f97fe10f17f0e2c6c2528072843",
      "0401655b5d3b7cfafaba30851d25edc44c6dd17d99410efbed8591303b4dbeea8cb1045d5255f9a60384c3bbd4a3386ae6e6fab341dc1f8db0eed5f0ab1aaac6d7838e00dadf8a1c2c64b48f89c633721e88369e54104b31368f26e35d04a442b0b428510fb23caada686add16492f333b0f7ba74c391d779b788df2c38d7a7f4778009d91",
      "003f64675fc8914ec9e2b3ecf13585b26dbaf3d5d805042ba487a5070b8c5ac1d39b17e2161771cc1b4d0a3ba6e866f4ea4808684b56af2a49b5e5111146d45d9326",
      "040013761e97007293d57de70962876b4926f69a52680b4714bee1d4236aa96c19b840c57e80b14e91258f0a350e3f7ba59f3f091633aede4c7ec4fa8918323aa45d5901076dec8eeb22899fda9ab9e1960003ff0535f53c02c40f2ae4cdc6070a3870b85b4bdd0bb77f1f889e7ee51f465a308f08c666ad3407f75dc046b2ff5a24dbe2ed",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "1316ed34bd52374854ed0e5cb0394ca0a79b2d8ce7f15d5104f21acdfb594286",
      "d9c64ec8deb8a0647fafe8ff",
      "942a2a92e0817cf032ce61abccf4f3a7c5d21b794ed943227e07b7df2d6dd92c9b8a9371949e65cca262448ab7",
      "6cb00ff99aebb2e4a05042ce0d048326dd2c03acd61a601b1038a65398406a96ab8b5da3187412b2324089ea16ba4ff7e6f4fe55d281fc8ae5f2049032b69ebd",
      {
          "a39502ef5ca116aa1317bd9583dd52f15b0502b71d900fc8a622d19623d0cb5d",
          "749eda112c4cfdd6671d84595f12cd13198fc3ef93ed72369178f344fe6e09c3",
          "f8b4e72cefbff4ca6c4eabb8c0383287082cfcbb953d900aed4959afd0017095",
      }
  },
  Params{
      Mode::Auth,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "470da194ec939201e6e57c36d8e67a9bdb22fdc3480172d33378c152321fe149d264100fe90e36ffb81e83cdfa8b34ae2f68691a55fe5f13edbf59cffaa4e84f",
      "4f6465206f6e2061204772656369616e2055726e",
      "0054109f96b8e312dd854ad3df5706c0ad6d19224c63d779585dd88fec37cc7fda8de430f6cb26a14110b780fd12861f3f7e5419e35d1d51f3148cb238ee73e98b1a",
      "0400d19e637f640b36e8d25a91f267ea590cbcf5e0e2a0e02ad7e486b3fe1ce34713ddda91232727274cb0d1a3e84f1543d69e8e91aa6b714d3b1d918c997a90b1936000296f83b54b7a362a87c5aef836cd81ad5f286f1bfa6a771ad1825e5f8d97c8a34883e276f9a9b1ee3ca713362a1d470951701cd6a9d16c2d44d03d0beb0041f296",
      "0118c813417d40b8edd14cec6fc04e67ede1967a9b26e8a19c20aa433251fb4dc76a7de2878177a44384800bae570da38e0f58193b6d1799227f27de33ef7eb2c76b",
      "04003c9de1cfc53be54b93f6625b07aae4e7ff8ecaebe121625ceec371c2efd83209487e83c776a36cd7937f66f829e9b2c4dcb5370d86546522210f731408f8aeeb84000e8033559064487ae5fd4748f1edbbf221ef467a3f259c5775ee79b76e12027c8e2364346f3f1bda51bd0fbab45d818a1a775ad01c06f7c8f540dd08a050605615",
      "00ab69acecec74b36e54e505c664e2f3b940a4528f9a770d9a1bbd92355d99b622fab6ffed999e8d7ec58204c49a3d53655964ff2b5396f03742c88d7e2094cb2227",
      "0400b880652e5b7de84d11246b873bb121cb99e8a2e7d884c331b1e3888f509c8131df4646f423678e85038dca6c1624e5a468c8da4d545a000ddb4269cbe96b59586001e352373c051af38e1daa8e0f42beb0642f3872f908bcf3ad674db18915c497ff5fdc088cbf346b2c13e950543867cc91f6968b59c93400e5824a0c17de3b2d7e46",
      "",
      "",
      "7d88479b678ab85e45db85f3e8b6c5c35600751973ca9929dffd743c4ffe6c1d",
      "a5c06c7297a23aa7e5009b6a",
      "39e0033eac3039372dc1ce46592c0c4dd2dcbe591e47da6b13d3845467a97379ab3ec8bb81c46ce22afee06f5f",
      "f1410ad4c18dea9338815bd3bab6851c4deae3fccdce17af3731e9f84d480658d2414868beaae9e59bc1ee4ce64b58c9f0bad942be3616ed576f1c478e403dc4",
      {
          "9905221c950d51c10e5a5db5d57282bca398bb311f64a64c2327492976b1a999",
          "e0765515034f51fdbf5e9a4de408b8e8a8c710f24266d1174f9293e256ad36cc",
          "59834b87a34da2a4a5755776433bb256f93405af062295fc8abc14f930000228",
      }
  },
  Params{
      Mode::AuthPsk,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "f762103fb8eb34ccbfa54f8ce4a8e53676ae047da7ffbdd66d68a788c8164f21dbb5312624ff8f92b768f781d2f12753a557cc511c9e5648550385be3d52393c",
      "4f6465206f6e2061204772656369616e2055726e",
      "019a861a0628b4aedc7f47f68750768b472256be2fae3ee65a29038adb18760239748c8b68be813714802429c989ef0c882271236983a67c6a80a9b47b383c215ce3",
      "0401981ba1148049207d76bc908afa6e199d66ba827942f65854d8639a412aa04414ab36c81e0b093bd4b25b4315c42199a82f639070e0c22e97eed8d37c2368e8face01c8e679eb8d192f8c894bd69fba735c8dfdd17775eb16bfedc2f9a34d7f10c6b289831ef411f9ce36ce1a1bd720f684bdfdb6502e569e4e686f967949cbeb5e2e04",
      "007f8fae6d32b959c91b3cc76b573307ab27a099ea9155a5f25fe3f2f0ec4ce323a70f3085f732d44ecd9fd36499101539f8dd9b2614e48ad15f22021fb40b480391",
      "0400afc1d169b7e8027b75156154e11b5754f13a96e548e5c47e242949f24f548e8269cb6d12d3a7533c5e13b860afc9901e7d8db21831690a5c542f4f4c6d095b025c0096b1a947ff2471242554dcfa7b7ad6cdf9c8d73fa1e106c482b6297c7ca5ec32c62fc25b7870768debf9ddd66106cb85988b97aa469f596ef23bc5af48e554a2be",
      "0147106ca69a1b194530545332d0a204c19dc51ab3b308a34bf3287ec8df8cc787d5853608ffbbf130b2816732274d6b825a28bdc279d8a01262dfeec8f945c3406c",
      "04005fcfe2fdc539fd13c51e068e4be3221d02500c47640c71a9a015ceac68b08744aa892592d1750fb270327eb436ec1bb9c481f6be3b59fc02ce524f1b97f3ae7946019043e72f8b11b60b71460ee2a5efe22d3478f503eb9ed38036e600f8491bafd0193cf772520e7464ac7b615a93bd97c9bdbd2743d91e51b69d7617dd64be8941af",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "698542f4438ffe5c9cafe3579bf9843368dc0c28e9593b9030e89152d80a9ed0",
      "383130f26a480c36c62db4f6",
      "08deb82f4f04940e7c69cb2d88dc7cf33b3c44631b3456580aa3804685c5420a55c8d4bec6b48b2c5a6e4d3a1d",
      "539c5af254cca2a5160b4f2cbc37b066206028ed9b88154dc4c91d24acd4b0a7d1251d3ee1a0577b1355f898118e4c3e55c016eedc80a029f1c508a1bb4011a4",
      {
          "02803c025e5389970a96e3f1b4fb82bffba55822a7638b0145a9386d04050810",
          "c0536f7ea79fb483d13e74fce10919def2a3b7e9de97822b475987cbf41e5739",
          "da1a7f0897e21869b5560658bfc8413f36ee79b918f30a56a1455e234ea94d0f",
      }
  },
  Params{
      Mode::Base,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "fe235ce991496c6c8395405da1c684f02206d24544d660f53412bb93bcb6ed6d1195414f020489f1c93e1df86c4d6ad71b7052b77e17f81960cb1b920edcedbc",
      "4f6465206f6e2061204772656369616e2055726e",
      "013fed613d0626aa01fef8d67ff6a683a9a173ce683fba2e833b1e4edf8598234736149ef1e91088e78a91ac4691d0f45e22505dc818a934897981955b3c6d4c080d",
      "0400edc201c9b32988897a7f7b19104ebb54fc749faa41a67e9931e87ec30677194898074afb9a5f40a97df2972368a0c594e5b60e90d1ff83e9e35f8ff3ad200fd6d70028b5645debe9f1f335dbc1225c066218e85cf82a05fbe361fa477740b906cb3083076e4d17232513d102627597d38e354762cf05b3bd0f33dc4d0fb78531afd3fd",
      "015b59f17366a1d4442e5b92d883a8f35fe8d88fea0e5bac6dfac7153c78fd0c6248c618b083899a7d62ba6e00e8a22cdde628dd5399b9a3377bb898792ff6f54ab9",
      "040084698a47358f06a92926ee826a6784341285ee45f4b8269de271a8c6f03d5e8e24f628de13f5c37377b7cabfbd67bc98f9e8e758dfbee128b2fe752cd32f0f3ccd0061baec1ed7c6b52b7558bc120f783e5999c8952242d9a20baf421ccfc2a2b87c42d7b5b806fea6d518d5e9cd7bfd6c85beb5adeb72da41ac3d4f27bba83cff24d7",
      "",
      "",
      "",
      "",
      "a0a8a428a5149b3ac93e07bbe8868945972a8964956fac14fc6a79e5c279d836",
      "9deefcbfd747d7a666450f00",
      "16d0a57d7dc5106a947b8ed6cb759af864fe8f60aa7f7e4665df083167aebecc9e423badf1ccb4937ac4ee96df",
      "bd98618e98c9856ca25cd63d9a72c3ef99af7fe55e29a8cc6773e315a670637bb07017ffbab0cf5e5a17aa0f63a6f3527d7f1725b28f92407fc27dbd6f34bffd",
      {
          "d8aebaa0381ef749d2108fea259d078bbb0941f6bd24a8a537f757a8e1a1a0c5",
          "48e64963c4941cea9a492567ceac487e8dbc4ef2582776cc395a775b9ac5093f",
          "2d712f50c15cced5f3f83f19b3925ef77c577a19f64eb29fa7d51feacd71d94b",
      }
  },
  Params{
      Mode::Psk,
      NamedGroup::secp521r1,
      HashFunction::Sha512,
      CipherSuite::TLS_CHACHA20_POLY1305_SHA256,
      "ebbd082d1fcf9eac2304cb48d70f2406f0f8a18f54a344c4d947a9e788a23954e0abee03bc886ea4efa8d6905f74defec757118dd98f79168f27547d896db339",
      "4f6465206f6e2061204772656369616e2055726e",
      "00545c9a59cb75f01a70a597911606380d21dade65949c2746e18ebe945f9b4cfa16c34a22739e477ee24bdd5662d2a9b765b7eb3225fc58fab27959a6d416f53525",
      "040073046b12656d7bdbf4ddf4f38f6f657861793f26f61fb5ce68798b8dab3ca239e4717ad4e76b807970f0bd353224ff48075415f41af17bb2a6845f47cb239d1dee001e311f82795bc49f5df716d2a38251cd2b9e9eb5e310f9078ff75a7f0615332571ec2a6d26e92a75988bf28b60f1a197dbfe06f26250666f04ed163207934142ab",
      "00fd82ee56c24eb02563aa1a5a4e082687f4dd2b6e5696255025cb688fccc81a673035060982e0269b68d80ff1dc7cdc2f5b15e2db20dc59bc0d4810efd35e963acb",
      "0401ab406318b4ee13c97b3154665b517cbf26cb507923cc617934fc77deff9470df98af6483285f6ce82e01f02c3529a2762294415626d9110b9cc34e26c1ccf7050b014f64fba39a23215af98ec36a2a32f18e57cb4d4c29fa4f1e65fb9b3b23bd710615034937f3a3cd2b8c97f34d759edaec1e75e60fc3288cd46e640aec92146dfc3e",
      "",
      "",
      "0247fd33b913760fa1fa51e1892d9f307fbe65eb171e8132c2af18555a738b82",
      "456e6e796e20447572696e206172616e204d6f726961",
      "e18b5c59550a61f02dd5b9e48489590731028a3a138155e00d943291bbaed34b",
      "04c09a0a7e9194a1a1730e95",
      "268e957e2b55b77a1737826c1164f1bf157c237a12f6a08354b8860529aff59be21b1940f729a38dcaa6a2083c",
      "cdff6de2b9d6190587f29c0fc7c1c2dad5bf278feb9223e3fd15a11186eeaf9f78e37cf082f44c44ecb7326cec825aab12dbfd8e3e528e2ed307107dab94a74b",
      {
          "e5cb78308c42b15722b1f446d597a97cba9d7efa2811c93a3d287667f5a93517",
          "740772bfa151260eb96de2cdf303231bbbf98a4c8676eb42a6619eb929ac1f61",
          "83ac3835390f7317131823b89b27391c53b29174d6eb7403607c410ce3ed5124",
      }
  }
};
// clang-format on

void testExportValues(
    std::unique_ptr<HpkeContext>& context,
    const std::vector<std::string>& exportValues) {
  const size_t exportLength = 32;
  const std::vector<std::string> contexts = {
      "", "00", "54657374436f6e74657874"};
  ASSERT_EQ(exportValues.size(), contexts.size());

  for (size_t testNum = 0; testNum < contexts.size(); ++testNum) {
    std::unique_ptr<folly::IOBuf> exporterContext =
        toIOBuf(contexts.at(testNum));
    auto secret =
        context->exportSecret(std::move(exporterContext), exportLength);
    auto expectedSecret = toIOBuf(exportValues.at(testNum));

    EXPECT_TRUE(folly::IOBufEqualTo()(secret, expectedSecret));
  }
}

MATCHER_P(TrafficKeyMatcher, expectedKey, "") {
  return folly::IOBufEqualTo()(expectedKey->iv, arg->iv) &&
      folly::IOBufEqualTo()(expectedKey->key, arg->key);
}

class HpkeMockX25519KeyExchange : public X25519KeyExchange {
 public:
  MOCK_METHOD(void, generateKeyPair, ());
};

class HpkeMockOpenSSLECKeyExchange : public openssl::OpenSSLECKeyExchange {
 public:
  using OpenSSLECKeyExchange::OpenSSLECKeyExchange;

  MOCK_METHOD0(generateKeyPair, void());

  template <typename T>
  static std::unique_ptr<HpkeMockOpenSSLECKeyExchange> makeMockKeyExchange() {
    return std::make_unique<HpkeMockOpenSSLECKeyExchange>(
        openssl::Properties<T>::curveNid, T::keyShareLength);
  }
};

template <class T>
void setOpenSSLPrivKey(
    openssl::OpenSSLECKeyExchange* kex,
    std::string& privKey,
    std::string& pubKey) {
  auto pubKeyBuf = toIOBuf(pubKey);
  std::string privKeyBin = folly::unhexlify(privKey);
  auto key = fizz::openssl::detail::decodeECPublicKey(
      pubKeyBuf->coalesce(), openssl::Properties<T>::curveNid);
  folly::ssl::EcKeyUniquePtr ecKey(EVP_PKEY_get1_EC_KEY(key.get()));
  folly::ssl::BIGNUMUniquePtr privateBn(
      BN_bin2bn((uint8_t*)privKeyBin.c_str(), privKeyBin.size(), nullptr));
  if (EC_KEY_set_private_key(ecKey.get(), privateBn.get()) != 1) {
    throw std::runtime_error("Failed to set private key");
  }
  if (EC_KEY_check_key(ecKey.get()) != 1) {
    throw std::runtime_error("Private key not valid");
  }
  kex->setPrivateKey(std::move(key));
}

std::unique_ptr<KeyExchange> generateAuthKex(
    NamedGroup group,
    std::string privateKey,
    std::string publicKey) {
  if (privateKey.empty()) {
    return nullptr;
  }
  switch (group) {
    case NamedGroup::x25519: {
      auto dKex = std::make_unique<X25519KeyExchange>();
      dKex->setKeyPair(toIOBuf(privateKey), toIOBuf(publicKey));
      return dKex;
    }
    case NamedGroup::secp256r1: {
      auto dKex = openssl::makeOpenSSLECKeyExchange<fizz::P256>();
      setOpenSSLPrivKey<fizz::P256>(dKex.get(), privateKey, publicKey);
      return dKex;
    }
    case NamedGroup::secp384r1: {
      auto dKex = openssl::makeOpenSSLECKeyExchange<fizz::P384>();
      setOpenSSLPrivKey<fizz::P384>(dKex.get(), privateKey, publicKey);
      return dKex;
    }
    case NamedGroup::secp521r1: {
      auto dKex = openssl::makeOpenSSLECKeyExchange<fizz::P521>();
      setOpenSSLPrivKey<fizz::P521>(dKex.get(), privateKey, publicKey);
      return dKex;
    }
    default:
      return nullptr;
  }
}

std::unique_ptr<fizz::hpke::Hkdf> genHKDF(HashFunction f) {
  return fizz::hpke::makeHpkeHkdf(
      folly::IOBuf::copyBuffer("HPKE-v1"), getKDFId(f));
}

SetupParam getSetupParam(
    std::unique_ptr<KeyExchange> kex,
    NamedGroup group,
    HashFunction func,
    CipherSuite suite,
    std::string privateKey,
    std::string publicKey,
    std::string authPrivateKey,
    std::string authPublicKey,
    std::unique_ptr<MockAeadCipher> cipher) {
  switch (group) {
    case NamedGroup::x25519: {
      auto dKex = dynamic_cast<X25519KeyExchange*>(kex.get());
      CHECK(dKex);
      dKex->setKeyPair(toIOBuf(privateKey), toIOBuf(publicKey));
      break;
    }
    case NamedGroup::secp256r1: {
      auto dKex = dynamic_cast<openssl::OpenSSLECKeyExchange*>(kex.get());
      CHECK(dKex);
      setOpenSSLPrivKey<fizz::P256>(dKex, privateKey, publicKey);
      break;
    }
    case NamedGroup::secp384r1: {
      auto dKex = dynamic_cast<openssl::OpenSSLECKeyExchange*>(kex.get());
      CHECK(dKex);
      setOpenSSLPrivKey<fizz::P384>(dKex, privateKey, publicKey);
      break;
    }
    case NamedGroup::secp521r1: {
      auto dKex = dynamic_cast<openssl::OpenSSLECKeyExchange*>(kex.get());
      CHECK(dKex);
      setOpenSSLPrivKey<fizz::P521>(dKex, privateKey, publicKey);
      break;
    }
    default:
      throw std::runtime_error("unsupported group");
  }

  std::unique_ptr<folly::IOBuf> suiteId =
      generateHpkeSuiteId(group, func, suite);

  return SetupParam{
      std::make_unique<DHKEM>(
          std::move(kex),
          generateAuthKex(group, authPrivateKey, authPublicKey),
          group,
          genHKDF(getHashFunctionForKEM(getKEMId(group)))),
      std::move(cipher),
      genHKDF(func),
      std::move(suiteId),
  };
}

TEST(HpkeTest, TestSetup) {
  for (const auto& testParam : HPKETestCases) {
    auto pkR = toIOBuf(testParam.pkR);
    auto info = toIOBuf(testParam.info);
    auto pkS = toIOBuf(testParam.pkS);

    auto encapCipher =
        std::make_unique<MockAeadCipher>(getCipher(testParam.suite));
    TrafficKey encapExpectedTrafficKey{
        toIOBuf(testParam.key), toIOBuf(testParam.iv)};
    EXPECT_CALL(
        *encapCipher, _setKey(TrafficKeyMatcher(&encapExpectedTrafficKey)))
        .Times(1);

    std::unique_ptr<KeyExchange> encapKex;
    switch (testParam.group) {
      case NamedGroup::x25519: {
        auto kex = std::make_unique<HpkeMockX25519KeyExchange>();
        EXPECT_CALL(*kex, generateKeyPair()).Times(1);
        encapKex = std::move(kex);
        break;
      }
      case NamedGroup::secp256r1: {
        auto kex =
            HpkeMockOpenSSLECKeyExchange::makeMockKeyExchange<fizz::P256>();
        EXPECT_CALL(*kex, generateKeyPair()).Times(1);
        encapKex = std::move(kex);
        break;
      }
      case NamedGroup::secp384r1: {
        auto kex =
            HpkeMockOpenSSLECKeyExchange::makeMockKeyExchange<fizz::P384>();
        EXPECT_CALL(*kex, generateKeyPair()).Times(1);
        encapKex = std::move(kex);
        break;
      }
      case NamedGroup::secp521r1: {
        auto kex =
            HpkeMockOpenSSLECKeyExchange::makeMockKeyExchange<fizz::P521>();
        EXPECT_CALL(*kex, generateKeyPair()).Times(1);
        encapKex = std::move(kex);
        break;
      }
      default:
        throw std::runtime_error("not supported");
    }

    SetupResult setupResult = setupWithEncap(
        testParam.mode,
        pkR->coalesce(),
        info->clone(),
        PskInputs(
            testParam.mode, toIOBuf(testParam.psk), toIOBuf(testParam.pskId)),
        getSetupParam(
            std::move(encapKex),
            testParam.group,
            testParam.hash,
            testParam.suite,
            testParam.skE,
            testParam.pkE,
            testParam.skS,
            testParam.pkS,
            std::move(encapCipher)));
    auto encryptContext = std::move(setupResult.context);

    auto enc = std::move(setupResult.enc);
    auto decapCipher =
        std::make_unique<MockAeadCipher>(getCipher(testParam.suite));
    TrafficKey decapExpectedTrafficKey{
        toIOBuf(testParam.key), toIOBuf(testParam.iv)};
    EXPECT_CALL(
        *decapCipher, _setKey(TrafficKeyMatcher(&decapExpectedTrafficKey)))
        .Times(1);

    std::unique_ptr<KeyExchange> decapKex;
    switch (testParam.group) {
      case NamedGroup::x25519: {
        decapKex = std::make_unique<X25519KeyExchange>();
        break;
      }
      case NamedGroup::secp256r1: {
        decapKex = openssl::makeKeyExchange<fizz::P256>();
        break;
      }
      case NamedGroup::secp384r1: {
        decapKex = openssl::makeKeyExchange<fizz::P384>();
        break;
      }
      case NamedGroup::secp521r1: {
        decapKex = openssl::makeKeyExchange<fizz::P521>();
        break;
      }
      default:
        throw std::runtime_error("unsupported");
    }

    auto decryptContext = setupWithDecap(
        testParam.mode,
        enc->coalesce(),
        pkS->coalesce(),
        std::move(info),
        PskInputs(
            testParam.mode, toIOBuf(testParam.psk), toIOBuf(testParam.pskId)),
        getSetupParam(
            std::move(decapKex),
            testParam.group,
            testParam.hash,
            testParam.suite,
            testParam.skR,
            testParam.pkR,
            testParam.skS,
            testParam.pkS,
            std::move(decapCipher)));

    // Test encrypt/decrypt
    std::unique_ptr<folly::IOBuf> aad = toIOBuf("436f756e742d30");
    std::unique_ptr<folly::IOBuf> plaintext =
        toIOBuf("4265617574792069732074727574682c20747275746820626561757479");

    auto ciphertext = encryptContext->seal(aad.get(), plaintext->clone());
    auto expectedCiphertext = testParam.ciphertext;
    EXPECT_TRUE(folly::IOBufEqualTo()(ciphertext, toIOBuf(expectedCiphertext)));

    auto gotPlaintext = decryptContext->open(aad.get(), std::move(ciphertext));
    EXPECT_TRUE(folly::IOBufEqualTo()(gotPlaintext, plaintext));

    // Test exporter secret
    auto gotExporterSecretE = encryptContext->getExporterSecret();
    auto gotExporterSecretD = decryptContext->getExporterSecret();
    auto expectedExporterSecret = toIOBuf(testParam.exporterSecret);
    EXPECT_TRUE(
        folly::IOBufEqualTo()(gotExporterSecretE, expectedExporterSecret));
    EXPECT_TRUE(
        folly::IOBufEqualTo()(gotExporterSecretD, expectedExporterSecret));

    // Test export values
    testExportValues(encryptContext, testParam.exportValues);
    testExportValues(decryptContext, testParam.exportValues);
  }
}

TEST(HpkeTest, TestKeySchedule) {
  for (const auto& testParam : HPKETestCases) {
    auto hkdf = genHKDF(testParam.hash);

    std::unique_ptr<MockAeadCipher> cipher =
        std::make_unique<MockAeadCipher>(getCipher(testParam.suite));
    std::unique_ptr<folly::IOBuf> suiteId =
        generateHpkeSuiteId(testParam.group, testParam.hash, testParam.suite);
    TrafficKey expectedTrafficKey{
        toIOBuf(testParam.key), toIOBuf(testParam.iv)};
    EXPECT_CALL(*cipher, _setKey(TrafficKeyMatcher(&expectedTrafficKey)))
        .Times(1);

    struct KeyScheduleParams keyScheduleParams {
      testParam.mode, toIOBuf(testParam.sharedSecret), toIOBuf(testParam.info),
          PskInputs(
              testParam.mode,
              toIOBuf(testParam.psk),
              toIOBuf(testParam.pskId)),
          std::move(cipher), std::move(hkdf), std::move(suiteId),
          fizz::hpke::HpkeContext::Role::Sender
    };
    auto context = keySchedule(std::move(keyScheduleParams));

    EXPECT_TRUE(folly::IOBufEqualTo()(
        context->getExporterSecret(), toIOBuf(testParam.exporterSecret)));
  }
}

} // namespace test
} // namespace hpke
} // namespace fizz
