// -*- mode: c++; c-indent-level: 4; c++-member-init-indent: 8; comment-column: 35; -*-

// EO
#include <eo>
#include <es.h>
#include <eoScalarFitness.h>
#include <utils/eoRealVectorBounds.h>

// core templates
#include <PO.h>
#include <eoRealParticle.h>
#include <eoBitParticle.h>

// the PSO algorithms
#include <eoEasyPSO.h>
#include <eoSSPSO.h>
#include <eoLSPSO.h>
#include <eoSyncEasyPSO.h>

// flights
#include <eoFlight.h>
#include <eoSigBinaryFlight.h>

// initialization utilities
#include <eoParticleBestInit.h>
#include <eoVelocityInit.h>
#include <psoApply.h>

// velocities
#include <eoVariableInertiaWeightedVelocity.h>
#include <eoFixedInertiaWeightedVelocity.h>
#include <eoConstrictedVelocity.h>
#include <eoConstrictedVariableWeightVelocity.h>
#include <eoStandardVelocity.h>

// topologies
#include <eoStarTopology.h>
#include <eoLinearTopology.h>

// others
#include <eoRandomRealWeightUp.h>
#include <eoGaussRealWeightUp.h>
#include <eoRealBoundModifier.h>


using namespace std;

// real particle
typedef eoRealParticle < eoMinimizingFitness >Indi;

// bit particle
typedef eoBitParticle < eoMinimizingFitness > binaryIndi;


double f1 (const Indi & _indi)
{
    double sum = 0;
    for (unsigned i = 0; i < _indi.size ()-1; i++)
        sum += pow(_indi[i],2);
    return (sum);
}

// return the number of "1" contained in a bit string
double binary_value(const binaryIndi & _indi)
{
  double sum = 0;
  for (unsigned i = 0; i < _indi.size(); i++)
    sum += _indi[i];
  return sum;
}


int
main (int __argc, char *__argv[])
{

    /********************** Common definitions ***********/

    const unsigned int VEC_SIZE = 2;	// Number of object variables
    const unsigned int BIN_VEC_SIZE = 8; // for the binary example

    const unsigned int POP_SIZE = 20;	// Size of population
    const unsigned int NEIGHBORHOOD_SIZE= 5; // for topologies
    const unsigned int MAX_GEN = 50;

    const double INIT_POSITION_MIN = -5.0;
    const double INIT_POSITION_MAX = 5.0;
    const double INIT_VELOCITY_MIN = -1;
    const double INIT_VELOCITY_MAX = 1;
    const double C1 = 2; // learning factors
    const double C2 = 2;
   
    // seed
    rng.reseed (time(0));

    //evaluation
    eoEvalFuncPtr<Indi, double, const Indi& > plainEval(  f1 );
    eoEvalFuncCounter < Indi > eval (plainEval);

    // position init
    eoUniformGenerator < double >uGen (INIT_POSITION_MIN, INIT_POSITION_MAX);
    eoInitFixedLength < Indi > random (VEC_SIZE, uGen);

    // velocity init
    eoUniformGenerator < double >sGen (INIT_VELOCITY_MIN, INIT_VELOCITY_MAX);
    eoVelocityInitFixedLength < Indi > veloRandom (VEC_SIZE, sGen);

    // local best init
    eoFirstIsBestInit < Indi > localInit;

    // flight
    eoRealVectorBounds bndsFlight(VEC_SIZE,INIT_POSITION_MIN,INIT_POSITION_MAX);
    eoStandardFlight < Indi > flight(bndsFlight);	


    /*********************** 1) TEST VARIABLE-WEIGHT-VELOCITY **********************/
    eoPop < Indi > pop1;

    // initialize the population, there will have a eoParticleFullInitializer later for this stuff
    pop1.append (POP_SIZE, random);
    apply < Indi > (eval, pop1);
    apply < Indi > (veloRandom, pop1);
    apply < Indi > (localInit, pop1);

    //define the topology
    eoLinearTopology<Indi> topology1(NEIGHBORHOOD_SIZE);
    topology1.setup(pop1);

    // define the bounds and how you want them to be updated
    eoRealVectorBounds bnds1(VEC_SIZE,INIT_VELOCITY_MIN,INIT_VELOCITY_MAX);
    eoRandomRealWeightUp weightUp1(0,1);

    // define the velocity
    eoValueParam<unsigned> generationCounter1(0, "Gen.");
    eoExpDecayingBoundModifier boundsModifier1(MAX_GEN,MAX_GEN/2,generationCounter1);
    eoVariableInertiaWeightedVelocity < Indi > velocity1 (topology1,weightUp1,C1,C2,bnds1,boundsModifier1);

    // define the continuator
    eoGenContinue < Indi > genCont1 (MAX_GEN);
    eoCheckPoint<Indi> checkpoint1(genCont1);
    eoIncrementor<unsigned> increment1(generationCounter1.value());
    checkpoint1.add(increment1);

    // the PSO
    eoEasyPSO < Indi > psa1 (checkpoint1, eval, velocity1, flight);
    psa1 (pop1); // run it !
    pop1.sort (); // sort the pop
    std::cout << "Final pop1 :\n" << pop1 << std::endl; // print the sorted final pop


  
    /*********************** 2) TEST EOSSPSO **********************/
    eoPop < Indi > pop2;

    pop2.append (POP_SIZE, random);
    apply < Indi > (eval, pop2);
    apply < Indi > (veloRandom, pop2);
    apply < Indi > (localInit, pop2);

    // the bounds ..
    eoRealVectorBounds bnds2(VEC_SIZE,INIT_VELOCITY_MIN,INIT_VELOCITY_MAX);
    eoValueParam<unsigned> generationCounter2(0, "Gen.");

    // define the continuator
    eoGenContinue < Indi > genCont2 (MAX_GEN);
    eoCheckPoint<Indi> checkpoint2(genCont2);
    eoIncrementor<unsigned> increment2(generationCounter2.value());
    checkpoint2.add(increment2);

    // the PSO
    eoSSPSO < Indi > psa2 (checkpoint2, eval,C1,C2,bnds2);
    psa2 (pop2); // run it 
    pop2.sort (); // sort the pop
    std::cout << "Final pop2 :\n" << pop2 << std::endl; // print the sorted final pop

    
    /*********************** 3) TEST EOSYNCEASYPSO **********************/
    eoPop < Indi > pop3;

    pop3.append (POP_SIZE, random);    
    apply < Indi > (eval, pop3);
    apply < Indi > (veloRandom, pop3);
    apply < Indi > (localInit, pop3);

    //define the topology
    eoLinearTopology<Indi> topology3(NEIGHBORHOOD_SIZE);
    topology3.setup(pop3);

    // define the velocity
    eoRealVectorBounds bnds3(VEC_SIZE,INIT_VELOCITY_MIN,INIT_VELOCITY_MAX);
    eoStandardVelocity < Indi > velocity3 (topology3,C1,C2,bnds3);

    // define the continuator
    eoGenContinue < Indi > genCont3 (MAX_GEN);
    eoCheckPoint<Indi> checkpoint3(genCont3);

    // the synchronous PSO
    eoSyncEasyPSO < Indi > psa3 (checkpoint3, eval, velocity3, flight);

    psa3 (pop3); // run it !
    pop3.sort (); // sort the pop
    std::cout << "Final pop3 :\n" << pop3 << std::endl; // print the sorted final pop
    
    
        
    /*********************** 4) TEST BINARY-FLIGHT-1 **********************/
    eoPop < binaryIndi > pop4;

    //evaluation
    eoEvalFuncPtr<binaryIndi, double, const binaryIndi& > binEval(  binary_value );
    eoEvalFuncCounter < binaryIndi > evalBinary (binEval);

    // position init - there will have a bit position initializer
    for (unsigned int igeno=0; igeno<POP_SIZE; igeno++)
    {
      binaryIndi v;           
      for (unsigned ivar=0; ivar<BIN_VEC_SIZE; ivar++)
        {
          bool r = rng.flip(); // new value, random in {0,1}
          v.push_back(r);      // append that random value to v
        }
      evalBinary(v);                 // evaluate it
      pop4.push_back(v);        // and put it in the population
    }
    
    // velocity init
    eoVelocityInitFixedLength < binaryIndi > veloBinRandom (BIN_VEC_SIZE, sGen);

    // local best init
    eoFirstIsBestInit < binaryIndi > localBinInit;
    
    apply < binaryIndi > (binEval, pop4);
    apply < binaryIndi > (veloBinRandom, pop4);
    apply < binaryIndi > (localBinInit, pop4);

    //define the topology.
    eoLinearTopology<binaryIndi> topology4(NEIGHBORHOOD_SIZE);
    topology4.setup(pop4);

    // define the velocity
    eoRealVectorBounds bnds4(BIN_VEC_SIZE,INIT_VELOCITY_MIN,INIT_VELOCITY_MAX);
    eoRandomRealWeightUp weightUp4(0,1);

    eoValueParam<unsigned> generationCounter4(0, "Gen.");
    eoExpDecayingBoundModifier boundsModifier4(MAX_GEN,MAX_GEN/2,generationCounter4);
    eoConstrictedVariableWeightVelocity < binaryIndi > velocity4 (topology4,0.97,weightUp4,C1,C2,bnds4,boundsModifier1);
    
    // flight
    eoSigBinaryFlight < binaryIndi > binFlight;
    
    // define the continuator
    eoGenContinue < binaryIndi > genCont4 (MAX_GEN);
    eoCheckPoint<binaryIndi> checkpoint4(genCont4);

    // the PSO
    eoEasyPSO < binaryIndi > psa4 (checkpoint4, evalBinary, velocity4, binFlight);
    psa4 (pop4); // run it !
    pop4.sort (); // sort the pop
    std::cout << "Final pop4 :\n" << pop4 << std::endl; // print the sorted final pop
}
