/*
 *  Copyright (c) 2012 Shirou Maruyama
 * 
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 * 
 *   1. Redistributions of source code must retain the above Copyright
 *      notice, this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above Copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the name of the authors nor the names of its contributors
 *      may be used to endorse or promote products derived from this
 *      software without specific prior written permission.
 */

#ifndef COMP_PERM_IDX_HPP_
#define COMP_PERM_IDX_HPP_

#include <vector>
#include <string>
#include <iostream>
#include <stdint.h>
#include "HufWatTree.hpp"

namespace cpi00 {

const unsigned char kDelimiter = 0;
const unsigned char kTerminator = 255;

#ifndef UINT64_MAX
  const uint64_t UINT64_MAX = (18446744073709551615ULL);
#endif

class CompPermIdx {
public:

  /**
   * Constructor
   */
  CompPermIdx() : WT_(), C_(), NumKeys_(0) {}

  /**
   * Build index.
   * @param keywords The input key list
   */
  void Build(std::vector<std::string>& keywords);

  /**
   * Rank(keyword) returns the rank of "keyword" in lexicographic order.
   * @param str THe query string. 
   */
  uint64_t Rank(const std::string& str);

  /**
   * Select(i) returns the ith keyword in lexicographic order.
   * @param i The position
   * @param ret The returned string. 
   */
  void Select(const uint64_t i, std::string& ret);

  /**
   * Membership(keywords) returns true if the keyword is in dictionary.
   * @param str The query string.
   */
  bool Membership(const std::string& str);

  /**
   * PrefixSearch("prefix") returns the string set prefixed by "prefix".
   * @param prefix The prefix query.
   * @param ret The returned string set.
   */
  void PrefixSearch(const std::string& prefix,
                    std::vector<std::string>& ret) const;

  /**
   * SuffixSearch("suffix") returns the string set prefixed by "suffix".
   * @param suffix The suffix query.
   * @param ret The returned string set.
   */
  void SuffixSearch(const std::string& suffix,
                    std::vector<std::string>& ret) const;

  /**
   * PrefixSuffixSearch("prefix", "suffix") returns 
   *  the string set prefixed by "prefix" and suffixed by "suffix".
   * @param prefix The prefix query.
   * @param suffix The suffix query.
   * @param ret The returned string set.
   */
  void PrefixSuffixSearch(const std::string& prefix,
                          const std::string& suffix,
                          std::vector<std::string>& ret) const;

  /**
   * SubstringSearch("substring") returns 
   *  the string set including "substring" as substring.
   * @param substring The subsring query.
   * @param ret The returned string set.
   */
  void SubstringSearch(const std::string& substring,
                       std::vector<std::string>& ret) const;

  /**
   * Read the index from istream, and return the index size in bytes. 
   * @param is The istream.
   */
  uint64_t Read(std::istream& is);

  /**
   * Write the index from ostream, and return the index size in bytes. 
   * @param is The ostream.
   */
  uint64_t Write(std::ostream& os) const;

  /**
   * Clear the index.
   */
  void Clear();

  /**
   * Return the number of registered keys.
   */
  uint64_t NumKeys() const { return NumKeys_; }

private:
  HufWatTree WT_;
  std::vector<uint64_t> C_;
  uint64_t NumKeys_;

  uint64_t Jump2End(const uint64_t pos) const;
  uint64_t BackStep(const uint64_t pos) const;
  uint64_t BackPermSearch(const std::vector<uint8_t>& query,
                          uint64_t& first, uint64_t& last) const;
  void GetString(const uint64_t pos, std::string& ret) const;
  void GetStringSet(const uint64_t first, const uint64_t last, 
                    std::vector<std::string>& ret) const;
  bool GetStringWithFilter(const uint64_t pos, std::string& ret, 
                           const uint64_t first, const uint64_t last,
                           std::vector<bool>& marks) const;
  void GetStringSetWithFilter(const uint64_t first, const uint64_t last,
                              std::vector<std::string>& ret) const;

  }; // class CompPermIdx

} // namespace cpi00

#endif // COMP_PERM_IDX_HPP_
