
/*
 * Copyright 2004-2019 Cray Inc.
 * Other additional copyright holders may be indicated within.
 * 
 * The entirety of this work is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * 
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
    /*

Chapel idiomatic wrappers for the LAPACK library.

.. note::

  Because of the volume of procedures provided, and because their behavior is virtually unchanged, in-depth documentation on each procedure's purpose and use is not given here.
  
Consult the `Netlibs LAPACK <https://www.netlib.org/lapack/>`_ site, and the
`Netlibs <https://www.netlib.org/lapack/explore-html/globals_func.html>`_ and
`Intel
<https://software.intel.com/en-us/mkl-developer-reference-fortran-lapack-routines>`_
LAPACK documents for that information.

Additionally, only a small set of LAPACK procedures have been tested for correctness.

Differences from LAPACK
-----------------------

The `XYYZZZ <https://www.netlib.org/lapack/individualroutines.html>`_ naming convention for procedures has been only slightly modified with the removal of the type identifier in the procedure name,
creating a set of polymorphic procedures under the name YYZZZ.

Procedure arguments have been simplified. 
Using the original Fortran documentation, groupings of arguments pertaining to matrices and arrays were found, and encapsulated by Chapel arrays.

For example, instead of 

.. code-block:: chapel

  LAPACKE_sgesv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

the arguments n, nrhs, lda, and ldb can be encapsulated by properties of Chapel arrays and their domains, giving the LAPACK procedure:

.. code-block:: chapel

  gesv(matrix_order : lapack_memory_order, a : [] real(32), ipiv : [] c_int, b : [] real(32) ): c_int;

This requires that your matrices are stored in a two dimensional form.

Type Adjustments
----------------

The LAPACKE types ``lapack_int``, ``lapack_float``, ``lapack_double``, ``lapack_complex_float``, ``lapack_complex_double``, and ``lapack_logical`` are not defined at all, but rather are replaced by the types ``c_int``, ``real(32)``, ``real(64)``, ``complex(64)``, ``complex(128)``, and ``c_int`` respectively.



The :enum:`lapack_memory_order` enumeration is a replacement for the ``c_int`` constants :data:`LAPACK_ROW_MAJOR` and :data:`LAPACK_COL_MAJOR` defined by LAPACK and used by ``matrix_order`` which is an argument present in essentially all procedures.
The value of each enumeration is equivalent (both in type and value) of the LAPACK constants.
However the constants are also provided and can be used as well.

The ``LAPACK_SELECT`` types are function pointer types, whose procedures need to be declared and implemented in C, then be declared as ``extern var`` on the Chapel side. 
See the dgees test (``$CHPL_HOME/test/library/packages/LAPACK/dgees.chpl``) for a concrete example.

Compiling with LAPACK
------------------------

Using this module require that you have LAPACK (binaries and C interface) on your system, as it is not bundled with Chapel. 

You can download and build the `Netlibs LAPACK <https://www.netlib.org/lapack/>`_ , if it is not already installed, and this section will assume usage of the Netlibs LAPACK.
You must also have libgfortran installed on your system (typically installed along with gcc).

To compile with LAPACK, Chapel needs to know:

1. Where the LAPACKE header (lapacke.h) is.

2. Where the libgfortran binary is (sometimes it is not to be found by ``ld``).

3. Where the various LAPACK binaries (lapacke, lapack, and refblas) are.

Once the details are worked out, compiling is quite simple and nearly identical to how one would compile a C program to work with LAPACK.

.. code-block:: sh

  chpl -I$PATH_TO_LAPACKE_INCLUDE_DIR \
       -L$PATH_TO_LIBGFORTRAN -lgfortran \
       -L$PATH_TO_LAPACK_BINARIES -llapacke -llapack -lrefblas \
       source.chpl

As an example,

.. code-block:: sh

  chpl -I$HOME/LAPACK/lapacke/include \
       -L/usr/lib/gcc/stuff -lgfortran \
       -L$HOME/LAPACK -llapacke -llapack -lrefblas \
       source.chpl

would be the command to use if Netlibs LAPACK had been built in ``$HOME/LAPACK``, and the libgfortran binary found to be located in ``/usr/lib/gcc/stuff``

You can refer to the COMPOPTS file of the LAPACK test suit (``$CHPL_HOME/test/library/packages/LAPACK/``) for an example

Future Work
-----------

We anticipate the following additions:

1. Better documentation on each individual procedure, mostly derived from the original Fortran documentation.

2. Larger test coverage of the provided procedures.

3. Enumerated values for arguments of procedures that currently take strings to denote different options.


*/
module LAPACK {

  /* Available LAPACK implementations for ``lapackImpl`` */
  enum LapackImpl {lapack, mkl, off};
  use LapackImpl;

  /*
    Specifies which header filename to include, based on the lapack
    implementation.

    Most LAPACK implementations rely on ``lapacke.h``, which is used when
    ``lapackImpl = lapack``, the default setting.

     - ``LapackImpl.lapack`` includes ``lapacke.h`` (default)
     - ``LapackImpl.mkl`` includes ``mkl_lapacke.h``
     - ``LapackImpl.off`` includes nothing

    .. warning::

      MKL does not currently support the low-level ``LAPACK_*`` interface, due
      to breaking naming conventions.

  */
  config param lapackImpl = LapackImpl.lapack;

  /*
     Manually specifies the header filename to include. This flag overrides.
     the header determined by ``lapackImpl``.

     This flag should only be necessary if using an ``LAPACK`` implementation
     with a unique header name that is not supported by ``lapackImpl``.
     However, no guarantees can be made about this module working with untested
     implementations.

  */
  config param lapackHeader = '';

  pragma "no doc"
  param header = if lapackHeader == '' then
                   if lapackImpl == LapackImpl.off then ''
                   else if lapackImpl == LapackImpl.mkl  then 'mkl_lapacke.h'
                   else 'lapacke.h'
                 else lapackHeader;

  use SysCTypes;

  /* Return `true` if type is supported by LAPACK */
  proc isLAPACKType(type t) param: bool {
    return isRealType(t) || isComplexType(t);
  }

/*External function pointer type LAPACK_C_SELECT1.*/
extern type LAPACK_C_SELECT1 ;
/*External function pointer type LAPACK_Z_SELECT1.
*/
extern type LAPACK_Z_SELECT1 ;
/*External function pointer type LAPACK_C_SELECT2.*/
extern type LAPACK_C_SELECT2 ;
/*External function pointer type LAPACK_D_SELECT2.*/
extern type LAPACK_D_SELECT2 ;
/*External function pointer type LAPACK_S_SELECT2.*/
extern type LAPACK_S_SELECT2 ;
/*External function pointer type LAPACK_Z_SELECT2.*/
extern type LAPACK_Z_SELECT2 ;
/*External function pointer type LAPACK_S_SELECT3.*/
extern type LAPACK_S_SELECT3 ;
/*External function pointer type LAPACK_D_SELECT3.*/
extern type LAPACK_D_SELECT3 ;


/*External const LAPACK_ROW_MAJOR.*/
extern const LAPACK_ROW_MAJOR : c_int ;
/*External const LAPACK_COL_MAJOR.*/
extern const LAPACK_COL_MAJOR : c_int ;
/*External const LAPACK_WORK_MEMORY_ERROR.*/
extern const LAPACK_WORK_MEMORY_ERROR : c_int ;
/*External const LAPACK_TRANSPOSE_MEMORY_ERROR.*/
extern const LAPACK_TRANSPOSE_MEMORY_ERROR : c_int ;


/* Mirrors the :data:`LAPACK_ROW_MAJOR` and :data:`LAPACK_COL_MAJOR` consts defined by LAPACK.

The enum values values are the same as the const value such that ``lapack_memory_order.row_major == LAPACK_ROW_MAJOR`` and ``lapack_memory_order.column_major == LAPACK_COL_MAJOR``, allowing the enum to be used with pure LAPACK procedures.
 */
enum lapack_memory_order{row_major = 101 : c_int, column_major = 102 : c_int};


pragma "no doc"
/*
 
*/
module ClassicLAPACK {

use SysCTypes;

pragma "no doc"
extern proc lapack_make_complex_float(re : c_float, im : c_float) : complex(64);

pragma "no doc"
extern proc lapack_make_complex_double(re : c_double, im : c_double) : complex(128);

pragma "no doc"
extern proc LAPACK_lsame(ref ca : c_char, ref cb : c_char, lca : c_int, lcb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sbdsdc(matrix_order : lapack_memory_order, uplo : c_char, compq : c_char, n : c_int, d : [] c_float, e : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, q : [] c_float, iq : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sbdsdc(matrix_order : lapack_memory_order, uplo : string, compq : string, n : c_int, d : [] c_float, e : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, q : [] c_float, iq : [] c_int) : c_int{
  require header;
  return LAPACKE_sbdsdc(matrix_order, uplo.toByte() : c_char, compq.toByte() : c_char, n, d, e, u, ldu, vt, ldvt, q, iq);
}

pragma "no doc"
extern proc LAPACKE_dbdsdc(matrix_order : lapack_memory_order, uplo : c_char, compq : c_char, n : c_int, d : [] c_double, e : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, q : [] c_double, iq : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dbdsdc(matrix_order : lapack_memory_order, uplo : string, compq : string, n : c_int, d : [] c_double, e : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, q : [] c_double, iq : [] c_int) : c_int{
  require header;
  return LAPACKE_dbdsdc(matrix_order, uplo.toByte() : c_char, compq.toByte() : c_char, n, d, e, u, ldu, vt, ldvt, q, iq);
}

pragma "no doc"
extern proc LAPACKE_sbdsqr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] c_float, ldvt : c_int, u : [] c_float, ldu : c_int, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sbdsqr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] c_float, ldvt : c_int, u : [] c_float, ldu : c_int, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sbdsqr(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dbdsqr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] c_double, ldvt : c_int, u : [] c_double, ldu : c_int, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dbdsqr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] c_double, ldvt : c_int, u : [] c_double, ldu : c_int, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dbdsqr(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cbdsqr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] complex(64), ldvt : c_int, u : [] complex(64), ldu : c_int, c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cbdsqr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] complex(64), ldvt : c_int, u : [] complex(64), ldu : c_int, c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cbdsqr(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zbdsqr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] complex(128), ldvt : c_int, u : [] complex(128), ldu : c_int, c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zbdsqr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] complex(128), ldvt : c_int, u : [] complex(128), ldu : c_int, c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zbdsqr(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sdisna(job : c_char, m : c_int, n : c_int, d : [] c_float, sep : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sdisna(job : string, m : c_int, n : c_int, d : [] c_float, sep : [] c_float) : c_int{
  require header;
  return LAPACKE_sdisna(job.toByte() : c_char, m, n, d, sep);
}

pragma "no doc"
extern proc LAPACKE_ddisna(job : c_char, m : c_int, n : c_int, d : [] c_double, sep : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ddisna(job : string, m : c_int, n : c_int, d : [] c_double, sep : [] c_double) : c_int{
  require header;
  return LAPACKE_ddisna(job.toByte() : c_char, m, n, d, sep);
}

pragma "no doc"
extern proc LAPACKE_sgbbrd(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int, pt : [] c_float, ldpt : c_int, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbbrd(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int, pt : [] c_float, ldpt : c_int, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sgbbrd(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dgbbrd(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int, pt : [] c_double, ldpt : c_int, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbbrd(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int, pt : [] c_double, ldpt : c_int, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dgbbrd(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cgbbrd(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int, pt : [] complex(64), ldpt : c_int, c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbbrd(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int, pt : [] complex(64), ldpt : c_int, c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cgbbrd(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zgbbrd(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int, pt : [] complex(128), ldpt : c_int, c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbbrd(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int, pt : [] complex(128), ldpt : c_int, c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zgbbrd(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sgbcon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbcon(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_sgbcon(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dgbcon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbcon(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dgbcon(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cgbcon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbcon(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_cgbcon(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zgbcon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbcon(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zgbcon(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_sgbequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sgbrfs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dgbrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dgbrfs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cgbrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbrfs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zgbrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbrfs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sgbrfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbrfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_sgbrfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dgbrfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbrfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dgbrfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_cgbrfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbrfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbrfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zgbrfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbrfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbrfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_sgbsv(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbsv(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbsv(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbsv(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int{
  require header;
  return LAPACKE_sgbsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_dgbsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int{
  require header;
  return LAPACKE_dgbsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_cgbsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_zgbsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_sgbsvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbsvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_sgbsvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dgbsvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbsvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dgbsvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_cgbsvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbsvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbsvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zgbsvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbsvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbsvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_sgbtrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbtrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbtrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbtrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbtrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbtrs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sgbtrs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dgbtrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbtrs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dgbtrs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cgbtrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbtrs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cgbtrs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zgbtrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbtrs(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zgbtrs(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sgebak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgebak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int{
  require header;
  return LAPACKE_sgebak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_dgebak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgebak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int{
  require header;
  return LAPACKE_dgebak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_cgebak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgebak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int{
  require header;
  return LAPACKE_cgebak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_zgebak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgebak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int{
  require header;
  return LAPACKE_zgebak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_sgebal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_float, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgebal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_float, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int{
  require header;
  return LAPACKE_sgebal(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_dgebal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_double, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgebal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_double, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int{
  require header;
  return LAPACKE_dgebal(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_cgebal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(64), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgebal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(64), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int{
  require header;
  return LAPACKE_cgebal(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_zgebal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(128), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgebal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(128), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int{
  require header;
  return LAPACKE_zgebal(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_sgebrd(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, d : [] c_float, e : [] c_float, tauq : [] c_float, taup : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgebrd(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, d : [] c_double, e : [] c_double, tauq : [] c_double, taup : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgebrd(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, d : [] c_float, e : [] c_float, tauq : [] complex(64), taup : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgebrd(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, d : [] c_double, e : [] c_double, tauq : [] complex(128), taup : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgecon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgecon(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_sgecon(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dgecon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgecon(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dgecon(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cgecon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgecon(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_cgecon(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zgecon(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgecon(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zgecon(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_sgeequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeequ(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeequb(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgees(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_S_SELECT2, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgees(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_S_SELECT2, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int) : c_int{
  require header;
  return LAPACKE_sgees(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, wr, wi, vs, ldvs);
}

pragma "no doc"
extern proc LAPACKE_dgees(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_D_SELECT2, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgees(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_D_SELECT2, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int) : c_int{
  require header;
  return LAPACKE_dgees(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, wr, wi, vs, ldvs);
}

pragma "no doc"
extern proc LAPACKE_cgees(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_C_SELECT1, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgees(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_C_SELECT1, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int) : c_int{
  require header;
  return LAPACKE_cgees(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, w, vs, ldvs);
}

pragma "no doc"
extern proc LAPACKE_zgees(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_Z_SELECT1, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgees(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_Z_SELECT1, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int) : c_int{
  require header;
  return LAPACKE_zgees(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, w, vs, ldvs);
}

pragma "no doc"
extern proc LAPACKE_sgeesx(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_S_SELECT2, sense : c_char, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int, ref rconde : c_float, ref rcondv : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgeesx(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_S_SELECT2, sense : string, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int, ref rconde : c_float, ref rcondv : c_float) : c_int{
  require header;
  return LAPACKE_sgeesx(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, wr, wi, vs, ldvs, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_dgeesx(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_D_SELECT2, sense : c_char, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int, ref rconde : c_double, ref rcondv : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgeesx(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_D_SELECT2, sense : string, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int, ref rconde : c_double, ref rcondv : c_double) : c_int{
  require header;
  return LAPACKE_dgeesx(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, wr, wi, vs, ldvs, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_cgeesx(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_C_SELECT1, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int, ref rconde : c_float, ref rcondv : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgeesx(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_C_SELECT1, sense : string, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int, ref rconde : c_float, ref rcondv : c_float) : c_int{
  require header;
  return LAPACKE_cgeesx(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_zgeesx(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_Z_SELECT1, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int, ref rconde : c_double, ref rcondv : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgeesx(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_Z_SELECT1, sense : string, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int, ref rconde : c_double, ref rcondv : c_double) : c_int{
  require header;
  return LAPACKE_zgeesx(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_sgeev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgeev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int) : c_int{
  require header;
  return LAPACKE_sgeev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_dgeev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgeev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int) : c_int{
  require header;
  return LAPACKE_dgeev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_cgeev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgeev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int) : c_int{
  require header;
  return LAPACKE_cgeev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_zgeev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgeev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int) : c_int{
  require header;
  return LAPACKE_zgeev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_sgeevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgeevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int{
  require header;
  return LAPACKE_sgeevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_dgeevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgeevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int{
  require header;
  return LAPACKE_dgeevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_cgeevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgeevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int{
  require header;
  return LAPACKE_cgeevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_zgeevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgeevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int{
  require header;
  return LAPACKE_zgeevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_sgehrd(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgehrd(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgehrd(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgehrd(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgejsv(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, jobr : c_char, jobt : c_char, jobp : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, stat : [] c_float, istat : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgejsv(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, jobr : string, jobt : string, jobp : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, stat : [] c_float, istat : [] c_int) : c_int{
  require header;
  return LAPACKE_sgejsv(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, jobr.toByte() : c_char, jobt.toByte() : c_char, jobp.toByte() : c_char, m, n, a, lda, sva, u, ldu, v, ldv, stat, istat);
}

pragma "no doc"
extern proc LAPACKE_dgejsv(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, jobr : c_char, jobt : c_char, jobp : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, stat : [] c_double, istat : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgejsv(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, jobr : string, jobt : string, jobp : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, stat : [] c_double, istat : [] c_int) : c_int{
  require header;
  return LAPACKE_dgejsv(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, jobr.toByte() : c_char, jobt.toByte() : c_char, jobp.toByte() : c_char, m, n, a, lda, sva, u, ldu, v, ldv, stat, istat);
}

pragma "no doc"
extern proc LAPACKE_sgelq2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelq2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelq2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelq2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgelqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgels(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgels(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sgels(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dgels(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgels(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dgels(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cgels(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgels(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cgels(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zgels(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgels(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zgels(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sgelsd(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelsd(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelsd(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelsd(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgelss(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelss(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelss(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelss(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgelsy(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, jpvt : [] c_int, rcond : c_float, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelsy(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, jpvt : [] c_int, rcond : c_double, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelsy(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, jpvt : [] c_int, rcond : c_float, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelsy(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, jpvt : [] c_int, rcond : c_double, ref rank : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqlf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqlf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqlf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqlf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqp3(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, jpvt : [] c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqp3(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, jpvt : [] c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqp3(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, jpvt : [] c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqp3(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, jpvt : [] c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqpf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, jpvt : [] c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqpf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, jpvt : [] c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqpf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, jpvt : [] c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqpf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, jpvt : [] c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqr2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqr2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqr2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqr2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqrfp(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqrfp(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqrfp(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqrfp(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgerfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgerfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sgerfs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dgerfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgerfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dgerfs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cgerfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgerfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cgerfs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zgerfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgerfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zgerfs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sgerfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgerfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_sgerfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dgerfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgerfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dgerfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_cgerfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgerfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_cgerfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zgerfsx(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgerfsx(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zgerfsx(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_sgerqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgerqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgerqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgerqf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgesdd(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesdd(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int) : c_int{
  require header;
  return LAPACKE_sgesdd(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt);
}

pragma "no doc"
extern proc LAPACKE_dgesdd(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesdd(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int) : c_int{
  require header;
  return LAPACKE_dgesdd(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt);
}

pragma "no doc"
extern proc LAPACKE_cgesdd(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesdd(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int) : c_int{
  require header;
  return LAPACKE_cgesdd(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt);
}

pragma "no doc"
extern proc LAPACKE_zgesdd(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesdd(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int) : c_int{
  require header;
  return LAPACKE_zgesdd(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt);
}

pragma "no doc"
extern proc LAPACKE_sgesv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgesv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgesv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgesv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dsgesv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zcgesv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgesvd(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, superb : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvd(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, superb : [] c_float) : c_int{
  require header;
  return LAPACKE_sgesvd(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
}

pragma "no doc"
extern proc LAPACKE_dgesvd(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, superb : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvd(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, superb : [] c_double) : c_int{
  require header;
  return LAPACKE_dgesvd(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
}

pragma "no doc"
extern proc LAPACKE_cgesvd(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int, superb : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesvd(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int, superb : [] c_float) : c_int{
  require header;
  return LAPACKE_cgesvd(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
}

pragma "no doc"
extern proc LAPACKE_zgesvd(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int, superb : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesvd(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int, superb : [] c_double) : c_int{
  require header;
  return LAPACKE_zgesvd(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, superb);
}

pragma "no doc"
extern proc LAPACKE_sgesvj(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, mv : c_int, v : [] c_float, ldv : c_int, stat : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvj(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, mv : c_int, v : [] c_float, ldv : c_int, stat : [] c_float) : c_int{
  require header;
  return LAPACKE_sgesvj(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, m, n, a, lda, sva, mv, v, ldv, stat);
}

pragma "no doc"
extern proc LAPACKE_dgesvj(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, mv : c_int, v : [] c_double, ldv : c_int, stat : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvj(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, mv : c_int, v : [] c_double, ldv : c_int, stat : [] c_double) : c_int{
  require header;
  return LAPACKE_dgesvj(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, m, n, a, lda, sva, mv, v, ldv, stat);
}

pragma "no doc"
extern proc LAPACKE_sgesvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int{
  require header;
  return LAPACKE_sgesvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_dgesvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int{
  require header;
  return LAPACKE_dgesvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_cgesvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, rpivot : [] c_float) : c_int{
  require header;
  return LAPACKE_cgesvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_zgesvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, rpivot : [] c_double) : c_int{
  require header;
  return LAPACKE_zgesvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, rpivot);
}

pragma "no doc"
extern proc LAPACKE_sgesvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_sgesvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dgesvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dgesvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_cgesvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_cgesvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zgesvxx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesvxx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zgesvxx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_sgetf2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgetf2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgetf2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgetf2(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgetrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgetrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgetrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgetrf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgetri(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgetri(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgetri(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgetri(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgetrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgetrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sgetrs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dgetrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgetrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dgetrs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cgetrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgetrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cgetrs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zgetrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgetrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zgetrs(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sggbak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggbak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int{
  require header;
  return LAPACKE_sggbak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_dggbak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggbak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int{
  require header;
  return LAPACKE_dggbak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_cggbak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggbak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int{
  require header;
  return LAPACKE_cggbak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_zggbak(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggbak(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int{
  require header;
  return LAPACKE_zggbak(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_sggbal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sggbal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float) : c_int{
  require header;
  return LAPACKE_sggbal(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
}

pragma "no doc"
extern proc LAPACKE_dggbal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dggbal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double) : c_int{
  require header;
  return LAPACKE_dggbal(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
}

pragma "no doc"
extern proc LAPACKE_cggbal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cggbal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float) : c_int{
  require header;
  return LAPACKE_cggbal(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
}

pragma "no doc"
extern proc LAPACKE_zggbal(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zggbal(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double) : c_int{
  require header;
  return LAPACKE_zggbal(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale);
}

pragma "no doc"
extern proc LAPACKE_sgges(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_S_SELECT3, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgges(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_S_SELECT3, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int) : c_int{
  require header;
  return LAPACKE_sgges(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr);
}

pragma "no doc"
extern proc LAPACKE_dgges(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_D_SELECT3, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgges(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_D_SELECT3, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int) : c_int{
  require header;
  return LAPACKE_dgges(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr);
}

pragma "no doc"
extern proc LAPACKE_cgges(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_C_SELECT2, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgges(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_C_SELECT2, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int) : c_int{
  require header;
  return LAPACKE_cgges(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
}

pragma "no doc"
extern proc LAPACKE_zgges(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_Z_SELECT2, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgges(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_Z_SELECT2, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int) : c_int{
  require header;
  return LAPACKE_zgges(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr);
}

pragma "no doc"
extern proc LAPACKE_sggesx(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_S_SELECT3, sense : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sggesx(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_S_SELECT3, sense : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float) : c_int{
  require header;
  return LAPACKE_sggesx(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_dggesx(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_D_SELECT3, sense : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dggesx(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_D_SELECT3, sense : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double) : c_int{
  require header;
  return LAPACKE_dggesx(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_cggesx(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_C_SELECT2, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cggesx(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_C_SELECT2, sense : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float) : c_int{
  require header;
  return LAPACKE_cggesx(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_zggesx(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_Z_SELECT2, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zggesx(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_Z_SELECT2, sense : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double) : c_int{
  require header;
  return LAPACKE_zggesx(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_sggev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int) : c_int{
  require header;
  return LAPACKE_sggev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_dggev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int) : c_int{
  require header;
  return LAPACKE_dggev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_cggev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int) : c_int{
  require header;
  return LAPACKE_cggev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_zggev(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggev(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int) : c_int{
  require header;
  return LAPACKE_zggev(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr);
}

pragma "no doc"
extern proc LAPACKE_sggevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sggevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int{
  require header;
  return LAPACKE_sggevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_dggevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dggevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int{
  require header;
  return LAPACKE_dggevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_cggevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cggevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float) : c_int{
  require header;
  return LAPACKE_cggevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_zggevx(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zggevx(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double) : c_int{
  require header;
  return LAPACKE_zggevx(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv);
}

pragma "no doc"
extern proc LAPACKE_sggglm(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, d : [] c_float, x : [] c_float, y : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dggglm(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, d : [] c_double, x : [] c_double, y : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cggglm(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, d : [] complex(64), x : [] complex(64), y : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zggglm(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, d : [] complex(128), x : [] complex(128), y : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgghrd(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgghrd(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sgghrd(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dgghrd(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgghrd(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dgghrd(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_cgghrd(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgghrd(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_cgghrd(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zgghrd(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgghrd(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zgghrd(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sgglse(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, c : [] c_float, d : [] c_float, x : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgglse(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, c : [] c_double, d : [] c_double, x : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgglse(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, c : [] complex(64), d : [] complex(64), x : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgglse(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, c : [] complex(128), d : [] complex(128), x : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sggqrf(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_float, lda : c_int, taua : [] c_float, b : [] c_float, ldb : c_int, taub : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dggqrf(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_double, lda : c_int, taua : [] c_double, b : [] c_double, ldb : c_int, taub : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cggqrf(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(64), lda : c_int, taua : [] complex(64), b : [] complex(64), ldb : c_int, taub : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zggqrf(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(128), lda : c_int, taua : [] complex(128), b : [] complex(128), ldb : c_int, taub : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sggrqf(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] c_float, lda : c_int, taua : [] c_float, b : [] c_float, ldb : c_int, taub : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dggrqf(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] c_double, lda : c_int, taua : [] c_double, b : [] c_double, ldb : c_int, taub : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cggrqf(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] complex(64), lda : c_int, taua : [] complex(64), b : [] complex(64), ldb : c_int, taub : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zggrqf(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] complex(128), lda : c_int, taua : [] complex(128), b : [] complex(128), ldb : c_int, taub : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sggsvd(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggsvd(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sggsvd(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

pragma "no doc"
extern proc LAPACKE_dggsvd(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggsvd(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dggsvd(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

pragma "no doc"
extern proc LAPACKE_cggsvd(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggsvd(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cggsvd(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

pragma "no doc"
extern proc LAPACKE_zggsvd(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggsvd(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zggsvd(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, iwork);
}

pragma "no doc"
extern proc LAPACKE_sggsvp(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggsvp(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int) : c_int{
  require header;
  return LAPACKE_sggsvp(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_dggsvp(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggsvp(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int) : c_int{
  require header;
  return LAPACKE_dggsvp(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_cggsvp(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggsvp(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int) : c_int{
  require header;
  return LAPACKE_cggsvp(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_zggsvp(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggsvp(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int) : c_int{
  require header;
  return LAPACKE_zggsvp(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_sgtcon(norm : c_char, n : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgtcon(norm : string, n : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_sgtcon(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dgtcon(norm : c_char, n : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgtcon(norm : string, n : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dgtcon(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cgtcon(norm : c_char, n : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgtcon(norm : string, n : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_cgtcon(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zgtcon(norm : c_char, n : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgtcon(norm : string, n : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zgtcon(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_sgtrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgtrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sgtrfs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dgtrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgtrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dgtrfs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cgtrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgtrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cgtrfs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zgtrfs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgtrfs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zgtrfs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sgtsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgtsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgtsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgtsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgtsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgtsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sgtsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dgtsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgtsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dgtsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cgtsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgtsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cgtsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zgtsvx(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgtsvx(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zgtsvx(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sgttrf(n : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgttrf(n : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgttrf(n : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgttrf(n : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgttrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgttrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sgttrs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dgttrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgttrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dgttrs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cgttrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgttrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cgttrs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zgttrs(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgttrs(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zgttrs(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chbev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chbev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhbev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhbev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chbevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chbevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhbevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhbevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chbevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chbevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhbevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhbevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_chbgst(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, x : [] complex(64), ldx : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgst(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, x : [] complex(64), ldx : c_int) : c_int{
  require header;
  return LAPACKE_chbgst(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
}

pragma "no doc"
extern proc LAPACKE_zhbgst(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, x : [] complex(128), ldx : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgst(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, x : [] complex(128), ldx : c_int) : c_int{
  require header;
  return LAPACKE_zhbgst(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
}

pragma "no doc"
extern proc LAPACKE_chbgv(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgv(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chbgv(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhbgv(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgv(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhbgv(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chbgvd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgvd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chbgvd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhbgvd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgvd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhbgvd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chbgvx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgvx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chbgvx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhbgvx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgvx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhbgvx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_chbtrd(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbtrd(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int) : c_int{
  require header;
  return LAPACKE_chbtrd(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_zhbtrd(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbtrd(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int) : c_int{
  require header;
  return LAPACKE_zhbtrd(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_checon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_checon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_checon(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zhecon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhecon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zhecon(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cheequb(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cheequb(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_cheequb(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_zheequb(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zheequb(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_zheequb(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_cheev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cheev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_cheev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_zheev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zheev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_zheev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_cheevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cheevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_cheevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_zheevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zheevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_zheevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_cheevr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cheevr(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_cheevr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_zheevr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zheevr(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_zheevr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_cheevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cheevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_cheevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_zheevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zheevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zheevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_chegst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chegst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chegst(matrix_order, itype, uplo.toByte() : c_char, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhegst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhegst(matrix_order, itype, uplo.toByte() : c_char, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chegv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chegv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_chegv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_zhegv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_zhegv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_chegvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chegvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_chegvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_zhegvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_zhegvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_chegvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chegvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chegvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhegvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhegvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_cherfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cherfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cherfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zherfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zherfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zherfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cherfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cherfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_cherfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zherfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zherfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zherfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_chesv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chesv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chesv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhesv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhesv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhesv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chesvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chesvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_chesvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zhesvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhesvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zhesvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_chesvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chesvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_chesvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zhesvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhesvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zhesvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_chetrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, d : [] c_float, e : [] c_float, tau : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chetrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, d : [] c_float, e : [] c_float, tau : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chetrd(matrix_order, uplo.toByte() : c_char, n, a, lda, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_zhetrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, d : [] c_double, e : [] c_double, tau : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, d : [] c_double, e : [] c_double, tau : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhetrd(matrix_order, uplo.toByte() : c_char, n, a, lda, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_chetrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chetrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_chetrf(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zhetrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zhetrf(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_chetri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chetri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_chetri(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zhetri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zhetri(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_chetrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chetrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chetrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhetrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhetrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chfrk(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_float, a : [] complex(64), lda : c_int, beta : c_float, c : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chfrk(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_float, a : [] complex(64), lda : c_int, beta : c_float, c : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chfrk(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_zhfrk(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_double, a : [] complex(128), lda : c_int, beta : c_double, c : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhfrk(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_double, a : [] complex(128), lda : c_int, beta : c_double, c : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhfrk(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_shgeqz(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, t : [] c_float, ldt : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_shgeqz(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, t : [] c_float, ldt : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_shgeqz(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alphar, alphai, beta, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dhgeqz(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, t : [] c_double, ldt : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dhgeqz(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, t : [] c_double, ldt : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dhgeqz(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alphar, alphai, beta, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chgeqz(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, t : [] complex(64), ldt : c_int, alpha : [] complex(64), beta : [] complex(64), q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chgeqz(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, t : [] complex(64), ldt : c_int, alpha : [] complex(64), beta : [] complex(64), q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chgeqz(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhgeqz(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, t : [] complex(128), ldt : c_int, alpha : [] complex(128), beta : [] complex(128), q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhgeqz(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, t : [] complex(128), ldt : c_int, alpha : [] complex(128), beta : [] complex(128), q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhgeqz(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chpcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chpcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_chpcon(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zhpcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zhpcon(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_chpev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chpev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhpev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhpev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chpevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chpevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhpevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhpevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chpevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chpevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhpevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhpevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_chpgst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chpgst(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_zhpgst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhpgst(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_chpgv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chpgv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhpgv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhpgv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chpgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chpgvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhpgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhpgvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chpgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chpgvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhpgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhpgvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_chprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_chprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zhprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zhprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_chpsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chpsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhpsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhpsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chpsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chpsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_chpsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zhpsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zhpsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_chptrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), d : [] c_float, e : [] c_float, tau : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chptrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), d : [] c_float, e : [] c_float, tau : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chptrd(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_zhptrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), d : [] c_double, e : [] c_double, tau : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), d : [] c_double, e : [] c_double, tau : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhptrd(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_chptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_chptrf(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zhptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zhptrf(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_chptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_chptri(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zhptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zhptri(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_chptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_shsein(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_shsein(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_shsein(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, wr, wi, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_dhsein(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dhsein(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_dhsein(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, wr, wi, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_chsein(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chsein(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_chsein(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_zhsein(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhsein(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_zhsein(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_shseqr(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_shseqr(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_shseqr(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, wr, wi, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dhseqr(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dhseqr(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dhseqr(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, wr, wi, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_chseqr(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chseqr(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_chseqr(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zhseqr(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhseqr(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zhseqr(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_clacgv(n : c_int, x : [] complex(64), incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlacgv(n : c_int, x : [] complex(128), incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slacn2(n : c_int, v : [] c_float, x : [] c_float, isgn : [] c_int, ref est : c_float, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlacn2(n : c_int, v : [] c_double, x : [] c_double, isgn : [] c_int, ref est : c_double, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_clacn2(n : c_int, v : [] complex(64), x : [] complex(64), ref est : c_float, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlacn2(n : c_int, v : [] complex(128), x : [] complex(128), ref est : c_double, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slacpy(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slacpy(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_slacpy(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dlacpy(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlacpy(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dlacpy(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_clacpy(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clacpy(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_clacpy(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zlacpy(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlacpy(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zlacpy(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_clacp2(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clacp2(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_clacp2(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zlacp2(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlacp2(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zlacp2(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zlag2c(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, sa : [] complex(64), ldsa : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slag2d(matrix_order : lapack_memory_order, m : c_int, n : c_int, sa : [] c_float, ldsa : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlag2s(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, sa : [] c_float, ldsa : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_clag2z(matrix_order : lapack_memory_order, m : c_int, n : c_int, sa : [] complex(64), ldsa : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slagge(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_float, a : [] c_float, lda : c_int, iseed : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlagge(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_double, a : [] c_double, lda : c_int, iseed : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_clagge(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_float, a : [] complex(64), lda : c_int, iseed : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlagge(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_double, a : [] complex(128), lda : c_int, iseed : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slamch(cmach : c_char) : c_float;

pragma "no doc"
inline proc LAPACKE_slamch(cmach : string) : c_float{
  require header;
  return LAPACKE_slamch(cmach.toByte() : c_char);
}

pragma "no doc"
extern proc LAPACKE_dlamch(cmach : c_char) : c_double;

pragma "no doc"
inline proc LAPACKE_dlamch(cmach : string) : c_double{
  require header;
  return LAPACKE_dlamch(cmach.toByte() : c_char);
}

pragma "no doc"
extern proc LAPACKE_slange(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int) : c_float;

pragma "no doc"
inline proc LAPACKE_slange(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] c_float, lda : c_int) : c_float{
  require header;
  return LAPACKE_slange(matrix_order, norm.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlange(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int) : c_double;

pragma "no doc"
inline proc LAPACKE_dlange(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] c_double, lda : c_int) : c_double{
  require header;
  return LAPACKE_dlange(matrix_order, norm.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_clange(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int) : c_float;

pragma "no doc"
inline proc LAPACKE_clange(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int) : c_float{
  require header;
  return LAPACKE_clange(matrix_order, norm.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlange(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int) : c_double;

pragma "no doc"
inline proc LAPACKE_zlange(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int) : c_double{
  require header;
  return LAPACKE_zlange(matrix_order, norm.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_clanhe(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_float;

pragma "no doc"
inline proc LAPACKE_clanhe(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_float{
  require header;
  return LAPACKE_clanhe(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlanhe(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_double;

pragma "no doc"
inline proc LAPACKE_zlanhe(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_double{
  require header;
  return LAPACKE_zlanhe(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_slansy(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int) : c_float;

pragma "no doc"
inline proc LAPACKE_slansy(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] c_float, lda : c_int) : c_float{
  require header;
  return LAPACKE_slansy(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlansy(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int) : c_double;

pragma "no doc"
inline proc LAPACKE_dlansy(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] c_double, lda : c_int) : c_double{
  require header;
  return LAPACKE_dlansy(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_clansy(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_float;

pragma "no doc"
inline proc LAPACKE_clansy(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_float{
  require header;
  return LAPACKE_clansy(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlansy(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_double;

pragma "no doc"
inline proc LAPACKE_zlansy(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_double{
  require header;
  return LAPACKE_zlansy(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_slantr(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int) : c_float;

pragma "no doc"
inline proc LAPACKE_slantr(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] c_float, lda : c_int) : c_float{
  require header;
  return LAPACKE_slantr(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlantr(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int) : c_double;

pragma "no doc"
inline proc LAPACKE_dlantr(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] c_double, lda : c_int) : c_double{
  require header;
  return LAPACKE_dlantr(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_clantr(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int) : c_float;

pragma "no doc"
inline proc LAPACKE_clantr(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int) : c_float{
  require header;
  return LAPACKE_clantr(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlantr(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int) : c_double;

pragma "no doc"
inline proc LAPACKE_zlantr(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int) : c_double{
  require header;
  return LAPACKE_zlantr(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_slarfb(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] c_float, ldv : c_int, t : [] c_float, ldt : c_int, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slarfb(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] c_float, ldv : c_int, t : [] c_float, ldt : c_int, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_slarfb(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dlarfb(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] c_double, ldv : c_int, t : [] c_double, ldt : c_int, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlarfb(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] c_double, ldv : c_int, t : [] c_double, ldt : c_int, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dlarfb(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_clarfb(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, t : [] complex(64), ldt : c_int, c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clarfb(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, t : [] complex(64), ldt : c_int, c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_clarfb(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zlarfb(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, t : [] complex(128), ldt : c_int, c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlarfb(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, t : [] complex(128), ldt : c_int, c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zlarfb(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_slarfg(n : c_int, ref alpha : c_float, x : [] c_float, incx : c_int, ref tau : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlarfg(n : c_int, ref alpha : c_double, x : [] c_double, incx : c_int, ref tau : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_clarfg(n : c_int, ref alpha : complex(64), x : [] complex(64), incx : c_int, ref tau : complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zlarfg(n : c_int, ref alpha : complex(128), x : [] complex(128), incx : c_int, ref tau : complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_slarft(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] c_float, ldv : c_int, tau : [] c_float, t : [] c_float, ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slarft(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] c_float, ldv : c_int, tau : [] c_float, t : [] c_float, ldt : c_int) : c_int{
  require header;
  return LAPACKE_slarft(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_dlarft(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] c_double, ldv : c_int, tau : [] c_double, t : [] c_double, ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlarft(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] c_double, ldv : c_int, tau : [] c_double, t : [] c_double, ldt : c_int) : c_int{
  require header;
  return LAPACKE_dlarft(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_clarft(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, tau : [] complex(64), t : [] complex(64), ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clarft(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, tau : [] complex(64), t : [] complex(64), ldt : c_int) : c_int{
  require header;
  return LAPACKE_clarft(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_zlarft(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, tau : [] complex(128), t : [] complex(128), ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlarft(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, tau : [] complex(128), t : [] complex(128), ldt : c_int) : c_int{
  require header;
  return LAPACKE_zlarft(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_slarfx(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] c_float, tau : c_float, c : [] c_float, ldc : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_slarfx(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] c_float, tau : c_float, c : [] c_float, ldc : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_slarfx(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_dlarfx(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] c_double, tau : c_double, c : [] c_double, ldc : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dlarfx(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] c_double, tau : c_double, c : [] c_double, ldc : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dlarfx(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_clarfx(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] complex(64), tau : complex(64), c : [] complex(64), ldc : c_int, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_clarfx(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] complex(64), tau : complex(64), c : [] complex(64), ldc : c_int, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_clarfx(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_zlarfx(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] complex(128), tau : complex(128), c : [] complex(128), ldc : c_int, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zlarfx(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] complex(128), tau : complex(128), c : [] complex(128), ldc : c_int, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zlarfx(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_slarnv(idist : c_int, iseed : [] c_int, n : c_int, x : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlarnv(idist : c_int, iseed : [] c_int, n : c_int, x : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_clarnv(idist : c_int, iseed : [] c_int, n : c_int, x : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zlarnv(idist : c_int, iseed : [] c_int, n : c_int, x : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_slaset(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : c_float, beta : c_float, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slaset(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : c_float, beta : c_float, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_slaset(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlaset(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : c_double, beta : c_double, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlaset(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : c_double, beta : c_double, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dlaset(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_claset(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : complex(64), beta : complex(64), a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_claset(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : complex(64), beta : complex(64), a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_claset(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlaset(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : complex(128), beta : complex(128), a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlaset(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : complex(128), beta : complex(128), a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zlaset(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_slasrt(id : c_char, n : c_int, d : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_slasrt(id : string, n : c_int, d : [] c_float) : c_int{
  require header;
  return LAPACKE_slasrt(id.toByte() : c_char, n, d);
}

pragma "no doc"
extern proc LAPACKE_dlasrt(id : c_char, n : c_int, d : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dlasrt(id : string, n : c_int, d : [] c_double) : c_int{
  require header;
  return LAPACKE_dlasrt(id.toByte() : c_char, n, d);
}

pragma "no doc"
extern proc LAPACKE_slaswp(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlaswp(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_claswp(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlaswp(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : c_char, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : string, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_slatms(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : c_char, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : string, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dlatms(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda);
}

pragma "no doc"
extern proc LAPACKE_clatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : c_char, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : string, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_clatms(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : c_char, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlatms(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : string, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zlatms(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda);
}

pragma "no doc"
extern proc LAPACKE_slauum(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slauum(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_slauum(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlauum(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlauum(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dlauum(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_clauum(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clauum(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_clauum(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlauum(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlauum(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zlauum(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_sopgtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, tau : [] c_float, q : [] c_float, ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sopgtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, tau : [] c_float, q : [] c_float, ldq : c_int) : c_int{
  require header;
  return LAPACKE_sopgtr(matrix_order, uplo.toByte() : c_char, n, ap, tau, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_dopgtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, tau : [] c_double, q : [] c_double, ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dopgtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, tau : [] c_double, q : [] c_double, ldq : c_int) : c_int{
  require header;
  return LAPACKE_dopgtr(matrix_order, uplo.toByte() : c_char, n, ap, tau, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_sopmtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, ap : [] c_float, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sopmtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, ap : [] c_float, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sopmtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, ap, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dopmtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, ap : [] c_double, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dopmtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, ap : [] c_double, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dopmtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, ap, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sorgbr(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sorgbr(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int{
  require header;
  return LAPACKE_sorgbr(matrix_order, vect.toByte() : c_char, m, n, k, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_dorgbr(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dorgbr(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int{
  require header;
  return LAPACKE_dorgbr(matrix_order, vect.toByte() : c_char, m, n, k, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_sorghr(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dorghr(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sorglq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dorglq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgql(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dorgql(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgqr(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dorgqr(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgrq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dorgrq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sorgtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int{
  require header;
  return LAPACKE_sorgtr(matrix_order, uplo.toByte() : c_char, n, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_dorgtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dorgtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int{
  require header;
  return LAPACKE_dorgtr(matrix_order, uplo.toByte() : c_char, n, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_sormbr(matrix_order : lapack_memory_order, vect : c_char, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormbr(matrix_order : lapack_memory_order, vect : string, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormbr(matrix_order, vect.toByte() : c_char, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormbr(matrix_order : lapack_memory_order, vect : c_char, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormbr(matrix_order : lapack_memory_order, vect : string, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormbr(matrix_order, vect.toByte() : c_char, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sormhr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormhr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormhr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, ilo, ihi, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormhr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormhr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormhr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, ilo, ihi, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sormlq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormlq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormlq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormlq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormlq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormlq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sormql(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormql(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormql(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormql(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormql(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormql(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sormqr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormqr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormqr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormqr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormqr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormqr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sormrq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormrq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormrq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormrq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormrq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormrq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sormrz(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormrz(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormrz(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, l, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormrz(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormrz(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormrz(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, l, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sormtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int) : c_int{
  require header;
  return LAPACKE_sormtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_dormtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int) : c_int{
  require header;
  return LAPACKE_dormtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_spbcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spbcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_spbcon(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dpbcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dpbcon(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cpbcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_cpbcon(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zpbcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zpbcon(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_spbequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spbequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_spbequ(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_dpbequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_dpbequ(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_cpbequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_cpbequ(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_zpbequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_zpbequ(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_spbrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spbrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_spbrfs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dpbrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dpbrfs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cpbrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cpbrfs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zpbrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zpbrfs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_spbstf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] c_float, ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbstf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] c_float, ldbb : c_int) : c_int{
  require header;
  return LAPACKE_spbstf(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_dpbstf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] c_double, ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbstf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] c_double, ldbb : c_int) : c_int{
  require header;
  return LAPACKE_dpbstf(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_cpbstf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] complex(64), ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbstf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] complex(64), ldbb : c_int) : c_int{
  require header;
  return LAPACKE_cpbstf(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_zpbstf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] complex(128), ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbstf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] complex(128), ldbb : c_int) : c_int{
  require header;
  return LAPACKE_zpbstf(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_spbsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spbsv(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpbsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpbsv(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpbsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpbsv(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpbsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpbsv(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spbsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spbsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_spbsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dpbsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dpbsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cpbsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cpbsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zpbsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zpbsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_spbtrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbtrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int) : c_int{
  require header;
  return LAPACKE_spbtrf(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_dpbtrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbtrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int) : c_int{
  require header;
  return LAPACKE_dpbtrf(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_cpbtrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbtrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int) : c_int{
  require header;
  return LAPACKE_cpbtrf(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_zpbtrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbtrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int) : c_int{
  require header;
  return LAPACKE_zpbtrf(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_spbtrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbtrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spbtrs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpbtrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbtrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpbtrs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpbtrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbtrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpbtrs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpbtrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbtrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpbtrs(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spftrf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spftrf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_float) : c_int{
  require header;
  return LAPACKE_spftrf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_dpftrf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpftrf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_double) : c_int{
  require header;
  return LAPACKE_dpftrf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_cpftrf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpftrf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpftrf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_zpftrf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpftrf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpftrf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_spftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spftri(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_float) : c_int{
  require header;
  return LAPACKE_spftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_dpftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpftri(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_double) : c_int{
  require header;
  return LAPACKE_dpftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_cpftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpftri(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_zpftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpftri(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_spftrs(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spftrs(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spftrs(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpftrs(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpftrs(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpftrs(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpftrs(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpftrs(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpftrs(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpftrs(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpftrs(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpftrs(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spocon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spocon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_spocon(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dpocon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpocon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dpocon(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cpocon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpocon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_cpocon(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zpocon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpocon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zpocon(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_spoequ(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dpoequ(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cpoequ(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zpoequ(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_spoequb(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dpoequb(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cpoequb(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zpoequb(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sporfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sporfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sporfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dporfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dporfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dporfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cporfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cporfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cporfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zporfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zporfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zporfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sporfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sporfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_sporfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dporfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dporfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dporfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_cporfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cporfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_cporfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zporfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zporfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zporfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_sposv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sposv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sposv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dposv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dposv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dposv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cposv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cposv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cposv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zposv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zposv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zposv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dsposv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsposv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref chlapack_iter : c_int) : c_int{
  require header;
  return LAPACKE_dsposv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, chlapack_iter);
}

pragma "no doc"
extern proc LAPACKE_zcposv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zcposv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref chlapack_iter : c_int) : c_int{
  require header;
  return LAPACKE_zcposv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, chlapack_iter);
}

pragma "no doc"
extern proc LAPACKE_sposvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sposvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sposvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dposvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dposvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dposvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cposvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cposvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cposvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zposvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zposvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zposvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sposvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sposvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_sposvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dposvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dposvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dposvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_cposvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cposvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_cposvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zposvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zposvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zposvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_spotrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spotrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_spotrf(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dpotrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpotrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dpotrf(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_cpotrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpotrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_cpotrf(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zpotrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpotrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zpotrf(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_spotri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spotri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_spotri(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dpotri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpotri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dpotri(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_cpotri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpotri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_cpotri(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zpotri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpotri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zpotri(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_spotrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spotrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spotrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpotrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpotrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpotrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpotrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpotrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpotrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpotrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpotrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpotrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sppcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sppcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_sppcon(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dppcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dppcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dppcon(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cppcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cppcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_cppcon(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zppcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zppcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zppcon(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_sppequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sppequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_sppequ(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_dppequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dppequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_dppequ(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_cppequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cppequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_cppequ(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_zppequ(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zppequ(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_zppequ(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_spprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_spprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dpprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dpprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cpprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cpprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zpprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zpprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sppsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sppsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sppsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dppsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dppsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dppsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cppsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cppsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cppsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zppsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zppsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zppsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sppsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sppsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sppsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dppsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dppsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dppsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cppsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cppsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cppsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zppsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zppsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zppsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_spptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float) : c_int{
  require header;
  return LAPACKE_spptrf(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_dpptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double) : c_int{
  require header;
  return LAPACKE_dpptrf(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_cpptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpptrf(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_zpptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpptrf(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_spptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float) : c_int{
  require header;
  return LAPACKE_spptri(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_dpptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double) : c_int{
  require header;
  return LAPACKE_dpptri(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_cpptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpptri(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_zpptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpptri(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_spptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spstrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spstrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float) : c_int{
  require header;
  return LAPACKE_spstrf(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol);
}

pragma "no doc"
extern proc LAPACKE_dpstrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpstrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double) : c_int{
  require header;
  return LAPACKE_dpstrf(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol);
}

pragma "no doc"
extern proc LAPACKE_cpstrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpstrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float) : c_int{
  require header;
  return LAPACKE_cpstrf(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol);
}

pragma "no doc"
extern proc LAPACKE_zpstrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpstrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double) : c_int{
  require header;
  return LAPACKE_zpstrf(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol);
}

pragma "no doc"
extern proc LAPACKE_sptcon(n : c_int, d : [] c_float, e : [] c_float, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dptcon(n : c_int, d : [] c_double, e : [] c_double, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cptcon(n : c_int, d : [] c_float, e : [] complex(64), anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zptcon(n : c_int, d : [] c_double, e : [] complex(128), anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_spteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_spteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dpteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dpteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_cpteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_cpteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zpteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zpteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sptrfs(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, df : [] c_float, ef : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dptrfs(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, df : [] c_double, ef : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cptrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cptrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cptrfs(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zptrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zptrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zptrfs(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sptsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dptsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cptsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zptsv(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sptsvx(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, df : [] c_float, ef : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sptsvx(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, df : [] c_float, ef : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sptsvx(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dptsvx(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, df : [] c_double, ef : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dptsvx(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, df : [] c_double, ef : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dptsvx(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cptsvx(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cptsvx(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cptsvx(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zptsvx(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zptsvx(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zptsvx(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_spttrf(n : c_int, d : [] c_float, e : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dpttrf(n : c_int, d : [] c_double, e : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cpttrf(n : c_int, d : [] c_float, e : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zpttrf(n : c_int, d : [] c_double, e : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_spttrs(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dpttrs(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cpttrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpttrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpttrs(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpttrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpttrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpttrs(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ssbev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_ssbev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dsbev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dsbev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_ssbevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_ssbevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dsbevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dsbevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_ssbevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_ssbevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_dsbevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dsbevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssbgst(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, x : [] c_float, ldx : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgst(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, x : [] c_float, ldx : c_int) : c_int{
  require header;
  return LAPACKE_ssbgst(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
}

pragma "no doc"
extern proc LAPACKE_dsbgst(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, x : [] c_double, ldx : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgst(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, x : [] c_double, ldx : c_int) : c_int{
  require header;
  return LAPACKE_dsbgst(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx);
}

pragma "no doc"
extern proc LAPACKE_ssbgv(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgv(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_ssbgv(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dsbgv(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgv(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dsbgv(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_ssbgvd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgvd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_ssbgvd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dsbgvd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgvd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dsbgvd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_ssbgvx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgvx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_ssbgvx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_dsbgvx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgvx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dsbgvx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssbtrd(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbtrd(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int) : c_int{
  require header;
  return LAPACKE_ssbtrd(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_dsbtrd(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbtrd(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int) : c_int{
  require header;
  return LAPACKE_dsbtrd(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_ssfrk(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_float, a : [] c_float, lda : c_int, beta : c_float, c : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssfrk(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_float, a : [] c_float, lda : c_int, beta : c_float, c : [] c_float) : c_int{
  require header;
  return LAPACKE_ssfrk(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_dsfrk(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_double, a : [] c_double, lda : c_int, beta : c_double, c : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsfrk(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_double, a : [] c_double, lda : c_int, beta : c_double, c : [] c_double) : c_int{
  require header;
  return LAPACKE_dsfrk(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_sspcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sspcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_sspcon(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dspcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dspcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dspcon(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_cspcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cspcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_cspcon(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zspcon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zspcon(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zspcon(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_sspev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sspev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dspev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dspev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sspevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sspevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dspevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dspevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sspevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_sspevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_dspevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dspevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_sspgst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float) : c_int{
  require header;
  return LAPACKE_sspgst(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_dspgst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double) : c_int{
  require header;
  return LAPACKE_dspgst(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_sspgv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sspgv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dspgv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dspgv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sspgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sspgvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dspgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dspgvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sspgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_sspgvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_dspgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dspgvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_ssprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dsprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dsprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_csprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_csprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zsprfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsprfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zsprfs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_sspsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sspsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dspsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dspsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cspsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cspsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cspsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zspsv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zspsv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zspsv(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sspsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sspsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_sspsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dspsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dspsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dspsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_cspsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cspsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_cspsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zspsvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zspsvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zspsvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ssptrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, d : [] c_float, e : [] c_float, tau : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, d : [] c_float, e : [] c_float, tau : [] c_float) : c_int{
  require header;
  return LAPACKE_ssptrd(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_dsptrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, d : [] c_double, e : [] c_double, tau : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, d : [] c_double, e : [] c_double, tau : [] c_double) : c_int{
  require header;
  return LAPACKE_dsptrd(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_ssptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_ssptrf(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_dsptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_dsptrf(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_csptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_csptrf(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zsptrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsptrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zsptrf(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_ssptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_ssptri(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_dsptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_dsptri(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_csptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_csptri(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zsptri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsptri(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zsptri(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_ssptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_ssptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dsptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dsptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_csptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_csptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zsptrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsptrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zsptrs(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sstebz(range : c_char, order : c_char, n : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, d : [] c_float, e : [] c_float, ref m : c_int, ref nsplit : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstebz(range : string, order : string, n : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, d : [] c_float, e : [] c_float, ref m : c_int, ref nsplit : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int) : c_int{
  require header;
  return LAPACKE_sstebz(range.toByte() : c_char, order.toByte() : c_char, n, vl, vu, il, iu, abstol, d, e, m, nsplit, w, iblock, isplit);
}

pragma "no doc"
extern proc LAPACKE_dstebz(range : c_char, order : c_char, n : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, d : [] c_double, e : [] c_double, ref m : c_int, ref nsplit : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstebz(range : string, order : string, n : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, d : [] c_double, e : [] c_double, ref m : c_int, ref nsplit : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int) : c_int{
  require header;
  return LAPACKE_dstebz(range.toByte() : c_char, order.toByte() : c_char, n, vl, vu, il, iu, abstol, d, e, m, nsplit, w, iblock, isplit);
}

pragma "no doc"
extern proc LAPACKE_sstedc(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstedc(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sstedc(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dstedc(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstedc(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dstedc(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_cstedc(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cstedc(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_cstedc(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zstedc(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zstedc(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zstedc(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sstegr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstegr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_sstegr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_dstegr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstegr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_dstegr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_cstegr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cstegr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_cstegr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_zstegr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zstegr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_zstegr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_sstein(matrix_order : lapack_memory_order, n : c_int, d : [] c_float, e : [] c_float, m : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int, z : [] c_float, ldz : c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dstein(matrix_order : lapack_memory_order, n : c_int, d : [] c_double, e : [] c_double, m : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int, z : [] c_double, ldz : c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cstein(matrix_order : lapack_memory_order, n : c_int, d : [] c_float, e : [] c_float, m : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int, z : [] complex(64), ldz : c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zstein(matrix_order : lapack_memory_order, n : c_int, d : [] c_double, e : [] c_double, m : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int, z : [] complex(128), ldz : c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sstemr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstemr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int{
  require header;
  return LAPACKE_sstemr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac);
}

pragma "no doc"
extern proc LAPACKE_dstemr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstemr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int{
  require header;
  return LAPACKE_dstemr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac);
}

pragma "no doc"
extern proc LAPACKE_cstemr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cstemr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int{
  require header;
  return LAPACKE_cstemr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac);
}

pragma "no doc"
extern proc LAPACKE_zstemr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zstemr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int) : c_int{
  require header;
  return LAPACKE_zstemr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac);
}

pragma "no doc"
extern proc LAPACKE_ssteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_ssteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dsteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dsteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_csteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_csteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zsteqr(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsteqr(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zsteqr(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_ssterf(n : c_int, d : [] c_float, e : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dsterf(n : c_int, d : [] c_double, e : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sstev(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstev(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sstev(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dstev(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstev(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dstev(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sstevd(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstevd(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sstevd(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dstevd(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstevd(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dstevd(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sstevr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstevr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_sstevr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_dstevr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstevr(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_dstevr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_sstevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstevx(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_sstevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_dstevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstevx(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dstevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssycon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssycon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_ssycon(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_dsycon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsycon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dsycon(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_csycon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csycon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_csycon(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_zsycon(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsycon(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_zsycon(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond);
}

pragma "no doc"
extern proc LAPACKE_ssyequb(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyequb(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_ssyequb(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_dsyequb(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyequb(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_dsyequb(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_csyequb(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csyequb(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_csyequb(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_zsyequb(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsyequb(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_zsyequb(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_ssyev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_ssyev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_dsyev(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyev(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_dsyev(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_ssyevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_ssyevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_dsyevd(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyevd(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_dsyevd(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w);
}

pragma "no doc"
extern proc LAPACKE_ssyevr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyevr(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_ssyevr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_dsyevr(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyevr(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int) : c_int{
  require header;
  return LAPACKE_dsyevr(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz);
}

pragma "no doc"
extern proc LAPACKE_ssyevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_ssyevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_dsyevx(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyevx(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dsyevx(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssygst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssygst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_ssygst(matrix_order, itype, uplo.toByte() : c_char, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dsygst(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsygst(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dsygst(matrix_order, itype, uplo.toByte() : c_char, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ssygv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssygv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_ssygv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_dsygv(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsygv(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_dsygv(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_ssygvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, w : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssygvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, w : [] c_float) : c_int{
  require header;
  return LAPACKE_ssygvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_dsygvd(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, w : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsygvd(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, w : [] c_double) : c_int{
  require header;
  return LAPACKE_dsygvd(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w);
}

pragma "no doc"
extern proc LAPACKE_ssygvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssygvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_ssygvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_dsygvx(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsygvx(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dsygvx(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssyrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_ssyrfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dsyrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dsyrfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_csyrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csyrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_csyrfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zsyrfs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsyrfs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zsyrfs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ssyrfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyrfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_ssyrfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dsyrfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyrfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dsyrfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_csyrfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csyrfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_csyrfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zsyrfsx(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsyrfsx(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zsyrfsx(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_ssysv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssysv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_ssysv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dsysv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsysv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dsysv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_csysv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csysv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_csysv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zsysv(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsysv(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zsysv(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ssysvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssysvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_ssysvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dsysvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsysvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dsysvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_csysvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csysvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_csysvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_zsysvx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsysvx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_zsysvx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ssysvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssysvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_ssysvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_dsysvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsysvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_dsysvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_csysvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csysvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float) : c_int{
  require header;
  return LAPACKE_csysvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_zsysvxx(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsysvxx(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double) : c_int{
  require header;
  return LAPACKE_zsysvxx(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params);
}

pragma "no doc"
extern proc LAPACKE_ssytrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, d : [] c_float, e : [] c_float, tau : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssytrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, d : [] c_float, e : [] c_float, tau : [] c_float) : c_int{
  require header;
  return LAPACKE_ssytrd(matrix_order, uplo.toByte() : c_char, n, a, lda, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_dsytrd(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, d : [] c_double, e : [] c_double, tau : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsytrd(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, d : [] c_double, e : [] c_double, tau : [] c_double) : c_int{
  require header;
  return LAPACKE_dsytrd(matrix_order, uplo.toByte() : c_char, n, a, lda, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_ssytrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssytrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_ssytrf(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_dsytrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsytrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_dsytrf(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_csytrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csytrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_csytrf(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zsytrf(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsytrf(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zsytrf(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_ssytri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssytri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_ssytri(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_dsytri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsytri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_dsytri(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_csytri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csytri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_csytri(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zsytri(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsytri(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zsytri(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv);
}

pragma "no doc"
extern proc LAPACKE_ssytrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssytrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_ssytrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dsytrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsytrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dsytrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_csytrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csytrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_csytrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zsytrs(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsytrs(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zsytrs(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_stbcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stbcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_stbcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, kd, ab, ldab, rcond);
}

pragma "no doc"
extern proc LAPACKE_dtbcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtbcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dtbcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, kd, ab, ldab, rcond);
}

pragma "no doc"
extern proc LAPACKE_ctbcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctbcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_ctbcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, kd, ab, ldab, rcond);
}

pragma "no doc"
extern proc LAPACKE_ztbcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztbcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_ztbcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, kd, ab, ldab, rcond);
}

pragma "no doc"
extern proc LAPACKE_stbrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stbrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_stbrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dtbrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtbrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dtbrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ctbrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctbrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_ctbrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ztbrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztbrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_ztbrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_stbtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stbtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_stbtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dtbtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtbtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dtbtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ctbtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctbtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ctbtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ztbtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztbtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ztbtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_stfsm(matrix_order : lapack_memory_order, transr : c_char, side : c_char, uplo : c_char, trans : c_char, diag : c_char, m : c_int, n : c_int, alpha : c_float, a : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stfsm(matrix_order : lapack_memory_order, transr : string, side : string, uplo : string, trans : string, diag : string, m : c_int, n : c_int, alpha : c_float, a : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_stfsm(matrix_order, transr.toByte() : c_char, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, m, n, alpha, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dtfsm(matrix_order : lapack_memory_order, transr : c_char, side : c_char, uplo : c_char, trans : c_char, diag : c_char, m : c_int, n : c_int, alpha : c_double, a : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtfsm(matrix_order : lapack_memory_order, transr : string, side : string, uplo : string, trans : string, diag : string, m : c_int, n : c_int, alpha : c_double, a : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dtfsm(matrix_order, transr.toByte() : c_char, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, m, n, alpha, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ctfsm(matrix_order : lapack_memory_order, transr : c_char, side : c_char, uplo : c_char, trans : c_char, diag : c_char, m : c_int, n : c_int, alpha : complex(64), a : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctfsm(matrix_order : lapack_memory_order, transr : string, side : string, uplo : string, trans : string, diag : string, m : c_int, n : c_int, alpha : complex(64), a : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ctfsm(matrix_order, transr.toByte() : c_char, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, m, n, alpha, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ztfsm(matrix_order : lapack_memory_order, transr : c_char, side : c_char, uplo : c_char, trans : c_char, diag : c_char, m : c_int, n : c_int, alpha : complex(128), a : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztfsm(matrix_order : lapack_memory_order, transr : string, side : string, uplo : string, trans : string, diag : string, m : c_int, n : c_int, alpha : complex(128), a : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ztfsm(matrix_order, transr.toByte() : c_char, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, m, n, alpha, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_stftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stftri(matrix_order : lapack_memory_order, transr : string, uplo : string, diag : string, n : c_int, a : [] c_float) : c_int{
  require header;
  return LAPACKE_stftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_dtftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtftri(matrix_order : lapack_memory_order, transr : string, uplo : string, diag : string, n : c_int, a : [] c_double) : c_int{
  require header;
  return LAPACKE_dtftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_ctftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_ctftri(matrix_order : lapack_memory_order, transr : string, uplo : string, diag : string, n : c_int, a : [] complex(64)) : c_int{
  require header;
  return LAPACKE_ctftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_ztftri(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_ztftri(matrix_order : lapack_memory_order, transr : string, uplo : string, diag : string, n : c_int, a : [] complex(128)) : c_int{
  require header;
  return LAPACKE_ztftri(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_stfttp(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] c_float, ap : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stfttp(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] c_float, ap : [] c_float) : c_int{
  require header;
  return LAPACKE_stfttp(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, ap);
}

pragma "no doc"
extern proc LAPACKE_dtfttp(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] c_double, ap : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtfttp(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] c_double, ap : [] c_double) : c_int{
  require header;
  return LAPACKE_dtfttp(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, ap);
}

pragma "no doc"
extern proc LAPACKE_ctfttp(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] complex(64), ap : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_ctfttp(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] complex(64), ap : [] complex(64)) : c_int{
  require header;
  return LAPACKE_ctfttp(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, ap);
}

pragma "no doc"
extern proc LAPACKE_ztfttp(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] complex(128), ap : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_ztfttp(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] complex(128), ap : [] complex(128)) : c_int{
  require header;
  return LAPACKE_ztfttp(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, ap);
}

pragma "no doc"
extern proc LAPACKE_stfttr(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] c_float, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stfttr(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] c_float, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_stfttr(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dtfttr(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] c_double, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtfttr(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] c_double, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dtfttr(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, a, lda);
}

pragma "no doc"
extern proc LAPACKE_ctfttr(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] complex(64), a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctfttr(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] complex(64), a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_ctfttr(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, a, lda);
}

pragma "no doc"
extern proc LAPACKE_ztfttr(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, arf : [] complex(128), a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztfttr(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, arf : [] complex(128), a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_ztfttr(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, arf, a, lda);
}

pragma "no doc"
extern proc LAPACKE_stgevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, s : [] c_float, lds : c_int, p : [] c_float, ldp : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stgevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, s : [] c_float, lds : c_int, p : [] c_float, ldp : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_stgevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, s, lds, p, ldp, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_dtgevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, s : [] c_double, lds : c_int, p : [] c_double, ldp : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtgevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, s : [] c_double, lds : c_int, p : [] c_double, ldp : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_dtgevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, s, lds, p, ldp, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ctgevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, s : [] complex(64), lds : c_int, p : [] complex(64), ldp : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctgevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, s : [] complex(64), lds : c_int, p : [] complex(64), ldp : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ctgevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, s, lds, p, ldp, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ztgevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, s : [] complex(128), lds : c_int, p : [] complex(128), ldp : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztgevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, s : [] complex(128), lds : c_int, p : [] complex(128), ldp : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ztgevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, s, lds, p, ldp, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_stgexc(matrix_order : lapack_memory_order, wantq : c_int, wantz : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int, ref ifst : c_int, ref ilst : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dtgexc(matrix_order : lapack_memory_order, wantq : c_int, wantz : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int, ref ifst : c_int, ref ilst : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_ctgexc(matrix_order : lapack_memory_order, wantq : c_int, wantz : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int, ifst : c_int, ilst : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_ztgexc(matrix_order : lapack_memory_order, wantq : c_int, wantz : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int, ifst : c_int, ilst : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_stgsen(matrix_order : lapack_memory_order, ijob : c_int, wantq : c_int, wantz : c_int, chlapack_select : [] c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int, ref m : c_int, ref pl : c_float, ref pr : c_float, dif : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dtgsen(matrix_order : lapack_memory_order, ijob : c_int, wantq : c_int, wantz : c_int, chlapack_select : [] c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int, ref m : c_int, ref pl : c_double, ref pr : c_double, dif : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_ctgsen(matrix_order : lapack_memory_order, ijob : c_int, wantq : c_int, wantz : c_int, chlapack_select : [] c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int, ref m : c_int, ref pl : c_float, ref pr : c_float, dif : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_ztgsen(matrix_order : lapack_memory_order, ijob : c_int, wantq : c_int, wantz : c_int, chlapack_select : [] c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int, ref m : c_int, ref pl : c_double, ref pr : c_double, dif : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_stgsja(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, tola : c_float, tolb : c_float, alpha : [] c_float, beta : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, ref ncycle : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stgsja(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, tola : c_float, tolb : c_float, alpha : [] c_float, beta : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, ref ncycle : c_int) : c_int{
  require header;
  return LAPACKE_stgsja(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, k, l, a, lda, b, ldb, tola, tolb, alpha, beta, u, ldu, v, ldv, q, ldq, ncycle);
}

pragma "no doc"
extern proc LAPACKE_dtgsja(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, tola : c_double, tolb : c_double, alpha : [] c_double, beta : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, ref ncycle : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtgsja(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, tola : c_double, tolb : c_double, alpha : [] c_double, beta : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, ref ncycle : c_int) : c_int{
  require header;
  return LAPACKE_dtgsja(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, k, l, a, lda, b, ldb, tola, tolb, alpha, beta, u, ldu, v, ldv, q, ldq, ncycle);
}

pragma "no doc"
extern proc LAPACKE_ctgsja(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, tola : c_float, tolb : c_float, alpha : [] c_float, beta : [] c_float, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, ref ncycle : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctgsja(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, tola : c_float, tolb : c_float, alpha : [] c_float, beta : [] c_float, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, ref ncycle : c_int) : c_int{
  require header;
  return LAPACKE_ctgsja(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, k, l, a, lda, b, ldb, tola, tolb, alpha, beta, u, ldu, v, ldv, q, ldq, ncycle);
}

pragma "no doc"
extern proc LAPACKE_ztgsja(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, tola : c_double, tolb : c_double, alpha : [] c_double, beta : [] c_double, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, ref ncycle : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztgsja(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, tola : c_double, tolb : c_double, alpha : [] c_double, beta : [] c_double, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, ref ncycle : c_int) : c_int{
  require header;
  return LAPACKE_ztgsja(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, k, l, a, lda, b, ldb, tola, tolb, alpha, beta, u, ldu, v, ldv, q, ldq, ncycle);
}

pragma "no doc"
extern proc LAPACKE_stgsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, s : [] c_float, dif : [] c_float, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stgsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, s : [] c_float, dif : [] c_float, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_stgsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, a, lda, b, ldb, vl, ldvl, vr, ldvr, s, dif, mm, m);
}

pragma "no doc"
extern proc LAPACKE_dtgsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, s : [] c_double, dif : [] c_double, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtgsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, s : [] c_double, dif : [] c_double, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_dtgsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, a, lda, b, ldb, vl, ldvl, vr, ldvr, s, dif, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ctgsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, s : [] c_float, dif : [] c_float, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctgsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, s : [] c_float, dif : [] c_float, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ctgsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, a, lda, b, ldb, vl, ldvl, vr, ldvr, s, dif, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ztgsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, s : [] c_double, dif : [] c_double, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztgsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, s : [] c_double, dif : [] c_double, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ztgsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, a, lda, b, ldb, vl, ldvl, vr, ldvr, s, dif, mm, m);
}

pragma "no doc"
extern proc LAPACKE_stgsyl(matrix_order : lapack_memory_order, trans : c_char, ijob : c_int, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, c : [] c_float, ldc : c_int, d : [] c_float, ldd : c_int, e : [] c_float, lde : c_int, f : [] c_float, ldf : c_int, ref scale : c_float, ref dif : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stgsyl(matrix_order : lapack_memory_order, trans : string, ijob : c_int, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, c : [] c_float, ldc : c_int, d : [] c_float, ldd : c_int, e : [] c_float, lde : c_int, f : [] c_float, ldf : c_int, ref scale : c_float, ref dif : c_float) : c_int{
  require header;
  return LAPACKE_stgsyl(matrix_order, trans.toByte() : c_char, ijob, m, n, a, lda, b, ldb, c, ldc, d, ldd, e, lde, f, ldf, scale, dif);
}

pragma "no doc"
extern proc LAPACKE_dtgsyl(matrix_order : lapack_memory_order, trans : c_char, ijob : c_int, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, c : [] c_double, ldc : c_int, d : [] c_double, ldd : c_int, e : [] c_double, lde : c_int, f : [] c_double, ldf : c_int, ref scale : c_double, ref dif : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtgsyl(matrix_order : lapack_memory_order, trans : string, ijob : c_int, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, c : [] c_double, ldc : c_int, d : [] c_double, ldd : c_int, e : [] c_double, lde : c_int, f : [] c_double, ldf : c_int, ref scale : c_double, ref dif : c_double) : c_int{
  require header;
  return LAPACKE_dtgsyl(matrix_order, trans.toByte() : c_char, ijob, m, n, a, lda, b, ldb, c, ldc, d, ldd, e, lde, f, ldf, scale, dif);
}

pragma "no doc"
extern proc LAPACKE_ctgsyl(matrix_order : lapack_memory_order, trans : c_char, ijob : c_int, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, c : [] complex(64), ldc : c_int, d : [] complex(64), ldd : c_int, e : [] complex(64), lde : c_int, f : [] complex(64), ldf : c_int, ref scale : c_float, ref dif : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctgsyl(matrix_order : lapack_memory_order, trans : string, ijob : c_int, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, c : [] complex(64), ldc : c_int, d : [] complex(64), ldd : c_int, e : [] complex(64), lde : c_int, f : [] complex(64), ldf : c_int, ref scale : c_float, ref dif : c_float) : c_int{
  require header;
  return LAPACKE_ctgsyl(matrix_order, trans.toByte() : c_char, ijob, m, n, a, lda, b, ldb, c, ldc, d, ldd, e, lde, f, ldf, scale, dif);
}

pragma "no doc"
extern proc LAPACKE_ztgsyl(matrix_order : lapack_memory_order, trans : c_char, ijob : c_int, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, c : [] complex(128), ldc : c_int, d : [] complex(128), ldd : c_int, e : [] complex(128), lde : c_int, f : [] complex(128), ldf : c_int, ref scale : c_double, ref dif : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztgsyl(matrix_order : lapack_memory_order, trans : string, ijob : c_int, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, c : [] complex(128), ldc : c_int, d : [] complex(128), ldd : c_int, e : [] complex(128), lde : c_int, f : [] complex(128), ldf : c_int, ref scale : c_double, ref dif : c_double) : c_int{
  require header;
  return LAPACKE_ztgsyl(matrix_order, trans.toByte() : c_char, ijob, m, n, a, lda, b, ldb, c, ldc, d, ldd, e, lde, f, ldf, scale, dif);
}

pragma "no doc"
extern proc LAPACKE_stpcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, ap : [] c_float, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stpcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, ap : [] c_float, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_stpcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap, rcond);
}

pragma "no doc"
extern proc LAPACKE_dtpcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, ap : [] c_double, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtpcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, ap : [] c_double, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dtpcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap, rcond);
}

pragma "no doc"
extern proc LAPACKE_ctpcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, ap : [] complex(64), ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctpcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, ap : [] complex(64), ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_ctpcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap, rcond);
}

pragma "no doc"
extern proc LAPACKE_ztpcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, ap : [] complex(128), ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztpcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, ap : [] complex(128), ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_ztpcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap, rcond);
}

pragma "no doc"
extern proc LAPACKE_stprfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stprfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_stprfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dtprfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtprfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dtprfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ctprfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctprfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_ctprfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ztprfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztprfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_ztprfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_stptri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, ap : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stptri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, ap : [] c_float) : c_int{
  require header;
  return LAPACKE_stptri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_dtptri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, ap : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtptri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, ap : [] c_double) : c_int{
  require header;
  return LAPACKE_dtptri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_ctptri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, ap : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_ctptri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, ap : [] complex(64)) : c_int{
  require header;
  return LAPACKE_ctptri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_ztptri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, ap : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_ztptri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, ap : [] complex(128)) : c_int{
  require header;
  return LAPACKE_ztptri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_stptrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stptrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_stptrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dtptrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtptrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dtptrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ctptrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctptrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ctptrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ztptrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztptrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ztptrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_stpttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, ap : [] c_float, arf : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_stpttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, ap : [] c_float, arf : [] c_float) : c_int{
  require header;
  return LAPACKE_stpttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, ap, arf);
}

pragma "no doc"
extern proc LAPACKE_dtpttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, ap : [] c_double, arf : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtpttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, ap : [] c_double, arf : [] c_double) : c_int{
  require header;
  return LAPACKE_dtpttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, ap, arf);
}

pragma "no doc"
extern proc LAPACKE_ctpttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, ap : [] complex(64), arf : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_ctpttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, ap : [] complex(64), arf : [] complex(64)) : c_int{
  require header;
  return LAPACKE_ctpttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, ap, arf);
}

pragma "no doc"
extern proc LAPACKE_ztpttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, ap : [] complex(128), arf : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_ztpttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, ap : [] complex(128), arf : [] complex(128)) : c_int{
  require header;
  return LAPACKE_ztpttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, ap, arf);
}

pragma "no doc"
extern proc LAPACKE_stpttr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_stpttr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_stpttr(matrix_order, uplo.toByte() : c_char, n, ap, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dtpttr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtpttr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dtpttr(matrix_order, uplo.toByte() : c_char, n, ap, a, lda);
}

pragma "no doc"
extern proc LAPACKE_ctpttr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctpttr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_ctpttr(matrix_order, uplo.toByte() : c_char, n, ap, a, lda);
}

pragma "no doc"
extern proc LAPACKE_ztpttr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztpttr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_ztpttr(matrix_order, uplo.toByte() : c_char, n, ap, a, lda);
}

pragma "no doc"
extern proc LAPACKE_strcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] c_float, lda : c_int, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_strcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, a : [] c_float, lda : c_int, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_strcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda, rcond);
}

pragma "no doc"
extern proc LAPACKE_dtrcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] c_double, lda : c_int, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, a : [] c_double, lda : c_int, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_dtrcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda, rcond);
}

pragma "no doc"
extern proc LAPACKE_ctrcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] complex(64), lda : c_int, ref rcond : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, a : [] complex(64), lda : c_int, ref rcond : c_float) : c_int{
  require header;
  return LAPACKE_ctrcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda, rcond);
}

pragma "no doc"
extern proc LAPACKE_ztrcon(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, n : c_int, a : [] complex(128), lda : c_int, ref rcond : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrcon(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, n : c_int, a : [] complex(128), lda : c_int, ref rcond : c_double) : c_int{
  require header;
  return LAPACKE_ztrcon(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda, rcond);
}

pragma "no doc"
extern proc LAPACKE_strevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] c_float, ldt : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_strevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] c_float, ldt : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_strevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_dtrevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] c_double, ldt : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] c_double, ldt : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_dtrevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ctrevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] complex(64), ldt : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] complex(64), ldt : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ctrevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ztrevc(matrix_order : lapack_memory_order, side : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] complex(128), ldt : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrevc(matrix_order : lapack_memory_order, side : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] complex(128), ldt : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ztrevc(matrix_order, side.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, mm, m);
}

pragma "no doc"
extern proc LAPACKE_strexc(matrix_order : lapack_memory_order, compq : c_char, n : c_int, t : [] c_float, ldt : c_int, q : [] c_float, ldq : c_int, ref ifst : c_int, ref ilst : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_strexc(matrix_order : lapack_memory_order, compq : string, n : c_int, t : [] c_float, ldt : c_int, q : [] c_float, ldq : c_int, ref ifst : c_int, ref ilst : c_int) : c_int{
  require header;
  return LAPACKE_strexc(matrix_order, compq.toByte() : c_char, n, t, ldt, q, ldq, ifst, ilst);
}

pragma "no doc"
extern proc LAPACKE_dtrexc(matrix_order : lapack_memory_order, compq : c_char, n : c_int, t : [] c_double, ldt : c_int, q : [] c_double, ldq : c_int, ref ifst : c_int, ref ilst : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrexc(matrix_order : lapack_memory_order, compq : string, n : c_int, t : [] c_double, ldt : c_int, q : [] c_double, ldq : c_int, ref ifst : c_int, ref ilst : c_int) : c_int{
  require header;
  return LAPACKE_dtrexc(matrix_order, compq.toByte() : c_char, n, t, ldt, q, ldq, ifst, ilst);
}

pragma "no doc"
extern proc LAPACKE_ctrexc(matrix_order : lapack_memory_order, compq : c_char, n : c_int, t : [] complex(64), ldt : c_int, q : [] complex(64), ldq : c_int, ifst : c_int, ilst : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrexc(matrix_order : lapack_memory_order, compq : string, n : c_int, t : [] complex(64), ldt : c_int, q : [] complex(64), ldq : c_int, ifst : c_int, ilst : c_int) : c_int{
  require header;
  return LAPACKE_ctrexc(matrix_order, compq.toByte() : c_char, n, t, ldt, q, ldq, ifst, ilst);
}

pragma "no doc"
extern proc LAPACKE_ztrexc(matrix_order : lapack_memory_order, compq : c_char, n : c_int, t : [] complex(128), ldt : c_int, q : [] complex(128), ldq : c_int, ifst : c_int, ilst : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrexc(matrix_order : lapack_memory_order, compq : string, n : c_int, t : [] complex(128), ldt : c_int, q : [] complex(128), ldq : c_int, ifst : c_int, ilst : c_int) : c_int{
  require header;
  return LAPACKE_ztrexc(matrix_order, compq.toByte() : c_char, n, t, ldt, q, ldq, ifst, ilst);
}

pragma "no doc"
extern proc LAPACKE_strrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_strrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_strrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_dtrrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_dtrrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ctrrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float) : c_int{
  require header;
  return LAPACKE_ctrrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_ztrrfs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrrfs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double) : c_int{
  require header;
  return LAPACKE_ztrrfs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, ferr, berr);
}

pragma "no doc"
extern proc LAPACKE_strsen(matrix_order : lapack_memory_order, job : c_char, compq : c_char, chlapack_select : [] c_int, n : c_int, t : [] c_float, ldt : c_int, q : [] c_float, ldq : c_int, wr : [] c_float, wi : [] c_float, ref m : c_int, ref s : c_float, ref sep : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_strsen(matrix_order : lapack_memory_order, job : string, compq : string, chlapack_select : [] c_int, n : c_int, t : [] c_float, ldt : c_int, q : [] c_float, ldq : c_int, wr : [] c_float, wi : [] c_float, ref m : c_int, ref s : c_float, ref sep : c_float) : c_int{
  require header;
  return LAPACKE_strsen(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, chlapack_select, n, t, ldt, q, ldq, wr, wi, m, s, sep);
}

pragma "no doc"
extern proc LAPACKE_dtrsen(matrix_order : lapack_memory_order, job : c_char, compq : c_char, chlapack_select : [] c_int, n : c_int, t : [] c_double, ldt : c_int, q : [] c_double, ldq : c_int, wr : [] c_double, wi : [] c_double, ref m : c_int, ref s : c_double, ref sep : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrsen(matrix_order : lapack_memory_order, job : string, compq : string, chlapack_select : [] c_int, n : c_int, t : [] c_double, ldt : c_int, q : [] c_double, ldq : c_int, wr : [] c_double, wi : [] c_double, ref m : c_int, ref s : c_double, ref sep : c_double) : c_int{
  require header;
  return LAPACKE_dtrsen(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, chlapack_select, n, t, ldt, q, ldq, wr, wi, m, s, sep);
}

pragma "no doc"
extern proc LAPACKE_ctrsen(matrix_order : lapack_memory_order, job : c_char, compq : c_char, chlapack_select : [] c_int, n : c_int, t : [] complex(64), ldt : c_int, q : [] complex(64), ldq : c_int, w : [] complex(64), ref m : c_int, ref s : c_float, ref sep : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrsen(matrix_order : lapack_memory_order, job : string, compq : string, chlapack_select : [] c_int, n : c_int, t : [] complex(64), ldt : c_int, q : [] complex(64), ldq : c_int, w : [] complex(64), ref m : c_int, ref s : c_float, ref sep : c_float) : c_int{
  require header;
  return LAPACKE_ctrsen(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, chlapack_select, n, t, ldt, q, ldq, w, m, s, sep);
}

pragma "no doc"
extern proc LAPACKE_ztrsen(matrix_order : lapack_memory_order, job : c_char, compq : c_char, chlapack_select : [] c_int, n : c_int, t : [] complex(128), ldt : c_int, q : [] complex(128), ldq : c_int, w : [] complex(128), ref m : c_int, ref s : c_double, ref sep : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrsen(matrix_order : lapack_memory_order, job : string, compq : string, chlapack_select : [] c_int, n : c_int, t : [] complex(128), ldt : c_int, q : [] complex(128), ldq : c_int, w : [] complex(128), ref m : c_int, ref s : c_double, ref sep : c_double) : c_int{
  require header;
  return LAPACKE_ztrsen(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, chlapack_select, n, t, ldt, q, ldq, w, m, s, sep);
}

pragma "no doc"
extern proc LAPACKE_strsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] c_float, ldt : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, s : [] c_float, sep : [] c_float, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_strsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] c_float, ldt : c_int, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, s : [] c_float, sep : [] c_float, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_strsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, s, sep, mm, m);
}

pragma "no doc"
extern proc LAPACKE_dtrsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] c_double, ldt : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, s : [] c_double, sep : [] c_double, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] c_double, ldt : c_int, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, s : [] c_double, sep : [] c_double, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_dtrsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, s, sep, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ctrsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] complex(64), ldt : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, s : [] c_float, sep : [] c_float, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] complex(64), ldt : c_int, vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, s : [] c_float, sep : [] c_float, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ctrsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, s, sep, mm, m);
}

pragma "no doc"
extern proc LAPACKE_ztrsna(matrix_order : lapack_memory_order, job : c_char, howmny : c_char, chlapack_select : [] c_int, n : c_int, t : [] complex(128), ldt : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, s : [] c_double, sep : [] c_double, mm : c_int, ref m : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrsna(matrix_order : lapack_memory_order, job : string, howmny : string, chlapack_select : [] c_int, n : c_int, t : [] complex(128), ldt : c_int, vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, s : [] c_double, sep : [] c_double, mm : c_int, ref m : c_int) : c_int{
  require header;
  return LAPACKE_ztrsna(matrix_order, job.toByte() : c_char, howmny.toByte() : c_char, chlapack_select, n, t, ldt, vl, ldvl, vr, ldvr, s, sep, mm, m);
}

pragma "no doc"
extern proc LAPACKE_strsyl(matrix_order : lapack_memory_order, trana : c_char, tranb : c_char, isgn : c_int, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, c : [] c_float, ldc : c_int, ref scale : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_strsyl(matrix_order : lapack_memory_order, trana : string, tranb : string, isgn : c_int, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, c : [] c_float, ldc : c_int, ref scale : c_float) : c_int{
  require header;
  return LAPACKE_strsyl(matrix_order, trana.toByte() : c_char, tranb.toByte() : c_char, isgn, m, n, a, lda, b, ldb, c, ldc, scale);
}

pragma "no doc"
extern proc LAPACKE_dtrsyl(matrix_order : lapack_memory_order, trana : c_char, tranb : c_char, isgn : c_int, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, c : [] c_double, ldc : c_int, ref scale : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrsyl(matrix_order : lapack_memory_order, trana : string, tranb : string, isgn : c_int, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, c : [] c_double, ldc : c_int, ref scale : c_double) : c_int{
  require header;
  return LAPACKE_dtrsyl(matrix_order, trana.toByte() : c_char, tranb.toByte() : c_char, isgn, m, n, a, lda, b, ldb, c, ldc, scale);
}

pragma "no doc"
extern proc LAPACKE_ctrsyl(matrix_order : lapack_memory_order, trana : c_char, tranb : c_char, isgn : c_int, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, c : [] complex(64), ldc : c_int, ref scale : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrsyl(matrix_order : lapack_memory_order, trana : string, tranb : string, isgn : c_int, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, c : [] complex(64), ldc : c_int, ref scale : c_float) : c_int{
  require header;
  return LAPACKE_ctrsyl(matrix_order, trana.toByte() : c_char, tranb.toByte() : c_char, isgn, m, n, a, lda, b, ldb, c, ldc, scale);
}

pragma "no doc"
extern proc LAPACKE_ztrsyl(matrix_order : lapack_memory_order, trana : c_char, tranb : c_char, isgn : c_int, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, c : [] complex(128), ldc : c_int, ref scale : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrsyl(matrix_order : lapack_memory_order, trana : string, tranb : string, isgn : c_int, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, c : [] complex(128), ldc : c_int, ref scale : c_double) : c_int{
  require header;
  return LAPACKE_ztrsyl(matrix_order, trana.toByte() : c_char, tranb.toByte() : c_char, isgn, m, n, a, lda, b, ldb, c, ldc, scale);
}

pragma "no doc"
extern proc LAPACKE_strtri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_strtri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_strtri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dtrtri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrtri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dtrtri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_ctrtri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrtri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_ctrtri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_ztrtri(matrix_order : lapack_memory_order, uplo : c_char, diag : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrtri(matrix_order : lapack_memory_order, uplo : string, diag : string, n : c_int, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_ztrtri(matrix_order, uplo.toByte() : c_char, diag.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_strtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_strtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_strtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dtrtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dtrtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ctrtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ctrtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ztrtrs(matrix_order : lapack_memory_order, uplo : c_char, trans : c_char, diag : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrtrs(matrix_order : lapack_memory_order, uplo : string, trans : string, diag : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_ztrtrs(matrix_order, uplo.toByte() : c_char, trans.toByte() : c_char, diag.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_strttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, arf : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_strttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, arf : [] c_float) : c_int{
  require header;
  return LAPACKE_strttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, arf);
}

pragma "no doc"
extern proc LAPACKE_dtrttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, arf : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, arf : [] c_double) : c_int{
  require header;
  return LAPACKE_dtrttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, arf);
}

pragma "no doc"
extern proc LAPACKE_ctrttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, arf : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, arf : [] complex(64)) : c_int{
  require header;
  return LAPACKE_ctrttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, arf);
}

pragma "no doc"
extern proc LAPACKE_ztrttf(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, arf : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrttf(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, arf : [] complex(128)) : c_int{
  require header;
  return LAPACKE_ztrttf(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, arf);
}

pragma "no doc"
extern proc LAPACKE_strttp(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, ap : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_strttp(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, ap : [] c_float) : c_int{
  require header;
  return LAPACKE_strttp(matrix_order, uplo.toByte() : c_char, n, a, lda, ap);
}

pragma "no doc"
extern proc LAPACKE_dtrttp(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, ap : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dtrttp(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, ap : [] c_double) : c_int{
  require header;
  return LAPACKE_dtrttp(matrix_order, uplo.toByte() : c_char, n, a, lda, ap);
}

pragma "no doc"
extern proc LAPACKE_ctrttp(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ap : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_ctrttp(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ap : [] complex(64)) : c_int{
  require header;
  return LAPACKE_ctrttp(matrix_order, uplo.toByte() : c_char, n, a, lda, ap);
}

pragma "no doc"
extern proc LAPACKE_ztrttp(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ap : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_ztrttp(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ap : [] complex(128)) : c_int{
  require header;
  return LAPACKE_ztrttp(matrix_order, uplo.toByte() : c_char, n, a, lda, ap);
}

pragma "no doc"
extern proc LAPACKE_stzrzf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dtzrzf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_ctzrzf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_ztzrzf(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_cungbr(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cungbr(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cungbr(matrix_order, vect.toByte() : c_char, m, n, k, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_zungbr(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zungbr(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zungbr(matrix_order, vect.toByte() : c_char, m, n, k, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_cunghr(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zunghr(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_cunglq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zunglq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_cungql(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zungql(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_cungqr(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zungqr(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_cungrq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zungrq(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_cungtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cungtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cungtr(matrix_order, uplo.toByte() : c_char, n, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_zungtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zungtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zungtr(matrix_order, uplo.toByte() : c_char, n, a, lda, tau);
}

pragma "no doc"
extern proc LAPACKE_cunmbr(matrix_order : lapack_memory_order, vect : c_char, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmbr(matrix_order : lapack_memory_order, vect : string, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmbr(matrix_order, vect.toByte() : c_char, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmbr(matrix_order : lapack_memory_order, vect : c_char, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmbr(matrix_order : lapack_memory_order, vect : string, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmbr(matrix_order, vect.toByte() : c_char, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cunmhr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmhr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmhr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, ilo, ihi, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmhr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmhr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmhr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, ilo, ihi, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cunmlq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmlq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmlq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmlq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmlq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmlq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cunmql(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmql(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmql(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmql(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmql(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmql(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cunmqr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmqr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmqr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmqr(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmqr(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmqr(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cunmrq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmrq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmrq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmrq(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmrq(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmrq(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cunmrz(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmrz(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmrz(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, l, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmrz(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmrz(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, l : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmrz(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, l, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cunmtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cunmtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cunmtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zunmtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zunmtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zunmtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, a, lda, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_cupgtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), tau : [] complex(64), q : [] complex(64), ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cupgtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), tau : [] complex(64), q : [] complex(64), ldq : c_int) : c_int{
  require header;
  return LAPACKE_cupgtr(matrix_order, uplo.toByte() : c_char, n, ap, tau, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_zupgtr(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), tau : [] complex(128), q : [] complex(128), ldq : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zupgtr(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), tau : [] complex(128), q : [] complex(128), ldq : c_int) : c_int{
  require header;
  return LAPACKE_zupgtr(matrix_order, uplo.toByte() : c_char, n, ap, tau, q, ldq);
}

pragma "no doc"
extern proc LAPACKE_cupmtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, ap : [] complex(64), tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cupmtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, ap : [] complex(64), tau : [] complex(64), c : [] complex(64), ldc : c_int) : c_int{
  require header;
  return LAPACKE_cupmtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, ap, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_zupmtr(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, ap : [] complex(128), tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zupmtr(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, ap : [] complex(128), tau : [] complex(128), c : [] complex(128), ldc : c_int) : c_int{
  require header;
  return LAPACKE_zupmtr(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, ap, tau, c, ldc);
}

pragma "no doc"
extern proc LAPACKE_sbdsdc_work(matrix_order : lapack_memory_order, uplo : c_char, compq : c_char, n : c_int, d : [] c_float, e : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, q : [] c_float, iq : [] c_int, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sbdsdc_work(matrix_order : lapack_memory_order, uplo : string, compq : string, n : c_int, d : [] c_float, e : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, q : [] c_float, iq : [] c_int, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sbdsdc_work(matrix_order, uplo.toByte() : c_char, compq.toByte() : c_char, n, d, e, u, ldu, vt, ldvt, q, iq, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dbdsdc_work(matrix_order : lapack_memory_order, uplo : c_char, compq : c_char, n : c_int, d : [] c_double, e : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, q : [] c_double, iq : [] c_int, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dbdsdc_work(matrix_order : lapack_memory_order, uplo : string, compq : string, n : c_int, d : [] c_double, e : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, q : [] c_double, iq : [] c_int, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dbdsdc_work(matrix_order, uplo.toByte() : c_char, compq.toByte() : c_char, n, d, e, u, ldu, vt, ldvt, q, iq, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_sbdsqr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] c_float, ldvt : c_int, u : [] c_float, ldu : c_int, c : [] c_float, ldc : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sbdsqr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] c_float, ldvt : c_int, u : [] c_float, ldu : c_int, c : [] c_float, ldc : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sbdsqr_work(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_dbdsqr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] c_double, ldvt : c_int, u : [] c_double, ldu : c_int, c : [] c_double, ldc : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dbdsqr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] c_double, ldvt : c_int, u : [] c_double, ldu : c_int, c : [] c_double, ldc : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dbdsqr_work(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_cbdsqr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] complex(64), ldvt : c_int, u : [] complex(64), ldu : c_int, c : [] complex(64), ldc : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cbdsqr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_float, e : [] c_float, vt : [] complex(64), ldvt : c_int, u : [] complex(64), ldu : c_int, c : [] complex(64), ldc : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_cbdsqr_work(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_zbdsqr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] complex(128), ldvt : c_int, u : [] complex(128), ldu : c_int, c : [] complex(128), ldc : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zbdsqr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ncvt : c_int, nru : c_int, ncc : c_int, d : [] c_double, e : [] c_double, vt : [] complex(128), ldvt : c_int, u : [] complex(128), ldu : c_int, c : [] complex(128), ldc : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_zbdsqr_work(matrix_order, uplo.toByte() : c_char, n, ncvt, nru, ncc, d, e, vt, ldvt, u, ldu, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_sdisna_work(job : c_char, m : c_int, n : c_int, d : [] c_float, sep : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sdisna_work(job : string, m : c_int, n : c_int, d : [] c_float, sep : [] c_float) : c_int{
  require header;
  return LAPACKE_sdisna_work(job.toByte() : c_char, m, n, d, sep);
}

pragma "no doc"
extern proc LAPACKE_ddisna_work(job : c_char, m : c_int, n : c_int, d : [] c_double, sep : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_ddisna_work(job : string, m : c_int, n : c_int, d : [] c_double, sep : [] c_double) : c_int{
  require header;
  return LAPACKE_ddisna_work(job.toByte() : c_char, m, n, d, sep);
}

pragma "no doc"
extern proc LAPACKE_sgbbrd_work(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int, pt : [] c_float, ldpt : c_int, c : [] c_float, ldc : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbbrd_work(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int, pt : [] c_float, ldpt : c_int, c : [] c_float, ldc : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sgbbrd_work(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_dgbbrd_work(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int, pt : [] c_double, ldpt : c_int, c : [] c_double, ldc : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbbrd_work(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int, pt : [] c_double, ldpt : c_int, c : [] c_double, ldc : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dgbbrd_work(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_cgbbrd_work(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int, pt : [] complex(64), ldpt : c_int, c : [] complex(64), ldc : c_int, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbbrd_work(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int, pt : [] complex(64), ldpt : c_int, c : [] complex(64), ldc : c_int, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbbrd_work(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgbbrd_work(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int, pt : [] complex(128), ldpt : c_int, c : [] complex(128), ldc : c_int, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbbrd_work(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, ncc : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int, pt : [] complex(128), ldpt : c_int, c : [] complex(128), ldc : c_int, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbbrd_work(matrix_order, vect.toByte() : c_char, m, n, ncc, kl, ku, ab, ldab, d, e, q, ldq, pt, ldpt, c, ldc, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgbcon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbcon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgbcon_work(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgbcon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbcon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgbcon_work(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgbcon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbcon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbcon_work(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgbcon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbcon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbcon_work(matrix_order, norm.toByte() : c_char, n, kl, ku, ab, ldab, ipiv, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgbequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgbrfs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgbrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgbrfs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgbrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbrfs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgbrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbrfs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgbrfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbrfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgbrfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgbrfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbrfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgbrfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgbrfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbrfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbrfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgbrfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbrfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbrfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgbsv_work(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbsv_work(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbsv_work(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbsv_work(matrix_order : lapack_memory_order, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgbsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgbsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgbsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgbsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgbsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgbsvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbsvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgbsvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgbsvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbsvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgbsvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgbsvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbsvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgbsvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgbsvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbsvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgbsvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, afb, ldafb, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgbtrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgbtrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgbtrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgbtrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgbtrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgbtrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sgbtrs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dgbtrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgbtrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dgbtrs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cgbtrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgbtrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cgbtrs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zgbtrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgbtrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, kl : c_int, ku : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zgbtrs_work(matrix_order, trans.toByte() : c_char, n, kl, ku, nrhs, ab, ldab, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sgebak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgebak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int{
  require header;
  return LAPACKE_sgebak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_dgebak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgebak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int{
  require header;
  return LAPACKE_dgebak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_cgebak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgebak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int{
  require header;
  return LAPACKE_cgebak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_zgebak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgebak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, scale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int{
  require header;
  return LAPACKE_zgebak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, scale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_sgebal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_float, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sgebal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_float, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int{
  require header;
  return LAPACKE_sgebal_work(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_dgebal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_double, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dgebal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_double, lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int{
  require header;
  return LAPACKE_dgebal_work(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_cgebal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(64), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgebal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(64), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float) : c_int{
  require header;
  return LAPACKE_cgebal_work(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_zgebal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(128), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgebal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(128), lda : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double) : c_int{
  require header;
  return LAPACKE_zgebal_work(matrix_order, job.toByte() : c_char, n, a, lda, ilo, ihi, scale);
}

pragma "no doc"
extern proc LAPACKE_sgebrd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, d : [] c_float, e : [] c_float, tauq : [] c_float, taup : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgebrd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, d : [] c_double, e : [] c_double, tauq : [] c_double, taup : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgebrd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, d : [] c_float, e : [] c_float, tauq : [] complex(64), taup : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgebrd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, d : [] c_double, e : [] c_double, tauq : [] complex(128), taup : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgecon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgecon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgecon_work(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgecon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgecon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgecon_work(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgecon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgecon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgecon_work(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgecon_work(matrix_order : lapack_memory_order, norm : c_char, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgecon_work(matrix_order : lapack_memory_order, norm : string, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgecon_work(matrix_order, norm.toByte() : c_char, n, a, lda, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgeequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeequ_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, r : [] c_float, c : [] c_float, ref rowcnd : c_float, ref colcnd : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeequb_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, r : [] c_double, c : [] c_double, ref rowcnd : c_double, ref colcnd : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgees_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_S_SELECT2, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int, work : [] c_float, lwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgees_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_S_SELECT2, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int, work : [] c_float, lwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgees_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, wr, wi, vs, ldvs, work, lwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_dgees_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_D_SELECT2, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int, work : [] c_double, lwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgees_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_D_SELECT2, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int, work : [] c_double, lwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgees_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, wr, wi, vs, ldvs, work, lwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_cgees_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_C_SELECT1, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgees_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_C_SELECT1, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cgees_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, w, vs, ldvs, work, lwork, rwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_zgees_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_Z_SELECT1, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgees_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_Z_SELECT1, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zgees_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, n, a, lda, sdim, w, vs, ldvs, work, lwork, rwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_sgeesx_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_S_SELECT2, sense : c_char, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int, ref rconde : c_float, ref rcondv : c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgeesx_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_S_SELECT2, sense : string, n : c_int, a : [] c_float, lda : c_int, ref sdim : c_int, wr : [] c_float, wi : [] c_float, vs : [] c_float, ldvs : c_int, ref rconde : c_float, ref rcondv : c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgeesx_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, wr, wi, vs, ldvs, rconde, rcondv, work, lwork, iwork, liwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_dgeesx_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_D_SELECT2, sense : c_char, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int, ref rconde : c_double, ref rcondv : c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgeesx_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_D_SELECT2, sense : string, n : c_int, a : [] c_double, lda : c_int, ref sdim : c_int, wr : [] c_double, wi : [] c_double, vs : [] c_double, ldvs : c_int, ref rconde : c_double, ref rcondv : c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgeesx_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, wr, wi, vs, ldvs, rconde, rcondv, work, lwork, iwork, liwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_cgeesx_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_C_SELECT1, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int, ref rconde : c_float, ref rcondv : c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgeesx_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_C_SELECT1, sense : string, n : c_int, a : [] complex(64), lda : c_int, ref sdim : c_int, w : [] complex(64), vs : [] complex(64), ldvs : c_int, ref rconde : c_float, ref rcondv : c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cgeesx_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv, work, lwork, rwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_zgeesx_work(matrix_order : lapack_memory_order, jobvs : c_char, sort : c_char, chlapack_select : LAPACK_Z_SELECT1, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int, ref rconde : c_double, ref rcondv : c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgeesx_work(matrix_order : lapack_memory_order, jobvs : string, sort : string, chlapack_select : LAPACK_Z_SELECT1, sense : string, n : c_int, a : [] complex(128), lda : c_int, ref sdim : c_int, w : [] complex(128), vs : [] complex(128), ldvs : c_int, ref rconde : c_double, ref rcondv : c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zgeesx_work(matrix_order, jobvs.toByte() : c_char, sort.toByte() : c_char, chlapack_select, sense.toByte() : c_char, n, a, lda, sdim, w, vs, ldvs, rconde, rcondv, work, lwork, rwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_sgeev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgeev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sgeev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dgeev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgeev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dgeev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_cgeev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgeev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgeev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgeev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgeev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgeev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgeevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgeevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_float, lda : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgeevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgeevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgeevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_double, lda : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgeevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, wr, wi, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgeevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgeevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(64), lda : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_float, ref abnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgeevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgeevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgeevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(128), lda : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, scale : [] c_double, ref abnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgeevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, w, vl, ldvl, vr, ldvr, ilo, ihi, scale, abnrm, rconde, rcondv, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgehrd_work(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgehrd_work(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgehrd_work(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgehrd_work(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgejsv_work(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, jobr : c_char, jobt : c_char, jobp : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgejsv_work(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, jobr : string, jobt : string, jobp : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgejsv_work(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, jobr.toByte() : c_char, jobt.toByte() : c_char, jobp.toByte() : c_char, m, n, a, lda, sva, u, ldu, v, ldv, work, lwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgejsv_work(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, jobr : c_char, jobt : c_char, jobp : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgejsv_work(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, jobr : string, jobt : string, jobp : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgejsv_work(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, jobr.toByte() : c_char, jobt.toByte() : c_char, jobp.toByte() : c_char, m, n, a, lda, sva, u, ldu, v, ldv, work, lwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_sgelq2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelq2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelq2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelq2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgelqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgels_work(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgels_work(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sgels_work(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dgels_work(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgels_work(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dgels_work(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_cgels_work(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgels_work(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, work : [] complex(64), lwork : c_int) : c_int{
  require header;
  return LAPACKE_cgels_work(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_zgels_work(matrix_order : lapack_memory_order, trans : c_char, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgels_work(matrix_order : lapack_memory_order, trans : string, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, work : [] complex(128), lwork : c_int) : c_int{
  require header;
  return LAPACKE_zgels_work(matrix_order, trans.toByte() : c_char, m, n, nrhs, a, lda, b, ldb, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sgelsd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelsd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelsd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelsd_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgelss_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelss_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelss_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, s : [] c_float, rcond : c_float, ref rank : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelss_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, s : [] c_double, rcond : c_double, ref rank : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgelsy_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, jpvt : [] c_int, rcond : c_float, ref rank : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgelsy_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, jpvt : [] c_int, rcond : c_double, ref rank : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgelsy_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, jpvt : [] c_int, rcond : c_float, ref rank : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgelsy_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, jpvt : [] c_int, rcond : c_double, ref rank : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqlf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqlf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqlf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqlf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqp3_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, jpvt : [] c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqp3_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, jpvt : [] c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqp3_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, jpvt : [] c_int, tau : [] complex(64), work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqp3_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, jpvt : [] c_int, tau : [] complex(128), work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqpf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, jpvt : [] c_int, tau : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqpf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, jpvt : [] c_int, tau : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqpf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, jpvt : [] c_int, tau : [] complex(64), work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqpf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, jpvt : [] c_int, tau : [] complex(128), work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqr2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqr2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqr2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqr2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgeqrfp_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgeqrfp_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgeqrfp_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgeqrfp_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgerfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgerfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgerfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgerfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgerfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgerfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgerfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgerfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgerfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgerfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgerfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgerfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgerfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgerfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgerfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgerfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgerfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgerfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgerfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgerfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgerfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgerfsx_work(matrix_order : lapack_memory_order, trans : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgerfsx_work(matrix_order : lapack_memory_order, trans : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgerfsx_work(matrix_order, trans.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, r, c, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgerqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgerqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgerqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgerqf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgesdd_work(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesdd_work(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgesdd_work(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgesdd_work(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesdd_work(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgesdd_work(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgesdd_work(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesdd_work(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cgesdd_work(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_zgesdd_work(matrix_order : lapack_memory_order, jobz : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesdd_work(matrix_order : lapack_memory_order, jobz : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zgesdd_work(matrix_order, jobz.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_sgesv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgesv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgesv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgesv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dsgesv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, work : [] c_double, swork : [] c_float, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zcgesv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, work : [] complex(128), swork : [] complex(64), rwork : [] c_double, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgesvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvd_work(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, u : [] c_float, ldu : c_int, vt : [] c_float, ldvt : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sgesvd_work(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dgesvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvd_work(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, u : [] c_double, ldu : c_int, vt : [] c_double, ldvt : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dgesvd_work(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_cgesvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesvd_work(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, u : [] complex(64), ldu : c_int, vt : [] complex(64), ldvt : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgesvd_work(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgesvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobvt : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesvd_work(matrix_order : lapack_memory_order, jobu : string, jobvt : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, u : [] complex(128), ldu : c_int, vt : [] complex(128), ldvt : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgesvd_work(matrix_order, jobu.toByte() : c_char, jobvt.toByte() : c_char, m, n, a, lda, s, u, ldu, vt, ldvt, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgesvj_work(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, mv : c_int, v : [] c_float, ldv : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvj_work(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, sva : [] c_float, mv : c_int, v : [] c_float, ldv : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sgesvj_work(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, m, n, a, lda, sva, mv, v, ldv, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dgesvj_work(matrix_order : lapack_memory_order, joba : c_char, jobu : c_char, jobv : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, mv : c_int, v : [] c_double, ldv : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvj_work(matrix_order : lapack_memory_order, joba : string, jobu : string, jobv : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, sva : [] c_double, mv : c_int, v : [] c_double, ldv : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dgesvj_work(matrix_order, joba.toByte() : c_char, jobu.toByte() : c_char, jobv.toByte() : c_char, m, n, a, lda, sva, mv, v, ldv, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sgesvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgesvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgesvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgesvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgesvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgesvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgesvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgesvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgesvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgesvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgesvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgesvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgesvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgesvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgesvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgesvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_float, c : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgesvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgesvxx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgesvxx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : string, r : [] c_double, c : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgesvxx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, r, c, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgetf2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgetf2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgetf2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgetf2_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgetrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgetrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgetrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgetrf_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgetri_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgetri_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgetri_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgetri_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgetrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgetrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sgetrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dgetrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgetrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dgetrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cgetrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgetrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cgetrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zgetrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgetrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zgetrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sggbak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggbak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] c_float, ldv : c_int) : c_int{
  require header;
  return LAPACKE_sggbak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_dggbak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggbak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] c_double, ldv : c_int) : c_int{
  require header;
  return LAPACKE_dggbak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_cggbak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggbak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_float, rscale : [] c_float, m : c_int, v : [] complex(64), ldv : c_int) : c_int{
  require header;
  return LAPACKE_cggbak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_zggbak_work(matrix_order : lapack_memory_order, job : c_char, side : c_char, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggbak_work(matrix_order : lapack_memory_order, job : string, side : string, n : c_int, ilo : c_int, ihi : c_int, lscale : [] c_double, rscale : [] c_double, m : c_int, v : [] complex(128), ldv : c_int) : c_int{
  require header;
  return LAPACKE_zggbak_work(matrix_order, job.toByte() : c_char, side.toByte() : c_char, n, ilo, ihi, lscale, rscale, m, v, ldv);
}

pragma "no doc"
extern proc LAPACKE_sggbal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sggbal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sggbal_work(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
}

pragma "no doc"
extern proc LAPACKE_dggbal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dggbal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dggbal_work(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
}

pragma "no doc"
extern proc LAPACKE_cggbal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cggbal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, work : [] c_float) : c_int{
  require header;
  return LAPACKE_cggbal_work(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
}

pragma "no doc"
extern proc LAPACKE_zggbal_work(matrix_order : lapack_memory_order, job : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zggbal_work(matrix_order : lapack_memory_order, job : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, work : [] c_double) : c_int{
  require header;
  return LAPACKE_zggbal_work(matrix_order, job.toByte() : c_char, n, a, lda, b, ldb, ilo, ihi, lscale, rscale, work);
}

pragma "no doc"
extern proc LAPACKE_sgges_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_S_SELECT3, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int, work : [] c_float, lwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgges_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_S_SELECT3, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int, work : [] c_float, lwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgges_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_dgges_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_D_SELECT3, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int, work : [] c_double, lwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgges_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_D_SELECT3, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int, work : [] c_double, lwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgges_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_cgges_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_C_SELECT2, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgges_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_C_SELECT2, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cgges_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_zgges_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_Z_SELECT2, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgges_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_Z_SELECT2, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zgges_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, work, lwork, rwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_sggesx_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_S_SELECT3, sense : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggesx_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_S_SELECT3, sense : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, ref sdim : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vsl : [] c_float, ldvsl : c_int, vsr : [] c_float, ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sggesx_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, iwork, liwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_dggesx_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_D_SELECT3, sense : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggesx_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_D_SELECT3, sense : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, ref sdim : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vsl : [] c_double, ldvsl : c_int, vsr : [] c_double, ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dggesx_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alphar, alphai, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, iwork, liwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_cggesx_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_C_SELECT2, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggesx_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_C_SELECT2, sense : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, ref sdim : c_int, alpha : [] complex(64), beta : [] complex(64), vsl : [] complex(64), ldvsl : c_int, vsr : [] complex(64), ldvsr : c_int, rconde : [] c_float, rcondv : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cggesx_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, rwork, iwork, liwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_zggesx_work(matrix_order : lapack_memory_order, jobvsl : c_char, jobvsr : c_char, sort : c_char, selctg : LAPACK_Z_SELECT2, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggesx_work(matrix_order : lapack_memory_order, jobvsl : string, jobvsr : string, sort : string, selctg : LAPACK_Z_SELECT2, sense : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, ref sdim : c_int, alpha : [] complex(128), beta : [] complex(128), vsl : [] complex(128), ldvsl : c_int, vsr : [] complex(128), ldvsr : c_int, rconde : [] c_double, rcondv : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, liwork : c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zggesx_work(matrix_order, jobvsl.toByte() : c_char, jobvsr.toByte() : c_char, sort.toByte() : c_char, selctg, sense.toByte() : c_char, n, a, lda, b, ldb, sdim, alpha, beta, vsl, ldvsl, vsr, ldvsr, rconde, rcondv, work, lwork, rwork, iwork, liwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_sggev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sggev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dggev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dggev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_cggev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cggev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cggev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zggev_work(matrix_order : lapack_memory_order, jobvl : c_char, jobvr : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zggev_work(matrix_order : lapack_memory_order, jobvl : string, jobvr : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zggev_work(matrix_order, jobvl.toByte() : c_char, jobvr.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_sggevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sggevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, iwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_dggevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dggevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alphar, alphai, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, iwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_cggevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] complex(64), beta : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_float, rscale : [] c_float, ref abnrm : c_float, ref bbnrm : c_float, rconde : [] c_float, rcondv : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cggevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, rwork, iwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_zggevx_work(matrix_order : lapack_memory_order, balanc : c_char, jobvl : c_char, jobvr : c_char, sense : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, bwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggevx_work(matrix_order : lapack_memory_order, balanc : string, jobvl : string, jobvr : string, sense : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] complex(128), beta : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, ref ilo : c_int, ref ihi : c_int, lscale : [] c_double, rscale : [] c_double, ref abnrm : c_double, ref bbnrm : c_double, rconde : [] c_double, rcondv : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, bwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zggevx_work(matrix_order, balanc.toByte() : c_char, jobvl.toByte() : c_char, jobvr.toByte() : c_char, sense.toByte() : c_char, n, a, lda, b, ldb, alpha, beta, vl, ldvl, vr, ldvr, ilo, ihi, lscale, rscale, abnrm, bbnrm, rconde, rcondv, work, lwork, rwork, iwork, bwork);
}

pragma "no doc"
extern proc LAPACKE_sggglm_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, d : [] c_float, x : [] c_float, y : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dggglm_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, d : [] c_double, x : [] c_double, y : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cggglm_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, d : [] complex(64), x : [] complex(64), y : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zggglm_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, d : [] complex(128), x : [] complex(128), y : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgghrd_work(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgghrd_work(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int) : c_int{
  require header;
  return LAPACKE_sgghrd_work(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_dgghrd_work(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgghrd_work(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int) : c_int{
  require header;
  return LAPACKE_dgghrd_work(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_cgghrd_work(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgghrd_work(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int) : c_int{
  require header;
  return LAPACKE_cgghrd_work(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_zgghrd_work(matrix_order : lapack_memory_order, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgghrd_work(matrix_order : lapack_memory_order, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int) : c_int{
  require header;
  return LAPACKE_zgghrd_work(matrix_order, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, a, lda, b, ldb, q, ldq, z, ldz);
}

pragma "no doc"
extern proc LAPACKE_sgglse_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, c : [] c_float, d : [] c_float, x : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgglse_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, c : [] c_double, d : [] c_double, x : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgglse_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, c : [] complex(64), d : [] complex(64), x : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgglse_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, p : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, c : [] complex(128), d : [] complex(128), x : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sggqrf_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_float, lda : c_int, taua : [] c_float, b : [] c_float, ldb : c_int, taub : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dggqrf_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] c_double, lda : c_int, taua : [] c_double, b : [] c_double, ldb : c_int, taub : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cggqrf_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(64), lda : c_int, taua : [] complex(64), b : [] complex(64), ldb : c_int, taub : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zggqrf_work(matrix_order : lapack_memory_order, n : c_int, m : c_int, p : c_int, a : [] complex(128), lda : c_int, taua : [] complex(128), b : [] complex(128), ldb : c_int, taub : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sggrqf_work(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] c_float, lda : c_int, taua : [] c_float, b : [] c_float, ldb : c_int, taub : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dggrqf_work(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] c_double, lda : c_int, taua : [] c_double, b : [] c_double, ldb : c_int, taub : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cggrqf_work(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] complex(64), lda : c_int, taua : [] complex(64), b : [] complex(64), ldb : c_int, taub : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zggrqf_work(matrix_order : lapack_memory_order, m : c_int, p : c_int, n : c_int, a : [] complex(128), lda : c_int, taua : [] complex(128), b : [] complex(128), ldb : c_int, taub : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sggsvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sggsvd_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sggsvd_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dggsvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dggsvd_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dggsvd_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cggsvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cggsvd_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, alpha : [] c_float, beta : [] c_float, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_cggsvd_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, rwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_zggsvd_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zggsvd_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, n : c_int, p : c_int, ref k : c_int, ref l : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, alpha : [] c_double, beta : [] c_double, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_zggsvd_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, n, p, k, l, a, lda, b, ldb, alpha, beta, u, ldu, v, ldv, q, ldq, work, rwork, iwork);
}

pragma "no doc"
extern proc LAPACKE_sggsvp_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, iwork : [] c_int, tau : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sggsvp_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] c_float, ldu : c_int, v : [] c_float, ldv : c_int, q : [] c_float, ldq : c_int, iwork : [] c_int, tau : [] c_float, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sggsvp_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, tau, work);
}

pragma "no doc"
extern proc LAPACKE_dggsvp_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, iwork : [] c_int, tau : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dggsvp_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] c_double, ldu : c_int, v : [] c_double, ldv : c_int, q : [] c_double, ldq : c_int, iwork : [] c_int, tau : [] c_double, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dggsvp_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, tau, work);
}

pragma "no doc"
extern proc LAPACKE_cggsvp_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, iwork : [] c_int, rwork : [] c_float, tau : [] complex(64), work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cggsvp_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, tola : c_float, tolb : c_float, ref k : c_int, ref l : c_int, u : [] complex(64), ldu : c_int, v : [] complex(64), ldv : c_int, q : [] complex(64), ldq : c_int, iwork : [] c_int, rwork : [] c_float, tau : [] complex(64), work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cggsvp_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work);
}

pragma "no doc"
extern proc LAPACKE_zggsvp_work(matrix_order : lapack_memory_order, jobu : c_char, jobv : c_char, jobq : c_char, m : c_int, p : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, iwork : [] c_int, rwork : [] c_double, tau : [] complex(128), work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zggsvp_work(matrix_order : lapack_memory_order, jobu : string, jobv : string, jobq : string, m : c_int, p : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, tola : c_double, tolb : c_double, ref k : c_int, ref l : c_int, u : [] complex(128), ldu : c_int, v : [] complex(128), ldv : c_int, q : [] complex(128), ldq : c_int, iwork : [] c_int, rwork : [] c_double, tau : [] complex(128), work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zggsvp_work(matrix_order, jobu.toByte() : c_char, jobv.toByte() : c_char, jobq.toByte() : c_char, m, p, n, a, lda, b, ldb, tola, tolb, k, l, u, ldu, v, ldv, q, ldq, iwork, rwork, tau, work);
}

pragma "no doc"
extern proc LAPACKE_sgtcon_work(norm : c_char, n : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgtcon_work(norm : string, n : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgtcon_work(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgtcon_work(norm : c_char, n : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgtcon_work(norm : string, n : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgtcon_work(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgtcon_work(norm : c_char, n : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cgtcon_work(norm : string, n : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cgtcon_work(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_zgtcon_work(norm : c_char, n : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zgtcon_work(norm : string, n : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zgtcon_work(norm.toByte() : c_char, n, dl, d, du, du2, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_sgtrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgtrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgtrfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgtrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgtrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgtrfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgtrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgtrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgtrfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgtrfs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgtrfs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgtrfs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgtsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgtsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgtsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgtsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgtsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgtsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, dlf : [] c_float, df : [] c_float, duf : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sgtsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dgtsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgtsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, dlf : [] c_double, df : [] c_double, duf : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dgtsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cgtsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cgtsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), dlf : [] complex(64), df : [] complex(64), duf : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cgtsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zgtsvx_work(matrix_order : lapack_memory_order, fact : c_char, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zgtsvx_work(matrix_order : lapack_memory_order, fact : string, trans : string, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), dlf : [] complex(128), df : [] complex(128), duf : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zgtsvx_work(matrix_order, fact.toByte() : c_char, trans.toByte() : c_char, n, nrhs, dl, d, du, dlf, df, duf, du2, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sgttrf_work(n : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dgttrf_work(n : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cgttrf_work(n : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zgttrf_work(n : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sgttrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sgttrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_float, d : [] c_float, du : [] c_float, du2 : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sgttrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dgttrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dgttrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] c_double, d : [] c_double, du : [] c_double, du2 : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dgttrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cgttrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cgttrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(64), d : [] complex(64), du : [] complex(64), du2 : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cgttrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zgttrs_work(matrix_order : lapack_memory_order, trans : c_char, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zgttrs_work(matrix_order : lapack_memory_order, trans : string, n : c_int, nrhs : c_int, dl : [] complex(128), d : [] complex(128), du : [] complex(128), du2 : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zgttrs_work(matrix_order, trans.toByte() : c_char, n, nrhs, dl, d, du, du2, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chbev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chbev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chbev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhbev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhbev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chbevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_chbevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zhbevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zhbevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_chbevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chbevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhbevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhbevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_chbgst_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, x : [] complex(64), ldx : c_int, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgst_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, x : [] complex(64), ldx : c_int, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chbgst_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhbgst_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, x : [] complex(128), ldx : c_int, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgst_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, x : [] complex(128), ldx : c_int, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhbgst_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chbgv_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgv_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chbgv_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhbgv_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgv_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhbgv_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chbgvd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgvd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_chbgvd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zhbgvd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgvd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zhbgvd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_chbgvx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chbgvx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(64), ldab : c_int, bb : [] complex(64), ldbb : c_int, q : [] complex(64), ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chbgvx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhbgvx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbgvx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] complex(128), ldab : c_int, bb : [] complex(128), ldbb : c_int, q : [] complex(128), ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhbgvx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_chbtrd_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chbtrd_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, d : [] c_float, e : [] c_float, q : [] complex(64), ldq : c_int, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chbtrd_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq, work);
}

pragma "no doc"
extern proc LAPACKE_zhbtrd_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhbtrd_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, d : [] c_double, e : [] c_double, q : [] complex(128), ldq : c_int, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhbtrd_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq, work);
}

pragma "no doc"
extern proc LAPACKE_checon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_checon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_checon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_zhecon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhecon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhecon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_cheequb_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cheequb_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cheequb_work(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax, work);
}

pragma "no doc"
extern proc LAPACKE_zheequb_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zheequb_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zheequb_work(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax, work);
}

pragma "no doc"
extern proc LAPACKE_cheev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cheev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cheev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zheev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zheev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zheev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_cheevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cheevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_cheevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zheevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zheevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zheevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_cheevr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cheevr_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_cheevr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zheevr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zheevr_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zheevr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_cheevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cheevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_cheevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_zheevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zheevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zheevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_chegst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chegst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chegst_work(matrix_order, itype, uplo.toByte() : c_char, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhegst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhegst_work(matrix_order, itype, uplo.toByte() : c_char, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chegv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chegv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chegv_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhegv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhegv_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_chegvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chegvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, w : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_chegvd_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zhegvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, w : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zhegvd_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_chegvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chegvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chegvx_work(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhegvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhegvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhegvx_work(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, b, ldb, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_cherfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cherfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cherfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zherfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zherfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zherfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_cherfsx_work(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cherfsx_work(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cherfsx_work(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zherfsx_work(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zherfsx_work(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zherfsx_work(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chesv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chesv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, work : [] complex(64), lwork : c_int) : c_int{
  require header;
  return LAPACKE_chesv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_zhesv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhesv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, work : [] complex(128), lwork : c_int) : c_int{
  require header;
  return LAPACKE_zhesv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_chesvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chesvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chesvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhesvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhesvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhesvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_chesvxx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chesvxx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chesvxx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhesvxx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhesvxx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ipiv : [] c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhesvxx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, ipiv, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chetrd_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, d : [] c_float, e : [] c_float, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chetrd_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, d : [] c_float, e : [] c_float, tau : [] complex(64), work : [] complex(64), lwork : c_int) : c_int{
  require header;
  return LAPACKE_chetrd_work(matrix_order, uplo.toByte() : c_char, n, a, lda, d, e, tau, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_zhetrd_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, d : [] c_double, e : [] c_double, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetrd_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, d : [] c_double, e : [] c_double, tau : [] complex(128), work : [] complex(128), lwork : c_int) : c_int{
  require header;
  return LAPACKE_zhetrd_work(matrix_order, uplo.toByte() : c_char, n, a, lda, d, e, tau, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_chetrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chetrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, work : [] complex(64), lwork : c_int) : c_int{
  require header;
  return LAPACKE_chetrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_zhetrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, work : [] complex(128), lwork : c_int) : c_int{
  require header;
  return LAPACKE_zhetrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_chetri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chetri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chetri_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_zhetri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhetri_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_chetrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chetrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chetrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhetrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhetrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhetrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chfrk_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_float, a : [] complex(64), lda : c_int, beta : c_float, c : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chfrk_work(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_float, a : [] complex(64), lda : c_int, beta : c_float, c : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chfrk_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_zhfrk_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_double, a : [] complex(128), lda : c_int, beta : c_double, c : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhfrk_work(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_double, a : [] complex(128), lda : c_int, beta : c_double, c : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhfrk_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_shgeqz_work(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, t : [] c_float, ldt : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_shgeqz_work(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, t : [] c_float, ldt : c_int, alphar : [] c_float, alphai : [] c_float, beta : [] c_float, q : [] c_float, ldq : c_int, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_shgeqz_work(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alphar, alphai, beta, q, ldq, z, ldz, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dhgeqz_work(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, t : [] c_double, ldt : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dhgeqz_work(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, t : [] c_double, ldt : c_int, alphar : [] c_double, alphai : [] c_double, beta : [] c_double, q : [] c_double, ldq : c_int, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dhgeqz_work(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alphar, alphai, beta, q, ldq, z, ldz, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_chgeqz_work(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, t : [] complex(64), ldt : c_int, alpha : [] complex(64), beta : [] complex(64), q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chgeqz_work(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, t : [] complex(64), ldt : c_int, alpha : [] complex(64), beta : [] complex(64), q : [] complex(64), ldq : c_int, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chgeqz_work(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhgeqz_work(matrix_order : lapack_memory_order, job : c_char, compq : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, t : [] complex(128), ldt : c_int, alpha : [] complex(128), beta : [] complex(128), q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhgeqz_work(matrix_order : lapack_memory_order, job : string, compq : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, t : [] complex(128), ldt : c_int, alpha : [] complex(128), beta : [] complex(128), q : [] complex(128), ldq : c_int, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhgeqz_work(matrix_order, job.toByte() : c_char, compq.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, t, ldt, alpha, beta, q, ldq, z, ldz, work, lwork, rwork);
}

pragma "no doc"
extern proc LAPACKE_chpcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chpcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chpcon_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_zhpcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhpcon_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_chpev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chpev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chpev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhpev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhpev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chpevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_chpevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zhpevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zhpevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_chpevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chpevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhpevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhpevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_chpgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chpgst_work(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_zhpgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhpgst_work(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_chpgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chpgv_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhpgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhpgv_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chpgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64), w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_chpgvd_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zhpgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128), w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zhpgvd_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_chpgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(64), bp : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(64), bp : [] complex(64), vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), rwork : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_chpgvx_work(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_zhpgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] complex(128), bp : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] complex(128), bp : [] complex(128), vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), rwork : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_zhpgvx_work(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, rwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_chprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chpsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chpsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chpsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhpsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhpsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_chpsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_chpsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_chpsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zhpsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zhpsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zhpsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_chptrd_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), d : [] c_float, e : [] c_float, tau : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chptrd_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), d : [] c_float, e : [] c_float, tau : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chptrd_work(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_zhptrd_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), d : [] c_double, e : [] c_double, tau : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptrd_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), d : [] c_double, e : [] c_double, tau : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhptrd_work(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_chptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_chptrf_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zhptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zhptrf_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_chptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_chptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_chptri_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_zhptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zhptri_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_chptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_chptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zhptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zhptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_shsein_work(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int, work : [] c_float, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_shsein_work(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, vl : [] c_float, ldvl : c_int, vr : [] c_float, ldvr : c_int, mm : c_int, ref m : c_int, work : [] c_float, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_shsein_work(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, wr, wi, vl, ldvl, vr, ldvr, mm, m, work, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_dhsein_work(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int, work : [] c_double, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dhsein_work(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, vl : [] c_double, ldvl : c_int, vr : [] c_double, ldvr : c_int, mm : c_int, ref m : c_int, work : [] c_double, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_dhsein_work(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, wr, wi, vl, ldvl, vr, ldvr, mm, m, work, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_chsein_work(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int, work : [] complex(64), rwork : [] c_float, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chsein_work(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), vl : [] complex(64), ldvl : c_int, vr : [] complex(64), ldvr : c_int, mm : c_int, ref m : c_int, work : [] complex(64), rwork : [] c_float, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_chsein_work(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, work, rwork, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_zhsein_work(matrix_order : lapack_memory_order, job : c_char, eigsrc : c_char, initv : c_char, chlapack_select : [] c_int, n : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int, work : [] complex(128), rwork : [] c_double, ifaill : [] c_int, ifailr : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhsein_work(matrix_order : lapack_memory_order, job : string, eigsrc : string, initv : string, chlapack_select : [] c_int, n : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), vl : [] complex(128), ldvl : c_int, vr : [] complex(128), ldvr : c_int, mm : c_int, ref m : c_int, work : [] complex(128), rwork : [] c_double, ifaill : [] c_int, ifailr : [] c_int) : c_int{
  require header;
  return LAPACKE_zhsein_work(matrix_order, job.toByte() : c_char, eigsrc.toByte() : c_char, initv.toByte() : c_char, chlapack_select, n, h, ldh, w, vl, ldvl, vr, ldvr, mm, m, work, rwork, ifaill, ifailr);
}

pragma "no doc"
extern proc LAPACKE_shseqr_work(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_shseqr_work(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_float, ldh : c_int, wr : [] c_float, wi : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_shseqr_work(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, wr, wi, z, ldz, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dhseqr_work(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dhseqr_work(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] c_double, ldh : c_int, wr : [] c_double, wi : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dhseqr_work(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, wr, wi, z, ldz, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_chseqr_work(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_chseqr_work(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(64), ldh : c_int, w : [] complex(64), z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int) : c_int{
  require header;
  return LAPACKE_chseqr_work(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, w, z, ldz, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_zhseqr_work(matrix_order : lapack_memory_order, job : c_char, compz : c_char, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zhseqr_work(matrix_order : lapack_memory_order, job : string, compz : string, n : c_int, ilo : c_int, ihi : c_int, h : [] complex(128), ldh : c_int, w : [] complex(128), z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int) : c_int{
  require header;
  return LAPACKE_zhseqr_work(matrix_order, job.toByte() : c_char, compz.toByte() : c_char, n, ilo, ihi, h, ldh, w, z, ldz, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_clacgv_work(n : c_int, x : [] complex(64), incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlacgv_work(n : c_int, x : [] complex(128), incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slacn2_work(n : c_int, v : [] c_float, x : [] c_float, isgn : [] c_int, ref est : c_float, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlacn2_work(n : c_int, v : [] c_double, x : [] c_double, isgn : [] c_int, ref est : c_double, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_clacn2_work(n : c_int, v : [] complex(64), x : [] complex(64), ref est : c_float, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlacn2_work(n : c_int, v : [] complex(128), x : [] complex(128), ref est : c_double, ref kase : c_int, isave : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slacpy_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slacpy_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_slacpy_work(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dlacpy_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlacpy_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dlacpy_work(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_clacpy_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clacpy_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_clacpy_work(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zlacpy_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlacpy_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zlacpy_work(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_clacp2_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clacp2_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_clacp2_work(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zlacp2_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlacp2_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zlacp2_work(matrix_order, uplo.toByte() : c_char, m, n, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zlag2c_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] complex(128), lda : c_int, sa : [] complex(64), ldsa : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slag2d_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, sa : [] c_float, ldsa : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlag2s_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, a : [] c_double, lda : c_int, sa : [] c_float, ldsa : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_clag2z_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, sa : [] complex(64), ldsa : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slagge_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_float, a : [] c_float, lda : c_int, iseed : [] c_int, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlagge_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_double, a : [] c_double, lda : c_int, iseed : [] c_int, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_clagge_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_float, a : [] complex(64), lda : c_int, iseed : [] c_int, work : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zlagge_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, kl : c_int, ku : c_int, d : [] c_double, a : [] complex(128), lda : c_int, iseed : [] c_int, work : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_claghe_work(matrix_order : lapack_memory_order, n : c_int, k : c_int, d : [] c_float, a : [] complex(64), lda : c_int, iseed : [] c_int, work : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zlaghe_work(matrix_order : lapack_memory_order, n : c_int, k : c_int, d : [] c_double, a : [] complex(128), lda : c_int, iseed : [] c_int, work : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_slagsy_work(matrix_order : lapack_memory_order, n : c_int, k : c_int, d : [] c_float, a : [] c_float, lda : c_int, iseed : [] c_int, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlagsy_work(matrix_order : lapack_memory_order, n : c_int, k : c_int, d : [] c_double, a : [] c_double, lda : c_int, iseed : [] c_int, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_clagsy_work(matrix_order : lapack_memory_order, n : c_int, k : c_int, d : [] c_float, a : [] complex(64), lda : c_int, iseed : [] c_int, work : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zlagsy_work(matrix_order : lapack_memory_order, n : c_int, k : c_int, d : [] c_double, a : [] complex(128), lda : c_int, iseed : [] c_int, work : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_slapmr_work(matrix_order : lapack_memory_order, forwrd : c_int, m : c_int, n : c_int, x : [] c_float, ldx : c_int, k : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlapmr_work(matrix_order : lapack_memory_order, forwrd : c_int, m : c_int, n : c_int, x : [] c_double, ldx : c_int, k : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_clapmr_work(matrix_order : lapack_memory_order, forwrd : c_int, m : c_int, n : c_int, x : [] complex(64), ldx : c_int, k : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlapmr_work(matrix_order : lapack_memory_order, forwrd : c_int, m : c_int, n : c_int, x : [] complex(128), ldx : c_int, k : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slartgp_work(f : c_float, g : c_float, ref cs : c_float, ref sn : c_float, ref r : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlartgp_work(f : c_double, g : c_double, ref cs : c_double, ref sn : c_double, ref r : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_slartgs_work(x : c_float, y : c_float, sigma : c_float, ref cs : c_float, ref sn : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlartgs_work(x : c_double, y : c_double, sigma : c_double, ref cs : c_double, ref sn : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_slapy2_work(x : c_float, y : c_float) : c_float;

pragma "no doc"
extern proc LAPACKE_dlapy2_work(x : c_double, y : c_double) : c_double;

pragma "no doc"
extern proc LAPACKE_slapy3_work(x : c_float, y : c_float, z : c_float) : c_float;

pragma "no doc"
extern proc LAPACKE_dlapy3_work(x : c_double, y : c_double, z : c_double) : c_double;

pragma "no doc"
extern proc LAPACKE_slamch_work(cmach : c_char) : c_float;

pragma "no doc"
inline proc LAPACKE_slamch_work(cmach : string) : c_float{
  require header;
  return LAPACKE_slamch_work(cmach.toByte() : c_char);
}

pragma "no doc"
extern proc LAPACKE_dlamch_work(cmach : c_char) : c_double;

pragma "no doc"
inline proc LAPACKE_dlamch_work(cmach : string) : c_double{
  require header;
  return LAPACKE_dlamch_work(cmach.toByte() : c_char);
}

pragma "no doc"
extern proc LAPACKE_slange_work(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, work : [] c_float) : c_float;

pragma "no doc"
inline proc LAPACKE_slange_work(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, work : [] c_float) : c_float{
  require header;
  return LAPACKE_slange_work(matrix_order, norm.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_dlange_work(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, work : [] c_double) : c_double;

pragma "no doc"
inline proc LAPACKE_dlange_work(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, work : [] c_double) : c_double{
  require header;
  return LAPACKE_dlange_work(matrix_order, norm.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_clange_work(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float;

pragma "no doc"
inline proc LAPACKE_clange_work(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float{
  require header;
  return LAPACKE_clange_work(matrix_order, norm.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_zlange_work(matrix_order : lapack_memory_order, norm : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double;

pragma "no doc"
inline proc LAPACKE_zlange_work(matrix_order : lapack_memory_order, norm : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double{
  require header;
  return LAPACKE_zlange_work(matrix_order, norm.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_clanhe_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float;

pragma "no doc"
inline proc LAPACKE_clanhe_work(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float{
  require header;
  return LAPACKE_clanhe_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_zlanhe_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double;

pragma "no doc"
inline proc LAPACKE_zlanhe_work(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double{
  require header;
  return LAPACKE_zlanhe_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_slansy_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, work : [] c_float) : c_float;

pragma "no doc"
inline proc LAPACKE_slansy_work(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, work : [] c_float) : c_float{
  require header;
  return LAPACKE_slansy_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_dlansy_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, work : [] c_double) : c_double;

pragma "no doc"
inline proc LAPACKE_dlansy_work(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, work : [] c_double) : c_double{
  require header;
  return LAPACKE_dlansy_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_clansy_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float;

pragma "no doc"
inline proc LAPACKE_clansy_work(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float{
  require header;
  return LAPACKE_clansy_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_zlansy_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double;

pragma "no doc"
inline proc LAPACKE_zlansy_work(matrix_order : lapack_memory_order, norm : string, uplo : string, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double{
  require header;
  return LAPACKE_zlansy_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_slantr_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, work : [] c_float) : c_float;

pragma "no doc"
inline proc LAPACKE_slantr_work(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, work : [] c_float) : c_float{
  require header;
  return LAPACKE_slantr_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_dlantr_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, work : [] c_double) : c_double;

pragma "no doc"
inline proc LAPACKE_dlantr_work(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, work : [] c_double) : c_double{
  require header;
  return LAPACKE_dlantr_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_clantr_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float;

pragma "no doc"
inline proc LAPACKE_clantr_work(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] complex(64), lda : c_int, work : [] c_float) : c_float{
  require header;
  return LAPACKE_clantr_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_zlantr_work(matrix_order : lapack_memory_order, norm : c_char, uplo : c_char, diag : c_char, m : c_int, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double;

pragma "no doc"
inline proc LAPACKE_zlantr_work(matrix_order : lapack_memory_order, norm : string, uplo : string, diag : string, m : c_int, n : c_int, a : [] complex(128), lda : c_int, work : [] c_double) : c_double{
  require header;
  return LAPACKE_zlantr_work(matrix_order, norm.toByte() : c_char, uplo.toByte() : c_char, diag.toByte() : c_char, m, n, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_slarfb_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] c_float, ldv : c_int, t : [] c_float, ldt : c_int, c : [] c_float, ldc : c_int, work : [] c_float, ldwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slarfb_work(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] c_float, ldv : c_int, t : [] c_float, ldt : c_int, c : [] c_float, ldc : c_int, work : [] c_float, ldwork : c_int) : c_int{
  require header;
  return LAPACKE_slarfb_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
}

pragma "no doc"
extern proc LAPACKE_dlarfb_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] c_double, ldv : c_int, t : [] c_double, ldt : c_int, c : [] c_double, ldc : c_int, work : [] c_double, ldwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlarfb_work(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] c_double, ldv : c_int, t : [] c_double, ldt : c_int, c : [] c_double, ldc : c_int, work : [] c_double, ldwork : c_int) : c_int{
  require header;
  return LAPACKE_dlarfb_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
}

pragma "no doc"
extern proc LAPACKE_clarfb_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, t : [] complex(64), ldt : c_int, c : [] complex(64), ldc : c_int, work : [] complex(64), ldwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clarfb_work(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, t : [] complex(64), ldt : c_int, c : [] complex(64), ldc : c_int, work : [] complex(64), ldwork : c_int) : c_int{
  require header;
  return LAPACKE_clarfb_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
}

pragma "no doc"
extern proc LAPACKE_zlarfb_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, direct : c_char, storev : c_char, m : c_int, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, t : [] complex(128), ldt : c_int, c : [] complex(128), ldc : c_int, work : [] complex(128), ldwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlarfb_work(matrix_order : lapack_memory_order, side : string, trans : string, direct : string, storev : string, m : c_int, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, t : [] complex(128), ldt : c_int, c : [] complex(128), ldc : c_int, work : [] complex(128), ldwork : c_int) : c_int{
  require header;
  return LAPACKE_zlarfb_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, direct.toByte() : c_char, storev.toByte() : c_char, m, n, k, v, ldv, t, ldt, c, ldc, work, ldwork);
}

pragma "no doc"
extern proc LAPACKE_slarfg_work(n : c_int, ref alpha : c_float, x : [] c_float, incx : c_int, ref tau : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlarfg_work(n : c_int, ref alpha : c_double, x : [] c_double, incx : c_int, ref tau : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_clarfg_work(n : c_int, ref alpha : complex(64), x : [] complex(64), incx : c_int, ref tau : complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zlarfg_work(n : c_int, ref alpha : complex(128), x : [] complex(128), incx : c_int, ref tau : complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_slarft_work(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] c_float, ldv : c_int, tau : [] c_float, t : [] c_float, ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slarft_work(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] c_float, ldv : c_int, tau : [] c_float, t : [] c_float, ldt : c_int) : c_int{
  require header;
  return LAPACKE_slarft_work(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_dlarft_work(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] c_double, ldv : c_int, tau : [] c_double, t : [] c_double, ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlarft_work(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] c_double, ldv : c_int, tau : [] c_double, t : [] c_double, ldt : c_int) : c_int{
  require header;
  return LAPACKE_dlarft_work(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_clarft_work(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, tau : [] complex(64), t : [] complex(64), ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clarft_work(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] complex(64), ldv : c_int, tau : [] complex(64), t : [] complex(64), ldt : c_int) : c_int{
  require header;
  return LAPACKE_clarft_work(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_zlarft_work(matrix_order : lapack_memory_order, direct : c_char, storev : c_char, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, tau : [] complex(128), t : [] complex(128), ldt : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlarft_work(matrix_order : lapack_memory_order, direct : string, storev : string, n : c_int, k : c_int, v : [] complex(128), ldv : c_int, tau : [] complex(128), t : [] complex(128), ldt : c_int) : c_int{
  require header;
  return LAPACKE_zlarft_work(matrix_order, direct.toByte() : c_char, storev.toByte() : c_char, n, k, v, ldv, tau, t, ldt);
}

pragma "no doc"
extern proc LAPACKE_slarfx_work(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] c_float, tau : c_float, c : [] c_float, ldc : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_slarfx_work(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] c_float, tau : c_float, c : [] c_float, ldc : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_slarfx_work(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_dlarfx_work(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] c_double, tau : c_double, c : [] c_double, ldc : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dlarfx_work(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] c_double, tau : c_double, c : [] c_double, ldc : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dlarfx_work(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_clarfx_work(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] complex(64), tau : complex(64), c : [] complex(64), ldc : c_int, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_clarfx_work(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] complex(64), tau : complex(64), c : [] complex(64), ldc : c_int, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_clarfx_work(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_zlarfx_work(matrix_order : lapack_memory_order, side : c_char, m : c_int, n : c_int, v : [] complex(128), tau : complex(128), c : [] complex(128), ldc : c_int, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zlarfx_work(matrix_order : lapack_memory_order, side : string, m : c_int, n : c_int, v : [] complex(128), tau : complex(128), c : [] complex(128), ldc : c_int, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zlarfx_work(matrix_order, side.toByte() : c_char, m, n, v, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_slarnv_work(idist : c_int, iseed : [] c_int, n : c_int, x : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dlarnv_work(idist : c_int, iseed : [] c_int, n : c_int, x : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_clarnv_work(idist : c_int, iseed : [] c_int, n : c_int, x : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zlarnv_work(idist : c_int, iseed : [] c_int, n : c_int, x : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_slaset_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : c_float, beta : c_float, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slaset_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : c_float, beta : c_float, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_slaset_work(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlaset_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : c_double, beta : c_double, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlaset_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : c_double, beta : c_double, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dlaset_work(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_claset_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : complex(64), beta : complex(64), a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_claset_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : complex(64), beta : complex(64), a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_claset_work(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlaset_work(matrix_order : lapack_memory_order, uplo : c_char, m : c_int, n : c_int, alpha : complex(128), beta : complex(128), a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlaset_work(matrix_order : lapack_memory_order, uplo : string, m : c_int, n : c_int, alpha : complex(128), beta : complex(128), a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zlaset_work(matrix_order, uplo.toByte() : c_char, m, n, alpha, beta, a, lda);
}

pragma "no doc"
extern proc LAPACKE_slasrt_work(id : c_char, n : c_int, d : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_slasrt_work(id : string, n : c_int, d : [] c_float) : c_int{
  require header;
  return LAPACKE_slasrt_work(id.toByte() : c_char, n, d);
}

pragma "no doc"
extern proc LAPACKE_dlasrt_work(id : c_char, n : c_int, d : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dlasrt_work(id : string, n : c_int, d : [] c_double) : c_int{
  require header;
  return LAPACKE_dlasrt_work(id.toByte() : c_char, n, d);
}

pragma "no doc"
extern proc LAPACKE_slaswp_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dlaswp_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_claswp_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zlaswp_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, k1 : c_int, k2 : c_int, ipiv : [] c_int, incx : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_slatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : c_char, a : [] c_float, lda : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_slatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : string, a : [] c_float, lda : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_slatms_work(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_dlatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : c_char, a : [] c_double, lda : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dlatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : string, a : [] c_double, lda : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dlatms_work(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_clatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : c_char, a : [] complex(64), lda : c_int, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_clatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_float, mode : c_int, cond : c_float, dmax : c_float, kl : c_int, ku : c_int, pack : string, a : [] complex(64), lda : c_int, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_clatms_work(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_zlatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : c_char, iseed : [] c_int, sym : c_char, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : c_char, a : [] complex(128), lda : c_int, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zlatms_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, dist : string, iseed : [] c_int, sym : string, d : [] c_double, mode : c_int, cond : c_double, dmax : c_double, kl : c_int, ku : c_int, pack : string, a : [] complex(128), lda : c_int, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zlatms_work(matrix_order, m, n, dist.toByte() : c_char, iseed, sym.toByte() : c_char, d, mode, cond, dmax, kl, ku, pack.toByte() : c_char, a, lda, work);
}

pragma "no doc"
extern proc LAPACKE_slauum_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_slauum_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_slauum_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dlauum_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dlauum_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dlauum_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_clauum_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_clauum_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_clauum_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zlauum_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zlauum_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zlauum_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_sopgtr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, tau : [] c_float, q : [] c_float, ldq : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sopgtr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, tau : [] c_float, q : [] c_float, ldq : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sopgtr_work(matrix_order, uplo.toByte() : c_char, n, ap, tau, q, ldq, work);
}

pragma "no doc"
extern proc LAPACKE_dopgtr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, tau : [] c_double, q : [] c_double, ldq : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dopgtr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, tau : [] c_double, q : [] c_double, ldq : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dopgtr_work(matrix_order, uplo.toByte() : c_char, n, ap, tau, q, ldq, work);
}

pragma "no doc"
extern proc LAPACKE_sopmtr_work(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, ap : [] c_float, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sopmtr_work(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, ap : [] c_float, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sopmtr_work(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, ap, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_dopmtr_work(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, ap : [] c_double, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dopmtr_work(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, ap : [] c_double, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dopmtr_work(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, ap, tau, c, ldc, work);
}

pragma "no doc"
extern proc LAPACKE_sorgbr_work(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sorgbr_work(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sorgbr_work(matrix_order, vect.toByte() : c_char, m, n, k, a, lda, tau, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dorgbr_work(matrix_order : lapack_memory_order, vect : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dorgbr_work(matrix_order : lapack_memory_order, vect : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dorgbr_work(matrix_order, vect.toByte() : c_char, m, n, k, a, lda, tau, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sorghr_work(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dorghr_work(matrix_order : lapack_memory_order, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sorglq_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dorglq_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgql_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dorgql_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgqr_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dorgqr_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgrq_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dorgrq_work(matrix_order : lapack_memory_order, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sorgtr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sorgtr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sorgtr_work(matrix_order, uplo.toByte() : c_char, n, a, lda, tau, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dorgtr_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dorgtr_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dorgtr_work(matrix_order, uplo.toByte() : c_char, n, a, lda, tau, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormbr_work(matrix_order : lapack_memory_order, vect : c_char, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormbr_work(matrix_order : lapack_memory_order, vect : string, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormbr_work(matrix_order, vect.toByte() : c_char, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormbr_work(matrix_order : lapack_memory_order, vect : c_char, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormbr_work(matrix_order : lapack_memory_order, vect : string, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormbr_work(matrix_order, vect.toByte() : c_char, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormhr_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormhr_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormhr_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, ilo, ihi, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormhr_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormhr_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, ilo : c_int, ihi : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormhr_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, ilo, ihi, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormlq_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormlq_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormlq_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormlq_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormlq_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormlq_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormql_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormql_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormql_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormql_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormql_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormql_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormqr_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormqr_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormqr_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormqr_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormqr_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormqr_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormrq_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormrq_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormrq_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormrq_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormrq_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormrq_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormrz_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormrz_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormrz_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, l, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormrz_work(matrix_order : lapack_memory_order, side : c_char, trans : c_char, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormrz_work(matrix_order : lapack_memory_order, side : string, trans : string, m : c_int, n : c_int, k : c_int, l : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormrz_work(matrix_order, side.toByte() : c_char, trans.toByte() : c_char, m, n, k, l, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_sormtr_work(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sormtr_work(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, a : [] c_float, lda : c_int, tau : [] c_float, c : [] c_float, ldc : c_int, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_sormtr_work(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dormtr_work(matrix_order : lapack_memory_order, side : c_char, uplo : c_char, trans : c_char, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dormtr_work(matrix_order : lapack_memory_order, side : string, uplo : string, trans : string, m : c_int, n : c_int, a : [] c_double, lda : c_int, tau : [] c_double, c : [] c_double, ldc : c_int, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dormtr_work(matrix_order, side.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, m, n, a, lda, tau, c, ldc, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_spbcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_spbcon_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dpbcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dpbcon_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cpbcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cpbcon_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zpbcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zpbcon_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_spbequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spbequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_spbequ_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_dpbequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_dpbequ_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_cpbequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_cpbequ_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_zpbequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_zpbequ_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_spbrfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbrfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_spbrfs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dpbrfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbrfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dpbrfs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cpbrfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbrfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cpbrfs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zpbrfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbrfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zpbrfs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_spbstf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] c_float, ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbstf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] c_float, ldbb : c_int) : c_int{
  require header;
  return LAPACKE_spbstf_work(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_dpbstf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] c_double, ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbstf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] c_double, ldbb : c_int) : c_int{
  require header;
  return LAPACKE_dpbstf_work(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_cpbstf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] complex(64), ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbstf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] complex(64), ldbb : c_int) : c_int{
  require header;
  return LAPACKE_cpbstf_work(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_zpbstf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kb : c_int, bb : [] complex(128), ldbb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbstf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kb : c_int, bb : [] complex(128), ldbb : c_int) : c_int{
  require header;
  return LAPACKE_zpbstf_work(matrix_order, uplo.toByte() : c_char, n, kb, bb, ldbb);
}

pragma "no doc"
extern proc LAPACKE_spbsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spbsv_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpbsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpbsv_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpbsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpbsv_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpbsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpbsv_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spbsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, afb : [] c_float, ldafb : c_int, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_spbsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dpbsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, afb : [] c_double, ldafb : c_int, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dpbsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cpbsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, afb : [] complex(64), ldafb : c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cpbsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zpbsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, afb : [] complex(128), ldafb : c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zpbsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, afb, ldafb, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_spbtrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbtrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int) : c_int{
  require header;
  return LAPACKE_spbtrf_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_dpbtrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbtrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int) : c_int{
  require header;
  return LAPACKE_dpbtrf_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_cpbtrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbtrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(64), ldab : c_int) : c_int{
  require header;
  return LAPACKE_cpbtrf_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_zpbtrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbtrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, ab : [] complex(128), ldab : c_int) : c_int{
  require header;
  return LAPACKE_zpbtrf_work(matrix_order, uplo.toByte() : c_char, n, kd, ab, ldab);
}

pragma "no doc"
extern proc LAPACKE_spbtrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spbtrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_float, ldab : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spbtrs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpbtrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpbtrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] c_double, ldab : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpbtrs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpbtrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpbtrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(64), ldab : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpbtrs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpbtrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpbtrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, kd : c_int, nrhs : c_int, ab : [] complex(128), ldab : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpbtrs_work(matrix_order, uplo.toByte() : c_char, n, kd, nrhs, ab, ldab, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spftrf_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spftrf_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_float) : c_int{
  require header;
  return LAPACKE_spftrf_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_dpftrf_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpftrf_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_double) : c_int{
  require header;
  return LAPACKE_dpftrf_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_cpftrf_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpftrf_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpftrf_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_zpftrf_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpftrf_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpftrf_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_spftri_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spftri_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_float) : c_int{
  require header;
  return LAPACKE_spftri_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_dpftri_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpftri_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] c_double) : c_int{
  require header;
  return LAPACKE_dpftri_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_cpftri_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpftri_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpftri_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_zpftri_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, a : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpftri_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, a : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpftri_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, a);
}

pragma "no doc"
extern proc LAPACKE_spftrs_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spftrs_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spftrs_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpftrs_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpftrs_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpftrs_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpftrs_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpftrs_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpftrs_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpftrs_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpftrs_work(matrix_order : lapack_memory_order, transr : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpftrs_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spocon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spocon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_spocon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dpocon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpocon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dpocon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cpocon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpocon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cpocon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zpocon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpocon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zpocon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_spoequ_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dpoequ_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cpoequ_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zpoequ_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_spoequb_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dpoequb_work(matrix_order : lapack_memory_order, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cpoequb_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zpoequb_work(matrix_order : lapack_memory_order, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sporfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sporfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sporfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dporfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dporfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dporfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cporfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cporfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cporfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zporfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zporfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zporfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sporfsx_work(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sporfsx_work(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sporfsx_work(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dporfsx_work(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dporfsx_work(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dporfsx_work(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cporfsx_work(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cporfsx_work(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cporfsx_work(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zporfsx_work(matrix_order : lapack_memory_order, uplo : c_char, equed : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zporfsx_work(matrix_order : lapack_memory_order, uplo : string, equed : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zporfsx_work(matrix_order, uplo.toByte() : c_char, equed.toByte() : c_char, n, nrhs, a, lda, af, ldaf, s, b, ldb, x, ldx, rcond, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sposv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sposv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sposv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dposv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dposv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dposv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cposv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cposv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cposv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zposv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zposv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zposv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dsposv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, work : [] c_double, swork : [] c_float, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsposv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, work : [] c_double, swork : [] c_float, ref chlapack_iter : c_int) : c_int{
  require header;
  return LAPACKE_dsposv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, work, swork, chlapack_iter);
}

pragma "no doc"
extern proc LAPACKE_zcposv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, work : [] complex(128), swork : [] complex(64), rwork : [] c_double, ref chlapack_iter : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zcposv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, work : [] complex(128), swork : [] complex(64), rwork : [] c_double, ref chlapack_iter : c_int) : c_int{
  require header;
  return LAPACKE_zcposv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb, x, ldx, work, swork, rwork, chlapack_iter);
}

pragma "no doc"
extern proc LAPACKE_sposvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sposvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sposvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dposvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dposvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dposvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cposvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cposvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cposvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zposvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zposvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zposvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sposvxx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sposvxx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, af : [] c_float, ldaf : c_int, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sposvxx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dposvxx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dposvxx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, af : [] c_double, ldaf : c_int, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dposvxx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cposvxx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cposvxx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, af : [] complex(64), ldaf : c_int, ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ref rpvgrw : c_float, berr : [] c_float, n_err_bnds : c_int, err_bnds_norm : [] c_float, err_bnds_comp : [] c_float, nparams : c_int, params : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cposvxx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zposvxx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zposvxx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, af : [] complex(128), ldaf : c_int, ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ref rpvgrw : c_double, berr : [] c_double, n_err_bnds : c_int, err_bnds_norm : [] c_double, err_bnds_comp : [] c_double, nparams : c_int, params : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zposvxx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, a, lda, af, ldaf, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, rpvgrw, berr, n_err_bnds, err_bnds_norm, err_bnds_comp, nparams, params, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_spotrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spotrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_spotrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dpotrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpotrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dpotrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_cpotrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpotrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_cpotrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zpotrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpotrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zpotrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_spotri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spotri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int) : c_int{
  require header;
  return LAPACKE_spotri_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_dpotri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpotri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int) : c_int{
  require header;
  return LAPACKE_dpotri_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_cpotri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpotri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int) : c_int{
  require header;
  return LAPACKE_cpotri_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_zpotri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpotri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int) : c_int{
  require header;
  return LAPACKE_zpotri_work(matrix_order, uplo.toByte() : c_char, n, a, lda);
}

pragma "no doc"
extern proc LAPACKE_spotrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spotrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spotrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpotrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpotrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] c_double, lda : c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpotrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpotrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpotrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(64), lda : c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpotrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpotrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpotrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, a : [] complex(128), lda : c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpotrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, a, lda, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sppcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sppcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sppcon_work(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dppcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dppcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dppcon_work(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cppcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cppcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), anorm : c_float, ref rcond : c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cppcon_work(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zppcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zppcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), anorm : c_double, ref rcond : c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zppcon_work(matrix_order, uplo.toByte() : c_char, n, ap, anorm, rcond, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sppequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sppequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_sppequ_work(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_dppequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dppequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_dppequ_work(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_cppequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cppequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), s : [] c_float, ref scond : c_float, ref amax : c_float) : c_int{
  require header;
  return LAPACKE_cppequ_work(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_zppequ_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zppequ_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), s : [] c_double, ref scond : c_double, ref amax : c_double) : c_int{
  require header;
  return LAPACKE_zppequ_work(matrix_order, uplo.toByte() : c_char, n, ap, s, scond, amax);
}

pragma "no doc"
extern proc LAPACKE_spprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_spprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dpprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dpprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cpprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cpprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zpprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zpprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sppsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sppsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sppsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dppsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dppsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dppsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cppsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cppsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cppsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zppsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zppsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zppsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sppsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ref equed : c_char, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sppsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ref equed : string, s : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sppsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dppsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ref equed : c_char, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dppsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ref equed : string, s : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dppsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cppsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ref equed : c_char, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cppsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ref equed : string, s : [] c_float, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cppsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zppsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ref equed : c_char, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zppsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ref equed : string, s : [] c_double, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zppsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, equed.toByte() : c_char, s, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_spptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float) : c_int{
  require header;
  return LAPACKE_spptrf_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_dpptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double) : c_int{
  require header;
  return LAPACKE_dpptrf_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_cpptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpptrf_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_zpptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpptrf_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_spptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float) : c_int{
  require header;
  return LAPACKE_spptri_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_dpptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double) : c_int{
  require header;
  return LAPACKE_dpptri_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_cpptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cpptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cpptri_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_zpptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zpptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zpptri_work(matrix_order, uplo.toByte() : c_char, n, ap);
}

pragma "no doc"
extern proc LAPACKE_spptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_spptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_spptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dpptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dpptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dpptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cpptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_spstrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spstrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float, work : [] c_float) : c_int{
  require header;
  return LAPACKE_spstrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol, work);
}

pragma "no doc"
extern proc LAPACKE_dpstrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpstrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dpstrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol, work);
}

pragma "no doc"
extern proc LAPACKE_cpstrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpstrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_float, work : [] c_float) : c_int{
  require header;
  return LAPACKE_cpstrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol, work);
}

pragma "no doc"
extern proc LAPACKE_zpstrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpstrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, piv : [] c_int, ref rank : c_int, tol : c_double, work : [] c_double) : c_int{
  require header;
  return LAPACKE_zpstrf_work(matrix_order, uplo.toByte() : c_char, n, a, lda, piv, rank, tol, work);
}

pragma "no doc"
extern proc LAPACKE_sptcon_work(n : c_int, d : [] c_float, e : [] c_float, anorm : c_float, ref rcond : c_float, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dptcon_work(n : c_int, d : [] c_double, e : [] c_double, anorm : c_double, ref rcond : c_double, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cptcon_work(n : c_int, d : [] c_float, e : [] complex(64), anorm : c_float, ref rcond : c_float, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_zptcon_work(n : c_int, d : [] c_double, e : [] complex(128), anorm : c_double, ref rcond : c_double, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_spteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_spteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_spteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_dpteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dpteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dpteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_cpteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cpteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_cpteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_zpteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zpteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_zpteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_sptrfs_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, df : [] c_float, ef : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dptrfs_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, df : [] c_double, ef : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cptrfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cptrfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cptrfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zptrfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zptrfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zptrfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sptsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dptsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cptsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zptsv_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sptsvx_work(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, df : [] c_float, ef : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sptsvx_work(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, df : [] c_float, ef : [] c_float, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sptsvx_work(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr, work);
}

pragma "no doc"
extern proc LAPACKE_dptsvx_work(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, df : [] c_double, ef : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dptsvx_work(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, df : [] c_double, ef : [] c_double, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dptsvx_work(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr, work);
}

pragma "no doc"
extern proc LAPACKE_cptsvx_work(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cptsvx_work(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), df : [] c_float, ef : [] complex(64), b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cptsvx_work(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zptsvx_work(matrix_order : lapack_memory_order, fact : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zptsvx_work(matrix_order : lapack_memory_order, fact : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), df : [] c_double, ef : [] complex(128), b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zptsvx_work(matrix_order, fact.toByte() : c_char, n, nrhs, d, e, df, ef, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_spttrf_work(n : c_int, d : [] c_float, e : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dpttrf_work(n : c_int, d : [] c_double, e : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_cpttrf_work(n : c_int, d : [] c_float, e : [] complex(64)) : c_int;

pragma "no doc"
extern proc LAPACKE_zpttrf_work(n : c_int, d : [] c_double, e : [] complex(128)) : c_int;

pragma "no doc"
extern proc LAPACKE_spttrs_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_float, e : [] c_float, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dpttrs_work(matrix_order : lapack_memory_order, n : c_int, nrhs : c_int, d : [] c_double, e : [] c_double, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cpttrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cpttrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_float, e : [] complex(64), b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cpttrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zpttrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zpttrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, d : [] c_double, e : [] complex(128), b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zpttrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, d, e, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_ssbev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_ssbev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_dsbev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dsbev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_ssbevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_ssbevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dsbevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dsbevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_ssbevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_ssbevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_dsbevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dsbevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssbgst_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, x : [] c_float, ldx : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgst_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, x : [] c_float, ldx : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_ssbgst_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work);
}

pragma "no doc"
extern proc LAPACKE_dsbgst_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, x : [] c_double, ldx : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgst_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, x : [] c_double, ldx : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dsbgst_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, x, ldx, work);
}

pragma "no doc"
extern proc LAPACKE_ssbgv_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgv_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_ssbgv_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_dsbgv_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgv_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dsbgv_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_ssbgvd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgvd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_ssbgvd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dsbgvd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgvd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dsbgvd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_ssbgvx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbgvx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_float, ldab : c_int, bb : [] c_float, ldbb : c_int, q : [] c_float, ldq : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_ssbgvx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_dsbgvx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbgvx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ka : c_int, kb : c_int, ab : [] c_double, ldab : c_int, bb : [] c_double, ldbb : c_int, q : [] c_double, ldq : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dsbgvx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ka, kb, ab, ldab, bb, ldbb, q, ldq, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssbtrd_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssbtrd_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] c_float, ldab : c_int, d : [] c_float, e : [] c_float, q : [] c_float, ldq : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_ssbtrd_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq, work);
}

pragma "no doc"
extern proc LAPACKE_dsbtrd_work(matrix_order : lapack_memory_order, vect : c_char, uplo : c_char, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsbtrd_work(matrix_order : lapack_memory_order, vect : string, uplo : string, n : c_int, kd : c_int, ab : [] c_double, ldab : c_int, d : [] c_double, e : [] c_double, q : [] c_double, ldq : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dsbtrd_work(matrix_order, vect.toByte() : c_char, uplo.toByte() : c_char, n, kd, ab, ldab, d, e, q, ldq, work);
}

pragma "no doc"
extern proc LAPACKE_ssfrk_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_float, a : [] c_float, lda : c_int, beta : c_float, c : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssfrk_work(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_float, a : [] c_float, lda : c_int, beta : c_float, c : [] c_float) : c_int{
  require header;
  return LAPACKE_ssfrk_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_dsfrk_work(matrix_order : lapack_memory_order, transr : c_char, uplo : c_char, trans : c_char, n : c_int, k : c_int, alpha : c_double, a : [] c_double, lda : c_int, beta : c_double, c : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsfrk_work(matrix_order : lapack_memory_order, transr : string, uplo : string, trans : string, n : c_int, k : c_int, alpha : c_double, a : [] c_double, lda : c_int, beta : c_double, c : [] c_double) : c_int{
  require header;
  return LAPACKE_dsfrk_work(matrix_order, transr.toByte() : c_char, uplo.toByte() : c_char, trans.toByte() : c_char, n, k, alpha, a, lda, beta, c);
}

pragma "no doc"
extern proc LAPACKE_sspcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sspcon_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dspcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dspcon_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cspcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_cspcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_cspcon_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_zspcon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zspcon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zspcon_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_sspev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sspev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sspev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_dspev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dspev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dspev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_sspevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_sspevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dspevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, ap : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dspevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_sspevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_sspevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_dspevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dspevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_sspgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float) : c_int{
  require header;
  return LAPACKE_sspgst_work(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_dspgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double) : c_int{
  require header;
  return LAPACKE_dspgst_work(matrix_order, itype, uplo.toByte() : c_char, n, ap, bp);
}

pragma "no doc"
extern proc LAPACKE_sspgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sspgv_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_dspgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgv_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dspgv_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_sspgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_sspgvd_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dspgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgvd_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dspgvd_work(matrix_order, itype, jobz.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, w, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_sspgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_float, bp : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_float, bp : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_sspgvx_work(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_dspgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : c_char, range : c_char, uplo : c_char, n : c_int, ap : [] c_double, bp : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspgvx_work(matrix_order : lapack_memory_order, itype : c_int, jobz : string, range : string, uplo : string, n : c_int, ap : [] c_double, bp : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dspgvx_work(matrix_order, itype, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, ap, bp, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_ssprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dsprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dsprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_csprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_csprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zsprfs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsprfs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zsprfs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_sspsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_sspsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dspsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dspsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_cspsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cspsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_cspsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zspsv_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zspsv_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zspsv_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sspsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sspsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, afp : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int, x : [] c_float, ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sspsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dspsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dspsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, afp : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int, x : [] c_double, ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dspsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_cspsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_cspsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), afp : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int, x : [] complex(64), ldx : c_int, ref rcond : c_float, ferr : [] c_float, berr : [] c_float, work : [] complex(64), rwork : [] c_float) : c_int{
  require header;
  return LAPACKE_cspsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_zspsvx_work(matrix_order : lapack_memory_order, fact : c_char, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zspsvx_work(matrix_order : lapack_memory_order, fact : string, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), afp : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int, x : [] complex(128), ldx : c_int, ref rcond : c_double, ferr : [] c_double, berr : [] c_double, work : [] complex(128), rwork : [] c_double) : c_int{
  require header;
  return LAPACKE_zspsvx_work(matrix_order, fact.toByte() : c_char, uplo.toByte() : c_char, n, nrhs, ap, afp, ipiv, b, ldb, x, ldx, rcond, ferr, berr, work, rwork);
}

pragma "no doc"
extern proc LAPACKE_ssptrd_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, d : [] c_float, e : [] c_float, tau : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptrd_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, d : [] c_float, e : [] c_float, tau : [] c_float) : c_int{
  require header;
  return LAPACKE_ssptrd_work(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_dsptrd_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, d : [] c_double, e : [] c_double, tau : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptrd_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, d : [] c_double, e : [] c_double, tau : [] c_double) : c_int{
  require header;
  return LAPACKE_dsptrd_work(matrix_order, uplo.toByte() : c_char, n, ap, d, e, tau);
}

pragma "no doc"
extern proc LAPACKE_ssptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_ssptrf_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_dsptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_dsptrf_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_csptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_csptrf_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_zsptrf_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsptrf_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int) : c_int{
  require header;
  return LAPACKE_zsptrf_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv);
}

pragma "no doc"
extern proc LAPACKE_ssptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_float, ipiv : [] c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_float, ipiv : [] c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_ssptri_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_dsptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] c_double, ipiv : [] c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] c_double, ipiv : [] c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dsptri_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_csptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(64), ipiv : [] c_int, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_csptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(64), ipiv : [] c_int, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_csptri_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_zsptri_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, ap : [] complex(128), ipiv : [] c_int, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zsptri_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, ap : [] complex(128), ipiv : [] c_int, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zsptri_work(matrix_order, uplo.toByte() : c_char, n, ap, ipiv, work);
}

pragma "no doc"
extern proc LAPACKE_ssptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_float, ipiv : [] c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_ssptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_dsptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] c_double, ipiv : [] c_int, b : [] c_double, ldb : c_int) : c_int{
  require header;
  return LAPACKE_dsptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_csptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_csptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(64), ipiv : [] c_int, b : [] complex(64), ldb : c_int) : c_int{
  require header;
  return LAPACKE_csptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_zsptrs_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zsptrs_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, nrhs : c_int, ap : [] complex(128), ipiv : [] c_int, b : [] complex(128), ldb : c_int) : c_int{
  require header;
  return LAPACKE_zsptrs_work(matrix_order, uplo.toByte() : c_char, n, nrhs, ap, ipiv, b, ldb);
}

pragma "no doc"
extern proc LAPACKE_sstebz_work(range : c_char, order : c_char, n : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, d : [] c_float, e : [] c_float, ref m : c_int, ref nsplit : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstebz_work(range : string, order : string, n : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, d : [] c_float, e : [] c_float, ref m : c_int, ref nsplit : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_sstebz_work(range.toByte() : c_char, order.toByte() : c_char, n, vl, vu, il, iu, abstol, d, e, m, nsplit, w, iblock, isplit, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dstebz_work(range : c_char, order : c_char, n : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, d : [] c_double, e : [] c_double, ref m : c_int, ref nsplit : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstebz_work(range : string, order : string, n : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, d : [] c_double, e : [] c_double, ref m : c_int, ref nsplit : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dstebz_work(range.toByte() : c_char, order.toByte() : c_char, n, vl, vu, il, iu, abstol, d, e, m, nsplit, w, iblock, isplit, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_sstedc_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstedc_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_sstedc_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dstedc_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstedc_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dstedc_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_cstedc_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cstedc_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int, work : [] complex(64), lwork : c_int, rwork : [] c_float, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_cstedc_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zstedc_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zstedc_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int, work : [] complex(128), lwork : c_int, rwork : [] c_double, lrwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zstedc_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work, lwork, rwork, lrwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_sstegr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstegr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_sstegr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dstegr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstegr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dstegr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_cstegr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cstegr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_cstegr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zstegr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zstegr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zstegr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_sstein_work(matrix_order : lapack_memory_order, n : c_int, d : [] c_float, e : [] c_float, m : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_dstein_work(matrix_order : lapack_memory_order, n : c_int, d : [] c_double, e : [] c_double, m : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_cstein_work(matrix_order : lapack_memory_order, n : c_int, d : [] c_float, e : [] c_float, m : c_int, w : [] c_float, iblock : [] c_int, isplit : [] c_int, z : [] complex(64), ldz : c_int, work : [] c_float, iwork : [] c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_zstein_work(matrix_order : lapack_memory_order, n : c_int, d : [] c_double, e : [] c_double, m : c_int, w : [] c_double, iblock : [] c_int, isplit : [] c_int, z : [] complex(128), ldz : c_int, work : [] c_double, iwork : [] c_int, ifailv : [] c_int) : c_int;

pragma "no doc"
extern proc LAPACKE_sstemr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstemr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_sstemr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dstemr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstemr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dstemr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_cstemr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_cstemr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, ref m : c_int, w : [] c_float, z : [] complex(64), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_cstemr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_zstemr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_zstemr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, ref m : c_int, w : [] c_double, z : [] complex(128), ldz : c_int, nzc : c_int, isuppz : [] c_int, ref tryrac : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_zstemr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, m, w, z, ldz, nzc, isuppz, tryrac, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_ssteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_ssteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_dsteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dsteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_csteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_csteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] complex(64), ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_csteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_zsteqr_work(matrix_order : lapack_memory_order, compz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_zsteqr_work(matrix_order : lapack_memory_order, compz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] complex(128), ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_zsteqr_work(matrix_order, compz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_ssterf_work(n : c_int, d : [] c_float, e : [] c_float) : c_int;

pragma "no doc"
extern proc LAPACKE_dsterf_work(n : c_int, d : [] c_double, e : [] c_double) : c_int;

pragma "no doc"
extern proc LAPACKE_sstev_work(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_sstev_work(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float) : c_int{
  require header;
  return LAPACKE_sstev_work(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_dstev_work(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dstev_work(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dstev_work(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz, work);
}

pragma "no doc"
extern proc LAPACKE_sstevd_work(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstevd_work(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_float, e : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_sstevd_work(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dstevd_work(matrix_order : lapack_memory_order, jobz : c_char, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstevd_work(matrix_order : lapack_memory_order, jobz : string, n : c_int, d : [] c_double, e : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dstevd_work(matrix_order, jobz.toByte() : c_char, n, d, e, z, ldz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_sstevr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstevr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_sstevr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dstevr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstevr_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dstevr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_sstevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_sstevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_float, e : [] c_float, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_sstevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_dstevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dstevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, n : c_int, d : [] c_double, e : [] c_double, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dstevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, n, d, e, vl, vu, il, iu, abstol, m, w, z, ldz, work, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssycon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssycon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] c_float, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_ssycon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_dsycon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsycon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] c_double, iwork : [] c_int) : c_int{
  require header;
  return LAPACKE_dsycon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond, work, iwork);
}

pragma "no doc"
extern proc LAPACKE_csycon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_csycon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, ipiv : [] c_int, anorm : c_float, ref rcond : c_float, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_csycon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_zsycon_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zsycon_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, ipiv : [] c_int, anorm : c_double, ref rcond : c_double, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zsycon_work(matrix_order, uplo.toByte() : c_char, n, a, lda, ipiv, anorm, rcond, work);
}

pragma "no doc"
extern proc LAPACKE_ssyequb_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float, work : [] c_float) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyequb_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_float, lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float, work : [] c_float) : c_int{
  require header;
  return LAPACKE_ssyequb_work(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax, work);
}

pragma "no doc"
extern proc LAPACKE_dsyequb_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double, work : [] c_double) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyequb_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] c_double, lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double, work : [] c_double) : c_int{
  require header;
  return LAPACKE_dsyequb_work(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax, work);
}

pragma "no doc"
extern proc LAPACKE_csyequb_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float, work : [] complex(64)) : c_int;

pragma "no doc"
inline proc LAPACKE_csyequb_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(64), lda : c_int, s : [] c_float, ref scond : c_float, ref amax : c_float, work : [] complex(64)) : c_int{
  require header;
  return LAPACKE_csyequb_work(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax, work);
}

pragma "no doc"
extern proc LAPACKE_zsyequb_work(matrix_order : lapack_memory_order, uplo : c_char, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double, work : [] complex(128)) : c_int;

pragma "no doc"
inline proc LAPACKE_zsyequb_work(matrix_order : lapack_memory_order, uplo : string, n : c_int, a : [] complex(128), lda : c_int, s : [] c_double, ref scond : c_double, ref amax : c_double, work : [] complex(128)) : c_int{
  require header;
  return LAPACKE_zsyequb_work(matrix_order, uplo.toByte() : c_char, n, a, lda, s, scond, amax, work);
}

pragma "no doc"
extern proc LAPACKE_ssyev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, w : [] c_float, work : [] c_float, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, w : [] c_float, work : [] c_float, lwork : c_int) : c_int{
  require header;
  return LAPACKE_ssyev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_dsyev_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, w : [] c_double, work : [] c_double, lwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyev_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, w : [] c_double, work : [] c_double, lwork : c_int) : c_int{
  require header;
  return LAPACKE_dsyev_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork);
}

pragma "no doc"
extern proc LAPACKE_ssyevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, w : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, w : [] c_float, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_ssyevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dsyevd_work(matrix_order : lapack_memory_order, jobz : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, w : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyevd_work(matrix_order : lapack_memory_order, jobz : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, w : [] c_double, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dsyevd_work(matrix_order, jobz.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, w, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_ssyevr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyevr_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, isuppz : [] c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_ssyevr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_dsyevr_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyevr_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, isuppz : [] c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, liwork : c_int) : c_int{
  require header;
  return LAPACKE_dsyevr_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, isuppz, work, lwork, iwork, liwork);
}

pragma "no doc"
extern proc LAPACKE_ssyevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssyevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_float, lda : c_int, vl : c_float, vu : c_float, il : c_int, iu : c_int, abstol : c_float, ref m : c_int, w : [] c_float, z : [] c_float, ldz : c_int, work : [] c_float, lwork : c_int, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_ssyevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_dsyevx_work(matrix_order : lapack_memory_order, jobz : c_char, range : c_char, uplo : c_char, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, ifail : [] c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_dsyevx_work(matrix_order : lapack_memory_order, jobz : string, range : string, uplo : string, n : c_int, a : [] c_double, lda : c_int, vl : c_double, vu : c_double, il : c_int, iu : c_int, abstol : c_double, ref m : c_int, w : [] c_double, z : [] c_double, ldz : c_int, work : [] c_double, lwork : c_int, iwork : [] c_int, ifail : [] c_int) : c_int{
  require header;
  return LAPACKE_dsyevx_work(matrix_order, jobz.toByte() : c_char, range.toByte() : c_char, uplo.toByte() : c_char, n, a, lda, vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, iwork, ifail);
}

pragma "no doc"
extern proc LAPACKE_ssygst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : c_char, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int;

pragma "no doc"
inline proc LAPACKE_ssygst_work(matrix_order : lapack_memory_order, itype : c_int, uplo : string, n : c_int, a : [] c_float, lda : c_int, b : [] c_float, ldb : c_int) : c_int{
  require header;
  return LAPACKE_ssygst_work(matrix_order, itype, uplo.toByte() : c_char, n